/*  Cprint 1.0 - a utility to convert C source files into Tex formatted files

Copyright (C) 1992  Tim Nali

    Cprint is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              */


/* Default path to look for macro files */
#define DEFAULTPATH "/usr/TeX/lib/tex/macros"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <curses.h>
#include <getopt.h>

FILE *infile, *outfile, *header;

/* various flags, see texconvert.c for explanation */
int EOFFLAG=FALSE, TTYPE=FALSE, COMMENTMYSTAR=FALSE, MYSTAR=FALSE, DELINSTYLE=FALSE, POUNDSTYLE=FALSE, STDOUT=FALSE, HEADLINE=FALSE, ASCII=FALSE, PARINDENT=FALSE;

void initialize(char *,char *,char *);
void finalize();
void texconvert(int);
FILE *open_file(char *,char *);

/* All possible command line options */
static const struct option long_options[]=
{
  "ttype",no_argument,&TTYPE,TRUE,
  "delinstyle",no_argument,&DELINSTYLE,TRUE,
  "star",no_argument,&MYSTAR,TRUE,
  "commentstar",no_argument,&COMMENTMYSTAR,TRUE,
  "poundstyle",no_argument,&POUNDSTYLE,TRUE,
  "stdout",no_argument,&STDOUT,TRUE,
  "headline",no_argument,&HEADLINE,TRUE,
  "ascii",no_argument,&ASCII,TRUE,
  "parindent",no_argument,&PARINDENT,TRUE,
  "output",required_argument,0,'o',
  "macro",required_argument,0,'m',
  0,0,0,0
};

void main(int argc, char *argv[])
{
  int c, longoptind, errflag=0;
  char *infilename=NULL, *outfilename=NULL, *headername=NULL;

  /* parse command line arguments */
  while ((c=getopt_long_only(argc,argv,"o:m:",long_options,&longoptind))!=EOF) {
    switch (c) {
    case 0:
      break;
    case 'o':
      outfilename=optarg;
      break;
    case 'm':
      headername=optarg;
      break;
    default:
      errflag++;
      break;
    }
  }
  /* input filename must be the last argument */
  infilename=argv[argc-1];
  if(errflag) {
    printf("usage:  cprint [options] [-m filename] [-o filename] filename\n");
    exit(2);
  }
  initialize(infilename,outfilename,headername);
  /* convert the file, one character at a time */
  while(!EOFFLAG && (c=getc(infile))!=EOF) {
    texconvert(c);
  }
  finalize();
  fclose(infile);
  fclose(outfile);  
}

/* Open input, output, and macro files.  Also copy the macro file
to the output file */
void initialize(char *inname, char *outname, char *headname)
{
  char headline[256], dummy[256];
  int c;
  
  if (STDOUT)
    outfile=stdout;
  else if (outname==NULL) {
    strncpy(dummy,inname,10);
    strcat(dummy,".tex");
    outfile=fopen(dummy, "w");
  }
  else 
    outfile=fopen(outname, "w");

  if(outfile==NULL) {
    fprintf(stderr,"Error opening output file %s\n",outname);
    exit(2);
  }
  
  if(headname!=NULL)
    header=open_file(headname,"r");
  else
    header=open_file("defaultmac.tex","r");
  if(header==NULL) {
    fprintf(stderr,"cprint:  Error opening macro file %s\n",headname);
    exit(2);
  }

  /* Read in headline */
  fgets(headline,256,header);

  /* Read in rest of macro file */
  while((c=getc(header))!=EOF)
    putc(c,outfile);
  
  /* wrtie headline to outfile */
  if (HEADLINE)
    fprintf(outfile,headline,inname);

  fclose(header);

  infile=open_file(inname,"r");
  if(infile==NULL) {
    fprintf(stderr,"cprint:  Error opening input file %s\n",inname);
    exit(2);
  }
}

/* send the proper codes to outfile for ending a TeX document */
void finalize()
{
  fprintf(outfile,"\n\\bye\n");
}

/* Tries to open specified file specified mode.  Tries looking in DEFAULTPATH
directory first.  If that fails, it tries looking in the current directory. 
returns a pointer to the file, or NULL if the file couldn't be found  */
FILE *open_file(char *name, char *mode)
{
  char path[256];
  FILE *file=NULL;

  strcpy(path,DEFAULTPATH);
  strcat(path,"/");
  strcat(path,name);
  if((file=fopen(path,mode))!=NULL)
    return(file);
  return(fopen(name,mode));
}







