/* driver.c	Framebuffer primitives */


#include <stdlib.h>
#include <string.h>
#include <vga.h>

#include "inlstring.h"		/* include inline string operations */
#include "vgagl.h"
#include "def.h"
#include "driver.h"


#define MAXBYTEWIDTH 4096	/* used in bitblt emulation */

#define NOTIMPL(s) notimplemented(s); return;

/* in: vp = video offset; out: rvp = video pointer, chunksize, page */
#define SETWRITEPAGED(vp, rvp, chunksize, page) \
	page = vp >> 16; \
	vga_setpage(page); \
	rvp = (vp & 0xffff) + VBUF; \
	chunksize = 0x10000 - (vp & 0xffff);


void notimplemented( char *s ) {
	printf("vgagl: %s not implemented.\n", s);
}



/* One byte per pixel frame buffer primitives */

#define ASSIGNVP8(x, y, vp) vp = VBUF + y * BYTEWIDTH + x;
#define ASSIGNVPOFFSET8(x, y, vp) vp = y * BYTEWIDTH + x;

int driver8_setpixel( int x, int y, int c ) {
	char *vp;
	ASSIGNVP8(x, y, vp);
	*vp = c;
}

int driver8p_setpixel( int x, int y, int c ) {
	int vp;
	ASSIGNVPOFFSET8(x, y, vp);
	vga_setpage(vp >> 16);
	*(VBUF + (vp & 0xffff)) = c;
}

int driver8_getpixel( int x, int y ) {
	char *vp;
	ASSIGNVP8(x, y, vp);
	return *vp;
}

int driver8p_getpixel( int x, int y ) {
	int vp;
	ASSIGNVPOFFSET8(x, y, vp);
	vga_setpage(vp >> 16);
	return *(VBUF + (vp & 0xffff));
}

int driver8_hline( int x1, int y, int x2, int c ) {
	char *vp;
	ASSIGNVP8(x1, y, vp);
	__memset(vp, c, x2 - x1 + 1);
}

int driver8p_hline( int x1, int y, int x2, int c ) {
	int vp;
	char *rvp;
	int l;
	int chunksize, page;
	ASSIGNVPOFFSET8(x1, y, vp);
	SETWRITEPAGED(vp, rvp, chunksize, page);
	l = x2 - x1 + 1;
	if (l <= chunksize)
		__memset(rvp, c, l);
	else {
		__memset(rvp, c, chunksize);
		vga_setpage(page + 1);
		__memset(VBUF, c, l - chunksize);
	}
}

int driver8_fillbox( int x, int y, int w, int h, int c ) {
	char *vp;
	int i;
	ASSIGNVP8(x, y, vp);
	for (i = 0; i < h; i++) {
		__memset(vp, c, w);
		vp += BYTEWIDTH;
	}
}

int driver8a_fillbox( int x, int y, int w, int h, int c ) {
	int vp;
	if (w * h < 128)
		driver8p_fillbox(x, y, w, h, c);
	else {
		ASSIGNVPOFFSET8(x, y, vp);
		vga_fillblt(vp, w, h, BYTEWIDTH, c);
	}
}

int driver8p_fillbox( int x, int y, int w, int h, int c ) {
	int vp;
	int page;
	int i;
	ASSIGNVPOFFSET8(x, y, vp);
	page = vp >> 16;
	vp &= 0xffff;
	vga_setpage(page);
	for (i = 0; i < h; i++) {
		if (vp + w > 0x10000)
			if (vp >= 0x10000) {
				page++;
				vga_setpage(page);
				vp &= 0xffff;
			}
			else {	/* page break within line */
				__memset(VBUF + vp, c, 0x10000 - vp);
				page++;
				vga_setpage(page);
				__memset(VBUF, c, (vp + w) & 0xffff);
				vp = (vp + BYTEWIDTH) & 0xffff;
				continue;
			}
		__memset(VBUF + vp, c, w);
		vp += BYTEWIDTH;
	}
}

inline int driver8_putbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP8(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(vp, bp, w);
		bp += bw;
		vp += BYTEWIDTH;
	}
}

int driver8p_putbox( int x, int y, int w, int h, void *b, int bw ) {
/* extra argument width of source bitmap, so that putboxpart can use this */
	int vp;
	int page;
	char *bp = b;
	int i;
	ASSIGNVPOFFSET8(x, y, vp);
	page = vp >> 16;
	vp &= 0xffff;
	vga_setpage(page);
	for (i = 0; i < h; i++) {
		if (vp + w > 0x10000)
			if (vp >= 0x10000) {
				page++;
				vga_setpage(page);
				vp &= 0xffff;
			}
			else {	/* page break within line */
				__memcpy(VBUF + vp, bp, 0x10000 - vp);
				page++;
				vga_setpage(page);
				__memcpy(VBUF, bp + 0x10000 - vp,
					(vp + w) & 0xffff);
				vp = (vp + BYTEWIDTH) & 0xffff;
				bp += bw;
				continue;
			}
		__memcpy(VBUF + vp, bp, w);
		bp += bw;
		vp += BYTEWIDTH;
	}
}

int driver8_getbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP8(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(bp, vp, w);
		bp += bw;
		vp += BYTEWIDTH;
	}
}

int driver8p_getbox( int x, int y, int w, int h, void *b, int bw ) {
	int vp;
	int page;
	char *bp = b;
	int i;
	ASSIGNVPOFFSET8(x, y, vp);
	page = vp >> 16;
	vp &= 0xffff;
	vga_setpage(page);
	for (i = 0; i < h; i++) {
		if (vp + w > 0x10000)
			if (vp >= 0x10000) {
				page++;
				vga_setpage(page);
				vp &= 0xffff;
			}
			else {	/* page break within line */
				__memcpy(bp, VBUF + vp, 0x10000 - vp);
				page++;
				vga_setpage(page);
				__memcpy(bp + 0x10000 - vp, VBUF,
					(vp + w) & 0xffff);
				vp = (vp + BYTEWIDTH) & 0xffff;
				bp += bw;
				continue;
			}
		__memcpy(bp, VBUF + vp, w);
		bp += bw;
		vp += BYTEWIDTH;
	}
}

int driver8_putboxmask( int x, int y, int w, int h, void *b ) {
	uchar *bp = b;
	uchar *vp;
	int i;
	ASSIGNVP8(x, y, vp);
	for (i = 0; i < h; i++) {
		uchar *endoflinebp = bp + w;
		while (bp < endoflinebp - 3) {
			unsigned int c4 = *(unsigned int *)bp;
			if (c4 & 0xff)
				*vp = (uchar)c4;
			c4 >>= 8;
			if (c4 & 0xff)
				*(vp + 1) = (uchar)c4;
			c4 >>= 8;
			if (c4 & 0xff)
				*(vp + 2) = (uchar)c4;
			c4 >>= 8;
			if (c4 & 0xff)
				*(vp + 3) = (uchar)c4;
			bp += 4;
			vp += 4;
			while (bp < endoflinebp) {
				uchar c = *bp++;
				if (c)
					*vp = c;
				vp++;
			}
			vp += BYTEWIDTH - w;
		}
	}
}

int driver8_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8_putbox(x, y, w, h, b + yo * ow + xo, ow);	/* inlined */
}

int driver8p_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8p_putbox(x, y, w, h, b + yo * ow + xo, ow);	/* inlined */
}

int driver8_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8_getbox(x, y, w, h, b + yo * ow + xo, ow);
}

int driver8p_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8p_getbox(x, y, w, h, b + yo * ow + xo, ow);
}

int driver8_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	char *svp, *dvp;
	if (y2 <= y1) {
		if (y1 == y2 && x2 >= x1) {	/* tricky */
			int i;
			if (x1 == x2)
				return;
			/* use a temporary buffer to store a line */
			/* using reversed movs would be much faster */
			ASSIGNVP8(x1, y1, svp);
			ASSIGNVP8(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				uchar linebuf[MAXBYTEWIDTH];
				__memcpy(linebuf, svp, w);
				__memcpy(dvp, linebuf, w);			
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
		else {	/* copy from top to bottom */ 
			int i;
			ASSIGNVP8(x1, y1, svp);
			ASSIGNVP8(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				__memcpy(dvp, svp, w);
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
	}
	else {	/* copy from bottom to top */
		int i;
		ASSIGNVP8(x1, y1 + h, svp);
		ASSIGNVP8(x2, y2 + h, dvp);
		for (i = 0; i < h; i++) {
			svp -= BYTEWIDTH;
			dvp -= BYTEWIDTH;
			__memcpy(dvp, svp, w);
		}
	}
}

int driver8a_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	int svp, dvp;
	ASSIGNVPOFFSET8(x1, y1, svp);
	ASSIGNVPOFFSET8(x2, y2, dvp);
	vga_bitblt(svp, dvp, w, h, BYTEWIDTH);
}



/* Two bytes per pixel graphics primitives */

#define ASSIGNVP16(x, y, vp) vp = VBUF + y * BYTEWIDTH + x * 2;
#define ASSIGNVPOFFSET16(x, y, vp) vp = y * BYTEWIDTH + x * 2;

int driver16_setpixel( int x, int y, int c ) {
	char *vp;
	ASSIGNVP16(x, y, vp);
	*(unsigned short *)vp = c;
}

int driver16p_setpixel( int x, int y, int c ) {
	int vp;
	ASSIGNVPOFFSET16(x, y, vp);
	vga_setpage(vp >> 16);
	*(unsigned short *)(VBUF + (vp & 0xffff)) = c;
}

int driver16_getpixel( int x, int y ) {
	char *vp;
	ASSIGNVP16(x, y, vp);
	return *(unsigned short *)vp;
}

int driver16p_getpixel( int x, int y ) {
	int vp;
	ASSIGNVPOFFSET16(x, y, vp);
	vga_setpage(vp >> 16);
	return *(unsigned short *)(VBUF + (vp & 0xffff));
}

int driver16_hline( int x1, int y, int x2, int c ) {
	char *vp;
	ASSIGNVP16(x1, y, vp);
	__memset2(vp, c, x2 - x1 + 1);
}

int driver16p_hline( int x1, int y, int x2, int c ) {
	int vp;
	char *rvp;
	int l;
	int chunksize, page;
	ASSIGNVPOFFSET16(x1, y, vp);
	SETWRITEPAGED(vp, rvp, chunksize, page);
	l = (x2 - x1 + 1) * 2;
	if (l <= chunksize)
		__memset2(rvp, c, l / 2);
	else {
		__memset2(rvp, c, chunksize / 2);
		vga_setpage(page + 1);
		__memset2(VBUF, c, (l - chunksize) / 2);
	}
}

int driver16_fillbox( int x, int y, int w, int h, int c ) {
	char *vp;
	int i;
	ASSIGNVP16(x, y, vp);
	for (i = 0; i < h; i++) {
		__memset2(vp, c, w);
		vp += BYTEWIDTH;
	}
}

int driver16p_fillbox( int x, int y, int w, int h, int c ) {
	int vp;
	int page;
	int i;
	ASSIGNVPOFFSET16(x, y, vp);
	page = vp >> 16;
	vp &= 0xffff;
	vga_setpage(page);
	for (i = 0; i < h; i++) {
		if (vp + w * 2 > 0x10000)
			if (vp >= 0x10000) {
				page++;
				vga_setpage(page);
				vp &= 0xffff;
			}
			else {	/* page break within line */
				__memset2(VBUF + vp, c, (0x10000 - vp) / 2);
				page++;
				vga_setpage(page);
				__memset2(VBUF, c, ((vp + w * 2) & 0xffff) / 2);
				vp = (vp + BYTEWIDTH) & 0xffff;
				continue;
			}
		__memset2(VBUF + vp, c, w);
		vp += BYTEWIDTH;
	}
}

inline int driver16_putbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP16(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(vp, bp, w * 2);
		bp += bw * 2;
		vp += BYTEWIDTH;
	}
}

int driver16p_putbox( int x, int y, int w, int h, void *b, int bw ) {
	driver8p_putbox(x * 2, y, w * 2, h, b, bw * 2);
}

int driver16_getbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP16(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(bp, vp, w * 2);
		bp += bw * 2;
		vp += BYTEWIDTH;
	}
}

int driver16p_getbox( int x, int y, int w, int h, void *b, int bw ) {
	driver8p_getbox(x * 2, y, w * 2, h, b, bw * 2);
}

int driver16_putboxmask( int x, int y, int w, int h, void *b ) {
	uchar *bp = b;
	uchar *vp;
	int i;
	ASSIGNVP16(x, y, vp);
	for (i = 0; i < h; i++) {
		uchar *endoflinebp = bp + w * 2;
		while (bp < endoflinebp - 7) {
			unsigned c2 = *(unsigned *)bp;
			if (c2 & 0xffff)
				*(ushort *)vp = (ushort)c2;
			c2 >>= 16;
			if (c2 & 0xffff)
				*(ushort *)(vp + 2) = (ushort)c2;
			c2 = *(unsigned *)(bp + 4);
			if (c2 & 0xffff)
				*(ushort *)(vp + 4) = (ushort)c2;
			c2 >>= 16;
			if (c2 & 0xffff)
				*(ushort *)(vp + 6) = (ushort)c2;
			bp += 8;
			vp += 8;
		}
		while (bp < endoflinebp) {
			ushort c = *(ushort *)bp;
			if (c)
				*(ushort *)vp = c;
			bp += 2;
			vp += 2;
		}
		vp += BYTEWIDTH - w * 2;
	}
}

int driver16_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8_putbox(x * 2, y, w * 2, h, b + yo * ow * 2 + xo * 2, ow * 2);
		/* inlined */
}

int driver16p_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8p_putbox(x * 2, y, w * 2, h, b + yo * ow * 2 + xo * 2, ow * 2);
}

int driver16_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver16_getbox(x, y, w, h, b + yo * ow + xo, ow);
}

int driver16p_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver16p_getbox(x, y, w, h, b + yo * ow + xo, ow);
}


int driver16_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	char *svp, *dvp;
	if (y2 <= y1) {
		if (y1 == y2 && x2 >= x1) {	/* tricky */
			int i;
			if (x1 == x2)
				return;
			/* use a temporary buffer to store a line */
			/* using reversed movs would be much faster */
			ASSIGNVP16(x1, y1, svp);
			ASSIGNVP16(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				uchar linebuf[MAXBYTEWIDTH];
				__memcpy(linebuf, svp, w * 2);
				__memcpy(dvp, linebuf, w * 2);
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
		else {	/* copy from top to bottom */ 
			int i;
			ASSIGNVP16(x1, y1, svp);
			ASSIGNVP16(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				__memcpy(dvp, svp, w * 2);
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
	}
	else {	/* copy from bottom to top */
		int i;
		ASSIGNVP8(x1, y1 + h, svp);
		ASSIGNVP8(x2, y2 + h, dvp);
		for (i = 0; i < h; i++) {
			svp -= BYTEWIDTH;
			dvp -= BYTEWIDTH;
			__memcpy(dvp, svp, w * 2);
		}
	}
}

int driver16a_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	int svp, dvp;
	ASSIGNVPOFFSET16(x1, y1, svp);
	ASSIGNVPOFFSET16(x2, y2, dvp);
	vga_bitblt(svp, dvp, w * 2, h, BYTEWIDTH);
}



/* Three bytes per pixel graphics primitives */

#define ASSIGNVP24(x, y, vp) vp = VBUF + y * BYTEWIDTH + x * 3;
#define ASSIGNVPOFFSET24(x, y, vp) vp = y * BYTEWIDTH + x * 3;
#define RGBEQUAL(c) ((c & 0xff) == ((c >> 8) & 0xff) && \
	(c & 0xff) == ((c >> 16) & 0xff))

int driver24_setpixel( int x, int y, int c ) {
	char *vp;
	ASSIGNVP24(x, y, vp);
	*(unsigned short *)vp = c;
	*(unsigned char *)(vp + 2) = c >> 16;
}

int driver24p_setpixel( int x, int y, int c ) {
	int vp, vpo;
	char *vbuf;
	int page;
	ASSIGNVPOFFSET24(x, y, vp);
	vbuf = VBUF;
	page = vp >> 16;
	vga_setpage(page);
	vpo = vp & 0xffff;
	if (vpo <= 0xfffd) {
		*(unsigned short *)(vbuf + vpo) = c;
		*(unsigned char *)(vbuf + vpo + 2) = c >> 16;
	}
	else
	if (vpo == 0xfffe) {
		*(unsigned short *)(vbuf + 0xfffe) = c;
		vga_setpage(page + 1);
		*(unsigned char *)vbuf = c >> 16;
	}
	else {	/* vpo == 0xffff */
		*(unsigned char *)(vbuf + 0xffff) = c;
		vga_setpage(page + 1);
		*(unsigned short *)vbuf = c >> 8;
	}
}

int driver24_getpixel( int x, int y ) {
	char *vp;
	ASSIGNVP24(x, y, vp);
	return *(unsigned short *)vp + (*(unsigned char *)(vp + 2) << 16);
}

int driver24p_getpixel( int x, int y ) {
	int vp, vpo;
	char *vbuf;
	int page;
	ASSIGNVPOFFSET24(x, y, vp);
	vbuf = VBUF;
	page = vp >> 16;
	vga_setpage(page);
	vpo = vp & 0xffff;
	if (vpo <= 0xfffd)
		return *(unsigned short *)(vbuf + vpo) +
		       (*(unsigned char *)(vbuf + vpo + 2) << 16);
	else
	if (vpo == 0xfffe) {
		int c;
		c = *(unsigned short *)(vbuf + 0xfffe) = c;
		vga_setpage(page + 1);
		return (*(unsigned char *)vbuf << 16) + c;
	}
	else {	/* vpo == 0xffff */
		int c;
		c = *(unsigned char *)(vbuf + 0xffff);
		vga_setpage(page + 1);
		return (*(unsigned short *)vbuf << 8) + c;
	}
}

int driver24_hline( int x1, int y, int x2, int c ) {
	char *vp;
	ASSIGNVP24(x1, y, vp);
	if (RGBEQUAL(c))
		__memset(vp, c, (x2 - x1 + 1) * 3);
	else
		__memset3(vp, c, x2 - x1 + 1);
}

int driver24p_hline( int x1, int y, int x2, int c ) {
/* Cannot handle page break within pixel */
	int vp;
	char *rvp;
	int l;
	int chunksize, page;
	ASSIGNVPOFFSET24(x1, y, vp);
	SETWRITEPAGED(vp, rvp, chunksize, page);
	l = (x2 - x1 + 1) * 3;
	if (l <= chunksize)
		__memset3(rvp, c, l / 3);
	else {
		__memset3(rvp, c, chunksize / 3);
		vga_setpage(page + 1);
		__memset3(VBUF, c, (l - chunksize) / 3);
	}
}

int driver24_fillbox( int x, int y, int w, int h, int c ) {
	char *vp;
	int i, j;
	ASSIGNVP24(x, y, vp);
	if (RGBEQUAL(c))
		for (i = 0; i < h; i++) {
			__memset(vp, c, w * 3);
			vp += BYTEWIDTH;
		}
	else
		for (j = 0; j < h; j++) {
			__memset3(vp, c, w);
			vp += BYTEWIDTH;
		}
}

int driver24p_fillbox( int x, int y, int w, int h, int c ) {
/* Cannot handle page break within pixel */
	int vp;
	int page;
	int i;
	ASSIGNVPOFFSET24(x, y, vp);
	page = vp >> 16;
	vp &= 0xffff;
	vga_setpage(page);
	if (RGBEQUAL(c)) {
		for (i = 0; i < h; i++) {
			if (vp + w * 3 > 0x10000)
				if (vp >= 0x10000) {
					page++;
					vga_setpage(page);
					vp &= 0xffff;
				}
				else {	/* page break within line */
					__memset(VBUF + vp, c, 0x10000 - vp);
					page++;
					vga_setpage(page);
					__memset(VBUF, c, (vp + w * 3) & 0xffff);
					vp = (vp + BYTEWIDTH) & 0xffff;
					continue;
				}
			__memset(VBUF + vp, c, w * 3);
			vp += BYTEWIDTH;
		}
	}
	else
		for (i = 0; i < h; i++) {
			if (vp + w * 3 > 0x10000)
				if (vp >= 0x10000) {
					page++;
					vga_setpage(page);
					vp &= 0xffff;
				}
				else {	/* page break within line */
					__memset3(VBUF + vp, c, (0x10000 - vp) / 3);
					page++;
					vga_setpage(page);
					__memset3(VBUF, c, ((vp + w * 3) & 0xffff) / 3);
					vp = (vp + BYTEWIDTH) & 0xffff;
					continue;
				}
			__memset3(VBUF + vp, c, w);
			vp += BYTEWIDTH;
		}
}

int driver24_putbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP24(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(vp, bp, w * 3);
		bp += bw * 3;
		vp += BYTEWIDTH;
	}
}

int driver24p_putbox( int x, int y, int w, int h, void *b, int bw ) {
	driver8p_putbox(x * 3, y, w * 3, h, b, bw * 3);
}

int driver24_getbox( int x, int y, int w, int h, void *b, int bw ) {
	char *vp;	/* screen pointer */
	char *bp;	/* bitmap pointer */
	int i;
	ASSIGNVP24(x, y, vp);
	bp = b;
	for (i = 0; i < h; i++) {
		__memcpy(bp, vp, w * 3);
		bp += bw * 3;
		vp += BYTEWIDTH;
	}
}

int driver24p_getbox( int x, int y, int w, int h, void *b, int bw ) {
	driver8p_getbox(x * 3, y, w * 3, h, b, bw * 3);
}

int driver24_putboxmask( int x, int y, int w, int h, void *b ) {
	uchar *bp = b;
	uchar *vp;
	int i;
	ASSIGNVP24(x, y, vp);
	for (i = 0; i < h; i++) {
		uchar *endoflinebp = bp + w * 3;
		while (bp < endoflinebp - 11) {
			unsigned c = *(unsigned *)bp;
			if (c & 0xffffff) {
				*(ushort *)vp = (ushort)c;
				*(vp + 2) = c >> 16;
			}
			c = *(unsigned *)(bp + 3);
			if (c & 0xffffff) {
				*(ushort *)(vp + 3) = (ushort)c;
				*(vp + 5) = c >> 16;
			}
			c = *(unsigned *)(bp + 6);
			if (c & 0xffffff) {
				*(ushort *)(vp + 6) = (ushort)c;
				*(vp + 8) = c >> 16;
			}
			c = *(unsigned *)(bp + 9);
			if (c & 0xffffff) {
				*(ushort *)(vp + 9) = (ushort)c;
				*(vp + 11) = c >> 16;
			} 
			bp += 12;
			vp += 12;
		}
		while (bp < endoflinebp) {
			uint c = *(uint *)bp;
			if (c & 0xffffff) {
				*(ushort *)vp = (ushort)c;
				*(vp + 2) = c >> 16;
			}
			bp += 3;
			vp += 3;
		}
		vp += BYTEWIDTH - w * 3;
	}
}

int driver24_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8_putbox(x * 3, y, w * 3, h, b + yo * ow * 3 + xo * 3, ow * 3);
		/* inlined */
}

int driver24p_putboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver8p_putbox(x * 3, y, w * 3, h, b + yo * ow * 3 + xo * 3, ow * 3);
}

int driver24_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver24_getbox(x, y, w, h, b + yo * ow + xo, ow);
}

int driver24p_getboxpart( int x, int y, int w, int h, int ow, int oh,
void *b, int xo, int yo ) {
	driver24p_getbox(x, y, w, h, b + yo * ow + xo, ow);
}

int driver24_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	char *svp, *dvp;
	if (y2 <= y1) {
		if (y1 == y2 && x2 >= x1) {	/* tricky */
			int i;
			if (x1 == x2)
				return;
			/* use a temporary buffer to store a line */
			/* using reversed movs would be much faster */
			ASSIGNVP24(x1, y1, svp);
			ASSIGNVP24(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				uchar linebuf[MAXBYTEWIDTH];
				__memcpy(linebuf, svp, w * 3);
				__memcpy(dvp, linebuf, w * 3);
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
		else {	/* copy from top to bottom */ 
			int i;
			ASSIGNVP24(x1, y1, svp);
			ASSIGNVP24(x2, y2, dvp);
			for (i = 0; i < h; i++) {
				__memcpy(dvp, svp, w * 3);
				svp += BYTEWIDTH;
				dvp += BYTEWIDTH;
			}
		}
	}
	else {	/* copy from bottom to top */
		int i;
		ASSIGNVP24(x1, y1 + h, svp);
		ASSIGNVP24(x2, y2 + h, dvp);
		for (i = 0; i < h; i++) {
			svp -= BYTEWIDTH;
			dvp -= BYTEWIDTH;
			__memcpy(dvp, svp, w * 3);
		}
	}
}

int driver24a_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	int svp, dvp;
	ASSIGNVPOFFSET24(x1, y1, svp);
	ASSIGNVPOFFSET24(x2, y2, dvp);
	vga_bitblt(svp, dvp, w * 3, h, BYTEWIDTH);
}



/* Memory primitives */

int driver_setread( GraphicsContext *gc, int i, void **vp ) {
	if (gc->modetype == CONTEXT_PAGED) {
		vga_setpage(i >> 16);
		*vp = (i & 0xffff) + gc->vbuf;
		return 0x10000 - (i & 0xffff);
	}
	else {
		*vp = gc->vbuf + i;
		return 0x10000;
	}
}

int driver_setwrite( GraphicsContext *gc, int i, void **vp ) {
	if (gc->modetype == CONTEXT_PAGED) {
		vga_setpage(i >> 16);
		*vp = (i & 0xffff) + gc->vbuf;
		return 0x10000 - (i & 0xffff);
	}
	else {
		*vp = gc->vbuf + i;
		return 0x10000;
	}
}



/* Functions that are not yet implemented */

int driver8p_putboxmask( int x, int y, int w, int h, void *b ) {
	NOTIMPL("8-bit paged putboxmask");
}

int driver8p_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	NOTIMPL("8-bit paged copybox (bitblt)");
}

int driver16p_putboxmask( int x, int y, int w, int h, void *b ) {
	NOTIMPL("16-bit paged putboxmask");
}

int driver16p_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	NOTIMPL("16-bit paged copybox");
}

int driver24p_putboxmask( int x, int y, int w, int h, void *b ) {
	NOTIMPL("24-bit paged putboxmask");
}

int driver24p_copybox( int x1, int y1, int w, int h, int x2, int y2 ) {
	NOTIMPL("24-bit paged copybox");
}
