/* VGAlib version 1.2 - (c) 1993 Tommy Frandsen 		   */
/*								   */
/* This library is free software; you can redistribute it and/or   */
/* modify it without any restrictions. This library is distributed */
/* in the hope that it will be useful, but without any warranty.   */

/* Multi-chipset support Copyright 1993 Harm Hanemaayer */
/* ET4000 code taken from VGAlib */
/* ET4000 code modified to handle HiColor modes better by David Monro */


#include <stdio.h>
#include "vga.h"
#include "vgamacros.h"
#include "chipsets.h"
#include "config.h"


/* Note: definition of ET4000_DAC in config.h determines hicolor DAC */
/* detection behaviour */


#define SEG_SELECT 0x3CD

#ifdef ET4000_BOCA

/* ET4000 registers for BOCA SVGA card */

static unsigned char et4000_g640x480x256_regs[71] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x50,0x60,0xE7,0x04,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xE3,
  0x00,0x00,0x08,0x00,0x43,0x0F,0x00,0xBC,0x00,0x00,0x00
};

static unsigned char et4000_g800x600x256_regs[71] = {
  0x7F,0x63,0x64,0x02,0x6A,0x1D,0x77,0xF0,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x5D,0x8F,0x57,0x64,0x60,0x5B,0x74,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xEF,
  0x00,0x00,0x08,0x00,0x43,0x0F,0x00,0xBC,0x00,0x00,0x00
};

/* BIOS mode 0x38 */
static unsigned char regs[71] = {
  0x99,0x7F,0x7F,0x1D,0x83,0x17,0x2F,0xF5,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x08,0x80,0xFF,0x80,0x60,0xFF,0x30,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0x27,
  0x00,0x00,0x0A,0x80,0x43,0x0F,0x00,0xBC,0x00,0x00,0x00
};

#else

/* ET4000 BIOS mode 2Eh - 640x480x256 */
static unsigned char et4000_g640x480x256_regs[71] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x50,0x60,0xE7,0x04,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xE3,
  0x28,0x00,0x08,0x00,0x03,0x0F,0x00,0xFC,0x00,0x00,0x00
};

/* ET4000 BIOS mode 30h - 800x600x256 */
static unsigned char et4000_g800x600x256_regs[71] = {
  0x7F,0x63,0x64,0x02,0x63,0x1D,0x77,0xF0,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x5D,0x8F,0x57,0x64,0x60,0x5B,0x74,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xE3,
  0x28,0x00,0x0A,0x00,0x03,0x0F,0x00,0xFC,0x00,0x00,0x00
};

/* ET4000 BIOS mode 38h - 1024x768x256 */
static unsigned char et4000_g1024x768x256_regs[71] = {
  0xA1,0x7F,0x80,0x04,0x89,0x99,0x26,0xFD,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x08,0x8A,0xFF,0x80,0x60,0x04,0x22,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xEB,
  0x28,0x00,0x08,0x00,0x03,0x0F,0x00,0xBC,0x00,0x00,0x00
};

#endif

static unsigned char et4000_g800x600x32k_regs[71] = {
  0xF9,0xC7,0xC9,0x9B,0xCF,0x8F,0x78,0xF0,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x5C,0x8E,0x57,0xC8,0x60,0x5B,0x75,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xEF,
  0x28,0x00,0x08,0x00,0x03,0x0F,0x00,0xBC,0x00,0x00,0x06
};

static unsigned char et4000_g640x480x16M_regs[71] = {
  0x27,0xEF,0xF2,0x88,0xF8,0x98,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x0C,0xDF,0xF0,0x60,0xE7,0x04,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xEF,
  0x28,0x00,0x0A,0x00,0x03,0x0F,0x01,0xBC,0x00,0x00,0x00
};

static unsigned char et4000_g640x480x32k_regs[71] = {
  0xC3,0x9F,0xA1,0x85,0xA7,0x1F,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0xA0,0x60,0xE7,0x04,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xE3,
  0x28,0x00,0x08,0x00,0x03,0x0F,0x00,0xBC,0x00,0x00,0x00
};

static int et4000_memory;
static int et4000_dac;

static int et4000_init( int, int );

static void dactopel();
static void dactocomm();


/* Mode table */

static unsigned char *et4000_moderegs[] = {
	et4000_g640x480x256_regs, et4000_g800x600x256_regs,	/* SVGA */
	et4000_g1024x768x256_regs,
	NULL, NULL, NULL,	/* 320x200 hicolor/truecolor */
	et4000_g640x480x32k_regs, et4000_g640x480x32k_regs,
	et4000_g640x480x16M_regs,
	et4000_g800x600x32k_regs, et4000_g800x600x32k_regs, NULL
};


/* Fill in chipset specific mode information */

static int et4000_getmodeinfo( int mode, vga_modeinfo *modeinfo ) {
	modeinfo->maxpixels = et4000_memory * 1024 / modeinfo->bytesperpixel;
	modeinfo->maxlogicalwidth = 4088;
	modeinfo->startaddressrange = 0xfffff;
	modeinfo->haveblit = 0;
}


/* Read and store chipset-specific registers */

static int et4000_saveregs(unsigned char regs[])
{
    int i;

    /* save extended CRT registers */
    for (i = 0; i < 6; i++) {
	 port_out(0x32+i, CRT_I); 
	 regs[EXT+i] = port_in(CRT_D); 
    }
    port_out(0x3f, CRT_I);
    regs[EXT+6] = port_in(CRT_D);

    /* save extended sequencer register */
    port_out(7, SEQ_I); 
    regs[EXT+7] = port_in(SEQ_D); 

    /* save some other ET4000 specific registers */
    regs[EXT+8] = port_in(0x3c3); 
    regs[EXT+9] = port_in(0x3cd); 

    /* save extended attribute register */
    port_in(IS1_R);    /* reset flip flop */
    port_out(0x16, ATT_IW);
    regs[EXT+10] = port_in(ATT_R);

    return;
}


/* Set chipset-specific registers */

static int et4000_setregs(unsigned char regs[])
{
    int i;

    /* write some ET4000 specific registers */
    port_out(regs[EXT+8], 0x3c3);
    port_out(regs[EXT+9], 0x3cd);

    /* write extended sequencer register */
    port_out(7, SEQ_I); 
    port_out(regs[EXT+7], SEQ_D); 

    /* write extended CRT registers */
    for (i = 0; i < 6; i++) {
	 port_out(0x32+i, CRT_I); 
	 port_out(regs[EXT+i], CRT_D); 
    }
    port_out(0x3f, CRT_I);
    port_out(regs[EXT+6], CRT_D);

    /* write extended attribute register */
    port_in(IS1_R);    /* reset flip flop */
    port_out(0x16, ATT_IW);
    port_out(regs[EXT+10], ATT_IW);
    
    return;
}


/* Return non-zero if mode is available */

static int et4000_modeavailable( int mode ) {
	if (STDVGAMODE(mode))
		return 1;		/* always available */
	switch (mode) {
		case G640x480x256 :
		case G800x600x256 :
			return (et4000_memory >= 512);
		case G1024x768x256 :
			return (et4000_memory == 1024);
		case G640x480x32K :
		case G800x600x32K :
			/* All HiColor dacs handle 15 bit */
			return (et4000_memory == 1024 && (et4000_dac & 1)); 
		case G640x480x64K :
		case G800x600x64K :
			/* Unfortunately, we can't tell the difference between a Sierra Mark 2 */
			/* and a Mark 3, and the mark 2 is only 15 bit. If this gives you trouble */
			/* change it to (8|16) rather than (2|8|16) */
			/* Currently allow Sierra Mark2/3, AT&T and AcuMos */
			return (et4000_memory == 1024 && (et4000_dac & (2|8|16)));
		case G640x480x16M :
			/* Don't know how to set dac command register for Diamond SS2410 Dac */
			/* Only allow AT&T and AcuMos */
			return (et4000_memory == 1024 && (et4000_dac & (8|16)));
		default :
			return 0;
	}
}


/* Set a mode */

static int et4000_setmode( int mode ) {
	if (!et4000_modeavailable(mode))
		return 1;	/* mode not available */
	if (et4000_dac)
		switch (vga_getmodeinfo(mode)->colors)
		{
		case 32768:
			vga_hicolor(1);
			break;
		case 65536:
			vga_hicolor(2);
			break;
		case 65536*256:
			vga_hicolor(3);
			break;
		default:
			vga_hicolor(0);
			break;
		}

	if (STDVGAMODE(mode))
		vga_chipsetfunctions[CHIPSET_SETMODE](mode);
	else {
		unsigned char *regs = et4000_moderegs[mode - G640x480x256];
		vga_setregs(regs);
		et4000_setregs(regs);
	}
}


/* Unlock chipset-specific registers */

static int et4000_unlock() {
	/* get access to extended registers */
	port_out(3, 0x3bf);
	if (port_in( 0x3cc ) & 1) 
		port_out(0xa0, 0x3d8);
	else
		port_out(0xa0, 0x3b8);
}


/* Relock chipset-specific registers */

static int et4000_lock() {
}


/* Indentify chipset; return non-zero if detected */

static int et4000_test()
{
    unsigned char new, old, val;
    int  base;

    /* test for Tseng clues */
    old = port_in(0x3cd);			  
    port_out(0x55, 0x3cd);
    new = port_in(0x3cd);			  
    port_out(old, 0x3cd);

    /* return false if not Tseng */
    if (new != 0x55)
	return 0;

    /* test for ET4000 clues */
    if (port_in(0x3cc) & 1) 
	base = 0x3d4;
    else 
	base = 0x3b4;
    port_out(0x33, base);
    old = port_in(base+1);
    new = old ^ 0xf;
    port_out(new, base+1);
    val = port_in(base+1);
    port_out(old, base+1);

    /* return true if ET4000 */    
    if (val == new) {
    	et4000_init(0, 0);
    	return 1;
    }
    return 0;
}


/* Bank switching function - set 64K bank number */

static int et4000_setpage( int page ) {
	port_out(page | (page << 4), SEG_SELECT);
}


/* Set display start address (not for 16 color modes) */
/* ET4000 supports any address in video memory (up to 1Mb) */

static int et4000_setdisplaystart( int address ) {
	outw(0x3d4, 0x0d + ((address >> 2) & 0x00ff) * 256);	/* sa2-sa9 */
	outw(0x3d4, 0x0c + ((address >> 2) & 0xff00));		/* sa10-sa17 */
	inb(0x3da);			/* set ATC to addressing mode */
	outb(0x3c0, 0x13 + 0x20);	/* select ATC reg 0x13 */
	outb(0x3c0, (inb(0x3c1) & 0xf0) | ((address & 3) << 1));
		/* write sa0-1 to bits 1-2 */
	outb(0x3d4, 0x33);
	outb(0x3d5, (inb(0x3d5) & 0xfc)
		| ((address & 0xc0000) >> 18));	/* write sa18-19 to bit 0-1 */
}


/* Set logical scanline length (usually multiple of 8) */
/* ET4000 supports multiples of 8 to 4088 */

static int et4000_setlogicalwidth( int width ) { 
	outw(0x3d4, 0x13 + (width >> 3) * 256);	/* lw3-lw11 */
	outb(0x3d4, 0x3f);
	outb(0x3d5, (inb(0x3d5) & 0x7f)
		| ((width & 0x800) >> 5));	/* write lw12 to bit 7 */
}


/* Function table (exported) */

int (*et4000_chipsetfunctions[])() = {
	et4000_saveregs,
	et4000_setregs,
	et4000_unlock,
	et4000_lock,
	et4000_test,
	et4000_init,
	et4000_setpage,
	et4000_setmode,
	et4000_modeavailable,
	et4000_setdisplaystart,
	et4000_setlogicalwidth,
	et4000_getmodeinfo
};


/* Hicolor DAC detection derived from vgadoc (looks tricky) */

static void dactocomm() {
	inb(0x3c8);	/* set PEL mode */
	inb(0x3c6);	/* read four times */
	inb(0x3c6);
	inb(0x3c6);
	inb(0x3c6);	
}

static void dactopel() {
	inb(0x3c8);
}

static int testdac()
{
	int x, y, z, v, oldcommreg, oldpelreg, retval;

	/* Use the following return values:
	 * 	0: Ordinary DAC
	 *	1: Sierra SC11486 (32k)
	 *	3: Sierra 32k/64k
	 *	5: SS2410 15/24bit DAC
	 *	9: AT&T 20c491/2
	 */

	#ifdef ET4000_DAC
	retval = ET4000_DAC;
	return retval;
	#endif

	retval = 0;	/* default to no fancy dac */
	dactopel();
	x=inb(0x3c6);
	do {		/* wait for repeat of value */
		y = x;
		x = inb(0x3c6);
	} while (x != y);
	z=x;
	inb(0x3c8);
	inb(0x3c6);
	inb(0x3c6);
	inb(0x3c6);
	x=inb(0x3c6);
	y=8;
	while ( (x!=0x8e) && (y>0) )
	{
		x=inb(0x3c6);
		y--;
	}
	if (x==0x8e)
	{
		/* We have an SS2410 */
		retval = 1 | 4;
		dactopel();
	}
	else
	{
		dactocomm();
		oldcommreg = inb(0x3c6);
		dactopel();
		oldpelreg = inb(0x3c6);
		x = oldcommreg ^ 0xFF;
		outb(0x3c6, x);
		dactocomm();
		v=inb(0x3c6);
		if (v != x)
		{
			dactocomm();
			x = oldcommreg ^ 0x60;
			outb(0x3c6, x);
			dactocomm();
			v = inb(0x3c6);
			/* We have a Sierra SC11486 */
			retval = 1;

			if ( (x & 0xe0) == (v & 0xe0) )
			{
				/* We have a Sierra 32k/64k */
				x = inb(0x3c6);
				dactopel();
				retval = 1 | 2;

				if ( x == inb(0x3c6) )
				{
					/* We have an ATT 20c491 or 20c492 */
					retval = 1 | 8;
					dactocomm();
					outb(0x3c6, 0xFF);
					dactocomm();
					if (inb(0x3c6) != 0xFF)
					{
						/* We have a weird dac, AcuMos ADAC1 */
						retval = 1 | 16;
					}
				}
			}
			dactocomm();
			outb(0x3c6, oldcommreg);
		}
		dactopel();
		outb(0x3c6, oldpelreg);
	}
	return retval;
}


/* Initialize chipset (called after detection) */

static int et4000_init( int force, int par1 ) {
	int value;
	if (force)
		et4000_memory = par1;
	else {
		outb(0x3d4,0x37);
		value = inb(0x3d5);
		switch (value & 3)

		{
		case 0:
		case 1:
			et4000_memory = 256;
			break;
		case 2:
			et4000_memory = 512;
			break;
		case 3:
			et4000_memory = 1024;
			break;
		default:
			break;
		}
	}
	et4000_dac = testdac();
	if (__driver_report) {
		char dacname[60];
		switch (et4000_dac & ~1)
		{
		case 0:
			strcpy(dacname, ", Hicolor Dac: Sierra SC11486");
			break;
		case 2:
			strcpy(dacname, ", Hicolor Dac: Sierra 32k/64k");
			break;
		case 4:
			strcpy(dacname, ", Hicolor Dac: SS2410");
			break;
		case 8:
			strcpy(dacname, ", Hicolor Dac: ATT20c491/2");
			break;
		case 16:
			strcpy(dacname, ", Hicolor Dac: ACUMOS ADAC1");
			break;
		default:
			strcpy(dacname, ", Hicolor Dac: Unknown");
			break;
		}

		printf("Using Tseng ET4000 driver (%d%s).\n", et4000_memory,
			et4000_dac & 1 ? dacname : "");
	}
	chipsetfunctions = et4000_chipsetfunctions;
}


int vga_hicolor(int mode)
/* Enters hicolor mode - 0 for no hi, 1 for 15 bit, 2 for 16, 3 for 24 */
/* For any modes it doesn't know about, etc, it attempts to turn hicolor off. */
{
	int x;
/*	ioperm(0x3c8, 1, 1);
	ioperm(0x3c6, 1, 1); */
	port_in(0x3C8);		/* Clears register funny mode */
	port_in(0x3C6);
	port_in(0x3C6);
	port_in(0x3C6);
	port_in(0x3C6);
	x = port_in(0x3c6);
	port_in(0x3C8);		/* Clears register funny mode */
	port_in(0x3C6);
	port_in(0x3C6);
	port_in(0x3C6);
	port_in(0x3C6);
	switch (et4000_dac & ~1)
	{
	    case 0:	/* Sierra SC11486 */
		if (mode == 1)
		    port_out( x | 0x80, 0x3c6);
		else
		    port_out( x & ~0x80, 0x3c6);
		break;
	    case 2:	/* Sierra Mark2/Mark3 */
		switch (mode)
		{
		case 1:
		    /* port_out( (x | 0x80) & ~0x40, 0x3c6); */
		    port_out( (x & 0x1f) | 0xa0, 0x3c6);
		    break;
		case 2:
		    /* port_out( x | 0xC0, 0x3c6); */
		    port_out( (x & 0x1f) | 0xe0, 0x3c6);
		    break;
		case 0:
		default:
		    port_out( x & ~0xC0, 0x3c6);
		    break;
		}
		break;
	    case 4:	/* Diamond SS2410 */
		if (mode == 1)
		    port_out( x | 0x80, 0x3c6);
		else
		    port_out( x & ~0x80, 0x3c6);
		break;
	    case 8:	/* AT&T 20c491/2 */
		switch (mode)
		{
		case 1:
		    port_out(x | 0xA0, 0x3C6);
		    break;
		case 2:
		    port_out(x | 0xC0, 0x3C6);
		    break;
		case 3:
		    port_out(x | 0xE0, 0x3C6);
		    break;
		case 0:
		default:
		    port_out(x & 0x1F, 0x3C6);
		    break;
		}
		break;
	    case 16:	/* AcuMos ADAC1 */
		switch (mode)
		{
		case 1:
		    port_out(0xF0, 0x3C6);
		    break;
		case 2:
		    port_out(0xE1, 0x3C6);
		    break;
		case 3:
		    port_out(0xE5, 0x3C6);
		    break;
		case 0:
		default:
		    port_out(0, 0x3C6);
		    break;
		}
		break;
	default:
		port_out(x & 0x1F, 0x3c6);
		break;
	}
	port_in(0x3C8);
}
