/*
 * e2fsck.c		- Consistency checker for the second extended
 *			  file system
 *
 * Copyright (C) 1992, 1993  Remy Card <card@masi.ibp.fr>
 *
 * This file is based on the minix file system programs fsck and mkfs
 * written and copyrighted by Linus Torvalds <Linus.Torvalds@cs.helsinki.fi>
 *
 * This file can be redistributed under the terms of the GNU General
 * Public License
 */

/*
 * History:
 * 93/05/26	- Creation
 * 93/10/20	- Added dtime checks for non deleted inodes
 * 93/10/21	- Added links_count check for deleted inodes
 * 93/10/31	- Checks the mount counter
 * 93/11/01	- Added -B option to allow recovery from an alternate 
 *		  super block
 * 93/11/13	- Removed the i_flags check (incompatible with attributes)
 * 93/11/14	- Added more descriptive messages in case of bad dir entries
 */

/*
 * Old history:
 * 09.11.91  -  made the first rudimetary functions
 *
 * 10.11.91  -  updated, does checking, no repairs yet.
 *		Sent out to the mailing-list for testing.
 *
 * 14.11.91  -	Testing seems to have gone well. Added some
 *		correction-code, and changed some functions.
 *
 * 15.11.91  -  More correction code. Hopefully it notices most
 *		cases now, and tries to do something about them.
 *
 * 16.11.91  -  More corrections (thanks to Mika Jalava). Most
 *		things seem to work now. Yeah, sure.
 *
 * 14.07.92  -  Modifications by Remy Card (card@masi.ibp.fr)
 *		Used the fsck source to create efsck.c
 *
 * 19.07.92  -	Many bugs fixed [the first release was full of bugs :-(]
 *		- init_zone_bitmap() and init_inode_bitmap() now check that
 *		  the zones/inodes free lists are not corrupted. If they are,
 *		  all zones/inodes are marked as unused and are corrected (I
 *		  hope) by check_counts).
 *		- init_zone_bitmap() and init_inode_bitmap() now check for
 *		  cycles in the free lists. Before, in case of a cycle, efsck
 *		  entered an infinite loop ...
 *		- salvage_inode_freelist() was wrong. It used inode-1 as index
 *		  instead of inode.
 *		- salvage_zone_freelist() and salvage_inode_freelist() now
 *		  try to keep the same first free zone/inode so there should
 *		  be less problems when checking a mounted file system (for
 *		  a mounted ext file system, the first free zone/inode numbers
 *		  are stored in the copy of the super block in memory and are
 *		  rewritten to disk on sync() and umount(), modifying the
 *		  first free zone/inode can lead to some inconsistency).
 *		- corrected the file name printing in get_inode().
 *		- corrected the "inode not used" test which was never true ...
 *		- added size checks : compare the size of each inode with the
 *		  number of blocks allocated for the inode.
 *		Remaining problem :
 *		- I think that there is some more work to do to correct the
 *		  free lists. Currently, efsck salvages (rebuilds) them and it
 *		  is a very primitive way to handle errors. Perhaps, it should
 *		  act in a more clever way by adding or removing zones/inodes
 *		  from the free lists. I don't know if it is very important ...
 *
 *  21.07.92  - Corrected check_sizes():
 *		- to count the last allocated block and	NOT the allocated
 *		  blocks count, so it now counts holes,
 *		- fixed the bug causing the message 'incorrect size 0
 *		  (counted = 0)'.
 *
 *  26.07.92  - efsck now understands triple indirection
 *
 *  11.08.92  -	Changes by Wayne Davison (davison@borland.com)
 *		- The badblock inode (2 on my system) is always read in and
 *		  the resulting bitmap is used to ensure that files don't use
 *		  these blocks in their data.  A minor tweak keeps efsck from
 *		  complaining about the inode not being referenced.
 *		- The -t option has been added to perform a read test of the
 *		  disk looking for new bad blocks.  Using -t gives efsck write-
 *		  permission for just the bad block changes (which might
 *		  include a rewrite of the free lists, inodes & root).  If no
 *		  repair options were specified and a file uses a bad block
 *		  only a warning is generated.  A block must be either unused
 *		  or "repaired" (dropped from the file) for it to be added to
 *		  the badblock inode.
 *		- Minor changes were made to the buffers to reduce their
 *		  number.
 *		- All the pluralizing outputs were changed so that 0's come out
 *		  plural (e.g. 0 bad blocks, not 0 bad block).
 *		- Fixed an off-by-one error in the INODE code.
 *		- Fixed a bug in the directory loop where it could infinite
 *		  loop
 *		  if we're checking bogus directory data (with a zero rec_len).
 *		- Fixed a bug in the size counting when dealing with the
 *		  triple-indirect zone list.
 *		- Twiddled the verbose code to use the counts stored in the
 *		  super block (which were just verified) rather than counting
 *		  them again.
 *		- The verbose code outputs the number of bad blocks on the
 *		  disk.
 *		- I removed 'm' from the usage list of options.
 *
 *  12.08.92  - Corrected (again) check_sizes() : it now complains only
 *		when the size of a file is less than the last allocated
 *		block.
 *
 *  13.08.92  - Changes to support the existing .badblocks file
 *	      - Added the -S argument which allows salvaging the free lists
 *
 *  16.08.92  - Added some sanity checks on the directory entries
 *	      - corrected the test on rec_len to ensure that it is greater than
 *		or equal to 8 (was 16 before) because there may be 8 bytes
 *		directory entries (the last unused entry in a block can be 8
 *		bytes long)
 *	      - Use getopt() to get arguments
 *
 *  25.08.92  - Changes by Wayne Davison to create a "new" bad blocks inode
 *		from the old ".badblocks" file
 *
 *  27.08.92  - Fixed two stupid errors :
 *		- when the free lists were salvaged, the tables were not
 *		  written back to the disk
 *		- when the free inodes count and/or the free blocks count
 *		  were modified, the tables were not written back to the
 *		  disk
 *
 *  28.08.92  - Corrected init_zone_bitmap() and init_inode_bitmap() : sanity
 *		checks on the free zones/inodes are done BEFORE accessing the
 *		bitmaps.  When it was done after accessing the bitmaps, efsck
 *		could dump core when the list(s) was(were) corrupted.
 *
 *  03.09.92  - Corrected check_sizes() to ignore special files
 *
 *  01.11.92  - return a status code
 *		try to correct bad directory entries by	truncating the
 *		directory
 *
 *  15.11.92  - display ZONES and FIRSTZONE when invalid blocks are detected
 *		check inodes mode
 *		check links to directories (enable detection of cycles)
 *		try to correct bad directory entries in a smarter way
 *
 *  29.11.92  - efsck now understands sockets (Thanks to Juergen Schoenwaeldner
 *		<schoenw@ibr.cs.tu-bs.de>)
 *
 *  06.12.92  - added the '-b filename' option which reads a bad block list
 *		from the file
 *
 *  12.12.92  - corrected a minor bug in the free blocks list initialization
 *		(Thanks to Stephen Tweedie <sct@dcs.ed.ac.uk>)
 *
 *  01.01.93  - adapted to gcc 2.3.3 / libc 4.2
 *
 *  10.01.93  - OK.  I just deleted the previous e2fsck.c (after making a
 *		backup :) and I converted efsck alpha 12 to e2fsck. 
 *		Then, I have added the new functionalities that I had added
 *		to the old e2fsck.
 *
 *  16.01.93  -	added i_blocks check
 *		added free blocks and inodes count checks for each group
 *
 *  17.01.93  -	added block nr checks in check_block_nr()
 *
 *  21.01.93  - corrected an off by one error.  Thanks to Thomas Winder
 *		<tom@vlsivie.tuwien.ac.at> and to Drew Eckhardt
 *		<drew@nag.cs.Colorado.Edu>
 *
 *  24.01.93  - adapted to the super block and descriptors backup
 *
 *  28.02.93  - corrected bugs in set_lost_dir_ent
 *
 *  01.03.93  - added the new used_dirs_count field check
 *
 *  13.03.93  - corrected a bug in set_lost_dir_ent which caused bad
 *		   directory entries
 *		count the fast symbolic links
 *		make -c an alias to -t
 *		change the bad blocks test
 *
 * 14.03.93   -	display the bad blocks list
 *
 * 18.03.93   - check lost+found contents in set_lost_dir_ent
 *
 * 27.03.93   - finally uses *much* less memory than before (but e2fsck
 *		is now slower :-( )
 *
 * 14.04.93   - lots of fixes
 *
 * 15.04.93   - optionnaly checks for mounted devices
 *		tries to print the file name for most error messages
 *		check for bad characters in file names
 *
 * 18.05.93   - removed old code
 *		changed the debug code
 *
 * I've had no time to add comments - hopefully the function names
 * are comments enough. As with all file system checkers, this assumes
 * the file system is quiescent - don't use it on a mounted device
 * unless you can be sure nobody is writing to it (and remember that the
 * kernel can write to it when it searches for files).
 *
 * Usage: e2fsck [-acdflmrstv] [-b file] device
 *	-a for automatic repairs
 *	-c for testing bad blocks on the fs
 *	-b for reading the bad blocks list from a file
 *	-d for debugging this program
 *	-f for checking the fs even if it is marked valid
 *	-l for a listing of all the filenames
 *	-m for checking for mounted file system
 *	-r for repairs (interactive)
 *	-s for super-block info
 *	-t for testing bad blocks on the fs
 *	-v for verbose (tells how many files)
 *
 * The device may be a block device or a image of one, but this isn't
 * enforced (but it's not much fun on a character device :-). 
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include <signal.h>
#include <termios.h>
#include <time.h>
#include <sys/stat.h>
#include <getopt.h>
#include <mntent.h>
#include "malloc.h"

#include <linux/ext2_fs.h>

#include "bitops.h"
#include "e2fsprogs.h"
#include "ckfunc.h"
#include "ckvar.h"

char blkbuf[EXT2_MAX_BLOCK_SIZE * TEST_BUFFER_BLOCKS];

#define UPPER(size,n)		((size + ((n) - 1)) / (n))
#define INODE_SIZE		(sizeof (struct ext2_inode))
#define INODE_BLOCKS		UPPER(INODES, EXT2_INODES_PER_BLOCK (Super))
#define INODE_BUFFER_SIZE	(INODE_BLOCKS * EXT2_BLOCK_SIZE (Super))

extern int isatty(int);

static struct termios termios, tmp_termios;

static char * program_name = "e2fsck";
static char * listfile = NULL;
static char * device_name = NULL;
static int dev;

/* Command line options */
int repair = 0;
int automatic = 0;
int verbose = 0;
int list = 0;
int show = 0;
int debug = 0;
int test_disk = 0;
int force = 0;
int check_mounted = 0;

int no_bad_inode = 0;
int no_lpf = 0;
int lpf_corrupted = 0;

/* Files counts */
int directory = 0;
int regular = 0;
int blockdev = 0;
int chardev = 0;
int links = 0;
int symlinks = 0;
int fast_symlinks = 0;
int fifo = 0;
int total = 0;
int badblocks = 0;
int sockets = 0;

int changed = 0; /* flags if the filesystem has been changed */

int retval = 0;

int inode_blocks_per_group;
int block_size;
int addr_per_block;

long sb = 1;	/* super block location */

/* File-name data */
#define MAX_DEPTH 50
int name_depth = 0;   /* -1 when we're parsing the bad block list */
char name_list[MAX_DEPTH][EXT2_NAME_LEN + 1];

static char super_block_buffer[EXT2_MIN_BLOCK_SIZE];

struct ext2_super_block * Super = (struct ext2_super_block *) super_block_buffer;

char * bad_map = NULL;
char * block_map = NULL;
char * inode_map = NULL;

unsigned short * inode_count = NULL;
unsigned short * block_count = NULL;

unsigned long group_desc_count;
unsigned long group_desc_size;
unsigned long desc_blocks;
struct ext2_group_desc * group_desc = NULL;

unsigned long last_block_read;

static void recursive_check (unsigned long ino);

static volatile void fatal_error (const char * fmt_string, int errcode)
{
	fflush (stdout);
	fprintf (stderr, fmt_string, program_name, device_name);
	if (repair && !automatic)
		tcsetattr (0, TCSANOW, &termios);
	exit (retval | errcode);
}

#define usage()		fatal_error("Usage: %s [-acdflrstv] [-B super_block]" \
				    "[-b filename] /dev/name\n", EXIT_USAGE)

volatile void die (const char * str, int errcode)
{
	strcpy (blkbuf, "%s: ");
	strcat (blkbuf, str);
	strcat (blkbuf, "\n");
	fatal_error (blkbuf, errcode);
}

/*
 * This simply goes through the file-name data and prints out the
 * current file.
 */
void print_current_name (void)
{
	int i = 0;

	while (i < name_depth)
		printf ("/%s", name_list[i++]);
	if (!name_depth)
		printf ("/");
	else if (name_depth == -1)
		printf ("{Bad-Block-List}");
	else if (name_depth == -2)
		printf ("{Free-Block-Map}");
}

int ask (const char * string, int def)
{
	int c;

	if (!repair)
	{
		printf ("%s? no\n", string);
		retval |= EXIT_UNCORRECTED;
		return 0;
	}
	if (automatic)
	{
		printf ("%s? %s\n", string, def ? "yes" : "no");
		return def;
	}
	printf (def ? "%s (y/n)? " : "%s (n/y)? ", string);
	for (;;)
	{
		fflush (stdout);
		if ((c = getchar()) == EOF)
			break;
		c = toupper(c);
		if (c == 'Y')
		{
			def = 1;
			break;
		}
		else if (c == 'N')
		{
			def = 0;
			break;
		}
		else if (c == ' ' || c == '\n')
			break;
	}
	if (def)
	{
		retval |= EXIT_DESTRUCT;
		printf ("yes\n");
	}
	else
	{
		retval |= EXIT_UNCORRECTED;
		printf ("no\n");
	}
	return def;
}

/*
 * check_block_nr checks to see that *nr is a valid block nr. If it
 * isn't, it will possibly be repaired. Check_block_nr returns != 0
 * if it changed the nr.
 */
int check_block_nr (unsigned long * nr)
{
	unsigned long group;

	DEBUG(("DEBUG: check_block_nr (%x)\n", nr));
	if (!*nr)
		return 0;
	if (*nr < FIRSTBLOCK)
		printf ("Block nr %lu < FIRSTBLOCK (%lu) in file `",
			*nr, FIRSTBLOCK);
	else if (*nr >= BLOCKS)
		printf ("Block nr %lu > BLOCKS (%lu) in file `", *nr, BLOCKS);
	else
	{
		group = (*nr - FIRSTBLOCK) / BLOCKSPERGROUP;
		if (*nr == group_desc[group].bg_block_bitmap)
			printf ("Block nr %lu is the block bitmap"
				" of group %lu in file `", *nr, group);
		else if (*nr == group_desc[group].bg_inode_bitmap)
			printf ("Block nr %lu is the inode bitmap"
				" of group %lu in file `", *nr, group);
		else if (*nr >= group_desc[group].bg_inode_table &&
			 *nr < (group_desc[group].bg_inode_table
				+ inode_blocks_per_group))
			printf ("Block nr %lu is in the inode table"
				" of group %lu in file `", *nr, group);
		else
			return 0;
	}
	print_current_name ();
	printf ("'. ");
	if (ask ("Remove block", 1))
	{
		*nr = 0;
		changed = 1;
		return 1;
	}
	return 0;
}

static void check_root (void)
{
	struct ext2_inode inode;

	DEBUG(("DEBUG: check_root()\n"));
	read_inode (dev, EXT2_ROOT_INO, &inode);
	if (!S_ISDIR (inode.i_mode))
		die ("root inode isn't a directory", EXIT_ERROR);
}

static void check_blocks (unsigned long ino)
{
	struct ext2_inode inode;
	int i;
	int inode_changed = 0;

	DEBUG(("DEBUG: check_blocks(%d)\n", ino));
	if (!ino || ino > INODES)
		return;
	if (inode_count[ino] > 1)  /* have we counted this file already? */
		return;
	read_inode (dev, ino, &inode);
	if (!S_ISDIR (inode.i_mode) && !S_ISREG (inode.i_mode) &&
	    !S_ISLNK (inode.i_mode))
		return;
	if (S_ISLNK (inode.i_mode) && inode.i_blocks == 0 &&
	    inode.i_size < EXT2_N_BLOCKS * sizeof (unsigned long))
		return;
	for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
		inode_changed |= add_block (dev, inode.i_block + i, 0);
	inode_changed |= add_block_ind (dev, inode.i_block + EXT2_IND_BLOCK, 0);
	inode_changed |= add_block_dind (dev, inode.i_block + EXT2_DIND_BLOCK, 0);
	inode_changed |= add_block_tind (dev, inode.i_block + EXT2_TIND_BLOCK, 0);
	if (inode_changed)
		write_inode (dev, ino, &inode);
}

unsigned short check_file (struct ext2_inode * dir, int * dir_changed,
			   unsigned int offset)
{
	struct ext2_inode inode;
	unsigned long ino;
	unsigned short rec_len;
	unsigned short name_len;
	char * name;
	char * error_msg = NULL;
	int original_depth = name_depth;
	int replace_slash = 0;
	int name_changed = 0;
	int i;

	DEBUG(("DEBUG: check_file(%d,%d)\n", (int)dir, offset));
	*dir_changed = mapped_read_block (dev, dir, offset / block_size, blkbuf);
	changed |= *dir_changed;
	name = blkbuf + (offset % block_size) + 8;
	ino = * (unsigned long *) (name - 8);
	rec_len = * (unsigned short *) (name - 4);
	name_len = * (unsigned short *) (name - 2);
	if (rec_len < EXT2_DIR_REC_LEN(1))
		error_msg = "rec_len is smaller than minimal";
	else if (rec_len % 4 != 0)
		error_msg = "rec_len % 4 != 0";
	else if (rec_len < EXT2_DIR_REC_LEN(name_len))
		error_msg = "rec_len is too small for name_len";
	else if (((offset % block_size) + rec_len) > block_size)
		error_msg = "directory entry across blocks";

	if (error_msg != NULL)
	{
		printf ("Bad directory entry in `");
		print_current_name ();
		printf ("' at offset %d.\n", offset);
		printf ("%s ", error_msg);
		if (((offset / 1024) + 1) * 1024 > dir->i_size)
			rec_len = dir->i_size - offset;
		else
			rec_len = block_size - (offset % block_size);
		if (ask ("Delete entry", 1))
		{
			* (unsigned long *) (name - 8) = 0;
			* (unsigned short *) (name - 4) = rec_len;
			* (unsigned short *) (name - 2) = rec_len - 8;
			write_block (dev, last_block_read, blkbuf);
			changed = 1;
		}
		else
			printf ("Skipping to next block\n");
		return rec_len;			
	}
	if (ino && strcmp (".", name) && strcmp ("..", name))
	{
		if (name_depth < MAX_DEPTH)
		{
			strncpy (name_list[name_depth], name, name_len);
			name_list[name_depth][name_len] = '\0';
		}
		name_depth++;	
	}
	if (!no_bad_inode && ino == EXT2_BAD_INO)
	{
		printf ("File `");
		print_current_name ();
		printf ("' consists of the badblock list. ");
		if (ask ("Delete it", 1))
		{
			/* Zero the inode and write out the dir block */
			* (unsigned long *) (name - 8) = 0;
			write_block (dev, last_block_read, blkbuf);
			changed = 1;
		}
		goto exit;
	}
	for (i = 0; i < name_len; i++)
	{
		if (name[i] == '/' || name[i] == '\0')
		{
			if (!replace_slash)
			{
				printf ("Bad file name %s (contains '/' or "
					" null) in directory `", name);
				name_depth--;
				print_current_name ();
				name_depth++;
				printf ("'. ");
			}
			if (replace_slash ||
			    ask ("Replace '/' or null by '.'", 1))
			{
				replace_slash = 1;
				name_changed = 1;
				name[i] = '.';
			}
		}
	}
	if (name_changed)
		write_block (dev, last_block_read, blkbuf);
	inode = get_inode (dev, ino);
	if (!offset)
		if (inode.i_mode == 0 || strcmp (".", name) != 0)
		{
			print_current_name ();
			printf (": bad directory: '.' isn't first\n");
		}
		else
			goto exit;
	if (offset == 12)
		if (inode.i_mode == 0 || strcmp ("..", name) != 0)
		{
			print_current_name ();
			printf (": bad directory: '..' isn't second\n");
		}
		else
			goto exit;
	if (ino && inode.i_mode == 0)
	{
		printf ("bad inode in directory `");
		name_depth--;
		print_current_name ();
		name_depth++;
		printf ("' at offset %d ", offset);
		if (ask ("Delete directory entry", 1))
		{
			* (unsigned long *) (name - 8) = 0;
			write_block (dev, last_block_read, blkbuf);
			changed = 1;
		}
	}
	if (inode.i_mode == 0)
		goto exit;
	if (list)
	{
		if (verbose)
			printf ("%6ld %07o ", ino, inode.i_mode);
		print_current_name ();
		if (S_ISDIR (inode.i_mode))
			printf (":\n");
		else
			printf ("\n");
	}
	check_blocks (ino);
#if 0	/* XXX - This check destroys i_flags implemented in ext2 fs 0.4 :-( */
	if (inode.i_flags)
	{
		print_current_name ();
		printf (" (inode %ld) has i_flags (currently not used) set. ",
			ino);
		if (ask ("Clear", 1))
		{
			inode.i_flags = 0;
			write_inode (dev, ino, &inode);
		}
	}
#endif
	if (inode.i_faddr)
	{
		print_current_name ();
		printf (" (inode %ld) has i_faddr (currently not used) set. ",
			ino);
		if (ask ("Clear", 1))
		{
			inode.i_faddr = 0;
			write_inode (dev, ino, &inode);
		}
	}
	if (inode.i_frag)
	{
		print_current_name ();
		printf (" (inode %ld) has i_frag (currently not used) set. ",
			ino);
		if (ask ("Clear", 1))
		{
			inode.i_frag = 0;
			write_inode (dev, ino, &inode);
		}
	}
	if (inode.i_fsize)
	{
		print_current_name ();
		printf (" (inode %ld) has i_fsize (currently not used) set. ",
			ino);
		if (ask ("Clear", 1))
		{
			inode.i_fsize = 0;
			write_inode (dev, ino, &inode);
		}
	}
	if (inode.i_file_acl)
	{
		print_current_name ();
		printf (" (inode %ld) has i_file_acl (currently not used) set. ",
			ino);
		if (ask ("Clear", 1))
		{
			inode.i_file_acl = 0;
			write_inode (dev, ino, &inode);
		}
	}
	if (inode.i_dir_acl)
	{
		print_current_name ();
		printf (" (inode %lu) has i_dir_acl"
			" (currently not used) set. ", ino);
		if (ask ("Clear", 1))
		{
			inode.i_dir_acl = 0;
			write_inode (dev, ino, &inode);
		}
	}
	if (inode.i_mode != 0 && S_ISDIR (inode.i_mode))
	{
		if (inode_count[ino] > 1)
		{
			printf ("link to a directory `");
			print_current_name ();
			printf ("'.\n");
			if (ask ("Delete directory entry", 1))
			{
				* (unsigned long *) (name - 8) = 0;
				write_block (dev, last_block_read, blkbuf);
				changed = 1;
				goto exit;
			}
		}
		recursive_check (ino);
	}
exit:
	name_depth = original_depth;
	return rec_len;
}

static void recursive_check (unsigned long ino)
{
	struct ext2_inode dir;
	unsigned long offset;
	unsigned short rec_len;
	int dir_changed = 0;

	DEBUG(("DEBUG: recursive_check(%d)\n", ino));
	read_inode (dev, ino, &dir);
	if (!S_ISDIR (dir.i_mode))
		die ("error in recursive_check: dir is not a directory", EXIT_ERROR);
	if (dir.i_size < EXT2_DIR_REC_LEN (1) + EXT2_DIR_REC_LEN (2))
	{
		print_current_name ();
		printf (": bad directory: size too short\n");
		return;
	}
	for (offset = 0; offset < dir.i_size; offset += rec_len)
	{
		rec_len = check_file (&dir, &dir_changed, offset);
		if (dir_changed)
			write_inode (dev, ino, &dir);
		if (rec_len < 8)
		{
			print_current_name ();
			printf (": bad directory: rec_len(%d) too short. ", rec_len);
			if (ask ("Truncate directory", 1))
			{
				dir.i_size = offset;
				truncate_file (dev, &dir);
				write_inode (dev, ino, &dir);
			}	    
			return;
		}
	}
}

static unsigned long get_free_block (void)
{
	static unsigned long blk = 0;

	if (!blk)
		blk = FIRSTBLOCK;
	while (blk < BLOCKS && block_in_use (blk))
		blk++;
	if (blk >= BLOCKS)
		die ("not enough good blocks", EXIT_ERROR);
	mark_block (blk);
	block_count[blk]++;
	FREEBLOCKSCOUNT--;
	return blk++;
}

static unsigned long next_new_bad (unsigned long block)
{
	if (!block)
		block = FIRSTBLOCK - 1;
	while (++block < BLOCKS)
		if (block_is_bad (block) && !block_count[block])
		{
			block_count[block]++;
			return block;
		}
	return 0;
}

static int next_block (unsigned long * znr, void * blk, unsigned long * pnr)
{
	if (*znr)
	{
		*pnr = *znr;
		read_block (dev, znr, blk);
		return 0;
	}
	*pnr = *znr = get_free_block ();
	memset (blk, 0, block_size);
	return 1;
}

static void update_bad_block (void)
{
	struct ext2_inode inode;
	int i, j, k;
	unsigned long block;
	unsigned long ind, dind, tind;
	int ind_dirty = 0, dind_dirty = 0, tind_dirty = 0;
	unsigned long * ind_block  = (unsigned long *)(blkbuf + block_size);
	unsigned long * dind_block = (unsigned long *)(blkbuf + block_size * 2);
	unsigned long * tind_block = (unsigned long *)(blkbuf + block_size * 3);

	read_inode (dev, EXT2_BAD_INO, &inode);
	inode.i_atime = inode.i_ctime = inode.i_mtime = time(NULL);
	inode.i_size = badblocks * block_size;
	if (!badblocks)
		return;
	if (!(block = next_new_bad (0)))
		return;
	for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
	{
		if (inode.i_block[i])
			continue;
		inode.i_block[i] = block;
		if (!(block = next_new_bad (block)))
			goto end_bad;
	}
	ind_dirty = next_block (inode.i_block + EXT2_IND_BLOCK, ind_block,
				&ind);
	for (i = 0; i < addr_per_block; i++)
	{
		if (ind_block[i])
			continue;
		ind_block[i] = block;
		ind_dirty = 1;
		if (!(block = next_new_bad (block)))
			goto end_bad;
	}
	dind_dirty = next_block (inode.i_block + EXT2_DIND_BLOCK,
				 dind_block, &dind);
	for (i = 0; i < addr_per_block; i++)
	{
		if (ind_dirty)
		{
			write_block (dev, ind, (char *) ind_block);
			ind_dirty = 0;
		}
		dind_dirty |= next_block (dind_block + i, ind_block, &ind);
		for (j = 0; j < addr_per_block; j++)
		{
			if (ind_block[j])
				continue;
			ind_block[j] = block;
			ind_dirty = 1;
			if (!(block = next_new_bad (block)))
				goto end_bad;
		}
	}
	tind_dirty = next_block (inode.i_block + EXT2_TIND_BLOCK,
				 tind_block, &tind);
	for (i = 0; i < addr_per_block; i++)
	{
		if (dind_dirty)
		{
			write_block (dev, dind, (char *) dind_block);
			dind_dirty = 0;
		}
		tind_dirty |= next_block (tind_block + i, dind_block, &dind);
		for (j = 0; j < addr_per_block; j++)
		{
			if (ind_dirty)
			{
				write_block (dev, ind, (char *) ind_block);
				ind_dirty = 0;
			}
			dind_dirty |= next_block(dind_block + j, ind_block,
						 &ind);
			for (k = 0; k < addr_per_block; k++)
			{
				if (ind_block[k])
					continue;
				ind_block[k] = block;
				ind_dirty = 1;
				if (!(block = next_new_bad (block)))
					goto end_bad;
			}
		}
	}
	printf ("Warning: there are too many bad blocks\n");
end_bad:
	if (ind_dirty)
		write_block (dev, ind, (char *) ind_block);
	if (dind_dirty)
		write_block (dev, dind, (char *) dind_block);
	if (tind_dirty)
		write_block (dev, tind, (char *) tind_block);
	write_inode (dev, EXT2_BAD_INO, &inode);
}

static void check_counts (void)
{
	unsigned long i, j;
	int gf;
	int dc;
	int group;
	unsigned long blocks;
	unsigned long inodes;
	unsigned long free;
	struct ext2_inode inode;
	
	DEBUG(("DEBUG: check_counts()\n"));
	if (verbose)
		printf ("Checking inodes counts\n");
	for (i = 1; i <= INODES; i++)
	{
		if (i != EXT2_ROOT_INO && i < EXT2_FIRST_INO)
		{
			if (!inode_in_use (i))
			{
				printf ("Reserved inode %lu marked"
					" not in use. ", i);
				if (ask ("Mark in use", 1))
					mark_inode (i);
			}
			continue;
		}
		read_inode (dev, i, &inode);
		if (!inode_count[i] && inode_in_use (i))
		{
			printf ("Inode %lu not used, "
				"not counted in the bitmap. ", i);
			if (!inode.i_dtime && inode.i_mode &&
			    inode.i_links_count && !no_lpf && !lpf_corrupted)
			{
				if (ask ("Connect to lost+found", 1))
					set_lost_dir_ent(dev, i, inode.i_mode);
			}
			else
			{
				if (lpf_corrupted)
					printf ("(lost+found is corrupted). ");
				if (no_lpf)
					printf ("(no directory lost+found). ");
				if (ask ("Mark not in use", 1))
					unmark_inode (i);
			}
		}
		else if (inode_count[i] && !inode_in_use (i))
		{
			printf ("Inode %lu used, not counted in the bitmap. ",
				i);
			if (ask ("Mark in use", 1))
				mark_inode (i);
		}
		if (!inode_in_use (i))
		{
			if (!inode.i_dtime && inode.i_mode)
			{
				printf ("Inode %lu not used with dtime null. ",
					i);
				if (ask ("Set dtime", 1))
				{
					inode.i_dtime = time (NULL);
					write_inode (dev, i, &inode);
				}
			}
			if (inode.i_links_count)
			{
				printf ("Inode %lu not used with links_count"
					" not null. ", i);
				if (ask ("Repair", 1))
				{
					inode.i_links_count = 0;
					write_inode (dev, i, &inode);
				}
			}
		}
		else if (inode.i_mode && inode.i_dtime)
		{
			printf ("Inode %lu used with dtime set. ", i);
			if (ask ("Set dtime to zero", 1))
			{
				inode.i_dtime = 0;
				write_inode (dev, i, &inode);
			}
		}
		if (inode_in_use (i) && inode.i_links_count != inode_count[i])
		{
			printf ("Inode %lu, i_nlinks=%d, counted=%d. ",
				i, inode.i_links_count, inode_count[i]);
			if (ask ("Set i_nlinks to count", 1))
			{
				inode.i_links_count = inode_count[i];
				write_inode (dev, i, &inode);
			}
		}
	}
	for (i = 1, free = 0, gf = 0, dc = 0, group = 0, inodes = 0;
	     i <= INODES; i++)
	{
		if (!inode_in_use (i))
		{
			gf++;
			free++;
		}
		else
		{
			read_inode (dev, i, &inode);
			if (S_ISDIR(inode.i_mode))
				dc++;
		}
		inodes++;
		if (inodes == INODESPERGROUP || i == INODES)
		{
			if (gf != group_desc[group].bg_free_inodes_count)
			{
				printf ("Free inodes count wrong for group %d (%d, counted=%d) ",
					group, group_desc[group].bg_free_inodes_count,
					gf);
				if (ask ("Repair", 1))
				{
					group_desc[group].bg_free_inodes_count = gf;
					changed = 1;
				}
			}
			if (dc != group_desc[group].bg_used_dirs_count)
			{
				printf ("Directories count wrong for group %d (%d, counted=%d) ",
					group, group_desc[group].bg_used_dirs_count,
					dc);
				if (ask ("Repair", 1))
				{
					group_desc[group].bg_used_dirs_count = dc;
					changed = 1;
				}
			}
			group++;
			inodes = 0;
			gf = 0;
			dc = 0;
		}
	}
	if (free != FREEINODESCOUNT)
	{
		printf ("Free inodes count wrong (%lu, counted=%lu). ",
			FREEINODESCOUNT, free);
		if (ask ("Repair", 1))
		{
			FREEINODESCOUNT = free;
			changed = 1;
		}
	}

	if (verbose)
		printf ("Checking blocks count\n");
	for (i = FIRSTBLOCK; i < BLOCKS; i++)
	{
		if (block_in_use (i) == block_count[i])
			continue;

		if (!block_count[i] && block_in_use (i))
		{
			if (block_is_bad (i))
				continue;
			printf ("Block %lu: marked in use, no file uses it. ",
				i);
			if (ask ("Unmark", 1))
				unmark_block (i);
			continue;
		}
		if (block_count[i] && !block_in_use (i))
		{
			printf ("Block %lu: marked not in use, counted=%u\n",
				i, (unsigned)block_count[i]);
			if (ask ("Mark", 1))
				mark_block (i);
		}
	}
	name_depth = -2;
	for (i = FIRSTBLOCK; i < BLOCKS; i++)
		if (!block_in_use (i))
		{
			j = i;
			if (check_block_nr (&j))
				mark_block (i);
		}
	for (i = FIRSTBLOCK, free = 0, gf = 0, group = 0, blocks = 0;
	     i < BLOCKS; i++)
	{
		if (!block_in_use (i))
		{
			gf++;
			free++;
		}
		blocks++;
		if (blocks == BLOCKSPERGROUP || i == (BLOCKS-1))
		{
			if (gf != group_desc[group].bg_free_blocks_count)
			{
				printf ("Free blocks count wrong for group %d (%d, counted=%d) ",
					group, group_desc[group].bg_free_blocks_count,
					gf);
				if (ask ("Repair", 1))
				{
					group_desc[group].bg_free_blocks_count = gf;
					changed = 1;
				}
			}
			group++;
			blocks = 0;
			gf = 0;
		}
	}
	if (free != FREEBLOCKSCOUNT)
	{
		printf ("Free blocks count wrong (%lu, counted=%lu). ",
			FREEBLOCKSCOUNT, free);
		if (ask ("Repair", 1))
		{
			FREEBLOCKSCOUNT = free;
			changed = 1;
		}
	}
}

static void check_bitmap_padding (void)
{
	int i,j,k;
	char block[block_size];

	DEBUG(("DEBUG: check_bitmap_padding()\n"));
	for (i = 0; i < group_desc_count; i++)
	{
		if (lseek (dev, group_desc[i].bg_block_bitmap * block_size,
			   SEEK_SET) !=
			group_desc[i].bg_block_bitmap * block_size)
			die ("seek failed in check_bitmap_padding", 
			     EXIT_ERROR);
		if (read (dev, block, block_size) != block_size)
			die ("read failed in check_bitmap_padding", 
			     EXIT_ERROR);
		j = BLOCKSPERGROUP;
		if ((BLOCKSPERGROUP*i + j) > (BLOCKS-FIRSTBLOCK))
			j = (BLOCKS-FIRSTBLOCK) - (BLOCKSPERGROUP*i);
		DEBUG(("DEBUG: Checking padding of block bitmap "
		       "for group %d offset %d.\n",
		       i, j));
		for (k=j; k < (block_size * 8); k++)
		{
			if (!testbit(block, k))
			{
				printf ("Padding bits unset in block bitmap "
					"for group %d. ", i);
				if (ask ("Set", 1))
				{
					changed = 1;
					for (k=j; k < (block_size * 8); k++)
						setbit(block, k);
					if (lseek (dev, group_desc[i].bg_block_bitmap * block_size,
						   SEEK_SET) !=
					    group_desc[i].bg_block_bitmap * block_size)
						die ("seek failed in check_bitmap_padding", 
						     EXIT_ERROR);
					if (write (dev, block, block_size) != block_size)
						die ("read failed in check_bitmap_padding", 
						     EXIT_ERROR);
				}
				break;
			}
		}
	}
}

static void count_ind (unsigned long * znr, unsigned long * last,
		       unsigned long * count)
{
	char * block = blkbuf + block_size;
	unsigned long * ptr;
	int i;

	if (!*znr || *znr > BLOCKS)
		return;
	*count += block_size / 512;
	read_block (dev, znr, block);
	for (i = 0, ptr = (unsigned long *) block; i < addr_per_block; i++)
		if (ptr[i])
		{
			*last = i + 1;
			*count += block_size / 512;
		}
}

static void count_dind (unsigned long * znr, unsigned long * last,
			unsigned long * count)
{
	char * block = blkbuf + block_size * 2;
	unsigned long * ptr;
	unsigned long tmp;
	int i;

	if (!*znr || !znr > BLOCKS)
		return;
	*count += block_size / 512;
	read_block (dev, znr, block);
	for (i = 0, ptr = (unsigned long *) block; i < addr_per_block; i++)
	{
		tmp = 0;
		count_ind (ptr + i, &tmp, count);
		if (tmp)
			*last = tmp + i * addr_per_block;
	}
}

static void count_tind (unsigned long * znr, unsigned long * last,
			unsigned long * count)
{
	char * block = blkbuf + block_size * 3;
	unsigned long * ptr;
	unsigned long tmp;
	int i;

	if (!*znr || !znr > BLOCKS)
		return;
	*count += block_size / 512;
	read_block (dev, znr, block);
	for (i = 0, ptr = (unsigned long *) block; i < addr_per_block; i++)
	{
		tmp = 0;
		count_dind (ptr + i, &tmp, count);
		if (tmp)
			*last = tmp + i * addr_per_block * addr_per_block;
	}
}

static void check_sizes (void)
{
	int i;
	unsigned long ino;
	struct ext2_inode inode;
	unsigned long count;
	unsigned long last;
	unsigned long tmp;

	DEBUG(("DEBUG: check_sizes()\n"));
	if (verbose)
		printf ("Checking file sizes\n");
	for (ino = 1; ino <= INODES; ino++)
	{
		if (!inode_in_use (ino))
			continue;
		read_inode (dev, ino, &inode);
		if (!S_ISDIR (inode.i_mode) && !S_ISREG (inode.i_mode) &&
		    !S_ISLNK (inode.i_mode))
			continue;
		if (S_ISLNK (inode.i_mode) && inode.i_blocks == 0 &&
		    inode.i_size < EXT2_N_BLOCKS * sizeof (unsigned long))
			continue;
		count = 0;
		last = 0;
		for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
			if (inode.i_block[i])
			{
				last = i + 1;
				count += block_size / 512;
			}
		tmp = 0;
		count_ind (inode.i_block + EXT2_IND_BLOCK, &tmp, &count);
		if (tmp)
			last = tmp + EXT2_NDIR_BLOCKS;
		tmp = 0;
		count_dind (inode.i_block + EXT2_DIND_BLOCK, &tmp, &count);
		if (tmp)
			last = tmp + EXT2_NDIR_BLOCKS + addr_per_block;
		tmp = 0;
		count_tind (inode.i_block + EXT2_TIND_BLOCK, &tmp, &count);
		if (tmp)
			last = tmp + EXT2_NDIR_BLOCKS + addr_per_block +
				addr_per_block * addr_per_block;
		if ((inode.i_size || last) &&
			inode.i_size < (last - 1) * block_size)
		{
			printf ("Inode %lu, incorrect size"
				" %ld (counted = %ld). ",
				ino, inode.i_size, count * block_size);
			if (ask ("Set size to counted", 1))
			{
				inode.i_size = count * block_size;
				write_inode (dev, ino, &inode);
			}
		}
		if (inode.i_blocks != count)
		{
			printf ("Inode %lu, incorrect i_blocks"
				" %lu (counted=%lu) .",
				ino, inode.i_blocks, count);
			if (ask ("Set i_blocks to counted", 1))
			{
				inode.i_blocks = count;
				write_inode (dev, ino, &inode);
			}
		}
	}
}

static void check (void)
{
	int i, j;
	unsigned long block;

	DEBUG(("DEBUG: check()\n"));
	memset (inode_count, 0, (INODES + 1) * sizeof (*inode_count));
	memset (block_count, 0, BLOCKS * sizeof (*block_count));

	block = FIRSTBLOCK;
	for (i = 0; i < group_desc_count; i++)
	{
		block_count[group_desc[i].bg_block_bitmap] = 1;
		block_count[group_desc[i].bg_inode_bitmap] = 1;
		for (j = 0; j < inode_blocks_per_group; j++)
			block_count[group_desc[i].bg_inode_table + j] = 1;
		block_count[block] = 1;
		for (j = 0; j < desc_blocks; j++)
			block_count[block + j + 1] = 1;
		block += BLOCKSPERGROUP;
	}
	read_bad_blocks (dev);
	if (test_disk)
		test_blocks (dev);
	if (listfile)
		read_bad_blocks_from_file (listfile);
	if (verbose)
		display_bad_blocks ();
	check_blocks (EXT2_ROOT_INO);
	if (verbose)
		printf ("Reading directories and checking files\n");
	recursive_check (EXT2_ROOT_INO);
	if (test_disk || listfile)
		update_bad_block ();
	check_connectivity (dev);
	check_counts ();
	check_sizes ();
}

static void check_mount (const char * device_name)
{
	FILE * f;
	struct mntent * mnt;
	struct termios termios, tmp;
	int c;

	if ((f = setmntent (MOUNTED, "r")) == NULL)
		return;
	while ((mnt = getmntent (f)) != NULL)
		if (strcmp (device_name, mnt->mnt_fsname) == 0)
		{
			printf ("%s is mounted. ", device_name);
			if (isatty (0) && isatty (1))
			{
				tcgetattr (0, &termios);
				tmp = termios;
				tmp.c_lflag &= ~(ICANON | ECHO);
				tcsetattr (0, TCSANOW, &tmp);
				printf ("Confirm check (y/n) ");
				do
				{
					fflush (stdout);
					if ((c = getchar ()) == EOF)
					{
						c = 'N';
						break;
					}
					c = toupper (c);
				} while (c != 'N' && c != 'Y');
				tcsetattr (0, TCSANOW, &termios);
				if (c == 'N')
					printf ("no\n");
				else
					printf ("yes\n");
			}
			else
				c = 'N';
			if (c == 'N')
			{
				printf ("check aborted.\n");
				exit (EXIT_OK);
			}
			endmntent (f);
			return;
		}
	endmntent (f);
}
					
void main (int argc, char ** argv)
{
	int count;
	char c;
	char * tmp;

	fprintf (stderr, "e2fsck %s, %s for EXT2 FS %s, %s\n",
		 E2FSPROGS_VERSION, E2FSPROGS_DATE,
		 EXT2FS_VERSION, EXT2FS_DATE);
	if (argc && *argv)
		program_name = *argv;
	if (INODE_SIZE * (EXT2_MIN_BLOCK_SIZE / INODE_SIZE) !=
	    EXT2_MIN_BLOCK_SIZE)
		die ("bad inode size", EXIT_ERROR);
	while ((c = getopt (argc, argv, "B:ab:cdflmrstv")) != EOF)
		switch (c)
		{
			case 'B':
				sb = strtol (optarg, &tmp, 0);
				if (*tmp || sb < 1)
				{
					fprintf (stderr, "%s: bad super block location: %s\n",
						 program_name, optarg);
					usage ();
				}
				break;
			case 'a':
				automatic = 1;
				repair = 1;
				break;
			case 'b':
				listfile = optarg;
				break;
			case 'c':
			case 't':
				test_disk = 1;
				break;
			case 'd':
				debug = 1;
				break;
			case 'f':
				force = 1;
				break;
			case 'l':
				list = 1;
				break;
			case 'm':
				check_mounted = 1;
				break;
			case 'r':
				automatic = 0;
				repair = 1;
				break;
			case 's':
				show = 1;
				break;
			case 'v':
				verbose = 1;
				break;
			default:
				usage ();
		}
	if (optind != argc - 1)
		usage ();
	device_name = argv[optind];
	if (check_mounted)
		check_mount (device_name);
	if (repair && !automatic)
	{
		if (!isatty (0) || !isatty (1))
			die ("need terminal for interactive repairs", EXIT_USAGE);
		tcgetattr (0, &termios);
		tmp_termios = termios;
		tmp_termios.c_lflag &= ~(ICANON | ECHO);
		tcsetattr (0, TCSANOW, &tmp_termios);
	}
	dev = open (device_name, repair || test_disk || listfile ? O_RDWR : O_RDONLY);
	if (dev < 0)
		die("unable to open '%s'", EXIT_ERROR);
	for (count = 0; count < 3; count++)
		sync();
	if (verbose)
		printf ("Reading tables\n");
	if (read_tables (dev, sb))
	{
		if (verbose)
			printf ("Checking root directory\n");
		check_root ();
		check ();
		if (changed)
		{
			write_tables (dev);
			cache_flush (dev);
		}
		check_bitmap_padding();
		if (changed)
		{
			printf ("----------------------------\n"
				"FILE SYSTEM HAS BEEN CHANGED\n"
				"----------------------------\n");
			for (count = 0; count < 3; count++)
				sync ();
		}
		else if (repair)
			write_super_block (dev);
		if (verbose)
		{
			int free;

			free = FREEINODESCOUNT;
			printf ("\n%6lu inode%s used (%lu%%)\n",
				(INODES - free),
				((INODES - free) != 1) ? "s" : "",
				100 * (INODES - free) / INODES);
			free = FREEBLOCKSCOUNT;
			printf ("%6lu block%s used (%lu%%)\n"
				"%6d bad block%s\n",
				(BLOCKS - free),
				((BLOCKS - free) != 1) ? "s" : "",
				100 * (BLOCKS - free) / BLOCKS,
				badblocks,
				badblocks != 1 ? "s" : "");
			printf ("\n%6d regular file%s\n"
				"%6d director%s\n"
				"%6d character device file%s\n"
				"%6d block device file%s\n"
				"%6d fifo%s\n"
				"%6d link%s\n"
				"%6d symbolic link%s (%d fast symbolic link%s)\n"
				"%6d socket%s\n"
				"------\n"
				"%6d file%s\n",
				regular, (regular != 1) ? "s" : "",
				directory, (directory != 1) ? "ies" : "y",
				chardev, (chardev != 1) ? "s" : "",
				blockdev, (blockdev != 1) ? "s" : "",
				fifo, (fifo != 1) ? "s" : "",
				links - 2 * directory + 1,
				((links - 2 * directory + 1) != 1) ? "s" : "",
				symlinks, (symlinks != 1) ? "s" : "",
				fast_symlinks, (fast_symlinks != 1) ? "s" : "",
				sockets, (sockets != 1) ? "s" : "",
				total - 2 * directory + 1,
				((total - 2 * directory + 1) != 1) ? "s" : "");
		}
	}
	else
		printf ("%s is clean, no check.\n", device_name);
	close (dev);
	if (repair && !automatic)
		tcsetattr (0, TCSANOW, &termios);
#ifdef DBMALLOC
	malloc_chain_check(1);
#endif
	exit (retval);
}
