/*  
    This is the heart of cprint.  It converts a character into the proper 
sequence needed by TeX.  Most characters go through unchanged, but characters 
like:
    { } ~ < > ^ _ * \
need special handling.  Texconvert is just a humongous switch statement, 
taking (hopefully) every possibility into account.  This code is horrendous; 
read at your own risk :-) 

Copyright (C) 1992  Tim Nali

    Cprint is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */


#include <stdio.h>
#include <curses.h>

extern FILE *infile, *outfile;

/* INCODE  - true if texconvert is in the body of a function declaration. Used
           - to control applying bodytype style.
COMMENT    - true if texconvert is in a comment.  Used to 
           - control applying commenttype style.
FIRSTQUOTE - true if texconvert has proccessed a ` " ' and is still in the 
           - same line.
NEWLINE    - true if texconvert has encountered a new line.  This stays true 
           - only for the next character (next time texconvert is called)
INQUOTE    - true if texconvert is in a string ( " string " ). A newline will 
           - reset inquote.
POUND      - true if a pound sign is encountered.  A newline will reset it.
SINGLEQUOTE- true if <'> is encountered.  stays true only for next read.
braces     - keeps track of how far nested in the code texconvert is. 
EOFFLAG    - true if EOF has been encountered.
TTYPE      - command line argument. True if specified. When this is specified,
           - different conversions are used to accomodate fonts like cmtt10.
COMMENTMYSTAR - command line argument.  true if specified.  Uses \mystar 
              - instead of plain asterisk with commetn delimiters.
MYSTAR     - command line argument.  true if specified.  Uses \mystar instead 
           - of asterisks in all places except comment delimiters.
DELINSTYLE - command line argument.  true if specified.  Makes the comment 
           - delimiters themselves in commenttype style.
POUNDSTYLE - command line argument.  true if specified. Applies poundtype 
           - style to lines starting with '#' */

int INCODE=FALSE, COMMENT=FALSE, FIRSTQUOTE=FALSE, NEWLINE=TRUE;
int INQUOTE=FALSE, POUND=FALSE, SINGLEQUOTE=FALSE, braces=0;
extern int EOFFLAG, TTYPE, COMMENTMYSTAR, MYSTAR, DELINSTYLE, POUNDSTYLE, ASCII, PARINDENT;

/* Tab needs to be converted to spaces.  one tab = eight spaces */
#define TEXTAB "\\        "

void texconvert(int c)
{
  int d;

  switch(c) {
/* TeX specifies that a simple escape (\\foo) is necessary */
  case '$': case '&': case '%': case '_':
    putc('\\',outfile);
    putc(c,outfile);
    break;

/* If the pound sign is the first character on the line, the line is put in poundtype style.  POUND is set true to signal this */
  case '#':
    if(NEWLINE && POUNDSTYLE) {
      fprintf(outfile,"{\\poundtype \\#");
      POUND=TRUE;
    }
    else {
      putc('\\',outfile);
      putc(c,outfile);
    }
    break;

/* More simple cases for TeX */
  case '^':
    fprintf(outfile,"\\char`\\^");
    break;

  case '~':
    fprintf(outfile,"\\char`\\~");
    break;

/* If this is an outermost brace, then the style bodytype is applied.  braces is increased to keep track of nesting.  If the brace is encountered instide of a string or after a single quote or comment, then the above is ignored. */
  case '{': 
    if (!COMMENT) {
      if(!INQUOTE && !SINGLEQUOTE) braces++;
      if (braces==1 && !SINGLEQUOTE && !INQUOTE) {
	fprintf(outfile,"{\\bodytype ");
	INCODE=TRUE;
      }
    }
    if (TTYPE) 
      fprintf(outfile,"\\char`\\{");
    else 
      fprintf(outfile,"{$\\mathtype \\char`\\{$}");
#ifdef DEBUG
    fprintf(stderr,"braces - %d\n",braces);
#endif
    break;

/* If this is an outermose brace, then bodytype style is turned off by closing the group.  As above, if it is on a string, single quote, or comment. Then no checks or closing is made. */
  case '}':
    if (INCODE && !COMMENT && !INQUOTE && !SINGLEQUOTE) braces--;
    if (TTYPE) 
      fprintf(outfile,"\\char`\\}");
    else
      fprintf(outfile,"{$\\mathtype \\char`\\}$}");
    if(!braces && INCODE) {
      fprintf(outfile," }");
      INCODE=FALSE;
    }
#ifdef DEBUG
    fprintf(stderr,"braces -%d\n",braces);
#endif
    break;

/* This tries to keep track of quote marks so it can put left-quotes and right-quotes around strings.  When a possible leftquote is encountered, FIRSTQUOTE is set to true.  When the next quote is encountered, it is made a right-quote.  Also, inquote is set so texconvert doesn't get confused while converting a string. */
  case '"':
    if (FIRSTQUOTE) {
      FIRSTQUOTE=FALSE;
      INQUOTE=FALSE;
      if (TTYPE) 
	putc(c,outfile);
      else {
	putc('\'',outfile);
	putc('\'',outfile);
      }
    }
    else {
      FIRSTQUOTE=TRUE;
      INQUOTE=TRUE;
      if (TTYPE)
	putc(c,outfile);
      else {
	putc('`',outfile);
	putc('`',outfile);
      }
    }
    break;

/* More simple cases */
  case '<':
    if (TTYPE) 
      fprintf(outfile,"\\char`\\<");
    else
      fprintf(outfile,"{$\\mathtype \\char`\\<$}");
    break;

  case '>':
    if (TTYPE)
      fprintf(outfile,"\\char`\\>");
    else
      fprintf(outfile,"{$\\mathtype \\char`\\>$}");
    break;

  case '\\':
    if (TTYPE)
      fprintf(outfile,"\\char`\\\\ ");
    else
      fprintf(outfile,"{$\\mathtype \\char`\\\\$}");
    break;

/*  This case reads ahead to see if it is encountering a '/*' or not.  If it is, then it applies commenttype style and outputs both characters (/ and *).  The flags DELINSTYLE and COMMENTMYSTAR control how exactly this is set up. If the next character is not a '*', then it is pushed back on the stream. */
  case '/':
    d=getc(infile);
    if (d != EOF && d=='*' && !INQUOTE && !SINGLEQUOTE) {
      if (DELINSTYLE) {
	if(COMMENTMYSTAR)
	  fprintf(outfile,"{\\commenttype /{\\mystar}");
	else
	  fprintf(outfile,"{\\commenttype /*");
      }
      else {
	if(COMMENTMYSTAR)
	  fprintf(outfile,"/{\\mystar}{\\commenttype ");
	else
	  fprintf(outfile,"/*{\\commenttype ");
      }
      COMMENT=TRUE;
#ifdef DEBUG
      fprintf(stderr,"comment begin\n");
#endif
    }
    else if (d!=EOF) {
      putc('/',outfile);
      ungetc(d,infile);
    }
    else { 
      putc(c,outfile);
      EOFFLAG=TRUE;
    }
    break;

/* TeX requires a double newline to be printed (well, not really.  I could require you to use the macro \obeylines). Many special cases are turned off when a newline is encountered.  Also, if the next character is a space, that space must be escaped for it and any following spaces to show up. Otherwise, indenting wouldn't work (I don't know why \obeyspaces doesn't work in this case). */
  case '\n':
    if (NEWLINE && !ASCII) fprintf(outfile,"\\ ");
    if (NEWLINE && ASCII) fprintf(outfile,"\n\\ \n");
    if (POUND) {
      POUND=FALSE;
      putc('}',outfile);
    }
    if (ASCII && !NEWLINE)
      fprintf(outfile,"\n");
    else
      fprintf(outfile,"\n\n");
    NEWLINE=TRUE;
    FIRSTQUOTE=FALSE;
    INQUOTE=FALSE;
    d=getc(infile);
    if(d==' ') {
      if (ASCII) fprintf(outfile,"\n");
      fprintf(outfile,"\\ ");
      if (ASCII && PARINDENT) fprintf(outfile,"\\myindent ");
      NEWLINE=FALSE;
    }
    else if (d=='\t') {
      if (ASCII) fprintf(outfile,"\n");
      fprintf(outfile,TEXTAB);
      NEWLINE=FALSE;
    } 
    else if(d!=EOF) ungetc(d,infile);
    else EOFFLAG=TRUE;
    return;
    break;

  case ' ':
    if(NEWLINE)
      fprintf(outfile,"\\");
    putc(' ',outfile);
    break;

  case '-':
    if (TTYPE)
      putc(c,outfile);
    else
      fprintf(outfile,"{$\\mathtype -$}",c);
    break;

/*  The next character is read in a attempt to find a end comment delimiter so it can end commenttype style group.  The macro \mystar is a userspecified macro for an alternate asterisk */
  case '*':
    d=getc(infile);
    if(COMMENT && d=='/'&& !INQUOTE && !SINGLEQUOTE) {
      if(DELINSTYLE) {
	if(COMMENTMYSTAR)
	  fprintf(outfile,"{\\mystar}/}");
	else
	  fprintf(outfile,"*/}");
	COMMENT=FALSE;
      }
      else {
	if(COMMENTMYSTAR)
	  fprintf(outfile,"}{\\mystar}/");
	else
	  fprintf(outfile,"}*/");
	COMMENT=FALSE;
      }
#ifdef DEBUG
      fprintf(stderr,"comment end\n");
#endif
    }
    else if (d!=EOF) {
      if(MYSTAR)
	fprintf(outfile,"{\\mystar}");
      else
	putc('*',outfile);
      ungetc(d,infile);
    }
    else {
      if(MYSTAR)
	fprintf(outfile,"{\\mystar}");
      else
	putc('*',outfile);
      EOFFLAG=TRUE;
    }
    break;

/* SINGLEQUOTE is set true.  The next time texconvert is called, it will be set false. */
  case '\'':
    putc(c,outfile);
    if(!INQUOTE) SINGLEQUOTE=TRUE;
    return;
    break;

/* Need to convert tab to spaces */
  case '\t':
    fprintf(outfile,TEXTAB);
    break;

  default:
    putc(c,outfile);
    NEWLINE=FALSE;
    break;
  }
  NEWLINE=FALSE;
  SINGLEQUOTE=FALSE;
}




