namespace Binge
{
	using System;
	using System.Collections;
	using System.Text;
	using Binge.Bits;
	using Binge.Parsers;
	using Binge.Generators;
	using Binge.Generators.CSharp;
	using Binge.Generators.CSharp.Qt;

	public class Application: Binge.Bits.Object
	{
		enum BindingTargets
		{
			None,
			CSharp,
			CSharpQt
		}

		static string    AnnotationFile   = String.Empty;
		static string    DefaultNamespace = String.Empty;
		static ArrayList SymbolFiles      = new ArrayList ();
		static string    TargetDirectory  = ".";
		static string    SupportDirectory = ".";
		static Encoding  FileEncoding     = new ASCIIEncoding ();
		static bool      NoPascalCase     = false;
		static bool      MakeThrottled    = false;
		static bool      MakePrivate      = false;
		static bool      MakeProtected    = false;
		static bool      ToStdout         = false;
		static bool      WarnXml          = false;
		static string    TargetLibrary    = String.Empty;
		static string    GlueLibrary      = String.Empty;
		static BindingTargets BindingTarget = BindingTargets.CSharpQt;

		// FIXME Replace with a real getopt implementation.
		private static string[] GetOptions (string[] args)
		{
			if (args.Length == 0)
			{
				Usage ();
				Environment.Exit (1);
			}

			ArrayList nargs = new ArrayList ();

			for (int i=0; i<args.Length; i++)
			{
				switch (args[i])
				{
				case "-a":
				case "--annotate":
					AnnotationFile = args[++i];
					break;
				case "-c":
				case "--to-stdout":
					ToStdout = true;
					break;
				case "-e":
				case "--encoding":
					FileEncoding = GetEncoding (args[++i].ToLower ());
					break;
				case "-h":
				case "--help":
					Usage ();
					Environment.Exit (0);
					break;
				case "-n":
				case "--namespace":
					DefaultNamespace = args[++i];
					break;
				case "-s":
				case "--symbols":
					SymbolFiles.Add (args[++i]);
					break;
				case "-t":
				case "--target":
					BindingTarget = GetBindingTarget (args[++i].ToLower ());
					break;
				case "-v":
				case "--verbose":
					Binge.Bits.Object.Verbose = true;
					break;
				case "--target-lib":
					TargetLibrary = args[++i];
					break;
				case "--glue-lib":
					GlueLibrary = args[++i];
					break;
				case "--target-dir":
					TargetDirectory = args[++i];
					break;
				case "--support-dir":
					SupportDirectory = args[++i];
					break;
				case "--warn-xml":
					WarnXml = true;
					break;
				case "--make-throttled":
					MakeThrottled = true;
					break;
				case "--make-private":
					MakePrivate = true;
					break;
				case "--make-protected":
					MakeProtected = true;
					break;
				case "--no-pascal-case":
					NoPascalCase = true;
					break;
				default:
					nargs.Add (args[i]);
					break;
				}
			}

			string[] ret = new string[nargs.Count];
			nargs.CopyTo (ret);

			return ret;
		}

		public static void Usage ()
		{
			Outl ("Usage: binge.exe [options] <apifile>");
			Outl ("options (* = is/has default):");
			Outl (" -a FILE, --annotate FILE   Use annotation file FILE");
			Outl (" -c,      --to-stdout       Generate bindings to stdout");
			Outl (" -e ENC,  --encoding ENC    Write files with encoding ENC *");
			Outl (" -h,      --help            Print help text");
			// Outl (" -n NS,   --namespace NS    Use default namespace NS");
			Outl (" -s FILE, --symbols FILE    Add FILE to mangled symbol file list");
			Outl (" -t TYPE, --target TYPE     Type of bindings to generate *");
			Outl (" -v,      --verbose         Verbose status output");
			Outl ("");
			Outl (" --target-lib NAME          Set target library file to NAME");
			Outl (" --glue-lib NAME            Set glue library file to NAME");
			Outl (" --target-dir DIR           Set binding output directory to DIR *");
			Outl (" --support-dir DIR          Set support output directory to DIR *");
			Outl (" --make-throttled           Parse throttled items");
			Outl (" --make-private             Generate private members");
			Outl (" --make-protected           Generate protected members");
			Outl (" --no-pascal-case           No automatic PascalCase name conversion");
			Outl (" --warn-xml                 Report non-critical errors in XML files");
			Outl ("");
			Outl (" Targets: csharp, *csharp-qt");
			Outl (" Encodings: *ascii, utf7, utf8, unicode");
		}

		static Encoding GetEncoding (string enc)
		{
			if (enc == "ascii")
				return new ASCIIEncoding ();
			else if (enc == "utf7")
				return new UTF7Encoding ();
			else if (enc == "utf8")
				return new UTF8Encoding();
			else if (enc == "unicode")
				return new UnicodeEncoding ();

			Errl ("Error: Unknown encoding: {0}", enc);
			Usage ();
			Environment.Exit (-10);
			return null;
		}

		static BindingTargets GetBindingTarget (string target)
		{
			if (target == "csharp")
				return BindingTargets.CSharp;
			else if (target == "csharp-qt")
				return BindingTargets.CSharpQt;
			else
				Errl ("Error: Unknown binding target: {0}", target);
				Usage ();
				Environment.Exit (-11);

			return BindingTargets.None; // To make the compiler happy.
		}

		public static int Main (string[] args)
		{
			// Configure binge...

			args = GetOptions (args);

			if (args.Length == 0)
			{
				Errl ("Error: No API file specified");
				Usage ();
				return -1;
			}

			// Setup run...

			Binge.Parsers.Parser parser = new XmlParser ();
			Binge.Parsers.Parser annotator = new XmlAnnotationParser ();
			Binge.Generators.Converter converter = null;
			Binge.Generators.GlueStick gluestick = null;
			Binge.Generators.Generator generator = null;
			IMangler mangler = new GnuMangler ();

			switch (BindingTarget)
			{
				case BindingTargets.CSharp:
					converter = new Binge.Generators.CSharp.Converter ();
					gluestick = new Binge.Generators.CSharp.GlueStick ();
					generator = new Binge.Generators.CSharp.Generator ();
					break;
				case BindingTargets.CSharpQt:
					converter = new Binge.Generators.CSharp.Qt.Converter ();
					gluestick = new Binge.Generators.CSharp.Qt.GlueStick ();
					generator = new Binge.Generators.CSharp.Generator ();
					if (TargetLibrary == String.Empty)
						TargetLibrary = "libqt-mt.so";
					if (GlueLibrary == String.Empty)
						GlueLibrary = "libqtsharp.so";
					break;
			}

			foreach (string file in SymbolFiles)
				mangler.Load (file);

			annotator.Namespaces = generator.Namespaces = parser.Namespaces;

			parser.Filename = args[0];
			parser.ParseThrottled = MakeThrottled;
			parser.ParsePrivate = MakePrivate;
			parser.ParseProtected = MakeProtected;
			parser.ShowWarnings = WarnXml;

			annotator.Filename = AnnotationFile;
			annotator.ShowWarnings = WarnXml;

			gluestick.TargetLibrary = TargetLibrary;
			gluestick.GlueLibrary = GlueLibrary;
			gluestick.Mangler = mangler;

			generator.FileEncoding = FileEncoding;
			generator.TargetDirectory = TargetDirectory;
			generator.SupportDirectory = SupportDirectory;
			generator.ToStdout = ToStdout;

			// Parse and annotate

			parser.Parse ();

			if (AnnotationFile != String.Empty)
				annotator.Parse ();

			// Convert and glue

			IDictionaryEnumerator de = parser.Namespaces.GetEnumerator ();
			
			while (de.MoveNext ())
			{
				Namespace ns = de.Value as Namespace;

				IDictionaryEnumerator cde = ns.Classes.GetEnumerator ();
				IDictionaryEnumerator ide = ns.Interfaces.GetEnumerator ();
				IDictionaryEnumerator ede = ns.Enums.GetEnumerator ();

				while (cde.MoveNext ())
				{
					Class klass = cde.Value as Class;

					converter.Convert (klass);

					foreach (Binge.Bits.Enum enm in klass.Enums)
						converter.Convert (enm);

					foreach (Field f in klass.Fields)
						converter.Convert (f);

					foreach (Property p in klass.Properties)
					{
						converter.Convert (p);
						gluestick.Glue (p);
					}

					foreach (Constructor ctor in klass.Constructors)
					{
						converter.Convert (ctor);
						gluestick.Glue (ctor);
						converter.Convert (ctor); // Glue might need further conversion...
					}

					foreach (Destructor dtor in klass.Destructors)
					{
						converter.Convert (dtor);
						gluestick.Glue (dtor);
					}

					foreach (Method method in klass.Methods)
					{
						converter.Convert (method);
						gluestick.Glue (method);
						converter.Convert (method); // Glue might need further conversion...
					}

					foreach (Method method in klass.ImportedMethods)
						converter.Convert (method);
				}

				while (ide.MoveNext ())
				{
					Interface iface = ide.Value as Interface;

					foreach (Property p in iface.Properties)
						converter.Convert (p);

					foreach (Method m in iface.Methods)
						converter.Convert (m);
				}

				while (ede.MoveNext ())
				{
					Binge.Bits.Enum enm = ede.Value as Binge.Bits.Enum;
					converter.Convert (enm);
				}
			}

			// Generate

			generator.Generate ();

			return 0;
		}
	}
}
