
/*	SC	A Spreadsheet Calculator
 *		Range Manipulations
 *
 *              Robert Bond, 4/87
 *
 *		$Revision: 7.7 $
 */

#include <sys/types.h>
#ifdef BSD42
#include <strings.h>
#else
#ifndef SYSIII
#include <string.h>
#endif
#endif

#include <stdio.h>
#include <curses.h>
#include <ctype.h>
#include "sc.h"

static struct range *rng_base;
static struct frange *frame_base;

void
add_range(char *name, struct ent_ptr left, struct ent_ptr right, int is_range)
{
    struct range *r;
    register char *p;
    int minr, minc, maxr, maxc;
    int minrf, mincf, maxrf, maxcf;
    register struct ent *rcp;
    struct range *prev = 0;

    if (left.vp->row < right.vp->row) {
	minr = left.vp->row; minrf = left.vf & FIX_ROW;
	maxr = right.vp->row; maxrf = right.vf & FIX_ROW;
    } else {
	minr = right.vp->row; minrf = right.vf & FIX_ROW;
	maxr = left.vp->row; maxrf = right.vf & FIX_ROW;
    } 

    if (left.vp->col < right.vp->col) {
	minc = left.vp->col; mincf = left.vf & FIX_COL;
	maxc = right.vp->col; maxcf = right.vf & FIX_COL;
    } else {
	minc = right.vp->col; mincf = right.vf & FIX_COL;
	maxc = left.vp->col; maxcf = left.vf & FIX_COL;
    } 

    left.vp = lookat(minr, minc);
    left.vf = minrf | mincf;
    right.vp = lookat(maxr, maxc);
    right.vf = maxrf | maxcf;

    if (!find_range(name, strlen(name), (struct ent *)0, (struct ent *)0,
	    &prev)) {
	error("Error: range name already defined");
	scxfree(name);
	return;
    }

    for (p=name; *p; p++)
	if (!(isalpha(*p) || (*p == '_') || (isdigit(*p) && p != name))) {
	    error("Invalid range name - illegal combination");
	    scxfree(name);
	    return;
	}
 
    p = name;
    if (isalpha(*p++) && (isdigit(*p) || (isalpha(*p++) && isdigit(*p)))) {
	p++;
	while (isdigit(*p))
	    p++;
	if (!(*p)) {
	    error("Invalid range name - ambiguous");
	    scxfree(name);
	    return;
	}
    }
 
    if (autolabel && minc>0 && !is_range) {
	rcp = lookat(minr, minc-1);
	if (rcp->label==0 && rcp->expr==0 && rcp->v==0)
		label(rcp, name, 0);
    }

    r = (struct range *)scxmalloc((unsigned)sizeof(struct range));
    r->r_name = name;
    r->r_left = left;
    r->r_right = right;
    r->r_is_range = is_range;
    if (prev) {
	r->r_next = prev->r_next;
	r->r_prev = prev;
	prev->r_next = r;
	if (r->r_next)
	    r->r_next->r_prev = r;
    } else {
	r->r_next = rng_base;
	r->r_prev = (struct range *)0;
	if (rng_base)
	    rng_base->r_prev = r;
	rng_base = r;
    }
    modflg++;
}

void
del_range(struct ent *left, struct ent *right)
{
    struct range *r;
    int minr, minc, maxr, maxc;

    minr = left->row < right->row ? left->row : right->row;
    minc = left->col < right->col ? left->col : right->col;
    maxr = left->row > right->row ? left->row : right->row;
    maxc = left->col > right->col ? left->col : right->col;

    left = lookat(minr, minc);
    right = lookat(maxr, maxc);

    if (find_range((char *)0, 0, left, right, &r)) 
	return;

    if (r->r_next)
        r->r_next->r_prev = r->r_prev;
    if (r->r_prev)
        r->r_prev->r_next = r->r_next;
    else
	rng_base = r->r_next;
    scxfree((char *)(r->r_name));
    scxfree((char *)r);
    modflg++;
}

void
clean_range()
{
    register struct range *r;
    register struct range *nextr;

    r = rng_base;
    rng_base = (struct range *)0;

    while (r) {
	nextr = r->r_next;
	scxfree((char *)(r->r_name));
	scxfree((char *)r);
	r = nextr;
    }
}

/* Match on name or lmatch, rmatch */

int
find_range(char *name, int len, struct ent *lmatch, struct ent *rmatch,
	struct range **rng)
{
    struct range *r;
    int cmp;

    if (name) {
	for (r = rng_base; r; r = r->r_next) {
	    if ((cmp = strncmp(name, r->r_name, len)) > 0)
		return (cmp);
	    *rng = r;
	    if (cmp == 0)
		return (cmp);
	}
	return (-1);
    }

    for (r = rng_base; r; r = r->r_next) {
	if ((lmatch == r->r_left.vp) && (rmatch == r->r_right.vp)) {
	    *rng = r;
	    return (0);
	}
    }
    return (-1);
}

void
sync_ranges()
{
    struct range *r;
    struct frange *fr;

    r = rng_base;
    while (r) {
	r->r_left.vp = lookat(r->r_left.vp->row, r->r_left.vp->col);
	r->r_right.vp = lookat(r->r_right.vp->row, r->r_right.vp->col);
	r = r->r_next;
    }
    fr = frame_base;
    while (fr) {
	fr->or_left = lookat(fr->or_left->row, fr->or_left->col);
	fr->or_right = lookat(fr->or_right->row, fr->or_right->col);
	fr->ir_left = lookat(fr->ir_left->row, fr->ir_left->col);
	fr->ir_right = lookat(fr->ir_right->row, fr->ir_right->col);
	fr = fr->r_next;
    }
}

void
write_range(FILE *f)
{
    register struct range *r;
    register struct range *nextr;

    for (r = nextr = rng_base; nextr; r = nextr, nextr = r->r_next) /* */ ;
    while (r) {
	(void) fprintf(f, "define \"%s\" %s%s%s%d",
			r->r_name,
			r->r_left.vf & FIX_COL ? "$":"",
			coltoa(r->r_left.vp->col), 
			r->r_left.vf & FIX_ROW ? "$":"",
			r->r_left.vp->row);
	if (r->r_is_range)
	    (void) fprintf(f, ":%s%s%s%d\n",
			    r->r_right.vf & FIX_COL ? "$":"",
			    coltoa(r->r_right.vp->col), 
			    r->r_right.vf & FIX_ROW ? "$":"",
			    r->r_right.vp->row);
	else
	    (void) fprintf(f, "\n");
	r = r->r_prev;
    }
}

void
list_range(FILE *f)
{
    register struct range *r;
    register struct range *nextr;

    (void) fprintf(f, "%-30s %s\n\n","Name","Definition");

    for (r = nextr = rng_base; nextr; r = nextr, nextr = r->r_next) /* */ ;
    while (r) {
	(void) fprintf(f, "%-30s %s%s%s%d",
			    r->r_name,
			    r->r_left.vf & FIX_COL ? "$":"",
			    coltoa(r->r_left.vp->col), 
			    r->r_left.vf & FIX_ROW ? "$":"",
			    r->r_left.vp->row);
	if (r->r_is_range)
	    (void) fprintf(f, ":%s%s%s%d\n",
			    r->r_right.vf & FIX_COL ? "$":"",
			    coltoa(r->r_right.vp->col), 
			    r->r_right.vf & FIX_ROW ? "$":"",
			    r->r_right.vp->row);
	else
	    (void) fprintf(f, "\n");
	r = r->r_prev;
    }
}

char *
v_name(int row, int col)
{
    struct ent *v;
    struct range *r;
    static char buf[20];

    v = lookat(row, col);
    if (!find_range((char *)0, 0, v, v, &r)) {
	return (r->r_name);
    } else {
        (void) sprintf(buf, "%s%d", coltoa(col), row);
	return (buf);
    }
}

char *
r_name(int r1, int c1, int r2, int c2)
{
    struct ent *v1, *v2;
    struct range *r;
    static char buf[100];

    v1 = lookat(r1, c1);
    v2 = lookat(r2, c2);
    if (!find_range((char *)0, 0, v1, v2, &r)) {
	return (r->r_name);
    } else {
        (void) sprintf(buf, "%s", v_name(r1, c1));
	(void) sprintf(buf+strlen(buf), ":%s", v_name(r2, c2));
	return (buf);
    }
}

int
are_ranges()
{
	return (rng_base != 0);
}

void
add_frange(struct ent *or_left, struct ent *or_right, struct ent *ir_left,
	struct ent *ir_right, int toprows, int bottomrows, int leftcols,
	int rightcols)
{
    struct frange *r;
    int minr, minc, maxr, maxc;

    minr = or_left->row < or_right->row ? or_left->row : or_right->row;
    minc = or_left->col < or_right->col ? or_left->col : or_right->col;
    maxr = or_left->row > or_right->row ? or_left->row : or_right->row;
    maxc = or_left->col > or_right->col ? or_left->col : or_right->col;

    or_left = lookat(minr, minc);
    or_right = lookat(maxr, maxc);

    if (ir_left) {
	minr = ir_left->row < ir_right->row ? ir_left->row : ir_right->row;
	minc = ir_left->col < ir_right->col ? ir_left->col : ir_right->col;
	maxr = ir_left->row > ir_right->row ? ir_left->row : ir_right->row;
	maxc = ir_left->col > ir_right->col ? ir_left->col : ir_right->col;

	ir_left = lookat(minr, minc);
	ir_right = lookat(maxr, maxc);

	if (ir_left->row < or_left->row ||
		ir_left->col < or_left->col ||
		ir_right->row > or_right->row ||
		ir_right->col > or_right->col) {
	    error("Invalid parameters");
	    return;
	}
    }

    if (frame_base) {
	/*
	 * Has this frange already been created?  If so, any negative
	 * parameters mean "don't change this value."
	 */
	for (r = frame_base; r; r = r->r_next) {
	    if ((r->or_left == or_left) && (r->or_right == or_right)) {
		if (ir_left) {
		    r->ir_left = ir_left;
		    r->ir_right = ir_right;
		} else {
		    if (toprows < 0)
			toprows = r->ir_left->row - r->or_left->row;
		    if (bottomrows < 0)
			bottomrows = r->or_right->row - r->ir_right->row;
		    if (leftcols < 0)
			leftcols = r->ir_left->col - r->or_left->col;
		    if (rightcols < 0)
			rightcols = r->or_right->col - r->ir_right->col;
		    r->ir_left = lookat(r->or_left->row + toprows,
			    r->or_left->col + leftcols);
		    r->ir_right = lookat(r->or_right->row - bottomrows,
			    r->or_right->col - rightcols);
		}

		/* If all frame sides are 0, delete the frange */
		if (r->ir_left == r->or_left && r->ir_right == r->or_right) {
		    if (r->r_next)
			r->r_next->r_prev = r->r_prev;
		    if (r->r_prev)
			r->r_prev->r_next = r->r_next;
		    else
			frame_base = r->r_next;
		    scxfree((char *)r);
		}
		modflg++;
		FullUpdate++;
		return;
	    }
	}
	/*
	 * See if the specified range overlaps any previously created frange.
	 */
	for (r = frame_base; r; r = r->r_next) {
	    if (  !(r->or_left->row > or_right->row ||
		    r->or_right->row < or_left->row ||
		    r->or_left->col > or_right->col ||
		    r->or_right->col < or_left->col)) {
		error("Framed ranges may not be nested or overlapping");
		return;
	    }
	}
    }

    if (ir_left != or_left || ir_right != or_right) {
	r = (struct frange *)scxmalloc((unsigned)sizeof(struct frange));
	r->or_left = or_left;
	r->or_right = or_right;

	if (ir_left) {
	    r->ir_left = ir_left;
	    r->ir_right = ir_right;
	} else {
	    if (toprows < 0) toprows = 0;
	    if (bottomrows < 0) bottomrows = 0;
	    if (leftcols < 0) leftcols = 0;
	    if (rightcols < 0) rightcols = 0;
	    r->ir_left = lookat(r->or_left->row + toprows,
		    r->or_left->col + leftcols);
	    r->ir_right = lookat(r->or_right->row - bottomrows,
		    r->or_right->col - rightcols);
	}

	r->r_next = frame_base;
	r->r_prev = (struct frange *)0;
	if (frame_base)
	    frame_base->r_prev = r;
	frame_base = r;
	modflg++;
	FullUpdate++;
    }
}

struct frange *
find_frange(int row, int col)
{
    struct frange *r;

    if (frame_base)
	for (r = frame_base; r; r = r->r_next) {
	    if ((r->or_left->row <= row) && (r->or_left->col <= col) &&
		    (r->or_right->row >= row) && (r->or_right->col >= col))
		return r;
	}
    return 0;
}

void
write_frange(FILE *f)
{
    register struct frange *r;
    register struct frange *nextr;

    for (r = nextr = frame_base; nextr; r = nextr, nextr = r->r_next) /* */ ;
    while (r) {
	fprintf(f, "frame %s",
		v_name(r->or_left->row, r->or_left->col));
	fprintf(f, ":%s",
		v_name(r->or_right->row, r->or_right->col));
	fprintf(f, " %s",
		v_name(r->ir_left->row, r->ir_left->col));
	fprintf(f, ":%s\n",
		v_name(r->ir_right->row, r->ir_right->col));
	r = r->r_prev;
    }
}
