#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <linux/unistd.h>

#include <ibcs/xout.h>

#include "x286emul.h"
#include "emu_signal.h"
#include "ldt.h"
#include "debug.h"


struct xexec *xexec = (struct xexec *)X286_MAP_ADDR;
struct xext *xext = (struct xext *)(X286_MAP_ADDR + sizeof(struct xexec));

#ifdef DEBUG
FILE *__dbf = NULL;
#endif

volatile struct sigcontext_struct *prog_sc;
unsigned long SEGV_STACK;
int LDATA,LTEXT;


#ifdef DEBUG
void
dump_state(unsigned char *laddr, unsigned short *stkladdr,
	struct sigcontext_struct *sc)
{
	int i;

	if (!__dbf)
		return;

	fprintf(__dbf, "Trap 0x%08lx, error 0x%08lx\n", sc->trapno, sc->err);
	fprintf(__dbf, "EIP: 0x%02x:0x%08lx (0x%08lx)\n",
		sc->cs, sc->eip, (unsigned long)laddr);
	fprintf(__dbf, "STK: 0x%02x:0x%08lx (0x%08lx) BP: 0x%08lx\n",
		sc->ss, sc->esp_at_signal, (unsigned long)stkladdr, sc->ebp);
	fprintf(__dbf, "EAX: 0x%08lx  EBX: 0x%08lx  ECX: 0x%08lx  EDX: 0x%08lx\n",
		sc->eax, sc->ebx, sc->ecx, sc->edx);
	fprintf(__dbf, "ESI: 0x%08lx  EDI: 0x%08lx\n", sc->esi, sc->edi);
	fprintf(__dbf, "DS:  0x%02x	ES:  0x%02x	SS:  0x%02x	FS:  0x%02x	GS:  0x%02x\n",
		sc->ds, sc->es, sc->ss, sc->fs, sc->gs);
	fprintf(__dbf, "STACK:");
	for (i=-8; i<0; i++)
		fprintf(__dbf, " %04x", stkladdr[i]);
	fprintf(__dbf, " |");
	for (i=0; i<8; i++)
		fprintf(__dbf, " %04x", stkladdr[i]);
	fprintf(__dbf, "\n");
	fprintf(__dbf, "CODE: ");
	for (i=-8; i<0; i++)
		fprintf(__dbf, " %02x", laddr[i]);
	fprintf(__dbf, " |");
	for (i=0; i<8; i++)
		fprintf(__dbf, " %02x", laddr[i]);
	fprintf(__dbf, "\n");
}
#endif


/* Libc sigaction buggers about with sa_restorer. */
int
sigaction(int sig, struct sigaction *new, struct sigaction *old)
{
	__asm__("int $0x80"
		: "=a" (sig)
		: "0" (__NR_sigaction), "b" (sig), "c" (new), "d" (old));
	if (sig >= 0)
		return 0;
	errno = -sig;
	return -1;
}


void
trap_signal(int signr, __sighandler_t func, int flags)
{
	struct sigaction sa;

	sa.sa_handler = func;
	sigfillset(&(sa.sa_mask));
	sigdelset(&sa.sa_mask,signr);
	sigdelset(&sa.sa_mask,SIGSEGV);		/* always allow SIGSEGV */
	sa.sa_flags = flags;

	sa.sa_restorer = (void (*)(void)) SEGV_STACK;

	sigaction(signr, &sa, NULL);
}


/*
 * Routine to replace floating point emulator calls
 * with the real floating point op codes.
 * This info comes via a nice man at SCO.
 * Note 'F8' and 'FA' are untested, i have never
 * seen a program use them.
 */
void
fpfixup(unsigned char *laddr)
{
	
	laddr[0] = 0x9b;
	switch (laddr[1] ){
		case 0xf8:
			laddr[1] = 0x26;
			laddr[2] = 0xd8 | (laddr[2] & 0x07);
			break;
		case 0xf9:
			laddr[1] = 0x90;
			break;
		case 0xfa:
			laddr[1] = 0x36;
			laddr[2] = 0xd8 | (laddr[2] & 0x07);
			break;
		default:
			laddr[1] = 0xd8 | (laddr[1] & 0x07);
		}
}


void
sig_segv(int nr, struct sigcontext_struct sc)
{
	unsigned char *laddr;
	unsigned short *stkladdr;

#ifndef NEW_SYSCALL
	prog_sc = &sc;	  /* SHOULD BE FIRST INSTRUCTION!!!!! */
#endif

#ifdef DEBUG_STACK
	if (__dbf) {
		unsigned int s,p;
		__asm__ volatile ("\tmovl $0,%%eax\n"
			"\tmov %%ss,%%ax\n"
			"\tmovl %%esp,%%ebx\n"
			: "=a" (s),"=b" (p)
			: );
		fprintf(__dbf, "x286emul: stack now -> 0x%02x:0x%04x (0x%08lx)!\n",
				s,
				p,
				ldt[s >> 3].base + ((ldt[s >> 3].base)?(p & 0xffff):(p)) );

	}
#endif

	/* Look up the linear address for the segment:offset of the
	 * eip and stack.
	 */
	laddr = (unsigned char *)(ldt[sc.cs >> 3].base + sc.eip);
	stkladdr = (unsigned short *)(ldt[sc.ss >> 3].base
			+ ((ldt[sc.ss >> 3].base)?(sc.esp_at_signal & 0xffff):(sc.esp_at_signal)));

#ifdef DEBUG_CALL
	if(__dbf){
		dump_state(laddr, stkladdr, &sc);
		fflush(__dbf);
	}
#endif

	if (ldt[sc.cs >> 3].base) {	/* Not in the emulator */

#ifndef NEW_SYSCALL
		/* Xenix 286 programs have a 128 byte patch header at the front of the
		 * program.  The default in this area is to use a "int 0x05" to do a
		 * system call we patch the header to use "int 0xef" because "int 0x05"
		 * is already used by the operating system.
		 * We would perhaps be better with a call gate?
		 */
		if (laddr[0] == 0xcd && laddr[1] == 0xef) {
			sc.eip = (sc.eip + 2) & 0xffff;
			x286syscall(&sc);
			/*
			 * Put flags on the stack just in case we were
			 * interrupted.
			 */
			stkladdr[-1] = (unsigned short)(sc.eflags & 0xffff);
			return;
		}
#endif
#ifdef DEBUG
		if (laddr[-1] == 0x9d && laddr[0] == 0xff && laddr[1] == 0x26) {
			if (__dbf) fprintf(__dbf, "x286emul: Interrupt return problem??\n");
		}
		if (laddr[0] == 0xcc){
			if (__dbf) fprintf(__dbf, "x286emul: Int 0x03 at 0x%02x:0x%04lx\n",sc.cs,sc.eip);
		}
#endif
		if (laddr[0] == 0xcd){
#ifdef DEBUG
			if (__dbf) fprintf(__dbf, "x286emul: Int 0x%02x at 0x%02x:0x%04lx\n",laddr[1],sc.cs,sc.eip);
#endif
			if ((laddr[1] >= 0xf0) && (laddr[1] <= 0xfa)){
				fpfixup(laddr);
				return;
			}
		}

	}

	fprintf(stderr, "x286emul: segv%s!\n", !ldt[sc.cs >> 3].base ? " in emulator" : "");
#ifdef DEBUG
	if (__dbf) {
		fprintf(__dbf, "x286emul: segv%s!\n",
			!ldt[sc.cs >> 3].base ? " in emulator" : "");
		dump_state(laddr, stkladdr, &sc);
	}
#endif
	exit(1);
}


#ifdef NEW_SYSCALL
/*
 * Patch the first 128 bytes of the x286 program to call
 * the emulator on a syscall
 */
unsigned char head_data[] = {	0xeb, 0x14,		/* jmp 0x18 */
				0xeb, 0x0e,		/* jmp 0x14 */
				0xeb, 0x0c,		/* jmp 0x14 */
				0xeb, 0x16,		/* jmp 0x20 */
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
				0xb8, 0x28, 0x08,	/* mov 0x2808,ax */
				0x90,			/* nop */
				0x90,			/* nop */
				0x9a, 0x00, 0x00, 0x00, 0x00,	/* lcall xx:xxxx */
				0xeb, 0x5f,		/* jmp 0x7f */
				0xf4,			/* hlt */
				0xff };
#else
/*
 * Patch the first 128 bytes of the x286 to do "int 0xef" to call
 * the emulator on a syscall
 */
unsigned char head_data[] = {	0xeb, 0x14,		/* jmp 0x18 */
				0xeb, 0x0e,		/* jmp 0x14 */
				0xeb, 0x0c,		/* jmp 0x14 */
				0xeb, 0x16,		/* jmp 0x20 */
				0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00,
				0xb8, 0x28, 0x08,	/* mov 0x2808,ax */
				0x90,			/* nop */
				0x90,			/* nop */
				0xcd, 0xef,		/* int 0xef */
				0x90, 0x90, 0x90,	/* nop */
				0xeb, 0x5f,		/* jmp 0x7f */
				0xf4,			/* hlt */
				0xff };
#endif

void
fix_header(void)
{
unsigned char *address;
int i;

#ifdef NEW_SYSCALL
int t_cs;
unsigned short *aloc;
void __x286syscall(void);
#endif

	address = (unsigned char *)(ldt[init_cs >> 3].base + init_entry +2);
	for (i=0; head_data[i] < 0xff; i++)
		address[i] = head_data[i];

#ifdef NEW_SYSCALL
	/* now patch in the address of "__x286syscall()" */
	aloc = (unsigned short *)(ldt[init_cs >> 3].base + init_entry + 0x1a);

	__asm__ volatile ("\tsubl %%eax,%%eax\n"
		"\tmov %%cs,%%ax\n"
		: "=a" (t_cs)
		: );

	aloc[0] = (unsigned short)__x286syscall;
	aloc[1] = (unsigned short)t_cs;
#endif
}


static unsigned short
set_frame(unsigned long base, unsigned long offset, char *argv[], char *envp[])
{
	unsigned char *sp;
	char *ap, *ep;
	int envc, argc, i;

	sp = (unsigned char *)(base + offset);

	for (envc=0; envp[envc]; envc++) {
		sp -= (strlen(envp[envc])+2) & ~1;
#ifdef DEBUG
		if (__dbf) fprintf(__dbf, "string: 0x%04lx (0x%08lx): \"%s\"\n",
				(unsigned long)(sp-base), (unsigned long)sp, envp[envc]);
#endif
		strcpy(sp, envp[envc]);
	}
	ep = sp;

	for (argc=0; argv[argc]; argc++) {
		sp -= strlen(argv[argc])+1;
#ifdef DEBUG
		if (__dbf) fprintf(__dbf, "string: 0x%04lx (0x%08lx): \"%s\"\n",
				(unsigned long)(sp-base), (unsigned long)sp, argv[argc]);
#endif
		strcpy(sp, argv[argc]);
	}
	ap = sp;

	*(--sp) = 0; *(--sp) = 0;
	*(--sp) = 0; *(--sp) = 0;
	if((int)sp & 1)
		*(--sp) = 0;
#ifdef DEBUG
	if(__dbf) fprintf(__dbf, "array:  0: 0x%04lx (0x%08lx) -> NULL\n",
			(unsigned long)(sp-base), (unsigned long)sp);
#endif
	for (i=envc; i--;) {
		sp -= 2;
		if(LDATA){
			*((unsigned short *)sp) = init_ds;
			sp -= 2;
			}
#ifdef DEBUG
		if(__dbf) fprintf(__dbf, "array: %2d: 0x%04lx (0x%08lx) -> 0x%04lx (0x%08lx)\n",
				i+1,
				(unsigned long)(sp-base), (unsigned long)sp,
				(unsigned long)(ep-base), (unsigned long)ep);
#endif
		*((unsigned short *)sp) = (unsigned long)(ep - base);
		while (*ep++);
		ep = (char *)((long)++ep & ~1);
	}

	if(LDATA){
		*(--sp) = 0; *(--sp) = 0;
		}
	*(--sp) = 0; *(--sp) = 0;
#ifdef DEBUG
	if(__dbf) fprintf(__dbf, "array:  0: 0x%04lx (0x%08lx) -> NULL\n",
			(unsigned long)(sp-base), (unsigned long)sp);
#endif

	for (i=argc; i--;) {
		sp -= 2;
		if(LDATA){
			*((unsigned short *)sp) = init_ds;
			sp -= 2;
			}
#ifdef DEBUG
		if(__dbf) fprintf(__dbf, "array: %2d: 0x%04lx (0x%08lx) -> 0x%04lx (0x%08lx)\n",
				i+1,
				(unsigned long)(sp-base), (unsigned long)sp,
				(unsigned long)(ap-base), (unsigned long)ap);
#endif
		*((unsigned short *)sp) = (unsigned long)(ap - base);
		while (*ap++);
	}

	sp -= 2;
	*((unsigned short *)sp) = argc;

	return (unsigned long)(sp-base);
}


int
main(int argc, char *argv[], char *envp[])
{
	int x286boot(void);
	unsigned long ds_base;

#ifdef DEBUG
	int fd,fd1;
	char *p;
	char p1[256];

	if ((p = getenv("X286DEBUG"))){		/* Open debug file and get file des out of the way */
		snprintf(p1,255,p,getpid());	/* of nasty programs */
		fd = open(p1,O_CREAT | O_APPEND | O_WRONLY, 0644);
		if(fd > 0){
			fd1 = dup2(fd,0x3f);
			close(fd);
			__dbf = fdopen(fd1, "a");
		}
	}

	if (__dbf) {
		setbuf(__dbf, 0);
		fprintf(__dbf, "Xenix 286 emulation: entry 0x%02lx:0x%04lx, data 0x%02lx, stack 0x%02lx:0x%04lx\n",
			init_cs, init_entry, init_ds, init_ds, limit_stk);
	}
#endif

	/* If the stack size isn't already specified in the header set
	 * it to the Xenix default of 0x1000. We need to know this at run
	 * time in order to implement the stkgro check.
	 */
	if (!(xexec->x_renv & XE_FS))
		xext->xe_stksize = 0x1000;

	LDATA = xexec->x_renv & (XE_LDATA | XE_HDATA);
	LTEXT = xexec->x_renv & XE_LTEXT;

	ldt_init();
	base_desc = init_ds >> 3;

	ds_base = (unsigned long)ldt[base_desc].base;
	init_stk = set_frame(ds_base, 0x10000, argv, envp);

	ldt[base_desc].limit = limit_stk | 1;
	ldt[base_desc].rlimit = init_stk - xext->xe_stksize - 1;
	if(ldt[base_desc].rlimit < ldt[base_desc].limit){
		fprintf(stderr, "x286emul: No Stack Space\n");
		exit(1);
		}

	fix_header();

#ifdef DEBUG
	if(__dbf) fprintf(__dbf, "stack addr = 0x%04lx\n", init_stk);
#endif

	{
	unsigned int s,p;
	__asm__ volatile ("\tmovl $0,%%eax\n"
		"\tmov %%ss,%%ax\n"
		"\tmovl %%esp,%%ebx\n"
		: "=a" (s),"=b" (p)
		: );
#ifdef DEBUG_STACK
	if (__dbf) fprintf(__dbf, "x286emul: stack now -> 0x%02x:0x%04x (0x%08lx)!\n",
			s,
			p,
			ldt[s >> 3].base + ((ldt[s >> 3].base)?(p & 0xffff):(p)) );
#endif
	SEGV_STACK = (p - 0x4000) & ~7;
	}

	init_sigs(); /* ONLY AFTER SEGV_STACK INITIALIZED */

	trap_signal(SIGSEGV, (__sighandler_t)sig_segv, SA_NOMASK);

	return x286boot();
}
