/*
 *  MonKT - a monitor for LinKT
 *  Copyright (C) 1998-2000 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qfiledialog.h>
#include "outputctrl.h"
#include "output.h"
#include "global.h"
#include "toolbox.h"
#include "main.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <kmessagebox.h>

#define BTN_WIDTH_MON 60
#define BTN_WIDTH_HEARD 50
#define BTN_WIDTH_SPY 25
#define BTN_HEIGHT 25


extern TopLevel *toplevel;


int currColors[OUTCOLOR_COUNT];


OutputCtrlEntry::OutputCtrlEntry( QWidget *parent, int type ) : QObject()
{
   QFont f(config->font_output, config->size_output);

   this->type = type;
   this->parent = parent;
   neu = false;
   nr = -1;
   spyid = -1;
   otherid = -1;
   calls = NULL;
   open = true;
   saving = false;
   fdsave = -1;


   // Der Kanal-Umschalte-Knopf
   btn = new QPushButton( parent );
   connect( btn, SIGNAL(clicked()), this, SLOT(slotBtnClicked()) );

   if (type != OCE_TYPE_MHLIST)
   {
	   kontextmenu = new QPopupMenu();
	   kontextmenu->insertItem( klocale->translate("&ignore QSO"), this, SLOT(slotIgnoreQSO()));
	   mnu_save = kontextmenu->insertItem( klocale->translate("&save QSO"), this, SLOT(slotSaveQSO()));
//		kontextmenu->insertSeparator( -1 );
	}

   switch (type)
   {
      case OCE_TYPE_MONITOR:
			output = new OutputWidget( parent );
			output->setFont(f);
			output->setBackgroundColor( colors[config->colors->background] );
			output->setGeometry( 0, BTN_HEIGHT, parent->width(), parent->height()-BTN_HEIGHT );
         updateColors();
			btn->setText("Monitor");
			break;
      case OCE_TYPE_MHLIST:
			mhlist = new MHList( parent );
			mhlist->setGeometry( 0, BTN_HEIGHT, parent->width(), parent->height()-BTN_HEIGHT );
         btn->setText("Heard");
      	break;
      case OCE_TYPE_SPY:
			output = new OutputWidget( parent );
			output->setFont(f);
			output->setBackgroundColor( colors[config->colors->background] );
			output->setGeometry( 0, BTN_HEIGHT, parent->width(), parent->height()-BTN_HEIGHT );
			connect( output, SIGNAL(kontextMenu()), this, SLOT(slotKontextMenu()) );
         updateColors();
			break;
   }
}


OutputCtrlEntry::~OutputCtrlEntry()
{
   switch (type)
   {
      case OCE_TYPE_MONITOR:
      case OCE_TYPE_SPY:
		   delete output;
		   delete kontextmenu;
         break;
      case OCE_TYPE_MHLIST:
      	delete mhlist;
         break;
	}
   delete btn;
}


void OutputCtrlEntry::slotBtnClicked()
{
   ((OutputCtrl *)parent)->chooseChannel( this );
}


//   void OutputCtrlEntry::setSpyData( char *call, t_qsoid spyid, t_qsoid otherid )
//
// Rufzeichen und andere Informationen bei einem Spy setzen. Das Rufzeichen
// wird als Tooltip fuer den Button verwendet.
void OutputCtrlEntry::setSpyData( char *call, t_qsoid spyid, t_qsoid otherid )
{
   QToolTip::add( btn, call );
   calls = (char *) strdup(call);
   this->spyid = spyid;
   this->otherid = otherid;
}


int OutputCtrlEntry::isSpyId( int id )
{
   if (id == spyid) return 1;
   if (id == otherid) return 2;
   return -1;
}


void OutputCtrlEntry::show()
{
   btn->show();

   switch (type)
   {
      case OCE_TYPE_MONITOR:
      case OCE_TYPE_SPY:
			output->show();
			break;
      case OCE_TYPE_MHLIST:
      	mhlist->show();
         break;
   }
}


void OutputCtrlEntry::hide()
{
   switch (type)
   {
      case OCE_TYPE_MONITOR:
      case OCE_TYPE_SPY:
			output->hide();
			break;
      case OCE_TYPE_MHLIST:
      	mhlist->hide();
         break;
   }
}


void OutputCtrlEntry::setNew( bool neu )
{
   if (neu == this->neu) return;

   QFont f(btn->font());

   if (neu)
      f.setBold( true );
   else
      f.setBold( false );
   btn->setFont(f);
   this->neu = neu;
}


int OutputCtrlEntry::getNr()
{
   return nr;
}


void OutputCtrlEntry::setNr( int nr )
{
   char tmp[10];

   sprintf(tmp, "%i", nr);
   btn->setText( tmp );
   this->nr = nr;
}


const char * OutputCtrlEntry::getCalls()
{
   return calls;
}


void OutputCtrlEntry::slotKontextMenu()
{
   kontextmenu->popup(QCursor::pos());
}


void OutputCtrlEntry::slotIgnoreQSO()
{
   ((OutputCtrl *)parent)->closeQSO( this );
}


void OutputCtrlEntry::slotSaveQSO()
{
   char tmp[1000];

   if (saving)
   {
      ::close(fdsave);
      saving = false;
      kontextmenu->changeItem(klocale->translate("&save QSO"), mnu_save);
      ((OutputCtrl *)parent)->chooseChannel( this );
      return;
   }

   strcpy(tmp, getenv("HOME"));
   QString f = QFileDialog::getSaveFileName(tmp);

   // Cancel
   if (f.isEmpty()) return;

   strcpy(tmp, (const char *)f);

   if (file_exist(tmp))
   {
      if (KMessageBox::questionYesNo( parent,
									klocale->translate("The specified file allready exists. Overwrite?"),
      							klocale->translate("Overwrite?"),
									klocale->translate("&Yes"),
									klocale->translate("&No")) != KMessageBox::Yes)
         return;
   }

   if ((fdsave = ::open(tmp, O_WRONLY|O_CREAT)) == -1)
   {
      printf("%s (%s)\n", strerror(errno), tmp);
      KMessageBox::error( parent, klocale->translate("Cannot create specified file!") );
      return;
   }

   // Zugriffsrechte einstellen
   fchmod(fdsave, S_IRUSR|S_IWUSR);

   saving = true;
   kontextmenu->changeItem(klocale->translate("Close &Save"), mnu_save);
   ((OutputCtrl *)parent)->chooseChannel( this );
}


void OutputCtrlEntry::getIDs( t_qsoid & id, t_qsoid & otherid )
{
   id = spyid;
   otherid = this->otherid;
}


bool OutputCtrlEntry::isSaving()
{
   return saving;
}


void OutputCtrlEntry::saveLine( char *data, int len )
{
   int i;
   char *tmp;


   if (saving && fdsave != -1)
   {
      tmp = (char *) malloc(len);
      for (i=0;i<len;i++)
         if (data[i] == '\r')
            tmp[i] = '\n';
         else
            tmp[i] = data[i];
      write(fdsave, tmp, len);
      free(tmp);
   }
}


void OutputCtrlEntry::setOtherId( t_qsoid id )
{
   otherid = id;
}


void OutputCtrlEntry::setFonts()
{
   QFont f(config->font_output, config->size_output);
   output->setFont(f);
}


void OutputCtrlEntry::updateColors()
{
	output->updateColors( &(currColors[0]) );
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


OutputCtrl::OutputCtrl( QWidget *parent ) : QWidget( parent )
{
	windows = new QList<OutputCtrlEntry>();
	windows->setAutoDelete( true );

	updateColors();

	// Das Monitor-Fenster erzeugen
	moni = new OutputCtrlEntry( this, OCE_TYPE_MONITOR );
	windows->append( moni );
   visible = moni;
/*	mhlist = new OutputCtrlEntry( this, OCE_TYPE_MHLIST );
	windows->append( mhlist );
   visible = mhlist;*/
}


OutputCtrl::~OutputCtrl()
{
   delete windows;
}


void OutputCtrl::resizeEvent(QResizeEvent *e)
{
   OutputCtrlEntry *tmp;
   int i=0;


   if ((tmp = windows->first()) == NULL) return;
   do
   {
	   switch (tmp->type)
	   {
	      case OCE_TYPE_MONITOR:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_MON, BTN_HEIGHT );
		      tmp->output->setGeometry( 0, BTN_HEIGHT, width(), height()-BTN_HEIGHT );
            i += BTN_WIDTH_MON;
				break;
	      case OCE_TYPE_SPY:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_SPY, BTN_HEIGHT );
		      tmp->output->setGeometry( 0, BTN_HEIGHT, width(), height()-BTN_HEIGHT );
            i += BTN_WIDTH_SPY;
				break;
	      case OCE_TYPE_MHLIST:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_HEARD, BTN_HEIGHT );
	      	tmp->mhlist->setGeometry( 0, BTN_HEIGHT, width(), height()-BTN_HEIGHT );
            i += BTN_WIDTH_HEARD;
	         break;
	   }
      i++;
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::repaintButtons()
{
   int i=0;
   OutputCtrlEntry *tmp;


   if ((tmp = windows->first()) == NULL) return;
   do
   {
	   switch (tmp->type)
	   {
	      case OCE_TYPE_MONITOR:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_MON, BTN_HEIGHT );
            i += BTN_WIDTH_MON;
				break;
	      case OCE_TYPE_SPY:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_SPY, BTN_HEIGHT );
            i += BTN_WIDTH_SPY;
				break;
	      case OCE_TYPE_MHLIST:
	         tmp->btn->setGeometry( i, 0, BTN_WIDTH_HEARD, BTN_HEIGHT );
            i += BTN_WIDTH_HEARD;
	         break;
	   }
      i++;
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::monitorWrite( char *text, int color )
{
   moni->output->writeText(text, color);
   if (visible != moni)
      moni->setNew( true );
}


void OutputCtrl::monitorNewLine()
{
   moni->output->newLine();
   if (visible != moni)
      moni->setNew( true );
}


//   bool OutputCtrl::addSpy( char *tocall, t_qsoid id, t_qsoid otherid )
//
// Oeffnet ein neues Spy-Fenster, wenn es denn noetig ist.
// Rueckgabe:
//    true, wenn der Spy neu ist und ein SPYID-Kommando abgeschickt
//          werden soll,
//    false wenn nicht.
bool OutputCtrl::addSpy( char *tocall, t_qsoid id, t_qsoid otherid )
{
   int nr, at;
   OutputCtrlEntry *spy;

   if ((spy = getSpy( otherid )) == NULL)
   {
      spy = new OutputCtrlEntry( this, OCE_TYPE_SPY );
      spy->setSpyData( tocall, id, otherid );
      getNextNr( nr, at );
      spy->setNr( nr );
      if (at == -1)
         windows->append( spy );
      else
         windows->insert( at, spy );
      spy->hide();
      spy->btn->show();
      repaintButtons();
      return true;
   }

   spy->setOtherId( id );
   return false;
}


void OutputCtrl::chooseChannel( OutputCtrlEntry *entry )
{
   OutputCtrlEntry *tmp;
   QString str;


   if ((tmp = windows->first()) == NULL) return;
   do
   {
      if (tmp == entry)
      {
         tmp->btn->setDown( true );
		   switch (tmp->type)
		   {
		      case OCE_TYPE_MONITOR:
		      case OCE_TYPE_SPY:
					tmp->output->show();
					break;
		      case OCE_TYPE_MHLIST:
		      	tmp->mhlist->show();
		         break;
		   }
         visible = tmp;
         tmp->setNew( false );
      }
      else
      {
         tmp->btn->setDown( false );
		   switch (tmp->type)
		   {
		      case OCE_TYPE_MONITOR:
		      case OCE_TYPE_SPY:
					tmp->output->hide();
					break;
		      case OCE_TYPE_MHLIST:
		      	tmp->mhlist->hide();
		         break;
		   }
      }
   }
   while ((tmp = windows->next()) != NULL);

   switch (entry->type)
   {
		case OCE_TYPE_MONITOR:
      	str = klocale->translate("MonKT - the LinKT Monitor") + " - " + klocale->translate("monitor");
         break;
		case OCE_TYPE_MHLIST:
      	str = klocale->translate("MonKT - the LinKT Monitor") + " - " + klocale->translate("heardlist");
         break;
		case OCE_TYPE_SPY:
	      if (entry->isSaving())
         	str = klocale->translate("MonKT - the LinKT Monitor") + " - " + entry->getCalls() + " - " + klocale->translate("saving");
	      else
         	str = klocale->translate("MonKT - the LinKT Monitor") + " - " + entry->getCalls();
			break;
   }

   toplevel->setCaption( str );
}


//   void OutputCtrl::getNextNr()
//
// Sucht die naechste freie Nummer in der SPY-Leiste 'raus.
void OutputCtrl::getNextNr( int & nr, int & at )
{
   OutputCtrlEntry *tmp;
   int expect=1, i;

   if ((tmp = windows->first()) == NULL) return;
   do
   {
   	if (tmp->type == OCE_TYPE_SPY)
      {
	      i = tmp->getNr();
	      if (i > expect && i != -1)
	      {
	         nr = expect;
	         at = windows->at();
	         return;
	      }
	      expect++;
		}
   }
   while ((tmp = windows->next()) != NULL);

   nr = expect;
   at = windows->at();
}


//   void OutputCtrl::spyData( char *data )
//
// Spy-Daten kommen an
void OutputCtrl::spyData( char *data, int len )
{
   OutputCtrlEntry *tmp;
   t_qsoid id;
   char *str1, *str2;
   bool ret;
   int newlen, i, co;


   memcpy(&id, data, sizeof(t_qsoid));
   len -= sizeof(t_qsoid);
   memmove(data, data+sizeof(t_qsoid), len);

   str1 = (char *) malloc(len+1);
   str2 = (char *) malloc(len+1);
   memcpy(str1, data, len);
   str1[len] = '\0';

   if ((tmp = windows->first()) == NULL) return;
   do
   {
      if ((co = tmp->isSpyId(id)) != -1)
      {
         tmp->saveLine(str1, len);
         switch (co)
         {
            case 1: co = OUTCOLOR_SPY1; break;
            case 2: co = OUTCOLOR_SPY2; break;
         }

         do
         {
            ret = false;

            if ((i = POS('\r', str1)) > -1)
            {
               memcpy(str2, str1, i);
               str2[i] = '\0';
               newlen = strlen(str1)-i-1;
               memmove(str1, str1+i+1, newlen);
               str1[newlen] = '\0';
               ret = true;
            }
            else
            {
               strcpy(str2, str1);
               str1[0] = '\0';
            }

            tmp->output->writeText( str2, co );
            if (visible != tmp)
               tmp->setNew( true );

            if (ret)
            {
               tmp->output->newLine();
               if (visible != tmp)
                  tmp->setNew( true );
            }
         }
         while (str1[0] != '\0');

         free(str1);
         free(str2);
         return;
      }
   }
   while ((tmp = windows->next()) != NULL);

   free(str1);
   free(str2);
}


//   OutputCtrlEntry * OutputCtrl::isOtherQSO( int id )
//
// Sucht den Spy 'raus, der zu dieser uebergebenen QSO-Id gehoert.
OutputCtrlEntry * OutputCtrl::getSpy( t_qsoid id )
{
   OutputCtrlEntry *tmp;

   if ((tmp = windows->first()) == NULL) return NULL;
   do
   {
      if (tmp->isSpyId(id) != -1)
         return tmp;
   }
   while ((tmp = windows->next()) != NULL);

   return NULL;
}


void OutputCtrl::closeQSO( OutputCtrlEntry *w )
{
   char tmp[100];
   t_qsoid id, otherid;


   if (w->open)
   {
      // Wenn der Spy noch offen ist, wollen wir dem ax25spyd sagen, dass
      // wir keine Frames mehr haben wollen.
      w->getIDs( id, otherid );
      sprintf(tmp, "SPYSTOPID %i", id);
      toplevel->sendMonitor(tmp);
      sprintf(tmp, "SPYSTOPID %i", otherid);
      toplevel->sendMonitor(tmp);
   }
   windows->remove( w );
   repaintButtons();

   // Auf den Monitor umschalten
   chooseChannel( moni );
}


void OutputCtrl::spyEnd( t_QSOMheard *data )
{
   OutputCtrlEntry *tmp;
   char str[100];


   if ((tmp = windows->first()) == NULL) return;

   do
   {
      if (tmp->isSpyId(data->qsoid) != -1)
         if (tmp->open)
         {
            tmp->open = false;
            tmp->output->newLine();
            sprintf(str, "<MonKT>: Spy-QSO %s closed.", tmp->getCalls());
            tmp->output->newLine();
            tmp->output->writeText( str, OUTCOLOR_SPYSTATUS );
            sprintf(str, "         Total: %i bytes (%i frames)", data->nInfoBytes, data->nFrames);
            tmp->output->newLine();
            tmp->output->writeText( str, OUTCOLOR_SPYSTATUS );
            sprintf(str, "         Playload: %i bytes", data->nInfoBytes);
            tmp->output->newLine();
            tmp->output->writeText( str, OUTCOLOR_SPYSTATUS );
            tmp->output->newLine();
            if (visible != tmp)
               tmp->setNew( true );
            return;
         }
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::setFonts()
{
   OutputCtrlEntry *tmp;


   if ((tmp = windows->first()) == NULL) return;

   do
   {
      tmp->setFonts();
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::slotBildAuf()
{
	switch (visible->type)
	{
		case OCE_TYPE_MONITOR:
		case OCE_TYPE_SPY:
			visible->output->slotPageUp();
			break;
		case OCE_TYPE_MHLIST:
			break;
	}
}


void OutputCtrl::slotBildAb()
{
	switch (visible->type)
	{
		case OCE_TYPE_MONITOR:
		case OCE_TYPE_SPY:
		   visible->output->slotPageDown();
			break;
		case OCE_TYPE_MHLIST:
			break;
	}
}


void OutputCtrl::updateColors()
{
   OutputCtrlEntry *tmp;

	currColors[OUTCOLOR_WINMARKCOL] = config->colors->qsoWinMarkColor;
	currColors[OUTCOLOR_WINMARKBACK] = config->colors->qsoWinMarkBack;
	currColors[OUTCOLOR_SPY1] = config->colors->spyColor1;
	currColors[OUTCOLOR_SPY2] = config->colors->spyColor2;
	currColors[OUTCOLOR_SPYSTATUS] = config->colors->spyColorStatus;
	currColors[OUTCOLOR_ERROR] = config->colors->colorError;
	currColors[OUTCOLOR_PORT] = config->colors->colorPort;
	currColors[OUTCOLOR_KISS] = config->colors->colorKiss;
	currColors[OUTCOLOR_BPQ] = config->colors->colorBpq;
	currColors[OUTCOLOR_DATA] = config->colors->colorData;
	currColors[OUTCOLOR_PROTOCOL] = config->colors->colorProtocol;
	currColors[OUTCOLOR_AXHDR] = config->colors->colorAxhdr;
	currColors[OUTCOLOR_ADDR] = config->colors->colorAddr;
	currColors[OUTCOLOR_IPHDR] = config->colors->colorIphdr;
	currColors[OUTCOLOR_TCPHDR] = config->colors->colorTcphdr;
	currColors[OUTCOLOR_ROSEHDR] = config->colors->colorRosehdr;
	currColors[OUTCOLOR_TIMESTAMP] = config->colors->colorTimestamp;
	currColors[OUTCOLOR_FLEXNET] = config->colors->colorFlexnet;
	currColors[OUTCOLOR_ZIERRAT] = config->colors->colorZierrat;
	currColors[OUTCOLOR_CALL] = config->colors->colorCall;
	currColors[OUTCOLOR_FMCALL] = config->colors->colorFmCall;
	currColors[OUTCOLOR_TOCALL] = config->colors->colorToCall;
	currColors[OUTCOLOR_VIACALL] = config->colors->colorViaCall;
	currColors[OUTCOLOR_HDRVAL] = config->colors->colorHdrVal;
	currColors[OUTCOLOR_BACK] = config->colors->background;


   if ((tmp = windows->first()) == NULL) return;
   do
   {
   	switch (tmp->type)
      {
			case OCE_TYPE_MONITOR:
			case OCE_TYPE_SPY:
         	tmp->updateColors();
         break;
      }
   }
   while ((tmp = windows->next()) != NULL);
}



//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------





