/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@alphalink.com.au>
	Lennert Buytenhek		<buytenh@scotty.dsv.nl>
        Matt Wilson, Red Hat Inc.       <msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef PED_DISK_H_INCLUDED
#define PED_DISK_H_INCLUDED

typedef struct _PedGeometry		PedGeometry;
typedef struct _PedDisk			PedDisk;
typedef struct _PedPartition		PedPartition;
typedef const struct _PedDiskOps	PedDiskOps;
typedef struct _PedDiskType		PedDiskType;

#include <parted/device.h>
#include <parted/filesys.h>

#define PED_MAX_PART_PER_DISK	16

struct _PedGeometry {
	PedDisk*		disk;
	PedSector		start;
	PedSector		length;
	PedSector		end;
};

typedef enum {
	PED_PARTITION_PRIMARY		= 0x00,
	PED_PARTITION_LOGICAL		= 0x01,
	PED_PARTITION_EXTENDED		= 0x02,
	PED_PARTITION_FREESPACE		= 0x04,
	PED_PARTITION_METADATA		= 0x08
} PedPartitionType;

struct _PedPartition {
	PedPartition*		prev;
	PedPartition*		next;

	PedGeometry		geom;
	int			num;
	int			hidden;

	PedPartitionType	type;
	PedPartition*		part_list;

	int			system;
	int			bootable;

	PedFileSystemType*	fs_type;
};

struct _PedDisk {
	PedDevice*		dev;
	PedDiskType*		type;
	PedPartition*		part_list;

/* office use only ;-) */
	int			update_mode;
};

/* align_geometry() should only be called on partitions that have geometry,
 * a PedPartitionType, and a number.
 * All of the bottom four operations are "private", and should only be called
 * by functions in disk.c.
 */
struct _PedDiskOps {
	int (*probe) (const PedDevice *dev);

	PedDisk* (*open) (PedDevice* dev);
	PedDisk* (*create) (PedDevice* dev);
	int (*close) (PedDisk* disk);
	int (*read) (PedDisk* disk);
	int (*write) (PedDisk* disk);

	int (*align_partition) (PedDisk* disk, PedPartition* part);
	int (*enumerate_partition) (PedDisk* disk, PedPartition* part);
	int (*alloc_metadata) (PedDisk* disk);
	int (*get_extended_system) ();
};

struct _PedDiskType {
	PedDiskType*		next;
	const char*		name;
	PedDiskOps* const	ops;
};

extern void ped_register_disk_type (PedDiskType* type);
extern void ped_unregister_disk_type (PedDiskType* type);
extern PedDiskType* ped_disk_type_get_next (PedDiskType* type);
extern PedDiskType* ped_disk_type_get (const char* name);
extern PedDiskType* ped_disk_probe (const PedDevice* dev);

extern PedDisk* ped_disk_open (PedDevice* dev);
extern PedDisk* ped_disk_create (PedDevice* dev, PedDiskType* type);
extern PedDisk* ped_disk_alloc (PedDevice* dev, PedDiskType* type);
extern int ped_disk_close (PedDisk* disk);
extern int ped_disk_read (PedDisk* disk);
extern int ped_disk_write (PedDisk* disk);
extern int ped_disk_is_busy (const PedDisk* disk);

extern PedGeometry* ped_geometry_new (const PedDisk* disk, PedSector start,
				      PedSector length);
extern PedGeometry* ped_geometry_duplicate (const PedGeometry* geom);
extern void ped_geometry_destroy (PedGeometry* geom);
extern void ped_geometry_set (PedGeometry* geom, PedSector start,
			      PedSector length);
extern void ped_geometry_set_start (PedGeometry* geom, PedSector start);
extern void ped_geometry_set_end (PedGeometry* geom, PedSector end);
extern int ped_geometry_test_overlap (PedGeometry* a, PedGeometry* b);
extern int ped_geometry_test_inside (PedGeometry* a, PedGeometry* b);

extern int ped_geometry_read (const PedGeometry* geom, void* buffer,
			      PedSector offset, PedSector count);
extern int ped_geometry_write (PedGeometry* geom, const void* buffer,
			       PedSector offset, PedSector count);
extern PedSector ped_geometry_check (PedGeometry* geom, void* buffer,
				     PedSector buffer_size, PedSector offset,
				     PedSector granularity, PedSector count);
extern int ped_geometry_sync (PedGeometry* geom);

/* returns -1 if "sector" is not within dest's space. */
extern PedSector ped_geometry_map (const PedGeometry* dst,
				   const PedGeometry* src,
				   PedSector sector);

extern PedPartition* ped_partition_new (const PedDisk* disk,
					PedPartitionType type,
					PedFileSystemType* fs_type,
					PedSector start,
					PedSector end);
extern PedPartition* ped_partition_duplicate (const PedPartition* part);
extern void ped_partition_destroy (PedPartition* part);
extern int ped_partition_is_active (const PedPartition* part);
extern int ped_partition_get_hide_state (const PedPartition* part);
extern int ped_partition_set_hide_state (PedPartition* part, int state);
extern int ped_partition_set_system (PedPartition* part,
				     PedFileSystemType* fs_type);
extern int ped_partition_is_busy (const PedPartition* part);

char* ped_partition_type_get_name (PedPartitionType type);

extern int ped_disk_add_partition (PedDisk* disk, PedPartition* part);
extern int ped_disk_delete_partition (PedDisk* disk, PedPartition* part);
extern int ped_disk_delete_all (PedDisk* disk);
extern int ped_disk_set_partition_geom (PedDisk* disk, PedPartition* part,
					PedSector start, PedSector end);
extern int ped_disk_maximize_partition (PedDisk* disk, PedPartition* part);
extern int ped_disk_minimize_extended_partition (PedDisk* disk);

extern PedPartition* ped_disk_next_partition (const PedDisk* disk,
					      const PedPartition* part);
extern PedPartition* ped_disk_get_partition (const PedDisk* disk, int num);
extern PedPartition* ped_disk_get_partition_by_sector (const PedDisk* disk,
						       PedSector sect);
extern PedPartition* ped_disk_extended_partition (const PedDisk* disk);
extern PedPartition* ped_disk_get_boot_partition (const PedDisk* disk);
extern int ped_disk_set_boot_partition (PedDisk* disk, PedPartition* part);

#endif /* PED_DISK_H_INCLUDED */

