#include <time.h>
#include <math.h>
#include <stdio.h>

#include <qbitmap.h>
#include <qimage.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpushbutton.h>

#include <kapp.h>
#include <kconfig.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstddirs.h>
#include <kmessagebox.h>

#include "world.h"

#include "kinputline.h"
#include "astro.h"
#include "sunclock.h"

time_t World::offset = 0;

void TimeTip::maybeTip(const QPoint &pos)
{
  QString text = ((World*)parentWidget())->getTip(pos);

  if (text.length() > 0)
    tip(QRect(pos,QSize(1,1)), text);
}


World::World()
  : QWidget(0), illuMask(0), tip(0)
{
  KConfig *config = kapp->config();
  config->setGroup("General");
  illumination = config->readBoolEntry("Illumination", TRUE);
}


World::World(QWidget *parent)
  : QWidget(parent), illuMask(0)
{
  QFont        font("Courier", 10);
  QFontMetrics metric(font);

  KConfig *config = kapp->config();
  config->setGroup("General");
  illumination = config->readBoolEntry("Illumination", TRUE);
  QString mapName = config->readEntry("Map");

  loadMap(mapName);

  loadFlags();
  repaint(FALSE);
  	
  // set up the menu
  QPopupMenu *flagsmenu = new QPopupMenu();
  flagsmenu->insertItem(redFlag, this, SLOT(addRedFlag()));
  flagsmenu->insertItem(greenFlag, this, SLOT(addGreenFlag()));
  flagsmenu->insertItem(blueFlag, this, SLOT(addBlueFlag()));
  flagsmenu->insertItem(yellowFlag, this, SLOT(addYellowFlag()));

  menu = new QPopupMenu();
  menu->insertItem(i18n("Select &map..."), this, SLOT(selectMap()));
  menu->insertSeparator();
  menu->insertItem(i18n("&Add a flag"),flagsmenu);
  menu->insertItem(i18n("&Delete nearest flag"), this, SLOT(deleteFlag()));
  menu->insertItem(i18n("A&nnotate flag..."), this, SLOT(annotateFlag()));
  menu->insertSeparator();
  menu->insertItem(i18n("Toggle &Illumination"), this, SLOT(illuminateMap()));
  menu->insertItem(i18n("Toggle &Simulation"), this, SLOT(toggleSimulation()));
  menu->insertSeparator();
  menu->insertItem(i18n("About &World Watch..."), this, SLOT(about()));

  setFont(font);
  setTime();
  setFixedSize(map.width(),map.height()+metric.height()+12);

  tip = new TimeTip(this);  

  show();

  offset = 0;
}


World::~World()
{
  delete tip;
  delete illuMask;
}


void World::loadMap(QString name)
{
  KIconLoader   iconLoader;
  QImage        image;

  redFlag = iconLoader.loadIcon("flag-red", KIcon::Toolbar);
  blueFlag = iconLoader.loadIcon("flag-blue", KIcon::Toolbar);
  greenFlag = iconLoader.loadIcon("flag-green", KIcon::Toolbar);
  yellowFlag = iconLoader.loadIcon("flag-yellow", KIcon::Toolbar);  

  cleanMap = QPixmap(name);
  if (cleanMap.isNull())
    cleanMap = QPixmap(locate("data", "kworldwatch/pics/depths_400.jpg"));
  if (cleanMap.isNull())
    {
      KMessageBox::error(this, i18n("Could not load the map!"), i18n("Image error"));
      abort();
    }

  image = cleanMap.convertToImage();
  QRgb rgb;
  int x,y;
  for (y=0; y<image.height(); y++)
    for(x=0; x<image.width(); x++)
      {
	rgb = image.pixel(x,y);
	image.setPixel(x,y,qRgb(qRed(rgb)/2,qGreen(rgb)/2,qBlue(rgb)/2));
      }
  darkMap.convertFromImage(image);

  map = cleanMap;

  delete illuMask;
  illuMask = new QBitmap(map.width(), map.height());  
}


void World::setTime()
{
  if (offset != 0)
    offset += 7*86500;

  time_t t = time(NULL)+offset;
  struct tm *tmp = gmtime(&t);
  time_t sec = tmp->tm_hour*60*60 + tmp->tm_min*60 + tmp->tm_sec;

  int old_position = gmt_position;
  gmt_position = map.width() * sec / 86400; // note: greenwich is in the middle!

  if (old_position != gmt_position)
    repaint(FALSE);
}


void World::dumpMap(QString fname, QSize size)
{
  int greenwich = map.width()/2;

  setTime();
  updateMap();

  QPixmap image(size);

  QPainter p;

  p.begin(&image);

  // calculate transformation
  if (size != map.size())
    p.scale((double) size.width() / (double) map.size().width(), 
	    (double) size.height() / (double) map.size().height());

  if (gmt_position >= greenwich)
    {	
      p.drawPixmap(gmt_position-greenwich, 0,
                   map, 
                   0, 0,
                   map.width()-gmt_position+greenwich); 
      p.drawPixmap(0,0,
                   map,
                   map.width()-gmt_position+greenwich, 0,
                   gmt_position-greenwich);	
    }
  else
    {	
      p.drawPixmap(0,0,
                   map,
                   greenwich-gmt_position, 0,
                   map.width()+gmt_position-greenwich);	
      p.drawPixmap(map.width()+gmt_position-greenwich, 0,
                   map, 
                   0, 0,
                   greenwich-gmt_position); 
    }

  p.end();

  image.save(fname, "PPM");
}


void World::paintEvent(QPaintEvent *)
{
  int greenwich = map.width()/2;

  updateMap();  
  QPainter p;

  p.begin(this);
  p.setFont(font());

  if (gmt_position >= greenwich)
    {	
      p.drawPixmap(gmt_position-greenwich, 0,
                   map, 
                   0, 0,
                   map.width()-gmt_position+greenwich); 
      p.drawPixmap(0,0,
                   map,
                   map.width()-gmt_position+greenwich, 0,
                   gmt_position-greenwich);	
    }
  else
    {	
      p.drawPixmap(0,0,
                   map,
                   greenwich-gmt_position, 0,
                   map.width()+gmt_position-greenwich);	
      p.drawPixmap(map.width()+gmt_position-greenwich, 0,
                   map, 
                   0, 0,
                   greenwich-gmt_position); 
    }

  // horizontal line
  p.drawLine(0,map.height()+1,width(),map.height()+1);
  // major lines
  for (int c=0; c<6; c++)
    p.drawLine(c*(map.width()-1)/4, map.height()+1, c*(map.width()-1)/4, map.height()+9);
  // minor lines
  for (int c=0; c<25; c++)
    p.drawLine(c*(map.width()-1)/24, map.height()+1, c*(map.width()-1)/24, map.height()+5);

  // text
  p.drawText(0,map.height()+10,width(),height(),AlignHCenter,"12:00");
  p.drawText(0,map.height()+10,width()/2,height(),AlignHCenter,"06:00");
  p.drawText(width()/2,map.height()+10,width()/2,height(),AlignHCenter,"18:00");
  p.drawText(0,map.height()+10,width(),height(),AlignLeft,"00:00");
  p.drawText(0,map.height()+10,width(),height(),AlignRight,"24:00");

  p.end();
}




void World::mousePressEvent(QMouseEvent *event)
{
  pos = event->pos();

  if ( (event->button() == RightButton) && (event->pos().y() <= map.height()))
    menu->popup(mapToGlobal(pos));
}


QString &World::getTip(const QPoint &pos)
{
  char buffer[20];

  if (pos.y() < map.height())
    { 
      time_t time=0;
      
      if (pos.x()>0)
       time = pos.x()*24*60*60/map.width();
      if (pos.x()>=map.width())
	time = 24*60*60;

      strftime(buffer, 20, "%H:%M", gmtime(&time));
      tipText = buffer;      

      int nearest; 
      double dist;
    
      nearest = nearestFlag(pos, dist);
  
      if (nearest>=0 && (unsigned int)nearest<flags.count() && dist<20.0)
        if (!flags.at(nearest)->note.isEmpty())
          {
            tipText += ", ";
            tipText += flags.at(nearest)->note;
          }
    }
  else
    tipText = ""; 
 
  return tipText;
}


void World::addRedFlag()
{
  appendFlag(pos.x(), pos.y()-15,0);
  repaint(FALSE);;
}


void World::addGreenFlag()
{
  appendFlag(pos.x(), pos.y()-15,1);
  repaint(FALSE);;
}


void World::addBlueFlag()
{
  appendFlag(pos.x(), pos.y()-15,2);
  repaint(FALSE);;
}


void World::addYellowFlag()
{
  appendFlag(pos.x(), pos.y()-15,3);
  repaint(FALSE);;
}


class AboutBox : public QDialog
{
public:
  
  AboutBox();

protected:

  void paintEvent(QPaintEvent *event);

private:

  QPixmap     back;
  QPushButton button;
};


AboutBox::AboutBox()
  : QDialog(NULL, "about", TRUE),
    button(i18n("Close"), this)
{
  KIconLoader iconLoader;
  QPixmap back(locate("data", "kworldwatch/pics/world2.png"));

  setCaption(i18n("About KDE World Watch"));
  setFixedSize(back.width(), back.height());
  setBackgroundPixmap(back);

  connect(&button, SIGNAL(clicked()), this, SLOT(accept()));
  
  button.adjustSize();
  button.move((width()-button.width())/2, height()-3*button.height()/2);
}


void AboutBox::paintEvent(QPaintEvent *)
{
  QPainter p(this);

  p.setFont(QFont("Courier", 14, QFont::Bold));
  p.setBackgroundMode(TransparentMode);
  p.drawText(0,0,width(),3*height()/4, AlignCenter, 
    i18n("The KDE World Wide Watch\n\nwritten by Matthias Hölzer\n(hoelzer@physik.uni-wuerzburg.de)"));
}


void World::about()
{
  AboutBox *box = new AboutBox();

  box->exec();

  delete box;
}


void World::loadFlags()
{
  KConfig *config = kapp->config();
  QString key;
  int count;
  Flag *flag;

  flags.setAutoDelete(TRUE);
  flags.clear();

  config->setGroup("General");
  count = config->readNumEntry("Number", -1);

  for (int i=0; i<count; i++)
    {
      flag = new Flag();

      key.sprintf("Flag %i", i);
      config->setGroup(key);
      flag->x = config->readDoubleNumEntry("x");
      flag->y = config->readDoubleNumEntry("y");
      flag->color = config->readNumEntry("flag", 0);
      flag->note = config->readEntry("note", "");
      flags.append(flag);
    }
}


void World::drawFlags()
{
  QPainter p(&map);

  for (Flag *flag=flags.first(); flag != NULL; flag=flags.next())
  {
    switch(flag->color)
    {
      case 0: p.drawPixmap(flag->x*map.width(), flag->y*map.height(), redFlag); break;
      case 1: p.drawPixmap(flag->x*map.width(), flag->y*map.height(), greenFlag); break;
      case 2: p.drawPixmap(flag->x*map.width(), flag->y*map.height(), blueFlag); break;
      case 3: p.drawPixmap(flag->x*map.width(), flag->y*map.height(), yellowFlag); break;
    }
  }
}


void World::appendFlag(int x, int y, int c)
{
  Flag *flag = new Flag();
  
  x = x+map.width()/2-gmt_position-5;
  if (x<0)
    x += map.width();
  if (x>map.width())
    x -= map.width();

  flag->x = (double)x/map.width();
  flag->y = (double)y/map.height();
  flag->color = c;

  flags.append(flag);

  saveFlags();
}


void World::saveFlags()
{
  KConfig *config = kapp->config();
  QString key;

  config->setGroup("General");
  config->writeEntry("Number", flags.count());

  for (unsigned int i=0; i<flags.count(); i++)
    {
      key.sprintf("Flag %i", i);
      config->setGroup(key);
      config->writeEntry("x", flags.at(i)->x);
      config->writeEntry("y", flags.at(i)->y);
      config->writeEntry("flag", flags.at(i)->color);
      config->writeEntry("note", flags.at(i)->note);
    }
}


double World::flagDistance(unsigned int i, const QPoint &pos)
{
  if (i>=flags.count())
    return 1e10;
  
  int x = (int)flags.at(i)->x*map.width();
  int y = (int)flags.at(i)->y*map.height();  

  x = x + gmt_position - map.width()/2;
  if (x<0)
    x += map.width();
  if (x>map.width())
    x -= map.width();

  x -= pos.x();
  y -= pos.y();
 
  return sqrt(x*x+y*y);
}


int World::nearestFlag(const QPoint &pos, double &dist)
{
  int nearest = -1;
  double distance = 1e9;

  for (unsigned int i=0; i<flags.count(); i++)
    if (flagDistance(i, pos) < distance)
      {
        distance = flagDistance(i, pos);
        nearest = i;
      }

  dist = distance;
  return nearest;
}


void World::deleteFlag()
{
  double dist;
  int nearest = nearestFlag(pos, dist);

  if (nearest >= 0 && (unsigned int)nearest<flags.count())
    flags.remove(nearest);
  repaint(FALSE);;

  saveFlags();
}


void World::annotateFlag()
{
  int i; 
  double dist;
 
  i = nearestFlag(pos, dist);
  
  if (i>=0 && (unsigned int)i<flags.count())
    {
      KInputLine il(this, i18n("Annotation"), flags.at(i)->note);
      il.setCaption(i18n("Annotate the nearest flag"));

      if (il.exec() == QDialog::Accepted)
        flags.at(i)->note = il.getText();
    }

  saveFlags();
}


void World::updateMap()
{
  time_t t;
  struct tm *tmp;
  double jt, sunra, sundec, sunrv, sunlong;
  short *wtab;  

  if (illumination)
    {
      map = darkMap;
 
      // calculate the position of the sun

      t = time(NULL) + offset;
      tmp = gmtime(&t);
      jt = jtime(tmp);
      sunpos(jt,FALSE, &sunra, &sundec, &sunrv, &sunlong);

      // calculate the illuminated area

      wtab = new short[map.height()];
      projillum(wtab,map.width(),map.height(),sundec);

      // draw illumination
      illuMask->fill(black);
      QPainter p;
      p.begin(illuMask);
     
      int start, stop;
      int middle = map.width()/2 + map.width()/2 - gmt_position;
      for (int y=0; y<map.height(); y++)
        if (wtab[y]>0)
          {
            start = middle - wtab[y];
            stop = middle + wtab[y];
            if (start < 0)
              {
                p.drawLine(0,y,stop,y);
                p.drawLine(map.width()+start,y,map.width(),y);
              }
            else
              if (stop > map.width())
                {
                  p.drawLine(start,y,map.width(),y);
                  p.drawLine(0,y,stop-map.width(),y);
                }
              else
                p.drawLine(start,y,stop,y);
          }
      p.end();
      
      QPainter mp(&map);
     
      cleanMap.setMask(*illuMask);
      mp.drawPixmap(0,0,cleanMap);

      // deallocate width table      
      delete [] wtab;
    }
  else
    {
      cleanMap.setMask(QBitmap(0,0));
      map = cleanMap;
    }

  // draw flags
  drawFlags();
}


void World::illuminateMap()
{
  illumination = !illumination;
  repaint(FALSE);

  KConfig *config = kapp->config();
  config->setGroup("General");
  config->writeEntry("Illumination", illumination);
}


void World::toggleSimulation()
{
  if (offset == 0)
    offset = 1;
  else
    offset = 0;
}


void World::selectMap()
{
  QFont        font("Courier", 10);
  QFontMetrics metric(font);
  
  KGlobal::dirs()->addResourceType("pics", KStandardDirs::kde_default("data")+"kworldwatch/pics/");
  QString name = KFileDialog::getOpenURL(KGlobal::dirs()->findResourceDir("pics","depths_400.jpg"),"*.jpg", this).filename();
  if (!name.isEmpty())
    {
      saveFlags();

      loadMap(name);

      KConfig *config = kapp->config();
      config->setGroup("General");
      config->writeEntry("Map", name);
      config->sync();

      loadFlags();

      setFixedSize(map.width(),map.height()+metric.height()+12);
    }
}
