/*
 *   WhoisDlg.cpp - Dialog for the whois command
 * 
 *   part of knu: KDE network utilities
 *
 *   Copyright (C) 1997-1999  Bertrand Leconte
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Id: WhoisDlg.cpp,v 1.7 1999/10/09 00:26:48 kalle Exp $
 */

#include <kconfig.h>
#include <klocale.h>
#include <qstringlist.h>

#include "WhoisDlg.h"
#include "WhoisDlg.moc"


// This is the unit used to separate widgets
#define SEPARATION 10

QStringList WhoisDlg::Server_List;

/*
 * Constructor
 */
WhoisDlg::WhoisDlg(QString commandName, 
		 QWidget* parent, const char* name)
  : CommandDlg(commandName, parent, name)
{
  KConfig *kc = kapp->config();

  layout1 = new QBoxLayout(commandBinOK, 
			   QBoxLayout::TopToBottom, SEPARATION);
  CHECK_PTR(layout1);
  
  layout2 = new QBoxLayout(QBoxLayout::LeftToRight, SEPARATION);
  CHECK_PTR(layout2);
  layout1->addLayout(layout2, 0);
  
  // Frame for options
  frame1 = new QFrame(commandBinOK, "frame_1");
  CHECK_PTR(frame1);
  frame1->setFrameStyle(QFrame::Box | QFrame::Sunken);
  layout1->addWidget(frame1, 0);
  
  layout3 = new QBoxLayout(frame1, QBoxLayout::LeftToRight, SEPARATION/2);
  CHECK_PTR(layout3);
  
  // Make the layout of CommandDlg
  layout2->addWidget(commandLbl1);
  layout2->addWidget(commandArgs);
  layout2->addSpacing(2*SEPARATION);
  layout2->addWidget(commandGoBtn);
  layout2->addWidget(commandStopBtn);
  
  // Layout of options
  layout3->addStretch(10);
  
  whoisCb2 = new QComboBox(FALSE, frame1, "cb_1");
  whoisCb2->setMinimumSize(3*fontMetrics().width("----------"), 
			   2*fontMetrics().height());
  whoisCb2->setMaximumSize(QLayout::unlimited, 2*fontMetrics().height());

  // We are reading the server list from the config file.
  kc->setGroup(configGroupName);
  if (!kc->hasKey("Servers")) {
    // The list is not there, so use a default one
    Server_List.append("whois.internic.net");
    Server_List.append("whois.ripe.net");
    Server_List.append("whois.arin.net");
    Server_List.append("whois.apnic.net");
    Server_List.append("whois.6bone.net");
    Server_List.append("whois.ra.net");
    kc->writeEntry("Servers", Server_List, ';'); // Write back configuration data
    
    for (QStringList::Iterator it = Server_List.begin(); it != Server_List.end(); ++it) {
      whoisCb2->insertItem(*it);
    }
  } else {
    Server_List = kc->readListEntry("Servers", ';');
    // I don't use  ->insertStringList() because I want to be able to remove spaces in 
    // the strings
    for (QStringList::Iterator it = Server_List.begin();
	 it != Server_List.end(); ++it) {
      whoisCb2->insertItem((*it).stripWhiteSpace());
    }
  }

  whoisLbl2 = new QLabel(whoisCb2, i18n("Ser&ver:"), frame1, "Label_1");
  whoisLbl2->setFixedSize(whoisLbl2->sizeHint());

  layout3->addWidget(whoisLbl2, 0);
  layout3->addWidget(whoisCb2, 10);

  layout3->addStretch(10);
  layout3->activate();
  
  layout1->addWidget(commandTextArea, 10);
  layout1->activate();

  /*
   * Look at the command binary to see which widget to display
   */
  kc->setGroup(configGroupName);
  if (!kc->hasKey("path")) {
    // It's the first execution, 
    // so we have to search for the pathname
    kc->writeEntry("path", commandName);
    checkBinaryAndDisplayWidget();
    if (commandFound) {
      // All is OK : we can enable this tab.
      if (!kc->hasKey("enable")) {
	kc->writeEntry("enable", 1);
      }
    }
  } else {
    checkBinaryAndDisplayWidget();
  }
  
  // Commit changes in configfile (if needed)
  kc->sync();
}

/*
 * Destructor
 */
WhoisDlg::~WhoisDlg()
{
}

/**
 * build the command line from widgets
 */
bool
WhoisDlg::buildCommandLine(QString args)
{
  QString s;
  KConfig *kc = kapp->config();
  
  kc->setGroup(configGroupName);
  s = kc->readEntry("path");
  if (s.isNull()) {
    return FALSE;
  } else {

    bool whoisBinary = true;  // default to whois behavior
    if (!stricmp(kc->readEntry("binaryType", "whois"), "fwhois")) {
       whoisBinary = false;
    }
    //debug("getExecutable = %s", (const char *)s);
    childProcess.clearArguments();
    childProcess.setExecutable(s);

    // Add arguments

    s = (kc->readEntry("arguments")).simplifyWhiteSpace();
    
    if (!s.isEmpty()) {
      while (s.contains(' ', FALSE) != 0) {
	int pos = s.find(' ', 0, FALSE);
	childProcess << s.left(pos);
	s = s.remove(0, pos+1);
      }
      childProcess << s;
    }
    

    if (whoisBinary) {
       childProcess << "-h";
       childProcess << whoisCb2->currentText(); // server name
       childProcess << (const char *)args;
//       debug(QString("==> whois: -h %1 %2").arg(whoisCb2->currentText()).arg(args));
    } else {
       childProcess << QString((const char *)args) + "@" + whoisCb2->currentText();
//       debug(QString("==> fwhois: %1@%2").arg(args).arg(whoisCb2->currentText()));
    }
    
    return TRUE;
  }
}



/* ******************************************************************** */

/**
 * make a new config object
 *
 * @param parent parent widget
 * @param name name of the widget
 */
WhoisCfgDlg::WhoisCfgDlg(const char *tcs,
			 QWidget *parent, const char *name)
  : CommandCfgDlg(tcs, parent, name)
{
  cfgListManager = 0;
  cfgLayoutGB = 0;
  cfgLayoutTB = 0;
}

/**
 * make a new config widget
 *
 * @param parent parent widget
 * @param makeLayouts name of the widget
 */
QWidget *
WhoisCfgDlg::makeWidget(QWidget *parent, bool makeLayouts)
{
#define SET_ADJUSTED_FIXED_SIZE(_wdgt) \
                  _wdgt->setFixedSize(_wdgt->sizeHint())

  debug("WhoisCfgDlg::makeWidget");
  (void)CommandCfgDlg::makeWidget(parent, FALSE);

  // Widgets
  cfgBG = new QButtonGroup(cfgWidget);
  CHECK_PTR(cfgBG);

  cfgWhoisBtn = new QRadioButton(i18n("&whois"), cfgBG);
  CHECK_PTR(cfgWhoisBtn);
  SET_ADJUSTED_FIXED_SIZE(cfgWhoisBtn);

  cfgFwhoisBtn = new QRadioButton(i18n("&fwhois"), cfgBG);
  CHECK_PTR(cfgFwhoisBtn);
  SET_ADJUSTED_FIXED_SIZE(cfgFwhoisBtn);

  if (!cfgListManager) {
    cfgListManager = new myListManager(i18n("Servers to ask"), 
				       cfgWidget, "lm");
    CHECK_PTR(cfgListManager);
  }

  if (makeLayouts) {
    cfgLayoutTB = new QBoxLayout(cfgWidget, QBoxLayout::TopToBottom, 10);
    CHECK_PTR(cfgLayoutTB);
    
    if (cfgWarning != 0) {
      cfgLayoutTB->addLayout(cfgWarningLayout);
      cfgWarningLayout->addStretch(10);
      cfgWarningLayout->addWidget(cfgWarningPm, 0);
      cfgWarningLayout->addWidget(cfgWarningLbl, 0);
      cfgWarningLayout->addStretch(10);
    }
    cfgLayoutTB->addWidget(cfgBinGB);
    
    cfgLayoutGB = new QGridLayout(cfgBinGB, 4, 3, 10);
    CHECK_PTR(cfgLayoutGB);
    
    cfgLayoutGB->addRowSpacing(0, 0);
    cfgLayoutGB->addWidget(cfgBinNameLbl, 1, 0, AlignRight|AlignVCenter);
    cfgLayoutGB->addWidget(cfgBinNameLE, 1, 1);
    cfgLayoutGB->addWidget(cfgBinNameBrowse, 1, 2);
    cfgLayoutGB->addWidget(cfgBinArgLbl, 2, 0, AlignRight|AlignVCenter);
    cfgLayoutGB->addWidget(cfgBinArgLE, 2, 1);
    cfgLayoutGB->setColStretch(0, 0);
    cfgLayoutGB->setColStretch(1, 10);
    cfgLayoutGB->activate();

    cfgLayout2 = new QBoxLayout(cfgBG, QBoxLayout::LeftToRight, 10);
    CHECK_PTR(cfgLayout2);
    
    cfgLayout2->addStretch(10);
    cfgLayout2->addWidget(cfgWhoisBtn);
    cfgLayout2->addStretch(10);
    cfgLayout2->addWidget(cfgFwhoisBtn);
    cfgLayout2->addStretch(10);
    cfgLayout2->activate();

    cfgLayoutTB->addWidget(cfgBG);
    cfgLayoutTB->addStretch(10);

    cfgListManager->setMinimumSize(150,150); // XXX
    /*
     * Initialisation de la string list. A voir s'il ne
     * faut pas dplacer ce code vers la lecture du fichier de configuration
     */
    cfgListManager->setStringList(WhoisDlg::Server_List); // Initialisation
    cfgLayoutTB->addWidget(cfgListManager, 10);

    //cfgLayoutTB->addStretch(10);

    cfgWidget->adjustSize();
    cfgLayoutTB->activate();

    cfgWidget->adjustSize();
    cfgWidget->setMinimumSize(cfgWidget->size());
    
    cfgLayoutTB->activate();
  }
  readConfig();
  return (cfgWidget);
#undef SET_ADJUSTED_FIXED_SIZE
}

/**
 * delete the config widget
 */
void
WhoisCfgDlg::deleteConfigWidget()
{
  debug("WhoisCfgDlg::deleteCondigWidget");
 
  delete cfgLayoutTB;
  cfgLayoutTB = 0;
  delete cfgLayoutGB;
  cfgLayoutGB = 0;
  delete cfgListManager;
  cfgListManager = 0;
  delete cfgBG;
  cfgBG = 0;
  delete cfgLayout2;
  cfgLayout2 = 0;
  delete cfgWhoisBtn;
  cfgWhoisBtn = 0;
  delete cfgFwhoisBtn;
  cfgFwhoisBtn = 0;
}

/**
 * commit changes to the configfile
 * 
 * @return if the change have been done
 */
bool
WhoisCfgDlg::commitChanges()
{ 
  debug("WhoisCfgDlg::commitChanges()");
  KConfig *kc = kapp->config();

  (void)CommandCfgDlg::commitChanges();
  if (cfgWhoisBtn->isChecked()) {
    kc->writeEntry("binaryType", "whois");
  } else {
    kc->writeEntry("binaryType", "fwhois");
  }
  return(TRUE);
}

/**
 * cancel changes to the configfile
 */
void
WhoisCfgDlg::cancelChanges()
{
  // Nothing to do...
  debug("WhoisCfgDlg::cancelChanges()");
}

/**
 * read the configfile
 */
void
WhoisCfgDlg::readConfig()
{
  debug("WhoisCfgDlg::readConfig()");
  QString s;
  KConfig *kc = kapp->config();

  kc->setGroup(configGroupName);
  
  if (kc->hasKey("binaryType")) {
    s = kc->readEntry("binaryType");
    if (!stricmp(s, "whois")) {
      cfgWhoisBtn->setChecked(TRUE);
    } else {
      cfgFwhoisBtn->setChecked(TRUE);
    }
  } else {
    cfgWhoisBtn->setChecked(TRUE);
  }
}

