#include "wizard.h"

#include <qvbuttongroup.h>
#include <qradiobutton.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qsignalmapper.h>
#include <qvgroupbox.h>
#include <qgrid.h>
#include <kapp.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdialogbase.h>
#include "netutils.h"
#include "types.h"
#include "defines.h"

MPWizard::MPWizard(const MPGameInfo &gi, ConnectionData &_cd,
				   QWidget *parent, const char *name)
: KWizard(parent, name, TRUE), cd(_cd)
{
	kconf = kapp->config();
	kconf->setGroup(MP_GROUP);

	setupTypePage();
	setupLocalPage(gi);
}

//-----------------------------------------------------------------------------
void MPWizard::setupTypePage()
{
	typePage = new QVBox(this);
	typePage->setMargin(KDialogBase::marginHint());
	
	QVButtonGroup *vbg = new QVButtonGroup(typePage);
	connect(vbg, SIGNAL(clicked(int)), SLOT(typeChanged(int)));
	QRadioButton *b;
	b = new QRadioButton(i18n("Create a local game"), vbg);
	b = new QRadioButton(i18n("Create a network game"), vbg);
	b = new QRadioButton(i18n("Join a network game"), vbg);
	type = (Type)kconf->readNumEntry(MP_GAMETYPE, 0);
	if ( type<0 || type>2 ) type = Local;
	vbg->setButton(type);

	typePage->setSpacing(KDialogBase::spacingHint());
	net = new QVGroupBox(i18n("Network settings"), typePage);
	QGrid *grid = new QGrid(2, net);
	lserver = new QLabel(" ", grid);
	grid->setSpacing(KDialogBase::spacingHint());
	eserver = new QLineEdit(grid);
	(void)new QLabel(i18n("Port"), grid);
	int p = kconf->readNumEntry(MP_PORT, (uint)MIN_USER_PORT);
	eport = new KIntNumInput(p, grid);
        eport->setRange(MIN_USER_PORT, MAX_USER_PORT, 1, false);
	
	addPage(typePage, i18n("Choose game type"));
	setHelpEnabled(typePage, FALSE);
	typeChanged(type);
}

//-----------------------------------------------------------------------------
void MPWizard::setupLocalPage(const MPGameInfo &gi)
{
	localPage = new QVBox(this);
	localPage->setMargin(KDialogBase::marginHint());
	
	wl = new WidgetList<PlayerLine>(5, localPage);
	QSignalMapper *husm = new QSignalMapper(this);
	if (gi.humanSettingSlot) connect(husm, SIGNAL(mapped(int)),
									 gi.humanSettingSlot);
	QSignalMapper *aism = new QSignalMapper(this);
	if (gi.AISettingSlot) connect(aism, SIGNAL(mapped(int)), gi.AISettingSlot);
	
	QString n;
	PlayerCheckBox::Type type;
	PlayerLine *pl;
	ASSERT( gi.maxNbLocalPlayers>0 );
	for (uint i=0; i<gi.maxNbLocalPlayers; i++) {
		type = (PlayerCheckBox::Type)
			kconf->readNumEntry(QString(MP_PLAYER_TYPE).arg(i),
								PlayerCheckBox::Empty);
		n = kconf->readEntry(QString(MP_PLAYER_NAME).arg(i),
							 i18n("Player %1").arg(i));
		pl = new PlayerLine(type, n, gi.humanSettingSlot, gi.AISettingSlot,
							i!=0, gi.AIAllowed, wl);
		connect(pl, SIGNAL(typeChanged(int)), SLOT(lineTypeChanged(int)));
		husm->setMapping(pl, i);
		connect(pl, SIGNAL(setHuman()), husm, SLOT(map()));
		aism->setMapping(pl, i);
		connect(pl, SIGNAL(setAI()), aism, SLOT(map()));
		wl->append(pl);
	}
	
	((QVBox *)localPage)->setSpacing(KDialogBase::spacingHint());
//	QHBox *hb = new QHBox(localPage);
	keys = new QPushButton(i18n("Configure keys"), localPage);
	connect(keys, SIGNAL(clicked()), SLOT(configureKeysSlot()));
	
	addPage(localPage, i18n("Local players settings"));
	setHelpEnabled(localPage, FALSE);
	lineTypeChanged(0);
}

QString MPWizard::name(uint i) const
{
	QString s = wl->widget(i)->name();
	if ( s.length()==0 ) s = i18n("Player #%1").arg(i);
	return s;
}

void MPWizard::typeChanged(int t)
{
	type = (Type)t;
	
	QString str;
	if ( type!=Client ) {
		if ( !getHostname(str) ) str = "";
		lserver->setText(i18n("Hostname"));
	} else {
		str = kconf->readEntry(MP_SERVER_ADDRESS, "pc.kde.org");
		lserver->setText(i18n("Server address"));
	}
	eserver->setText(str);
	eserver->setEnabled(type==Client);
	eport->setEnabled(type!=Local);
	net->setEnabled(type!=Local);
}

void MPWizard::lineTypeChanged(int)
{
	bool b = FALSE;
	for (uint i=0; i<wl->size(); i++)
		if ( wl->widget(i)->type()==PlayerCheckBox::Human ) {
			b = TRUE;
			break;
		}
	keys->setEnabled(b);
}

void MPWizard::accept()
{
	kconf->setGroup(MP_GROUP);
	
	cd.network         = ( type!=Local );
	cd.server          = ( type!=Client );
	BoardData bd;
	for (uint i=0; i<wl->size(); i++) {
		if ( wl->widget(i)->type()==PlayerCheckBox::Empty ) continue;
		bd.name = name(i);
		bd.type = wl->widget(i)->type();
		cd.rhd.bds += bd;
	}
	
	if (cd.network) {
		QString serror;
		/* compute addresses */
		if ( !getInetAddresses(eserver->text(), cd.serverAddress,
							   cd.serverName, &serror) )
			R_ERROR_BOX((cd.server ? i18n("Host error")
						 : i18n("Server error")), serror);

		if (!cd.server) {
			QString str;
			if ( !getHostname(str, &serror) ||
				 !getInetAddresses(str, cd.hostAddress, cd.hostName, &serror) )
				R_ERROR_BOX(i18n("Hostname error"), serror);
		}

		/* create a socket */
		if ( !createInetSocket(cd.rhd.socket, &serror) )
			R_ERROR_BOX(serror, QString::null);

		/* bind or connect socket */
		bool err;
		if (cd.server) err = !bindInetSocket(cd.rhd.socket, eport->value(), 1,
											 &serror);
		else err = !connectInetSocket(cd.rhd.socket, eport->value(),
									  cd.serverName, &serror);
		if (err) R_ERROR_BOX(serror, QString::null);
		
		/* ok */
		if ( !cd.server ) kconf->writeEntry(MP_SERVER_ADDRESS,
											eserver->text());
		kconf->writeEntry(MP_PORT, eport->value());
	}
	
	kconf->writeEntry(MP_GAMETYPE, (int)type);
	for (uint i=0; i<wl->size(); i++) {
		kconf->writeEntry(QString(MP_PLAYER_TYPE).arg(i),
						  (int)wl->widget(i)->type());
		kconf->writeEntry(QString(MP_PLAYER_NAME).arg(i), name(i));
	}
	
	KWizard::accept();
}

void MPWizard::showPage(QWidget *page)
{
	if ( page==localPage ) setFinishEnabled(localPage, TRUE);
	KWizard::showPage(page);
}

void MPWizard::configureKeysSlot()
{
	uint nb = 0;
	for (uint i=0; i<wl->size(); i++)
		if ( wl->widget(i)->type()==PlayerCheckBox::Human ) nb++;
	emit configureKeys(nb);
}
