/*
 * localetime.cpp
 *
 * Copyright (c) 1999-2000 Hans Petter Bieker <bieker@kde.org>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qwhatsthis.h>

#include <kglobal.h>
#include <kdialog.h>

#include <kconfig.h>
#include <ksimpleconfig.h>
#include <kstddirs.h>

#include "klocaleadv.h"
#include "toplevel.h"
#include "localetime.h"
#include "localetime.moc"

extern KLocaleAdvanced *locale;

KLocaleConfigTime::KLocaleConfigTime(QWidget *parent, const char*name)
 : QWidget(parent, name)
{
  // Time
  QGridLayout *lay = new QGridLayout(this, 4, 2, 
				     KDialog::marginHint(),
				     KDialog::spacingHint());
  lay->setAutoAdd(TRUE);

  labTimeFmt = new QLabel(this, I18N_NOOP("Time format:"));
  edTimeFmt = new QLineEdit(this);
  connect( edTimeFmt, SIGNAL( textChanged(const QString &) ), this, SLOT( slotTimeFmtChanged(const QString &) ) );

  labDateFmt = new QLabel(this, I18N_NOOP("Date format:"));
  edDateFmt = new QLineEdit(this);
  connect( edDateFmt, SIGNAL( textChanged(const QString &) ), this, SLOT( slotDateFmtChanged(const QString &) ) );

  labDateFmtShort = new QLabel(this, I18N_NOOP("Short date format:"));
  edDateFmtShort = new QLineEdit(this);
  connect( edDateFmtShort, SIGNAL( textChanged(const QString &) ), this, SLOT( slotDateFmtShortChanged(const QString &) ) );
  
  lay->setColStretch(1, 1);
}

KLocaleConfigTime::~KLocaleConfigTime()
{
}

/**
 * Load stored configuration.
 */
void KLocaleConfigTime::load()
{
  KConfig *config = KGlobal::config();
  KConfigGroupSaver saver(config, QString::fromLatin1("Locale"));

  KSimpleConfig ent(locate("locale", QString::fromLatin1("l10n/") + locale->time() + QString::fromLatin1("/entry.desktop")), true);
  ent.setGroup(QString::fromLatin1("KCM Locale"));

  // different tmp variables
  QString str;

  // TimeFormat
  str = config->readEntry(QString::fromLatin1("TimeFormat"));
  if (str.isNull())
    str = ent.readEntry(QString::fromLatin1("TimeFormat"), QString::fromLatin1("%I:%M:%S %p"));
  locale->setTimeFormat(str);
  // DateFormat
  str = config->readEntry(QString::fromLatin1("DateFormat"));
  if (str.isNull())
    str = ent.readEntry(QString::fromLatin1("DateFormat"), QString::fromLatin1("%A %d %B %Y"));
  locale->setDateFormat(str);

  // DateFormatShort
  str = config->readEntry(QString::fromLatin1("DateFormatShort"));
  if (str.isNull())
    str = ent.readEntry(QString::fromLatin1("DateFormatShort"), QString::fromLatin1("%m/%d/%y"));
  locale->setDateFormatShort(str);

  // update the widgets
  edTimeFmt->setText(locale->timeFormat());
  edDateFmt->setText(locale->dateFormat());
  edDateFmtShort->setText(locale->dateFormatShort());
}

void KLocaleConfigTime::save()
{
  KSimpleConfig *c = new KSimpleConfig(QString::fromLatin1("kdeglobals"), false);
  c->setGroup(QString::fromLatin1("Locale"));
  // Write something to the file to make it dirty
  c->writeEntry(QString::fromLatin1("TimeFormat"), QString::null);

  c->deleteEntry(QString::fromLatin1("TimeFormat"), false);
  c->deleteEntry(QString::fromLatin1("DateFormat"), false);
  c->deleteEntry(QString::fromLatin1("DateFormatShort"), false);
  delete c;

  KConfigBase *config = new KConfig;
  config->setGroup(QString::fromLatin1("Locale"));

  KSimpleConfig ent(locate("locale", QString::fromLatin1("l10n/") + locale->time() + QString::fromLatin1("/entry.desktop")), true);
  ent.setGroup(QString::fromLatin1("KCM Locale"));

  QString str;

  str = ent.readEntry(QString::fromLatin1("TimeFormat"), QString::fromLatin1("%I:%M:%S %p"));
  str = config->readEntry(QString::fromLatin1("TimeFormat"), str);
  if (str != locale->timeFormat())
    config->writeEntry(QString::fromLatin1("TimeFormat"), locale->timeFormat(), true, true);

  str = ent.readEntry(QString::fromLatin1("DateFormat"), QString::fromLatin1("%A %d %B %Y"));
  str = config->readEntry(QString::fromLatin1("DateFormat"), str);
  if (str != locale->dateFormat())
    config->writeEntry(QString::fromLatin1("DateFormat"), locale->dateFormat(), true, true);

  str = ent.readEntry(QString::fromLatin1("DateFormatShort"), QString::fromLatin1("%m/%d/%y"));
  str = config->readEntry(QString::fromLatin1("DateFormatShort"), str);
  if (str != locale->dateFormatShort())
    config->writeEntry(QString::fromLatin1("DateFormatShort"), locale->dateFormatShort(), true, true);

  delete config;
}

void KLocaleConfigTime::defaults()
{
  reset();
}

void KLocaleConfigTime::slotTimeFmtChanged(const QString &t)
{
  locale->setTimeFormat(t);
  emit resample();
}

void KLocaleConfigTime::slotDateFmtChanged(const QString &t)
{
  locale->setDateFormat(t);
  emit resample();
}

void KLocaleConfigTime::slotDateFmtShortChanged(const QString &t)
{
  locale->setDateFormatShort(t);
  emit resample();
}

void KLocaleConfigTime::reset()
{
  KSimpleConfig ent(locate("locale", QString::fromLatin1("l10n/") + locale->time() + QString::fromLatin1("/entry.desktop")), true);
  ent.setGroup(QString::fromLatin1("KCM Locale"));

  locale->setTimeFormat(ent.readEntry(QString::fromLatin1("TimeFormat"), QString::fromLatin1("%I:%M:%S %p")));
  locale->setDateFormat(ent.readEntry(QString::fromLatin1("DateFormat"), QString::fromLatin1("%A %d %B %Y")));
  locale->setDateFormatShort(ent.readEntry(QString::fromLatin1("DateFormatShort"), QString::fromLatin1("%m/%d/%y")));

  edTimeFmt->setText(locale->timeFormat());
  edDateFmt->setText(locale->dateFormat());
  edDateFmtShort->setText(locale->dateFormatShort());
}

void KLocaleConfigTime::reTranslate()
{
  QString str;

  str = locale->translate
    ("The text in this textbox will be used to format "
     "time strings. The sequences below will be replaced:\n"
     "\n"
     "%H The hour as a decimal number using a 24-hour clock (00-23).\n"
     "%k The hour (24-hour clock) as a decimal number (0-23).\n"
     "%I The  hour as a decimal number using a 12-hour clock (01-12).\n"
     "%l The hour (12-hour clock) as a decimal number (1-12).\n"
     "%M The minute as a decimal number (00-59).\n"
     "%S The second as a decimal number (00-59).\n"
     "%p Either AM or PM according to the given time "
     "value. Noon is treated as PM and midnight as AM.");
  QWhatsThis::add( labTimeFmt, str );
  QWhatsThis::add( edTimeFmt,  str );

  QString datecodes = locale->translate(
    "\n"
    "%Y\tThe year with century as a decimal number.\n"
    "%y\tThe year without century as a decimal number (00-99).\n"
    "%m\tThe month as a decimal number (01-12).\n"
    "%n\tThe month as a decimal number (1-12).\n"
    "%b\tThe first three characters of the month name.\n"
    "%B\tThe full month name.\n"
    "%d\tThe day of month as a decimal number (01-31).\n"
    "%e\tThe day of month as a decimal number (1-31).\n"
    "%a\tThe first three characters of the weekday name.\n"
    "%A\tThe full weekday name.");

  str = locale->translate
    ( "The text in this textbox will be used to format long "
      "dates. The sequences below will be replaced:\n") + datecodes;
  QWhatsThis::add( labDateFmt, str );
  QWhatsThis::add( edDateFmt,  str );
  
  str = locale->translate
    ( "The text in this textbox will be used to format short "
      "dates. For instance, this is used when listing files. "
      "The sequences below will be replaced:\n") + datecodes;
  QWhatsThis::add( labDateFmtShort, str );
  QWhatsThis::add( edDateFmtShort,  str );
}
