/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#define SLOPPY_FOCUS

#include "icewm.h"

YFrameWindow::YFrameWindow(YWindow *parent, YFrameClient *client): YWindow(parent) {
    fClient = 0;
    fFocused = 0;
    fNext = fPrev = 0;
    fPopupActive = 0;

    style = 0;
    normalX = 0;
    normalY = 0;
    normalWidth = 1;
    normalHeight = 1;
    fSavedFrameState = fState = WithdrawnState;
    movingWindow = 0;
    sizingWindow = 0;
    indicatorsVisible = 0;
    fFrameFunctions = 0;
    fFrameDecors = 0;
    fFrameOptions = 0;
    fFrameIcon = 0;
    fWindowLayer = 0;
    fTaskBarApp = 0;
    fWinListItem = 0;

    createPointerWindows();

    fClientContainer = new YClientContainer(this, this);
    fClientContainer->show();
 
    fTitleBar = new YFrameTitleBar(this, this);
    fTitleBar->show();
    fMaximizeButton = new YFrameButton(fTitleBar, this, cmdMaximize, cmdMaximizeVert);
    fMaximizeButton->show();
    fMinimizeButton = new YFrameButton(fTitleBar, this, cmdMinimize, cmdHide);
    fMinimizeButton->show();
    fCloseButton = new YFrameButton(fTitleBar, this, cmdClose, cmdClose);
    if (useXButton)
        fCloseButton->show();
    fMenuButton = new YFrameButton(fTitleBar, this, cmdSysMenu);
    fMenuButton->show();

    assert(client != 0);
    manage(client);

    getFrameHints();
    if (!(frameOptions() & foFullKeys))
        grabKeys();

    if (frameOptions() & foOnTop)
        fWindowLayer = 1;

    getClientIcon();

    insert();
    fClientContainer->grabButtons();
    if (!(frameOptions() & foIgnoreTaskBar))
            fTaskBarApp = taskBar->addApp(this);
    fWinListItem = windowList->addWindow(this);
    app->restackWindows(this, 1);
    if (frameOptions() & foAllWorkspaces)
        wmOccupyAll();
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geWindowOpened);
#endif
}

YFrameWindow::~YFrameWindow() {
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geWindowClosed);
#endif
    if (movingWindow || sizingWindow)
        endMoveSize();
    if (app->colormapWindow() == this)
        app->setColormapWindow(0);
    if (fTaskBarApp) {
        taskBar->removeApp(this);
        fTaskBarApp = 0;
    }
    if (fWinListItem) {
        windowList->removeWindow(fWinListItem);
        fWinListItem = 0;
    }
    fClientContainer->releaseButtons();
    // perhaps should be done another way
    switchWindow->destroyed(this);
    deactivate();
    remove();

    
    if (fPopupActive)
        fPopupActive->cancelPopup();
    
    if (fClient != 0) {
        XRemoveFromSaveSet(app->display(), client()->clientWindow());
        XDeleteContext(app->display(), client()->clientWindow(), frameContext);
    }
    delete fClient;
    delete fTitleBar;
    delete fMenuButton;
    delete fCloseButton;
    delete fMaximizeButton;
    delete fMinimizeButton;
    delete fClientContainer;

    XDestroyWindow(app->display(), topSide);
    XDestroyWindow(app->display(), leftSide);
    XDestroyWindow(app->display(), rightSide);
    XDestroyWindow(app->display(), bottomSide);
    XDestroyWindow(app->display(), topLeftCorner);
    XDestroyWindow(app->display(), topRightCorner);
    XDestroyWindow(app->display(), bottomLeftCorner);
    XDestroyWindow(app->display(), bottomRightCorner);
}


// create 8 windows that are used to show the proper pointer
// on frame (for resize)
void YFrameWindow::createPointerWindows() {
    XSetWindowAttributes attributes;

    attributes.event_mask = 0;

    attributes.cursor = sizeTopPointer;
    topSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, InputOnly, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeLeftPointer;
    leftSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, InputOnly, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeRightPointer;
    rightSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, InputOnly, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeBottomPointer;
    bottomSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, InputOnly, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeTopLeftPointer;
    topLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                  CopyFromParent, InputOnly, CopyFromParent,
                                  CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeTopRightPointer;
    topRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                   CopyFromParent, InputOnly, CopyFromParent,
                                   CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomLeftPointer;
    bottomLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                     CopyFromParent, InputOnly, CopyFromParent,
                                     CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomRightPointer;
    bottomRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                      CopyFromParent, InputOnly, CopyFromParent,
                                      CWCursor | CWEventMask, &attributes);

    XMapSubwindows(app->display(), handle());
    indicatorsVisible = 1;
}

void YFrameWindow::grabKeys() {
    grabKey(XK_Tab, Mod1Mask);
    grabKey(XK_Tab, Mod1Mask | ShiftMask);
    grabKey(XK_F2, Mod1Mask);
    grabKey(XK_F3, Mod1Mask);
    grabKey(XK_F4, Mod1Mask);
    grabKey(XK_F5, Mod1Mask);
    grabKey(XK_F6, Mod1Mask);
    grabKey(XK_F7, Mod1Mask);
    grabKey(XK_F8, Mod1Mask);
    grabKey(XK_F9, Mod1Mask);
    grabKey(XK_F10, Mod1Mask);
    grabKey(XK_F11, Mod1Mask);
    grabKey(XK_F12, Mod1Mask);
    grabKey(' ', Mod1Mask);
}

void YFrameWindow::manage(YFrameClient *client) {
    assert(client != 0);
    fClient = client;
    
    XSetWindowBorderWidth(app->display(),
                          client->handle(),
                          0);
    
    XAddToSaveSet(app->display(), client->clientWindow());

    client->reparent(fClientContainer, 0, 0);

    client->setFrame(this);
}

void YFrameWindow::unmanage() {
    assert(fClient != 0);
     
    XSetWindowBorderWidth(app->display(),
                          client()->handle(),
                          client()->border());

    client()->reparent(app->root(),
                       x() + borderX(),
                       y() + borderY());

    XRemoveFromSaveSet(app->display(), client()->clientWindow());

    client()->setFrame(0);

    fClient = 0;
}

void YFrameWindow::configureClient(const XConfigureRequestEvent &configureRequest) {
    int cx, cy, cwidth, cheight;

    client()->setBorder((configureRequest.value_mask & CWBorderWidth) ? configureRequest.border_width : client()->border());
    cx = (configureRequest.value_mask & CWX) ? configureRequest.x + DO_BORDER(client()->border()) : x();
    cy = (configureRequest.value_mask & CWY) ? configureRequest.y + DO_BORDER(client()->border()) : y();
    cwidth = (configureRequest.value_mask & CWWidth) ? configureRequest.width : client()->width();
    cheight = (configureRequest.value_mask & CWHeight) ? configureRequest.height : client()->height();

    cx += borderX();
    cy += borderY();

    setGeometry(cx - borderX(),
                cy - borderY(),
                cwidth + 2 * borderX(),
                cheight + 2 * borderY() + titleY());

    if (configureRequest.value_mask & CWStackMode) {
        YFrameWindow *sibling = 0;
        XWindowChanges xwc;

        if ((configureRequest.value_mask & CWSibling) &&
            XFindContext(app->display(),
                         configureRequest.above,
                         frameContext,
                         (XPointer *)&sibling) == 0)
            xwc.sibling = sibling->handle();
        else
            xwc.sibling = configureRequest.above;

        xwc.stack_mode = configureRequest.detail;

        /* !!! implement the rest, and possibly fix these: */

        if (sibling && xwc.sibling != None) {
            switch (xwc.stack_mode) {
            case Above:
                setAbove(sibling);
                break;
            case Below:
                setBelow(sibling);
                break;
            default:
                return ;
            }
            XConfigureWindow (app->display(),
                              handle(),
                              configureRequest.value_mask & (CWSibling | CWStackMode),
                              &xwc);
        } else if (xwc.sibling == None && app->top(layer()) != 0) {
            switch (xwc.stack_mode) {
            case Above:
                wmRaise();
		activate();
                break;
            case Below:
                wmLower();
                break;
            default:
                return ;
            }
        } else
            return ;
    }
}

void YFrameWindow::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int /*count*/) {
    if (up.button == 3) {
        updateMenu();
        windowMenu()->popup(0, this,
                            up.x_root, up.y_root, -1, -1,
                            YPopupWindow::pfCanFlipVertical |
                            YPopupWindow::pfCanFlipHorizontal |
                            YPopupWindow::pfPopupMenu);
        
    }
}


void YFrameWindow::handleCrossing(const XCrossingEvent &crossing) {
    if (crossing.type == EnterNotify) {
        if (!clickFocus)
            activate();
#ifndef SLOPPY_FOCUS
    } else if (crossing.type == LeaveNotify && fFocused) {
        if (crossing.detail != NotifyInferior &&
            crossing.mode == NotifyNormal)
        {
            if (!clickFocus) {
                deactivate();
            }
        }
#endif
    }
}

void YFrameWindow::raise() {
    if (this != app->top(layer())) {
        YWindow::raise();
        setAbove(app->top(layer()));
    }
}

void YFrameWindow::lower() {
    if (this != app->bottom(layer())) {
        YWindow::lower();
        setAbove(0);
    }
}

void YFrameWindow::remove() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before removing", this);
#endif
    if (prev())
        prev()->setNext(next());
    else
        app->setTop(layer(), next());

    if (next())
        next()->setPrev(prev());
    else
        app->setBottom(layer(), prev());
 
    setPrev(0);
    setNext(0);

#ifdef DEBUG
    if (debug_z) dumpZorder("after removing", this);
#endif
}

void YFrameWindow::insert() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before inserting", this);
#endif
    setNext(app->top(layer()));
    setPrev(0);
    if (next())
        next()->setPrev(this);
    else
        app->setBottom(layer(), this);
    app->setTop(layer(), this);
#ifdef DEBUG
    if (debug_z) dumpZorder("after inserting", this);
#endif
}

void YFrameWindow::setAbove(YFrameWindow *aboveFrame) {
#ifdef DEBUG
    if (debug_z) dumpZorder("before setAbove", this, aboveFrame);
#endif
    if (aboveFrame != next() && aboveFrame != this) {
        if (prev())
            prev()->setNext(next());
        else
            app->setTop(layer(), next());

        if (next())
            next()->setPrev(prev());
        else
            app->setBottom(layer(), prev());
        
        setNext(aboveFrame);
        if (next()) {
            setPrev(next()->prev());
            next()->setPrev(this);
        } else {
            setPrev(app->bottom(layer()));
            app->setBottom(layer(), this);
        }
        if (prev())
            prev()->setNext(this);
        else
            app->setTop(layer(), this);
#ifdef DEBUG
        if (debug_z) dumpZorder("after setAbove", this, aboveFrame);
#endif
    }
}

void YFrameWindow::setBelow(YFrameWindow *belowFrame) {
    if (belowFrame != next())
        setAbove(belowFrame->next());
}

YFrameWindow *YFrameWindow::findWindow(int flags) {
    YFrameWindow *p = this;

     if (flags & fwfNext) 
         goto next;
     
     do {
         if ((flags & fwfVisible) && !p->visible())
             goto next;
         if ((flags & fwfFocusable) && !p->focusable())
             goto next;
         if ((flags & fwfWorkspace) && !p->visibleOn(app->activeWorkspace()))
             goto next;
         
         return p;

     next:
         if (flags & fwfBackward) 
             p = (flags & fwfLayers) ? p->prevLayer() : p->prev();
         else
             p = (flags & fwfLayers) ? p->nextLayer() : p->next();
         if (p == 0)
             if (!(flags & fwfCycle))
                 return 0;
             else if (flags & fwfBackward)
                 p = (flags & fwfLayers) ? app->bottomLayer() : app->bottom(layer());
             else 
                 p = (flags & fwfLayers) ? app->topLayer() : app->top(layer());
     } while (p != this);

     if (!(flags & fwfSame))
         return 0;
     if ((flags & fwfVisible) && !p->visible())
         return 0;
     if ((flags & fwfFocusable) && !p->focusable())
         return 0;
     if ((flags & fwfWorkspace) && !p->visibleOn(app->activeWorkspace()))
         return 0;
     
     return this;
}

void YFrameWindow::sendConfigure() {
    XEvent xev;
    
    xev.xconfigure.type = ConfigureNotify;
    xev.xconfigure.display = app->display();
    xev.xconfigure.event = client()->handle();
    xev.xconfigure.window = client()->handle();
    xev.xconfigure.x = x() + borderX() - DO_BORDER(client()->border());
    xev.xconfigure.y = y() + borderY()
#ifndef TITLEBAR_BOTTOM
        + titleY()
#endif
        - DO_BORDER(client()->border());
    xev.xconfigure.width = client()->width();
    xev.xconfigure.height = client()->height();
    xev.xconfigure.border_width = DO_BORDER(client()->border());

    assert(titlebar() != 0);
    xev.xconfigure.above = None; //titlebar()->handle();
    xev.xconfigure.override_redirect = False;

#ifdef DEBUG_C
    Status rc = 
#endif
        XSendEvent(app->display(),
               client()->clientWindow(),
               False,
               StructureNotifyMask,
               &xev);

#ifdef DEBUG_C
    MSG(("sent %d: x=%d, y=%d, width=%d, height=%d",
         rc,
         x(),
         y(),
         client()->width(),
         client()->height()));
#endif
}

void YFrameWindow::handleCommand(WMCommand command, void *context) {
    switch (command) {
    case cmdActivate: activate(); break;
    case cmdRestore: wmRestore(); break;
    case cmdMinimize: wmMinimize(); break;
    case cmdMaximize: wmMaximize(); break;
    case cmdMaximizeVert: wmMaximizeVert(); break;
    case cmdLower: wmLower(); break;
    case cmdRaise: wmRaise(); break;
    case cmdShade: wmShade(); break;
    case cmdClose: wmClose(); break;
    case cmdKill: wmKill(); break;
    case cmdHide: wmHide(); break;
    case cmdMove: wmMove(); break;
    case cmdSize: wmSize(); break;
    case cmdOccupyAll: wmOccupyAll(); break;
    case cmdOccupyAllOrCurrent: wmOccupyAllOrCurrent(); break;
    case cmdOccupyWorkspace: wmOccupyWorkspace((int)context); break;
    case cmdOccupyOnlyWorkspace: wmOccupyOnlyWorkspace((int)context); break;
    default:
	app->handleCommand(command, context);
        break;
    }
}

void YFrameWindow::wmMove() {
    startMoveSize(1, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmSize() {
    startMoveSize(0, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmRestore() {
    if (style & fsHidden)
        wmHide();
    else if (style & fsMinimized)
        wmMinimize();
    else if (style & fsShaded)
        wmShade();
    else if (style & fsMaximized)
        wmMaximize();
}

void YFrameWindow::wmMinimize() {
#ifdef DEBUG_S
    MSG(("wmMinimize - Frame: %d", visible()));
    MSG(("wmMinimize - Client: %d", client()->visible()));
#endif
    if (style & fsMinimized) {
        style &= ~fsMinimized;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        changeState(NormalState);
        if (this == app->focus() && clickFocus)
            app->focusTopWindow();
    } else {
        if (!canMinimize())
            return ;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowMin);
#endif
        style |= fsMinimized;
        changeState(IconicState);
        if (this == app->focus() && clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::DoMaximize(int what) {
    if (style & fsShaded)
        wmShade();

    XSizeHints *sh = client()->sizeHints();
    
    if (style & fsMaximized) {
        style &= ~fsMaximized;
        fMaximizeButton->setCommand(cmdMaximize, cmdMaximizeVert);

        if (sh) {
            normalWidth = normalWidth * sh->width_inc + sh->base_width;
            normalHeight = normalHeight * sh->height_inc + sh->base_height;
        }
        
        setGeometry(normalX, normalY,
                    normalWidth + 2 * borderX(),
                    normalHeight + 2 * borderY() + titleY());
        if (style & fsMinimized)
            wmMinimize();
    } else {
        if (!canMaximize())
            return ;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowMax);
#endif

        style |= fsMaximized;
        fMaximizeButton->setCommand(cmdRestore, cmdRestore);
        
        normalX = x();
        normalY = y();
        normalWidth = client()->width();
        normalHeight = client()->height();

        if (sh) {
            normalWidth = (normalWidth - sh->base_width) / sh->width_inc;
            normalHeight = (normalHeight - sh->base_height) / sh->height_inc;
        }
        
        int nx = x() + borderX();
        int ny = y() + borderY();
        int nw = (what & 2) ? app->maxWidth() : client()->width();
        int nh = (what & 1) ? app->maxHeight() - titleY() : client()->height();
        
        client()->constrainSize(nw, nh);
        
        //nx = x();
        //ny = y();
        //app->constrainPosition(x, y, nw, nh);

        if (nx + nw > int(app->maxX()))
            nx = app->minX(); //app->maxX() - nw - borderX();

        if (ny + nh + int(wsTitleBar) > int(app->maxY()))
            ny = app->minY(); //app->maxY() - nh - titleY() - borderY();

        if (nx < app->minX())
            nx = app->minX();
        if (ny < app->minY())
            ny = app->minY();
        
        nx -= borderX();
        ny -= borderY();
        nw += 2 * borderX();
        nh += 2 * borderY() + titleY();
        
        setGeometry(nx, ny, nw, nh);

        if (style & fsMinimized)
            wmMinimize();
    }
    
}

void YFrameWindow::wmMaximize() {
    DoMaximize(3);

}

void YFrameWindow::wmMaximizeVert() {
    DoMaximize(1);
}

void YFrameWindow::wmMaximizeHorz() {
    DoMaximize(2);
}

void YFrameWindow::wmShade() {
#ifdef DEBUG_S
    MSG(("wmShade - Frame: %d", visible()));
    MSG(("wmShade - Client: %d", client()->visible()));
#endif
    if (style & fsShaded) {
        style &= ~fsShaded;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        if (!(style & fsMinimized)) {
            changeState(NormalState);
            setGeometry(x(),
                        y()
#ifdef TITLEBAR_BOTTOM
                        - client()->height()
#endif
                        ,
                        client()->width() + 2 * borderX(),
                        client()->height() + 2 * borderY() + titleY());
        }
        if (focused() && clickFocus)
            setFocus();
    } else {
        if (!canShade())
            return ;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowShade);
#endif
        style |= fsShaded;
        changeState(IconicState);
        setGeometry(x(),
                    y()
#ifdef TITLEBAR_BOTTOM
                    + client()->height()
#endif
                    ,
                    client()->width() + 2 * borderX(),
                    2 * borderY() + titleY());

    }
    if (focused()) {
        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);
    }
#ifdef DEBUG_S
    MSG(("/wmShade - Frame: %d", visible()));
    MSG(("/wmShade - Client: %d", client()->visible()));
#endif
}

void YFrameWindow::wmHide() {
    if (style & fsHidden) {
        style &= ~fsHidden;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        changeState(NormalState);
        if (fTaskBarApp == 0 && !(frameOptions() & foIgnoreTaskBar))
            fTaskBarApp = taskBar->addApp(this);
        if (clickFocus)
            app->focusTopWindow();
    } else {
        if (!canHide())
            return ;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowHide);
#endif
        style |= fsHidden;
        changeState(IconicState);
        if (fTaskBarApp) {
            taskBar->removeApp(this);
            fTaskBarApp = 0;
        }
        if (clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::wmLower() {
    if (this != app->bottom(layer())) {
        YFrameWindow *w = this;
        YFrameClient *o;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowLower);
#endif
        while (w) {
            w->doLower();

            o = w->client()->owner();
            if (o)
                w = o->frame();
            else
                w = 0;
        }
        app->restackWindows(this, -1);
        if (clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::doLower() {
    setAbove(0);
}

void YFrameWindow::wmRaise() {
    doRaise();
    app->restackWindows(this, 1);
}

void YFrameWindow::doRaise() {
#ifdef DEBUG
    if (debug_z) dumpZorder("wmRaise: ", this);
#endif
    if (this != app->top(layer())) {
        setAbove(app->top(layer()));
        {
            YFrameClient *c = client()->transient();
            while (c) {
                if (c->frame())
                    c->frame()->doRaise();
                c = c->nextTransient();
            }
        }
#ifdef DEBUG
        if (debug_z) dumpZorder("wmRaise after raise: ", this);
#endif
    }
}

void YFrameWindow::wmClose() {
    if (!canClose())
        return ;

    /* race condition otherwise (???) */
    XGrabServer(app->display());
    client()->getProtocols();

    if (client()->protocols() & YFrameClient::wpDeleteWindow)
        client()->sendMessage(_XA_WM_DELETE_WINDOW);
    else {
#ifdef DEBUG
//        if (debug) 
              msg("No WM_DELETE_WINDOW protocol");
#endif
        XBell(app->display(), 100);
        XKillClient(app->display(), client()->clientWindow());
    }
    XUngrabServer(app->display());
}

void YFrameWindow::wmKill() {
    // must ask user!
    ///return ;
    
    if (!canClose())
        return ;
    XBell(app->display(), 100);
    XKillClient(app->display(), client()->clientWindow());
}

void YFrameWindow::wmPrevWindow() {
    if (next() != this)
        app->activate(findWindow(fwfNext | fwfBackward | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
}

void YFrameWindow::wmNextWindow() {
    if (next() != this) {
        wmLower();
        app->activate(findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
    }
}

void YFrameWindow::wmLastWindow() {
    if (next() != this)
        app->activate(findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
}

void YFrameWindow::loseFocus() {
    if (fFocused) {
        fFocused = 0;
        if (focusOnClickClient || raiseOnClickClient)
            if (fClientContainer)
                fClientContainer->grabButtons();
        titlebar()->deactivate();
        repaint();
        if (fTaskBarApp)
            fTaskBarApp->repaint();
    }
}

void YFrameWindow::setFocus() {
    if (!fFocused) {

        fFocused = 1;
        titlebar()->activate();
        repaint();
        if (fTaskBarApp)
            fTaskBarApp->repaint();
        
        if (warpPointer) { /* ugh :-( */ 
            Window child, root;
            int root_x, root_y, win_x, win_y;
            unsigned int mask;
            
            XQueryPointer(app->display(), app->root()->handle(), &root, &child, 
                          &root_x, &root_y, &win_x, &win_y, &mask);

            if (child != handle())
                XWarpPointer(app->display(), 
                             None, app->root()->handle(),
                             0, 0, 0, 0, x(), y());

        }

        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);

        if (focusOnClickClient &&
           !(raiseOnClickClient && (this != app->top(layer()))))
            fClientContainer->releaseButtons();

        if (client()->protocols() & YFrameClient::wpTakeFocus)
            client()->sendMessage(_XA_WM_TAKE_FOCUS);
    }
}

void YFrameWindow::activate() {
    if (!visibleNow()) {
        for (int w = 0; w < workspaceCount; w++)
            if (visibleOn(w)) {
                app->activateWorkspace(w);
                break;
            }
    }
    // recover lost (offscreen) windows
    if (x() >= int(app->root()->width()) ||
        y() >= int(app->root()->height()) ||
        x() <= - int(width()) ||
        y() <= - int(height()))
    {
        int newX = x();
        int newY = y();
        
        if (x() >= int(app->root()->width()))
            newX = int(app->root()->width() - width() + borderX());
        if (y() >= int(app->root()->height()))
            newY = int(app->root()->height() - height() + borderY());
        
        if (newX < int(- borderX()))
            newX = int(- borderX());
        if (newY < int(- borderY()))
            newY = int(- borderY());
        setPosition(newX, newY);
    }
    
    if (raiseOnFocus)
        wmRaise();
    if (style & fsHidden)
        wmHide();
    if (style & fsMinimized)
        wmMinimize();
    if (focusable())
        app->setFocus(this);
}

void YFrameWindow::deactivate() {
    if (app->focus() == this)
        app->loseFocus(this);
}

int YFrameWindow::focusable() {
    return client() ? client()->focusable() : 0;
}

void YFrameWindow::setFrameState(FrameState state) {
    fState = state;
    client()->setWMState(fState);
}

void YFrameWindow::changeState(FrameState state) {
    if (state == IconicState) {
        if (frameState() == NormalState || frameState() == WithdrawnState || (style & fsShaded)) {
            if (visibleNow()) {
#ifdef DEBUG_S
                MSG(("->Iconic"));
#endif
                if (!(style & fsShaded) || (style & (fsMinimized | fsHidden))) {
#ifdef DEBUG_S
                    MSG(("--- Hide Frame: %d", visible()));
#endif
                    hide();
                }
#ifdef DEBUG_S
                MSG(("--- Hide Client: %d", client()->visible()));
#endif
                fClientContainer->hide();
                client()->hide();
                if (frameState() != IconicState)
                    setFrameState(IconicState);
            } else {
                fSavedFrameState = state;
            }
        }
    } else if (state == NormalState) {
        if (frameState() == IconicState || frameState() == WithdrawnState) {
            if (visibleNow()) {
#ifdef DEBUG_S
                MSG(("->Normal"));
#endif
                if (!(style & fsShaded)) {
#ifdef DEBUG_S
                    MSG(("--- Show Client: %d", client()->visible()));
#endif
                    client()->show();
                    fClientContainer->show();
                }
#ifdef DEBUG_S
                MSG(("--- Show Frame: %d", visible()));
#endif
                show();
                if (!(style & fsShaded))
                    if (frameState() != NormalState)
                        setFrameState(NormalState);
            } else {
                fSavedFrameState = state;
            }
        }
    }
#ifdef DEBUG_S
    MSG(("- Frame: %d", visible()));
    MSG(("- Client: %d", client()->visible()));
#endif
}

void YFrameWindow::paint(Graphics &g, int , int , unsigned int , unsigned int ) {
    YColor *bg;

    if (focused())
        bg = activeBorderBg;
    else
        bg = inactiveBorderBg;

    g.setColor(bg);
    switch (wmLook) {
#if defined(CONFIG_LOOK_WIN95) || defined(CONFIG_LOOK_WARP4) || defined(CONFIG_LOOK_NICE)
#ifdef CONFIG_LOOK_WIN95
    case lookWin95:
#endif
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4:
#endif
#ifdef CONFIG_LOOK_NICE
    case lookNice:
#endif
        g.fillRect(1, 1, width() - 3, height() - 3);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
        break;
#endif
#if defined(CONFIG_LOOK_MOTIF) || defined(CONFIG_LOOK_WARP3)
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif:
#endif
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3:
#endif
        g.fillRect(1, 1, width() - 2, height() - 2);
        g.draw3DRect(0, 0, width() - 1, height() - 1, true);
        g.draw3DRect(borderX() - 1, borderY() - 1,
                     width() - 2 * borderX() + 1, height() - 2 * borderY() + 1,
                     false);

#ifdef CONFIG_LOOK_MOTIF
        if (wmLook == lookMotif && canSize()) {
            YColor *b = bg->brighter();
            YColor *d = bg->darker();


            g.setColor(d);
            g.drawLine(wsCornerX - 1, 0, wsCornerX - 1, height() - 1);
            g.drawLine(width() - wsCornerX - 1, 0, width() - wsCornerX - 1, height() - 1);
            g.drawLine(0, wsCornerY - 1, width(),wsCornerY - 1);
            g.drawLine(0, height() - wsCornerY - 1, width(), height() - wsCornerY - 1);
            g.setColor(b);
            g.drawLine(wsCornerX, 0, wsCornerX, height() - 1);
            g.drawLine(width() - wsCornerX, 0, width() - wsCornerX, height() - 1);
            g.drawLine(0, wsCornerY, width(), wsCornerY);
            g.drawLine(0, height() - wsCornerY, width(), height() - wsCornerY);
        }
        break;
#endif
#endif
    default:
        break;
    }
}

void YFrameWindow::setPopupActive(YPopupWindow *popup) {
    MSG(("setting popup for frame to %ld", popup));
    fPopupActive = popup;
}

void YFrameWindow::popupSystemMenu() {
    if (fPopupActive == 0)
        fMenuButton->popupMenu();
}

void YFrameWindow::updateTitle() {
    titlebar()->repaint();
}

void YFrameWindow::updateIconTitle() {
    if (fTaskBarApp)
        fTaskBarApp->repaint();
}

int YFrameWindow::visibleNow() {
    return (style & fsWorkspaceHidden) ? 0 : 1;
}

void YFrameWindow::wmOccupyAllOrCurrent() {
    if (client()->allWorkspaces()) {
        client()->setWorkspaces(1 << app->activeWorkspace());
        client()->setAllWorkspaces(0);
    } else {
        client()->setAllWorkspaces(1);
    }
}

void YFrameWindow::wmOccupyAll() {
    if (client()->allWorkspaces()) {
        client()->setAllWorkspaces(0);
    } else {
        client()->setAllWorkspaces(1);
    }
}

void YFrameWindow::wmOccupyWorkspace(int workspace) {
    assert(workspace >= 0 && workspace < workspaceCount);
    unsigned long nw = client()->workspaces() ^ (1 << workspace);
    if (nw == 0)
        nw = 1 << app->activeWorkspace();
    client()->setWorkspaces(nw);
}

void YFrameWindow::wmOccupyOnlyWorkspace(int workspace) {
    assert(workspace >= 0 && workspace < workspaceCount);
    client()->setWorkspaces(1 << workspace);
    client()->setAllWorkspaces(0);
}

void YFrameWindow::workspaceShow() {
    if (style & fsWorkspaceHidden) {
        style &= ~fsWorkspaceHidden;

        fState = fSavedFrameState; 
        if (frameState() == NormalState)
            show();
        if (style & fsShaded)
            show();

        if (!taskBarShowAllWindows)
            if (fTaskBarApp)
                fTaskBarApp->setShown(1);
        
        if (clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::workspaceHide() {
    if (!(style & fsWorkspaceHidden)) {
        style |= fsWorkspaceHidden;

        fSavedFrameState = frameState();
        /*setFrameState(IconicState);
        hide();
        fClientContainer->hide();
        client()->hide();*/

        hide();
        fState = IconicState;
        
        if (!taskBarShowAllWindows)
            if (fTaskBarApp)
                fTaskBarApp->setShown(0);

        if (clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::addToMenu(YMenu *menu) {
    YMenu::YMenuItem *item =
        new YMenu::YMenuItem((char *)client()->windowTitle(), -1, 0,
                             cmdActivateWindow, 0, this);
    if (item) {
        item->setPixmap(clientIcon()->small());
        menu->add(item);
    }
}

void YFrameWindow::getFrameHints() {
#ifndef NO_MWM_HINTS
    CARD32 decors = client()->mwmDecors();
    CARD32 functions = client()->mwmFunctions();

    fFrameFunctions = ffMove;
    fFrameDecors = 0;
    fFrameOptions = 0;
    
    if (decors & MWM_DECOR_BORDER)      fFrameDecors |= fdBorder;
    if (decors & MWM_DECOR_RESIZEH)     fFrameDecors |= fdResize;
    if (decors & MWM_DECOR_TITLE)       fFrameDecors |= fdTitleBar;
    if (decors & MWM_DECOR_MENU)        fFrameDecors |= fdSysMenu;
    if (decors & MWM_DECOR_MAXIMIZE)    fFrameDecors |= fdMaximize;
    if (decors & MWM_DECOR_MINIMIZE)    fFrameDecors |= fdMinimize;

    if (functions & MWM_FUNC_RESIZE)    fFrameFunctions |= ffResize;
    if (functions & MWM_FUNC_MAXIMIZE)  fFrameFunctions |= ffMaximize;
    if (functions & MWM_FUNC_MINIMIZE)
        fFrameFunctions |= ffMinimize | ffHide | ffShade;
    if (functions & MWM_FUNC_CLOSE) {
        fFrameFunctions |= ffClose;
        fFrameDecors |= fdClose; /* hack */
    }
#else
    fFrameFunctions =
        ffMove | ffResize | ffClose |
        ffMinimize | ffMaximize | ffHide | ffShade;
    fFrameDecors =
        fdTitleBar | fdSysMenu | fdBorder | fdResize |
        fdClose | fdMinimize | fdMaximize;
    fFrameOptions = 0;
#endif

#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;

    getWindowOptions(wo);
    fFrameFunctions &= ~wo.function_mask;
    fFrameFunctions |= wo.functions;
    fFrameDecors &= ~wo.decor_mask;
    fFrameDecors |= wo.decors;
    fFrameOptions &= ~wo.option_mask;
    fFrameOptions |= wo.options;
#endif
}

#ifndef NO_WINDOW_OPTIONS
static void combineOptions(WindowOption &cm, WindowOption &n) {
    if (cm.icon == 0)
        if (n.icon != 0)
            cm.icon = n.icon;
    cm.functions |= n.functions & ~cm.function_mask;
    cm.function_mask |= n.function_mask;
    cm.decors |= n.decors & ~cm.decor_mask;
    cm.decor_mask |= n.decor_mask;
    cm.options |= n.options & ~cm.option_mask;
    cm.option_mask |= n.option_mask;
}

void YFrameWindow::getWindowOptions(WindowOption &opt) {
    memset((void *)&opt, 0, sizeof(opt));
    
    XClassHint *h = client()->classHint();
    WindowOption *wo;
    
    if (!h)
        return;

    if (h->res_name && h->res_class) {
        char *both = (char *) malloc(strlen(h->res_name) + 1 +
                                     strlen(h->res_class) + 1);
        if (both) {
            strcpy(both, h->res_class);
            strcat(both, ".");
            strcat(both, h->res_name);
        }
        wo = both ? getWindowOption(both, 0) : 0;
        if (wo) combineOptions(opt, *wo);
    }
    if (h->res_class) {
        wo = getWindowOption(h->res_class, 0);
        if (wo) combineOptions(opt, *wo);
    }
    if (h->res_name) {
        wo = getWindowOption(h->res_name, 0);
        if (wo) combineOptions(opt, *wo);
    }
    wo = getWindowOption(0, 0);
    if (wo) combineOptions(opt, *wo);
}
#endif

void YFrameWindow::getClientIcon() {
    delete fFrameIcon;
    fFrameIcon = 0;

#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;
    
    getWindowOptions(wo);
    
    if (wo.icon)
        fFrameIcon = getIcon(wo.icon);
#endif
}

YFrameWindow *YFrameWindow::nextLayer() {
    return fNext ? fNext : ((layer() == 1) ? app->top(0) : 0);
}
YFrameWindow *YFrameWindow::prevLayer() {
    return fPrev ? fPrev : ((layer() == 0) ? app->bottom(1) : 0);
}

YMenu *YFrameWindow::windowMenu() {
    //if (frameOptions() & foFullKeys)
    //    return windowMenuNoKeys;
    //else
    return ::windowMenu;
}
