/*
 *  MonKT - a monitor for LinKT
 *  Copyright (C) 1998-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qfiledialog.h>
#include "outputctrl.h"
#include "outputctrl.moc"
#include "output.h"
#include "global.h"
#include "toolbox.h"
#include "main.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <kmsgbox.h>

#define BTN_WIDTH 60
#define BTN_WIDTH_NRS 25
#define BTN_HEIGHT 25


extern TopLevel *toplevel;


OutputCtrlEntry::OutputCtrlEntry( QWidget *parent, int type ) : QObject()
{
   QFont f(config->font_output, config->size_output);

   this->type = type;
   this->parent = parent;
   neu = false;
   nr = -1;
   spyid = -1;
   otherid = -1;
   calls = NULL;
   open = true;
   saving = false;
   fdsave = -1;


   // Der Kanal-Umschalte-Knopf
   btn = new QPushButton( parent );
   connect( btn, SIGNAL(clicked()), this, SLOT(slotBtnClicked()) );

   kontextmenu = new QPopupMenu();
   kontextmenu->insertItem( klocale->translate("&ignore QSO"), this, SLOT(slotIgnoreQSO()));
   mnu_save = kontextmenu->insertItem( klocale->translate("&save QSO"), this, SLOT(slotSaveQSO()));
//   kontextmenu->insertSeparator( -1 );

   switch (type)
   {
      case OCE_TYPE_MONITOR:
           output = new OutputWidget( parent );
           output->setFont(f);
           output->setBackgroundColor( colors[config->colors->background] );
           output->setGeometry( 0, BTN_HEIGHT, parent->width(), parent->height()-BTN_HEIGHT );
           btn->setText("Monitor");
           break;
      case OCE_TYPE_SPY:
           output = new OutputWidget( parent );
           output->setFont(f);
           output->setBackgroundColor( colors[config->colors->background] );
           output->setGeometry( 0, BTN_HEIGHT, parent->width(), parent->height()-BTN_HEIGHT );
           connect( output, SIGNAL(kontextMenu()), this, SLOT(slotKontextMenu()) );
           break;
   }
}


OutputCtrlEntry::~OutputCtrlEntry()
{
   delete output;
   delete btn;
   delete kontextmenu;
}


void OutputCtrlEntry::slotBtnClicked()
{
   ((OutputCtrl *)parent)->chooseChannel( this );
}


//   void OutputCtrlEntry::setSpyData( char *call, t_qsoid spyid, t_qsoid otherid )
//
// Rufzeichen und andere Informationen bei einem Spy setzen. Das Rufzeichen
// wird als Tooltip fuer den Button verwendet.
void OutputCtrlEntry::setSpyData( char *call, t_qsoid spyid, t_qsoid otherid )
{
   QToolTip::add( btn, call );
   calls = (char *) strdup(call);
   this->spyid = spyid;
   this->otherid = otherid;
}


int OutputCtrlEntry::isSpyId( int id )
{
   if (id == spyid) return 1;
   if (id == otherid) return 2;
   return -1;
}


void OutputCtrlEntry::show()
{
   btn->show();

   switch (type)
   {
      case OCE_TYPE_MONITOR:
      case OCE_TYPE_SPY:
           output->show();
           break;
   }
}


void OutputCtrlEntry::hide()
{
   switch (type)
   {
      case OCE_TYPE_MONITOR:
      case OCE_TYPE_SPY:
           output->hide();
           break;
   }
}


void OutputCtrlEntry::setNew( bool neu )
{
   if (neu == this->neu) return;

   QFont f(btn->font());

   if (neu)
      f.setBold( true );
   else
      f.setBold( false );
   btn->setFont(f);
   this->neu = neu;
}


int OutputCtrlEntry::getNr()
{
   return nr;
}


void OutputCtrlEntry::setNr( int nr )
{
   char tmp[10];

   sprintf(tmp, "%i", nr);
   btn->setText( tmp );
   this->nr = nr;
}


char * OutputCtrlEntry::getCalls()
{
   return calls;
}


void OutputCtrlEntry::slotKontextMenu()
{
   kontextmenu->popup(QCursor::pos());
}


void OutputCtrlEntry::slotIgnoreQSO()
{
   ((OutputCtrl *)parent)->closeQSO( this );
}


void OutputCtrlEntry::slotSaveQSO()
{
   char tmp[1000];

   if (saving)
   {
      ::close(fdsave);
      saving = false;
      kontextmenu->changeItem(klocale->translate("&save QSO"), mnu_save);
      ((OutputCtrl *)parent)->chooseChannel( this );
      return;
   }

   strcpy(tmp, getenv("HOME"));
   QString f = QFileDialog::getSaveFileName(tmp);

   // Cancel
   if (f.isEmpty()) return;

   strcpy(tmp, (const char *)f);

   if (file_exist(tmp))
   {
      if (KMsgBox::yesNo(NULL, klocale->translate("Overwrite?"),
                               klocale->translate("The specified file allready exists. Overwrite?"),
                               0,
                               klocale->translate("&Yes"),
                               klocale->translate("&No")) != 1)
         return;
   }

   if ((fdsave = ::open(tmp, O_WRONLY|O_CREAT)) == -1)
   {
      printf("%s (%s)\n", strerror(errno), tmp);
      KMsgBox::message(NULL, klocale->translate("Failure"),
                             klocale->translate("Cannot create specified file!"));
      return;
   }

   // Zugriffsrechte einstellen
   fchmod(fdsave, S_IRUSR|S_IWUSR);

   saving = true;
   kontextmenu->changeItem(klocale->translate("Close &Save"), mnu_save);
   ((OutputCtrl *)parent)->chooseChannel( this );
}


void OutputCtrlEntry::getIDs( t_qsoid & id, t_qsoid & otherid )
{
   id = spyid;
   otherid = this->otherid;
}


bool OutputCtrlEntry::isSaving()
{
   return saving;
}


void OutputCtrlEntry::saveLine( char *data, int len )
{
   int i;
   char *tmp;


   if (saving && fdsave != -1)
   {
      tmp = (char *) malloc(len);
      for (i=0;i<len;i++)
         if (data[i] == '\r')
            tmp[i] = '\n';
         else
            tmp[i] = data[i];
      write(fdsave, tmp, len);
      free(tmp);
   }
}


void OutputCtrlEntry::setOtherId( t_qsoid id )
{
   otherid = id;
}


void OutputCtrlEntry::setFonts()
{
   QFont f(config->font_output, config->size_output);
   output->setFont(f);
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


OutputCtrl::OutputCtrl( QWidget *parent ) : QWidget( parent )
{
   windows = new QList<OutputCtrlEntry>();
   windows->setAutoDelete( true );

   // Das Monitor-Fenster erzeugen
   moni = new OutputCtrlEntry( this, OCE_TYPE_MONITOR );
   windows->append( moni );
   visible = moni;
}


OutputCtrl::~OutputCtrl()
{
   delete windows;
}


void OutputCtrl::resizeEvent(QResizeEvent *e)
{
   OutputCtrlEntry *tmp;
   int i=0;

   if ((tmp = windows->first()) == NULL) return;
   do
   {
      tmp->output->setGeometry( 0, BTN_HEIGHT, width(), height()-BTN_HEIGHT );
      if (i == 0)
         tmp->btn->setGeometry( 0, 0, BTN_WIDTH, BTN_HEIGHT );
      else
         tmp->btn->setGeometry( ((i-1)*BTN_WIDTH_NRS)+BTN_WIDTH+i, 0, BTN_WIDTH_NRS, BTN_HEIGHT );
      i++;
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::repaintButtons()
{
   int i=0;
   OutputCtrlEntry *tmp;

   if ((tmp = windows->first()) == NULL) return;
   do
   {
      if (i == 0)
         tmp->btn->setGeometry( 0, 0, BTN_WIDTH, BTN_HEIGHT );
      else
         tmp->btn->setGeometry( ((i-1)*BTN_WIDTH_NRS)+BTN_WIDTH+i, 0, BTN_WIDTH_NRS, BTN_HEIGHT );
      i++;
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::monitorWrite( char *text, QColor color )
{
   moni->output->writeText(text, color);
   if (visible != moni)
      moni->setNew( true );
}


void OutputCtrl::monitorNewLine()
{
   moni->output->newLine();
   if (visible != moni)
      moni->setNew( true );
}


//   bool OutputCtrl::addSpy( char *tocall, t_qsoid id, t_qsoid otherid )
//
// Oeffnet ein neues Spy-Fenster, wenn es denn noetig ist.
// Rueckgabe:
//    true, wenn der Spy neu ist und ein SPYID-Kommando abgeschickt
//          werden soll,
//    false wenn nicht.
bool OutputCtrl::addSpy( char *tocall, t_qsoid id, t_qsoid otherid )
{
   int nr, at;
   OutputCtrlEntry *spy;

   if ((spy = getSpy( otherid )) == NULL)
   {
      spy = new OutputCtrlEntry( this, OCE_TYPE_SPY );
      spy->setSpyData( tocall, id, otherid );
      getNextNr( nr, at );
      spy->setNr( nr );
      if (at == -1)
         windows->append( spy );
      else
         windows->insert( at, spy );
      spy->hide();
      spy->btn->show();
      repaintButtons();
      return true;
   }

   spy->setOtherId( id );
   return false;
}


void OutputCtrl::chooseChannel( OutputCtrlEntry *entry )
{
   OutputCtrlEntry *tmp;
   char str[100];


   if ((tmp = windows->first()) == NULL) return;
   do
   {
      if (tmp == entry)
      {
         tmp->btn->setDown( true );
         tmp->output->show();
         visible = tmp;
         tmp->setNew( false );
      }
      else
      {
         tmp->btn->setDown( false );
         tmp->output->hide();
      }
   }
   while ((tmp = windows->next()) != NULL);

   if ((entry == moni) || (entry->getCalls() == NULL))
      strcpy(str, klocale->translate("MonKT - the LinKT Monitor"));
   else
      if (entry->isSaving())
         sprintf(str, "%s -- %s - %s", klocale->translate("MonKT - the LinKT Monitor"),
                                       entry->getCalls(),
                                       klocale->translate("saving"));
      else
         sprintf(str, "%s -- %s", klocale->translate("MonKT - the LinKT Monitor"),
                                  entry->getCalls());

   toplevel->setCaption( str );
}


//   void OutputCtrl::getNextNr()
//
// Sucht die naechste freie Nummer in der SPY-Leiste 'raus.
void OutputCtrl::getNextNr( int & nr, int & at )
{
   OutputCtrlEntry *tmp;
   int expect=0, i;

   if ((tmp = windows->first()) == NULL) return;
   do
   {
      i = tmp->getNr();
      if (i > expect && i != -1)
      {
         nr = expect;
         at = windows->at();
         return;
      }
      expect++;
   }
   while ((tmp = windows->next()) != NULL);

   nr = expect;
   at = windows->at();
}


//   void OutputCtrl::spyData( char *data )
//
// Spy-Daten kommen an
void OutputCtrl::spyData( char *data, int len )
{
   OutputCtrlEntry *tmp;
   t_qsoid id;
   char *str1, *str2;
   bool ret;
   int newlen, i, co;


   memcpy(&id, data, sizeof(t_qsoid));
   len -= sizeof(t_qsoid);
   memmove(data, data+sizeof(t_qsoid), len);

   str1 = (char *) malloc(len+1);
   str2 = (char *) malloc(len+1);
   memcpy(str1, data, len);
   str1[len] = '\0';

   if ((tmp = windows->first()) == NULL) return;
   do
   {
      if ((co = tmp->isSpyId(id)) != -1)
      {
         tmp->saveLine(str1, len);
         switch (co)
         {
            case 1: co = config->colors->spyColor1; break;
            case 2: co = config->colors->spyColor2; break;
         }

         do
         {
            ret = false;

            if ((i = POS('\r', str1)) > -1)
            {
               memcpy(str2, str1, i);
               str2[i] = '\0';
               newlen = strlen(str1)-i-1;
               memmove(str1, str1+i+1, newlen);
               str1[newlen] = '\0';
               ret = true;
            }
            else
            {
               strcpy(str2, str1);
               str1[0] = '\0';
            }

            tmp->output->writeText( str2, colors[co] );
            if (visible != tmp)
               tmp->setNew( true );

            if (ret)
            {
               tmp->output->newLine();
               if (visible != tmp)
                  tmp->setNew( true );
            }
         }
         while (str1[0] != '\0');

         free(str1);
         free(str2);
         return;
      }
   }
   while ((tmp = windows->next()) != NULL);

   free(str1);
   free(str2);
}


//   OutputCtrlEntry * OutputCtrl::isOtherQSO( int id )
//
// Sucht den Spy 'raus, der zu dieser uebergebenen QSO-Id gehoert.
OutputCtrlEntry * OutputCtrl::getSpy( t_qsoid id )
{
   OutputCtrlEntry *tmp;

   if ((tmp = windows->first()) == NULL) return NULL;
   do
   {
      if (tmp->isSpyId(id) != -1)
         return tmp;
   }
   while ((tmp = windows->next()) != NULL);

   return NULL;
}


void OutputCtrl::closeQSO( OutputCtrlEntry *w )
{
   char tmp[100];
   t_qsoid id, otherid;


   if (w->open)
   {
      // Wenn der Spy noch offen ist, wollen wir dem ax25spyd sagen, dass
      // wir keine Frames mehr haben wollen.
      w->getIDs( id, otherid );
      sprintf(tmp, "SPYSTOPID %i", id);
      toplevel->sendMonitor(tmp);
      sprintf(tmp, "SPYSTOPID %i", otherid);
      toplevel->sendMonitor(tmp);
   }
   windows->remove( w );
   repaintButtons();

   // Auf den Monitor umschalten
   chooseChannel( moni );
}


void OutputCtrl::spyEnd( t_QSOMheard *data )
{
   OutputCtrlEntry *tmp;
   char str[100];


   if ((tmp = windows->first()) == NULL) return;

   do
   {
      if (tmp->isSpyId(data->qsoid) != -1)
         if (tmp->open)
         {
            tmp->open = false;
            tmp->output->newLine();
            sprintf(str, "<MonKT>: Spy-QSO %s closed.", tmp->getCalls());
            tmp->output->newLine();
            tmp->output->writeText( str, colors[config->colors->spyColorStatus] );
            sprintf(str, "         Total: %i bytes (%i frames)", data->nInfoBytes, data->nFrames);
            tmp->output->newLine();
            tmp->output->writeText( str, colors[config->colors->spyColorStatus] );
            sprintf(str, "         Playload: %i bytes", data->nInfoBytes);
            tmp->output->newLine();
            tmp->output->writeText( str, colors[config->colors->spyColorStatus] );
            tmp->output->newLine();
            if (visible != tmp)
               tmp->setNew( true );
            return;
         }
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::setFonts()
{
   OutputCtrlEntry *tmp;


   if ((tmp = windows->first()) == NULL) return;

   do
   {
      tmp->setFonts();
   }
   while ((tmp = windows->next()) != NULL);
}


void OutputCtrl::slotBildAuf()
{
   visible->output->slotPageUp();
}


void OutputCtrl::slotBildAb()
{
   visible->output->slotPageDown();
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////


QSOListe::QSOListe( QWidget *parent )
{
   setNumCols(1);
   setNumRows(0);
   setCellWidth(0);
   setCellHeight(20);

   setTableFlags( Tbl_clipCellPainting | Tbl_autoVScrollBar );
   setBackgroundMode( PaletteBase );
   setFrameStyle( QFrame::WinPanel | QFrame::Sunken );
   setFocusPolicy( StrongFocus );
}


QSOListe::~QSOListe()
{
}



