/*
 * server.c: Things dealing with server connections, etc. 
 *
 * Written By Michael Sandrof
 *
 * Copyright(c) 1990 
 *
 * See the COPYRIGHT file, or do a HELP IRCII COPYRIGHT 
 */
#ifdef IRIX
#define _HAVE_SIN_LEN 1
#define _HAVE_SA_LEN 1
#define MAXDNAME 100
#endif

#include "irc.h"
#include "struct.h"

#include "parse.h"

#include <stdarg.h>

#include "server.h"
#include "commands.h"
#include "ircaux.h"
#include "input.h"
#include "who.h"
#include "lastlog.h"
#include "exec.h"
#include "window.h"
#include "output.h"
#include "names.h"
#include "hook.h"
#include "vars.h"
#include "screen.h"
#include "notify.h"
#include "misc.h"
#include "status.h"

#ifdef WDIDENT
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#endif


#ifdef IRIX
#undef sa_len
#endif

static	char *	set_umode (int du_index);

const	char *  umodes = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

/* server_list: the list of servers that the user can connect to,etc */
	Server	*server_list = NULL;

/* number_of_servers: in the server list */
	int	number_of_servers = 0;

	int	primary_server = -1;
	int	from_server = -1;
	int	never_connected = 1;		/* true until first connection
						 * is made */
	int	connected_to_server = 0;	/* true when connection is
						 * confirmed */
	int	parsing_server_index = -1;
	int	last_server = -1;

extern	int	dgets_errno;
	int	identd = -1;

#if defined(WINNT) || defined(__EMX__) || defined(WANT_IDENTD)
	int	already_identd = 0;
#endif

int (*serv_open_func) (int, unsigned long, int) = NULL;
int (*serv_output_func) (int, int, char *, int) = NULL;
int (*serv_input_func)  (int, char *, int, int, int) = NULL;
int (*serv_close_func) (int, long, int) = NULL;

QueueSend *serverqueue = NULL;

/*
 * close_server: Given an index into the server list, this closes the
 * connection to the corresponding server.  It does no checking on the
 * validity of the index.  It also first sends a "QUIT" to the server being
 * closed 
 */
void	close_server (int cs_index, char *message)
{
	char	buffer[BIG_BUFFER_SIZE/4 + 1];

	if (cs_index < 0 || cs_index > number_of_servers)
	{
		return;
	}
	if (server_list[cs_index].read == -1)
		return;
		
	if (serv_close_func)
		(*serv_close_func)(cs_index, server_list[cs_index].local_addr.s_addr, server_list[cs_index].port);
	clean_server_queues(from_server);

	if (waiting_out > waiting_in)
		waiting_out = waiting_in = 0;

	if (server_list[cs_index].reconnecting)
	        set_waiting_channel(cs_index);
	else
		clear_channel_list(cs_index);

	clear_link(&server_list[cs_index].server_last);
	clear_link(&server_list[cs_index].tmplink);

	server_list[cs_index].reconnecting = 0;
	server_list[cs_index].operator = 0;
	server_list[cs_index].connected = 0;
	server_list[cs_index].buffer = NULL;

	new_free(&server_list[cs_index].away);
        if (server_list[cs_index].write > -1)
        {
               if (message && *message && !server_list[cs_index].closing)
               {
               		server_list[cs_index].closing = 1;
			if (x_debug & DEBUG_OUTBOUND)
				   yell("Closing server %d because [%s]", 
					   cs_index, message ? message : empty_string);
                         snprintf(buffer, BIG_BUFFER_SIZE, "QUIT :%s\n", message);
                         send(server_list[cs_index].write, buffer, strlen(buffer), 0);
                }
		new_close(server_list[cs_index].write);
        }
	if (server_list[cs_index].read > -1)
		new_close(server_list[cs_index].read);
	server_list[cs_index].write = server_list[cs_index].read = -1;
	if (identd != -1)
		set_socketflags(identd, 0);
#if defined(WINNT) || defined(__EMX__) || defined(WANT_IDENTD)
	already_identd = 0;
#endif
}

int close_all_servers(char *message)
{
	int i;
	for (i = 0; i < number_of_servers; i++)
	{
		server_list[i].reconnecting = 0;
		close_server(i, message);
	}
	return 0;
}

/*
 * set_server_bits: Sets the proper bits in the fd_set structure according to
 * which servers in the server list have currently active read descriptors.  
 */
void	set_server_bits (fd_set *rd, fd_set *wr)
{
	int	i;

	for (i = 0; i < number_of_servers; i++)
	{
		if (server_list[i].read > -1)
			FD_SET(server_list[i].read, rd);
#ifdef NON_BLOCKING_CONNECTS
		if (!(server_list[i].login_flags & (LOGGED_IN|CLOSE_PENDING)) &&
		    server_list[i].write > -1)
			FD_SET(server_list[i].write, wr);
#endif
	}
}

int in_timed_server = 0;

int timed_server (void *args)
{
char *p = (char *)args;
static int retry = 0;
	if (!connected_to_server && number_of_servers && p && *p)
	{
		get_connected(atol(p), from_server);
		bitchsay("Servers exhausted. Restarting. [%d]", ++retry);
	}
	new_free(&p);
	args = NULL;
	in_timed_server = 0;
	return 0;
}

/*
 * do_server: check the given fd_set against the currently open servers in
 * the server list.  If one have information available to be read, it is read
 * and and parsed appropriately.  If an EOF is detected from an open server,
 * one of two things occurs. 1) If the server was the primary server,
 * get_connected() is called to maintain the connection status of the user.
 * 2) If the server wasn't a primary server, connect_to_server() is called to
 * try to keep that connection alive. 
 */
void	do_server (fd_set *rd, fd_set *wr)
{
	char	buffer[BIG_BUFFER_SIZE + 1];
	int	des,
		i;
	static	int	times = 1;
static	time_t	last_timeout = 0;

	for (i = 0; i < number_of_servers; i++)
	{
#ifdef NON_BLOCKING_CONNECTS
		if (((des = server_list[i].write) > -1) && FD_ISSET(des, wr) && !(server_list[i].login_flags & LOGGED_IN))
		{
			struct sockaddr_in sa;
			int salen = sizeof(struct sockaddr_in);
			if (getpeername(des, (struct sockaddr *) &sa, &salen) != -1)
				register_server(i, server_list[i].d_nickname);	
		}
#endif
		if (((des = server_list[i].read) > -1) && FD_ISSET(des, rd))
		{
			int	junk;
			char 	*bufptr;
			errno	= 0;
			last_server = from_server = i;
			bufptr = buffer;
			if (serv_input_func)
				junk = (*serv_input_func)(i, bufptr, des, 1, BIG_BUFFER_SIZE);
			else
				junk = dgets(bufptr, des, 1, BIG_BUFFER_SIZE);
			switch (junk)
			{
				case 0: /* timeout */
					break;
				case -1: /* EOF condition */
				{
					int old_serv = i;
					close_server(i, empty_string);
					last_timeout = 0;
					server_list[i].reconnecting = 1;
					say("Connection closed from %s: %s", server_list[i].name, (dgets_errno == -1) ? "Remote end closed connection" : strerror(dgets_errno));

					if (times++ >= get_int_var(MAX_SERVER_RECONNECT_VAR))
					{
						clean_server_queues(i);
						if(do_hook(DISCONNECT_LIST,"No Connection"))
							put_it("%s", convert_output_format(fget_string_var(FORMAT_DISCONNECT_FSET), "%s %s", update_clock(GET_TIME), "No connection"));
						times = 1;
						i++;
					}
					get_connected(i, old_serv);
					break;
				}
				default:
				{
					last_timeout = 0;
					parsing_server_index = i;
					server_list[i].last_msg = now;
					parse_server(buffer);
					new_free(&server_list[i].buffer);
					parsing_server_index = -1;
					message_from(NULL, LOG_CRAP);
					break;
				}
			}
			from_server = primary_server;
		}
		if (primary_server == -1 || !is_server_connected(primary_server))
			window_check_servers();
		if (server_list[i].read != -1 && (errno == ENETUNREACH || errno == EHOSTUNREACH))
		{
			if (last_timeout == 0)
				last_timeout = now;
			else if (now - last_timeout > 600)
			{
				close_server(i, empty_string);
				get_connected(i, -1);
			}
		}
	}
}

/*
 * find_in_server_list: given a server name, this tries to match it against
 * names in the server list, returning the index into the list if found, or
 * -1 if not found 
 */
extern	int	find_in_server_list (char *server, int port)
{
	int	i,
		len;

	len = strlen(server);
	for (i = 0; i < number_of_servers; i++)
	{
		if (port && server_list[i].port && port != server_list[i].port && port != -1)
			continue;

#define MATCH_WITH_COMPLETION(n1, n2) 			\
{							\
	size_t l1 = strlen(n1);				\
	size_t l2 = strlen(n2);				\
	size_t l3 = l1 > l2 ? l2 : l1;			\
							\
	if (!my_strnicmp(n1, n2, l3))			\
		return i;				\
}


		MATCH_WITH_COMPLETION(server, server_list[i].name);
		
		if (!server_list[i].itsname)
			continue;
		MATCH_WITH_COMPLETION(server, server_list[i].itsname);
	}
	return (-1);
}

/*
 * parse_server_index:  given a string, this checks if it's a number, and if
 * so checks it validity as a server index.  Otherwise -1 is returned 
 */
int	parse_server_index (char *str)
{
	int	i;

	if (is_number(str))
	{
		i = my_atol(str);
		if ((i >= 0) && (i < number_of_servers))
			return (i);
	}
	return (-1);
}

/*
 * This replaces ``get_server_index''.
 */
int 	find_server_refnum (char *server, char **rest)
{
	int 	refnum;
	int	port = irc_port;
	char 	*cport = NULL, 
		*password = NULL,
		*nick = NULL,
		*snetwork = NULL;

	/*
	 * First of all, check for an existing server refnum
	 */
	if ((refnum = parse_server_index(server)) != -1)
		return refnum;
	/*
	 * Next check to see if its a "server:port:password:nick:network"
	 */
	else if (index(server, ':'))
		parse_server_info(server, &cport, &password, &nick, &snetwork);

	else if (index(server, '['))
	{
		int i;
		server++; 
		chop(server, 1);
		if (!server || !*server)
			return from_server;
		for (i = 0; i < number_of_servers; i++)
		{
			if (server && server_list[i].snetwork && !strcasecmp(server, server_list[i].snetwork))
				return i;
		}
	}
	/*
	 * Next check to see if its "server port password nick"
	 */
	else if (rest && *rest)
	{
		cport = next_arg(*rest, rest);
		password = next_arg(*rest, rest);
		nick = next_arg(*rest, rest);
		snetwork = next_arg(*rest, rest);
	}

	if (cport && *cport)
		port = my_atol(cport);

	/*
	 * Add to the server list (this will update the port
	 * and password fields).
	 */
	add_to_server_list(server, port, password, nick, snetwork, 1);
	return from_server;
}


/*
 * add_to_server_list: adds the given server to the server_list.  If the
 * server is already in the server list it is not re-added... however, if the
 * overwrite flag is true, the port and passwords are updated to the values
 * passes.  If the server is not on the list, it is added to the end. In
 * either case, the server is made the current server. 
 */
void 	add_to_server_list (char *server, int port, char *password, char *nick, char *snetwork, int overwrite)
{
extern int default_swatch;
	if ((from_server = find_in_server_list(server, port)) == -1)
	{
		from_server = number_of_servers++;
		RESIZE(server_list, Server, number_of_servers+1);
		memset(&server_list[from_server], 0, sizeof(Server));		
		server_list[from_server].name = m_strdup(server);
		if (snetwork)
			server_list[from_server].snetwork = m_strdup(snetwork);
		server_list[from_server].read = -1;
		server_list[from_server].write = -1;
		server_list[from_server].lag = -1;
		server_list[from_server].motd = 1;
		server_list[from_server].ircop_flags = default_swatch;
		server_list[from_server].port = port;
		malloc_strcpy(&server_list[from_server].umodes, umodes);
		if (password && *password)
			malloc_strcpy(&(server_list[from_server].password), password);

		if (nick && *nick)
			malloc_strcpy(&(server_list[from_server].d_nickname), nick);
		else if (!server_list[from_server].d_nickname)
			malloc_strcpy(&(server_list[from_server].d_nickname), nickname);

		make_notify_list(from_server);
		set_umode(from_server);
	}
	else
	{
		if (overwrite)
		{
			server_list[from_server].port = port;
			if (password || !server_list[from_server].password)
			{
				if (password && *password)
					malloc_strcpy(&(server_list[from_server].password), password);
				else
					new_free(&(server_list[from_server].password));
			}
			if (nick || !server_list[from_server].d_nickname)
			{
				if (nick && *nick)
					malloc_strcpy(&(server_list[from_server].d_nickname), nick);
				else
					new_free(&(server_list[from_server].d_nickname));
			}
		}
		if (strlen(server) > strlen(server_list[from_server].name))
			malloc_strcpy(&(server_list[from_server].name), server);
	}
}

void 	remove_from_server_list (int i)
{
	Window	*tmp = NULL;

	if (i < 0 || i >= number_of_servers)
		return;

	say("Deleting server [%d]", i);

	clean_server_queues(i);

	new_free(&server_list[i].name);
	new_free(&server_list[i].snetwork);
	new_free(&server_list[i].itsname);
	new_free(&server_list[i].password);
	new_free(&server_list[i].away);
	new_free(&server_list[i].version_string);
	new_free(&server_list[i].nickname);
	new_free(&server_list[i].s_nickname);
	new_free(&server_list[i].d_nickname);
	new_free(&server_list[i].umodes);
	
	/* 
	 * this should save a coredump.  If number_of_servers drops
	 * down to zero, then trying to do a realloc ends up being
	 * a free, and accessing that is a no-no.
	 */
	if (number_of_servers == 1)
	{
		say("Sorry, the server list is empty and I just dont know what to do.");
		irc_exit(1, NULL, NULL);
	}

	memmove(&server_list[i], &server_list[i + 1], (number_of_servers - i - 1) * sizeof(Server));
	number_of_servers--;
	RESIZE(server_list, Server, number_of_servers);

	/* update all he structs with server in them */
	channel_server_delete(i);
	exec_server_delete(i);
        if (i < primary_server)
                --primary_server;
        if (i < from_server)
                --from_server;
	while ((traverse_all_windows(&tmp)))
		if (tmp->server > i)
			tmp->server--;
}



/*
 * parse_server_inFo:  This parses a single string of the form
 * "server:portnum:password:nickname:snetwork".  It the points port to the portnum
 * portion and password to the password portion.  This chews up the original
 * string, so * upon return, name will only point the the name.  If portnum
 * or password are missing or empty,  their respective returned value will
 * point to null. 
 */
void	parse_server_info (char *name, char **port, char **password, char **nick, char **snetwork)
{
	char *ptr;

	*port = *password = *nick = NULL;
	if ((ptr = (char *) strchr(name, ':')) != NULL)
	{
		*(ptr++) = (char) 0;
		if (strlen(ptr) == 0)
			*port = NULL;
		else
		{
			*port = ptr;
			if ((ptr = (char *) strchr(ptr, ':')) != NULL)
			{
				*(ptr++) = (char) 0;
				if (strlen(ptr) == 0)
					*password = 0;
				else
				{
					*password = ptr;
					if ((ptr = (char *) strchr(ptr, ':'))
							!= NULL)
					{
						*(ptr++) = 0;
						if (!strlen(ptr))
							*nick = NULL;
						else
						{
							*nick = ptr;
							if ((ptr = strchr(ptr, ':')) !=NULL)
							{
								*(ptr++) = 0;
								if  (!strlen(ptr))
									*snetwork = NULL;
								else
									*snetwork = ptr;
							}
						}
					}
				}
			}
		}
	}
}

/*
 * build_server_list: given a whitespace separated list of server names this
 * builds a list of those servers using add_to_server_list().  Since
 * add_to_server_list() is used to added each server specification, this can
 * be called many many times to add more servers to the server list.  Each
 * element in the server list case have one of the following forms: 
 *
 * servername 
 * servername:port 
 * servername:port:password 
 * servername::password 
 * servernetwork
 * servername:port:password:nick:servernetwork
 * Note also that this routine mucks around with the server string passed to it,
 * so make sure this is ok 
 */
static char *default_network = NULL;

int	build_server_list (char *servers)
{
	char	*host,
		*rest,
		*password = NULL,
		*port = NULL,
		*nick = NULL,
		*snetwork = NULL;

	int	port_num;
	int	i = 0;
	
	if (servers == NULL)
		return 0;

	while (servers)
	{
		if ((rest = (char *) strchr(servers, '\n')) != NULL)
			*rest++ = 0;
		while ((host = new_next_arg(servers, &servers)) != NULL)
		{
			if (!*host)
				break;
			if (*host == '[')
			{
				host++;
				if (host[strlen(host)-1] != ']' && servers && *servers)
				{
					char *ptr = NULL;
					host[strlen(host)] = ' ';
					if ((ptr = MatchingBracket(host, '[', ']')))
					{
						*ptr++ = 0; 
						servers = ptr;
					}
				}
				if (host[strlen(host)-1] == ']')
					chop(host, 1);
				malloc_strcpy(&default_network, host);
				snetwork = NULL;
				continue;
			}
			parse_server_info(host, &port, &password, &nick, &snetwork);
                        if (port && *port)
                        {
                                port_num = my_atol(port);
                                if (!port_num)
                                        port_num = irc_port;
                        }
			else
				port_num = irc_port;

			add_to_server_list(host, port_num, password, nick, snetwork ? snetwork : default_network, 0);
			i++;
		}
		servers = rest;
	}
	return i;
}

int read_and_parse_server(char **filename, char *buffer)
{
FILE *fp;
int i = 0;
	if ((fp = uzfopen(filename, "r", 0)))
	{
		char *p;
		while (fgets(buffer, BIG_BUFFER_SIZE, fp))
		{
			chop(buffer, 1);
			if ((p = strchr(buffer, '#')))
				*p = 0;
			i += build_server_list(buffer);
		}
		fclose(fp);
	} 
	return i;
}

/*
 * read_server_file: reads hostname:portnum:password:network server information from
 * a file and adds this stuff to the server list.  See build_server_list()/ 
 */
int read_server_file (char *servers_file)
{
	int some = 0;
	char *file_path = NULL;
	char	buffer[BIG_BUFFER_SIZE + 1];
	int old_window_display = window_display;
	char    *expanded;

	window_display = 0;

	if (getenv("IRC_SERVERS_FILE"))
	{
		malloc_strcpy(&file_path, getenv("IRC_SERVERS_FILE"));
		expanded = expand_twiddle(file_path);
		some = read_and_parse_server(&expanded, buffer);
		new_free(&file_path);
		new_free(&expanded);
	}

#ifdef SERVERS_FILE
	if (SERVERS_FILE[0] == '/')
		malloc_strcpy(&file_path, irc_lib);
	malloc_strcat(&file_path, SERVERS_FILE);
	some += read_and_parse_server(&file_path, buffer);
	new_free(&file_path);
#endif	

	if (*servers_file == '/')
		file_path = m_strdup(servers_file);
	else
		malloc_sprintf(&file_path, "~/%s", servers_file);
	some += read_and_parse_server(&file_path, buffer);
	new_free(&file_path);

	window_display = old_window_display;
	return some;
}

/*
 * connect_to_server_direct: handles the tcp connection to a server.  If
 * successful, the user is disconnected from any previously connected server,
 * the new server is added to the server list, and the user is registered on
 * the new server.  If connection to the server is not successful,  the
 * reason for failure is displayed and the previous server connection is
 * resumed uniterrupted. 
 *
 * This version of connect_to_server() connects directly to a server 
 */
static	int	connect_to_server_direct (char *server_name, int port)
{
	int		new_des;
struct sockaddr_in 	localaddr;
	int		address_len;
	unsigned short	this_sucks;
#ifdef WDIDENT
	struct stat sb;
	struct passwd *pw;
	char lockfile[1024];
	FILE *fp;
	char candofilestuff=0;	
	struct hostent *hp=NULL;
	struct sockaddr_in raddr;
#endif

	oper_command = 0;
	errno = 0;
	memset(&localaddr, 0, sizeof(localaddr));
	this_sucks = (unsigned short)port;

#ifdef WDIDENT
	pw=getpwuid(getuid());
	if(!pw)
		goto noidentwd;
	sprintf(lockfile, "%s/.identwd", pw->pw_dir);
	
	if(*server_name=='/')
		goto noidentwd;
	
	if(stat(lockfile, &sb))
		goto noidentwd;
	
	if(!(sb.st_mode & S_IFDIR))
		goto noidentwd;
	
	hp=resolv(server_name);
	if(!hp)
		goto noidentwd;
	if(hp->h_addrtype != AF_INET)
		goto noidentwd;
	
	memcpy(&raddr.sin_addr, hp->h_addr, hp->h_length);
	sprintf(lockfile, "%s/.identwd/%s.%i.LOCK", pw->pw_dir,
		inet_ntoa((struct in_addr)raddr.sin_addr), port);
	if ((fp=fopen(lockfile, "w")))
	{
		fprintf(fp, "WAIT\n");
		fclose(fp);
		candofilestuff=1;
	}
noidentwd:
#endif
	new_des = connect_by_number(server_name, &this_sucks, SERVICE_CLIENT, PROTOCOL_TCP, 0);
	port = this_sucks;
	if (new_des < 0)
	{
		if (x_debug)
			say("new_des is %d", new_des);
		say("Unable to connect to port %d of server %s: %s", port,
				server_name, errno ? strerror(errno) :"unknown host");
		if ((from_server != -1)&& (server_list[from_server].read != -1))
			say("Connection to server %s resumed...", server_list[from_server].name);
#ifdef WDIDENT
	if(candofilestuff)
		remove(lockfile);
#endif
		return (-1);
	}

	if (*server_name != '/')
	{
		address_len = sizeof(struct sockaddr_in);
		getsockname(new_des, (struct sockaddr *) &localaddr, &address_len);
	}
#ifdef WDIDENT
	if(candofilestuff && (fp = fopen(lockfile, "w")) )
	{
		fprintf(fp, "%s %i %s", inet_ntoa(localaddr.sin_addr),
			htons(localaddr.sin_port), username);
		fclose(fp);
	}
#endif

	update_all_status(current_window, NULL, 0);
	add_to_server_list(server_name, port, NULL, NULL, NULL, 1);

	server_list[from_server].closing = 0;
	if (port)
	{
		server_list[from_server].read = new_des;
		server_list[from_server].write = new_des;
	}
	else
		server_list[from_server].read = new_des;

	new_open(new_des);
        server_list[from_server].local_addr.s_addr = localaddr.sin_addr.s_addr;
	server_list[from_server].operator = 0;
	if (identd != -1)
		set_socketflags(identd, now);
	return (0);
}

int 	connect_to_server_by_refnum (int refnum, int c_server)
{
	char *sname;
	int sport;
	int conn;
	if (refnum == -1)
	{
		say ("Connecting to refnum -1.  That makes no sense.");
		return -1;		/* XXXX */
	}

	sname = server_list[refnum].name;
	sport = server_list[refnum].port;

	if (server_list[refnum].read == -1)
	{
		if (sport == -1)
			sport = irc_port;

		from_server = refnum;
		say("Connecting to port %d of server %s [refnum %d]", sport, sname, refnum);
		conn = connect_to_server_direct(sname, sport);

		if (conn)
			return -1;

		if (serv_open_func)
			(*serv_open_func)(from_server, server_list[from_server].local_addr.s_addr, server_list[from_server].port);
		if ((c_server != -1) && (c_server != from_server))
		{
			server_list[c_server].reconnecting =1;
			close_server(c_server, "changing servers");
		}
		
		if (!server_list[from_server].d_nickname)
			malloc_strcpy(&(server_list[from_server].d_nickname), nickname);

		if (server_list[from_server].password)
			send_to_server("PASS %s", server_list[from_server].password);

		register_server(from_server, server_list[from_server].d_nickname);
		server_list[refnum].last_msg = now;
		server_list[refnum].eof = 0;
		server_list[refnum].connected = 1;
		*server_list[refnum].umode = 0;
		server_list[refnum].operator = 0;
		set_umode(refnum);
	}
	else
	{
		say("Connected to port %d of server %s", sport, sname);
		from_server = refnum;
	}
	message_from(NULL, LOG_CRAP);
	update_all_status(current_window, NULL, 0);
	return 0;
}

int next_server(int server)
{
   int previous_server = server;

   server++;
   if (server == number_of_servers)
      server = 0;


   if(get_int_var(SERVER_GROUPS_VAR) != 0 && server_list[previous_server].snetwork != NULL)
      {
      while(strcmp(server_list[server].snetwork, server_list[previous_server].snetwork) != 0)
         {
            server++;
	    if (server == number_of_servers)
	       server = 0;
         }
      }
   return server;
}

/*
 * get_connected: This function connects the primary server for IRCII.  It
 * attempts to connect to the given server.  If this isn't possible, it
 * traverses the server list trying to keep the user connected at all cost.  
 */
void	get_connected (int server, int old_server)
{
	int	s,
		ret = -1;
	int	times = 0;

	if (server_list)
	{
		if (server >= number_of_servers)
			server = 0;
		else if (server < 0)
			server = number_of_servers - 1;

		s = server;
		if (connect_to_server_by_refnum(server, primary_server))
		{
			int this_loop = 1;
			times = 1;
			while (server_list[server].read == -1)
			{
				times++;
				from_server = server;
				ret = connect_to_server_by_refnum(server, old_server);
				if (++this_loop < get_int_var(MAX_SERVER_RECONNECT_VAR))
					continue;
				this_loop = 0;
				server = next_server(server);
				if (server == s && (times > (get_int_var(MAX_SERVER_RECONNECT_VAR) * (number_of_servers - 1))))
				{
				    clean_server_queues(server);
				    from_server = -1;
				    return;
				}
			}
		}
		times = 0;
		change_server_channels(old_server, from_server);
		set_window_server(-1, from_server, 1);
		server_list[from_server].reconnects++;
	}
	else
	{
                if (do_hook(DISCONNECT_LIST,"No Server List"))
			put_it("%s", convert_output_format(fget_string_var(FORMAT_DISCONNECT_FSET), "%s %s", update_clock(GET_TIME), "You are not connected to a server. Use /SERVER to connect."));
	}
}

/* display_server_list: just guess what this does */
void display_server_list (void)
{
	int	i;
	char *netw = NULL;
	
	if (server_list)
	{
		if (from_server != -1)
			say("Current server: %s %d",
					server_list[from_server].name,
					server_list[from_server].port);
		else
			say("Current server: <None>");

		if (primary_server != -1)
			say("Primary server: %s %d",
				server_list[primary_server].name,
				server_list[primary_server].port);
		else
			say("Primary server: <None>");

		say("Server list:");
		for (i = 0; i < number_of_servers; i++)
		{
			if (!netw && server_list[i].snetwork)
			{
				netw = server_list[i].snetwork;
				say("[%s]", netw);
			}
			else if (!netw && !server_list[i].snetwork)
			{
				netw = "unknown";
				say("[%s]", netw);
			}
			else if (netw && server_list[i].snetwork && strcasecmp(netw, server_list[i].snetwork))
			{
				netw = server_list[i].snetwork;
				say("[%s]", netw);
			}
			if (!server_list[i].nickname)
			{
				if (server_list[i].read == -1)
					say("\t%2d) %s %d", i,
						server_list[i].name,
						server_list[i].port);
				else
					say("\t%2d) %s %d", i,
						server_list[i].name,
						server_list[i].port);
			}
			else
			{
				if (server_list[i].read == -1)
					say("\t%2d) %s %d (was %s)", i,
						server_list[i].name,
						server_list[i].port,
						server_list[i].nickname);
				else
					say("\t%2d) %s %d (%s)", i,
						server_list[i].name,
						server_list[i].port,
						server_list[i].nickname);
			}
		}
	}
	else
		say("The server list is empty");
}

/*
 * set_server_password: this sets the password for the server with the given
 * index.  If password is null, the password for the given server is returned 
 */
char	*set_server_password (int ssp_index, char *password)
{

	if (server_list)
	{
		if (password)
			malloc_strcpy(&(server_list[ssp_index].password), password);
		return (server_list[ssp_index].password);
	}
	else
		return (NULL);
}

/* server_list_size: returns the number of servers in the server list */
int server_list_size (void)
{
	return (number_of_servers);
}

/*
 * server: the /SERVER command. Read the SERVER help page about 
 */
BUILT_IN_COMMAND(servercmd)
{
	char	*server = NULL;
	int	i;

	if (!(server = next_arg(args, &args)))
	{
		display_server_list();
		return;
	}
	
	/*
	 * Delete an existing server
	 */
	if (strlen(server) > 1 && !my_strnicmp(server, "-DELETE", strlen(server)))
	{
		if ((server = next_arg(args, &args)) != NULL)
		{
			if ((i = parse_server_index(server)) == -1)
			{
				if ((i = find_in_server_list(server, 0)) == -1)
				{
					say("No such server in list");
					return;
				}
			}
			if (server_list[i].connected)
			{
				say("Can not delete server that is already open");
				return;
			}
			remove_from_server_list(i);
		}
		else
			say("Need server number for -DELETE");
	}
		/*
	 * Add a server, but dont connect
	 */
	else if (strlen(server) > 1 && !my_strnicmp(server, "-ADD", strlen(server)))
	{
		if ((server = new_next_arg(args, &args)))
			(void) find_server_refnum(server, &args);
		else
			say("Need server info for -ADD");
	}
		/*
	 * The difference between /server +foo.bar.com  and
	 * /window server foo.bar.com is that this can support
	 * doing a server number.  That makes it a tad bit more
	 * difficult to parse, too. :P  They do the same thing,
	 * though.
	 */
	else if (*server == '+')
	{
		if (*++server)
		{
			i = find_server_refnum(server, &args);
			if (!connect_to_server_by_refnum(i, -1))
				set_window_server(0, i, 0);
		}
		else
			get_connected(primary_server + 1, from_server);
	}
	/*
	 * You can only detach a server using its refnum here.
	 */
	else if (*server == '-')
	{
		if (*++server)
		{
			i = find_server_refnum(server, &args);
			if (i == primary_server)
			{
			    say("You can't close your primary server!");
			    return;
			}
			close_server(i, "closing server");
			window_check_servers();
		}
		else
			get_connected(from_server - 1, from_server);
	}
	/*
	 * Just a naked /server with no flags
	 */
	else
	{
		int j = from_server;
		i = find_server_refnum(server, &args);
		if (!connect_to_server_by_refnum(i, j))
		{
			if (j != -1 > -1 && i != j &&
			    !get_server_away(i) && get_server_away(j))
			    	set_server_away(i, get_server_away(j), 0);
			set_window_server(-1, from_server, 1);
			change_server_channels(j, i);
		}
	}
}

/*
 * flush_server: eats all output from server, until there is at least a
 * second delay between bits of servers crap... useful to abort a /links. 
 */
void flush_server (void)
{
	fd_set rd;
	struct timeval timeout;
	int	flushing = 1;
	int	des;
	char	buffer[BIG_BUFFER_SIZE + 1];

	if ((des = server_list[from_server].read) == -1)
		return;
	timeout.tv_usec = 0;
	timeout.tv_sec = 1;
	while (flushing)
	{
		FD_ZERO(&rd);
		FD_SET(des, &rd);
		switch (new_select(&rd, NULL, &timeout))
		{
			case -1:
			case 0:
				flushing = 0;
				break;
			default:
				if (FD_ISSET(des, &rd))
				{
					if (!dgets(buffer, des, 0, BIG_BUFFER_SIZE))
						flushing = 0;
				}
				break;
		}
	}
	/* make sure we've read a full line from server */
	FD_ZERO(&rd);
	FD_SET(des, &rd);
	if (new_select(&rd, NULL, &timeout) > 0)
		dgets(buffer, des, 1, BIG_BUFFER_SIZE);
}


static char *set_umode (int du_index)
{
	char *c = server_list[du_index].umode;
	long flags = server_list[du_index].flags;
	long flags2 = server_list[du_index].flags2;
	int i;
	
	for (i = 0; umodes[i]; i++)
	{
		if (umodes[i] == 'o')
			continue;
		if (i > 31)
		{
			if (flags2 & (0x1 << (i - 32)))
				*c++ = server_list[du_index].umodes[i];
		}
		else
		{
			if (flags & (0x1 << i))
				*c++ = server_list[du_index].umodes[i];
		}
	}
	if (get_server_operator(du_index))
		*c++ = 'O';
	*c = 0;	
	return server_list[du_index].umode;
}

char *get_possible_umodes (int gu_index)
{
	if (gu_index == -1)
		gu_index = primary_server;
	else if (gu_index >= number_of_servers)
		return empty_string;

	return server_list[gu_index].umodes;
}

char *get_umode (int gu_index)
{
	if (gu_index == -1)
		gu_index = primary_server;
	else if (gu_index >= number_of_servers)
		return empty_string;

	return server_list[gu_index].umode;
}

void clear_user_modes (int gindex)
{
	if (gindex == -1)
		gindex = primary_server;
	else if (gindex >= number_of_servers)
		return;
	server_list[gindex].flags = 0;
	server_list[gindex].flags2 = 0;
	set_umode(gindex);
}

/*
 * Encapsulates everything we need to change our AWAY status.
 * This improves greatly on having everyone peek into that member.
 * Also, we can deal centrally with someone changing their AWAY
 * message for a server when we're not connected to that server
 * (when we do connect, then we send out the AWAY command.)
 * All this saves a lot of headaches and crashes.
 */
void	set_server_away (int ssa_index, char *message, int silent)
{
	int old_from_server = from_server;

	from_server = ssa_index;
	if (ssa_index == -1)
		say("You are not connected to a server.");
	else if (message && *message)
	{
		if (server_list[ssa_index].away != message)
			malloc_strcpy(&server_list[ssa_index].away, message);

		if (!server_list[ssa_index].awaytime)
			server_list[ssa_index].awaytime = now;

		if (get_int_var(MSGLOG_VAR))
			log_toggle(1, NULL);
		if (!server_list[ssa_index].connected)
		{
			from_server = old_from_server;
			return;
		}
		if (fget_string_var(FORMAT_AWAY_FSET) && !silent)
		{
			char buffer[BIG_BUFFER_SIZE+1];
			if (get_int_var(SEND_AWAY_MSG_VAR))
			{
				char *p = NULL;
				ChannelList *chan;
				for (chan = server_list[ssa_index].chan_list; chan; chan = chan->next)
					m_s3cat(&p, ",", chan->channel);
				if (p)
					send_to_server("PRIVMSG %s :ACTION %s", p, 
						stripansicodes(convert_output_format(fget_string_var(FORMAT_AWAY_FSET), "%s [\002BX\002-MsgLog %s] %s",update_clock(GET_TIME), get_int_var(MSGLOG_VAR)?"On":"Off", message)));
				new_free(&p);
			}
			send_to_server("%s :%s", "AWAY", stripansicodes(convert_output_format(fget_string_var(FORMAT_AWAY_FSET), "%s [\002BX\002-MsgLog %s] %s", update_clock(GET_TIME), get_int_var(MSGLOG_VAR)?"On":"Off",message)));
			strncpy(buffer, convert_output_format(fget_string_var(FORMAT_SEND_ACTION_FSET), "%s %s $C ", update_clock(GET_TIME), server_list[ssa_index].nickname), BIG_BUFFER_SIZE);
			strncat(buffer, convert_output_format(fget_string_var(FORMAT_AWAY_FSET), "%s [\002BX\002-MsgLog %s] %s", update_clock(GET_TIME), get_int_var(MSGLOG_VAR)?"On":"Off", message), BIG_BUFFER_SIZE);
			put_it("%s", buffer);
		}
		else
			send_to_server("%s :%s", "AWAY", stripansicodes(convert_output_format(message, NULL)));
	}
	else
	{
		server_list[ssa_index].awaytime = 0;
		new_free(&server_list[ssa_index].away);
		if (server_list[ssa_index].connected)
			send_to_server("AWAY :");
	}
	from_server = old_from_server;
}

char *	get_server_away (int gsa_index)
{
	if (gsa_index == -1)
		return NULL;
	if (gsa_index == -2)
	{
		int	i;
		for (i = 0; i < number_of_servers; i++)
		{
			if (server_list[i].connected && server_list[i].away)
				return server_list[i].away;
		}
		return NULL;
	}
	if (gsa_index < 0 || gsa_index > number_of_servers)
		return NULL;
	return server_list[gsa_index].away;
}


void	set_server_flag (int ssf_index, int flag, int value)
{
	if (ssf_index == -1)
		ssf_index = primary_server;
	else if (ssf_index >= number_of_servers)
		return;
	if (flag > 31)
	{
		if (value)
			server_list[ssf_index].flags2 |= 0x1 << (flag - 32);
		else
			server_list[ssf_index].flags2 &= ~(0x1 << (flag - 32));
	}
	else
	{
		if (value)
			server_list[ssf_index].flags |= 0x1 << flag;
		else
			server_list[ssf_index].flags &= ~(0x1 << flag);
	}
	set_umode(ssf_index);
}

int	get_server_flag (int gsf_index, int value)
{
	if (gsf_index == -1)
		gsf_index = primary_server;
	else if (gsf_index >= number_of_servers)
		return 0;
	if (value > 31)
		return server_list[gsf_index].flags2 & (0x1 << (value - 32));
	else
		return server_list[gsf_index].flags & (0x1 << value);
}

/*
 * set_server_version: Sets the server version for the given server type.  A
 * zero version means pre 2.6, a one version means 2.6 aso. (look server.h
 * for typedef)
 */
void	set_server_version (int ssv_index, int version)
{
	if (ssv_index == -1)
		ssv_index = primary_server;
	else if (ssv_index >= number_of_servers)
		return;
	server_list[ssv_index].version = version;
}

/*
 * get_server_version: returns the server version value for the given server
 * index 
 */
int	get_server_version (int gsv_index)
{
	if (gsv_index == -1)
		gsv_index = primary_server;
	else if (gsv_index >= number_of_servers)
		return 0;
	if (gsv_index == -1)
		return 0;
	return (server_list[gsv_index].version);
}

/* get_server_name: returns the name for the given server index */
char	*get_server_name (int gsn_index)
{
	if (gsn_index == -1)
		gsn_index = primary_server;
	if (gsn_index == -1 || gsn_index >= number_of_servers)
		return empty_string;

	return (server_list[gsn_index].name);
}

char	*get_server_network (int gsn_index)
{
	if (gsn_index == -1)
		gsn_index = primary_server;
	if (gsn_index == -1 || gsn_index >= number_of_servers)
		return empty_string;

	return (server_list[gsn_index].snetwork);
}

/* get_server_itsname: returns the server's idea of its name */
char	*get_server_itsname (int gsi_index)
{
	if (gsi_index==-1)
		gsi_index=primary_server;
	else if (gsi_index >= number_of_servers)
		return empty_string;

	/* better check gsi_index for -1 here CDE */
	if (gsi_index == -1)
		return empty_string;

	if (server_list[gsi_index].itsname)
		return server_list[gsi_index].itsname;
	else
		return server_list[gsi_index].name;
}

void	set_server_itsname (int ssi_index, char *name)
{
	if (ssi_index==-1)
		ssi_index=primary_server;
	else if (ssi_index >= number_of_servers)
		return;

	malloc_strcpy(&server_list[ssi_index].itsname, name);
}

/*
 * is_server_open: Returns true if the given server index represents a server
 * with a live connection, returns false otherwise 
 */
int	is_server_open (int iso_index)
{
	if (iso_index < 0 || iso_index >= number_of_servers) 
		return (0);
	return (server_list[iso_index].read != -1);
}

/*
 * is_server_connected: returns true if the given server is connected.  This
 * means that both the tcp connection is open and the user is properly
 * registered 
 */
int	is_server_connected (int isc_index)
{
	if (isc_index >= 0 && isc_index < number_of_servers)
		return (server_list[isc_index].connected);
	return 0;
}

void	clear_sent_to_server (int servnum)
{
	server_list[servnum].sent = 0;
}

int	sent_to_server (int servnum)
{
	return server_list[servnum].sent;
}


/* get_server_port: Returns the connection port for the given server index */
int	get_server_port (int gsp_index)
{
	if (gsp_index == -1)
		gsp_index = primary_server;
	else if (gsp_index >= number_of_servers)
		return 0;

	return (server_list[gsp_index].port);
}

/*
 * get_server_nickname: returns the current nickname for the given server
 * index 
 */
char	*get_server_nickname (int gsn_index)
{
	if (gsn_index >= number_of_servers)
		return empty_string;
	else if (gsn_index != -1 && server_list[gsn_index].nickname)
		return (server_list[gsn_index].nickname);
	else
		return "<Nickname not registered yet>";
}


/* 
 * set_server2_8 - set the server as a 2.8 server 
 * This is used if we get a 001 numeric so that we dont bite on
 * the "kludge" that ircd has for older clients
 */
void 	set_server2_8 (int ss2_index, int value)
{
	if (ss2_index < number_of_servers)
		server_list[ss2_index].server2_8 = value;
	return;
}

/* get_server2_8 - get the server as a 2.8 server */
int 	get_server2_8 (int gs2_index)
{
	if (gs2_index == -1)
		gs2_index = primary_server;
	else if (gs2_index >= number_of_servers)
		return 0;
	return (server_list[gs2_index].server2_8);
}
	
/*
 * get_server_operator: returns true if the user has op privs on the server,
 * false otherwise 
 */
int	get_server_operator (int gso_index)
{
	if ((gso_index < 0) || (gso_index >= number_of_servers))
		return 0;
	return (server_list[gso_index].operator);
}

/*
 * set_server_operator: If flag is non-zero, marks the user as having op
 * privs on the given server.  
 */
void	set_server_operator (int sso_index, int flag)
{
	if (sso_index < 0 || sso_index >= number_of_servers)
		return;
	server_list[sso_index].operator = flag;
	oper_command = 0;
	set_umode(sso_index);
}

/*
 * set_server_nickname: sets the nickname for the given server to nickname.
 * This nickname is then used for all future connections to that server
 * (unless changed with NICK while connected to the server 
 */
void 	set_server_nickname (int ssn_index, char *nick)
{
	if (ssn_index != -1 && ssn_index < number_of_servers)
	{
		malloc_strcpy(&(server_list[ssn_index].nickname), nick);
		if (ssn_index == primary_server)
			strmcpy(nickname,nick, NICKNAME_LEN );
	}
	update_all_status(current_window, NULL, 0);
}


void 	set_server_redirect (int s, const char *who)
{
	malloc_strcpy(&server_list[s].redirect, who);
}

char	*get_server_redirect(int s)
{
	return server_list[s].redirect;
}

int	check_server_redirect (char *who)
{
	if (!who || !server_list[from_server].redirect)
		return 0;

	if (!strncmp(who, "***", 3) && !strcmp(who+3, server_list[from_server].redirect))
	{
		set_server_redirect(from_server, NULL);
		return 1;
	}

	return 0;
}

void	register_server (int ssn_index, char *nick)
{
	int old_from_server = from_server;
	if (server_list[ssn_index].password)
		my_send_to_server(ssn_index, "PASS %s", server_list[ssn_index].password);
		
	my_send_to_server(ssn_index, "USER %s %s %s :%s", username, 
			(send_umode && *send_umode) ? send_umode : 
			(LocalHostName?LocalHostName:hostname), 
			username, *realname ? realname : space);

	change_server_nickname(ssn_index, nick);

	server_list[ssn_index].login_flags &= ~LOGGED_IN;
	server_list[ssn_index].login_flags &= ~CLOSE_PENDING;
	server_list[ssn_index].last_msg = now;
	server_list[ssn_index].eof = 0;
	server_list[ssn_index].connected = 1;
	*server_list[ssn_index].umode = 0;
	server_list[ssn_index].operator = 0;
	set_umode(ssn_index);
        server_list[ssn_index].login_flags |= LOGGED_IN;
	from_server = old_from_server;
}

void	set_server_cookie (int ssm_index, char *cookie)
{
	if (ssm_index != -1 && ssm_index < number_of_servers && cookie)
		malloc_strcpy(&server_list[ssm_index].cookie, cookie);
}

char	*get_server_cookie(int ssm_index)
{
	char *s = NULL;
	if (ssm_index != -1 && ssm_index < number_of_servers)
		s = server_list[ssm_index].cookie;
	return s;
}

void 	set_server_motd (int ssm_index, int flag)
{
	if (ssm_index != -1 && ssm_index < number_of_servers)
		server_list[ssm_index].motd = flag;
}

int	get_server_lag (int gso_index)
{
	if ((gso_index < 0 || gso_index >= number_of_servers))
		return 0;
	return(server_list[gso_index].lag);
}

void set_server_lag (int gso_index, int secs)
{
	if ((gso_index != -1 && gso_index < number_of_servers))
		server_list[gso_index].lag = secs;
}

int 	get_server_motd (int gsm_index)
{
	if (gsm_index != -1 && gsm_index < number_of_servers)
		return(server_list[gsm_index].motd);
	return (0);
}

void 	server_is_connected (int sic_index, int value)
{
	if (sic_index < 0 || sic_index >= number_of_servers)
		return;

	server_list[sic_index].connected = value;
	if (value)
		server_list[sic_index].eof = 0;
}

void	set_server_version_string (int servnum, const char *ver)
{
	malloc_strcpy(&server_list[servnum].version_string, ver);
}

char *	get_server_version_string (int servnum)
{
	 return server_list[servnum].version_string;
}

unsigned long get_server_ircop_flags(int servnum)
{
	if (servnum >= 0 && (servnum <= number_of_servers))
		return server_list[servnum].ircop_flags;
	return 0;
}

void set_server_ircop_flags(int servnum, unsigned long flag)
{
	if (servnum < 0 || servnum >= number_of_servers)
		return;
	server_list[servnum].ircop_flags = flag;
}


char	*get_server_userhost (int gsu_index)
{
	if (gsu_index >= number_of_servers)
		return empty_string;
	else if (gsu_index != -1 && server_list[gsu_index].userhost)
		return (server_list[gsu_index].userhost);
	else
		return get_userhost();
}

/*
 * got_my_userhost -- callback function, XXXX doesnt belong here
 */
void 	got_my_userhost (UserhostItem *item, char *nick, char *stuff)
{
	new_free(&server_list[from_server].userhost);
	server_list[from_server].userhost = m_3dup(item->user, "@", item->host);
}



static int write_to_server(int server, int des, char *buffer)
{
int err = 0;
	if (do_hook(SEND_TO_SERVER_LIST, "%d %d %s", server, des, buffer))
	{
		if (serv_output_func)
			err = (*serv_output_func)(server, des, buffer, strlen(buffer));
		else
			err = write(des, buffer, strlen(buffer));
		if ((err == -1) && !get_int_var(NO_FAIL_DISCONNECT_VAR))
		{
			say("Write to server failed.  Closing connection.");
			close_server(server, strerror(errno));
			get_connected(server, server);
		}
	}
	return err;
}

void send_from_server_queue(void)
{
QueueSend *tmp;
	if ((tmp = serverqueue))
	{
		if (now - server_list[tmp->server].last_sent >= get_int_var(QUEUE_SENDS_VAR))
		{
			serverqueue = tmp->next;
			if (server_list[tmp->server].connected)
				write_to_server(tmp->server, tmp->des, tmp->buffer);
			else
				put_it("ERR in server queue. not connected.");
#ifdef QUEUE_DEBUG
			put_it("sending 1 at %d", now - server_list[tmp->server].last_sent);
#endif
			server_list[tmp->server].last_sent = now;
			new_free(&tmp->buffer);
			new_free(&tmp);
		}
	}	
}

void add_to_server_queue(int server, int des, char *buffer)
{
QueueSend *tmp, *tmp1;
	tmp = (QueueSend *)new_malloc(sizeof(QueueSend));
	tmp->server = server;
	tmp->des = des;
	tmp->buffer = m_strdup(buffer);
	for (tmp1 = serverqueue; tmp1; tmp1 = tmp1->next)
		if (tmp1->next == NULL)
			break;
	if (tmp1)
		tmp1->next = tmp;
	else
	{
		serverqueue = tmp;	
		server_list[server].last_sent = now;
	}
}

static void vsend_to_server(int type, const char *format, va_list args)
{
	char	buffer[BIG_BUFFER_SIZE + 1];	/* make this buffer *much*
						 * bigger than needed */
	char 	*buf = buffer;
	int server;
	int 	len,
		des;
	if ((server = from_server) == -1)
		server = primary_server;

	if (server != -1 && ((des = server_list[server].write) != -1) && format)
	{
		vsnprintf(buf, BIG_BUFFER_SIZE, format, args);

		if (outbound_line_mangler)
			mangle_line(buf, outbound_line_mangler);
			
		server_list[server].sent = 1;
		len = strlen(buffer);
		if (len > (IRCD_BUFFER_SIZE - 2) || len == -1)
			buffer[IRCD_BUFFER_SIZE - 2] = (char) 0;
		if (x_debug & DEBUG_OUTBOUND)
			debugyell("[%d] -> [%s]", des, buffer);
		strmcat(buffer, "\r\n", IRCD_BUFFER_SIZE);

		if (get_int_var(QUEUE_SENDS_VAR) && (type == QUEUE_SEND))
		{
			add_to_server_queue(server, des, buffer);
			return;
		}

		write_to_server(server, des, buffer);
		if (oper_command)
			memset(buffer, 0, len);
	}
	else if (from_server == -1)
        {
                if (do_hook(DISCONNECT_LIST,"No Connection to %d", server))
			put_it("%s", convert_output_format(fget_string_var(FORMAT_DISCONNECT_FSET), "%s %s", update_clock(GET_TIME), "You are not connected to a server. Use /SERVER to connect."));
        }
}

/* send_to_server: sends the given info the the server */
void 	send_to_server (const char *format, ...)
{
	va_list args;

	va_start(args, format);
	vsend_to_server(IMMED_SEND, format, args);
	va_end(args);
}

/* send_to_server: sends the given info the the server */
void 	my_send_to_server (int refnum, const char *format, ...)
{
	int old_from_server = from_server;
	va_list args;

	from_server = refnum;
	va_start(args, format);
	vsend_to_server(IMMED_SEND, format, args);
	va_end(args);
	from_server = old_from_server;

}

void queue_send_to_server(int refnum, const char *format, ...)
{
	int old_from_server = from_server;
	va_list args;

	from_server = refnum;
	va_start(args, format);
	vsend_to_server(QUEUE_SEND, format, args);
	va_end(args);
	from_server = old_from_server;
	
}


/*
 * close_all_server: Used when creating new screens to close all the open
 * server connections in the child process...
 */
extern	void close_all_server (void)
{
	int	i;

	for (i = 0; i < number_of_servers; i++)
	{
		if (server_list[i].read != -1)
			new_close(server_list[i].read);
		if (server_list[i].write != -1)
			new_close(server_list[i].write);
	}
}

extern	char *create_server_list (char *input)
{
	int	i;
	int	do_read = 0;
	char	*value = NULL;
	char buffer[BIG_BUFFER_SIZE + 1];
	if (input && *input && *input == '1')
		do_read = 1;
	*buffer = '\0';
	for (i = 0; i < number_of_servers; i++)
	{
		if (server_list[i].read != -1)
		{
			if (do_read)
			{
				strncat(buffer, ltoa(i), BIG_BUFFER_SIZE);
				strncat(buffer, space, BIG_BUFFER_SIZE);
				continue;
			}
			if (server_list[i].itsname)
			{
				strncat(buffer, server_list[i].itsname, BIG_BUFFER_SIZE);
				strncat(buffer, space, BIG_BUFFER_SIZE);
			}
			else
				yell("Warning: server_list[%d].itsname is null and it shouldnt be", i);
				
		}
	}
	malloc_strcpy(&value, buffer);

	return value;
}

void server_disconnect(int i, char *args)
{
char	*message;
        /*
         * XXX - this is a major kludge.  i should never equal -1 at
         * this point.  we only do this because something has gotten
         * *really* confused at this point.  .mrg.
	 *
	 * Like something so obvious as already being disconnected?
         */
        if (i == -1)
        {
		if (connected_to_server)
		{
			for (i = 0; i < number_of_servers; i++)
			{
				clear_channel_list(i);
				clean_server_queues(i);
				server_list[i].eof = -1;
				new_close(server_list[i].read);
				new_close(server_list[i].write);
			}
		}
		goto done;
	}
 
	if (i >= 0 && i < number_of_servers)
	{
		if (!args || !*args)
			message = "Disconnecting";
		else
			message = args;
		put_it("%s", convert_output_format(fget_string_var(FORMAT_DISCONNECT_FSET), "%s %s %s", update_clock(GET_TIME), "Disconnecting from server", get_server_itsname(i)));
		clear_channel_list(i);
		close_server(i, message);
		server_list[i].eof = 1;
	}
done:
	window_check_servers();
	if (!connected_to_server)
		if(do_hook(DISCONNECT_LIST,"Disconnected by User request"))
			put_it("%s", convert_output_format(fget_string_var(FORMAT_DISCONNECT_FSET), "%s %s", update_clock(GET_TIME), "You are not connected to a server. Use /SERVER to connect."));
}


BUILT_IN_COMMAND(disconnectcmd)
{
	char    *server; 
	int	i;

	if (args && *args && isdigit(*args) && (server = next_arg(args, &args)) != NULL)
	{
		i = parse_server_index(server);
		if (-1 == i)
		{
			say("No such server!");
			return;
		}
	}
	else
		i = get_window_server(0);

	server_disconnect(i, args);
} 

/*
 * This is the function to attempt to make a nickname change.  You
 * cannot send the NICK command directly to the server: you must call
 * this function.  This function makes sure that the neccesary variables
 * are set so that if the NICK command fails, a sane action can be taken.
 *
 * If ``nick'' is NULL, then this function just tells the server what
 * we're trying to change our nickname to.  If we're not trying to change
 * our nickname, then this function does nothing.
 */
void	change_server_nickname (int ssn_index, char *nick)
{
	Server *s = &server_list[ssn_index];
	char    *n;

	if (nick)
	{
		n = LOCAL_COPY(nick);
		if ((n = check_nickname(n)) != NULL)
		{
			malloc_strcpy(&s->d_nickname, n);
			malloc_strcpy(&s->s_nickname, n);
		}
		else
			reset_nickname(ssn_index);
	}

	if (server_list[ssn_index].s_nickname)
		my_send_to_server(ssn_index, "NICK %s", server_list[ssn_index].s_nickname);
}

void	accept_server_nickname (int ssn_index, char *nick)
{
	malloc_strcpy(&server_list[ssn_index].nickname, nick);
	malloc_strcpy(&server_list[ssn_index].d_nickname, nick);
	new_free(&server_list[ssn_index].s_nickname);
	server_list[ssn_index].fudge_factor = 0;

	if (ssn_index == primary_server)
		strmcpy(nickname, nick, NICKNAME_LEN);

	update_all_status(current_window, NULL, 0);
	update_input(UPDATE_ALL);
}

void	nick_command_is_pending (int servnum, int value)
{
	server_list[servnum].nickname_pending = value;
}



/* 
 * This will generate up to 18 nicknames plus the 9-length(nickname)
 * that are unique but still have some semblance of the original.
 * This is intended to allow the user to get signed back on to
 * irc after a nick collision without their having to manually
 * type a new nick every time..
 * 
 * The func will try to make an intelligent guess as to when it is
 * out of guesses, and if it ever gets to that point, it will do the
 * manually-ask-you-for-a-new-nickname thing.
 */
void fudge_nickname (int servnum)
{
	char l_nickname[BIG_BUFFER_SIZE + 1];
	Server *s = &server_list[from_server];
	/*
	 * If we got here because the user did a /NICK command, and
	 * the nick they chose doesnt exist, then we just dont do anything,
	 * we just cancel the pending action and give up.
	 */
	if (s->nickname_pending)
	{
		new_free(&s->s_nickname);
		return;
	}

	/*
	 * Ok.  So we're not doing a /NICK command, so we need to see
	 * if maybe we're doing some other type of NICK change.
	 */
	if (s->s_nickname)
		strmcpy(l_nickname, s->s_nickname, NICKNAME_LEN);
	else if (s->nickname)
		strmcpy(l_nickname, s->nickname, NICKNAME_LEN);
	else
		strmcpy(l_nickname, nickname, NICKNAME_LEN);


	if (s->fudge_factor < strlen(l_nickname))
		s->fudge_factor = strlen(l_nickname);
	else
	{
		s->fudge_factor++;
		if (s->fudge_factor == 17)
		{
			/* give up... */
			reset_nickname(servnum);
			s->fudge_factor = 0;
			return;
		}
	}

	/* 
	 * Process of fudging a nickname:
	 * If the nickname length is less then 9, add an underscore.
	 */
	if (strlen(l_nickname) < 9)
		strmcat(l_nickname, "_", NICKNAME_LEN);

	/* 
	 * The nickname is 9 characters long. roll the nickname
	 */
	else
	{
		char tmp = l_nickname[8];
		l_nickname[8] = l_nickname[7];
		l_nickname[7] = l_nickname[6];
		l_nickname[6] = l_nickname[5];
		l_nickname[5] = l_nickname[4];
		l_nickname[4] = l_nickname[3];
		l_nickname[3] = l_nickname[2];
		l_nickname[2] = l_nickname[1];
		l_nickname[1] = l_nickname[0];
		l_nickname[0] = tmp;
	}
	if (!strcmp(l_nickname, "_________"))
	{
		reset_nickname(servnum);
		return;
	}
	change_server_nickname(servnum, l_nickname);
}


/*
 * -- Callback function
 */
void nickname_sendline (char *data, char *nick)
{
	int	new_server;

	new_server = atoi(data);
	change_server_nickname(new_server, nick);
}

/*
 * reset_nickname: when the server reports that the selected nickname is not
 * a good one, it gets reset here. 
 * -- Called by more than one place
 */
void reset_nickname (int servnum)
{
	char	server_num[10];

	kill(getpid(), SIGINT);
	say("You have specified an illegal nickname");
	if (!dumb_mode)
	{
		say("Please enter your nickname");
		strcpy(server_num, ltoa(servnum));
		add_wait_prompt("Nickname: ", nickname_sendline, server_num,
			WAIT_PROMPT_LINE, 1);
	}
	update_all_status(current_window, NULL, 0);
}



/*
 * password_sendline: called by send_line() in get_password() to handle
 * hitting of the return key, etc 
 * -- Callback function
 */
void password_sendline (char *data, char *line)
{
	int	new_server;

	if (line && *line)
	{
		new_server = atoi(next_arg(line, &line));
		set_server_password(new_server, line);
		connect_to_server_by_refnum(new_server, -1);
	}
}

char *get_pending_nickname(int servnum)
{
	return server_list[servnum].s_nickname;
}

BUILT_IN_COMMAND(evalserver)
{
	if (args && *args)
	{
		int old_server = from_server;
		char *p;
		p = next_arg(args, &args);
		if (is_number(p))
		{
			if (is_server_connected(my_atol(p)))
				from_server = my_atol(args);
		} 
		else if (*args)
			*(args-1) = ' ';
		else
			args = p;
		if (*args == '{')
		{
			
			char *ptr;
			ptr = MatchingBracket(args+1, *args, *args == '{' ? '}':')');
			*ptr = 0;
			args++;
		}
		parse_line(NULL, args, subargs, 0, 0, 1);
		from_server = old_server;
	}
}

#if defined(WINNT) || defined(__EMX__) || defined(WANT_IDENTD)

void identd_read(int s)
{
char buffer[100];
char *bufptr;
unsigned int lport = 0, rport = 0;
	*buffer = 0;
	bufptr = buffer;

	already_identd++;
	if (recv(s, buffer, sizeof(buffer)-1, 0) <=0)
	{
		put_it("ERROR in identd request");
		close_socketread(s);
		already_identd = 0;
		return;
	}
	if (sscanf(bufptr, "%d , %d", &lport, &rport) == 2)
	{
		if (lport < 1 || rport < 1 || lport > 65534 || rport > 65534)
		{
			close_socketread(s);
			put_it("ERROR port for identd bad [%d:%d]", lport, rport);
			already_identd = 0;
			return;
		}
		sprintf(buffer, "%hu , %hu : USERID : UNIX : %s", lport, rport, username);
		dcc_printf(s, "%s\r\n", buffer);
		put_it("Sent IDENTD request %s", buffer);
		set_socketflags(identd, now);
	}
	close_socketread(s);
}

void identd_handler(int s)
{
struct  sockaddr_in     remaddr;
int sra = sizeof(struct sockaddr_in);
int ret = 1;
int sock = -1;
	if ((sock = accept(s, (struct sockaddr *) &remaddr, &sra)) > -1)
	{
		add_socketread(sock, s, 0, inet_ntoa(remaddr.sin_addr), identd_read, NULL);
		add_sockettimeout(sock, 20);
	}
}

#endif

void start_identd(void)
{
#if defined(WINNT) || defined(__EMX__) || defined(WANT_IDENTD)
int sock = -1;
unsigned short port = 113;
	if ((sock = connect_by_number(NULL, &port, SERVICE_SERVER, PROTOCOL_TCP, 1)) > -1)
		add_socketread(sock, port, 0, NULL, identd_handler, NULL);
	identd = sock;
#endif
}

