/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"
#include "texcache.h"
#include "Printerr.h"
#include "tlib.h"
#include "scrncptr.h"
#include "util.h"
#include "intro.h"
#include "linux_utils.h"

#define MAX_POINTS  1000
#define BRUSH_THICKNESS  8
#define BRUSH_INTENSITY  50
#define CURVE_SUBDIVS  10

#define SHIFT_KEY_DOWN       0x1
#define CONTROL_KEY_DOWN     0x2
#define LEFT_ARROW_KEY_DOWN  0x4
#define RIGHT_ARROW_KEY_DOWN 0x8
#define DOWN_ARROW_KEY_DOWN  0x10
#define UP_ARROW_KEY_DOWN    0x20

#define LEFT_MOUSE_BUTTON_DOWN   0x1
#define MIDDLE_MOUSE_BUTOTN_DOWN 0x2
#define RIGHT_MOUSE_BUTTON_DOWN  0x4

static unsigned int gKeyboardState = 0;
static unsigned int gMouseState = 0;

// local variables
static GrTexInfo gIntroTexture[4], gLightMapTexture;
static FxU16 *gWholeData, *gOriginalData;
static FxU32 gMemReq;
static POINT gMousePos;
static float gCurrStep, gPoints[MAX_POINTS][2];
static int gTraversals, gNumPoints = 0;
static FxBool gEditCurve;

// global variables
FxBool gIntroRunning;

// external variables
#ifdef USE_GLIDE3
extern GrContext_t gContext;
#endif // USE_GLIDE3

extern int SCREEN_RES, SCREEN_REFRESH;
extern float SCREEN_RES_X, SCREEN_RES_Y;
extern int HI_SCREEN_RES, HI_SCREEN_RES_X, HI_SCREEN_RES_Y;
extern int LO_SCREEN_RES, LO_SCREEN_RES_X, LO_SCREEN_RES_Y;
extern int gNumTMUsUsed;
extern int gNumScreenShots, gNumScreensToCapture, gScreenCaptureEnabled;

// external functions
extern void InitGlideState();

// local function prototypes
static void ReloadPartialTextures(int s0, int t0, int s1, int t1);
static void BrushAt(int center_s, int center_t, int radius, int intensity);
static void ClearAt(int center_s, int center_t, int radius);
void DrawBezierCurve();
static void SetupCurve();


FxBool InitIntro()
{
	FxU32 i, log_width, log_height, size;
	char str[64];

	gIntroRunning = FXFALSE;

	for (i=0; i<4; i++)
	{
		sprintf(str, "data/intro%d.dat", i);
		if (!LoadGlideTextureTGA(str, &gIntroTexture[i], GR_TEXFMT_ALPHA_INTENSITY_88, 127, 0))
		{
			return FXFALSE;
		}
	}

#ifdef USE_GLIDE3
	log_width = gIntroTexture[0].largeLodLog2;
	log_height = log_width - gIntroTexture[0].aspectRatioLog2;
#else
	log_width = GR_LOD_1 - gIntroTexture[0].largeLod;
	log_height = log_width - (GR_ASPECT_1x1 - gIntroTexture[0].aspectRatio);
#endif // USE_GLIDE3

	// the texture must be 256x256
	if (log_width != 8 || log_height != 8)
	{
		for (i=0; i<4; i++)
		{
			delete [] gIntroTexture[i].data;
		}
		return FXFALSE;
	}

	size = 1<<(log_width+log_height);

	gWholeData = new FxU16[size*4];
	gOriginalData = new FxU16[size*4];
	if (!gWholeData || !gOriginalData)
	{
		for (i=0; i<4; i++)
		{
			delete [] gIntroTexture[i].data;
		}
		return FXFALSE;
	}

	// copy the 4 gIntroTextures in one big buffer gWholeData
	// so that everything can work off of gWholeData
	for (i=0; i<4; i++)
	{
		memcpy(&gWholeData[size*i], gIntroTexture[i].data, sizeof(FxU16)*size);
		delete [] gIntroTexture[i].data;
		gIntroTexture[i].data = &gWholeData[size*i];
	}

	// store the original texture data in gOriginalData for later use
	memcpy(gOriginalData, gWholeData, sizeof(FxU16)*size*4);

	// this is the memory required for each gIntroTexture
	gMemReq = grTexTextureMemRequired(GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[0]);

	// load all 4 textures in the texture cache
	for (i=0; i<4; i++)
	{
		grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
		if (GetNumTMUs() > 1)
		{
			grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
		}
	}

	// create and load the light map texture
	if (!CreateLightMapTexture(&gLightMapTexture))
	{
		CleanupIntro();
		return FXFALSE;
	}
	grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - 5*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gLightMapTexture);
	if (GetNumTMUs() > 1)
	{
		grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - 5*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gLightMapTexture);
	}

	// initialize the Bezier curve's control vertices
	SetupCurve();
	gCurrStep = 0.0f;
	gTraversals = 0;	

	// set up the font stuff
	tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
	tlConClear();

	gIntroRunning = FXTRUE;

	if (GetNumTMUs() > 1)
	{
#ifdef USE_GLIDE3
		grVertexLayout(GR_PARAM_ST1, offsetof(GrVertex, tmuvtx[1]), GR_PARAM_ENABLE);
#else // USE_GLIDE3
		grHints(GR_HINT_STWHINT, GR_STWHINT_ST_DIFF_TMU1);
#endif // USE_GLIDE3
	}

	return FXTRUE;
}

FxBool CleanupIntro()
{
	int i;

	if (gWholeData)
	{
		delete [] gWholeData;
		gWholeData = NULL;
	}
	if (gOriginalData)
	{
		delete [] gOriginalData;
		gOriginalData = NULL;
	}
	if (gLightMapTexture.data)
	{
		delete [] gLightMapTexture.data;
		gLightMapTexture.data = NULL;
	}
	for (i=0; i<4; i++)
	{
		gIntroTexture[i].data = NULL;
	}

	return FXTRUE;
}

void RenderIntro()
{
	GrVertex verts[4];
	float s_offset, t_offset, screen_width, screen_height;
	int i, x, y, time, color;
	float light_x, light_y, light_radius, rad_sin, rad_cos;
	const float edge_thickness = 40.0f;

	time = linux_timeGetTime()>>6;

	light_radius = 0.25f*SCREEN_RES_X;
	fsincos(DEG_TO_RAD(time), &rad_sin, &rad_cos);
	s_offset = 2.0f*rad_cos;
	t_offset = -2.0f*rad_sin;
	rad_sin *= 0.5f*SCREEN_RES_Y;
	rad_cos *= 0.5f*SCREEN_RES_Y;
	light_x = 0.5f*SCREEN_RES_X + rad_cos;
	light_y = 0.5f*SCREEN_RES_Y + rad_sin;

	color = 0x7f00ff00;
	if (time & 256)
	{
		color |= ((time & 255)<<16) | (255 - (time & 255));
	}
	else
	{
		color |= ((255 - (time & 255))<<16) | (time & 255);
	}
	grConstantColorValue(color);

	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_NONE, FXFALSE);
	grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
								 GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_TEXTURE, FXFALSE);

	screen_width = 0.5f*SCREEN_RES_X;
	screen_height = 0.5f*SCREEN_RES_Y;

	verts[0].oow = 1.0f;
	verts[1].oow = 1.0f;
	verts[2].oow = 1.0f;
	verts[3].oow = 1.0f;

	for (i=0; i<4; i++)
	{
		x = (i & 1);
		y = 1-(i>>1);

		verts[0].x = x*screen_width;
		verts[0].y = y*screen_height;
		verts[1].x = (x+1)*screen_width;
		verts[1].y = y*screen_height;
		verts[2].x = x*screen_width;
		verts[2].y = (y+1)*screen_height;
		verts[3].x = (x+1)*screen_width;
		verts[3].y = (y+1)*screen_height;

		if (gNumTMUsUsed > 1)
		{
			grTexSource(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
			grTexSource(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);

			grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL_ADD_LOCAL_ALPHA, GR_COMBINE_FACTOR_ONE,
									 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
			grTexCombine(GR_TMU1, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
									 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
			grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);

			verts[0].tmuvtx[0].sow =   0.0f + s_offset;
			verts[0].tmuvtx[0].tow = 256.0f + t_offset;
			verts[1].tmuvtx[0].sow = 256.0f + s_offset;
			verts[1].tmuvtx[0].tow = 256.0f + t_offset;
			verts[2].tmuvtx[0].sow =   0.0f + s_offset;
			verts[2].tmuvtx[0].tow =   0.0f + t_offset;
			verts[3].tmuvtx[0].sow = 256.0f + s_offset;
			verts[3].tmuvtx[0].tow =   0.0f + t_offset;

			verts[0].tmuvtx[1].sow =   0.0f;
			verts[0].tmuvtx[1].tow = 256.0f;
			verts[1].tmuvtx[1].sow = 256.0f;
			verts[1].tmuvtx[1].tow = 256.0f;
			verts[2].tmuvtx[1].sow =   0.0f;
			verts[2].tmuvtx[1].tow =   0.0f;
			verts[3].tmuvtx[1].sow = 256.0f;
			verts[3].tmuvtx[1].tow =   0.0f;

			grDrawTriangle(&verts[0], &verts[1], &verts[2]);
			grDrawTriangle(&verts[1], &verts[3], &verts[2]);
		}
		else // 1 TMU
		{
			grTexSource(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);

			grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
									 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
			grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);

			verts[0].tmuvtx[0].sow =   0.0f;
			verts[0].tmuvtx[0].tow = 256.0f;
			verts[1].tmuvtx[0].sow = 256.0f;
			verts[1].tmuvtx[0].tow = 256.0f;
			verts[2].tmuvtx[0].sow =   0.0f;
			verts[2].tmuvtx[0].tow =   0.0f;
			verts[3].tmuvtx[0].sow = 256.0f;
			verts[3].tmuvtx[0].tow =   0.0f;

			grDrawTriangle(&verts[0], &verts[1], &verts[2]);
			grDrawTriangle(&verts[1], &verts[3], &verts[2]);

			grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
									 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXTRUE, FXFALSE);
			grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO);
			grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
										 GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_TEXTURE, FXFALSE);

			verts[0].tmuvtx[0].sow += s_offset;
			verts[0].tmuvtx[0].tow += t_offset;
			verts[1].tmuvtx[0].sow += s_offset;
			verts[1].tmuvtx[0].tow += t_offset;
			verts[2].tmuvtx[0].sow += s_offset;
			verts[2].tmuvtx[0].tow += t_offset;
			verts[3].tmuvtx[0].sow += s_offset;
			verts[3].tmuvtx[0].tow += t_offset;

			grDrawTriangle(&verts[0], &verts[1], &verts[2]);
			grDrawTriangle(&verts[1], &verts[3], &verts[2]);
		}
	}

	grColorCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grAlphaBlendFunction(GR_BLEND_DST_COLOR, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);

	verts[0].oow = verts[1].oow = verts[2].oow = verts[3].oow = 1.0f;

	// blend the left edge
	verts[0].x = 0.0f;
	verts[0].y = 0.0f;
	verts[1].x = edge_thickness;
	verts[1].y = 0.0f;
	verts[2].x = 0.0f;
	verts[2].y = SCREEN_RES_Y;
	verts[3].x = edge_thickness;
	verts[3].y = SCREEN_RES_Y;
	verts[0].r = verts[2].r = 0.0f;
	verts[0].g = verts[2].g = 0.0f;
	verts[0].b = verts[2].b = 0.0f;
	verts[1].r = verts[3].r = 255.0f;
	verts[1].g = verts[3].g = 255.0f;
	verts[1].b = verts[3].b = 255.0f;
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);

	// blend the right edge
	verts[0].x = SCREEN_RES_X-edge_thickness;
	verts[0].y = 0.0f;
	verts[1].x = SCREEN_RES_X;
	verts[1].y = 0.0f;
	verts[2].x = SCREEN_RES_X-edge_thickness;
	verts[2].y = SCREEN_RES_Y;
	verts[3].x = SCREEN_RES_X;
	verts[3].y = SCREEN_RES_Y;
	verts[0].r = verts[2].r = 255.0f;
	verts[0].g = verts[2].g = 255.0f;
	verts[0].b = verts[2].b = 255.0f;
	verts[1].r = verts[3].r = 0.0f;
	verts[1].g = verts[3].g = 0.0f;
	verts[1].b = verts[3].b = 0.0f;
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);

	// blend the top edge
	verts[0].x = 0.0f;
	verts[0].y = 0.0f;
	verts[1].x = SCREEN_RES_X;
	verts[1].y = 0.0f;
	verts[2].x = 0.0f;
	verts[2].y = edge_thickness;
	verts[3].x = SCREEN_RES_X;
	verts[3].y = edge_thickness;
	verts[0].r = verts[1].r = 0.0f;
	verts[0].g = verts[1].g = 0.0f;
	verts[0].b = verts[1].b = 0.0f;
	verts[2].r = verts[3].r = 255.0f;
	verts[2].g = verts[3].g = 255.0f;
	verts[2].b = verts[3].b = 255.0f;
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);

	// blend the bottom edge
	verts[0].x = 0.0f;
	verts[0].y = SCREEN_RES_Y-edge_thickness;
	verts[1].x = SCREEN_RES_X;
	verts[1].y = SCREEN_RES_Y-edge_thickness;
	verts[2].x = 0.0f;
	verts[2].y = SCREEN_RES_Y;
	verts[3].x = SCREEN_RES_X;
	verts[3].y = SCREEN_RES_Y;
	verts[0].r = verts[1].r = 255.0f;
	verts[0].g = verts[1].g = 255.0f;
	verts[0].b = verts[1].b = 255.0f;
	verts[2].r = verts[3].r = 0.0f;
	verts[2].g = verts[3].g = 0.0f;
	verts[2].b = verts[3].b = 0.0f;
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);

	// draw the light map
	verts[0].x = light_x - light_radius;
	verts[0].y = light_y - light_radius;
	verts[1].x = light_x + light_radius;
	verts[1].y = light_y - light_radius;
	verts[2].x = light_x - light_radius;
	verts[2].y = light_y + light_radius;
	verts[3].x = light_x + light_radius;
	verts[3].y = light_y + light_radius;
	grConstantColorValue(0xffffffff);
	grTexSource(GR_TMU0, grTexMaxAddress(GR_TMU0) - 5*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gLightMapTexture);
	grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
							 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
	grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_LOCAL,
								 GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
	grAlphaBlendFunction(GR_BLEND_SRC_COLOR, GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO);
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);

	DrawBezierCurve();

	if (gNumScreensToCapture)
	{
		char str[64];

		gNumScreensToCapture--;

		sprintf(str, "scrn%d.tga", gNumScreenShots);
		if (ScreenCaptureTGA(str))
		{
			gNumScreenShots++;
		}
	}

	// draw the console stuff
      	tlConClear();
	tlConOutput("press the space key to continue ...");
	tlConRender();

	// draw a rotating "+" cursor
	if (gEditCurve)
	{
		grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ONE_MINUS_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO);
		grColorCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_NONE, FXFALSE);
		grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_NONE, FXFALSE);
		grConstantColorValue(0x7f7f0000);

		fsincos(DEG_TO_RAD(linux_timeGetTime()>>2), &rad_sin, &rad_cos);
		rad_sin *= 8.0f;
		rad_cos *= 8.0f;

		verts[0].x = (float)gMousePos.x - rad_cos;
		verts[0].y = (float)gMousePos.y - rad_sin;
		verts[1].x = (float)gMousePos.x + rad_cos;
		verts[1].y = (float)gMousePos.y + rad_sin;
		verts[2].x = (float)gMousePos.x + rad_sin;
		verts[2].y = (float)gMousePos.y - rad_cos;
		verts[3].x = (float)gMousePos.x - rad_sin;
		verts[3].y = (float)gMousePos.y + rad_cos;

		grDrawLine(&verts[0], &verts[1]);
		grDrawLine(&verts[2], &verts[3]);
	}

	grBufferSwap(1);
}

// given the s, t rectangle in the big texture (gWholeTexture)
// find the smaller rectangles in each of the 4 textures
// making up the big texture, and reload them in the texture cache
// NOTE: gWholeTexture is 512x512 and each gIntroTexture is 256x256
static void ReloadPartialTextures(int s0, int t0, int s1, int t1)
{
	int i, minu, maxu, minv, maxv, u0, v0, u1, v1;

	for (i=0; i<4; i++)
	{
		minu = (i & 1)<<8;
		maxu = minu + 256;
		minv = (i>>1)<<8;
		maxv = minv + 256;
		u0 = CLAMP(s0, minu, maxu-1);
		v0 = CLAMP(t0, minv, maxv-1);
		u1 = CLAMP(s1, minu, maxu-1);
		v1 = CLAMP(t1, minv, maxv-1);
		// if the clamped rectangle has some area, we need to reload it
		if ((u1-u0) | (v1-v0))
		{
			// transform it to the smaller texture's coords
			u0 &= 255;
			v0 &= 255;
			u1 &= 255;
			v1 &= 255;
#ifdef USE_GLIDE3
			grTexDownloadMipMapLevelPartial(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, gIntroTexture[i].largeLodLog2, gIntroTexture[i].largeLodLog2,
																			gIntroTexture[i].aspectRatioLog2, gIntroTexture[i].format, GR_MIPMAPLEVELMASK_BOTH,
																			((FxU16 *)gIntroTexture[i].data) + (v0<<8), v0, v1);
			if (GetNumTMUs() > 1)
			{
				grTexDownloadMipMapLevelPartial(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, gIntroTexture[i].largeLodLog2, gIntroTexture[i].largeLodLog2,
																				gIntroTexture[i].aspectRatioLog2, gIntroTexture[i].format, GR_MIPMAPLEVELMASK_BOTH,
																				((FxU16 *)gIntroTexture[i].data) + (v0<<8), v0, v1);
			}
#else
			grTexDownloadMipMapLevelPartial(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, gIntroTexture[i].largeLod, gIntroTexture[i].largeLod,
																			gIntroTexture[i].aspectRatio, gIntroTexture[i].format, GR_MIPMAPLEVELMASK_BOTH,
																			((FxU16 *)gIntroTexture[i].data) + (v0<<8), v0, v1);
			if (GetNumTMUs() > 1)
			{
				grTexDownloadMipMapLevelPartial(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, gIntroTexture[i].largeLod, gIntroTexture[i].largeLod,
																				gIntroTexture[i].aspectRatio, gIntroTexture[i].format, GR_MIPMAPLEVELMASK_BOTH,
																				((FxU16 *)gIntroTexture[i].data) + (v0<<8), v0, v1);
			}
#endif // USE_GLIDE3
		}
	}
}

static void BrushAt(int center_s, int center_t, int radius, int intensity)
{
	FxU16 *data, color;
	int s0, t0, s1, t1, s, t;
	int dist_squared, radius_squared;
	int radius_squared_inv; // fixed 16.16

	radius_squared = radius*radius;
	radius_squared_inv = 65535/radius_squared;

	s0 = CLAMP(center_s-radius, 0, 511);
	t0 = CLAMP(center_t-radius, 0, 511);
	s1 = CLAMP(center_s+radius, 0, 511);
	t1 = CLAMP(center_t+radius, 0, 511);

	for (t=t0; t<t1; t++)
	{
		for (s=s0; s<s1; s++)
		{
			dist_squared = SQR(s - center_s) + SQR(t - center_t);
			if (dist_squared < radius_squared)
			{
				data = &gWholeData[((t+(s & 256)+(t & 256))<<8) + (s & 255)];
				color = CLAMP(((*data) & 0x00ff) + (intensity - ((intensity*dist_squared*radius_squared_inv)>>16)), 0, 255);
				*data = ((*data) & 0xff00) | color;
			}
		}
	}

	ReloadPartialTextures(s0, t0, s1, t1);
}

static void ClearAt(int center_s, int center_t, int radius)
{
	int index, s0, t0, s1, t1, s, t;
	int dist_squared, radius_squared;
	int radius_squared_inv; // fixed 16.16

	radius_squared = radius*radius;
	radius_squared_inv = 65535/radius_squared;

	s0 = CLAMP(center_s-radius, 0, 511);
	t0 = CLAMP(center_t-radius, 0, 511);
	s1 = CLAMP(center_s+radius, 0, 511);
	t1 = CLAMP(center_t+radius, 0, 511);

	for (t=t0; t<t1; t++)
	{
		for (s=s0; s<s1; s++)
		{
			dist_squared = SQR(s - center_s) + SQR(t - center_t);
			if (dist_squared < radius_squared)
			{
				index = ((t+(s & 256)+(t & 256))<<8) + (s & 255);
				gWholeData[index] = gOriginalData[index];
			}
		}
	}

	ReloadPartialTextures(s0, t0, s1, t1);
}

//////////////////////////////////////////////////////////
void linux_UIIntro(XEvent report)
{
	KeySym key;
	int i, dist_squared, min_dist_squared;
	static int min_index = 0;

	switch(report.type)	{

		case KeyPress:
			key = XLookupKeysym(&report.xkey, 0);

			switch(key)	{
				case XK_Shift_L:
				case XK_Shift_R:
					gKeyboardState |= SHIFT_KEY_DOWN;
				break;

				case XK_Control_L:
				case XK_Control_R:
					gKeyboardState |= CONTROL_KEY_DOWN;
				break;

				case XK_Escape:
				case XK_space:
					gIntroRunning = FXFALSE;
				break;

				case XK_A:
				case XK_a:
					if (gEditCurve) {
						if (gNumPoints > 2 && gNumPoints < MAX_POINTS-2 && (gNumPoints & 0x3) == 0) {
							gPoints[gNumPoints][X] = gPoints[gNumPoints-1][X];
							gPoints[gNumPoints][Y] = gPoints[gNumPoints-1][Y];
							gPoints[gNumPoints+1][X] = gPoints[gNumPoints][X] + (gPoints[gNumPoints-1][X] - gPoints[gNumPoints-2][X]);
							gPoints[gNumPoints+1][Y] = gPoints[gNumPoints][Y] + (gPoints[gNumPoints-1][Y] - gPoints[gNumPoints-2][Y]);
							gNumPoints+=2;
						}
					}
				break;

				case XK_D:
				case XK_d:
					if (gEditCurve)	{
						gNumPoints = MAX(0, gNumPoints-1);
						min_index = -1;
					}
				break;

				case XK_E:
				case XK_e:
					gEditCurve ^= 1;
					if (gEditCurve) {

						gCurrStep = 0.0f;
						gTraversals = 0;

						memcpy(gWholeData, gOriginalData, sizeof(FxU16)*512*512);
						for (i = 0; i < 4; i++) {
							grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
							if (GetNumTMUs() > 1) {
								grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
							}
						}
					}
				break;

				case XK_M:
				case XK_m:
					if (SCREEN_RES == LO_SCREEN_RES) {
						SCREEN_RES = HI_SCREEN_RES;
						SCREEN_RES_X = (float)HI_SCREEN_RES_X;
						SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;

						// convert points to the new resolution
						for (i = 0; i < gNumPoints; i++) {
							gPoints[i][X] *= (float)LO_SCREEN_RES_X/(float)HI_SCREEN_RES_X;
							gPoints[i][Y] *= (float)LO_SCREEN_RES_Y/(float)HI_SCREEN_RES_Y;
						}
					} else {
						SCREEN_RES = LO_SCREEN_RES;
						SCREEN_RES_X = (float)LO_SCREEN_RES_X;
						SCREEN_RES_Y = (float)LO_SCREEN_RES_Y;

						// convert points to the new resolution
						for (i=0; i < gNumPoints; i++) {
							gPoints[i][X] *= (float)HI_SCREEN_RES_X/(float)LO_SCREEN_RES_X;
							gPoints[i][Y] *= (float)HI_SCREEN_RES_Y/(float)LO_SCREEN_RES_Y;
						}
					}


					// reset the curve
					gCurrStep = 0.0f;
					gTraversals = 0;

					// close glide window before reopening it
#ifdef USE_GLIDE3
					grSstWinClose(gContext);
#else
					grSstWinClose();
#endif // USE_GLIDE3
					// try to get a triple-buffer with depth-buffer frame buffer
					if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1)) {
						// now try to get a double-buffer with depth-buffer frame buffer
						if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1)) {

							grGlideShutdown();
							linux_ShutdownXWindow();

							exit(0xdeadbeef);
						}
					}

					InitGlideState();

					// reset the console
					//tlConSet(0.1f, 0.94f, 0.5f, 0.98f, 40, 1, 0xff7f0000);
					tlConSet(0.0f, 0.0f, 1.0f, 1.0f, SCREEN_RES_X, SCREEN_RES_Y, 80, 40, 0xff00007f);
					linux_ResizeXWindow( SCREEN_RES_X, SCREEN_RES_Y );
					// reset the bumpmap texture
					memcpy(gWholeData, gOriginalData, sizeof(FxU16)*512*512);

					for (i=0; i< 4; i++) {
						grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
						if (GetNumTMUs() > 1) {
							grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
						}
					}

					grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);

					if (GetNumTMUs() >= 2) {
						grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
					}

					SetupCurve();

				break;

				case XK_P:
				case XK_p:
					if (gEditCurve)	{
						PrintError("gNumPoints = %d;\n", gNumPoints);
						for (i=0; i<gNumPoints; i++) {
							PrintError("gPoints[%d][X] = %3.0f.0f;\n", i, gPoints[i][X]);
							PrintError("gPoints[%d][Y] = %3.0f.0f;\n", i, gPoints[i][Y]);
						}
					}
				break;

				case XK_R:
				case XK_r:
					if (gEditCurve) {
						SetupCurve();
					}
				break;

				case XK_U:
				case XK_u:
					if (GetNumTMUs() > 1) {
						gNumTMUsUsed ^= 3;
					}
				break;

				default: break;
			} // switch on key pressed //

		break;

		case KeyRelease:
			key = XLookupKeysym(&report.xkey, 0);

			switch(key) {
			case XK_Return:
			case XK_KP_Enter:
					gNumScreensToCapture = gScreenCaptureEnabled;
				break;

				case XK_Shift_L:
				case XK_Shift_R:
					gKeyboardState &= ~SHIFT_KEY_DOWN;
				break;

				case XK_Control_L:
			        case XK_Control_R:
					gKeyboardState &= ~CONTROL_KEY_DOWN;
				break;

			} /* switch */

		break;

		case ButtonPress:

			switch(report.xbutton.button) {
				case (1): /* MOUSE_LEFT */
					gMouseState |= LEFT_MOUSE_BUTTON_DOWN;

					gMousePos.x = report.xbutton.x;
					gMousePos.y = report.xbutton.y;

					min_dist_squared = 10*10;
					min_index = -1;

					for (i = 0; i < gNumPoints; i++) {
						dist_squared = (int)(SQR(gPoints[i][X]-gMousePos.x) + SQR(gPoints[i][Y]-gMousePos.y));
						if (dist_squared < min_dist_squared) {
							min_dist_squared = dist_squared;
							min_index = i;
						}
					}
					if (gKeyboardState & CONTROL_KEY_DOWN) {
						if (min_index != -1) {
							if (min_index+1 < gNumPoints &&
									gPoints[min_index+1][X] == gPoints[min_index][X] &&
									gPoints[min_index+1][Y] == gPoints[min_index][Y])
							{
								gPoints[min_index+1][X] = (float)gMousePos.x;
								gPoints[min_index+1][Y] = (float)gMousePos.y;
							}
							else if (min_index-1 > 0 &&
									gPoints[min_index-1][X] == gPoints[min_index][X] &&
									gPoints[min_index-1][Y] == gPoints[min_index][Y])
							{
								gPoints[min_index-1][X] = (float)gMousePos.x;
								gPoints[min_index-1][Y] = (float)gMousePos.y;
							}
							gPoints[min_index][X] = (float)gMousePos.x;
							gPoints[min_index][Y] = (float)gMousePos.y;
						}
					}
					else if (gKeyboardState & SHIFT_KEY_DOWN)
					{
						if (min_index != -1)
						{
							gPoints[gNumPoints][X] = gPoints[min_index][X];
							gPoints[gNumPoints][Y] = gPoints[min_index][Y];
						}
						else
						{
							gPoints[gNumPoints][X] = (float)gMousePos.x;
							gPoints[gNumPoints][Y] = (float)gMousePos.y;
						}
						gNumPoints = MIN(MAX_POINTS-1, gNumPoints+1);
					}
					else
					{
						gPoints[gNumPoints][X] = (float)gMousePos.x;
						gPoints[gNumPoints][Y] = (float)gMousePos.y;
						gNumPoints = MIN(MAX_POINTS-1, gNumPoints+1);
					}

				break;

				case (2): /* MOUSE_MIDDLE */
				break;

				case (3): /* MOUSE_RIGHT */ 
					gMouseState |= RIGHT_MOUSE_BUTTON_DOWN;
				break;

				default: break;
			} /* switch on which mouse button was pressed */

		break;

		case ButtonRelease:

			switch(report.xbutton.button) {

				case (1): /* MOUSE_LEFT */
					gMouseState &= ~LEFT_MOUSE_BUTTON_DOWN;
					min_index = -1;
				break;

				case (2): /* MOUSE_MIDDLE */
				break;

				case (3): /* MOUSE_RIGHT */
					gMouseState &= ~RIGHT_MOUSE_BUTTON_DOWN;
				break;
			}

		break;

		case MotionNotify:

			gMousePos.x = report.xmotion.x;
			gMousePos.y = report.xmotion.y;

			if ((gKeyboardState & (CONTROL_KEY_DOWN | SHIFT_KEY_DOWN)) == (CONTROL_KEY_DOWN | SHIFT_KEY_DOWN)) {
				if (min_index != -1) {
					if (min_index+1 < gNumPoints &&
							gPoints[min_index+1][X] == gPoints[min_index][X] &&
							gPoints[min_index+1][Y] == gPoints[min_index][Y]) {
						gPoints[min_index+1][X] = (float)gMousePos.x;
						gPoints[min_index+1][Y] = (float)gMousePos.y;

					} else if (min_index-1 > 0 &&
							gPoints[min_index-1][X] == gPoints[min_index][X] &&
							gPoints[min_index-1][Y] == gPoints[min_index][Y]) {
						gPoints[min_index-1][X] = (float)gMousePos.x;
						gPoints[min_index-1][Y] = (float)gMousePos.y;
					}

					gPoints[min_index][X] = (float)gMousePos.x;
					gPoints[min_index][Y] = (float)gMousePos.y;
				}
			}

		break;

		default: break;
	}
}

//////////////////////////////////////////////////////////

// Need convert to LINUX //
// add to line 362 of main.cpp!
/*
long IntroUI(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	int i, dist_squared, min_dist_squared;
	RECT rect;
	static int min_index;

	switch (msg)
	{
		case WM_MOUSEMOVE:
			gMousePos.x = lParam & 0xffff;
			gMousePos.y = lParam>>16;
			if ((wParam & (MK_CONTROL | MK_LBUTTON)) == (MK_CONTROL | MK_LBUTTON))
			{
				if (min_index != -1)
				{
					if (min_index+1 < gNumPoints &&
							gPoints[min_index+1][X] == gPoints[min_index][X] &&
							gPoints[min_index+1][Y] == gPoints[min_index][Y])
					{
						gPoints[min_index+1][X] = (float)gMousePos.x;
						gPoints[min_index+1][Y] = (float)gMousePos.y;
					}
					else if (min_index-1 > 0 &&
							gPoints[min_index-1][X] == gPoints[min_index][X] &&
							gPoints[min_index-1][Y] == gPoints[min_index][Y])
					{
						gPoints[min_index-1][X] = (float)gMousePos.x;
						gPoints[min_index-1][Y] = (float)gMousePos.y;
					}
					gPoints[min_index][X] = (float)gMousePos.x;
					gPoints[min_index][Y] = (float)gMousePos.y;
				}
			}
			return 0;

    case WM_LBUTTONDOWN:
      SetCapture(hwnd);
			gMousePos.x = lParam & 0xffff;
			gMousePos.y = lParam>>16;

			min_dist_squared = 10*10;
			min_index = -1;
			for (i=0; i<gNumPoints; i++)
			{
				dist_squared = (int)(SQR(gPoints[i][X]-gMousePos.x) + SQR(gPoints[i][Y]-gMousePos.y));
				if (dist_squared < min_dist_squared)
				{
					min_dist_squared = dist_squared;
					min_index = i;
				}
			}
			if (wParam & MK_CONTROL)
			{
				if (min_index != -1)
				{
					if (min_index+1 < gNumPoints &&
							gPoints[min_index+1][X] == gPoints[min_index][X] &&
							gPoints[min_index+1][Y] == gPoints[min_index][Y])
					{
						gPoints[min_index+1][X] = (float)gMousePos.x;
						gPoints[min_index+1][Y] = (float)gMousePos.y;
					}
					else if (min_index-1 > 0 &&
							gPoints[min_index-1][X] == gPoints[min_index][X] &&
							gPoints[min_index-1][Y] == gPoints[min_index][Y])
					{
						gPoints[min_index-1][X] = (float)gMousePos.x;
						gPoints[min_index-1][Y] = (float)gMousePos.y;
					}
					gPoints[min_index][X] = (float)gMousePos.x;
					gPoints[min_index][Y] = (float)gMousePos.y;
				}
			}
			else if (wParam & MK_SHIFT)
			{
				if (min_index != -1)
				{
					gPoints[gNumPoints][X] = gPoints[min_index][X];
					gPoints[gNumPoints][Y] = gPoints[min_index][Y];
				}
				else
				{
					gPoints[gNumPoints][X] = (float)gMousePos.x;
					gPoints[gNumPoints][Y] = (float)gMousePos.y;
				}
				gNumPoints = MIN(MAX_POINTS-1, gNumPoints+1);
			}
			else
			{
				gPoints[gNumPoints][X] = (float)gMousePos.x;
				gPoints[gNumPoints][Y] = (float)gMousePos.y;
				gNumPoints = MIN(MAX_POINTS-1, gNumPoints+1);
			}
      return 0;

    case WM_LBUTTONUP:
      ReleaseCapture();
			min_index = -1;
      return 0;

    case WM_KEYDOWN:
			switch (wParam)
			{
				case VK_SPACE: case VK_ESCAPE:
					gIntroRunning = FXFALSE;
					break;

				case 'A':
					if (gEditCurve)
					{
						if (gNumPoints > 2 && gNumPoints < MAX_POINTS-2 && (gNumPoints & 0x3) == 0)
						{
							gPoints[gNumPoints][X] = gPoints[gNumPoints-1][X];
							gPoints[gNumPoints][Y] = gPoints[gNumPoints-1][Y];
							gPoints[gNumPoints+1][X] = gPoints[gNumPoints][X] + (gPoints[gNumPoints-1][X] - gPoints[gNumPoints-2][X]);
							gPoints[gNumPoints+1][Y] = gPoints[gNumPoints][Y] + (gPoints[gNumPoints-1][Y] - gPoints[gNumPoints-2][Y]);
							gNumPoints+=2;
						}
					}
					break;

				case 'D':
					if (gEditCurve)
					{
						gNumPoints = MAX(0, gNumPoints-1);
						min_index = -1;
					}
					break;

				case 'E':
					gEditCurve ^= 1;
					if (gEditCurve)
					{
						gCurrStep = 0.0f;
						gTraversals = 0;
						memcpy(gWholeData, gOriginalData, sizeof(FxU16)*512*512);
						for (i=0; i<4; i++)
						{
							grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
							if (GetNumTMUs() > 1)
							{
								grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
							}
						}
					}
					break;

				case 'M':
					if (SCREEN_RES == LO_SCREEN_RES)
					{
						SCREEN_RES = HI_SCREEN_RES;
						SCREEN_RES_X = (float)HI_SCREEN_RES_X;
						SCREEN_RES_Y = (float)HI_SCREEN_RES_Y;
						// convert points to the new resolution
						for (i=0; i<gNumPoints; i++)
						{
							gPoints[i][X] *= (float)LO_SCREEN_RES_X/(float)HI_SCREEN_RES_X;
							gPoints[i][Y] *= (float)LO_SCREEN_RES_Y/(float)HI_SCREEN_RES_Y;
						}
					}
					else
					{
						SCREEN_RES = LO_SCREEN_RES;
						SCREEN_RES_X = (float)LO_SCREEN_RES_X;
						SCREEN_RES_Y = (float)LO_SCREEN_RES_Y;
						// convert points to the new resolution
						for (i=0; i<gNumPoints; i++)
						{
							gPoints[i][X] *= (float)HI_SCREEN_RES_X/(float)LO_SCREEN_RES_X;
							gPoints[i][Y] *= (float)HI_SCREEN_RES_Y/(float)LO_SCREEN_RES_Y;
						}
					}

					rect.left = 0;
					rect.top = 0;
					rect.right = (int)SCREEN_RES_X;
					rect.bottom = (int)SCREEN_RES_Y;
					ClipCursor(&rect);

					// reset the curve
					gCurrStep = 0.0f;
					gTraversals = 0;

					// close glide window before reopening it
#ifdef USE_GLIDE3
					grSstWinClose(gContext);
#else
					grSstWinClose();
#endif // USE_GLIDE3
					// try to get a triple-buffer with depth-buffer frame buffer
					if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, GetNumColorBuffers(), 1))
					{
						// now try to get a double-buffer with depth-buffer frame buffer
						if (!grSstWinOpen(0, SCREEN_RES, SCREEN_REFRESH, GR_COLORFORMAT_ABGR, GR_ORIGIN_UPPER_LEFT, 2, 1))
						{
							grGlideShutdown();
							exit(0xdeadbeef);
						}
					}

					InitGlideState();
					// reset the console
					tlConSet(0.1f, 0.94f, 0.5f, 0.98f, SCREEN_RES_X, SCREEN_RES_Y, 40, 1, 0xff7f0000);
					// reset the bumpmap texture
					memcpy(gWholeData, gOriginalData, sizeof(FxU16)*512*512);
					for (i=0; i<4; i++)
					{
						grTexDownloadMipMap(GR_TMU0, grTexMaxAddress(GR_TMU0) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
						if (GetNumTMUs() > 1)
						{
							grTexDownloadMipMap(GR_TMU1, grTexMaxAddress(GR_TMU1) - (i+1)*gMemReq, GR_MIPMAPLEVELMASK_BOTH, &gIntroTexture[i]);
						}
					}

					grTexClampMode(GR_TMU0, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
					if (GetNumTMUs() >= 2)
					{
						grTexClampMode(GR_TMU1, GR_TEXTURECLAMP_CLAMP, GR_TEXTURECLAMP_CLAMP);
					}
					SetupCurve();
					break;

				case 'P':
					if (gEditCurve)
					{
						PrintError("gNumPoints = %d;\n", gNumPoints);
						for (i=0; i<gNumPoints; i++)
						{
							PrintError("gPoints[%d][X] = %3.0f.0f;\n", i, gPoints[i][X]);
							PrintError("gPoints[%d][Y] = %3.0f.0f;\n", i, gPoints[i][Y]);
						}
					}
					break;

				case 'R':
					if (gEditCurve)
					{
						SetupCurve();
					}
					break;

				case 'U':
					if (GetNumTMUs() > 1)
					{
						gNumTMUsUsed ^= 3;
					}
					break;
			}
			return 0;

    case WM_KEYUP:
			switch (wParam)
			{
				case VK_RETURN:
					gNumScreensToCapture = gScreenCaptureEnabled;
					break;
			}
			return 0;
	}

  return DefWindowProc(hwnd, msg, wParam, lParam);
}
*/



void DrawBezierSection(float *p0, float *p1, float *p2, float *p3)
{
	int i;
	float t, t2, t3, one_minus_t, one_minus_t2, one_minus_t3;
	GrVertex verts[CURVE_SUBDIVS+1];

	// Bezier basis functions at t=0 are simplified
	verts[0].x = p0[X];
	verts[0].y = p0[Y];

	verts[0].oow = 1.0f;
	verts[0].r = 255.0f;
	verts[0].g =   0.0f;
	verts[0].b =   0.0f;
	verts[0].a = 255.0f;

	for (i=1; i<=CURVE_SUBDIVS; i++)
	{
		t = (float)i/(float)CURVE_SUBDIVS;
		t2 = t*t;
		t3 = t*t2;
		one_minus_t = 1.0f-t;
		one_minus_t2 = one_minus_t*one_minus_t;
		one_minus_t3 = one_minus_t*one_minus_t2;

		// compute the Bezier basis functions
		verts[i].x = one_minus_t3*p0[X] + 3.0f*t*one_minus_t2*p1[X] + 3.0f*t2*one_minus_t*p2[X] + t3*p3[X];
		verts[i].y = one_minus_t3*p0[Y] + 3.0f*t*one_minus_t2*p1[Y] + 3.0f*t2*one_minus_t*p2[Y] + t3*p3[Y];

		verts[i].oow = 1.0f;
		verts[i].r = 255.0f;
		verts[i].g =   0.0f;
		verts[i].b =   0.0f;
		verts[i].a = 255.0f;

		grDrawLine(&verts[i-1], &verts[i]);

		// f(t) = [(1-t)^3]*P0 + [3*t*(1-t)^2]*P1 + [3*t^2*(1-t)]*P2 + [t^3]*P3
		// f'(t) = [-3*(1-t)^2]*P0 + [3-12*t+9*t^2]*P1 + [6*t - 9*t^2]*P2 + [3*t^2]*P3
		float xvel, yvel, vel;
		GrVertex v0, v1;
		v0.oow = v1.oow = 1.0f;
		v0.r = v1.r =   0.0f;
		v0.g = v1.g = 255.0f;
		v0.b = v1.b =   0.0f;
		xvel = (-3.0f*one_minus_t2)*p0[X] + (3.0f - 12.0f*t + 9.0f*t2)*p1[X] + (6.0f*t - 9.0f*t2)*p2[X] + (3.0f*t2)*p3[X];
		yvel = (-3.0f*one_minus_t2)*p0[Y] + (3.0f - 12.0f*t + 9.0f*t2)*p1[Y] + (6.0f*t - 9.0f*t2)*p2[Y] + (3.0f*t2)*p3[Y];
		vel = fsqrt(xvel*xvel + yvel*yvel);
		v0.x = verts[i].x;
		v0.y = verts[i].y;
		v1.x = v0.x + 0.1f*xvel;
		v1.y = v0.y + 0.1f*yvel;
		grDrawLine(&v0, &v1);
	}
}

void DrawBezierCurve()
{
	int i;
	float steps, t, t2, t3, one_minus_t, one_minus_t2, one_minus_t3;
	float x, y, xvel, yvel, vel;
	GrVertex vert0, vert1;

	grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);
	grColorCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);

	if (!gEditCurve)
	{
		steps = 10.0f*(gNumPoints & ~3);
		i = (int)(gCurrStep*(gNumPoints>>2)/steps);
		t = steps/(float)(gNumPoints>>2);
		t = (gCurrStep - i*t)/t;
		i <<= 2;
		t2 = t*t;
		t3 = t*t2;
		one_minus_t = 1.0f-t;
		one_minus_t2 = one_minus_t*one_minus_t;
		one_minus_t3 = one_minus_t*one_minus_t2;

		vert0.oow = 1.0f;
		vert0.r = 255.0f;
		vert0.g = 255.0f;
		vert0.b =   0.0f;
		vert0.a = 255.0f;
		vert1.oow = 1.0f;
		vert1.r = 255.0f;
		vert1.g = 255.0f;
		vert1.b =   0.0f;
		vert1.a = 255.0f;

		// f(t) = [(1-t)^3]*P0 + [3*t*(1-t)^2]*P1 + [3*t^2*(1-t)]*P2 + [t^3]*P3
		x = one_minus_t3*gPoints[i][X] + 3.0f*t*one_minus_t2*gPoints[i+1][X] + 3.0f*t2*one_minus_t*gPoints[i+2][X] + t3*gPoints[i+3][X];
		y = one_minus_t3*gPoints[i][Y] + 3.0f*t*one_minus_t2*gPoints[i+1][Y] + 3.0f*t2*one_minus_t*gPoints[i+2][Y] + t3*gPoints[i+3][Y];

		// f'(t) = [-3*(1-t)^2]*P0 + [3-12*t+9*t^2]*P1 + [6*t - 9*t^2]*P2 + [3*t^2]*P3
		xvel = (-3.0f*one_minus_t2)*gPoints[i][X] + (3.0f - 12.0f*t + 9.0f*t2)*gPoints[i+1][X] + (6.0f*t - 9.0f*t2)*gPoints[i+2][X] + (3.0f*t2)*gPoints[i+3][X];
		yvel = (-3.0f*one_minus_t2)*gPoints[i][Y] + (3.0f - 12.0f*t + 9.0f*t2)*gPoints[i+1][Y] + (6.0f*t - 9.0f*t2)*gPoints[i+2][Y] + (3.0f*t2)*gPoints[i+3][Y];
		vel = fsqrt(xvel*xvel + yvel*yvel);

		// factor in the velocity to make sure we move at a constant speed
		gCurrStep += 200.0f/vel;

		// transform screen coords to 512x512 texture coords
		x = 512.0f*x/SCREEN_RES_X;
		y = 512.0f*(SCREEN_RES_Y-y)/SCREEN_RES_Y;

		if (gTraversals & 1)
		{
			ClearAt((int)x, (int)y, BRUSH_THICKNESS);
		}
		else
		{
			BrushAt((int)x, (int)y, BRUSH_THICKNESS, -BRUSH_INTENSITY);
		}
		/*
		else if (gTraversals == 0)
		{
			BrushAt((int)x, (int)y, BRUSH_THICKNESS, -BRUSH_INTENSITY);
		}
		else // gTraversals == 2
		{
			BrushAt((int)x, (int)y, BRUSH_THICKNESS, BRUSH_INTENSITY);
		}
		*/

		if (gCurrStep >= steps)
		{
			gTraversals = (gTraversals+1) & 0x3;
			gCurrStep = 0.0f;
		}
	}
	else
	{
		for (i=0; i<(gNumPoints & ~3); i+=4)
		{
			DrawBezierSection(gPoints[i], gPoints[i+1], gPoints[i+2], gPoints[i+3]);
		}

		for (i=0; i<gNumPoints; i++)
		{
			vert0.oow = vert1.oow = 1.0f;
			vert0.r = vert1.r =   0.0f;
			vert0.g = vert1.g =   0.0f;
			vert0.b = vert1.b = 255.0f;
			vert0.x = gPoints[i][X] - 3.0f;
			vert0.y = gPoints[i][Y] - 3.0f;
			vert1.x = gPoints[i][X] + 3.0f;
			vert1.y = gPoints[i][Y] + 3.0f;
			grDrawLine(&vert0, &vert1);
			vert0.x += 6.0f;
			vert1.x -= 6.0f;
			grDrawLine(&vert0, &vert1);
		}
	}
}

// these control points spell out "donut" in cursive
// these points were created in 800x600 resolution
// so convert if necessary
static void SetupCurve()
{
	int i;

	gNumPoints = 100;
	gPoints[0][X] = 117.0f;
	gPoints[0][Y] = 345.0f;
	gPoints[1][X] =  53.0f;
	gPoints[1][Y] = 349.0f;
	gPoints[2][X] =  59.0f;
	gPoints[2][Y] = 428.0f;
	gPoints[3][X] = 107.0f;
	gPoints[3][Y] = 426.0f;
	gPoints[4][X] = 107.0f;
	gPoints[4][Y] = 426.0f;
	gPoints[5][X] = 150.0f;
	gPoints[5][Y] = 427.0f;
	gPoints[6][X] = 167.0f;
	gPoints[6][Y] = 334.0f;
	gPoints[7][X] = 204.0f;
	gPoints[7][Y] = 214.0f;
	gPoints[8][X] = 204.0f;
	gPoints[8][Y] = 214.0f;
	gPoints[9][X] = 233.0f;
	gPoints[9][Y] =  94.0f;
	gPoints[10][X] = 185.0f;
	gPoints[10][Y] = 112.0f;
	gPoints[11][X] = 171.0f;
	gPoints[11][Y] = 261.0f;
	gPoints[12][X] = 171.0f;
	gPoints[12][Y] = 261.0f;
	gPoints[13][X] = 155.0f;
	gPoints[13][Y] = 352.0f;
	gPoints[14][X] = 157.0f;
	gPoints[14][Y] = 414.0f;
	gPoints[15][X] = 189.0f;
	gPoints[15][Y] = 426.0f;
	gPoints[16][X] = 189.0f;
	gPoints[16][Y] = 426.0f;
	gPoints[17][X] = 226.0f;
	gPoints[17][Y] = 434.0f;
	gPoints[18][X] = 233.0f;
	gPoints[18][Y] = 350.0f;
	gPoints[19][X] = 288.0f;
	gPoints[19][Y] = 347.0f;
	gPoints[20][X] = 288.0f;
	gPoints[20][Y] = 347.0f;
	gPoints[21][X] = 297.0f;
	gPoints[21][Y] = 344.0f;
	gPoints[22][X] = 264.0f;
	gPoints[22][Y] = 347.0f;
	gPoints[23][X] = 247.0f;
	gPoints[23][Y] = 373.0f;
	gPoints[24][X] = 247.0f;
	gPoints[24][Y] = 373.0f;
	gPoints[25][X] = 226.0f;
	gPoints[25][Y] = 391.0f;
	gPoints[26][X] = 237.0f;
	gPoints[26][Y] = 433.0f;
	gPoints[27][X] = 278.0f;
	gPoints[27][Y] = 432.0f;
	gPoints[28][X] = 278.0f;
	gPoints[28][Y] = 432.0f;
	gPoints[29][X] = 328.0f;
	gPoints[29][Y] = 426.0f;
	gPoints[30][X] = 317.0f;
	gPoints[30][Y] = 364.0f;
	gPoints[31][X] = 299.0f;
	gPoints[31][Y] = 352.0f;
	gPoints[32][X] = 299.0f;
	gPoints[32][Y] = 352.0f;
	gPoints[33][X] = 301.0f;
	gPoints[33][Y] = 335.0f;
	gPoints[34][X] = 315.0f;
	gPoints[34][Y] = 355.0f;
	gPoints[35][X] = 355.0f;
	gPoints[35][Y] = 342.0f;
	gPoints[36][X] = 355.0f;
	gPoints[36][Y] = 342.0f;
	gPoints[37][X] = 402.0f;
	gPoints[37][Y] = 326.0f;
	gPoints[38][X] = 361.0f;
	gPoints[38][Y] = 373.0f;
	gPoints[39][X] = 344.0f;
	gPoints[39][Y] = 408.0f;
	gPoints[40][X] = 344.0f;
	gPoints[40][Y] = 408.0f;
	gPoints[41][X] = 319.0f;
	gPoints[41][Y] = 452.0f;
	gPoints[42][X] = 345.0f;
	gPoints[42][Y] = 418.0f;
	gPoints[43][X] = 361.0f;
	gPoints[43][Y] = 383.0f;
	gPoints[44][X] = 361.0f;
	gPoints[44][Y] = 383.0f;
	gPoints[45][X] = 380.0f;
	gPoints[45][Y] = 341.0f;
	gPoints[46][X] = 389.0f;
	gPoints[46][Y] = 338.0f;
	gPoints[47][X] = 428.0f;
	gPoints[47][Y] = 335.0f;
	gPoints[48][X] = 428.0f;
	gPoints[48][Y] = 335.0f;
	gPoints[49][X] = 465.0f;
	gPoints[49][Y] = 337.0f;
	gPoints[50][X] = 428.0f;
	gPoints[50][Y] = 375.0f;
	gPoints[51][X] = 412.0f;
	gPoints[51][Y] = 410.0f;
	gPoints[52][X] = 412.0f;
	gPoints[52][Y] = 410.0f;
	gPoints[53][X] = 409.0f;
	gPoints[53][Y] = 449.0f;
	gPoints[54][X] = 441.0f;
	gPoints[54][Y] = 429.0f;
	gPoints[55][X] = 459.0f;
	gPoints[55][Y] = 412.0f;
	gPoints[56][X] = 459.0f;
	gPoints[56][Y] = 412.0f;
	gPoints[57][X] = 485.0f;
	gPoints[57][Y] = 384.0f;
	gPoints[58][X] = 495.0f;
	gPoints[58][Y] = 364.0f;
	gPoints[59][X] = 503.0f;
	gPoints[59][Y] = 352.0f;
	gPoints[60][X] = 503.0f;
	gPoints[60][Y] = 352.0f;
	gPoints[61][X] = 523.0f;
	gPoints[61][Y] = 318.0f;
	gPoints[62][X] = 512.0f;
	gPoints[62][Y] = 346.0f;
	gPoints[63][X] = 479.0f;
	gPoints[63][Y] = 399.0f;
	gPoints[64][X] = 479.0f;
	gPoints[64][Y] = 399.0f;
	gPoints[65][X] = 462.0f;
	gPoints[65][Y] = 445.0f;
	gPoints[66][X] = 493.0f;
	gPoints[66][Y] = 435.0f;
	gPoints[67][X] = 513.0f;
	gPoints[67][Y] = 431.0f;
	gPoints[68][X] = 513.0f;
	gPoints[68][Y] = 431.0f;
	gPoints[69][X] = 541.0f;
	gPoints[69][Y] = 420.0f;
	gPoints[70][X] = 551.0f;
	gPoints[70][Y] = 399.0f;
	gPoints[71][X] = 571.0f;
	gPoints[71][Y] = 364.0f;
	gPoints[72][X] = 571.0f;
	gPoints[72][Y] = 364.0f;
	gPoints[73][X] = 596.0f;
	gPoints[73][Y] = 312.0f;
	gPoints[74][X] = 585.0f;
	gPoints[74][Y] = 352.0f;
	gPoints[75][X] = 559.0f;
	gPoints[75][Y] = 393.0f;
	gPoints[76][X] = 559.0f;
	gPoints[76][Y] = 393.0f;
	gPoints[77][X] = 534.0f;
	gPoints[77][Y] = 443.0f;
	gPoints[78][X] = 572.0f;
	gPoints[78][Y] = 441.0f;
	gPoints[79][X] = 598.0f;
	gPoints[79][Y] = 417.0f;
	gPoints[80][X] = 598.0f;
	gPoints[80][Y] = 417.0f;
	gPoints[81][X] = 628.0f;
	gPoints[81][Y] = 382.0f;
	gPoints[82][X] = 647.0f;
	gPoints[82][Y] = 355.0f;
	gPoints[83][X] = 673.0f;
	gPoints[83][Y] = 279.0f;
	gPoints[84][X] = 673.0f;
	gPoints[84][Y] = 279.0f;
	gPoints[85][X] = 740.0f;
	gPoints[85][Y] = 114.0f;
	gPoints[86][X] = 712.0f;
	gPoints[86][Y] = 121.0f;
	gPoints[87][X] = 686.0f;
	gPoints[87][Y] = 212.0f;
	gPoints[88][X] = 686.0f;
	gPoints[88][Y] = 212.0f;
	gPoints[89][X] = 657.0f;
	gPoints[89][Y] = 288.0f;
	gPoints[90][X] = 630.0f;
	gPoints[90][Y] = 395.0f;
	gPoints[91][X] = 645.0f;
	gPoints[91][Y] = 421.0f;
	gPoints[92][X] = 645.0f;
	gPoints[92][Y] = 421.0f;
	gPoints[93][X] = 669.0f;
	gPoints[93][Y] = 450.0f;
	gPoints[94][X] = 705.0f;
	gPoints[94][Y] = 434.0f;
	gPoints[95][X] = 747.0f;
	gPoints[95][Y] = 400.0f;
	gPoints[96][X] = 616.0f;
	gPoints[96][Y] = 203.0f;
	gPoints[97][X] = 651.0f;
	gPoints[97][Y] = 205.0f;
	gPoints[98][X] = 722.0f;
	gPoints[98][Y] = 200.0f;
	gPoints[99][X] = 772.0f;
	gPoints[99][Y] = 190.0f;

	// convert from 800x600 to current resolution
	for (i=0; i<gNumPoints; i++)
	{
		gPoints[i][X] *= SCREEN_RES_X/800.0f;
		gPoints[i][Y] *= SCREEN_RES_Y/600.0f;
	}
}

FxBool CreateLightMapTexture(GrTexInfo *tex_info)
{
	FxU16 *data;
	int s, t, dist_squared;
	float factor;

	tex_info->data = new FxU16[256*256];
	if (!tex_info->data)
	{
		return FXFALSE;
	}
#ifdef USE_GLIDE3
	tex_info->largeLodLog2 = GR_LOD_LOG2_256;
	tex_info->smallLodLog2 = GR_LOD_LOG2_256;
	tex_info->aspectRatioLog2 = GR_ASPECT_LOG2_1x1;
#else
	tex_info->largeLod = GR_LOD_256;
	tex_info->smallLod = GR_LOD_256;
	tex_info->aspectRatio = GR_ASPECT_1x1;
#endif // USE_GLIDE3
	tex_info->format = GR_TEXFMT_ALPHA_INTENSITY_88;

	data = (FxU16 *)tex_info->data;
	for (t=0; t<256; t++)
	{
		for (s=0; s<256; s++)
		{
			dist_squared = SQR(s-128) + SQR(t-128);
			*data = 0xff00;
			if (dist_squared <= 128*128)
			{
				factor = fsqrt((float)dist_squared/(128.0f*128.0f));
				*data |= 255 - (int)(255.0f*factor);
			}
			data++;
		}
	}

	return FXTRUE;
}
