/* vi:set ts=8 sts=0 sw=8:
 * $Id: htmltags.c,v 1.6 1999/01/21 21:36:57 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <string.h>
#include <gtk/gtk.h>
#include "htmltags.h"
#include "toolbar.h"
#include "doc.h"
#include "dialog.h"
#include "prefs.h"
#include "misc.h"


/*** local types ***/
#define HTS_BIG		"Big"
#define HTS_BOLD	"Bold"
#define HTS_CENT	"Center"
#define HTS_CMNT	"Comment"
#define HTS_EMPH	"Emphasis"
#define HTS_HDR1	"Header1"
#define HTS_HDR2	"Header2"
#define HTS_HDR3	"Header3"
#define HTS_HDR4	"Header4"
#define HTS_HDR5	"Header5"
#define HTS_HDR6	"Header6"
#define HTS_IMG		"Image"
#define HTS_ITAL	"Italic"
#define HTS_LBRK	"LineBreak"
#define HTS_LEFT	"LeftJust"
#define HTS_LINK	"Link"
#define HTS_LITM	"ListItem"
#define HTS_ORDR	"Ordered"
#define HTS_PARA	"Paragraph"
#define HTS_RGHT	"RightJust"
#define HTS_SEP		"Separator"
#define HTS_SMLL	"Small"
#define HTS_STRG	"Strong"
#define HTS_STRK	"Strikeout"
#define HTS_SUB		"Subscript"
#define HTS_SUP		"Superscript"
#define HTS_TARG	"Target"
#define HTS_TTL		"Title"
#define HTS_TTY		"Typewriter"
#define HTS_ULNE	"Underline"
#define HTS_UORD	"Unordered"

/*
 * there are basically two types of html tags we'll be handling:
 *
 * 1. the simple kind, where the start of the tag and the end of the tag are
 *    more or less "symmetrical" (plus the addition of a forward slash).  for
 *    example, "<B></B>", "<I></I>".  simple tags use the callback function
 *    html_tag_simple_cb().
 *
 * 2. the somewhat more complex kind, where the start and end are asymmetrical.
 *    that is, the beginning of the tag may contain other arguments, which
 *    don't have a corresponding counterpart in the end of the tag.  for
 *    example, "<A HREF="http://www.foo.com"></A>".  complex tags use the
 *    callback function html_tag_complex_cb().
 *
 * in both cases, the beginning of the tag will be referred to as the "prefix",
 * while the end of the tag is referred to as the "suffix".
 *
 * additionally, for readability purposes, we may prepend a carriage return
 * before the prefix, or append one after the suffix.
 *
 * the enum and struct below handle these various options.
 *
 * and yes, we'll be using lots and lots of tables instead of being a bonehead
 * and writing data specific code (which leads to serious code bloat) for
 * inserting html tags like in another GTK based text editor.
 */
typedef enum {
	Before,		/* before the tag prefix */
	Between,	/* between the prefix and suffix */
	After,		/* after the tag suffix */
	NOPOS
} CursorPos;

typedef struct {
	html_action_t	action;		/* see htmltags.h */
	char *		tagname;	/* see above #define's */
	char *		prefix;		/* see above for prefix & suffix */
	char *		suffix;		/* "/" if exists, NULL if none */
	char *		data;		/* text that goes in prfx, not sffx */
	CursorPos	cursorpos;	/* where to place the cursor */
	gboolean	cr_before;	/* CR before prefix? */
	gboolean	cr_after;	/* CR after suffix? */
} htmltags_t;


/*
 * each page item/entry for a complex html dialog box can be of only two types:
 * an entry widget or a combo (pulldown) widget (note that the pulldown is
 * actually an extension of the entry widget).
 *
 * the third "widget" type, VertSpace, is used as a separator.
 */
typedef enum {
	EntryWgt,
	PdownWgt,
	VertSpace	/* this isn't really a widget */
} widget_t;


/*
 * this structure defines what each page item/entry represents.  e.g., what the
 * option name as part of the HTML tag itself.
 */
typedef struct {
	char *		itemname;	/* "SPACE" if VertSpace */
	widget_t	wgttype;	/* see above */
	gboolean	is_assign;	/* is assignment? */
	gboolean	quoted;		/* need quotes? */
	char *		tagtext;	/* text to insert as part of the tag */
	char **		itemlist;	/* pulldown items */
	GtkWidget *	wgt;		/* pointer to widget */
} html_wgt_info_t;


/*
 * this defines what a page is: a page name, and a list of widgets representing
 * the entries on that page.
 */
typedef struct {
	char *		pgname;	/* goes into notebook tab */
	html_wgt_info_t	*hwip;
} html_page_t;


/*
 * callback functions in this file really need more than one piece of informa-
 * tion for callback data.  this structure packages up the various things that
 * are usually needed.
 */
typedef struct {
	dialog_entry_t edp;	/* in dialog.h */
	doc_t *d;
	htmltags_t *htp;
	html_page_t *dlg_pages;
} html_entry_dlg_t;


/*** local function prototypes ***/
static void html_tag_simple_cb(GtkWidget *wgt, gpointer cbdata);
static void html_tag_complex_cb(GtkWidget *wgt, gpointer cbdata);
static void html_tag_insert(doc_t *d, char *buf, htmltags_t *htp);
static void html_entry_dlg(doc_t *d, htmltags_t *htp, char *title, char *msg,
			   GtkSignalFunc);
static void html_comment_exec(GtkWidget *wgt, gpointer cbdata);
static void html_target_exec(GtkWidget *wgt, gpointer cbdata);
static void html_link_img_exec(GtkWidget *wgt, gpointer cbdata);
static void html_dlg_cancel(GtkWidget *wgt, gpointer cbdata);
static void html_link_img_dlg(html_page_t *dlg_pages, doc_t *d,
			      htmltags_t *htp, char *title,
			      GtkSignalFunc execute_cb,
			      GtkSignalFunc cancel_cb);
static GtkWidget *html_tag_pulldown_create(char **itemlist);
static void html_dlg_destroy(GtkWidget *wgt, gpointer cbdata);
static void html_link_img_dlg_pages_add(html_page_t *dlg_pages,
					int maxentries, GtkNotebook *nb);
static void html_link_img_dlg_items_add(GtkTable *table, html_page_t *hpp);
static void html_link_img_dlg_buttons_add(GtkWidget *toplev,
					  GtkSignalFunc execute_cb,
					  GtkSignalFunc cancel_cb,
					  html_entry_dlg_t *hedp);
static int html_link_img_calc_len(html_entry_dlg_t *hedp);
static void html_link_img_attr_add(html_page_t *dlg_pages, char *buf);
static char *html_tag_simple_buf_build(htmltags_t *htp);
static char *html_link_img_buf_build(int len, html_entry_dlg_t *hedp);
static toolbar_data_t *html_find_tb_nfo(char *tagname);
static void html_menu_cb_common(html_action_t action, gpointer cbdata);
static htmltags_t *html_find_tag_by_name(char *tagname);
static void html_tag_simple_common(htmltags_t *htp, win_t *w);
static void html_tag_complex_common(htmltags_t *htp, doc_t *d);
static htmltags_t *html_find_tag_by_action(html_action_t action);
#if 0
static char *html_action_to_name(html_action_t action);
static html_action_t html_name_to_action(char *tagname);
#endif


/*** local variables ***/
static htmltags_t htmltags[] = {
	{ HtmlBig,	HTS_BIG,  "BIG",    "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlBold,	HTS_BOLD, "B",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlCent,	HTS_CENT, "CENTER", "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlCmnt,	HTS_CMNT, "!--",    "/",  NULL,
			After,   FALSE, FALSE },
	{ HtmlEmph,	HTS_EMPH, "EM",     "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlH1,	HTS_HDR1, "H1",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlH2,	HTS_HDR2, "H2",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlH3,	HTS_HDR3, "H3",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlH4,	HTS_HDR4, "H4",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlH5,	HTS_HDR5, "H5",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlH6,	HTS_HDR6, "H6",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlImg,	HTS_IMG,  "IMG",    NULL, NULL,
			After,   FALSE, FALSE },
	{ HtmlItal,	HTS_ITAL, "I",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlLbrk,	HTS_LBRK, "BR",     NULL, NULL,
			After,   FALSE, FALSE },
	{ HtmlLeft,	HTS_LEFT, "DIV",    "/",  "ALIGN=left",
			Between, TRUE,  TRUE  },
	{ HtmlLink,	HTS_LINK, "A",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlLitm,	HTS_LITM, "LI",     NULL, NULL,
			After,   FALSE, FALSE },
	{ HtmlOrdr,	HTS_ORDR, "OL",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlPara,	HTS_PARA, "P",      NULL, NULL,
			After,   TRUE,  TRUE  },
	{ HtmlRght,	HTS_RGHT, "DIV",    "/",  "ALIGN=right",
			Between, TRUE,  TRUE  },
	{ HtmlSep,	HTS_SEP,  "HR",     NULL, NULL,
			After,   TRUE,  TRUE  },
	{ HtmlSmll,	HTS_SMLL, "SMALL",  "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlStrg,	HTS_STRG, "STRONG", "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlStrk,	HTS_STRK, "S",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlSub,	HTS_SUB,  "SUB",    "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlSup,	HTS_SUP,  "SUP",    "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlTarg,	HTS_TARG, "A",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlTtl,	HTS_TTL,  "TITLE",  "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlTty,	HTS_TTY,  "TT",     "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlUlne,	HTS_ULNE, "U",      "/",  NULL,
			Between, FALSE, FALSE },
	{ HtmlUord,	HTS_UORD, "UL",     "/",  NULL,
			Between, FALSE, TRUE  },
	{ HtmlNONE }
}; /* htmltags[] */

static char *shape_items[] =
	{ "none", "rect", "circle", "poly", "default", NULL };
static char *dir_items[] =
	{ "auto", "ltr", "rtl", NULL };
static char *align_items[] =
	{ "none", "top", "middle", "bottom", "left", "right", NULL };
static char *ismap_items[] =
	{ "AUTO", "ISMAP", NULL };

/* BEGIN: these are the pages that go into the LINK dialog notebook */
/*
 * SHAPE = { rect, circle, poly, default, NONE }  no quotes
 * DIR = { implied, ltr, rtl }  no quotes
 */
static html_wgt_info_t link_page_main[] = {
	{ " Link Destination ",	 EntryWgt, TRUE, TRUE,  "HREF" },
	{ " Named Link End ",	 EntryWgt, TRUE, TRUE,  "NAME" },
	{ " Link Relations ",	 EntryWgt, TRUE, TRUE,  "REL" },
	{ " Reverse Relations ", EntryWgt, TRUE, TRUE,  "REV" },
	{ " Target Frame Name ", EntryWgt, TRUE, TRUE,  "TARGET" },
	{ " Coordinates ",	 EntryWgt, TRUE, TRUE,  "COORDS" },
	{ "SPACE",		 VertSpace },
	{ " Shape ",		 PdownWgt, TRUE, FALSE, "SHAPE", shape_items },
	{ " Content Type ",	 EntryWgt, TRUE, TRUE,  "TYPE" },
	{ NULL }
}; /* link_page_main */

static html_wgt_info_t link_page_core[] = {
	{ " Character Encoding ", EntryWgt, TRUE, TRUE,  "CHARSET" },
	{ " Language Code ",	  EntryWgt, TRUE, TRUE,  "HREFLANG" },
	{ " Title ",		  EntryWgt, TRUE, TRUE,  "TITLE" },
	{ " Link End ID ",	  EntryWgt, TRUE, TRUE,  "ID" },
	{ " Class ",		  EntryWgt, TRUE, TRUE,  "CLASS" },
	{ " Style ",		  EntryWgt, TRUE, TRUE,  "STYLE" },
	{ "SPACE",		  VertSpace },
	{ " Language ",		  EntryWgt, TRUE, TRUE,  "LANG" },
	{ " Text Direction ",	  PdownWgt, TRUE, FALSE, "DIR", dir_items },
	{ NULL }
}; /* link_page_core */

static html_wgt_info_t link_page_events[] = {
	{ " On Click ",		EntryWgt, TRUE, TRUE, "ONCLICK" },
	{ " On Double Click ",	EntryWgt, TRUE, TRUE, "ONDBLCLICK" },
	{ " On Mouse Over ",	EntryWgt, TRUE, TRUE, "ONMOUSEOVER" },
	{ " On Mouse Down ",	EntryWgt, TRUE, TRUE, "ONMOUSEDOWN" },
	{ " On Mouse Move ",	EntryWgt, TRUE, TRUE, "ONMOUSEMOVE" },
	{ " On Mouse Out ",	EntryWgt, TRUE, TRUE, "ONMOUSEOUT" },
	{ " On Mouse Up ",	EntryWgt, TRUE, TRUE, "ONMOUSEUP" },
	{ "SPACE",		VertSpace },
	{ " On Key Down ",	EntryWgt, TRUE, TRUE, "ONKEYDOWN" },
	{ " On Key Press ",	EntryWgt, TRUE, TRUE, "ONKEYPRESS" },
	{ " On Key Up ",	EntryWgt, TRUE, TRUE, "ONKEYUP" },
	{ NULL }
}; /* link_page_events */

static html_wgt_info_t link_page_extra[] = {
	{ " Accessibility Key ", EntryWgt, TRUE, TRUE, "ACCESS" },
	{ " Focus ",		 EntryWgt, TRUE, TRUE, "ONFOCUS" },
	{ " Blur ",		 EntryWgt, TRUE, TRUE, "ONBLUR" },
	{ " Tab Index ",	 EntryWgt, TRUE, TRUE, "TABINDEX" },
	{ NULL }
}; /* link_page_extra */
/* END: these are the pages that go into the LINK dialog notebook */

/* BEGIN: these are the pages that go into the IMAGE dialog notebook */
/*
 * DIR = { implied, ltr, rtl }  no quotes
 * ALIGN = { none, top, middle, bottom, left, right }   no quotes
 * ISMAP = { none }	if true, no assignment
 */
static html_wgt_info_t img_page_main[] = {
	{ " Graphics URL ",	EntryWgt, TRUE, TRUE, "SRC" },
	{ " Alternative Text ",	EntryWgt, TRUE, TRUE, "ALT" },
	{ " Width ",		EntryWgt, TRUE, TRUE, "WIDTH" },
	{ " Height ",		EntryWgt, TRUE, TRUE, "HEIGHT" },
	{ " Border ",		EntryWgt, TRUE, TRUE, "BORDER" },
	{ NULL }
}; /* img_page_main */

static html_wgt_info_t img_page_core[] = {
	{ " Title ",		EntryWgt, TRUE, TRUE,  "TITLE" },
	{ " Link End ID ",	EntryWgt, TRUE, TRUE,  "ID" },
	{ " Class ",		EntryWgt, TRUE, TRUE,  "CLASS" },
	{ " Style ",		EntryWgt, TRUE, TRUE,  "STYLE" },
	{ " Language ",		EntryWgt, TRUE, TRUE,  "LANG" },
	{ " Text Direction ",	PdownWgt, TRUE, FALSE, "DIR", dir_items },
	{ NULL }
}; /* img_page_core */

static html_wgt_info_t img_page_events[] = {
	{ " On Click ",		EntryWgt, TRUE, TRUE, "ONCLICK" },
	{ " On Double Click ",	EntryWgt, TRUE, TRUE, "ONDBLCLICK" },
	{ " On Mouse Over ",	EntryWgt, TRUE, TRUE, "ONMOUSEOVER" },
	{ " On Mouse Down ",	EntryWgt, TRUE, TRUE, "ONMOUSEDOWN" },
	{ " On Mouse Move ",	EntryWgt, TRUE, TRUE, "ONMOUSEMOVE" },
	{ " On Mouse Out ",	EntryWgt, TRUE, TRUE, "ONMOUSEOUT" },
	{ " On Mouse Up ",	EntryWgt, TRUE, TRUE, "ONMOUSEUP" },
	{ "SPACE",		VertSpace },
	{ " On Key Down ",	EntryWgt, TRUE, TRUE, "ONKEYDOWN" },
	{ " On Key Press ",	EntryWgt, TRUE, TRUE, "ONKEYPRESS" },
	{ " On Key Up ",	EntryWgt, TRUE, TRUE, "ONKEYUP" },
	{ NULL }
}; /* imgk_page_events */

static html_wgt_info_t img_page_align[] = {
	{ " Align ",		  PdownWgt, TRUE, FALSE, "ALIGN", align_items },
	{ " Horizontal Spacing ", EntryWgt, TRUE, TRUE,  "HSPACE" },
	{ " Vertical Spacing ",	  EntryWgt, TRUE, TRUE,  "VSPACE" },
	{ NULL }
}; /* img_page_align */

static html_wgt_info_t img_page_extra[] = {
	{ " Sensitive Map ",	PdownWgt, FALSE, FALSE, "ISMAP", ismap_items },
	{ " Client Usemap ",	EntryWgt, TRUE, TRUE,   "USEMAP" },
	{ " Long Desc Link ",	EntryWgt, TRUE, TRUE,   "LONGDESC" },
	{ NULL }
}; /* img_page_extra */
/* END: these are the pages that go into the IMAGE dialog notebook */

static html_page_t link_tag_dlg[] = {
	{ " Main ",	  link_page_main },
	{ " Core, i18n ", link_page_core },
	{ " Events ",	  link_page_events },
	{ " Extra ",	  link_page_extra },
	{ NULL }
}; /* link_tag_dlg[] */

static html_page_t img_tag_dlg[] = {
	{ " Main ",	  img_page_main },
	{ " Core, i18n ", img_page_core },
	{ " Events ",	  img_page_events },	/* same as before */
	{ " Align ",	  img_page_align },
	{ " Extra ",	  img_page_extra },
	{ NULL }
}; /* img_tag_dlg[] */


/*
 * note that the tooltip private name/data must match with the value of the
 * tagname field in htmltags_t.
 */
static toolbar_data_t html_tbdata[] = {
	{ "Title",	"Title",	HTS_TTL,
		"tb_title.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "Link",	"Link",		HTS_LINK,
		"tb_link.xpm",		(GtkSignalFunc)html_tag_complex_cb },
	{ "Target",	"Target",	HTS_TARG,
		"tb_target.xpm",	(GtkSignalFunc)html_tag_complex_cb },
	{ "Image",	"Image",	HTS_IMG,
		"tb_image.xpm",		(GtkSignalFunc)html_tag_complex_cb },
	{ "Comment",	"Comment",	HTS_CMNT,
		"tb_comment.xpm",	(GtkSignalFunc)html_tag_complex_cb },
	{ "<H1>",	"Header1",	HTS_HDR1,
		"tb_h1.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "<H2>",	"Header2",	HTS_HDR2,
		"tb_h2.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "<H3>",	"Header3",	HTS_HDR3,
		"tb_h3.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "Left",	"Align Left",	HTS_LEFT,
		"tb_leftjust.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "Center",	"Center",	HTS_CENT,
		"tb_centjust.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "Right",	"Align Right",	HTS_RGHT,
		"tb_rightjust.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<B>",	"Bold",		HTS_BOLD,
		"tb_bold.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "<I>",	"Italic",	HTS_ITAL,
		"tb_italic.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<U>",	"Underline",	HTS_ULNE,
		"tb_underline.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<TT>",	"Typewriter",	HTS_TTY,
		"tb_typewriter.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<S>",	"Strikeout",	HTS_STRK,
		"tb_strikeout.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<EM>",	"Emphasis",	HTS_EMPH,
		"tb_emphasis.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "Strong",	"Strong",	HTS_STRG,
		"tb_strong.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<SUP>",	"Superscript",	HTS_SUP,
		"tb_superscript.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<SUB>",	"Subscript",	HTS_SUB,
		"tb_subscript.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<HR>",	"Separator",	HTS_SEP,
		"tb_separator.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<P>",	"Paragraph",	HTS_PARA,
		"tb_paragraph.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<BR>",	"Line Break",	HTS_LBRK,
		"tb_linebreak.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<BIG>",	"Big",		HTS_BIG,
		"tb_big.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "Small",	"Small",	HTS_SMLL,
		"tb_small.xpm",		(GtkSignalFunc)html_tag_simple_cb },
	{ "<OL>",	"Ordered List",	HTS_ORDR,
		"tb_ordered.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<UL>",	"Unordered List",	HTS_UORD,
		"tb_unordered.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ "<LI>",	"List Item",	HTS_LITM,
		"tb_listitem.xpm",	(GtkSignalFunc)html_tag_simple_cb },
	{ NULL }
}; /* html_tbdata */


/*
 * since both the link and image dialogs share the tables up above, there can
 * be at most one of each at any time.  these are the top-level dialog widgets
 * for each.  they are non-NULL when one of them is already present, and get
 * reset to NULL when destroyed (in html_dlg_destroy()).
 */
static GtkWidget *link_dlg = NULL;
static GtkWidget *img_dlg = NULL;


/*** global function defintions ***/
/*
 * PUBLIC: html_tb_create
 *
 * in:
 *	w: current window
 *	parent: parent widget (box type)
 *
 * out:
 *	tb: the toolbar
 *	tb_h: toolbar handle
 */
void
html_tb_create(win_t *w, GtkWidget *parent, GtkWidget **tb, GtkWidget **tb_h)
{
	toolbar_create(w, parent, html_tbdata, tb, tb_h);
	if (IS_SHOW_HTML_TOOLBAR())
		gtk_widget_show(*tb_h);
	else
		gtk_widget_hide(*tb_h);
} /* html_tb_create */


#ifdef GTK_HAVE_FEATURES_1_1_0

/*
 * PUBLIC: html_menu_cb
 *
 * all HTML insertions from the main menu use this callback.  see menu.c
 */
void
html_menu_cb(GtkWidget *wgt, gpointer cbdata, guint cbaction)
{
	html_menu_cb_common((html_action_t)cbaction, cbdata);
} /* html_menu_cb */

#else

/*
 * the next bunch of callbacks is kind of inefficient since we already have
 * html_tag_simple_cb() and html_tag_complex_cb() because the MenuFactory is a
 * bit limited.
 */
void
html_title_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTtl, cbdata);
}

void
html_comment_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlCmnt, cbdata);
}

void
html_link_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLink, cbdata);
}

void
html_target_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTarg, cbdata);
}

void
html_image_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlImg, cbdata);
}

void
html_h1_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH1, cbdata);
}

void
html_h2_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH2, cbdata);
}

void
html_h3_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH3, cbdata);
}

void
html_h4_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH4, cbdata);
}

void
html_h5_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH5, cbdata);
}

void
html_h6_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH6, cbdata);
}

void
html_center_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlCent, cbdata);
}

void
html_right_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlRght, cbdata);
}

void
html_paragraph_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlPara, cbdata);
}

void
html_linebreak_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLbrk, cbdata);
}

void
html_separator_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSep, cbdata);
}

void
html_bold_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlBold, cbdata);
}

void
html_italic(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlItal, cbdata);
}

void
html_underline_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlUlne, cbdata);
}

void
html_typewriter_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTty, cbdata);
}

void
html_strikeout_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlStrk, cbdata);
}

void
html_emphasis_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlEmph, cbdata);
}

void
html_strong_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlStrg, cbdata);
}

void
html_subscript_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSub, cbdata);
}

void
html_superscript_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSup, cbdata);
}

void
html_big_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlBig, cbdata);
}

void
html_small_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSmll, cbdata);
}

void
html_ordered_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlOrdr, cbdata);
}

void
html_unordered_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlUord, cbdata);
}

void
html_listitem_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLitm, cbdata);
}

#endif	/* gtk+-1.0.x */


/*** local function defintions ***/
/*
 * PRIVATE: html_menu_cb_common
 *
 * common routine called by all the braindead menu callbacks for inserting html
 * tags.
 */
static void
html_menu_cb_common(html_action_t action, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	toolbar_data_t *tbdp;
	htmltags_t *htp;

	htp = html_find_tag_by_action(action);
	tbdp = html_find_tb_nfo(htp->tagname);

	if (tbdp->callback == GTK_SIGNAL_FUNC(html_tag_simple_cb)) {
		html_tag_simple_common(htp, w);
	} else if (tbdp->callback == GTK_SIGNAL_FUNC(html_tag_complex_cb)) {
		html_tag_complex_common(htp, (doc_t *)w->curdoc);
	}
} /* html_menu_cb_common */


/*
 * PRIVATE: html_find_tag_by_action
 *
 * finds the HTML tag name by its action enumeration
 */
static htmltags_t *
html_find_tag_by_action(html_action_t action)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->action != HtmlNONE; htp++) {
		if (htp->action == action)
			break;
	}
	g_assert(htp->action != HtmlNONE);

	return htp;
} /* html_htp_by_action */


#if 0
static html_action_t
html_name_to_action(char *tagname)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->action != HtmlNONE; htp++) {
		if (strcmp(htp->tagname, tagname) == 0)
			break;
	}
	g_assert(htp->action != HtmlNONE);

	return (htp->action);
} /* html_name_to_action */


static char *
html_action_to_name(html_action_t action)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->action != HtmlNONE; htp++) {
		if (htp->action == action)
			break;
	}
	g_assert(htp->action != HtmlNONE);

	return (htp->tagname);
} /* html_action_to_name */
#endif


/*
 * PRIVATE: html_tag_simple_cb
 *
 * handles simple tags.  e.g., tags that are symmetrical such as "<B></B>"
 *
 * more complex tags that are asymetrical are handled elsewhere.
 */
static void
html_tag_simple_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GtkTooltipsData *tooltipsdata;
	htmltags_t *htp;

	/* find the html tag info in the table.  the method used is a little */
	tooltipsdata = gtk_tooltips_data_get(wgt);
	g_assert(tooltipsdata != NULL);
	htp = html_find_tag_by_name(tooltipsdata->tip_private);

	html_tag_simple_common(htp, w);
} /* html_tag_simple_cb */


/*
 * PRIVATE: html_find_tb_nfo
 *
 * finds the entry in the toolbar data table by tagname
 */
static toolbar_data_t *
html_find_tb_nfo(char *tagname)
{
	toolbar_data_t *tbdp = NULL;

	for (tbdp = html_tbdata; tbdp->tooltip_private_text != NULL; tbdp++) {
		if (strcmp(tbdp->tooltip_private_text, tagname) == 0)
			break;
	}

	g_assert(tbdp != NULL);
	return tbdp;
} /* html_find_tb_nfo */


/*
 * PRIVATE: html_find_tag_by_name
 *
 * finds the entry in the htmltags table by tagname
 */
static htmltags_t *
html_find_tag_by_name(char *tagname)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->tagname != NULL; htp++) {
		if (strcmp(tagname, htp->tagname) == 0)
			break;
	}
	g_assert(htp->tagname != NULL);

	return htp;
} /* html_find_tag_by_name */


/*
 * PRIVATE: html_tag_simple_common
 *
 * common routine for simple html tags
 */
static void
html_tag_simple_common(htmltags_t *htp, win_t *w)
{
	char *buf;

	/* build the buffer */
	buf = html_tag_simple_buf_build(htp);

	/* insert it into the text widget and adjust cursor position */
	html_tag_insert((doc_t *)w->curdoc, buf, htp);

	g_free(buf);
} /* html_tag_simple_common */


/*
 * PRIVATE: html_tag_simple_buf_build
 *
 * builds and returns a text buffer containing a simple/symmetrical HTML tag. 
 * called only from html_tag_simple_cb().
 */
static char *
html_tag_simple_buf_build(htmltags_t *htp)
{
	int len;
	char *buf;

	/* build a buffer for the html tag text to insert */
	len = strlen(htp->prefix) + 2 +
	      (htp->data ? (strlen(htp->data) + 1) : 0) +
	      (htp->suffix ? (1 + strlen(htp->prefix) + 2) : 1) +
	      (htp->cr_before ? 1 : 0) + (htp->cr_after ? 1 : 0) + 3;
	buf = (char *)g_malloc(len);
	/* insert prefix */
	if (htp->cr_before) {
		sprintf(buf, "\n<%s", htp->prefix);
	} else
		sprintf(buf, "<%s", htp->prefix);
	if (htp->data) {
		strcat(buf, " ");
		strcat(buf, htp->data);
	}
	strcat(buf, ">");

	/* add suffix */
	if (htp->suffix) {
		strcat(buf, "</");
		strcat(buf, htp->prefix);
		strcat(buf, ">");
	}
	if (htp->cr_after)
		strcat(buf, "\n");

	GNPDBG_HTML(("html_tag_simple_buf_build: inserting buf (len = %d,"
		     "strlen = %ld) '%s'\n", len, (long)strlen(buf), buf));
	g_assert((size_t)strlen(buf) <= (size_t)len);

	return buf;
} /* html_tag_simple_buf_build */


/*
 * PRIVATE: html_tag_complex_cb
 *
 * toplevel callback to create a dialog box to let the user enter information
 * for a complex/asymmetrical HTML tag. 
 */
static void
html_tag_complex_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d = (doc_t *)w->curdoc;
	GtkTooltipsData *tooltipsdata;
	htmltags_t *htp;

	/* find the html tag info in the table */
	tooltipsdata = gtk_tooltips_data_get(wgt);
	g_assert(tooltipsdata != NULL);
	htp = html_find_tag_by_name(tooltipsdata->tip_private);

	html_tag_complex_common(htp, d);
} /* html_tag_complex_cb */


/*
 * PRIVATE: html_tag_complex_common
 *
 * common routine for all complex html tags
 */
static void
html_tag_complex_common(htmltags_t *htp, doc_t *d)
{
	switch (htp->action) {
	case HtmlTarg:
		html_entry_dlg(d, htp, "Insert HTML Target", "Target Name:",
			       GTK_SIGNAL_FUNC(html_target_exec));
		break;
	case HtmlCmnt:
		html_entry_dlg(d, htp, "Insert Comment", "Comment Text:",
			       GTK_SIGNAL_FUNC(html_comment_exec));
		break;
	case HtmlLink:
		html_link_img_dlg(link_tag_dlg, d, htp, "HTML Link Tag",
				  GTK_SIGNAL_FUNC(html_link_img_exec),
				  GTK_SIGNAL_FUNC(html_dlg_cancel));
		break;
	case HtmlImg:
		html_link_img_dlg(img_tag_dlg, d, htp, "HTML Image Tag",
				  GTK_SIGNAL_FUNC(html_link_img_exec),
				  GTK_SIGNAL_FUNC(html_dlg_cancel));
		break;
	default:
		g_warning("html_tag_complex_common: unknown action type = %d"
			  " (tagname='%s')\n", htp->action, htp->tagname);
		break;
	}
} /* html_tag_complex_common */


/*
 * PRIVATE: html_link_img_dlg
 *
 * routine to create a dialog box for either the link tag or the image tag.
 * called only from html_tag_complex_cb().
 *
 * dlg_pages: points to either link_tag_dlg[] or img_tag_dlg[]
 */
static void
html_link_img_dlg(html_page_t *dlg_pages, doc_t *d, htmltags_t *htp,
		  char *title, GtkSignalFunc execute_cb,
		  GtkSignalFunc cancel_cb)
{
	int maxentries;
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	html_entry_dlg_t *hedp;
	GtkWidget *dlg, *nb;

	if (dlg_pages == link_tag_dlg && link_dlg) {
		gdk_window_raise(link_dlg->window);
		return;
	}

	if (dlg_pages == img_tag_dlg && img_dlg) {
		gdk_window_raise(img_dlg->window);
		return;
	}

	/* create top-level window and notebook */
	dlg = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dlg), (title) ? title : "Edit Tag");
	nb = gtk_notebook_new();
	gtk_container_border_width(GTK_CONTAINER(nb), 6);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(nb), GTK_POS_TOP);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(nb));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), nb, TRUE, TRUE, 0);

	/* need to pass cur doc and htp ptr as callback data */
	hedp = (html_entry_dlg_t *)g_malloc(sizeof(html_entry_dlg_t));
	hedp->d = d;
	hedp->htp = htp;
	hedp->dlg_pages = dlg_pages;
	gtk_signal_connect(GTK_OBJECT(dlg), "destroy",
			   GTK_SIGNAL_FUNC(html_dlg_destroy), (gpointer)hedp);
	/* track which dialog window it is */
	if (dlg_pages == link_tag_dlg) {
		link_dlg = dlg;
		hedp->edp.toplev = link_dlg;
	} else if (dlg_pages == img_tag_dlg) {
		img_dlg = dlg;
		hedp->edp.toplev = img_dlg;
	}

	/* count max number of entries across all pages */
	for (maxentries = 0, hpp = dlg_pages; hpp->pgname; hpp++) {
		int count = 0;

		for (hwip = hpp->hwip; hwip->itemname; hwip++)
			count++;
		if (count > maxentries)
			maxentries = count;
	}

	/* add pages */
	html_link_img_dlg_pages_add(dlg_pages, maxentries, GTK_NOTEBOOK(nb));

	/* now the buttons */
	html_link_img_dlg_buttons_add(dlg, execute_cb, cancel_cb, hedp);

	gtk_widget_show_all(dlg);
} /* html_link_img_dlg */


/*
 * PRIVATE: html_link_img_dlg_buttons_add
 *
 * creates the "Ok" and "Cancel" buttons for the link/img dialog box.  called
 * only from html_link_img_dlg().
 */
static void
html_link_img_dlg_buttons_add(GtkWidget *toplev, GtkSignalFunc execute_cb,
			      GtkSignalFunc cancel_cb, html_entry_dlg_t *hedp)
{
	GtkWidget *hbox, *tmp;

	hbox = GTK_DIALOG(toplev)->action_area;
	gtk_container_border_width(GTK_CONTAINER(hbox), 5);

	tmp = misc_button_new_w_label(" Ok ", GTK_SIGNAL_FUNC(execute_cb),
				      hedp, hbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NORMAL,
#endif
				      NULL, 0, 0, 0, 0);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);

	tmp = misc_button_new_w_label(" Cancel ", GTK_SIGNAL_FUNC(cancel_cb),
				      hedp, hbox,
#ifdef GTK_HAVE_FEATURES_1_1_0
				      GTK_RELIEF_NORMAL,
#endif
				      NULL, 0, 0, 0, 0);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);
} /* html_link_img_dlg_buttons_add */


/*
 * PRIVATE: html_link_img_dlg_pages_add
 *
 * adds pages into the notebook for the link/img dialog box.  called only from
 * html_link_img_dlg().
 */
static void
html_link_img_dlg_pages_add(html_page_t *dlg_pages, int maxentries,
	GtkNotebook *nb)
{
	GtkWidget *frame, *table, *tmp;
	html_page_t *hpp;

	for (hpp = dlg_pages; hpp->pgname; hpp++) {

		GNPDBG_HTML(("html_link_img_dlg: page '%s'\n", hpp->pgname));

		/* it's a new page, so create a frame and table */
		frame = gtk_frame_new(NULL);
		gtk_container_border_width(GTK_CONTAINER(frame), 4);
		table = gtk_table_new(maxentries, 2, TRUE);
		gtk_container_border_width(GTK_CONTAINER(table), 4);
		gtk_container_add(GTK_CONTAINER(frame), table);
		tmp = gtk_label_new(hpp->pgname);
		gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);

		/* add each item for current page */
		html_link_img_dlg_items_add(GTK_TABLE(table), hpp);
	} /* for each page */
} /* html_link_img_dlg_pages_add */


/*
 * PRIVATE: html_link_img_dlg_items_add
 *
 * adds each item/entry into a page of a notebook.  called only from
 * html_link_img_dlg_pages_add().
 */
static void
html_link_img_dlg_items_add(GtkTable *table, html_page_t *hpp)
{
	html_wgt_info_t *hwip;
	GtkWidget *tmp = NULL;
	int row;

	for (row = 0, hwip = hpp->hwip; hwip->itemname; row++, hwip++) {
		/* add the entry field (or pulldown list) */
		switch (hwip->wgttype) {
		case PdownWgt:
			g_assert(hwip->itemlist);
			tmp = html_tag_pulldown_create(hwip->itemlist);
			break;
		case EntryWgt:
			tmp = gtk_entry_new();
			break;
		case VertSpace:
			continue;
		default:
			g_warning("html_link_img_dlg_items_add: fatal, unknown "
				  "wgttype (%d, itemname='%s')\n",
				  hwip->wgttype, hwip->itemname);
			continue;
		} /* switch (hwip->wgttype) */

		g_assert(tmp != NULL);
		gtk_table_attach_defaults(table, tmp, 1, 2, row, row+1);

		/* need to store (keep track of) the widget */
		hwip->wgt = tmp;

		/* add label with text */
		tmp = gtk_label_new(hwip->itemname);
		gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
		gtk_table_attach_defaults(table, tmp, 0, 1, row, row+1);

		GNPDBG_HTML(("html_link_img_dlg: added item '%s'\n",
			     hwip->itemname));
	} /* for each item on current page */
} /* html_link_img_dlg_items_add */


/*
 * PRIVATE: html_tag_pulldown_create
 *
 * given itemlist, creates a combo box (a pulldown list).  called only from
 * html_link_img_dlg_items_add().
 */
static GtkWidget *
html_tag_pulldown_create(char **itemlist)
{
	GtkWidget *combo;
	GList *list;
	char **item;

	list = NULL;
	for (item = itemlist; *item; item++)
		list = g_list_append(list, (gpointer)(*item));

	combo = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry),
			   (char *)g_list_nth_data(list, 0));
	return combo;
} /* html_tag_pulldown_create */


/*
 * PRIVATE: html_link_img_exec
 *
 * for the Link and Image tag dialogs, this is the callback for the "Ok"
 * button.
 */
static void
html_link_img_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf;
	int len;

	/* count up the total length needed */
	len = html_link_img_calc_len(hedp);
	g_assert(len > 0);

	/* build the buffer */
	buf = html_link_img_buf_build(len, hedp);

	/* and away it goes! */
	html_tag_insert(hedp->d, buf, hedp->htp);
	g_free(buf);
	gtk_widget_destroy(hedp->edp.toplev);
} /* html_link_img_exec */


/*
 * PRIVATE: html_link_img_buf_build
 *
 * builds and returns a text buffer which will be inserted into the current
 * document.  called only from html_link_img_exec().
 */
static char *
html_link_img_buf_build(int len, html_entry_dlg_t *hedp)
{
	char *buf;

	buf = (char *)g_malloc(sizeof(char) * len);

	/* stuff in prefix */
	if (hedp->htp->cr_before) {
		sprintf(buf, "\n<%s ", hedp->htp->prefix);
	} else
		sprintf(buf, "<%s ", hedp->htp->prefix);

	/* add each tag attribute */
	html_link_img_attr_add(hedp->dlg_pages, buf);

	/* append the suffix, if any */
	buf[strlen(buf) - 1] = '\0';	/* remove trailing space */
	strcat(buf, ">");
	if (hedp->htp->suffix)
		sprintf(buf + strlen(buf), "</%s>", hedp->htp->prefix);
	if (hedp->htp->cr_after)
		strcat(buf, "\n");

	GNPDBG_HTML(("html_link_img_buf_build: inserting buf (len = %d," 
		     "strlen = %ld) = '%s'\n", len, (long)strlen(buf), buf));
	return buf;
} /* html_link_img_buf_build */


/*
 * PRIVATE: html_link_img_attr_add
 *
 * appends each optional attribute for the link/img tag into the text buffer
 * that is to be inserted into the current document.  called only from
 * html_link_img_buf_build().
 */
static void
html_link_img_attr_add(html_page_t *dlg_pages, char *buf)
{
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	char *tmp;

	for (hpp = dlg_pages; hpp->pgname; hpp++) {
		for (hwip = hpp->hwip; hwip->itemname; hwip++) {
			if (hwip->wgttype == VertSpace)
				continue;

			if (hwip->wgttype == EntryWgt)
				tmp = gtk_entry_get_text(GTK_ENTRY(hwip->wgt));
			else
				tmp = gtk_entry_get_text(GTK_ENTRY(
						GTK_COMBO(hwip->wgt)->entry));

			if (tmp && tmp[0] != '\0') {
				if (hwip->wgttype == PdownWgt && 
					(g_strcasecmp(tmp, "auto") == 0 ||
					 g_strcasecmp(tmp, "none") == 0))
					break;

				strcat(buf, hwip->tagtext);
				if (hwip->is_assign)
					strcat(buf, "=");
				if (hwip->quoted)
					strcat(buf, "\"");


				if (hwip->is_assign)
					strcat(buf, tmp);
				if (hwip->quoted)
					strcat(buf, "\"");
				strcat(buf, " ");
			} /* widget exists */
		} /* for each item on page */
	} /* for each page */
} /* html_link_img_attr_add */


/*
 * PRIVATE: html_link_img_calc_len
 *
 * calculates an upper bound on the length of the text buffer to be inserted
 * for the link/img tag.  this is done by scanning through the tables, and if
 * an entry widget is non-empty, adding its length and the length of the
 * supporting characters.
 */
static int
html_link_img_calc_len(html_entry_dlg_t *hedp)
{
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	char *tmp;
	int len = 0;

	for (hpp = hedp->dlg_pages; hpp->pgname; hpp++) {
		for (hwip = hpp->hwip; hwip->itemname; hwip++) {
			if (hwip->wgttype == VertSpace)
				continue;

			if (hwip->wgttype == EntryWgt)
				tmp = gtk_entry_get_text(GTK_ENTRY(hwip->wgt));
			else
				tmp = gtk_entry_get_text(
					GTK_ENTRY(GTK_COMBO(hwip->wgt)->entry));

			if (tmp && tmp[0] != '\0') {
				/* if "auto" or "none", don't add this attr */
				if (hwip->wgttype == PdownWgt && 
					(g_strcasecmp(tmp, "auto") == 0 ||
					 g_strcasecmp(tmp, "none") == 0))
					break;

				len += strlen(tmp);

				g_assert(hwip->tagtext);
				g_assert(hwip->wgt);

				len += strlen(hwip->tagtext) +
					((hwip->is_assign) ? 1 : 0) +
					((hwip->quoted) ? 2 : 0) +
					1;	/* +1 for space character */
			} /* widget exists */
		} /* for each item on page */
	} /* for each page */

	/* add on prefix and suffix lengths */
	len += strlen(hedp->htp->prefix) + 4 +	/* "<%s " and "</%s>" */
		(hedp->htp->suffix ? (strlen(hedp->htp->prefix) + 3) : 1) +
		(hedp->htp->cr_before ? 1 : 0) +
		(hedp->htp->cr_after ? 1 : 0) +
		3;	/* 1 for \0, +2 extra just in case i can't count */

	return len;
} /* html_link_img_calc_len */


/*
 * PRIVATE: html_tag_insert
 *
 * insert buf into the text widget and position cursor wrt to inserted tag
 */
static void
html_tag_insert(doc_t *d, char *buf, htmltags_t *htp)
{
	int pt;

	gtk_text_freeze(GTK_TEXT(d->data));

	/* sync up insertion point with current cursor position */

#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_text_set_point(GTK_TEXT(d->data),
			   gtk_editable_get_position(GTK_EDITABLE(d->data)));
#else
	gtk_text_set_point(GTK_TEXT(d->data),
			   GTK_EDITABLE(d->data)->current_pos);
#endif

	/* insert text */
        gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL, buf, strlen(buf));

	/* update cursor position */
	pt = gtk_text_get_point(GTK_TEXT(d->data));
	switch (htp->cursorpos) {
#ifdef GTK_HAVE_FEATURES_1_1_0
	case Before:
		gtk_editable_set_position(GTK_EDITABLE(d->data),
					  pt - strlen(buf) - 1);
		break;
	case Between:
		gtk_editable_set_position(GTK_EDITABLE(d->data),
			pt - strlen(htp->prefix) - (htp->cr_after ? 4 : 3));
		break;
	case After:
		gtk_editable_set_position(GTK_EDITABLE(d->data), pt);
		break;
#else
	case Before:
		GTK_EDITABLE(d->data)->current_pos = pt - strlen(buf) - 1;
		break;
	case Between:
		GTK_EDITABLE(d->data)->current_pos =
			pt - strlen(htp->prefix) - (htp->cr_after ? 4 : 3);
		break;
	case After:
		GTK_EDITABLE(d->data)->current_pos = pt;
		break;
#endif
	default:
		break;
	}
	gtk_text_thaw(GTK_TEXT(d->data));
	gtk_signal_emit_by_name(GTK_OBJECT(d->data), "changed", d);
} /* html_tag_insert */


/*
 * PRIVATE: html_entry_dlg
 *
 * creates a simply entry dialog box to get input for a tag.  called only from
 * html_tag_complex_cb().
 */
static void
html_entry_dlg(doc_t *d, htmltags_t *htp, char *title, char *msg,
	       GtkSignalFunc execute_cb)
{
	html_entry_dlg_t *hedp;

	hedp = (html_entry_dlg_t *)g_malloc(sizeof(html_entry_dlg_t));
	hedp->d = d;
	hedp->htp = htp;
	get_dialog_entry(title, NULL, msg, 128, NULL,
			 GTK_SIGNAL_FUNC(execute_cb),       (gpointer)hedp,
			 GTK_SIGNAL_FUNC(html_dlg_cancel),  (gpointer)hedp,
			 GTK_SIGNAL_FUNC(html_dlg_destroy), (gpointer)hedp,
			 &(hedp->edp.toplev), &(hedp->edp.entry));
} /* html_entry_dlg */


/*
 * PRIVATE: html_dlg_destroy
 *
 * common callback to destroy a dialog
 */
static void
html_dlg_destroy(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;

	if (hedp->edp.toplev && hedp->edp.toplev == link_dlg)
		link_dlg = NULL;
	else if (hedp->edp.toplev && hedp->edp.toplev == img_dlg)
		img_dlg = NULL;

	g_free(hedp);
	hedp = NULL;
} /* html_dlg_destroy */


/*
 * PRIVATE: html_dlg_cancel
 *
 * common callback for the "Cancel" button.  simply calls gtk_widget_destroy()
 * on the top level widget.
 */
static void
html_dlg_cancel(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_dlg_cancel */


/*
 * PRIVATE: html_target_exec
 *
 * inserts the text buffer for the target HTML tag.
 */
static void
html_target_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf, *entrytext;

	entrytext = gtk_entry_get_text(GTK_ENTRY(hedp->edp.entry));
	if (strlen(entrytext) > 0) {
		buf = (char *)g_malloc(strlen(entrytext) + 20);
		g_snprintf(buf, strlen(entrytext) + 20, "<A NAME=\"%s\"></A>",
				entrytext);
		html_tag_insert(hedp->d, buf, hedp->htp);
		g_free(buf);
	}

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_target_exec */


/*
 * PRIVATE: html_comment_exec
 *
 * inserts the text buffer for the HTML comment.
 */
static void
html_comment_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf, *comment;

	comment = gtk_entry_get_text(GTK_ENTRY(hedp->edp.entry));
	if (strlen(comment) > 0) {
		buf = (char *)g_malloc(strlen(comment) + 12);
		g_snprintf(buf, strlen(comment) + 12, "<!-- %s -->", comment);
		html_tag_insert(hedp->d, buf, hedp->htp);
		g_free(buf);
	}

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_comment_exec */


/* the end */
