/* Directory routines
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <sys/stat.h>

/* unistd.h defines _POSIX_VERSION on POSIX.1 systems. */
#if defined(HAVE_DIRENT_H) || defined(_POSIX_VERSION)
#include <dirent.h>
#define NLENGTH(dirent) (strlen ((dirent)->d_name))
#else
#define dirent direct
#define NLENGTH(dirent) ((dirent)->d_namlen)
#ifdef HAVE_SYS_NDIR_H
#include <sys/ndir.h>
#endif /* HAVE_SYS_NDIR_H */
#ifdef HAVE_SYS_DIR_H
#include <sys/dir.h>
#endif /* HAVE_SYS_DIR_H */
#ifdef HAVE_NDIR_H
#include <ndir.h>
#endif /* HAVE_NDIR_H */
#endif /* not (HAVE_DIRENT_H or _POSIX_VERSION) */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "mad.h"
#include "global.h"
#define DIR_H_INCLUDE_HANDLE_DIRENT
#include "dir.h"
#include "util.h"
typedef struct Panel Panel;
#include "tree.h"

static char rcsid [] = "$Id: dir.c,v 1.13 1995/01/27 02:34:38 miguel Exp $";

/* If true show files starting with a dot */
int show_dot_files = 1;

/* If true show files ending in ~ */
int show_backups = 0;

/* If false then directories are shown separately from files */
int mix_all_files = 0;

/* Reverse flag */
static int reverse = 1;

#define MY_ISDIR(x) ( (S_ISDIR (x->buf.st_mode) || x->f.link_to_dir) ? 1 : 0)

sort_orders_t sort_orders [SORT_TYPES] = {
    { "Unsorted",  unsorted },
    { "Name",      sort_name },
    { "Extension", sort_ext },
    { "Time",      sort_time },
    { "Size",      sort_size },
    { "Inode",     sort_inode }
};

int unsorted (const file_entry *a, const file_entry *b)
{
    return 0;
}

int sort_name (const file_entry *a, const file_entry *b)
{
    int ad = MY_ISDIR (a);
    int bd = MY_ISDIR (b);

    if (ad == bd || mix_all_files)
	return strcmp (a->fname, b->fname) * reverse;
    else
	return bd-ad;
}

int sort_ext (const file_entry *a, const file_entry *b)
{
    char *exta, *extb;
    int r;
    int ad = MY_ISDIR (a);
    int bd = MY_ISDIR (b);

    if (ad == bd || mix_all_files){
	exta = extension (a->fname);
	extb = extension (b->fname);
	r = strcmp (exta, extb);
	if (r)
	    return r * reverse;
	else
	    return sort_name (a, b);
    } else
	return bd-ad;
}

int sort_time (const file_entry *a, const file_entry *b)
{
    int ad = MY_ISDIR (a);
    int bd = MY_ISDIR (b);

    if (ad == bd || mix_all_files)
	return (a->buf.st_mtime - b->buf.st_mtime) * reverse;
    else
	return bd-ad;
}

int sort_inode (const file_entry *a, const file_entry *b)
{
    int ad = MY_ISDIR (a);
    int bd = MY_ISDIR (b);

    if (ad == bd || mix_all_files)
	return (a->buf.st_ino - b->buf.st_ino) * reverse;
    else
	return bd-ad;
}

int sort_size (const file_entry *a, const file_entry *b)
{
    int ad = MY_ISDIR (a);
    int bd = MY_ISDIR (b);

    if (ad == bd || mix_all_files)
	return (b->buf.st_size - a->buf.st_size) * reverse;
    else
	return bd-ad;
}

void do_sort (dir_list *list, sortfn *sort, int top, int reverse_f)
{
    reverse = reverse_f ? -1 : 1;
    qsort (&(list->list) [1], top, sizeof (file_entry), sort);
}

void clean_dir (dir_list *list, int count)
{
    int i;

    for (i = 0; i < count; i++){
	free (list->list [i].fname);
	list->list [i].fname = 0;
    }
}

/* Used to set up a directory list when there is no access to a directory */
int set_zero_dir (dir_list *list)
{
    (list->list) [0].fname = strdup ("/");
    (list->list) [0].f.marked = 0;
    lstat ("/", &((list->list) [0].buf));
	
    return 1;
}

/* Return values: -1 = failure, 0 = don't add, 1 = add to the list */
int handle_dirent (dir_list *list, char *filter, struct dirent *dp,
		   struct stat *buf1, int next_free, int *link_to_dir)
{
    if (dp->d_name [0] == '.' && dp->d_name [1] == 0)
	return 0;
    if (!show_dot_files){
	if (dp->d_name [0] == '.'){
	    if (!(dp->d_name [1] == 0))
		if (!(dp->d_name [1] == '.' && NLENGTH (dp) == 2))
		    return 0;
	}
    }
    if (!show_backups && dp->d_name [NLENGTH (dp)-1] == '~')
	return 0;
    lstat (dp->d_name, buf1);

    if (S_ISDIR (buf1->st_mode))
	do_tree_check (dp->d_name);
    
    /* A link to a file or a directory? */
    *link_to_dir = 0;
    if (buf1->st_mode & S_IFLNK){
	struct stat buf2;
	if (!stat (dp->d_name, &buf2))
	    *link_to_dir = (buf2.st_mode & S_IFDIR) != 0;
    }
    if (!((buf1->st_mode & S_IFDIR) || *link_to_dir) && filter &&
	!regexp_match (filter, dp->d_name, match_file))
	return 0;

    /* Need to grow the *list? */
    if (next_free == list->size){
	list->list = realloc (list->list, sizeof (file_entry) *
			      (list->size + RESIZE_STEPS));
	if (!list->list)
	    return -1;
	list->size += RESIZE_STEPS;
    }
    return 1;
}

int do_load_dir(dir_list *list, sortfn *sort, int reverse, char *filter)
{
    DIR           *dirp;
    struct dirent *dp;
    int           status, link_to_dir;
    int           next_free = 0;
    struct stat   buf;

    start_tree_check ();
    dirp = opendir (".");
    if (!dirp){
	return set_zero_dir (list);
    }
    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	status = handle_dirent (list, filter, dp, &buf, next_free, &link_to_dir);
	if (status == 0)
	    continue;
	if (status == -1)
	    return next_free;
	list->list [next_free].fnamelen = NLENGTH (dp);
	list->list [next_free].fname = strdup (dp->d_name);
	list->list [next_free].f.marked = 0;
	list->list [next_free].f.link_to_dir = link_to_dir;
	list->list [next_free].buf = buf;
	next_free++;
	if (!(next_free % 32))
	    rotate_dash ();
    }
    if (next_free)
	do_sort (list, sort, next_free-1, reverse);
    closedir (dirp);
    end_tree_check ();
    return next_free;
}

int link_isdir (file_entry *file)
{
    struct stat b;
    
    if (S_ISLNK (file->buf.st_mode)){
	stat (file->fname, &b);
	if (S_ISDIR (b.st_mode))
	    return 1;
    }
    return 0;
}
 
int if_link_is_exe (file_entry *file)
{
    struct stat b;

    if (S_ISLNK (file->buf.st_mode)){
	stat (file->fname, &b);
	return is_exe (b.st_mode);
    }
    return 1;
}

static dir_list dir_copy = { 0, 0 };

static void alloc_dir_copy (int size)
{
    int i;
	    
    if (dir_copy.size < size){
	if (dir_copy.list){

	    for (i = 0; i < dir_copy.size; i++)
		if (dir_copy.list [i].fname)
		    free (dir_copy.list [i].fname);
	    free (dir_copy.list);
	    dir_copy.list = 0;
	}

	dir_copy.list = xmalloc (sizeof (file_entry) * size, "alloc_dir_copy");
	for (i = 0; i < size; i++)
	    dir_copy.list [i].fname = 0;
	dir_copy.size = size;
    }
}

/* If filter is null, then it is a match */
int do_reload_dir (dir_list *list, sortfn *sort, int count, int rev,
		   char *filter)
{
    DIR           *dirp;
    struct dirent *dp;
    int           next_free = 0;
    int           i, found, status, link_to_dir;
    struct stat   buf;
    int		  tmp_len;  /* For optimisation */

    start_tree_check ();
    dirp = opendir (".");
    if (!dirp)
	return set_zero_dir (list);

    alloc_dir_copy (list->size);
    for (i = 0; i < count; i++){
	dir_copy.list [i].fnamelen = list->list [i].fnamelen;
	dir_copy.list [i].fname =    list->list [i].fname;
	dir_copy.list [i].f.marked = list->list [i].f.marked;
	dir_copy.list [i].f.link_to_dir = list->list [i].f.link_to_dir;
    }

    for (dp = readdir (dirp); dp; dp = readdir (dirp)){
	status = handle_dirent (list, filter, dp, &buf, next_free, &link_to_dir);
	if (status == 0)
	    continue;
	if (status == -1)
	    return next_free;
	
	tmp_len = NLENGTH (dp);
	for (found = i = 0; i < count; i++)
	    if (tmp_len == dir_copy.list [i].fnamelen
		&& !strcmp (dp->d_name, dir_copy.list [i].fname)){
		list->list [next_free].f.marked = dir_copy.list [i].f.marked;
		found = 1;
		break;
	    }
	
	if (!found)
	    list->list [next_free].f.marked = 0;
	
	list->list [next_free].fnamelen = tmp_len;
	list->list [next_free].fname = strdup (dp->d_name);
	list->list [next_free].f.link_to_dir = link_to_dir;
	list->list [next_free].buf = buf;
	next_free++;
	if (!(next_free % 16))
	    rotate_dash ();
    }
    closedir (dirp);
    end_tree_check ();
    do_sort (list, sort, next_free-1, rev);
    clean_dir (&dir_copy, count);
    return next_free;
}

char *sort_type_to_name (sortfn *sort_fn)
{
    int i;

    for (i = 0; i < SORT_TYPES; i++)
	if ((sortfn *) (sort_orders [i].sort_fn) == sort_fn)
	    return sort_orders [i].sort_name;

    return "Unknown";
}

sortfn *sort_name_to_type (char *sort_name)
{
    int i;

    for (i = 0; i < SORT_TYPES; i++)
	if (strcasecmp (sort_orders [i].sort_name, sort_name) == 0)
	    return (sortfn *) sort_orders [i].sort_fn;

    /* default case */
    return (sortfn *) sort_name;
}

