/* Keyboard support routines.
   Copyright (C) 1994 Miguel de Icaza.
   Copyright (C) 1994, 1995 Janne Kukonlehto
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>		/* FD_ZERO et al */
#include <sys/time.h>		/* struct timeval */
#if defined(AIX) || defined(_AIX) || defined(__aix__)
    #include <sys/select.h>
#endif
#include <ncurses.h>
#include <ctype.h>
#include <errno.h>
#include <malloc.h> 
#include "util.h"		/* For xmalloc prototype */
#include "mad.h"		/* The memory debugger */
#include "global.h"
#include "mouse.h"
#include "input.h"
#include "key.h"
#include "main.h"

static char rcsid [] = "$Id: key.c,v 1.17 1995/01/27 02:35:20 miguel Exp $";

/* This macros were stolen from gpm 0.15 */
#define GET_TIME(tv) (gettimeofday(&tv, (struct timezone *)NULL))
#define DIF_TIME(t1,t2) ((t2.tv_sec -t1.tv_sec) *1000+ \
			 (t2.tv_usec-t1.tv_usec)/1000)

int mou_auto_repeat = 100;
int double_click_speed = 250;
int old_esc_mode = 0;

static int input_fd;
static fd_set select_set;
static int disabled_channels = 0; /* Disable channels checking */

/* File descriptor monitoring add/remove routines */
typedef struct SelectList {
    int fd;
    select_fn callback;
    struct SelectList *next;
} SelectList;

SelectList *select_list = 0;

void add_select_channel (int fd, select_fn callback)
{
    SelectList *new;

    new = xmalloc (sizeof (SelectList), "add_select_channel");
    new->fd = fd;
    new->callback = callback;
    new->next = select_list;
    select_list = new;
}

void delete_select_channel (int fd)
{
    SelectList *p = select_list;
    SelectList *prev = 0;
    
    while (p){
	if (p->fd == fd){
	    if (prev)
		prev->next = p->next;
	    else
		select_list = p->next;
	    free (p);
	}
	prev = p;
    }
}

void channels_down (void)
{
    disabled_channels ++;
}

void channels_up (void)
{
    if (!disabled_channels)
	fprintf (stderr,
		 "Error: channels_up called with disabled_channels = 0\n");
    disabled_channels--;
}

inline static int add_selects (fd_set *select_set)
{
    SelectList *p;
    int        top_fd = 0;

    if (disabled_channels)
	return 0;
    
    for (p = select_list; p; p = p->next){
	FD_SET (p->fd, select_set);
	if (p->fd > top_fd)
	    top_fd = p->fd;
    }
    return top_fd;
}

static void check_selects (fd_set *select_set)
{
    SelectList *p;

    if (disabled_channels)
	return;
    
    for (p = select_list; p; p = p->next)
	if (FD_ISSET (p->fd, select_set))
	    (*p->callback)(p->fd);
}

void init_key (void)
{
    input_fd = fileno (stdin);
}

void xmouse_get_event (Gpm_Event *ev)
{
    int btn;
    static struct timeval tv1 = { 0, 0 }; /* Force first click as single */
    static struct timeval tv2;
    static int clicks;

    /* Decode Xterm mouse information to a GPM style event */

    /* Variable btn has following meaning: */
    /* 0 = btn1 dn, 1 = btn2 dn, 2 = btn3 dn, 3 = btn up */
    btn = xgetch () - 32;
    
    /* There seems to be no way of knowing which button was released */
    /* So we assume all the buttons were released */

    if (btn == 3){
        ev->type = GPM_UP | (GPM_SINGLE << clicks);
        ev->buttons = 0;
	GET_TIME (tv1);
	clicks = 0;
    } else {
        ev->type = GPM_DOWN;
	GET_TIME (tv2);
	if (tv1.tv_sec && (DIF_TIME (tv1,tv2) < double_click_speed)){
	    clicks++;
	    clicks %= 3;
	} else
	    clicks = 0;
	
        switch (btn) {
	case 0:
            ev->buttons |= GPM_B_LEFT;
            break;
	case 1:
            ev->buttons |= GPM_B_MIDDLE;
            break;
	case 2:
            ev->buttons |= GPM_B_RIGHT;
            break;
	default:
            /* Nothing */
            break;
        }
    }
    /* Coordinates are 33-based */
    /* Transform them to 1-based */
    ev->x = xgetch () - 32;
    ev->y = xgetch () - 32;
}

/* If set timeout is set, then we wait 0.1 seconds, else, we block */
static void try_channels (int set_timeout)
{
    struct timeval timeout;
    static fd_set select_set;
    struct timeval *timeptr;
    int v;

    while (1){
	FD_ZERO (&select_set);
	FD_SET  (1, &select_set);	/* Add stdin */
	add_selects (&select_set);
	
	if (set_timeout){
	    timeout.tv_sec = 0;
	    timeout.tv_usec = 100000;
	    timeptr = &timeout;
	} else
	    timeptr = 0;
	
	v = select (FD_SETSIZE, &select_set, NULL, NULL, timeptr);
	if (v > 0){
	    check_selects (&select_set);
	    if (FD_ISSET (1, &select_set))
		return;
	}
    }
}

/* Workaround for System V Curses vt100 bug */
static int getch_with_delay (void)
{
    int c;

    /* This routine could be used on systems without mouse support,
       so we need to do the select check :-( */
    while (1){
	try_channels (0);

	/* Try to get a character */
	c = xgetch ();
	if (c != ERR)
	    break;
	/* Failed -> wait 0.1 secs and try again */
	try_channels (1);
    }
    /* Success -> return the character */
    return c;
}

#ifndef HAVE_LIBGPM
#define gpm_flag 0
#endif

/* Returns a character read from stdin with appropriate interpretation */
/* Also takes care of generated mouse events */
/* The routine will block only if block is true */
int mouse_getch (int block)
{
    int c, d;
    static int flag;			/* Return value from select */
    static int xmouse_event_flag = 0;	/* Flag: xterm mouse event waiting? */
    static Gpm_Event ev;		/* Mouse event */
    struct timeval timeout;
    struct timeval *time_addr = NULL;
    int redo_event = 0;
#ifdef DEBUGMOUSE
    extern FILE *log;
#endif

#ifdef HAVE_DOUPDATE
    doupdate ();
#endif
    
    doupdate ();
    /* Repeat if using mouse */
    while (xmouse_flag || gpm_flag)
    {
	if ((xmouse_flag && !xmouse_event_flag) || gpm_flag)
	{
	    FD_ZERO (&select_set);
	    FD_SET  (input_fd, &select_set);
	    add_selects (&select_set);

#ifdef HAVE_LIBGPM
	    if (gpm_flag) {
		FD_SET  (gpm_fd, &select_set);
	    }
#endif

	    timeout.tv_sec = 0;
	    timeout.tv_usec = mou_auto_repeat * 1000;

	    if (!block){
		time_addr = &timeout;
		timeout.tv_sec = 0;
		timeout.tv_usec = 0;
	    }
	    flag = select (FD_SETSIZE, &select_set, NULL, NULL, time_addr);
	    if (flag == 0){
		if (redo_event & MOU_REPEAT){
		    redo_event = redo_mouse (&ev);
		    continue;
		}
		if (!block)
		    return -1;
	    }
	    if (flag == -1 && errno == EINTR)
	        continue;
	    check_selects (&select_set);
	    
	    if (FD_ISSET (input_fd, &select_set))
	        break;
	}
	if (
#ifdef HAVE_LIBGPM
	    (gpm_flag && FD_ISSET (gpm_fd, &select_set)) ||
#endif
	    xmouse_event_flag){
	    xmouse_event_flag = 0;
#ifdef HAVE_LIBGPM
	    if (gpm_flag){
		Gpm_GetEvent (&ev);
		Gpm_FitEvent (&ev);
	    }
#endif
	    DEBUGM ((log, "+%s\n", redo_event & MOU_REPEAT?"redo":"nor"));
	    if (redo_event & MOU_REPEAT){
	        redo_event = redo_mouse (&ev);
		continue;
	    } else {
	        redo_event = mouse_handler (&ev);
	    }
	    DEBUGM ((log, "%s\n", redo_event & MOU_REPEAT?"redo":"nor"));

	    if (redo_event & MOU_ENDLOOP)
	        return -1;
	    time_addr = redo_event ? &timeout : NULL;
	    if (quit){
 		return -1;
	    }
#if 0
	    if (!(redo_event & MOU_REPEAT))
		return -1;
#endif
	    continue;
	}
    }
    untouchwin (stdscr);
    c = getch_with_delay ();
 process_char:
    /* This is needed on some OS that do not support ncurses and */
    /* do some magic after read()ing the data */
    if (c == '\r')
	c = '\n';

#if defined(AIX) || defined(_AIX) || defined(__aix__)
    if (c == KEY_SCANCEL)
	return '\t';
#endif

    if (c == KEY_F(0))
	return KEY_F(10);
    
    if (c != ESC_CHAR)
	return c;

    /* If curses didn't find a match for ESC-whatever, then the rest */
    /* of the characters are available for reading.   */

    if (old_esc_mode){
        #ifdef BUGGY_CURSES
	    wtimeout(stdscr, 500);
        #else
	    nodelay (stdscr, TRUE);
        #endif
    }
    
    d = xgetch ();

    if (old_esc_mode){
        #ifdef BUGGY_CURSES
	    notimeout (stdscr, TRUE);
        #else
	    nodelay (stdscr, FALSE);
        #endif
    }
    
    if (d == ERR)
	return c;

    if (d == ESC_CHAR)
	return ESC_CHAR;
    
    switch (d){
    case '0':
	return KEY_F(10); /* special case */
    case '<':
	return KEY_HOME;
    case '>':
	return KEY_END;
    case '[':
	d = xgetch ();
	if (d == 'M' && xmouse_flag){
	    /* Get a xterm mouse event */
	    xmouse_get_event (&ev);
	    xmouse_event_flag = 1;
	    /* Handle mouse event */
	    return mi_getch ();
	} else if (d == '1'){
	    d = xgetch ();
	    if (xgetch () != '~')
	        return -1;
	    switch (d){
	    case '1':
		return KEY_F(1);
	    case '2':
		return KEY_F(2);
	    case '3':
		return KEY_F(3);
	    case '4':
		return KEY_F(4);
	    case '5':
		return KEY_F(5);
	    case '7':
		return KEY_F(6);
	    case '8':
		return KEY_F(7);
	    case '9':
		return KEY_F(8);
	    default:
		return -1;
	    }
	} else if (d == '2'){
	    d = xgetch ();
	    if (xgetch () != '~')
	        return -1;
	    switch (d){
	    case '0':
		return KEY_F(9);
	    case '1':
		return KEY_F(10);
	    default:
		return -1;
	    }
	} else {
	    ungetch (d);
	    return '[';
	}
	break;
    default:
        if (isdigit(d))
	    return KEY_F(d-'0');
	if ((d >= 'a' && d <= 'z') || (d >= 'A' && d <= 'Z' )
	    || d == '\n' || d == '\t' || d == XCTRL ('h') || d == KEY_BACKSPACE ||
	    d == 127)
	    return ALT(d);
	else {
	    return c;
	}
    }
    return c;
}

/* Classification routines */
int is_abort_char (int c)
{
    return (c == XCTRL('c') || c == XCTRL('g') || c == ESC_CHAR ||
	    c == KEY_F(10));
}

int is_quit_char (int c)
{
    return (c == XCTRL('g') || (c == ESC_CHAR) || (c == KEY_F(10)));
}

