/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 */

/*
 * Make sure you include "input.h" first--mbase.h is well-behaved with respect
 * to common definitions (min, max, TRUE, FALSE, etc), but curses.h usually
 * isn't.  If you still get compiler complaints, see the dox, chapter 6, for
 * compile-time switches that may help.
 *
 */

#include "input.h"
#include <mbase.h>


/*
 * DEFINITIONS ----------------------------------------------------------------
 *
 */

#ifdef MSDOS
#define Standout() standout()
#define Standend() standout(); standend()
#else
#define Standout() standout()
#define Standend() standend()
#endif

#define PERPAGE 18


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

   void   main          XARGS( (int, char **) );
   bool   parseArgs     XARGS( (int, char **) );
   void   paintScreen   XARGS( (bool) );
   void   fillPage      XARGS( (void) );
   void   cleanup       XARGS( (void) );
   void   doError       XARGS( (char *) );
   void   doLine        XARGS( (int, int) );
   bool   verify        XARGS( (char *) );
   bool   readOnly      XARGS( (void) );
   bool   fnService     XARGS( (mb_service) );
   int    lenDisplay    XARGS( (int) );

   void   getRelName    XARGS( (void) );
   void   getIndex      XARGS( (void) );
   bool   getRelation   XARGS( (char *) );
   bool   getEditor     XARGS( (void) );

   bool   recZero       XARGS( (void) );
   void   recSelect     XARGS( (int) );
   void   recDelete     XARGS( (void) );
   void   recAddUpdate  XARGS( (bool) );


/*
 * VARIABLES ------------------------------------------------------------------
 *
 */

extern bool _fUpdateNow;  /* hidden deep within mb_input.c */

relation *rel = RNULL;
char     *key = NULL;
bool      fInCurses  = FALSE;
bool      fIsError   = FALSE;
bool      fReadOnly  = FALSE;
bool      fNeedFill  = FALSE;  /* Need to redisplay page? */
char      err[80] = "";

int       pg, pgmax, pglst, idx;
dataptr   rec;

int   gNum = 13;
int   gOpt[20] = { 0, 6, 11, 16, 21, 26, 30, 34, 41, 45, 49, 53, 57 };
char *gLine = "First Last Next Prev Gteq Equ Idx Screen Add Upd Del Rel Quit";

char *strOneOne = "%s.rel -- %ld record%s, %d index -- Page %d of %d";
char *strOneTwo = "%s.rel -- %ld record%s, %d indices -- Page %d of %d";
char *strTwoOne = "%s.rel -- %ld records%s, %d index -- Page %d of %d";
char *strTwoTwo = "%s.rel -- %ld records%s, %d indices -- Page %d of %d";

char *strNone = "";
char *strSome = " (%ld in queue)";

char *strAction;


/*
 * PRIMARY ROUTINES -----------------------------------------------------------
 *
 */

void
main  (argc, argv)
int    argc;
char **argv;
{
   bool   fRequest;
   int    pos;
   int    ch;


   setbuf (stdout, NULL);
   setbuf (stderr, NULL);

   SetService (fnService);

/*
 * First we've gotta parse the arguments...
 *
 */

   if (! parseArgs (argc, argv))
      {
      fprintf (stderr, "format: vr [[-k enckey] relation]\n");
      exit (1);
      }


/*
 * Then we initialize curses, and paint the startup screen
 *
 */

   fNeedFill = TRUE;
   paintScreen (TRUE);


/*
 * And finally we wait for the user to try to do something
 *
 */

   for (pos = 0; ; )
      {

/*
 * First comes the primary wait-for-a-key routine...
 *
 */

      if (fNeedFill)
         {
         fillPage ();
         if (err[0])
            {
            doError (err);
            err[0] = 0;
            }
         fNeedFill = FALSE;
         }

      movech (ch, 0, gOpt[pos]);

      if (fIsError)
         {
         doError ("");
         }

/*
 * Then figure out what the user wants to do.
 *
 */

      switch (ch)               /* CHECK FOR FIELD NAVIGATION -------------- */
         {
         case ' ':       pos++;   break;
         case AR_RIGHT:  pos++;   break;
         case '\b':      pos--;   break;
         case 127:       pos--;   break;
         case AR_LEFT:   pos--;   break;

         case '?':       paintScreen (FALSE);
                        break;
         case CTRL_L:    clearok (win, TRUE);
                         refresh ();
                        break;
         }

      fRequest = TRUE;

      switch (ch=tolower(ch))   /* CHECK FOR ACTION REQUEST ---------------- */
         {
         case '\n':                     break;
         case '\r':                     break;
         case  'f':  pos =  0;          break;
         case  'l':  pos =  1;          break;
         case  'n':  pos =  2;          break;
         case  'p':  pos =  3;          break;
         case  'g':  pos =  4;          break;
         case  'e':  pos =  5;          break;
         case  'i':  pos =  6;          break;
         case  's':  pos =  7;          break;
         case  'a':  pos =  8;          break;
         case  'u':  pos =  9;          break;
         case  'd':  pos = 10;          break;
         case  'r':  pos = 11;          break;
         case  'q':  pos = -1;          break;
         default:    fRequest = FALSE;  break;
         }
      if (pos == -1)    pos = gNum-1;
      if (pos == gNum)  pos = 0;

      if (fRequest)
         {
         if ((pos >= 0 && pos <= 10) && rel == RNULL)    /* ANY OPERATION -- */
            {
            doError ("specify new relation first");
            fRequest = FALSE;
            }
         if ((pos == 9 || pos == 10) && rel->pos == 0L)  /* UPDATE/DELETE -- */
            {
            doError ("select a record first");
            fRequest = FALSE;
            }
         }

      if (! fRequest)
         {
         continue;
         }

      switch (pos)
         {
         case  0:  recSelect (FIRST);     break;  /* SELECT FIRST RECORD --- */
         case  1:  recSelect (LAST);      break;  /* SELECT LAST RECORD ---- */
         case  2:  recSelect (NEXT);      break;  /* SELECT NEXT RECORD ---- */
         case  3:  recSelect (PREVIOUS);  break;  /* SELECT PREV RECORD ---- */
         case  4:  recSelect (GTEQ);      break;  /* FIND A RECORD, GTEQ --- */
         case  5:  recSelect (EQUAL);     break;  /* FIND AN EXACT RECORD -- */

         case  6:  getIndex();            break;  /* CHANGE INDICES -------- */

         case  7:  if ( (++pg) > pgmax )          /* CHANGE DISPLAY PAGE --- */
                      pg = 1;
                   fNeedFill = TRUE;
                   paintScreen (FALSE);
                  break;

         case  8:  recAddUpdate (TRUE);   break;  /* ADD A NEW RECORD ------ */
         case  9:  recAddUpdate (FALSE);  break;  /* UPDATE CURRENT RECORD - */
         case 10:  recDelete ();          break;  /* DELETE CURRENT RECORD - */

         case 11:  getRelName();          break;  /* CHANGE RELATIONS ------ */
         }

      if (pos == 12)                              /* EXIT ------------------ */
         break;
      }

   cleanup();
   exit_curses();

   mb_exit (0);
}


/*
 * DISPLAY ROUTINES -----------------------------------------------------------
 *
 */

bool
parseArgs (argc, argv)
int        argc;
char           **argv;
{
   char  *pch;

   for (--argc,++argv; argc; --argc,++argv)
      {
      if (*(pch = *argv) != '-')
         break;

      switch (*(++pch))
         {
         case 'R':  fReadOnly = TRUE;
                   break;
         case 'k':  if (*(key = (++pch)) == 0)
                       {
                       if (! --argc)
                          return FALSE;
                       key = *(++argv);
                       }
                   break;
         default:   return FALSE;
                   break;
         }
      }

   if ((key != NULL && !argc) || argc > 1)
      return FALSE;

   if (argc)
      {
      if (! getRelation (*argv))
         {
         fprintf (stderr, "%s\n", mb_error);
         exit (1);
         }
      }

   return TRUE;
}

void
paintScreen (fInit)
bool         fInit;
{
   char temp[80], temp2[30], *pch;
   long n;

   if (fInit)
      {
      if (! fInCurses)
         {
         init_curses ();
         fInCurses = TRUE;
         }

      mvaddstr (0,  0, gLine);
      mvaddstr (0, 66, "MetalBase 5.1");
      mvaddstr (1,  0, "-------------------------------------------------------------------------------");
      }

   if (rel == RNULL)
      {
      strcpy (temp, "no relation given");
      }
   else
      {
      if ((n = mb_num_q (rel)) == 0L)
         strcpy (temp2,  strNone);
      else
         sprintf (temp2, strSome, n);

      if ((n = mb_num (rel)) == 1L)
         {
         if (rel->nIdx == 1)  pch = strOneOne;
         else                 pch = strOneTwo;
         }
      else
         {
         if (rel->nIdx == 1)  pch = strTwoOne;
         else                 pch = strTwoTwo;
         }

      sprintf (temp, pch, rel->relname, n, temp2, rel->nIdx, pg, pgmax);
      }

   move (2, 0);
   clrtoeol();
   mvaddstr (2, (80-strlen(temp))/2, temp);

   refresh();
}

void
cleanup ()
{
   if (rec)
      {
      recFree (rel, rec);
      free (rec);
      rec = NULL;
      }
   if (rel)
      {
      mb_rmv (rel);
      rel = RNULL;
      }
   idx = pg = pgmax = pglst = 0;
}

/*
 * INTERACTIVE ROUTINES -------------------------------------------------------
 *
 */

void
getIndex ()
{
   int   i;
   char  temp[80];


   fNeedFill = TRUE;  /* We're going to erase the page here */

   move (4, 0);
   clrtobot();

   for (i = 0; i < min (rel->nIdx, 16); i++)
      {
      sprintf (temp, "index #%2d - %s", i+1, rel->idxName[i]);
      mvaddstr (6+i, 10, temp);
      }
   doError ("select index");

   mvaddstr (4, 10, "index name or number : ");
   refresh();

   strcpy (temp, rel->idxName[idx]);

   if ((input (temp, T_CHAR, 20) < 0) || ! temp[0])
      {
      strcpy (err, "index selection aborted");
      return;
      }

   if ((i = atoi (temp)-1) == -1)
      {
      i = idxnum (rel, temp);
      }
   if (i < 0 || i >= min (rel->nIdx, 16))
      {
      strcpy (err, "invalid index -- selection aborted");
      return;
      }

   idx = i;
   sprintf (err, "new index: #%d (%s)", idx+1, rel->idxName[idx]);
}

void
getRelName ()
{
   char  name[80];


   fNeedFill  = TRUE;

   move (4, 0);
   clrtobot();

   addstr ("relation name      : ");
   refresh();

   name[0] = 0;
   input (name, T_CHAR, 60);

   if (name[0])
      {
      getRelation (name);
      }
}

bool
getRelation (name)
char        *name;
{
   char  temp[80];

   if (mb_tst (name) != MB_OKAY)
      return FALSE;

   cleanup ();

#ifdef NO_ENCRYPT
   key = NULL;
#else
   if (! fInCurses)
      {
      if (key == NULL)
         {
         printf ("encryption password: ");
         gets (temp);
         key = temp;
         }
      }
   else
      {
      move (5, 0);  addstr ("encryption password: ");  refresh();
      temp[0] = 0;
      input (temp, T_CHAR, 60);
      move (4, 0);  clrtobot();  refresh();
      key = temp;
      }
#endif

   if ((rel = mb_inc (name, key)) == RNULL)
      return FALSE;

   if ((rec = (dataptr)malloc (1+ rel->cbRecord)) == (dataptr)0)
      {
      SetError (MB_NO_MEMORY);
      return FALSE;
      }
   if (recInit (rel, rec) != MB_OKAY)
      {
      return FALSE;
      }
   if (! recZero ())
      {
      mb_rmv (rel);
      rel = NULL;
      return FALSE;
      }

   pgmax = rel->nFld / PERPAGE + 1;
   pglst = rel->nFld % PERPAGE;
   pg    = 1;

   return TRUE;
}

bool
getEditor ()
{
   fIsError = TRUE;  /* We're dirtying up the bottom line here */
   move (23, 0);
   clrtoeol();
   addstr ("editor to use (ESC or CTRL-C to abort): ");
   if (input (editorname, T_CHAR, 20) == -1)
      return FALSE;
   doError ("loading editor...");
   return TRUE;
}

void
doError (line)
char    *line;
{
   if (! fInCurses)
      {
      fIsError = FALSE;
      if (line && *line)
         {
         printf ("%s\n", line);
         }
      return;
      }

   move (23, 0);
   clrtoeol();
   fIsError = FALSE;

   if (! line || ! *line)
      {
      refresh();
      return;
      }

   Standout();  addstr (line);
   Standend();
   refresh();
   fIsError = TRUE;
}

void
recSelect (act)
mb_action  act;
{
   char  rc;
   int   i, n, f, m;
   long  pos;
   char  temp[70];
   file  fh;


   if (act == GTHAN || act == EQUAL)
      {
      move (4, 0);
      clrtobot ();
      fNeedFill = TRUE;

      if (! recZero ())
         return;

      m = min (PERPAGE, rel->nIdxFld[idx]);

      for (i = 0; i < m; i++)
         {
         doLine (4+i, rel->idxFld[idx][i]);
         }
      doError (strAction = "enter comparison values");
      refresh();

      for (i = 0; ; )
         {
         move (4+i, 18);

         n = lenDisplay( f = rel->idxFld[idx][i] );

         if ( (rel->fldType[f] == T_MCHAR) || (rel->fldType[f] == T_MBYTE) )
            {
            temp[0] = 0;
            rc = input (temp, (rel->fldType[f]==T_MCHAR)? T_CHAR:T_BYTE, n);
            if ((fh = fieldOpen (rel, rec, f)) > 0)
               {
               writx (fh, temp, strlen(temp));
               close (fh);
               }
            }
         else
            {
            rc = input ((char *)rec + rel->cbStart[f], rel->fldType[f], n);
            }

         if (rc < 0 || rc == 1 || (rc == 0 && i == m-1))
            break;

         switch (rc)
            {
            case '-':     i--;  break;
            case 'k':     i--;  break;
            case AR_UP:   i--;  break;
            case  0:      i++;  break;
            case '+':     i++;  break;
            case 'j':     i++;  break;
            case AR_DOWN: i++;  break;
            }
         if (i == -1)  i = m-1;
         if (i ==  m)  i = 0;
         }

      if (rc < 0)
         {
         strcpy (err, "search aborted");
         (void)mb_sel (rel, idx, rec, CURRENT, NULL);
         return;
         }
      }

   fNeedFill = TRUE;

   pos = rel->pos;

   if (mb_sel (rel, idx, rec, act, NULL) != MB_OKAY)
      {
      rel->pos = pos;
      (void)mb_sel (rel, idx, rec, CURRENT, NULL);
      }
}

void
fillPage ()
{
   int   i, n;

   move (4, 0);
   clrtobot();

   if (! rel)
      {
      return;
      }
   if (! rel->pos)
      {
      if (! recZero())
         return;
      }

   n = (pg-1) * PERPAGE;

   for (i = 0; i < (pg == pgmax ? pglst : PERPAGE); i++,n++)
      {
      doLine (4+i, n);
      }

   paintScreen (FALSE);
}


/*
 * SERVICE ROUTINES -----------------------------------------------------------
 *
 */

void
doLine (y, n)
int     y, n;
{
   char    temp[80], *pch;
   int     len;


   move (y, 0);  clrtoeol();

   sprintf (temp, "%-13.13s -", rel->fldName[n]);
   mvaddstr (y,  0, temp);

   len = lenDisplay (n);

   switch (rel->fldType[n])
      {
      case T_MCHAR:
      case T_CHAR:    mvaddch (y, 17, '\"');
                      mvaddch (y, 18 +len, '\"');
                     break;
      case T_MBYTE:
      case T_BYTE:    mvaddstr (y, 16, "0x");
                      mvaddch  (y, 18 +3*len -1, '.');
                     break;
      case T_SHORT:   mvaddstr (y, 17, "[      ]");                 break;
      case T_USHORT:  mvaddstr (y, 17, "[      ]");                 break;
      case T_LONG:    mvaddstr (y, 17, "[           ]");            break;
      case T_ULONG:   mvaddstr (y, 17, "[           ]");            break;
      case T_FLOAT:   mvaddstr (y, 17, "[           ]");            break;
      case T_DOUBLE:  mvaddstr (y, 17, "[              ]");         break;
      case T_MONEY:   mvaddstr (y, 17, "$              ");          break;
      case T_TIME:    mvaddstr (y, 17, "(  :  :  )");               break;
      case T_DATE:    mvaddstr (y, 17, "(  /  /    )");             break;
      case T_SERIAL:  mvaddstr (y, 17, "(           )");            break;
      case T_PHONE:   mvaddstr (y, 17, "[                     ]");  break;
      }

   if (rel->pos == 0L)
      return;

   pch = (char *)rec +rel->cbStart[n];

   _fUpdateNow = FALSE;   /* Tell display() not to refresh() yet */

   move (y, 18);
   display (pch, rel->fldType[n], len);
}

void
recAddUpdate (fAdd)
bool          fAdd;
{
   int     fstart, fend, f;
   int     pgstart;
   int     len;
   char    ch;
   long    posOld;


   if (! readOnly ())
      return;


   fNeedFill = TRUE;

   if (fAdd)
      {
      posOld = rel->pos;
      rel->pos = 1L;       /* So fillPage() will clear current values */

      if (! recZero())
         return;
      }

   pgstart = pg;
   pg = 1;

   for (f = 0; ; )
      {
      fillPage ();

      if (fAdd)
         doError (strAction = "enter data to add");
      else
         doError (strAction = "enter new data for this record");

      fstart = (pg-1) * PERPAGE;
      fend = fstart + (pg == pgmax ? pglst : PERPAGE);

      for (;;)
         {
         move (4 + f-fstart, 18);

         len = lenDisplay (f);

         if (rel->fldType[f] == T_SERIAL)
            {
            if (rel->nFld == 1)  ch = 1;    /* Nothing but serial#?  Done. */
            else                 ch = '+';  /* Don't let 'em enter serial# */
            }
         else
            {
            ch = input ((char *)rec +rel->cbStart[f], rel->fldType[f], len);
            }

         if (ch == 0)
            {
            if (f == fend-1 && pg == pgmax)  ch = 1;
            else                             ch = '+';
            }

         if (ch < 0 || ch == 1)
            break;

         switch (ch)
            {
            case '-': case 'k': case AR_UP:   f--;  ch='-';  break;
            case '+': case 'j': case AR_DOWN: f++;  ch='+';  break;
            }

         if (ch == '-' && rel->fldType[f] == T_SERIAL)
            f--;

         if (f >= fend)
            {
            ch = 0;
            pg++;  f = fend;
            break;
            }
         if (f <  fstart)
            {
            ch = 0;
            pg--;  f = fstart-1;
            break;
            }
         }

      if (pg > pgmax)
         {
         pg = 1;
         f = 0;
         }
      if (pg == 0)
         {
         pg = pgmax;
         f  = rel->nFld-1;
         }

      if (ch != 0)
         break;
      }

   if (fAdd)
      {
      rel->pos = posOld;  /* Restore this. */
      }

   pg = pgstart;

   if (ch < 0)
      {
      if (fAdd)  strcpy (err, "addition aborted");
      else       strcpy (err, "change aborted");
      (void)mb_sel (rel, idx, rec, CURRENT, NULL);
      return;
      }

   if (fAdd)  mb_add (rel, rec);
   else       mb_upd (rel, rec);

   if (mb_errno == MB_OKAY)
      {
      if (fAdd)  strcpy (err, "record successfully added");
      else       strcpy (err, "record successfully updated");
      }

   (void)mb_sel (rel, idx, rec, CURRENT, NULL);
}

void
recDelete ()
{
   if (! readOnly ())
      return;

   if (! verify ("delete this record ? "))
      {
      strcpy (err, "delete aborted");
      return;
      }

   fNeedFill = TRUE;

   if (mb_del (rel) == MB_OKAY)
      {
      strcpy (err, "record deleted");
      }
}

bool
verify (str)
char   *str;
{
   char  ch;

   doError (str);

   for (;;)
      {
      ch = getarr();
      switch (tolower (ch))
         {
         case 'q':
         case 'n':
         case ' ':
         case ESC:
         case '\r':
         case '\n':  doError ("");  return 0;  break;
         case 'y':   doError ("");  return 1;  break;
         }
      }
}

bool
recZero ()
{
   int      i, len;
   char    *pch;

   if ((! rel) || (! rec))
      {
      return FALSE;
      }

   if (recClean (rel, rec) != MB_OKAY)
      {
      return FALSE;
      }

   for (i = 0; i < rel->nFld; i++)
      {
      pch = (char *)rec +rel->cbStart[i];
      len = rel->cbLen[i];

      switch (rel->fldType[i])
         {
         case T_CHAR:    *pch = 0;                        break;
         case T_SHORT:   *(short  *)pch = (short)0;       break;
         case T_USHORT:  *(ushort *)pch = (ushort)0;      break;
         case T_LONG:    *(long   *)pch = (long)0;        break;
         case T_ULONG:   *(ulong  *)pch = (ulong)0;       break;
         case T_FLOAT:   *(float  *)pch = (float)0.0;     break;
         case T_DOUBLE:  *(double *)pch = (double)0.0;    break;
         case T_MONEY:   *(double *)pch = (double)0.0;    break;
         case T_TIME:    *(mb_time*)pch = curtime();      break;
         case T_DATE:    *(mb_date*)pch = curdate();      break;
         case T_SERIAL:  *(long   *)pch = (long)0;        break;
         case T_PHONE:   ((mb_phone *)pch)->number = 0L;  break;
         case T_BYTE:    numcpy (pch, NULL, len);         break;
         }
      }

   return TRUE;
}

bool
readOnly ()
{
   if (! rel->fReadOnly && ! fReadOnly)
      {
      return TRUE;
      }

   strcpy (err, "relation opened in read-only mode");

   return FALSE;
}

bool
fnService (svc)
mb_service svc;
{
   switch (svc)
      {
      case svcERROR:      if (err[0] == 0)  strcpy (err, mb_error);  break;

      case svcBUSY:       doError ("wait...");                       break;
      case svcPAUSE:      doError ("request queued - wait...");      break;
      case svcTIMEOUT:    doError ("timeout in queue - wait...");    break;

      case svcIN_START:   doError ("press ctrl-e to edit field");    break;
      case svcIN_END:     doError (strAction);                       break;
      case svcIN_EDITOR:  return getEditor();                        break;
      }

   return TRUE;
}

int
lenDisplay (fld)
int         fld;
{
   int  len;

   switch (rel->fldType[fld])
      {
      case T_MCHAR:  len = 60;                         break;
      case T_MBYTE:  len = 60;                         break;
      case T_CHAR:   len = min (rel->cbLen[fld], 60);  break;
      case T_BYTE:   len = min (rel->cbLen[fld], 20);  break;
      case T_SHORT:  len = 6;                          break;
      case T_USHORT: len = 6;                          break;
      case T_LONG:   len = 11;                         break;
      case T_ULONG:  len = 11;                         break;
      case T_FLOAT:  len = 11;                         break;
      case T_SERIAL: len = 11;                         break;
      case T_DOUBLE: len = 14;                         break;
      case T_MONEY:  len = 14;                         break;
      case T_TIME:   len = 8;                          break;
      case T_DATE:   len = 10;                         break;
      case T_PHONE:  len = 21;                         break;
      }
   return len;
}

