------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 5                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.27 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Util; use Exp_Util;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Rtsfind;  use Rtsfind;
with Sinfo;    use Sinfo;
with Sem;      use Sem;
with Sem_Util; use Sem_Util;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Exp_Ch5 is

   -----------------------------------
   -- Expand_N_Assignment_Statement --
   -----------------------------------

   --  For tagged types, create a record type C with a size equivalent to the
   --  the one of the type to be copied. If the target is class wide (i.e.
   --  dynamically tagged) the construction of this type involves a dispatch
   --  call to the size attribute. Only the last field of this record will be
   --  copied in order to avoid copying the tag and the finalization chain
   --  pointers which are not technically part of the value).

   procedure Expand_N_Assignment_Statement (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Lhs : constant Node_Id    := Name (N);
      Rhs : constant Node_Id    := Expression (N);
      Typ : constant Entity_Id  := Etype (Lhs);

      Node : Node_Id;

   begin

      --  ??? The evaluate once doesn't work here for mysterious reason.
      --  Furthermore, New_Copy is used instead of sharing the nodes because
      --  sometimes the analysis of the expanded nodes rewrite-substitute their
      --  arguments which is pretty awful when the argument is shared
      --  Set_Evaluate_Once (Lhs, True);
      --  Set_Evaluate_Once (Rhs, True);

      --  Generate the code:

      --  (lines marked <CTRL> only concern controlled types)
      --
      --
      --    Finalize (lhs);                                        <CTRL>
      --
      --    <if Lhs is class-wide and Tag_Checks are on>
      --        if Lhs._Tag /= Rhs._Tag then
      --           raise constraint_Error;
      --        end if;
      --
      --    <tagged_copy> (Lhs, Rhs);
      --    Adjust (lhs);                                          <CTRL>

      if Is_Tagged_Type (Typ) then
         Replace_Substitute_Tree (N,
           Make_Tagged_Copy (Loc, Lhs, Rhs, Typ));

         --  generate tag equality check for class-wide targets

         if Is_Class_Wide_Type (Typ)
           and then (not Tag_Checks_Suppressed (Typ))
         then

            --  <if lhs is class-wide and Tag_Checks are on>
            --      if lhs._tag /= rhs._Tag then
            --         raise constraint_Error;
            --      end if;

            Node :=
              Make_If_Statement (Loc,
                Condition =>
                  Make_Op_Ne (Loc,
                    Left_Opnd =>
                      Make_Selected_Component (Loc,
                        Prefix => New_Copy (Lhs),
                        Selector_Name => Make_Identifier (Loc, Name_uTag)),
                    Right_Opnd =>
                      Make_Selected_Component (Loc,
                        Prefix => New_Copy (Rhs),
                        Selector_Name => Make_Identifier (Loc, Name_uTag))),
                Then_Statements => New_List (New_Constraint_Error (Loc)));

            Insert_Before (N, Node);
            Analyze (Node);
         end if;

         if Is_Controlled (Typ) then
            Node := Make_Finalize_Call (New_Copy (Lhs), Typ);
            Insert_Before (N, Node);
            Analyze (Node);
            Insert_After (N, Make_Adjust_Call (New_Copy (Lhs), Typ));
         end if;

         Analyze (N);
      end if;
   end Expand_N_Assignment_Statement;

   ------------------------------
   -- Expand_N_Block_Statement --
   ------------------------------

   --  Establish master if the block is a master
   --  Activate tasks if the block is an activator

   procedure Expand_N_Block_Statement (N : Node_Id) is
   begin
      Build_Task_Activation_Call (N);

      if Is_Task_Master (N) then
         Establish_Task_Master (N);
      end if;

   end Expand_N_Block_Statement;

   -----------------------------
   -- Expand_N_Case_Statement --
   -----------------------------

   --  If the last alternative is not an Others choice replace it with an
   --  N_Others_Choice. Note that we do not bother to call Analyze on the
   --  modified case statement, since it's only effect would be to compute
   --  the contents of the Others_Discrete_Choices node laboriously, and of
   --  course we already know the list of choices that corresponds to the
   --  others choice (it's the list we are replacing!)

   procedure Expand_N_Case_Statement (N : Node_Id) is
      Altnode     : constant Node_Id := Last (Alternatives (N));
      Others_Node : Node_Id;

   begin
      if Nkind (First (Discrete_Choices (Altnode))) /= N_Others_Choice then
         Others_Node := Make_Others_Choice (Sloc (Altnode));
         Set_Others_Discrete_Choices
           (Others_Node, Discrete_Choices (Altnode));
         Set_Discrete_Choices (Altnode, New_List (Others_Node));
      end if;
   end Expand_N_Case_Statement;

   ---------------------------
   -- Expand_N_If_Statement --
   ---------------------------

   --  Add traceback before the IF. In addition, if an ELSE/ELSIF parts
   --  are present, add traceback calls at the start of the THEN, ELSIF
   --  and ELSE statements, so we know which way control went.

   procedure Expand_N_If_Statement (N : Node_Id) is
      Elsf : Node_Id;

   begin
      Traceback_Store (N);

      Traceback_Store (First (Then_Statements (N)));

      if Present (Elsif_Parts (N)) then
         Elsf := First (Elsif_Parts (N));

         while Present (Elsf) loop
            Traceback_Store (First (Then_Statements (Elsf)));
            Elsf := Next (Elsf);
         end loop;
      end if;

      if Present (Else_Statements (N)) then
         Traceback_Store (First (Else_Statements (N)));
      end if;
   end Expand_N_If_Statement;

end Exp_Ch5;
