------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              C O M P E R R                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.15 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains routines called when a fatal internal compiler
--  error is detected. Calls to these routines cause termination of the
--  current compilation with appropriate error output.

with Atree;            use Atree;
with Debug;            use Debug;
with Errout;           use Errout;
with Excep;            use Excep;
with Osint;            use Osint;
with Output;           use Output;
with Sinput;           use Sinput;
with Sprint;           use Sprint;
with System.Traceback; use System.Traceback;
with Treepr;           use Treepr;

package body Comperr is

   --------------------
   -- Compiler_Abort --
   --------------------

   procedure Compiler_Abort (X : String; Code : Integer) is
   begin
      --  If debug flag -dk is set, then we do an immediate abort. This is
      --  useful if we want to enter the debugger immediately at the abort
      --  without doing any processing at all before the abort occurs.

      if Debug_Flag_K then
         Exit_Program (E_Abort);

      --  If errors have already occured, then we guess that the abort may
      --  well be caused by previous errors, and we don't make too much fuss
      --  about it, since we want to let the programmer fix the errors first.

      elsif Errors_Detected /= 0 then
         raise Unrecoverable_Error;

      --  Otherwise give message with details of the abort'

      else
         Set_Standard_Error;
         Write_Location (Sloc (Fatal_Error_Node));
         Write_Str (": internal error: ");
         Write_Str (X);

         if Code /= 0 then
            Write_Str (" (code = ");
            Write_Int (Int (Code));
            Write_Str (", ");
         else
            Write_Str (" (");
         end if;

         Write_Str ("Node_Id = ");
         Write_Int (Int (Fatal_Error_Node));
         Write_Char (')');

         Write_Eol;
         Write_Location (Sloc (Fatal_Error_Node));
         Write_Str (": please report this to gnat-report@cs.nyu.edu");
         Write_Eol;

         if Debug_Flag_3 then
            Write_Eol;
            Print_Tree_Node (Fatal_Error_Node);
            Write_Eol;
         end if;

         Set_Standard_Output;

         --  If tracebacks are stored (compiler was compiled with -db debug
         --  switch, then output the tracebacks, then output any requested
         --  tree and/or source dumps, and finally exit without aborting,
         --  since we assume the reason that tracebacks are being used is
         --  that the abort does not give useful information.

         if Tracebacks_Stored then
            Output_Traceback;
            Tree_Dump;
            Source_Dump;
            Exit_Program (E_Errors);

         --  If no tracebacks are stored, then just exit with an abort
         --  after outputting any requested tree and/or source dumps.

         else
            Tree_Dump;
            Source_Dump;
            Exit_Program (E_Abort);
         end if;
      end if;

   end Compiler_Abort;

end Comperr;
