/****************************************************************************/
/*                                                                          */
/*                         GNAT COMPILER COMPONENTS                         */
/*                                                                          */
/*                             A - T R A N S 4                              */
/*                                                                          */
/*                          C Implementation File                           */
/*                                                                          */
/*                            $Revision: 1.68 $                             */
/*                                                                          */
/*           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          */
/*                                                                          */
/* GNAT is free software;  you can  redistribute it  and/or modify it under */
/* terms of the  GNU General Public License as published  by the Free Soft- */
/* ware  Foundation;  either version 2,  or (at your option) any later ver- */
/* sion.  GNAT is distributed in the hope that it will be useful, but WITH- */
/* OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY */
/* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License */
/* for  more details.  You should have  received  a copy of the GNU General */
/* Public License  distributed with GNAT;  see file COPYING.  If not, write */
/* to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */
/*                                                                          */
/****************************************************************************/

#include "config.h"
#include "tree.h"
#include "flags.h"
#include "a-ada.h"
#include "a-types.h"
#include "a-atree.h"
#include "a-nlists.h"
#include "a-elists.h"
#include "a-sinfo.h"
#include "a-einfo.h"
#include "a-namet.h"
#include "a-snames.h"
#include "a-string.h"
#include "a-uintp.h"
#include "a-trans.h"
#include "a-trans3.h"
#include "a-trans4.h"
#include "a-misc.h"

/* Prepare expr to be an argument of a TRUTH_NOT_EXPR or other logical
   operation.

   This preparation consists of taking the ordinary
   representation of an expression expr and producing a valid tree
   boolean expression describing whether expr is nonzero.  We could
   simply always do build_binary_op (NE_EXPR, expr, integer_zero_node, 1),
   but we optimize comparisons, &&, ||, and !.

   The resulting type should always be the same as the input type.
   This function is simpler than the corresponding C version since
   the only possible operands will be things of Boolean type.  */

tree
truthvalue_conversion (expr)
     tree expr;
{
  register enum tree_code code;
  tree type = TREE_TYPE (expr);

  switch (TREE_CODE (expr))
    {
    case EQ_EXPR:  case NE_EXPR: case LE_EXPR: case GE_EXPR:
    case LT_EXPR:  case GT_EXPR:
    case TRUTH_ANDIF_EXPR:
    case TRUTH_ORIF_EXPR:
    case TRUTH_AND_EXPR:
    case TRUTH_OR_EXPR:
    case TRUTH_XOR_EXPR:
    case ERROR_MARK:
      return expr;

    case COND_EXPR:
      /* Distribute the conversion into the arms of a COND_EXPR.  */
      return fold (build (COND_EXPR, type, TREE_OPERAND (expr, 0),
			  truthvalue_conversion (TREE_OPERAND (expr, 1)),
			  truthvalue_conversion (TREE_OPERAND (expr, 2))));
    }

  return build_binary_op (NE_EXPR, type, expr,
			  convert (type, integer_zero_node));
}

/* Return the base type of TYPE.  */

static tree
get_base_type (type)
     tree type;
{
  while (TREE_TYPE (type) != 0
	 && (TREE_CODE (type) == INTEGER_TYPE
	     || TREE_CODE (type) == REAL_TYPE))
    type = TREE_TYPE (type);

  return type;
}


/* We have a comparison or assignment operation on two array types, T1 and T2.
   Return the type that both operands should be converted to, if any.
   Otherwise return zero.  */

static tree
find_common_array_type (t1, t2)
     tree t1, t2;
{
  /* If either type is non-BLKmode, use it.  Note that we know that we will
     not have any alignment problems since if we did the non-BLKmode
     type could not have been used.  */
  if (TYPE_MODE (t1) != BLKmode)
    return t1;
  else if (TYPE_MODE (t2) != BLKmode)
    return t2;

  /* Otherwise, return the type that has a constant size.  */
  if (TREE_CONSTANT (TYPE_SIZE (t1)))
    return t1;
  else if (TREE_CONSTANT (TYPE_SIZE (t2)))
    return t2;

  /* In this case, both types have variable size.  It's probably
     best to leave the "type mismatch" because changing it could
     case a bad self-referential reference.  */
  return 0;
}

/* Return an expression tree representing an equality comparison of
   A1 and A2, two objects of ARRAY_TYPE.  The returned expression should
   be of type RESULT_TYPE

   Two arrays are equal if the lengths in each dimension are equal
   and the data is equal.  We perform the length tests in as efficient
   a manner as possible.  */

static tree
compare_arrays (result_type, a1, a2)
     tree a1, a2;
     tree result_type;
{
  tree t1 = TREE_TYPE (a1);
  tree t2 = TREE_TYPE (a2);
  tree result = convert (result_type, integer_one_node);
  int length_zero_p = 0;

  /* Process each dimension separately and compare the lengths.  If any
     dimension has a size known to be zero, set SIZE_ZERO_P to 1 to
     suppress the comparison of the data.  */
  while (TREE_CODE (t1) == ARRAY_TYPE)
    {
      tree lb1 = TYPE_MIN_VALUE (TYPE_DOMAIN (t1));
      tree ub1 = TYPE_MAX_VALUE (TYPE_DOMAIN (t1));
      tree lb2 = TYPE_MIN_VALUE (TYPE_DOMAIN (t2));
      tree ub2 = TYPE_MAX_VALUE (TYPE_DOMAIN (t2));
      tree bt = get_base_type (TREE_TYPE (lb1));
      tree length1 = fold (build (MINUS_EXPR, bt, ub1, lb1));
      tree length2 = fold (build (MINUS_EXPR, bt, ub2, lb2));
      tree tem;
      tree comparison;

      /* If the length of the first array is a constant, swap our operands
	 unless the length of the second array is the constant zero.  
	 Note that we have set the `length' values to the length - 1.  */
      if (TREE_CODE (length1) == INTEGER_CST
	  && ! integer_zerop (fold (build (PLUS_EXPR, bt, length2,
					   convert (bt, integer_one_node)))))
	{
	  tem = a1, a1 = a2, a2 = tem;
	  tem = t1, t1 = t2, t2 = tem;
	  tem = lb1, lb1 = lb2, lb2 = tem;
	  tem = ub1, ub1 = ub2, ub2 = tem;
	  tem = length1, length1 = length2, length2 = tem;
	}

      /* If the length of this dimension in the second array is the constant
	 zero, we can just go inside the original bounds for the first
	 array and see if last < first.  */
      if (integer_zerop (fold (build (PLUS_EXPR, bt, length2,
				      convert (bt, integer_one_node)))))
	{
	  tree ub = TYPE_MAX_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t1)));
	  tree lb = TYPE_MIN_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t1)));

	  comparison = build_binary_op (LT_EXPR, result_type, ub, lb);

	  if (contains_placeholder_p (comparison))
	    comparison = build (WITH_RECORD_EXPR, result_type,
				comparison, a1);

	  length_zero_p = 1;
	}

      /* If the length is some other constant value, we know that the
	 this dimension in the first array cannot be superflat, so we
	 can just use its length from the actual stored bounds.  */
      else if (TREE_CODE (length2) == INTEGER_CST)
	{
	  ub1 = TYPE_MAX_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t1)));
	  lb1 = TYPE_MIN_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t1)));
	  ub2 = TYPE_MAX_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t2)));
	  lb2 = TYPE_MIN_VALUE (TYPE_INDEX_TYPE (TYPE_DOMAIN (t2)));
	  bt = get_base_type (TREE_TYPE (ub1));

	  comparison
	    = build_binary_op (EQ_EXPR, result_type, 
			       build_binary_op (MINUS_EXPR, bt, ub1, lb1),
			       build_binary_op (MINUS_EXPR, bt, ub2, lb2));

	  /* Note that we know that UB2 and LB2 are constant and hence
	     cannot contain a PLACEHOLDER_EXPR.  */

	  if (contains_placeholder_p (comparison))
	    comparison = build (WITH_RECORD_EXPR, result_type, comparison, a1);
	}

      /* Otherwise compare the computed lengths.  */
      else
	{
	  if (contains_placeholder_p (length1))
	    length1 = build (WITH_RECORD_EXPR, bt, length1, a1);
	  if (contains_placeholder_p (length2))
	    length2 = build (WITH_RECORD_EXPR, bt, length2, a2);

	  comparison
	    = build_binary_op (EQ_EXPR, result_type, length1, length2);
	}

      result = build_binary_op (TRUTH_ANDIF_EXPR, result_type,
				result, comparison);

      t1 = TREE_TYPE (t1);
      t2 = TREE_TYPE (t2);
    }

  /* Unless the size of some bound is known to be zero, compare the
     data in the array.  */
  if (! length_zero_p)
    {
      tree type = find_common_array_type (TREE_TYPE (a1), TREE_TYPE (a2));

      if (type != 0)
	a1 = convert (type, a1), a2 = convert (type, a2);


      result
	= build_binary_op (TRUTH_ANDIF_EXPR, result_type, result,
			   build (EQ_EXPR, result_type, a1, a2));
    }

  return result;
}

/* Make a binary operation of kind OP_CODE.  RESULT_TYPE is the type
   desired for the result.  Usually the operation is to be performed
   in that type.  For MODIFY_EXPR and ARRAY_REF, RESULT_TYPE may be 0
   in which case the type to be used will be derived from the operands.

   This function is very much unlike the ones for C and C++ since we
   have already done any type conversion and matching required.  All we
   have to do here is validate the work done by SEM and handle subtypes.  */

tree
build_binary_op (op_code, result_type, left_operand, right_operand)
     enum tree_code op_code;
     tree result_type;
     tree left_operand;
     tree right_operand;
{
  tree left_type  = TREE_TYPE (left_operand);
  tree right_type = TREE_TYPE (right_operand);
  tree left_base_type = get_base_type (left_type);
  tree right_base_type = get_base_type (right_type);
  tree operation_type = (result_type != 0 && TYPE_EXTRA_SUBTYPE_P (result_type)
			 ? get_base_type (result_type) : result_type);
  tree modulus = (operation_type != 0 && TYPE_MODULAR_P (operation_type)
		  ? TYPE_MODULUS (operation_type) : 0);
  tree result;
  int has_side_effects = 0;

  switch (op_code)
    {
    case MODIFY_EXPR:
      if (operation_type == 0)
	operation_type = left_type;

      /* If we are copying one array to another, find the best type to use.  */
      if (TREE_CODE (left_type) == ARRAY_TYPE
	  && TREE_CODE (right_type) == ARRAY_TYPE)
	{
	  tree best_type = find_common_array_type (left_type, right_type);

	  if (best_type && left_type != best_type)
	    left_operand = convert (best_type, left_operand);
	  if (best_type && right_type != best_type)
	    right_operand = convert (best_type, right_operand);

	  if (best_type)
	    operation_type = best_type;
	}
      else if (TREE_TYPE (right_operand) != operation_type)
	right_operand = convert (operation_type, right_operand);

      has_side_effects = 1;
      modulus = 0;
      break;

    case ARRAY_REF:
      if (operation_type == 0)
	operation_type = TREE_TYPE (left_type);

      if (right_type != TYPE_DOMAIN (left_type))
	right_operand = convert (TYPE_DOMAIN (left_type), right_operand);

      modulus = 0;
      break;

    case GE_EXPR:
    case LE_EXPR:
    case GT_EXPR:
    case LT_EXPR:
      if (TREE_CODE (left_type) == POINTER_TYPE)
	gigi_abort (501);

      /* ... fall through ... */

    case EQ_EXPR:
    case NE_EXPR:
      /* If both objects are records, compare them specially.  */
      if (TREE_CODE (left_type) == ARRAY_TYPE
	  && TREE_CODE (right_type) == ARRAY_TYPE)
	{
	  result = compare_arrays (result_type, left_operand, right_operand);

	  if (op_code == EQ_EXPR)
	    ;
	  else if (op_code == NE_EXPR)
	    result = invert_truthvalue (result);
	  else
	    gigi_abort (502);

	  return result;
	}

      /* Otherwise, the base types must be the same unless the objects are
	 records.  If we have records, use the best type and convert both
	 operands to that type.  */
      if (left_base_type != right_base_type)
	{
	  if (TREE_CODE (left_base_type) == RECORD_TYPE
	      && TREE_CODE (right_base_type) == RECORD_TYPE)
	    {
	      /* The only way these are permitted to be the same is if both
		 types have the same name.  In that case, one of them must
		 not be self-referential.  Use that one as the best type.
		 Even better is if one is of fixed size.  */
	      tree best_type = 0;

	      if (TYPE_NAME (left_base_type) == 0
		  || TYPE_NAME (left_base_type) != TYPE_NAME (right_base_type))
		gigi_abort (503);

	      if (TREE_CONSTANT (TYPE_SIZE (left_base_type)))
		best_type = left_base_type;
	      else if (TREE_CONSTANT (TYPE_SIZE (right_base_type)))
		best_type = right_base_type;
	      else if (! contains_placeholder_p (TYPE_SIZE (left_base_type)))
		best_type = left_base_type;
	      else if (! contains_placeholder_p (TYPE_SIZE (right_base_type)))
		best_type = right_base_type;
	      else
		gigi_abort (504);

	      left_operand = convert (best_type, left_operand);
	      right_operand = convert (best_type, right_operand);
	    }
	  else
	    gigi_abort (505);
	}

      /* If we are comparing a fat pointer against zero, we need to 
	 just compare the template pointer.  */
      else if (TYPE_FAT_POINTER_P (left_base_type)
	       && TREE_CODE (right_operand) == CONSTRUCTOR
	       && integer_zerop (TREE_VALUE (TREE_OPERAND (right_operand, 1))))
	{
	  right_operand
	    = build_component_ref (left_operand,
				   NULL_TREE,
				   TREE_CHAIN (TYPE_FIELDS (left_base_type)));
	  left_operand = convert (TREE_TYPE (right_operand),
				  integer_zero_node);
	}
      else
	{
	  left_operand = convert (left_base_type, left_operand);
	  right_operand = convert (right_base_type, right_operand);
	}

      modulus = 0;
      break;

    case PREINCREMENT_EXPR:
    case PREDECREMENT_EXPR:
    case POSTINCREMENT_EXPR:
    case POSTDECREMENT_EXPR:
      /* In these, the result type and the left operand type should be the
	 same.  Do the operation in the base type of those and convert the
	 right operand (which is an integer) to that type.

	 Note that these operations are only used in loop control where
	 we guarantee that no overflow can occur.  So nothing special need
	 be done for modular types.  */

      if (left_type != result_type)
	gigi_abort (506);

      operation_type = get_base_type (result_type);
      left_operand = convert (operation_type, left_operand);
      right_operand = convert (operation_type, right_operand);
      has_side_effects = 1;
      modulus = 0;
      break;

    case LSHIFT_EXPR:
    case RSHIFT_EXPR:
      /* The RHS of a shift can be any type.  In addition, we don't support
	 them on modular types.  Otherwise, processing is the same as
	 normal.  */
      if (operation_type != left_base_type || modulus != 0)
	gigi_abort (514);

      left_operand = convert (operation_type, left_operand);
      break;

    case TRUTH_ANDIF_EXPR:
    case TRUTH_ORIF_EXPR:
    case TRUTH_AND_EXPR:
    case TRUTH_OR_EXPR:
    case TRUTH_XOR_EXPR:
      left_operand = truthvalue_conversion (left_operand);
      right_operand = truthvalue_conversion (right_operand);
      /* ... fall through ... */
    default:
      /* The result type should be the same as the base types of the
	 both operands (and they should be the same).  Convert
	 everything to the result type.  */

      if (operation_type != left_base_type
	  || left_base_type != right_base_type)
	gigi_abort (507);

      left_operand = convert (operation_type, left_operand);
      right_operand = convert (operation_type, right_operand);
    }

  /* For non-binary modular types, do the operation as signed so that we get
     mathematically correct answer.  */
  if (modulus != 0 && TREE_UNSIGNED (operation_type))
    {
      operation_type = signed_type (operation_type);
      left_operand = convert (operation_type, left_operand);
      right_operand = convert (operation_type, right_operand);
    }

  result = build (op_code, operation_type, left_operand, right_operand);
  TREE_SIDE_EFFECTS (result) = has_side_effects;

  result = fold (result);
  TREE_CONSTANT (result)
    = TREE_CONSTANT (left_operand) & TREE_CONSTANT (right_operand);

  /* If we are working with modular types, perform the MOD operation.  */
  if (modulus != 0)
    result = fold (build (FLOOR_MOD_EXPR, operation_type, result,
			  convert (operation_type, modulus)));

  if (result_type != 0 && result_type != operation_type)
    result = convert (result_type, result);

  return result;
}

/* Similar, but for unary operations.  */

tree
build_unary_op (op_code, result_type, operand)
     enum tree_code op_code;
     tree result_type;
     tree operand;
{
  tree type = TREE_TYPE (operand);
  tree base_type = get_base_type (type);
  tree operation_type = (result_type != 0 && TYPE_EXTRA_SUBTYPE_P (result_type)
			 ? get_base_type (result_type) : result_type);
  tree modulus = (operation_type && TYPE_MODULAR_P (operation_type)
		  ? TYPE_MODULUS (operation_type) : 0);
  tree result;
  int side_effects = 0;

  switch (op_code)
    {
    case TRUTH_NOT_EXPR:
      if (result_type != base_type)
	gigi_abort (508);

      result = invert_truthvalue (truthvalue_conversion (operand));
      break;

    case ADDR_EXPR:
      if (TREE_CODE (operand) == INDIRECT_REF
	  || TREE_CODE (operand) == UNCONSTRAINED_ARRAY_REF)
	result = TREE_OPERAND (operand, 0);
      else
	result = fold (build1 (op_code, build_pointer_type (type), operand));

      TREE_CONSTANT (result) = staticp (operand) || TREE_CONSTANT (operand);
      modulus = 0;
      break;

    case INDIRECT_REF:
      /* If we want to refer to an entire unconstrained array,
	 make up an expression to do so.  This will never survive to
	 the backend.  */
      if (TYPE_FAT_POINTER_P (type))
	result = build1 (UNCONSTRAINED_ARRAY_REF,
			 TYPE_UNCONSTRAINED_ARRAY (type), operand);
      else
	result = fold (build1 (op_code, TREE_TYPE (type), operand));

      modulus = 0;
      side_effects = flag_volatile 
	|| (! TYPE_FAT_POINTER_P (type) && TYPE_VOLATILE (TREE_TYPE (type)));
      break;

    default:
      if (operation_type != base_type)
	gigi_abort (509);

      /* For non-binary modular types, do the operation as signed so that we
	 get mathematically correct answer.  */
      if (modulus != 0 && TREE_UNSIGNED (operation_type))
	{
	  operation_type = signed_type (operation_type);
	  operand = convert (operation_type, operand);
	}

      result = fold (build1 (op_code, operation_type, convert (operation_type,
							       operand)));
    }

  if (side_effects)
    {
      TREE_SIDE_EFFECTS (result) = 1;
      if (TREE_CODE (result) == INDIRECT_REF)
	TREE_THIS_VOLATILE (result) = TYPE_VOLATILE (TREE_TYPE (result));
    }

  if (result_type != 0 && TREE_TYPE (result) != result_type)
    result = convert (result_type, result);

  /* If we are working with modular types, perform the MOD operation.  */
  if (modulus != 0)
    result = fold (build (FLOOR_MOD_EXPR, operation_type, result,
			  convert (operation_type,
				   TYPE_MODULUS (operation_type))));

  return result;
}

/* Similar, but for COND_EXPR.  */

tree
build_cond_expr (result_type, condition_operand, true_operand, false_operand)
     tree result_type;
     tree condition_operand;
     tree true_operand;
     tree false_operand;
{
  /* Front-end verifies that result, true and false operands have same base
     type. Convert everything to the result type.  */

  true_operand  = convert (result_type, true_operand);
  false_operand = convert (result_type, false_operand);

  return fold (build (COND_EXPR, result_type, condition_operand,
		      true_operand, false_operand));
}


/* Build a CALL_EXPR to call FUNDECL with one argument, ARG.  Return
   the CALL_EXPR.  */
tree
build_call_1_expr (fundecl, arg)
     tree fundecl;
     tree arg;
{
  tree call = build (CALL_EXPR, TREE_TYPE (TREE_TYPE (fundecl)),
		     build_unary_op (ADDR_EXPR, NULL_TREE, fundecl),
		     chainon (NULL_TREE, build_tree_list (NULL_TREE, arg)),
		     NULL_TREE);

  TREE_SIDE_EFFECTS (call) = 1;

  return call;
}

/* Likewise to call FUNDECL with no arguments.  */

tree
build_call_0_expr (fundecl)
     tree fundecl;
{
  tree call = build (CALL_EXPR, TREE_TYPE (TREE_TYPE (fundecl)),
		     build_unary_op (ADDR_EXPR, NULL_TREE, fundecl),
		     NULL_TREE, NULL_TREE);

  TREE_SIDE_EFFECTS (call) = 1;

  return call;
}


/* Return a CONSTRUCTOR of TYPE whose list is LIST.  */

tree
build_constructor (type, list)
     tree type;
     tree list;
{
  tree elmt;
  int allconstant = 1;
  int side_effects = 0;
  tree result;

  for (elmt = list; elmt; elmt = TREE_CHAIN (elmt))
    {
      if (! TREE_CONSTANT (TREE_VALUE (elmt)))
	allconstant = 0;

      if (TREE_SIDE_EFFECTS (TREE_VALUE (elmt)))
	side_effects = 1;
    }

  result = build (CONSTRUCTOR, type, NULL_TREE, list);
  TREE_CONSTANT (result) = allconstant;
  TREE_STATIC (result) = allconstant;
  TREE_SIDE_EFFECTS (result) = side_effects;

  return result;
}

/* Return a COMPONENT_REF to access a field that is given by COMPONENT,
   an IDENTIFIER_NODE giving the name of the field, FIELD, a FIELD_DECL,
   for the field, or both.

   We also handle the fact that we might have been passed a pointer to the
   actual record and know how to look for fields in variant parts.  */

tree
build_simple_component_ref (record_variable, component, field)
     tree record_variable;
     tree component;
     tree field;
{
  tree record_type = TREE_TYPE (record_variable);
  tree ref;

  /* Handle added pointer for pass-by-reference values.  */
  if (TREE_CODE (record_type) == POINTER_TYPE)
    {
      record_variable
	= build_unary_op (INDIRECT_REF, NULL_TREE, record_variable);
      record_type = TREE_TYPE (record_variable);
    }

  if ((TREE_CODE (record_type) != RECORD_TYPE
       && TREE_CODE (record_type) != UNION_TYPE
       && TREE_CODE (record_type) != QUAL_UNION_TYPE)
      || TYPE_SIZE (record_type) == 0)
    gigi_abort (510);

  if (field == 0 || DECL_CONTEXT (field) != record_type)
    /* Check if there is a field with name COMPONENT in the record.  */
    {
      if (component == 0)
	gigi_abort (511);

      /* ??? Explore later if we can use the TYPE_LANG_SPECIFIC optimization
	 that appears in C version of this function.  */

      for (field = TYPE_FIELDS (record_type); field;
	   field = TREE_CHAIN (field))
	{
	  if (DECL_NAME (field) == component)
	    break;
	  else if (DECL_NAME (field) == 0)
	     {
	      tree field_ref
		= build_simple_component_ref (record_variable, 
					      NULL_TREE, field);
	      ref = build_simple_component_ref (field_ref, 
						component, NULL_TREE);

	      if (ref != 0)
		return ref;
	    }
	}
    }

  if (!field)
    return 0;

  ref = build (COMPONENT_REF, TREE_TYPE (field), record_variable, field);

  if (TREE_READONLY (record_variable) || TREE_READONLY (field))
    TREE_READONLY (ref) = 1;
  if (TREE_THIS_VOLATILE (record_variable) || TREE_THIS_VOLATILE (field))
    TREE_THIS_VOLATILE (ref) = 1;

  return ref;
}

/* Like build_simple_component_ref, except that we look in the field __parent
   if the field is not defined at this level */

tree
build_component_ref (record_variable, component, field)
     tree record_variable;
     tree component;
     tree field;
{
  tree local_field;
  tree parent_ref;
  static tree parent_comp = 0;
  tree comp_ref;

  if (!parent_comp) 
    parent_comp = get_identifier (Get_Name_String (Name_uParent)); 

  /* See if the field is present at this level.  */
  comp_ref = build_simple_component_ref (record_variable, component, field);
  if (comp_ref)
    return comp_ref;

  /* If it is not present, look recursively in the parent.  */

  parent_ref = build_simple_component_ref (record_variable, 
					   parent_comp, NULL_TREE);
  if (parent_ref)
    return build_component_ref (parent_ref, component, field);
  else
    /* The field was not found in the hierarchy. Should not happen. */
    gigi_abort (512);
}

/* Build a GCC tree to correspond to allocating an object of TYPE whose
   initial value is INIT, if INIT is nonzero.  Convert the expression to
   RESULT_TYPE, which must be some type of pointer.  Return the tree. 
   GNAT_TYPE is the type of the underlying object in case we need to 
   call a record initialization procedure.  */

tree
build_allocator (type, init, gnat_type, result_type)
     tree type;
     tree init;
     Entity_Id gnat_type;
     tree result_type;
{
  /* Counts number of allocators we were able to do by statically allocating
     memory when at top level.  */
  static int alloc_var_index = 0;
  tree size = TYPE_SIZE (type);
  tree ptr_type;
  tree result;

  /* If TYPE is an unconstrained array, it must be the case that INIT
     is nonzero and an UNCONSTRAINED_ARRAY_REF.  Set SIZE to be the sum of
     the sizes of the object, its template, and the fat pointer.  Allocate
     the whole thing and fill in the parts.  */
  if (TREE_CODE (type) == UNCONSTRAINED_ARRAY_TYPE)
    {
      tree fat_ptr_type = TYPE_POINTER_TO (type);
      tree array_type = TREE_TYPE (TREE_TYPE (TYPE_FIELDS (fat_ptr_type)));
      tree template_type
	= TREE_TYPE (TREE_TYPE (TREE_CHAIN (TYPE_FIELDS (fat_ptr_type))));
      tree fat_ptr_size = TYPE_SIZE (fat_ptr_type);
      tree template_size = TYPE_SIZE (template_type);
      tree storage;
      tree old_template_ptr, old_array_ptr, new_template_ptr, new_array_ptr;

      if (init == 0 || TREE_CODE (init) != UNCONSTRAINED_ARRAY_REF)
	gigi_abort (513);

      size = size_binop (PLUS_EXPR, 
			 build (WITH_RECORD_EXPR, sizetype,
				TYPE_SIZE (array_type),
				TREE_OPERAND (init, 0)),
			 size_binop (PLUS_EXPR, fat_ptr_size, template_size));

      storage = build (CALL_EXPR, build_pointer_type (char_type_node),
		       build_unary_op (ADDR_EXPR, NULL_TREE, malloc_decl),
		       build_tree_list (NULL_TREE,
					size_binop (CEIL_DIV_EXPR,
						    size,
						    size_int (BITS_PER_UNIT))),
		       NULL_TREE);
      TREE_SIDE_EFFECTS (storage) = 1;
      storage = save_expr (storage);

      /* Skip the fat pointer to get the template, then skip both to get
	 the memory for the array.  */
      new_template_ptr
	= convert (build_pointer_type (template_type),
		   build (PLUS_EXPR, TREE_TYPE (storage),
			  storage,
			  convert (TREE_TYPE (storage),
				   size_in_bytes (fat_ptr_type))));

      new_array_ptr
	= convert
	  (build_pointer_type (array_type),
	   build (PLUS_EXPR, TREE_TYPE (storage),
		  storage,
		  convert (TREE_TYPE (storage),
			   size_binop (PLUS_EXPR,
				       size_in_bytes (fat_ptr_type),
				       size_in_bytes (template_type)))));

      old_template_ptr = build_component_ref (TREE_OPERAND (init, 0),
					      get_identifier ("p_template"),
					      NULL_TREE);

      old_array_ptr = build_component_ref (TREE_OPERAND (init, 0),
					   get_identifier ("p_array"),
					   NULL_TREE);

      result = build_unary_op (INDIRECT_REF, fat_ptr_type,
			       convert (build_pointer_type (fat_ptr_type),
					storage));

      result = build (COMPOUND_EXPR, fat_ptr_type,
		      build_binary_op (MODIFY_EXPR, template_type,
				       build_unary_op (INDIRECT_REF, NULL_TREE,
						       new_template_ptr),
				       build_unary_op (INDIRECT_REF, NULL_TREE,
						       old_template_ptr)),
		      result);

      result = build (COMPOUND_EXPR, fat_ptr_type,
		      build_binary_op (MODIFY_EXPR, array_type,
				       build_unary_op (INDIRECT_REF, NULL_TREE,
						       new_array_ptr),
				       build_unary_op (INDIRECT_REF, NULL_TREE,
						       old_array_ptr)),
		      result);

      return
	build_binary_op
	  (MODIFY_EXPR, fat_ptr_type,
	   result,
	   build_constructor
	   (fat_ptr_type,
	    tree_cons (TYPE_FIELDS (fat_ptr_type), new_array_ptr,
		       tree_cons (TREE_CHAIN (TYPE_FIELDS (fat_ptr_type)),
				  new_template_ptr, NULL_TREE))));
    }

  /* If TYPE is an discriminated record, see if there is an initializing
     expression.  If there is, make the object we are allocating that size.
     Otherwise, allocate an object that is the maximum size of the record.
     Note that the record must have had default discriminant or this would
     not be valid Ada.  */
  if (TREE_CODE (size) != INTEGER_CST && contains_placeholder_p (size))
    {
      if (init)
	size = build (WITH_RECORD_EXPR, sizetype, size, init);
      else
	size = max_size (size, 1);
    }

  /* If we are at top-level and SIZE is a constant, we can actually
     allocate an object of TYPE and point to it.  */
  if (global_bindings_p () && TREE_CODE (size) == INTEGER_CST)
    {
      char name[20];

      sprintf (name, "__V%d", alloc_var_index++);
      result = create_var_decl (name, NULL_PTR, type, init, 0, 0, 0, 1);
      result = build_unary_op (ADDR_EXPR, NULL_TREE, result);
      init = 0;
    }
  else
    {
      /* Call the allocator and return an object whose type is a pointer to
	 TYPE.  Note that we pass the size in bytes and convert to
	 sizetype.  */

      size = size_binop (CEIL_DIV_EXPR, convert (sizetype, size),
			 size_int (BITS_PER_UNIT));

      result
	= build (CALL_EXPR, build_pointer_type (type),
		 build_unary_op (ADDR_EXPR, NULL_TREE, malloc_decl),
		 chainon (NULL_TREE, build_tree_list (NULL_TREE, size)),
		 NULL_TREE);
      TREE_SIDE_EFFECTS (result) = 1;
    }

  /* If we have an initial value, put the new address into a SAVE_EXPR, assign
     the value, and return the address.  Do this with a COMPOUND_EXPR.  */

  if (init)
    {
      result = save_expr (result);
      result = build (COMPOUND_EXPR, TREE_TYPE (result),
		      build_binary_op (MODIFY_EXPR, type,
				       build_unary_op (INDIRECT_REF, type,
						       result),
				       init),
		      result);
    }

  /* If the result is not of the proper type, convert it.  But check for
     converting a pointer to a constrained type into a pointer to
     an unconstrained type.  In that case, we have to allocate the template
     from memory as well.  */
  if (TREE_TYPE (result) != result_type)
    {
      if (TYPE_FAT_POINTER_P (result_type)
	  & TREE_CODE (TREE_TYPE (TREE_TYPE (result))) == ARRAY_TYPE)
	result = convert_to_unconstrained (result_type, result, 1);
      else
	result = convert (result_type, result);
    }

  return result;
}

/* Indicate that we need to make the address of EXPR_NODE and it therefore
   should not be allocated in a register. Return 1 if successful.  */

int
mark_addressable (tree expr_node)
{
  while (1)
    switch (TREE_CODE (expr_node))
      {
      case ADDR_EXPR:
      case COMPONENT_REF:
      case ARRAY_REF:
      case REALPART_EXPR:
      case IMAGPART_EXPR:
	expr_node = TREE_OPERAND (expr_node, 0);
	break;

      case CONSTRUCTOR:
	TREE_ADDRESSABLE (expr_node) = 1;
	return 1;

      case VAR_DECL:
      case CONST_DECL:
      case PARM_DECL:
      case RESULT_DECL:
	put_var_into_stack (expr_node);
	TREE_ADDRESSABLE (expr_node) = 1;
	return 1;

      case FUNCTION_DECL:
	TREE_ADDRESSABLE (expr_node) = 1;
	return 1;

      default:
	return 1;
    }
}
