/*
 * straycats.c
 *
 * Copyright (C), 1994, Graeme W. Wilford. (Wilf.)
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * code to detect stray cats - somewhat tricky with the FSSTND. It Will find
 * stray cats for both FSSTND compliant man trees and old style/local user
 * type man trees.
 *
 * Tue May  3 21:24:51 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk)
 */

#define MAN_MAIN        /* to not define config_file */
#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "mydbm.h"
#include "straycats.h"
#include "manp.h"
#include "dbver.h"

void ver(void);
char *prognam;
int debug;
static short all;	/* check all man trees, even unwritable ones */

static short create_place;	/* create place-holders */
static short del_rel_stray;	/* delete TRUE straycats !! */
static short del_nonrel_stray;	/* delete non-rel straycats */
static short del_place;  	/* delete place-holders */

void usage(void)
{
	printf("usage: %s [[-da] [-N|-C|-S|-R|-X]] | [-h] | [-V]\n", prognam);
	
	printf(	
	  "\n\t\t-d: produce debugging info.\n"
  	  "\t\t-a: check all man trees regardless of write perms.\n"
	  "\t\t-h: show this usage message.\n"
	  "\t\t-V: show version/author message.\n"
  	  "\n\tUse the following options with care, creation/deletion with your\n"
  	  "\t(root included) priveledges will occur. Read the man page.\n\n"
  	  "\t\t-N: default CREATION/DELETION, invokes -S -C.\n"
	  "\t\t-C: CREATE place-holders for the straycats.\n"
  	  "\t\t-S: DELETE straycats that have a non-relative source file.\n"
	  "\t\t-R: DELETE any place-holders found\n"
	  "\t\t-X: DELETE straycats with no source file. WARNING: Don't use!\n\n"
	);
}

int main(int argc, char *argv[])
{
	char *c1, *path;
	int c;

	prognam = *argv;

	while ((c = getopt(argc, argv, "daNSCRXhV")) != -1){

		switch (c){

			case 'd':
				debug++;
				break;
			case 'a':
				all++;
				break;
			case 'N':
				del_nonrel_stray = 1;
				create_place = 1;
				break;
			case 'S':
				del_nonrel_stray = 1;
				break;	
			case 'C':
				create_place = 1;
				break;
			case 'R':
				del_place++;
				break;
			case 'X':
				del_rel_stray++;
				break;
			case 'V':
				ver();
				exit(0);
			case 'h':
				usage();
				exit(0);
			default:
				usage();
				exit(1);
		}
	}
	
	if (del_rel_stray && create_place) {
		usage();
		fputs(
		  "\nWill not delete straycats AND touch relative src files: pointless",
		  stderr);
		exit(1);
	}
	if ( (create_place && del_place)
	  || (del_rel_stray && del_nonrel_stray) ) {
	  	usage();
	  	fputs("\npointless combination of options", stderr);
	  	exit(1);
	}
	
	if (!create_place && !del_rel_stray && !del_place && !del_nonrel_stray)
		printf("WARNING: Only looking for straycats, use an option for creation/deletion.\n");

	path = manpath(0);

	/* 
	 * should check write perms in directories first and only deal 
	 * with ours.
	 */

	while( (c1 = strrchr(path, ':')) != NULL){
		*c1 = '\0';

		compare_dirs(++c1);
	}
	compare_dirs(path);

	return 0;
}

void compare_dirs(char *path)
{
	DIR *cdir;
	MYDBM_FILE dbf;
	struct dirent *catlist;
	char catdir[BUFSIZ], mandir[BUFSIZ], database[BUFSIZ];
	size_t catlen, manlen;
	char *t1;

	strcpy(mandir, path);

	/* 
	 * check for write perms
	 */

	if (!all && access(mandir, W_OK) == -1){
		printf("skipping %s: no write permission.\n",
		  path);
		return;
	}
	printf("Looking at %s tree\n", path);

	/* 
	 * if it's /usr, it'll be in the global database
	 * if not, it'll be in the users local database.
	 */
	 
	if (strncmp(mandir, MAN_ROOT, sizeof MAN_ROOT -1) != 0) {
		strcpy(database, mandir);
		strcat(database, LOCAL_DB);
	}
	else
		strcpy(database, DB);

	if ( (dbf = MYDBM_REOPEN(database)) == NULL) {
		fprintf(stderr, "%s: compare_dirs: ", prognam);
		perror(database);
		exit(1);
	}

	dbver_rd(dbf);
	
#ifdef FSSTND
	if (strncmp(mandir, MAN_ROOT, sizeof MAN_ROOT -1) != 0)
		strcpy(catdir, mandir);
	else if (strlen(mandir) < 9) /* sizeof "/usr/man" */
		strcpy(catdir, CAT_ROOT);
	else 
		strcpy(catdir, convert_name_sc(mandir));
#else
	strcpy(catdir, mandir);
#endif

/*
	if (debug)
		fprintf(stderr, "raw manpathdir: %s, catpathdir: %s\n", mandir, catdir);
 */
	if ( (cdir = opendir(catdir)) == NULL){
		fprintf(stderr,
		  "%s: compare_dirs: could not open dir for reading: ",
		  prognam);
		perror(catdir);
		exit(1);
	}
	strcat(catdir, "/");
	strcat(mandir, "/");
	catlen = strlen(catdir);
	manlen = strlen(mandir);
		
	while ((catlist = readdir(cdir)) != NULL){
		if (*catlist->d_name != 'c')
			continue;
		strcpy(catdir + catlen, catlist->d_name);
		strcpy(mandir + manlen, catlist->d_name);
		t1 = strrchr(mandir, '/');
		*(t1 + 1) = 'm';
		*(t1 + 3) = 'n';
		
/*		if (debug)
			fprintf(stderr, "catdir: %s\nmandir: %s\n",
			  catdir, mandir);
 */
		touch_stray_src(dbf, catdir, mandir);
	}
	MYDBM_CLOSE(dbf);
}

void touch_stray_src(MYDBM_FILE dbf, char *catdir, char *mandir)
{
	DIR *cdir;
	FILE *fd;
	datum key, content;
	struct dirent *catlist;
	struct stat buf;
	size_t len, lencat;
	char *filename;
	
	if ( (cdir = opendir(catdir)) == NULL){
		fprintf(stderr,
		  "%s: touch_stray_src: could not open dir for reading: ",
		  prognam);
		perror(catdir);
		exit(1);
	}

	strcat(mandir, "/");
	strcat(catdir, "/");
	len = strlen(mandir);
	lencat = strlen(catdir);
		
	while ((catlist = readdir(cdir)) != NULL) {
		if (*catlist->d_name == '.')
			continue;

		strcpy(catdir + lencat, catlist->d_name);
		strcpy(mandir + len, catlist->d_name);
		*(strrchr(mandir, '.')) = '\0'; /* for .Z or .gz */

		/* 
		 * mandir now holds the would-be relative
		 * source man file for the located zcat file
		 */ 

		if (debug)
			fprintf(stderr, "%s: stat: %d, errorno: %d, error: %s\n",
			  mandir, stat(mandir, &buf), errno, strerror(errno));

		if (stat(mandir, &buf) == -1 && errno == 2) {

			/* source non existent */

			key.dptr = strdup(catlist->d_name);
			*(strrchr(key.dptr, '.')) = '\0'; /* .Z or .gz */
			*(strrchr(key.dptr, '.')) = '\0'; /* .# */
			key.dsize = strlen(key.dptr) + 1;

			content = MYDBM_FETCH(dbf, key);
			free(key.dptr);

			/* 
			 * check the db, it might exist in another 
			 * dimension. (source tree)
			 * Make sure that what we've found belongs to
			 * the right section. Check all paths, not again...
			 */

			if ( (filename = exists(content.dptr, catlist->d_name)) 
			  == NULL) {

				/* 
				 * It absolutely doesn't exist. 
				 * We can either bin it, or be more
				 * friendly and create a place-holder 
				 * for it.
				 */
				 	
				printf("%s: STRAYCAT\n", catdir);

				if (del_rel_stray) {
					if (unlink(catdir) != 0) {
						fprintf(stderr,
						  "unable to remove ");
						perror(catdir);
					}
				}

				if (!create_place)
					continue;

				if ( (fd = fopen(mandir, "w")) == NULL){
					fprintf(stderr,
					  "%s: creation: ", prognam);
					perror(mandir);
					continue;
				}

				printf("%s: created place holder\n",
				  mandir);
					  
				if (fclose(fd) != 0){
					fprintf(stderr,
					  "%s: could not close: ", prognam);
					perror(mandir);
					exit(1);
				}
			} else {
				/* 
				 * this is where it gets tricky.
				 * we have a stray cat, but we have the 
				 * source in
				 * a different man tree :-)
				 *
				 * I guess that the best thing to do is 
				 * tell the user
				 * and then get rid of the stray cat.
				 *
			  	 * check to see if database file 
			  	 * actually
			  	 * exists in reality. If not, db is
			  	 * out of sync. die.
			  	 */ 

			  	if (del_place)
			  		continue;

				if (stat(filename, &buf) == -1 
				  && errno == 2) {
				  	fputs("ERROR: Global database is out of date with the filesystem, run mandb first.\n", stderr);
				  	exit(0);
				}
				
			 	printf("%s: STRAYCAT, but has a NON RELATIVE source file:\n\t%s\n",
			 	  catdir, filename);

				if (del_nonrel_stray) {
				 	if (unlink(catdir) != 0) {
				 		fputs("unable to remove ", stderr);
				 		perror(catdir);
				 	}
				}
				MYDBM_FREE(content.dptr);
			}
			
		} else if (buf.st_size == 0) {
	
			/* we have a cat place holder */
	
			printf("%s: placeholder for stray cat\n", mandir);

			if (del_place) {
				if (unlink(mandir) != 0) {
				 	fputs("unable to remove ", stderr);
				 	perror(mandir);
				} else
					printf("%s: deleted place-holder\n", mandir);
			}
		}
	}
}

#ifdef FSSTND

/*
 * Change name of form "/usr/.../man" to "/var/catman/..." 
 */

char *convert_name_sc(char *name)
{
	char *to_name, *newto_name;
	char *t2, *t1;

	to_name = strdup (name);

	t1 = strrchr (to_name, '/');
	*t1 = '\0';
	t2 = strrchr (to_name, '/');

	newto_name = (char *) malloc (sizeof CAT_ROOT + strlen(t2) + 2);
	strcpy(newto_name, CAT_ROOT);
	strcat(newto_name, t2);

	free(to_name);
	return newto_name;
}
#endif

char *exists(char *list, char *catname)
{
	char *t1;
	size_t len;

	if (list == NULL)
	return NULL;

	len = strlen(catname) - (sizeof COMPRESS_EXT - 1);  /* - 2; */

	while ( (t1 = strrchr(list, ':')) != NULL) {
		*t1 = '\0';
		if (strncmp(strrchr(++t1, '/') + 1, catname, len) == 0)
		  	return t1;
	}
	if (strncmp(strrchr(list, '/') + 1, catname, len) == 0)
		return list;

	return NULL;
}
