/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "gtest/gtest.h"
#include "MediaData.h"
#include "mozilla/ArrayUtils.h"
#include "mp4_demuxer/BufferStream.h"
#include "mp4_demuxer/MP4Metadata.h"
#include "mp4_demuxer/MoofParser.h"

using namespace mozilla;
using namespace mp4_demuxer;

TEST(stagefright_MP4Metadata, EmptyStream)
{
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(0);
  nsRefPtr<BufferStream> stream = new BufferStream(buffer);

  EXPECT_FALSE(MP4Metadata::HasCompleteMetadata(stream));
  nsRefPtr<MediaByteBuffer> metadataBuffer = MP4Metadata::Metadata(stream);
  EXPECT_FALSE(metadataBuffer);

  MP4Metadata metadata(stream);
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kUndefinedTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kAudioTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kVideoTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kTextTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(static_cast<TrackInfo::TrackType>(-1)));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kUndefinedTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kAudioTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kVideoTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kTextTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(static_cast<TrackInfo::TrackType>(-1), 0));
  // We can seek anywhere in any MPEG4.
  EXPECT_TRUE(metadata.CanSeek());
  EXPECT_FALSE(metadata.Crypto().valid);
}

TEST(stagefright_MoofParser, EmptyStream)
{
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(0);
  nsRefPtr<BufferStream> stream = new BufferStream(buffer);

  Monitor monitor("MP4Metadata::gtest");
  MonitorAutoLock mon(monitor);
  MoofParser parser(stream, 0, false, &monitor);
  EXPECT_EQ(0u, parser.mOffset);
  EXPECT_TRUE(parser.ReachedEnd());

  nsTArray<MediaByteRange> byteRanges;
  byteRanges.AppendElement(MediaByteRange(0, 0));
  EXPECT_FALSE(parser.RebuildFragmentedIndex(byteRanges));

  EXPECT_TRUE(parser.GetCompositionRange(byteRanges).IsNull());
  EXPECT_TRUE(parser.mInitRange.IsNull());
  EXPECT_EQ(0u, parser.mOffset);
  EXPECT_TRUE(parser.ReachedEnd());
  EXPECT_FALSE(parser.HasMetadata());
  nsRefPtr<MediaByteBuffer> metadataBuffer = parser.Metadata();
  EXPECT_FALSE(metadataBuffer);
  EXPECT_TRUE(parser.FirstCompleteMediaSegment().IsNull());
  EXPECT_TRUE(parser.FirstCompleteMediaHeader().IsNull());
}

uint8_t test_case_mp4[] = {
  0x00, 0x00, 0x00, 0x20,  'f',  't',  'y',  'p',  'i',  's',  'o',  'm',
  0x00, 0x00, 0x02, 0x00,  'i',  's',  'o',  'm',  'i',  's',  'o',  '2',
   'a',  'v',  'c',  '1',  'm',  'p',  '4',  '1', 0x00, 0x00, 0x00, 0x08,
   'f',  'r',  'e',  'e', 0x00, 0x00, 0x07, 0xcc,  'm',  'd',  'a',  't',
  0x00, 0x00, 0x02, 0xaf, 0x06, 0x05, 0xff, 0xff, 0xab, 0xdc, 0x45, 0xe9,
  0xbd, 0xe6, 0xd9, 0x48, 0xb7, 0x96, 0x2c, 0xd8, 0x20, 0xd9, 0x23, 0xee,
  0xef, 0x78, 0x32, 0x36, 0x34, 0x20, 0x2d, 0x20, 0x63, 0x6f, 0x72, 0x65,
  0x20, 0x31, 0x34, 0x36, 0x20, 0x72, 0x32, 0x35, 0x33, 0x38, 0x20, 0x31,
  0x32, 0x31, 0x33, 0x39, 0x36, 0x63, 0x20, 0x2d, 0x20, 0x48, 0x2e, 0x32,
  0x36, 0x34, 0x2f, 0x4d, 0x50, 0x45, 0x47, 0x2d, 0x34, 0x20, 0x41, 0x56,
  0x43, 0x20, 0x63, 0x6f, 0x64, 0x65, 0x63, 0x20, 0x2d, 0x20, 0x43, 0x6f,
  0x70, 0x79, 0x6c, 0x65, 0x66, 0x74, 0x20, 0x32, 0x30, 0x30, 0x33, 0x2d,
  0x32, 0x30, 0x31, 0x35, 0x20, 0x2d, 0x20, 0x68, 0x74, 0x74, 0x70, 0x3a,
  0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x76, 0x69, 0x64, 0x65, 0x6f, 0x6c,
  0x61, 0x6e, 0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x78, 0x32, 0x36, 0x34, 0x2e,
  0x68, 0x74, 0x6d, 0x6c, 0x20, 0x2d, 0x20, 0x6f, 0x70, 0x74, 0x69, 0x6f,
  0x6e, 0x73, 0x3a, 0x20, 0x63, 0x61, 0x62, 0x61, 0x63, 0x3d, 0x31, 0x20,
  0x72, 0x65, 0x66, 0x3d, 0x33, 0x20, 0x64, 0x65, 0x62, 0x6c, 0x6f, 0x63,
  0x6b, 0x3d, 0x31, 0x3a, 0x30, 0x3a, 0x30, 0x20, 0x61, 0x6e, 0x61, 0x6c,
  0x79, 0x73, 0x65, 0x3d, 0x30, 0x78, 0x33, 0x3a, 0x30, 0x78, 0x31, 0x31,
  0x33, 0x20, 0x6d, 0x65, 0x3d, 0x68, 0x65, 0x78, 0x20, 0x73, 0x75, 0x62,
  0x6d, 0x65, 0x3d, 0x37, 0x20, 0x70, 0x73, 0x79, 0x3d, 0x31, 0x20, 0x70,
  0x73, 0x79, 0x5f, 0x72, 0x64, 0x3d, 0x31, 0x2e, 0x30, 0x30, 0x3a, 0x30,
  0x2e, 0x30, 0x30, 0x20, 0x6d, 0x69, 0x78, 0x65, 0x64, 0x5f, 0x72, 0x65,
  0x66, 0x3d, 0x31, 0x20, 0x6d, 0x65, 0x5f, 0x72, 0x61, 0x6e, 0x67, 0x65,
  0x3d, 0x31, 0x36, 0x20, 0x63, 0x68, 0x72, 0x6f, 0x6d, 0x61, 0x5f, 0x6d,
  0x65, 0x3d, 0x31, 0x20, 0x74, 0x72, 0x65, 0x6c, 0x6c, 0x69, 0x73, 0x3d,
  0x31, 0x20, 0x38, 0x78, 0x38, 0x64, 0x63, 0x74, 0x3d, 0x31, 0x20, 0x63,
  0x71, 0x6d, 0x3d, 0x30, 0x20, 0x64, 0x65, 0x61, 0x64, 0x7a, 0x6f, 0x6e,
  0x65, 0x3d, 0x32, 0x31, 0x2c, 0x31, 0x31, 0x20, 0x66, 0x61, 0x73, 0x74,
  0x5f, 0x70, 0x73, 0x6b, 0x69, 0x70, 0x3d, 0x31, 0x20, 0x63, 0x68, 0x72,
  0x6f, 0x6d, 0x61, 0x5f, 0x71, 0x70, 0x5f, 0x6f, 0x66, 0x66, 0x73, 0x65,
  0x74, 0x3d, 0x2d, 0x32, 0x20, 0x74, 0x68, 0x72, 0x65, 0x61, 0x64, 0x73,
  0x3d, 0x31, 0x32, 0x20, 0x6c, 0x6f, 0x6f, 0x6b, 0x61, 0x68, 0x65, 0x61,
  0x64, 0x5f, 0x74, 0x68, 0x72, 0x65, 0x61, 0x64, 0x73, 0x3d, 0x31, 0x20,
  0x73, 0x6c, 0x69, 0x63, 0x65, 0x64, 0x5f, 0x74, 0x68, 0x72, 0x65, 0x61,
  0x64, 0x73, 0x3d, 0x30, 0x20, 0x6e, 0x72, 0x3d, 0x30, 0x20, 0x64, 0x65,
  0x63, 0x69, 0x6d, 0x61, 0x74, 0x65, 0x3d, 0x31, 0x20, 0x69, 0x6e, 0x74,
  0x65, 0x72, 0x6c, 0x61, 0x63, 0x65, 0x64, 0x3d, 0x30, 0x20, 0x62, 0x6c,
  0x75, 0x72, 0x61, 0x79, 0x5f, 0x63, 0x6f, 0x6d, 0x70, 0x61, 0x74, 0x3d,
  0x30, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x74, 0x72, 0x61, 0x69, 0x6e, 0x65,
  0x64, 0x5f, 0x69, 0x6e, 0x74, 0x72, 0x61, 0x3d, 0x30, 0x20, 0x62, 0x66,
  0x72, 0x61, 0x6d, 0x65, 0x73, 0x3d, 0x33, 0x20, 0x62, 0x5f, 0x70, 0x79,
  0x72, 0x61, 0x6d, 0x69, 0x64, 0x3d, 0x32, 0x20, 0x62, 0x5f, 0x61, 0x64,
  0x61, 0x70, 0x74, 0x3d, 0x31, 0x20, 0x62, 0x5f, 0x62, 0x69, 0x61, 0x73,
  0x3d, 0x30, 0x20, 0x64, 0x69, 0x72, 0x65, 0x63, 0x74, 0x3d, 0x31, 0x20,
  0x77, 0x65, 0x69, 0x67, 0x68, 0x74, 0x62, 0x3d, 0x31, 0x20, 0x6f, 0x70,
  0x65, 0x6e, 0x5f, 0x67, 0x6f, 0x70, 0x3d, 0x30, 0x20, 0x77, 0x65, 0x69,
  0x67, 0x68, 0x74, 0x70, 0x3d, 0x32, 0x20, 0x6b, 0x65, 0x79, 0x69, 0x6e,
  0x74, 0x3d, 0x32, 0x35, 0x30, 0x20, 0x6b, 0x65, 0x79, 0x69, 0x6e, 0x74,
  0x5f, 0x6d, 0x69, 0x6e, 0x3d, 0x32, 0x35, 0x20, 0x73, 0x63, 0x65, 0x6e,
  0x65, 0x63, 0x75, 0x74, 0x3d, 0x34, 0x30, 0x20, 0x69, 0x6e, 0x74, 0x72,
  0x61, 0x5f, 0x72, 0x65, 0x66, 0x72, 0x65, 0x73, 0x68, 0x3d, 0x30, 0x20,
  0x72, 0x63, 0x5f, 0x6c, 0x6f, 0x6f, 0x6b, 0x61, 0x68, 0x65, 0x61, 0x64,
  0x3d, 0x34, 0x30, 0x20, 0x72, 0x63, 0x3d, 0x63, 0x72, 0x66, 0x20, 0x6d,
  0x62, 0x74, 0x72, 0x65, 0x65, 0x3d, 0x31, 0x20, 0x63, 0x72, 0x66, 0x3d,
  0x32, 0x33, 0x2e, 0x30, 0x20, 0x71, 0x63, 0x6f, 0x6d, 0x70, 0x3d, 0x30,
  0x2e, 0x36, 0x30, 0x20, 0x71, 0x70, 0x6d, 0x69, 0x6e, 0x3d, 0x30, 0x20,
  0x71, 0x70, 0x6d, 0x61, 0x78, 0x3d, 0x36, 0x39, 0x20, 0x71, 0x70, 0x73,
  0x74, 0x65, 0x70, 0x3d, 0x34, 0x20, 0x69, 0x70, 0x5f, 0x72, 0x61, 0x74,
  0x69, 0x6f, 0x3d, 0x31, 0x2e, 0x34, 0x30, 0x20, 0x61, 0x71, 0x3d, 0x31,
  0x3a, 0x31, 0x2e, 0x30, 0x30, 0x00, 0x80, 0x00, 0x00, 0x04, 0xfb, 0x65,
  0x88, 0x84, 0x00, 0x25, 0xff, 0xd7, 0x2b, 0x47, 0x53, 0x79, 0x61, 0xc8,
  0xdf, 0x88, 0x8e, 0xdc, 0xef, 0xc0, 0x85, 0x00, 0x99, 0x10, 0x58, 0x46,
  0xf1, 0x28, 0xf2, 0x0c, 0xfc, 0x41, 0x82, 0xa0, 0x45, 0x81, 0xb5, 0x7e,
  0xcc, 0x27, 0x62, 0x2f, 0xac, 0x5c, 0xe8, 0xa9, 0xcf, 0x0e, 0x54, 0x17,
  0x68, 0x13, 0x48, 0x0d, 0xa1, 0x5d, 0xd1, 0x8b, 0xdf, 0x6e, 0xd0, 0xc8,
  0x8f, 0xb3, 0xc2, 0xaf, 0xd1, 0xdc, 0x98, 0x74, 0x7e, 0x0f, 0x69, 0xbe,
  0x54, 0xa2, 0x83, 0x4a, 0xc4, 0xf1, 0x79, 0x35, 0x60, 0x10, 0xb4, 0xfb,
  0xa7, 0x52, 0x3a, 0x99, 0x5e, 0x39, 0x0d, 0x04, 0xad, 0x24, 0x49, 0xfd,
  0xbe, 0x5f, 0x49, 0xdf, 0x36, 0x91, 0x1f, 0x62, 0x7f, 0x00, 0xa1, 0xba,
  0x3c, 0xe9, 0xf3, 0x38, 0x08, 0xb7, 0x87, 0x9a, 0x11, 0xae, 0x5f, 0xf2,
  0x66, 0x14, 0xc3, 0xa9, 0xf0, 0x76, 0x69, 0x5f, 0x1d, 0x20, 0x19, 0xc0,
  0x50, 0x94, 0xb4, 0x7b, 0x44, 0xaf, 0x21, 0x24, 0x80, 0x3f, 0x2a, 0x81,
  0x1b, 0xe4, 0xae, 0xfe, 0xb0, 0x99, 0xd3, 0xa5, 0x1d, 0x3f, 0x27, 0x0c,
  0x61, 0x54, 0x4d, 0x39, 0x8b, 0x1d, 0x12, 0x65, 0xe1, 0xb3, 0x4d, 0xc1,
  0x9d, 0xef, 0xfe, 0x48, 0xc6, 0xc9, 0xf4, 0x9a, 0x09, 0xc1, 0x3c, 0x5c,
  0xe3, 0xa1, 0x16, 0xec, 0x83, 0xb6, 0xb9, 0xc2, 0x94, 0x64, 0xc9, 0xf2,
  0x91, 0x51, 0x4c, 0xaf, 0xe5, 0xc3, 0x8a, 0x99, 0x1b, 0x22, 0x86, 0x55,
  0x70, 0xbf, 0xc7, 0x99, 0xc3, 0x07, 0x15, 0xd6, 0xdb, 0x27, 0x46, 0x08,
  0x19, 0x7e, 0x0a, 0x9b, 0x82, 0xe0, 0x15, 0x1d, 0x57, 0x8a, 0x2c, 0xc4,
  0x05, 0xf8, 0x09, 0xfb, 0x3c, 0x47, 0x4f, 0x55, 0xc6, 0x3e, 0x0b, 0x9d,
  0xe5, 0x03, 0x3e, 0xc9, 0x4c, 0xeb, 0xa2, 0x1b, 0x04, 0xfa, 0x99, 0x44,
  0x29, 0x06, 0x2e, 0x57, 0x61, 0xa3, 0x63, 0x00, 0x53, 0x08, 0x20, 0xcb,
  0x15, 0x50, 0x95, 0x13, 0x24, 0x63, 0xc9, 0x4f, 0xc0, 0xb0, 0x88, 0xe6,
  0x38, 0xef, 0x96, 0x99, 0xd2, 0x96, 0x47, 0xc7, 0xc8, 0x61, 0xac, 0xcc,
  0x24, 0x0a, 0x1f, 0x79, 0xce, 0x91, 0x88, 0xf5, 0xe9, 0xe2, 0x2d, 0x88,
  0xaf, 0x37, 0xca, 0xa3, 0x90, 0x5b, 0x83, 0x1a, 0x9c, 0x92, 0xda, 0x33,
  0x69, 0xe6, 0x7c, 0x14, 0xa6, 0x85, 0x00, 0x42, 0x46, 0x51, 0xcf, 0x50,
  0xed, 0x34, 0xe8, 0x8e, 0xb5, 0x78, 0xeb, 0x09, 0x37, 0x6e, 0x82, 0xea,
  0x7b, 0xe1, 0xfd, 0x45, 0x61, 0x77, 0x14, 0x6a, 0x75, 0xbf, 0xb8, 0x13,
  0x10, 0xc0, 0x08, 0xca, 0x1c, 0x51, 0xc7, 0x3a, 0x0a, 0x2b, 0x89, 0x5d,
  0x3c, 0x8d, 0xf4, 0x11, 0xc0, 0x3a, 0x90, 0x02, 0xc0, 0x0c, 0xe8, 0x59,
  0x11, 0xac, 0xe0, 0x5d, 0xe2, 0x95, 0x78, 0x09, 0xd5, 0xfa, 0x26, 0xf5,
  0x23, 0x30, 0x0a, 0xfd, 0x54, 0x0b, 0x73, 0x1c, 0x81, 0x22, 0x48, 0x8b,
  0x79, 0xe0, 0xf9, 0xe8, 0xde, 0x7a, 0x79, 0xba, 0xef, 0xdd, 0x5a, 0xd6,
  0x44, 0xb7, 0xa8, 0x41, 0xd0, 0xa1, 0xeb, 0x45, 0xf8, 0x4d, 0x7f, 0x97,
  0x48, 0x82, 0xd2, 0x81, 0x61, 0x08, 0x15, 0x9e, 0xfe, 0x78, 0xba, 0xdf,
  0xb0, 0x13, 0x92, 0x59, 0x4e, 0x4d, 0xfd, 0xff, 0x5d, 0x66, 0x14, 0xd4,
  0x0b, 0x43, 0x9f, 0xd8, 0x62, 0x4e, 0x54, 0x83, 0xf9, 0x59, 0x48, 0x2f,
  0x26, 0x21, 0xd4, 0xf0, 0x98, 0x6b, 0x14, 0x61, 0x4b, 0xf6, 0x00, 0xcf,
  0xe3, 0x24, 0x4b, 0x2f, 0xd4, 0x5e, 0x6d, 0x40, 0x4b, 0x52, 0xea, 0xa5,
  0x89, 0x94, 0x0f, 0xd2, 0xeb, 0x02, 0x54, 0x68, 0x26, 0xcf, 0x2d, 0x83,
  0x0f, 0x62, 0x1b, 0x9e, 0x75, 0x81, 0x65, 0x30, 0xdd, 0x03, 0xb0, 0xc1,
  0xda, 0x4a, 0xc9, 0xd8, 0x64, 0x29, 0x6a, 0xe2, 0x83, 0xf1, 0xb8, 0x4e,
  0xc5, 0xaf, 0x8e, 0xf0, 0x8c, 0xbb, 0xda, 0xea, 0x8a, 0xbc, 0xcc, 0xa3,
  0xf2, 0x19, 0xe8, 0xeb, 0x7e, 0x18, 0x64, 0x91, 0x1f, 0xdd, 0xc8, 0xc8,
  0xf8, 0xad, 0xc8, 0x3b, 0xf9, 0x92, 0x74, 0x03, 0x9b, 0x90, 0x51, 0xb7,
  0xb5, 0xe3, 0x80, 0x4f, 0x8a, 0x84, 0xfa, 0xa4, 0xd6, 0x9c, 0x53, 0x0b,
  0x81, 0xc3, 0xee, 0x9e, 0x70, 0xa3, 0xbd, 0x3f, 0xbb, 0x60, 0x2d, 0x97,
  0x65, 0xa7, 0x69, 0x2f, 0x22, 0x49, 0x65, 0xf4, 0xb0, 0x33, 0xbc, 0xd2,
  0x80, 0x1b, 0x2c, 0x3a, 0xe3, 0x04, 0x8c, 0x49, 0x42, 0x25, 0xa0, 0x6d,
  0x3c, 0xfe, 0xfa, 0x70, 0x90, 0x6a, 0x30, 0xf4, 0x0c, 0xe4, 0x3f, 0x78,
  0xf9, 0xba, 0x55, 0xb9, 0xfa, 0xd7, 0xce, 0x05, 0xbc, 0xe9, 0xc9, 0xad,
  0x4a, 0x37, 0xa0, 0xf7, 0x8d, 0x96, 0x22, 0xf6, 0x38, 0x8d, 0xf4, 0xf6,
  0xe6, 0x8b, 0x45, 0xac, 0x13, 0xc5, 0xe6, 0x05, 0x1e, 0x09, 0xd7, 0x98,
  0xb3, 0xb6, 0x59, 0xe4, 0x3b, 0x47, 0x16, 0x6e, 0xdf, 0xac, 0x7f, 0x38,
  0x6e, 0xf9, 0xcf, 0xaa, 0x68, 0x98, 0xdb, 0x22, 0x89, 0x6e, 0xad, 0xbe,
  0xed, 0xb1, 0x82, 0xa0, 0xc2, 0x9b, 0xd5, 0x79, 0x89, 0x96, 0xf6, 0xd9,
  0x8f, 0x58, 0x77, 0x15, 0x2c, 0x73, 0xeb, 0x89, 0xcc, 0xf0, 0x37, 0x4d,
  0x41, 0x70, 0xc5, 0x58, 0xae, 0x77, 0xab, 0x30, 0xcf, 0x6c, 0x7c, 0x1c,
  0x52, 0x9a, 0x62, 0xf6, 0xf8, 0x0a, 0x65, 0x92, 0x83, 0x01, 0xc3, 0x60,
  0xed, 0xfd, 0x4d, 0x9a, 0x4b, 0xd4, 0xa5, 0xe1, 0xc4, 0xe2, 0xe1, 0x8c,
  0x64, 0xce, 0x54, 0x9c, 0xa9, 0x7f, 0xb9, 0x34, 0x88, 0xb1, 0x17, 0xde,
  0x85, 0x27, 0x43, 0x81, 0x3b, 0x37, 0x27, 0x25, 0xbf, 0xba, 0x6c, 0x69,
  0xb4, 0xce, 0xcd, 0xef, 0x7c, 0xee, 0x48, 0xb9, 0x8a, 0x09, 0x1f, 0x42,
  0x8e, 0xc3, 0x14, 0xe4, 0xfd, 0xda, 0xa0, 0xfa, 0x7d, 0x0b, 0x68, 0x6a,
  0x81, 0xb1, 0x96, 0xf9, 0x07, 0xf2, 0xed, 0x32, 0xf3, 0x52, 0x15, 0x00,
  0x2f, 0x5f, 0x6e, 0x55, 0xc6, 0x85, 0x4b, 0xdc, 0x3d, 0x7c, 0xa6, 0xa7,
  0xeb, 0x80, 0xab, 0xf3, 0xfe, 0x21, 0xe6, 0x1d, 0xbd, 0xca, 0x33, 0x29,
  0x9c, 0x94, 0xa4, 0x7f, 0xec, 0x67, 0x6c, 0xc0, 0x0e, 0xea, 0x5f, 0x9a,
  0x30, 0x54, 0x9c, 0xf2, 0x8f, 0xaa, 0x5f, 0xc3, 0x3e, 0x61, 0x54, 0x41,
  0x8a, 0xbf, 0x7f, 0xff, 0x8a, 0xfb, 0x7f, 0x8c, 0x40, 0xe4, 0x5a, 0xbe,
  0xe5, 0x39, 0xa4, 0xdd, 0x9b, 0xa6, 0xab, 0xd7, 0xee, 0x15, 0x32, 0x04,
  0xd1, 0xbd, 0x7c, 0xe4, 0x98, 0x3f, 0x3f, 0x40, 0x87, 0xbc, 0x01, 0x36,
  0xe7, 0xcd, 0x7b, 0xcf, 0xf7, 0xe9, 0x60, 0x1d, 0xea, 0xe2, 0x6e, 0x13,
  0x3c, 0xd0, 0x28, 0x2d, 0xa8, 0x9d, 0x25, 0x06, 0x99, 0xf8, 0xdc, 0x9f,
  0x9f, 0xc5, 0x5a, 0xfd, 0x8f, 0x31, 0xcf, 0xc3, 0xe5, 0xe6, 0xc9, 0x99,
  0xae, 0x56, 0x72, 0xe2, 0x2f, 0xdf, 0x2c, 0x0e, 0x7d, 0x51, 0xc8, 0x35,
  0x40, 0x23, 0x9e, 0x52, 0x44, 0x6f, 0x17, 0x40, 0x01, 0xd1, 0x85, 0x07,
  0x55, 0xef, 0x10, 0xc9, 0xe5, 0xb9, 0xef, 0xbf, 0xf4, 0xe5, 0x38, 0xd3,
  0x1f, 0x2b, 0x91, 0x51, 0x60, 0x75, 0xc8, 0x95, 0xcc, 0x9d, 0x5a, 0xfa,
  0x69, 0xae, 0x9a, 0x16, 0x41, 0x07, 0x9c, 0x94, 0x40, 0xb7, 0xa9, 0xdd,
  0x8d, 0xcb, 0xce, 0xc5, 0xf2, 0x5e, 0x59, 0x67, 0x69, 0x07, 0xd3, 0x04,
  0xa4, 0x99, 0x56, 0xd9, 0xdc, 0x04, 0x9a, 0x66, 0x62, 0x7b, 0x67, 0xc8,
  0xd0, 0x34, 0x69, 0x5b, 0x4a, 0xce, 0x6e, 0x53, 0x0e, 0x62, 0xf7, 0x85,
  0xe0, 0xd7, 0xb7, 0x27, 0x55, 0x3a, 0x4d, 0x36, 0x47, 0xf2, 0x74, 0x02,
  0xb3, 0x66, 0x2c, 0xda, 0xc3, 0xb3, 0x38, 0x94, 0x67, 0x82, 0x44, 0xf4,
  0x12, 0xe4, 0x1c, 0x8f, 0x22, 0x4d, 0x32, 0x35, 0xf0, 0xe3, 0x41, 0x0a,
  0x7d, 0xe4, 0xb4, 0x6e, 0x10, 0x4f, 0xa9, 0x46, 0xd5, 0xab, 0x90, 0x4c,
  0xad, 0x2c, 0x30, 0xd0, 0x9e, 0x68, 0x2c, 0xc4, 0x3c, 0xf7, 0x05, 0xdf,
  0x22, 0xaa, 0xb0, 0x82, 0xbb, 0x2c, 0x67, 0x8c, 0xfd, 0x1b, 0x04, 0x41,
  0xf1, 0x4f, 0x77, 0xa4, 0xdb, 0xfb, 0xca, 0x1d, 0xd7, 0x61, 0x8a, 0x3e,
  0x89, 0x40, 0x88, 0xf2, 0xda, 0x35, 0x2b, 0x9d, 0xbf, 0xd8, 0x98, 0x55,
  0x4e, 0x60, 0xac, 0xc1, 0x1b, 0xd4, 0xe0, 0xb8, 0x6d, 0x13, 0xa0, 0xa3,
  0x24, 0x80, 0xa0, 0xe6, 0x12, 0xad, 0x27, 0x36, 0xee, 0xd7, 0x55, 0x4b,
  0xb4, 0x1a, 0xd2, 0x87, 0x31, 0x1a, 0x00, 0x53, 0xe9, 0x0f, 0xb7, 0x50,
  0xeb, 0xdb, 0x63, 0xfe, 0xc3, 0xd0, 0xb1, 0x25, 0xdc, 0x63, 0x66, 0xcc,
  0xe6, 0x99, 0xa3, 0x34, 0x0b, 0x1d, 0xdd, 0x84, 0x88, 0x3c, 0xfc, 0x79,
  0xf5, 0x13, 0x0a, 0xe0, 0xca, 0x9e, 0x02, 0xeb, 0x06, 0xab, 0x6d, 0x80,
  0xeb, 0x06, 0x3d, 0x9a, 0xbb, 0x97, 0xd5, 0xd2, 0x23, 0x22, 0x17, 0xca,
  0x7a, 0x34, 0x09, 0xfe, 0x53, 0xfa, 0xc1, 0x34, 0x2a, 0x2c, 0xcb, 0x07,
  0xd3, 0x92, 0x86, 0x9c, 0x7b, 0xd6, 0xdc, 0xe9, 0x5d, 0xa9, 0xcd, 0xb3,
  0x72, 0xc1, 0x5d, 0xcd, 0x3f, 0xc2, 0x9b, 0xcf, 0x5a, 0x54, 0xbe, 0x50,
  0x84, 0xbc, 0xe1, 0x33, 0xbd, 0xfd, 0xb6, 0x59, 0x49, 0x11, 0x25, 0xc9,
  0x01, 0x57, 0x78, 0x8a, 0xef, 0x16, 0x7e, 0x15, 0x2d, 0x9d, 0x30, 0x4b,
  0x68, 0xa4, 0x3b, 0x99, 0xbf, 0x11, 0x70, 0x0f, 0x17, 0x33, 0xd5, 0x6e,
  0x31, 0x86, 0x7a, 0xea, 0x12, 0xdb, 0xb9, 0xbc, 0x67, 0x4e, 0x79, 0x58,
  0x2f, 0x81, 0x00, 0x00, 0x00, 0x0e, 0x41, 0x9a, 0x21, 0x6c, 0x42, 0xdf,
  0x18, 0xd3, 0x2d, 0x01, 0x43, 0x7f, 0x24, 0x38, 0x00, 0x00, 0x03, 0x1f,
   'm',  'o',  'o',  'v', 0x00, 0x00, 0x00, 0x6c,  'm',  'v',  'h',  'd',
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x03, 0xe8, 0x00, 0x00, 0x00, 0x50, 0x00, 0x01, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02, 0x49,  't',  'r',  'a',  'k',
  0x00, 0x00, 0x00, 0x5c,  't',  'k',  'h',  'd', 0x00, 0x00, 0x00, 0x03,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
  0x00, 0xa0, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24,
   'e',  'd',  't',  's', 0x00, 0x00, 0x00, 0x1c,  'e',  'l',  's',  't',
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x50,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0xc1,
   'm',  'd',  'i',  'a', 0x00, 0x00, 0x00, 0x20,  'm',  'd',  'h',  'd',
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x04, 0x00, 0x15, 0xc7, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x2d, 0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x76, 0x69, 0x64, 0x65, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 0x69, 0x64, 0x65,
  0x6f, 0x48, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x72, 0x00, 0x00, 0x00, 0x01,
  0x6c, 0x6d, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00, 0x14, 0x76, 0x6d, 0x68,
  0x64, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x24, 0x64, 0x69, 0x6e, 0x66, 0x00, 0x00, 0x00,
  0x1c, 0x64, 0x72, 0x65, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x0c, 0x75, 0x72, 0x6c, 0x20, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x01, 0x2c, 0x73, 0x74, 0x62, 0x6c, 0x00, 0x00, 0x00,
  0xac, 0x73, 0x74, 0x73, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x9c, 0x61, 0x76, 0x63, 0x31, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
  0x5a, 0x00, 0x48, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x18, 0xff, 0xff, 0x00, 0x00, 0x00, 0x36, 0x61, 0x76, 0x63, 0x43, 0x01,
  0x64, 0x00, 0x0a, 0xff, 0xe1, 0x00, 0x1d, 0x67, 0x64, 0x00, 0x0a, 0xac,
  0xd9, 0x42, 0x0d, 0xf9, 0x3f, 0xf0, 0x00, 0x50, 0x00, 0x41, 0x00, 0x00,
  0x03, 0x00, 0x01, 0x00, 0x00, 0x03, 0x00, 0x32, 0x0f, 0x12, 0x25, 0x96,
  0x01, 0x00, 0x06, 0x68, 0xeb, 0xe3, 0xcb, 0x22, 0xc0, 0x00, 0x00, 0x00,
  0x10, 0x70, 0x61, 0x73, 0x70, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
  0x04, 0x00, 0x00, 0x00, 0x18, 0x73, 0x74, 0x74, 0x73, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x02,
  0x00, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x73, 0x73, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x1c, 0x73, 0x74, 0x73, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x1c, 0x73, 0x74, 0x73, 0x7a, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x07,
  0xb2, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x73, 0x74, 0x63,
  0x6f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x30, 0x00, 0x00, 0x00, 0x62, 0x75, 0x64, 0x74, 0x61, 0x00, 0x00, 0x00,
  0x5a, 0x6d, 0x65, 0x74, 0x61, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x21, 0x68, 0x64, 0x6c, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x6d, 0x64, 0x69, 0x72, 0x61, 0x70, 0x70, 0x6c, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x2d, 0x69, 0x6c,
  0x73, 0x74, 0x00, 0x00, 0x00, 0x25, 0xa9, 0x74, 0x6f, 0x6f, 0x00, 0x00,
  0x00, 0x1d, 0x64, 0x61, 0x74, 0x61, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
  0x00, 0x00, 0x4c, 0x61, 0x76, 0x66, 0x35, 0x36, 0x2e, 0x33, 0x33, 0x2e,
  0x31, 0x30, 0x31
};
static const size_t test_case_mp4_len = ArrayLength(test_case_mp4);

TEST(stagefright_MPEG4Metadata, test_case_mp4)
{
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(test_case_mp4_len);
  buffer->AppendElements(test_case_mp4, test_case_mp4_len);
  nsRefPtr<BufferStream> stream = new BufferStream(buffer);

  EXPECT_TRUE(MP4Metadata::HasCompleteMetadata(stream));
  nsRefPtr<MediaByteBuffer> metadataBuffer = MP4Metadata::Metadata(stream);
  EXPECT_TRUE(metadataBuffer);

  MP4Metadata metadata(stream);
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kUndefinedTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kAudioTrack));
  EXPECT_EQ(1u, metadata.GetNumberTracks(TrackInfo::kVideoTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(TrackInfo::kTextTrack));
  EXPECT_EQ(0u, metadata.GetNumberTracks(static_cast<TrackInfo::TrackType>(-1)));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kUndefinedTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kAudioTrack, 0));
  UniquePtr<TrackInfo> track = metadata.GetTrackInfo(TrackInfo::kVideoTrack, 0);
  EXPECT_TRUE(!!track);
  EXPECT_FALSE(metadata.GetTrackInfo(TrackInfo::kTextTrack, 0));
  EXPECT_FALSE(metadata.GetTrackInfo(static_cast<TrackInfo::TrackType>(-1), 0));
  // We can see anywhere in any MPEG4.
  EXPECT_TRUE(metadata.CanSeek());
  EXPECT_FALSE(metadata.Crypto().valid);
}

TEST(stagefright_MPEG4Metadata, test_case_mp4_skimming)
{
  static const size_t step = 4u;
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(test_case_mp4_len);
  buffer->AppendElements(test_case_mp4, test_case_mp4_len);
  for (size_t offset = 0; offset < test_case_mp4_len - step; offset += step) {
    nsRefPtr<BufferStream> stream = new BufferStream(buffer);

    // Just exercizing the parser starting at different points through the file,
    // making sure it doesn't crash.
    // No checks because results would differ for each position.
    MP4Metadata::HasCompleteMetadata(stream);
    nsRefPtr<MediaByteBuffer> metadataBuffer = MP4Metadata::Metadata(stream);
    MP4Metadata metadata(stream);

    buffer->RemoveElementsAt(0, step);
  }
}

TEST(stagefright_MoofParser, test_case_mp4)
{
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(test_case_mp4_len);
  buffer->AppendElements(test_case_mp4, test_case_mp4_len);
  nsRefPtr<BufferStream> stream = new BufferStream(buffer);

  Monitor monitor("MP4Metadata::HasCompleteMetadata");
  MonitorAutoLock mon(monitor);
  MoofParser parser(stream, 0, false, &monitor);
  EXPECT_EQ(0u, parser.mOffset);
  EXPECT_FALSE(parser.ReachedEnd());

  nsTArray<MediaByteRange> byteRanges;
  byteRanges.AppendElement(MediaByteRange(0, 0));
  EXPECT_FALSE(parser.RebuildFragmentedIndex(byteRanges));

  EXPECT_TRUE(parser.GetCompositionRange(byteRanges).IsNull());
  EXPECT_TRUE(parser.mInitRange.IsNull());
  EXPECT_EQ(0u, parser.mOffset);
  EXPECT_FALSE(parser.ReachedEnd());
  EXPECT_TRUE(parser.HasMetadata());
  nsRefPtr<MediaByteBuffer> metadataBuffer = parser.Metadata();
  EXPECT_TRUE(metadataBuffer);
  EXPECT_TRUE(parser.FirstCompleteMediaSegment().IsNull());
  EXPECT_TRUE(parser.FirstCompleteMediaHeader().IsNull());
}

TEST(stagefright_MoofParser, test_case_mp4_skimming)
{
  const size_t step = 4u;
  nsRefPtr<MediaByteBuffer> buffer = new MediaByteBuffer(test_case_mp4_len);
  buffer->AppendElements(test_case_mp4, test_case_mp4_len);
  Monitor monitor("MP4Metadata::HasCompleteMetadata");
  MonitorAutoLock mon(monitor);
  for (size_t offset = 0; offset < test_case_mp4_len - step; offset += step) {
    nsRefPtr<BufferStream> stream = new BufferStream(buffer);

    // Just exercizing the parser starting at different points through the file,
    // making sure it doesn't crash.
    // No checks because results would differ for each position.
    MoofParser parser(stream, 0, false, &monitor);
    nsTArray<MediaByteRange> byteRanges;
    byteRanges.AppendElement(MediaByteRange(0, 0));
    EXPECT_FALSE(parser.RebuildFragmentedIndex(byteRanges));
    parser.GetCompositionRange(byteRanges);
    parser.HasMetadata();
    nsRefPtr<MediaByteBuffer> metadataBuffer = parser.Metadata();
    parser.FirstCompleteMediaSegment();
    parser.FirstCompleteMediaHeader();

    buffer->RemoveElementsAt(0, step);
  }
}
