/*
 * bltGraph.c --
 *
 *	This module implements a graph widget for the
 *	Tk toolkit.
 *
 * Copyright 1991-1993 by AT&T Bell Laboratories.
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that the copyright notice and warranty
 * disclaimer appear in supporting documentation, and that the
 * names of AT&T Bell Laboratories any of their entities not be used
 * in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * AT&T disclaims all warranties with regard to this software, including
 * all implied warranties of merchantability and fitness.  In no event
 * shall AT&T be liable for any special, indirect or consequential
 * damages or any damages whatsoever resulting from loss of use, data
 * or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or
 * performance of this software.
 *
 * Graph widget created by Sani Nassif and George Howlett.
 */

/* To do:
 *
 * 1) Fix log manual scale for log axes.
 *
 * 2) Update manual pages.
 *
 * 3) Update internal comments.
 *
 * 4) Pie charts
 *
 * 5) Contour and flow graphs
 *
 * 6) Account for roundoff error when calculating bar widths
 *
 * 7) Arrows for line tags
 *
 * 8) Cross and plus symbols look dopey when < 5 pixels (use lines instead)
 *
 * 9) Make sure reasonable defaults show for configuration items
 *
 * 10) Rubberbox for zooming??? 
 */
#include <ctype.h>
#include <errno.h>

#include <tk.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "bltGraph.h"
#include "bltGrTag.h"
#include "bltGrElem.h"

#ifndef GRAPH_VERSION
#define GRAPH_VERSION "2.0"
#endif

#define TRUE  1
#define FALSE 0

#ifdef hpux
#define BLACK		"black"
#define WHITE		"white"
#define LIGHTGREY       "lightgrey"
#define BISQUE1		"bisque1"
#else
#define BLACK		"#000000"
#define WHITE		"#ffffff"
#define LIGHTGREY       "#d3d3d3"
#define BISQUE1		"#ffe4c4"
#endif

static char *classNames[] =
{
    "Blt_graph", "Blt_barchart", "Blt_piechart"
};

static int configFlags[] =
{
    XYGRAPH_MASK, BARCHART_MASK, PIECHART_MASK
};

#define DEF_BAR_WIDTH		"0.95"
#define DEF_GRAPH_BG_COLOR	BISQUE1
#define DEF_GRAPH_BG_MONO	WHITE
#define DEF_GRAPH_CURSOR  	"crosshair"
#define DEF_GRAPH_FG_COLOR	BLACK
#define DEF_GRAPH_FG_MONO	BLACK
#define DEF_GRAPH_FONT		"*-Helvetica-Bold-R-Normal-*-120-*"
#define DEF_GRAPH_HEIGHT	"400"
#define DEF_GRAPH_WIDTH		"400"
#define DEF_GRAPH_TITLE		(char *)NULL
#define DEF_EXT_BORDER_WIDTH   	(char *)NULL
#define DEF_EXT_BORDER_RELIEF	"flat"
#define DEF_PLOT_BG_COLOR	LIGHTGREY
#define DEF_PLOT_BG_MONO	WHITE
#define DEF_INT_RELIEF_COLOR	"sunken"
#define DEF_INT_RELIEF_MONO	"flat"
#define DEF_INT_BW_MONO       	(char *)NULL
#define DEF_INT_BW_COLOR      	"2"
#define DEF_INT_BG_MONO	 	WHITE
#define DEF_INT_BG_COLOR       	LIGHTGREY

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_GRAPH_BG_COLOR, Tk_Offset(Graph, border),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_GRAPH_BG_MONO, Tk_Offset(Graph, border),
	TK_CONFIG_MONO_ONLY | ALL_MASK},
    {TK_CONFIG_DOUBLE, "-barwidth", "barWidth", "BarWidth",
	DEF_BAR_WIDTH, Tk_Offset(Graph, barWidth), BARCHART_MASK},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL, (char *)NULL, 0,
	ALL_MASK},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0,
	ALL_MASK},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	(char *)NULL, Tk_Offset(Graph, extBWidth), ALL_MASK},
    {TK_CONFIG_PIXELS, "-bottommargin", "bottomMargin", "Margin",
	(char *)NULL, Tk_Offset(Graph, bottomMargin), ALL_MASK},
    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_GRAPH_CURSOR, Tk_Offset(Graph, cursor),
	ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_FONT, "-font", "font", "Font",
	DEF_GRAPH_FONT, Tk_Offset(Graph, fontPtr), ALL_MASK},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *)NULL, (char *)NULL, 0,
	ALL_MASK},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_GRAPH_FG_COLOR, Tk_Offset(Graph, fgColorPtr),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_GRAPH_FG_MONO, Tk_Offset(Graph, fgColorPtr),
	TK_CONFIG_MONO_ONLY | ALL_MASK},
    {TK_CONFIG_PIXELS, "-height", "height", "Height",
	DEF_GRAPH_HEIGHT, Tk_Offset(Graph, reqHeight), ALL_MASK},
    {TK_CONFIG_PIXELS, "-leftmargin", "leftMargin", "Margin",
	(char *)NULL, Tk_Offset(Graph, leftMargin), ALL_MASK},
    {TK_CONFIG_COLOR, "-plotbackground", "plotBackground", "Background",
	DEF_PLOT_BG_MONO, Tk_Offset(Graph, intBgColorPtr),
	TK_CONFIG_MONO_ONLY | ALL_MASK},
    {TK_CONFIG_COLOR, "-plotbackground", "plotBackground", "Background",
	DEF_PLOT_BG_COLOR, Tk_Offset(Graph, intBgColorPtr),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_INT, "-plotborderwidth", "plotBorderWidth", "BorderWidth",
	DEF_INT_BW_COLOR, Tk_Offset(Graph, intBWidth),
	ALL_MASK | TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_INT, "-plotborderwidth", "plotBorderWidth", "BorderWidth",
	(char *)NULL, Tk_Offset(Graph, intBWidth),
	ALL_MASK | TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_RELIEF, "-plotrelief", "plotRelief", "Relief",
	DEF_INT_RELIEF_MONO, Tk_Offset(Graph, intRelief),
	TK_CONFIG_MONO_ONLY | TK_CONFIG_DONT_SET_DEFAULT | ALL_MASK},
    {TK_CONFIG_RELIEF, "-plotrelief", "plotRelief", "Relief",
	DEF_INT_RELIEF_COLOR, Tk_Offset(Graph, intRelief),
	TK_CONFIG_COLOR_ONLY | ALL_MASK},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_EXT_BORDER_RELIEF, Tk_Offset(Graph, extRelief), ALL_MASK},
    {TK_CONFIG_PIXELS, "-rightmargin", "rightMargin", "Margin",
	(char *)NULL, Tk_Offset(Graph, rightMargin), ALL_MASK},
    {TK_CONFIG_STRING, "-title", "title", "Title",
	(char *)NULL, Tk_Offset(Graph, title),
	ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-topmargin", "topMargin", "Margin",
	(char *)NULL, Tk_Offset(Graph, topMargin), ALL_MASK},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_GRAPH_WIDTH, Tk_Offset(Graph, reqWidth), ALL_MASK},
    {TK_CONFIG_STRING, "-xtitle", "xTitle", "Title",
	"X", Tk_Offset(Graph, xTitle), ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_STRING, "-ytitle", "yTitle", "Title",
	"Y", Tk_Offset(Graph, yTitle), ALL_MASK | TK_CONFIG_NULL_OK},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

double Blt_negInfinity, Blt_posInfinity;

extern int Blt_CreateAxis _ANSI_ARGS_((Graph *, AxisType, int));
extern int Blt_AxisCmd _ANSI_ARGS_((Graph *, GraphAxis *, int, char **, int));
extern void Blt_ComputeAxes _ANSI_ARGS_((Graph *));

extern int Blt_LegendCmd _ANSI_ARGS_((Graph *, int, char **));
extern int Blt_CreateLegend _ANSI_ARGS_((Graph *));

extern int Blt_CreatePostScript _ANSI_ARGS_((Graph *));

extern int Blt_CreateCrosshairs _ANSI_ARGS_((Graph *));
extern int Blt_CrosshairsCmd _ANSI_ARGS_((Graph *, int, char **));

static void DisplayGraph _ANSI_ARGS_((ClientData));
static void DestroyGraph _ANSI_ARGS_((ClientData));
static int GraphWidgetCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int,
	char **));
extern int Blt_TagCmd _ANSI_ARGS_((Graph *, int, char **));
extern int Blt_ElementCmd _ANSI_ARGS_((Graph *, int, char **));
extern int Blt_FindCmd _ANSI_ARGS_((Tcl_Interp *, char *, ClientData *));
extern char *sys_errlist[];

/*
 *--------------------------------------------------------------
 *
 * Blt_EventuallyRedraw --
 *
 *	Tell the Tk dispatcher to call the graph display routine at
 *	the next idle point.  This request is made only if the window
 *	is mapped and no other redraw request is pending.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The window is eventually redisplayed.
 *
 *--------------------------------------------------------------
 */
void
Blt_EventuallyRedraw(graphPtr)
    Graph *graphPtr;		/* Graph widget record */
{
    if (Tk_IsMapped(graphPtr->tkwin) && !(graphPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(DisplayGraph, (ClientData)graphPtr);
	graphPtr->flags |= REDRAW_PENDING;
    }
}

/*
 *--------------------------------------------------------------
 *
 * GraphEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on graphs.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get cleaned
 *	up.  When it gets exposed, the graph is eventually redisplayed.
 *
 *--------------------------------------------------------------
 */
static void
GraphEventProc(clientData, eventPtr)
    ClientData clientData;	/* Graph widget record */
    register XEvent *eventPtr;	/* Event which triggered call to routine */
{
    register Graph *graphPtr = (Graph *)clientData;

    if (eventPtr->type == Expose) {
	if (eventPtr->xexpose.count == 0) {
	    graphPtr->flags |= REDRAW_ALL;
	    Blt_EventuallyRedraw(graphPtr);
	}
    } else if (eventPtr->type == DestroyNotify) {
	Tcl_DeleteCommand(graphPtr->interp, Tk_PathName(graphPtr->tkwin));
	graphPtr->tkwin = NULL;
	if (graphPtr->flags & REDRAW_PENDING) {
	    Tk_CancelIdleCall(DisplayGraph, (ClientData)graphPtr);
	}
	Tk_EventuallyFree((ClientData)graphPtr, DestroyGraph);
    } else if (eventPtr->type == ConfigureNotify) {
	graphPtr->flags |= (LAYOUT_PENDING | LAYOUT_ALL | REDRAW_ALL);
	Blt_EventuallyRedraw(graphPtr);
    }
}

/*
 *--------------------------------------------------------------
 *
 * LocatePointer --
 *
 *	This procedure returns a list of the graph coordinate
 *	values corresponding with the given window X and Y
 *	coordinate positions.
 *
 * Results:
 *	Returns a standard Tcl result.  The interp->result field is
 *	a Tcl list of the corresponding graph X and Y coordinates.
 *	If an error occurred while parsing the window positions,
 *	TCL_ERROR is returned, and interp->result will contain
 *	the error message.
 *
 *--------------------------------------------------------------
 */
static int
LocatePointer(graphPtr, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    int argc;
    char **argv;
{
    int winX, winY;		/* Integer window coordinates representation */
    char string[200];

    if (argc != 4) {
	Tcl_AppendResult(graphPtr->interp, "wrong # args: should be \"",
	    argv[0], " locate winX winY\"", NULL);
	return TCL_ERROR;
    }
    if (Tcl_GetInt(graphPtr->interp, argv[2], &winX) != TCL_OK ||
	Tcl_GetInt(graphPtr->interp, argv[3], &winY) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Perform the reverse transformation from window coordinates to data
     */
    Tcl_PrintDouble(graphPtr->interp, Blt_WorldX(graphPtr, winX), string);
    Tcl_AppendElement(graphPtr->interp, string);
    Tcl_PrintDouble(graphPtr->interp, Blt_WorldY(graphPtr, winY), string);
    Tcl_AppendElement(graphPtr->interp, string);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyGraph --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of a graph at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the widget is freed up.
 *
 *----------------------------------------------------------------------
 */
static void
DestroyGraph(clientData)
    ClientData clientData;
{
    register Graph *graphPtr = (Graph *)clientData;
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch cursor;
    Element *elemPtr;
    Tag *tagPtr;

    /*
     * Destroy the individual components of the graph: elements, tags, X and Y
     * axes, legend, display lists etc.
     */
    for (entryPtr = Tcl_FirstHashEntry(&(graphPtr->elemTable), &cursor);
	entryPtr != NULL; entryPtr = Tcl_NextHashEntry(&cursor)) {
	elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
	(*elemPtr->destroyProc) (graphPtr, elemPtr);
    }
    Tcl_DeleteHashTable(&(graphPtr->elemTable));
    Blt_ClearList(&(graphPtr->elemList));
    for (entryPtr = Tcl_FirstHashEntry(&(graphPtr->tagTable), &cursor);
	entryPtr != NULL; entryPtr = Tcl_NextHashEntry(&cursor)) {
	tagPtr = (Tag *)Tcl_GetHashValue(entryPtr);
	(*tagPtr->destroyProc) (graphPtr, tagPtr);
    }
    Tcl_DeleteHashTable(&(graphPtr->tagTable));
    Blt_ClearList(&(graphPtr->tagList));

    (*graphPtr->X->destroyProc) (graphPtr, graphPtr->X);
    (*graphPtr->Y->destroyProc) (graphPtr, graphPtr->Y);
    (*graphPtr->legendPtr->destroyProc) (graphPtr);
    (*graphPtr->postscript->destroyProc) (graphPtr);
    (*graphPtr->crosshairs->destroyProc) (graphPtr);
    /*
     * Release allocated X resources and memory.
     */
    if (graphPtr->extGC != NULL) {
	Tk_FreeGC(graphPtr->display, graphPtr->extGC);
    }
    if (graphPtr->extBgGC != NULL) {
	Tk_FreeGC(graphPtr->display, graphPtr->extBgGC);
    }
    Tk_FreeOptions(configSpecs, (char *)graphPtr, graphPtr->display,
	configFlags[(int)graphPtr->type]);
    free((char *)graphPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * CreateGraph --
 *
 *	This procedure creates and initializes a new widget.
 *
 * Results:
 *	The return value is a pointer to a structure describing
 *	the new widget.  If an error occurred, then the return
 *	value is NULL and an error message is left in interp->result.
 *
 * Side effects:
 *	Memory is allocated, a Tk_Window is created, etc.
 *
 *----------------------------------------------------------------------
 */

static Graph *
CreateGraph(interp, parent, pathName, type)
    Tcl_Interp *interp;
    Tk_Window parent;
    char *pathName;
    GraphClassType type;
{
    register Graph *graphPtr;
    Tk_Window tkwin;
    int flags = configFlags[(int)type];

    tkwin = Tk_CreateWindowFromPath(interp, parent, pathName, (char *)NULL);
    if (tkwin == (Tk_Window)NULL) {
	return (Graph *) NULL;
    }
    graphPtr = (Graph *)calloc(1, sizeof(Graph));

    if (graphPtr == (Graph *)NULL) {
	interp->result = "Can't allocate graph structure";
	return (Graph *) NULL;
    }
    Tk_SetClass(tkwin, classNames[(int)type]);

    /*
     * Initialize the data structure for the graph.
     */
    graphPtr->tkwin = tkwin;
    graphPtr->pathName = Tk_PathName(tkwin);
    graphPtr->display = Tk_Display(tkwin);
    graphPtr->interp = interp;
    graphPtr->type = type;

    Tcl_InitHashTable(&(graphPtr->elemTable), TCL_STRING_KEYS);
    Blt_InitLinkedList(&(graphPtr->elemList), TCL_STRING_KEYS);
    Tcl_InitHashTable(&(graphPtr->tagTable), TCL_STRING_KEYS);
    Blt_InitLinkedList(&(graphPtr->tagList), TCL_STRING_KEYS);

    graphPtr->nextTagId = 1;
    /*
     * Axes and tags (text, bitmap) need private GCs, so create the window now.
     */
    Tk_MakeWindowExist(tkwin);
    if (Blt_CreateAxis(graphPtr, X_AXIS_TYPE, flags) != TCL_OK) {
	goto error;
    }
    if (Blt_CreateAxis(graphPtr, Y_AXIS_TYPE, flags) != TCL_OK) {
	goto error;
    }
    if (Blt_CreateCrosshairs(graphPtr) != TCL_OK) {
	goto error;
    }
    if (Blt_CreateLegend(graphPtr) != TCL_OK) {
	goto error;
    }
    if (Blt_CreatePostScript(graphPtr) != TCL_OK) {
	goto error;
    }
    /* Compute the initial axis ticks and labels */
    Blt_ComputeAxes(graphPtr);
    Tk_CreateEventHandler(graphPtr->tkwin, ExposureMask | StructureNotifyMask,
	GraphEventProc, (ClientData)graphPtr);
    Tcl_CreateCommand(interp, Tk_PathName(graphPtr->tkwin), GraphWidgetCmd,
	(ClientData)graphPtr, (Tcl_CmdDeleteProc *)NULL);
    return (graphPtr);

  error:
    if (tkwin != (Tk_Window)NULL) {
	Tk_DestroyWindow(tkwin);
    }
    return (Graph *) NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureGraph --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a graph widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for graphPtr;  old resources get freed, if there
 *	were any.  The graph is redisplayed.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureGraph(graphPtr, argc, argv, flags)
    register Graph *graphPtr;	/* Graph widget record */
    int argc;			/* Number of configuration arguments */
    char **argv;		/* Configuration arguments */
    int flags;			/* Configuration flags */
{
    XColor *borderColor;
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    if (Tk_ConfigureWidget(graphPtr->interp, graphPtr->tkwin, configSpecs,
	    argc, argv, (char *)graphPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((graphPtr->reqWidth < 1) || (graphPtr->reqHeight < 1)) {
	Tcl_AppendResult(graphPtr->interp,
	    "impossible width/height specifications for \"",
	    Tk_PathName(graphPtr->tkwin), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    Tk_GeometryRequest(graphPtr->tkwin, graphPtr->reqWidth, 
	graphPtr->reqHeight);
    Tk_SetInternalBorder(graphPtr->tkwin, graphPtr->extBWidth);
    Tk_SetBackgroundFromBorder(graphPtr->tkwin, graphPtr->border);

    borderColor = Tk_3DBorderColor(graphPtr->border);

    /*
     * Update background color for axis text GCs
     */
    Blt_UpdateAxisBackgrounds(graphPtr, borderColor);

    /*
     * Create GCs for interior and exterior regions, and a background GC for
     * clearing the margins with XFillRectangle
     */
    gcValues.foreground = graphPtr->fgColorPtr->pixel;
    gcValues.background = borderColor->pixel;
    gcValues.font = graphPtr->fontPtr->fid;
    gcMask = (GCForeground | GCBackground | GCFont);
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (graphPtr->extGC != NULL) {
	Tk_FreeGC(graphPtr->display, graphPtr->extGC);
    }
    graphPtr->extGC = newGC;
    gcValues.foreground = borderColor->pixel;
    gcValues.background = graphPtr->fgColorPtr->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (graphPtr->extBgGC != NULL) {
	Tk_FreeGC(graphPtr->display, graphPtr->extBgGC);
    }
    graphPtr->extBgGC = newGC;
    gcValues.foreground = graphPtr->intBgColorPtr->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (graphPtr->intGC != NULL) {
	Tk_FreeGC(graphPtr->display, graphPtr->intGC);
    }
    graphPtr->intGC = newGC;

    /*
     * Title font, title strings, or margins may have changed layout
     */
    graphPtr->flags |= REDRAW_ALL | LAYOUT_PENDING | LAYOUT_ALL;
    Blt_EventuallyRedraw(graphPtr);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GraphCmd --
 *
 *	Creates a new window and Tcl command representing an
 *	instance of a graph widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static int
GraphCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tk_Window tkwin = (Tk_Window)clientData;
    register Graph *graphPtr;
    GraphClassType type;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"", argv[0],
	    " pathName ?options?\"", (char *)NULL);
	return TCL_ERROR;
    }
    if (strcmp(argv[0], "blt_graph") == 0) {
	type = GRAPH_TYPE;
    } else if (strcmp(argv[0], "blt_barchart") == 0) {
	type = BARCHART_TYPE;
    } else if (strcmp(argv[0], "blt_contour") == 0) {
	type = CONTOUR_TYPE;
    } else if (strcmp(argv[0], "blt_piechart") == 0) {
	type = PIECHART_TYPE;
    } else {
	Tcl_AppendResult(interp, "unknown graph-creation command \"", argv[0],
	    "\"", (char *)NULL);
	return TCL_ERROR;
    }
    /*
     * HUGE_VAL may be a function instead of a numeric constant.
     */
    Blt_negInfinity = -HUGE_VAL;
    Blt_posInfinity = HUGE_VAL;

    graphPtr = CreateGraph(interp, tkwin, argv[1], type);
    if (graphPtr == NULL) {
	return TCL_ERROR;
    }
    if (ConfigureGraph(graphPtr, argc - 2,
	    argv + 2, configFlags[(int)type]) != TCL_OK) {
	Tk_DestroyWindow(graphPtr->tkwin);
	return TCL_ERROR;
    }
    interp->result = graphPtr->pathName;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * GraphWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static int
GraphWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register Graph *graphPtr = (Graph *)clientData;
    int result = TCL_ERROR;
    Tk_Window tkwin = graphPtr->tkwin;
    char c;
    int length;
    int flags;
    GraphClassType type;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
	    " option ?arg arg ...?\"", (char *)NULL);
	return TCL_ERROR;
    }
    Tk_Preserve((ClientData)graphPtr);

    c = argv[1][0];
    length = strlen(argv[1]);
    type = graphPtr->type;
    flags = configFlags[(int)type] | TK_CONFIG_ARGV_ONLY;

    if ((c == 'c') && (length > 2) &&
	(strncmp(argv[1], "configure", length) == 0)) {
	if (argc == 2)
	    result = Tk_ConfigureInfo(interp, tkwin, configSpecs,
		(char *)graphPtr, (char *)NULL, flags);
	else if (argc == 3)
	    result = Tk_ConfigureInfo(interp, tkwin, configSpecs,
		(char *)graphPtr, argv[2], flags);
	else {
	    result = ConfigureGraph(graphPtr, argc - 2, argv + 2, flags);
	}
    } else if ((c == 'c') && (length > 1) &&
	(strncmp(argv[1], "crosshairs", length) == 0)) {
	result = Blt_CrosshairsCmd(graphPtr, argc, argv);
    } else if ((c == 'e') && (strncmp(argv[1], "element", length) == 0)) {
	result = Blt_ElementCmd(graphPtr, argc, argv);
    } else if ((c == 't') && (strncmp(argv[1], "tag", length) == 0)) {
	result = Blt_TagCmd(graphPtr, argc, argv);
    } else if ((c == 'x') && (strncmp(argv[1], "xaxis", length) == 0)) {
	result = Blt_AxisCmd(graphPtr, graphPtr->X, argc, argv, flags);
    } else if ((c == 'y') && (strncmp(argv[1], "yaxis", length) == 0)) {
	result = Blt_AxisCmd(graphPtr, graphPtr->Y, argc, argv, flags);
    } else if ((c == 'l') && (length > 1) &&
	(strncmp(argv[1], "legend", length) == 0)) {
	result = Blt_LegendCmd(graphPtr, argc, argv);
    } else if ((c == 'l') && (length > 1) &&
	(strncmp(argv[1], "locate", length) == 0)) {
	result = LocatePointer(graphPtr, argc, argv);
    } else if ((c == 'p') && (strncmp(argv[1], "postscript", length) == 0)) {
	result = (*graphPtr->postscript->printProc)(graphPtr, argc, argv);
    } else if ((c == 'p') && (length > 1) &&
	(strncmp(argv[1], "psconfigure", length) == 0)) {
	result = (*graphPtr->postscript->configProc)(graphPtr, argc, argv);
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1], "\": should be\
 configure, crosshairs, element, legend, locate, postscript, psconfigure, tag,\
 xaxis, or yaxis", (char *)NULL);
    }
    Tk_Release((ClientData)graphPtr);
    return result;
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_LayoutGraph --
 *
 *	Based upon the computes sizing of the plotting area, calls
 *	routines to calculate the positions for the graph components.
 *	This routine is called when the layout changes.  It is called
 *	from DisplayGraph and PrintPostScript.
 *
 * Results:
 *	None
 *
 * Side Effects:
 *	Old storage is freed and new memory is allocated for holding
 *	component positions.  These positions will eventually affect
 *	where the components are drawn in the graph window.
 *
 * -----------------------------------------------------------------
 */
void
Blt_LayoutGraph(graphPtr)
    Graph *graphPtr;
{
    Blt_ListEntry *entryPtr;
    Element *elemPtr;
    Tag *tagPtr;

    /*
     * Layout both X and Y axis
     */
    if (graphPtr->flags & LAYOUT_ALL) {
	(*graphPtr->X->layoutProc) (graphPtr);
	(*graphPtr->Y->layoutProc) (graphPtr);
    }
    /*
     * Layout elements
     */
    for (entryPtr = Blt_FirstListEntry(&(graphPtr->elemList));
	entryPtr != NULL; entryPtr = Blt_NextListEntry(entryPtr)) {
	elemPtr = (Element *)Blt_GetListValue(entryPtr);
	if ((graphPtr->flags & LAYOUT_ALL) ||
	    (elemPtr->flags & LAYOUT_PENDING)) {
	    (*elemPtr->layoutProc) (graphPtr, elemPtr);
	    elemPtr->flags &= ~LAYOUT_PENDING;
	}
    }

    /*
     * Layout tags
     */
    for (entryPtr = Blt_FirstListEntry(&(graphPtr->tagList));
	entryPtr != NULL; entryPtr = Blt_NextListEntry(entryPtr)) {
	tagPtr = (Tag *)Blt_GetListValue(entryPtr);
	if ((graphPtr->flags & LAYOUT_ALL) ||
	    (tagPtr->flags & LAYOUT_PENDING)) {
	    (*tagPtr->layoutProc) (graphPtr, tagPtr);
	    tagPtr->flags &= ~LAYOUT_PENDING;
	}
    }
    graphPtr->flags &= ~(LAYOUT_ALL | LAYOUT_PENDING);
}

/*
 * -----------------------------------------------------------------
 *
 * DisplayElements --
 *
 *	Calls the individual element drawing routines for each
 *	element.
 *
 * Results:
 *	None
 *
 * Side Effects:
 *	Elements are drawn into the drawable (pixmap) which will
 *	eventually be displayed in the graph window.
 *
 * -----------------------------------------------------------------
 */
static void
DisplayElements(graphPtr)
    Graph *graphPtr;
{
    Blt_ListEntry *entryPtr;
    register Element *elemPtr;

    for (entryPtr = Blt_FirstListEntry(&(graphPtr->elemList));
	entryPtr != NULL; entryPtr = Blt_NextListEntry(entryPtr)) {
	elemPtr = (Element *)Blt_GetListValue(entryPtr);
	(*elemPtr->displayProc) (graphPtr, elemPtr);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * DisplayTags --
 *
 *	Calls the individual tag drawing routines for each tag.
 *	A tag will not be drawn if there is an element associated
 *	with the tag (elemId is non-NULL) and that element is
 *	not in the element display list.
 *
 * Results:
 *	None
 *
 * Side Effects:
 *	Tags are drawn into the drawable (pixmap) which will eventually
 *	be displayed in the graph window.
 *
 * -----------------------------------------------------------------
 */
static void
DisplayTags(graphPtr)
    Graph *graphPtr;
{
    Blt_ListEntry *listPtr;
    Tag *tagPtr;
    int mapped;

    for (listPtr = Blt_FirstListEntry(&(graphPtr->tagList)); listPtr != NULL;
	listPtr = Blt_NextListEntry(listPtr)) {
	tagPtr = (Tag *)Blt_GetListValue(listPtr);
	mapped = TRUE;

	if (tagPtr->elemId != NULL) {
	    Tcl_HashEntry *entryPtr;
	    Element *elemPtr;

	    entryPtr = Tcl_FindHashEntry(&(graphPtr->elemTable),
		(char *)tagPtr->elemId);
	    if (entryPtr == NULL) {
		continue;
	    }
	    elemPtr = (Element *)Tcl_GetHashValue(entryPtr);
	    mapped = elemPtr->mapped;
	}
	if (mapped) {
	    (*tagPtr->displayProc) (graphPtr, tagPtr);
	}
    }
}

/*
 * -----------------------------------------------------------------
 *
 * DisplayExterior --
 *
 * 	Draws the exterior region of the graph (axes, ticks, titles, etc)
 *	onto a pixmap. The interior region is defined by the given 
 *	rectangle structure. 
 *
 *		X coordinate axis
 *		Y coordinate axis
 *		legend
 *		interior border
 *		exterior border
 *		titles (X and Y axis, graph)
 *
 * Returns:
 *	None.
 * 
 * Side Effects:
 *	Exterior of graph is displayed in its window.
 *
 * -----------------------------------------------------------------
 */
static void
DisplayExterior(graphPtr, rectPtr)
    Graph *graphPtr;
    XRectangle *rectPtr;	/* Interior plotting region */
{
    int x, y;
    XRectangle rectArr[4];
    TextAttr textAttr;
    int titlePos;

    /*
     * Draw the four outer rectangles which encompass the plotting
     * surface. This clears the surrounding area and clips the plot.
     */

    rectArr[0].x = rectArr[0].y = rectArr[3].x = rectArr[1].x = 0;
    rectArr[0].width = rectArr[3].width = Tk_Width(graphPtr->tkwin);
    rectArr[0].height = rectPtr->y;
    rectArr[3].y = rectPtr->y + rectPtr->height;
    rectArr[3].height = Tk_Height(graphPtr->tkwin) - rectArr[3].y;
    rectArr[2].y = rectArr[1].y = rectPtr->y;
    rectArr[1].width = rectPtr->x;
    rectArr[2].height = rectArr[1].height = rectPtr->height;
    rectArr[2].x = rectPtr->x + rectPtr->width;
    rectArr[2].width = Tk_Width(graphPtr->tkwin) - rectArr[2].x;
    XFillRectangles(graphPtr->display, graphPtr->canvas, graphPtr->extBgGC,
	rectArr, 4);

    /* Interior 3D border */
    if ((graphPtr->intRelief != TK_RELIEF_FLAT) && (graphPtr->intBWidth > 0)) {
	Tk_Draw3DRectangle(graphPtr->display, graphPtr->canvas,
	    graphPtr->border, rectPtr->x, rectPtr->y, rectPtr->width,
	    rectPtr->height, graphPtr->intBWidth, graphPtr->intRelief);
    }
    /* Legend */
    if (graphPtr->legendPtr->position.x == DEF_POSITION) {
	(*graphPtr->legendPtr->displayProc) (graphPtr);
    }
    /* Graph and Axis titles */
    textAttr.theta = 0.0;
    textAttr.anchor = TK_ANCHOR_CENTER;
    textAttr.gc = graphPtr->extGC;
    textAttr.bgColorPtr = Tk_3DBorderColor(graphPtr->border);
    textAttr.fontPtr = graphPtr->fontPtr;
    titlePos = graphPtr->extBWidth + TEXTHEIGHT(graphPtr->fontPtr);

    x = (graphPtr->extreme.x + graphPtr->origin.x) / 2;
    if (graphPtr->title != NULL) {
	Blt_DrawText(graphPtr->display, graphPtr->canvas, graphPtr->title,
	    &textAttr, x, titlePos);
    }
    if (graphPtr->xTitle != NULL) {
	Blt_DrawText(graphPtr->display, graphPtr->canvas, graphPtr->xTitle,
	    &textAttr, x, Tk_Height(graphPtr->tkwin) - titlePos);
    }
    if (graphPtr->yTitle != NULL) {
	y = (graphPtr->origin.y + graphPtr->extreme.y) / 2;
	textAttr.theta = 90.0;
	Blt_DrawText(graphPtr->display, graphPtr->canvas, graphPtr->yTitle,
	    &textAttr, titlePos, y);
    }
    /* X and Y Axes */
    (*graphPtr->X->displayProc) (graphPtr);
    (*graphPtr->Y->displayProc) (graphPtr);

    /* Exterior 3D border */
    if ((graphPtr->extRelief != TK_RELIEF_FLAT) &&
	(graphPtr->extBWidth > 0)) {
	Tk_Draw3DRectangle(graphPtr->display, graphPtr->canvas,
	    graphPtr->border, 0, 0, Tk_Width(graphPtr->tkwin), 
	    Tk_Height(graphPtr->tkwin), graphPtr->extBWidth, 
	    graphPtr->extRelief);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * DisplayGraph --
 *
 *	This procedure is invoked to display a graph widget.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Commands are output to X to display the graph in its
 *	current mode.
 *
 *----------------------------------------------------------------------
 */
static void
DisplayGraph(clientData)
    ClientData clientData;
{
    register Graph *graphPtr = (Graph *)clientData;
    XRectangle plotRect;
    int twiceBW;
    GraphLegend *legendPtr;

    /* If the window has been deleted or is no longer mapped, do nothing */
    graphPtr->flags &= ~REDRAW_PENDING;
    if ((graphPtr->tkwin == NULL) || !Tk_IsMapped(graphPtr->tkwin)) {
	return;
    }
#ifdef notdef
    fprintf(stderr, "Calling DisplayGraph\n");
#endif

    graphPtr->width = Tk_Width(graphPtr->tkwin);
    graphPtr->height = Tk_Height(graphPtr->tkwin);

    if (graphPtr->flags & LAYOUT_PENDING) {
	if (Blt_ComputeLayout(graphPtr) != TCL_OK) {
	    return;		/* Not enough room to plot graph */
	}
	Blt_LayoutGraph(graphPtr);
    }
    twiceBW = (graphPtr->intBWidth * 2);
    /*
     * Determine the plotting surface region of the graph window
     */
    plotRect.x = graphPtr->origin.x - graphPtr->intBWidth;
    plotRect.y = graphPtr->extreme.y - graphPtr->intBWidth;
    plotRect.width = (graphPtr->extreme.x - graphPtr->origin.x) + twiceBW;
    plotRect.height = (graphPtr->origin.y - graphPtr->extreme.y) + twiceBW;

    /*
     * Create a canvas the size of the window, filling it with the bg color
     */
    graphPtr->canvas = XCreatePixmap(graphPtr->display,
	Tk_WindowId(graphPtr->tkwin), Tk_Width(graphPtr->tkwin), 
	Tk_Height(graphPtr->tkwin), Tk_Depth(graphPtr->tkwin));
    XFillRectangles(graphPtr->display, graphPtr->canvas, graphPtr->intGC,
	&plotRect, 1);
    legendPtr = graphPtr->legendPtr;
    if (legendPtr->position.x != DEF_POSITION) {
	(*legendPtr->displayProc) (graphPtr);
    }
    /* Draw the graph elements and tags on the canvas */
    if (graphPtr->type == GRAPH_TYPE) {
	DisplayTags(graphPtr);
	DisplayElements(graphPtr);
    } else {
	DisplayElements(graphPtr);
	DisplayTags(graphPtr);
    }

    /* Disable crosshairs and update lengths before drawing */
    (*graphPtr->crosshairs->toggleProc) (graphPtr);
    (*graphPtr->crosshairs->updateProc) (graphPtr);

    if (graphPtr->flags & REDRAW_ALL) {
	DisplayExterior(graphPtr, &plotRect);
	XCopyArea(graphPtr->display, graphPtr->canvas,
	    Tk_WindowId(graphPtr->tkwin), graphPtr->extGC, 0, 0,
	    Tk_Width(graphPtr->tkwin), Tk_Height(graphPtr->tkwin), 0, 0);
    } else {
	/* Draw just the interior plotting region */

	plotRect.x += graphPtr->intBWidth;
	plotRect.y += graphPtr->intBWidth;
	plotRect.width -= (graphPtr->intBWidth * 2);
	plotRect.height -= (graphPtr->intBWidth * 2);
	XCopyArea(graphPtr->display, graphPtr->canvas,
	    Tk_WindowId(graphPtr->tkwin), graphPtr->extGC, plotRect.x, 
	    plotRect.y, (unsigned int)plotRect.width, 
	    (unsigned int)plotRect.height, plotRect.x, plotRect.y);
    }
    (*graphPtr->crosshairs->toggleProc) (graphPtr);

    XFreePixmap(graphPtr->display, graphPtr->canvas);
    graphPtr->flags &= ~REDRAW_ALL;
}

void
Blt_GraphInit(interp)
    Tcl_Interp *interp;
{
    Tk_Window tkwin;

    if (Blt_FindCmd(interp, "blt_graph", (ClientData *)NULL) == TCL_OK) {
	return;			/* Command already exists */
    }
    tkwin = Tk_MainWindow(interp);
    if (tkwin == NULL) {
	return;			/* Not a Tk application */
    }
    Tcl_SetVar2(interp, "blt_versions", "blt_graph", GRAPH_VERSION,
	TCL_GLOBAL_ONLY);
    Tcl_CreateCommand(interp, "blt_graph", GraphCmd, (ClientData)tkwin,
	(Tcl_CmdDeleteProc *)NULL);
}

void
Blt_BarchartInit(interp)
    Tcl_Interp *interp;
{
    Tk_Window tkwin;

    if (Blt_FindCmd(interp, "blt_barchart", (ClientData *)NULL) == TCL_OK) {
	return;			/* Command already exists */
    }
    tkwin = Tk_MainWindow(interp);
    if (tkwin == NULL) {
	return;			/* Not a Tk application */
    }
    Tcl_SetVar2(interp, "blt_versions", "blt_barchart", GRAPH_VERSION,
	TCL_GLOBAL_ONLY);
    Tcl_CreateCommand(interp, "blt_barchart", GraphCmd, (ClientData)tkwin,
	(Tcl_CmdDeleteProc *)NULL);
}
