#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <ctype.h>
#include <unistd.h>
#define __GUS_PATH_APP__
#include "../../include/libgus.h"
#include "../../include/gusfiles.h"
#include GUS_INCLUDE_NCURSES

#define VERSION		"1.02"

static int quiet;
static int verbose;
static int quit;
static int prev;
static int next;
static int time_break;

static int m_format;
static char *m_format_name;
static int m_tracks;
static int m_pause;

static void signal_terminate( int signal )
{
  gus_midiplay_stop( m_pause = 1 );
} 

static void m_select( void *private, fd_set *read_fds, fd_set *write_fds )
{
  if ( FD_ISSET( fileno( stdin ), read_fds ) )
    {
      int key;
      
      for ( key = getch(); key != EOF; key = getch() )
        switch ( tolower( key ) ) {
          case 'q':
            gus_midiplay_stop( quit = m_pause = 1 );
            break;
          case 'p':
            if ( !(m_pause & 1) )
              {
                m_pause ^= 2;
                if ( !quiet )
                  {
                    if ( m_pause & 2 )
                      printf( "Waiting... " );
                     else
                      printf( "\r                                                " );
                    fflush( stdout );
                  }
                gus_midiplay_stop( m_pause );
              }
            break;
          case 'n':
          case ' ':
          case '+':
          case KEY_RIGHT:
            gus_midiplay_stop( next = m_pause = 1 );
            break;
          case '-':
          case KEY_LEFT:
            gus_midiplay_stop( prev = m_pause = 1 );
            break;
        }
    }
}

static void m_info( void *private, int format, char *format_name, int tracks )
{
  m_format = format;
  if ( m_format_name ) free( m_format_name );
  m_format_name = strdup( format_name );
  m_tracks = 0;
  if ( !quiet )
    {
      printf( "\nFormat: %s, Tracks: %i...\n", format_name, tracks );
      fflush( stdout );
    }
}

static void m_download_program( void *private, gus_midi_device_t *mdevice, unsigned int program )
{
#if 0
  printf( "download_program: device = %i, program = %i/%i\n", mdevice -> device, program >> 16, program & 0xffff );
#endif
  if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
    gus_midi_icfg_download_program( mdevice -> device, &program, 1 );
}

static void m_total_time( void *private, unsigned int time )
{
  if ( !quiet )
    {
      printf( "Total time   = %i.%i second%s\n", time / 10, time % 10, time / 10 == 1 ? "" : "s" );
      fflush( stdout );
    }
}

static void m_time( void *private, unsigned int time )
{
  if ( !quiet )
    {
      printf( "\rCurrent time = %i.%i second%s     ", time / 10, time % 10, time / 10 == 1 ? "" : "s" );
      fflush( stdout );
    }
  if ( time_break > 0 && time_break <= time / 10 )
    gus_midiplay_stop( next = m_pause = 1 );
}

static void m_text( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Text: %s\n", track, text );
}

static void m_copyright( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Copyright: %s\n", track, text );
}

static void m_sequence( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Sequence name: %s\n", track, text );
}

static void m_instrument( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Intrument: %s\n", track, text );
}

static void m_lyric( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Lyric: %s\n", track, text );
}

static void m_marker( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Marker: %s\n", track, text );
}

static void m_cuepoint( void *private, int track, char *text )
{
  if ( verbose )
    printf( "-%i- Cue-point: %s\n", track, text );
}

static gus_midiplay_callbacks_t m_callbacks = {
  15,
  NULL,
  0,
  {},
  {},
  m_select,
  m_info,
  m_download_program,
  m_total_time,
  m_time,
  m_text,
  m_copyright,
  m_sequence,
  m_instrument,
  m_lyric,
  m_marker,
  m_cuepoint
};

static void help( void )
{
  printf( "Available switches:\n\n"
  	  "  -h         help\n"
  	  "  -q         quiet mode\n"
  	  "  -f <file>  configuration file (default /usr/local/etc/gus-midi.conf)\n"
  	  "  -i <file>  instrument file (default /usr/local/etc/gus-midi-gf1.conf)\n"
  	  "  -c #       select output card (1-%i)\n"
  	  "  -d <dev>   select output device (synth,uart)\n"
  	  "  -p         use only .pat files as instrument source\n"
  	  "  -1         use 1MB gmfull1m.XXX file as instrument source (or onboard ROM)\n"
  	  "  -4         use 4MB gsfull4m.XXX file as instrument source\n"
  	  "  -u <bank>  use user bank (megabank, utopia or utopia_mono)\n"
  	  "		as instrument source\n"
  	  "  -r <val>   reverb effect type for InterWave (-1 = disable, 0-7, default=4)\n"
  	  "  -s <val>   chorus effect type for InterWave (-1 = disable, 0-7, default=2)\n"
  	  "  -g         Roland GS mode on\n"
  	  "  -m         General MIDI mode on\n"
  	  "  -3         MT-32 mode on\n"
  	  "  -v         verbose mode\n"
  	  "  -t <sec>   scan mode - set timelimit to <sec>\n",
  	  	GUS_CARDS
  );
}

#define SOURCE_DEFAULT		0
#define SOURCE_PATCH		1
#define SOURCE_IW_FILE_1MB	3
#define SOURCE_IW_FILE_4MB	4
#define SOURCE_MEGABANK		5
#define SOURCE_UTOPIA_STEREO	6
#define SOURCE_UTOPIA_MONO	7

int main( int argc, char *argv[] )
{
  int morehelp = 0, card = -1, device = -1;
  int instr_source = SOURCE_DEFAULT;
  int reverb_type = 4, chorus_type = 2;
  gus_midi_device_t *mdevice;
  char *conf_file = NULL;
  char *instr_file = NULL;
  int emul = -1, old_emul = -1;
  int optind_first;

  quiet = 0;
  verbose = 0;
  time_break = 0;
  while ( 1 )
    {
      int c;
    
      if ( ( c = getopt( argc, argv, "14u:hqvc:d:f:i:pr:s:gm3t" ) ) < 0 ) break;
      switch ( c ) {
        case 'h':
          morehelp++;
          break;
        case 'q':
          quiet = 1;
          break;
        case 'v':
          verbose = 1;
          break;
        case 'c':
          card = atoi( optarg );
          if ( card < 0 || card >= GUS_CARDS )
            {
              fprintf( stderr, "card out of range (1-%i)\n", GUS_CARDS );
              morehelp++;
            }
          card--;
          if ( device < 0 ) device = GUS_MIDID_SYNTH;
          break;
        case 'd':
          if ( !strcmp( optarg, "synth" ) ) device = GUS_MIDID_SYNTH; else
          if ( !strcmp( optarg, "uart" ) ) device = GUS_MIDID_UART; else
            {
              fprintf( stderr, "invalid device (synth or uart)\n" );
              morehelp++;
            }
          if ( card < 0 ) card = 0;
          break;
        case 'f':
          free( conf_file );
          conf_file = strdup( optarg );
          break;
        case 'i':
          free( instr_file );
          instr_file = strdup( optarg );
          break;
        case 'p':
          instr_source = SOURCE_PATCH;
          break;
        case '1':
          instr_source = SOURCE_IW_FILE_1MB;
          break;
        case '4':
          instr_source = SOURCE_IW_FILE_4MB;
          break;
        case 'u':
          if ( !strcmp( optarg, "megabank" ) )
            instr_source = SOURCE_MEGABANK;
           else
          if ( !strcmp( optarg, "utopia" ) )
            instr_source = SOURCE_UTOPIA_STEREO;
           else
          if ( !strcmp( optarg, "utopia_mono" ) )
            instr_source = SOURCE_UTOPIA_MONO;
          break;
        case 'r':
          reverb_type = atoi( optarg );
          if ( reverb_type < 0 ) reverb_type = -1;
          if ( reverb_type > 7 )
            {
              fprintf( stderr, "unknown reverb effect type\n" );
              morehelp++;
            }
          break;
        case 's':
          chorus_type = atoi( optarg );
          if ( chorus_type < 0 ) reverb_type = -1;
          if ( chorus_type > 7 )
            {
              fprintf( stderr, "unknown chorus effect type\n" );
              morehelp++;
            }
          break;
        case 'm':
          emul = GUS_MIDI_EMUL_GM;
          break;
        case 'g':
          emul = GUS_MIDI_EMUL_GS;
          break;
        case '3':
          emul = GUS_MIDI_EMUL_MT32;
          break;
        case 't':
          time_break = atoi( optarg );
          if ( time_break < 0 ) time_break = 0;
          break;
        case '?':
          printf( "\07Invalid switch or option needs an argument\n" );
      }          
    }

  if ( optind >= argc || morehelp )
    {
      printf( "Usage: %s [switches] <easy.mid> ... \n", argv[ 0 ] );
      if ( morehelp )
        help();
       else
        printf( "Type %s -h for more help.\n", argv[ 0 ] );
      return -1;
    }

  if ( !quiet )
    {
      /* initialize curses code */
      atexit((void*)endwin);
      initscr();
      cbreak();
      nodelay(stdscr, TRUE);
      noecho();
    }

  signal( SIGINT, signal_terminate );
  signal( SIGTERM, signal_terminate );
      
  if ( !quiet )
    printf( "UltraMidi version " VERSION " by Jaroslav Kysela (Perex soft)\n" );

  if ( verbose ) quiet = 0;	/* ok... why verbose and quiet??? */

  if ( card < 0 || device < 0 )
    device = GUS_MIDID_COMMON;
   else
    device = ( card << 4 ) | device;

  if ( device == GUS_MIDID_COMMON )
    {
      if ( gus_midi_open_intelligent( GUS_MIDI_OUT, conf_file, 8 * 1024, GUS_MIDI_OF_ECHO_ENABLE ) )
        {
          fprintf( stderr, "midi open: %s\n", gus_midi_error );
          return -1;
        }
    }
   else
    {
      mdevice = malloc( sizeof( gus_midi_device_t ) );
      gus_midi_fill_device_structure( mdevice, GUS_MIDI_OPEN_MODE_WRITE, device, 0xffff );
      if ( gus_midi_open( GUS_MIDI_OUT, mdevice, 8 * 1024, GUS_MIDI_OF_ECHO_ENABLE ) )
        {
          fprintf( stderr, "midi open: %s\n", gus_midi_error );
          return -1;
        }
    }
  
  putenv( "GUS_APP_CONFIGURATION=ON" );
  switch ( instr_source ) {
    case SOURCE_PATCH:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_GF1_PATCH_ONLY=ON" );
      break;
    case SOURCE_IW_FILE_1MB:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_USE_IW_1MB_FILE=ON" );
      break;
    case SOURCE_IW_FILE_4MB:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_USE_IW_4MB_FILE=ON" );
      break;
    case SOURCE_MEGABANK:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_USE_MEGABANK=ON" );
      break;
    case SOURCE_UTOPIA_STEREO:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_USE_UTOPIA_STEREO=ON" );
      break;
    case SOURCE_UTOPIA_MONO:
      putenv( "GUS_APP_USER_RESET=ON" );
      putenv( "GUS_APP_USE_UTOPIA_MONO=ON" );
      break;
  }

  for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
    {
      if ( emul >= 0 )
        {
          old_emul = gus_midi_emulation_get( mdevice -> device );
          if ( old_emul != emul )
            gus_midi_emulation_set( mdevice -> device, emul );
           else
            old_emul = -1;
        }
      if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
        {
          gus_info_t ginfo;
          struct GUS_STRU_EFFECT effect;
      
          if ( gus_midi_icfg_open( mdevice -> device, instr_file ) )
            {
              fprintf( stderr, "icfg open error\n" );
              gus_midi_close();
              return -1;
            }
          if ( gus_midi_synth_info( mdevice -> device, &ginfo ) >= 0 )
            if ( ginfo.flags & GUS_STRU_INFO_F_ENHANCED )
              {
                gus_effect_interwave( &effect, reverb_type, chorus_type );
                gus_midi_effect_setup( mdevice -> device, &effect );
              }
       }
    }

  quit = 0;  
  optind_first = optind;
  while ( optind < argc && !quit ) 
    {
      FILE *in;
      long size, rsize;
      unsigned char *song;
      char *song_name;
      struct stat st;
      
      song_name = argv[ optind++ ];
      if ( stat( song_name, &st ) < 0 ) continue;
      if ( ( st.st_mode & S_IFMT ) != S_IFREG ) continue;
      in = fopen( song_name, "rb" );
      if ( !in ) continue;
      size = 0;
      song = malloc( 10 * 1024 );
      if ( !song )
        {
          fprintf( stderr, "Alloc error.\n" );
          return -1;
        }
      while ( 1 )
        {
          rsize = fread( &song[ size ], 1, 10 * 1024, in );
          size += rsize;
          if ( rsize < 10 * 1024 )
            {
              song = realloc( song, size );
              if ( !song )
                {
                  fprintf( stderr, "Alloc error.\n" );
                  return -1;
                }
              break;
            }
           else
            {
              song = realloc( song, size + 10 * 1024 );
              if ( !song )
                {
                  fprintf( stderr, "Alloc error.\n" );
                  return -1;
                }
            }
        }
      fclose( in );

      if ( ( rsize = gus_midiplay_songs( song, size ) ) > 1 )
        printf( "Warning! Midi file '%s' contains %li songs\n", argv[ optind - 1 ], rsize );
      if ( !rsize )
        {
          free( song );
          printf( "File '%s' have unknown format.\n", argv[ optind - 1 ] );
          continue;
        }

      m_callbacks.last_fd = gus_midi_get_handle();
      FD_ZERO( &m_callbacks.read_fds );
      FD_ZERO( &m_callbacks.write_fds );
      if ( in != stdin && !quiet )
        FD_SET( fileno( stdin ), &m_callbacks.read_fds );
      if ( !quiet )
        {
          printf( "Playing song '%s'... ", song_name );
          fflush( stdout );
        }
      m_format_name = NULL;
      m_pause = 0;
      if ( gus_midiplay_song( device, song, size, 0, &m_callbacks ) < 0 )
        fprintf( stderr, "play error\n" );

      for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
        if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
          gus_midi_memory_reset( mdevice -> device );		/* ok.. clean instruments */

      free( m_format_name ); m_format_name = NULL;
      if ( !quiet )
        {
          printf( "\r" );
          if ( next || prev )
            {
              printf( "Skipped..." );
              if ( time_break > 0 )
                printf( " Scan mode..." );
            }
           else
            printf( "Done..." );
          printf( "                              \n" );
          fflush( stdout );
        }
      if ( next ) next = 0;
      if ( prev )
        {
          optind -= 2;
          if ( optind < optind_first ) optind = optind_first;
          prev = 0;
        }
    }

  for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
    {
      if ( mdevice -> cap & GUS_MIDI_CAP_MEMORY )
        gus_midi_icfg_close( mdevice -> device );
      if ( old_emul >= 0 )
        gus_midi_emulation_set( mdevice -> device, old_emul );
    }

  gus_midi_close();
  free( conf_file );
  free( instr_file );
  unsetenv( "GUS_APP_CONFIGURATION" );
  switch ( instr_source ) {
    case SOURCE_PATCH:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_GF1_PATCH_ONLY" );
      break;
    case SOURCE_IW_FILE_1MB:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_USE_IW_1MB_FILE" );
      break;
    case SOURCE_IW_FILE_4MB:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_USE_IW_4MB_FILE" );
      break;
    case SOURCE_MEGABANK:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_USE_MEGABANK" );
      break;
    case SOURCE_UTOPIA_STEREO:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_USE_UTOPIA_STEREO" );
      break;
    case SOURCE_UTOPIA_MONO:
      unsetenv( "GUS_APP_USER_RESET" );
      unsetenv( "GUS_APP_USE_UTOPIA_MONO" );
      break;
  }

  signal( SIGINT, SIG_DFL );
  signal( SIGTERM, SIG_DFL );

  return 0;
}
