/*************************************************************************
 *  ultra_midi.c
 *
 *  Copyright (C) 1995-97 Jaroslav Kysela (perex@pf.jcu.cz)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include "musserver.h"

#undef PRELOAD
#define INSTRUMENT_CONFIG	"/usr/local/etc/gus-midi-gf1.conf"

extern char *waddir;

int have_midi = 0;
int use_synth = 0;
static int main_volume = 8;

void vol_change(int volume);

void reset_midi(void)
{
  int i;

  gus_midi_timer_stop();
  gus_midi_reset();
  for ( i = 0; i < 16; i++ )
    {
      if ( i != 9 )
        gus_midi_control( GUS_MIDID_COMMON, i, GUS_MCTL_E3_CHORUS_DEPTH, 20 );
    }
#ifndef PRELOAD
  gus_midi_memory_reset_all();
#endif
  vol_change( main_volume );
}

void midi_setup(int pref_dev, int num_dev)
{
  char filename[ 257 ];
  char *cfg_filename;
  struct stat info;
  gus_midi_device_t *device;
  gus_info_t ginfo;
  struct GUS_STRU_EFFECT effect;

  have_midi = 0;
  use_synth = 0;
  sprintf( filename, "%s/gus-midi.conf", waddir );
  if ( stat( filename, &info ) == -1 )
    cfg_filename = NULL;
   else
    cfg_filename = filename;
  if ( gus_midi_open_intelligent( GUS_MIDI_OUT, cfg_filename, 1024, GUS_MIDI_OF_ECHO_ENABLE ) < 0 )
    {
      printf( "musserver: midi open error > %s\n", gus_midi_error );
      return;
    }
  have_midi = 1;
  use_synth = 1;
  gus_midi_timer_base( 100 );
  gus_midi_timer_tempo( 128 );
  for ( device = gus_midi_output_devices(); device; device = device -> next )
    if ( device -> cap & GUS_MIDI_CAP_SYNTH )
      {
        if ( gus_midi_icfg_open( device -> device, INSTRUMENT_CONFIG ) < 0 )
          {
            gus_midi_close();
            fprintf( stderr, "musserver: icfg open error" );
            exit(1);
          }
        if ( gus_midi_synth_info( device -> device, &ginfo ) >= 0 )
          if ( ginfo.flags & GUS_STRU_INFO_F_ENHANCED )
            {
              gus_effect_interwave( &effect, GUS_EFFECT_INTERWAVE_1_HALL2, GUS_EFFECT_INTERWAVE_3_CHORUS3 );
              gus_midi_effect_setup( device -> device, &effect );
            }
      }
#ifdef PRELOAD
  gus_midi_preload_bank( GUS_MIDID_COMMON, 0, "ALL" );
#endif
}

void cleanup_midi(void)
{
  gus_midi_device_t *device;

  gus_midi_timer_stop();
  for ( device = gus_midi_output_devices(); device; device = device -> next )
    if ( device -> cap & GUS_MIDI_CAP_SYNTH )
      gus_midi_icfg_close( device -> device );
  gus_midi_close();
}

void note_off(int note, int channel, int volume)
{
  gus_midi_note_off( GUS_MIDID_COMMON, channel, note, volume );
}

void note_on(int note, int channel, int volume)
{
  gus_midi_note_on( GUS_MIDID_COMMON, channel, note, volume );
}

void pitch_bend(int channel, signed int value)
{
  gus_midi_bender( GUS_MIDID_COMMON, channel, value * 64 );
}

void control_change(int controller, int channel, int value)
{
  gus_midi_control( GUS_MIDID_COMMON, channel, controller, value );
}

void patch_change(int patch, int channel)
{
  gus_midi_program_change( GUS_MIDID_COMMON, channel, patch );
}

void midi_wait(float ftime)
{
  static gus_midi_callbacks_t callbacks = {
    17,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
  };
  static int echo_time = 0;
  fd_set read_fds, write_fds;
  int handle = gus_midi_get_handle();
  int time, time1;

  time = ftime;
  while ( time )
    {
      time1 = time;
      if ( time1 > 64 ) time1 = 64;
      time -= time1;
      gus_midi_wait( time1 );
      echo_time += time;
      if ( echo_time > 64 )
        {
          gus_midi_echo( "1234", 4 );
          echo_time -= 64;
        }
    }
  gus_midi_write();
  do {
    FD_ZERO( &read_fds );
    FD_ZERO( &write_fds );
    FD_SET( handle, &read_fds );
    FD_SET( handle, &write_fds );
    select( handle + 1, &read_fds, &write_fds, NULL, NULL );
    if ( FD_ISSET( handle, &read_fds ) )
      gus_midi_input( &callbacks );
  } while ( !FD_ISSET( handle, &write_fds ) );
}

void midi_flush( void )
{
  gus_midi_flush();
}

void midi_timer(int action)
{
  switch (action)
    {
    case 0:
      gus_midi_timer_start();
      break;
    case 1:
      gus_midi_timer_stop();
      break;
    case 2:
      gus_midi_timer_continue();
      break;
    }
}

void vol_change(int volume)
{
  main_volume = volume;
  gus_midi_master_volume( GUS_MIDID_COMMON, volume > 15 ? 127 : volume << 3 );
}

void patch_preload(int patch, int channel)
{
#ifndef PRELOAD
  gus_midi_device_t *card;

  if ( channel == 9 && patch < 128 ) return;	/* error? */
  card = gus_midi_output_devices();
  while ( card )
    {
      if ( (card -> channels & ( 1 << channel )) && (card -> cap & GUS_MIDI_CAP_MEMORY) ) 
        gus_midi_icfg_download_program( card -> device, &patch, 1 );
      card = card -> next;
    }
#endif
}
