/*
 *  GUSMIX
 *
 *  AUTHOR: Jaroslav Kysela <perex@pf.jcu.cz>
 *  CHANGES: Tim Janik <Tim.Janik@Hamburg.Netsurf.DE>
 *           T Taneli Vahakangas <vahakang@cc.helsinki.fi>
 *  BASED ON: aumix 0.2 from Savio Lam <lam836@cs.cuhk.hk>
 *
 *  HISTORY:
 * 
 *  v1.03d
 *     1. 8.1997 - updates for libgus 3.3.0
 *
 *  v1.03c
 *     4. 7.1997 - support for 80+ colums added
 *		 - patch by T Taneli Vahakangas <vahakang@cc.helsinki.fi>
 *
 *  v1.03b
 *    23. 5.1997 - check for 25 lines added
 *
 *  v1.03a
 *    24. 1.1997 - support for Effect volume added
 *
 *  v1.02b
 *    20.12.1996 - support for InterWave serial transfer added
 *
 *  v1.02a
 *     9.12.1996 - patch by Tim Janik
 *		   - more keycode processing
 *		   - fixed few size/spelling bugs
 *		   - added numeric values
 *
 *  v1.01e
 *    30.11.1996 - support for card identification
 *
 *  v1.01d
 *     5. 8.1996 - support for loopback (CODEC) added
 *
 *  v1.01c
 *    17. 3.1996 - support for GUS PnP mixer added
 *
 *  v1.01b
 *    19.12.1995 - support for more cards added (-c switch)
 *
 *  v1.01a
 *    25.11.1995 - signal handler added for clean abort (Ctrl-C)
 *		 - mute function added
 *
 *  v1.00a
 *    14.10.1995 - first release
 *
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/signal.h>
#define __GUS_PATH_APP__
#include "../../include/libgus.h"
#include "../../include/gusfiles.h"
#include GUS_INCLUDE_NCURSES

#define VERSION			"1.03c"

/* colors */

#define TITLE1_COLOR		1
#define TITLE2_COLOR		2
#define TITLE3_COLOR		3
#define BORDER_COLOR		4
#define RECSRC_COLOR		5
#define ACTIVE_RECSRC_COLOR	6
#define BAR1_COLOR		7
#define BAR2_COLOR		8
#define BAR3_COLOR		9
#define BAR_LABEL_COLOR		10
#define BAR_FRAME_COLOR		11
#define SELECT_COLOR		12
#define HELP_BORDER_COLOR	13
#define HELP_TITLE_COLOR	14
#define HELP_HOTKEY_COLOR	15
#define HELP_TEXT_COLOR		16

void initColors( void );

/*
 *  Variables
 */

WINDOW *w_panel;

int colormode = 0;	/* 0 = no color, 1 = color */
int bars = 0;
int mversion = 0, mmaskversion = 0;
int devmask = 0, recmask = 0, recsrc = 0, stereodevs = 0;

struct channelInfo {
  char *name;
  int id;
  int mask;
  int (*read)( void );
  int (*write)( int );
} ichannels[] = {
  { "Master",	GUS_MIXER_ID_MASTER,	GUS_MIXER_MASK_MASTER,
		gus_mixer_read_volume,	gus_mixer_write_volume	},
  { "Synth ",	GUS_MIXER_ID_SYNTH,	GUS_MIXER_MASK_SYNTH,
		gus_mixer_read_synth,	gus_mixer_write_synth	},
  { "Effect",	GUS_MIXER_ID_EFFECT,	GUS_MIXER_MASK_EFFECT,
		gus_mixer_read_effect,	gus_mixer_write_effect },
  { " PCM  ",	GUS_MIXER_ID_PCM,	GUS_MIXER_MASK_PCM,
		gus_mixer_read_pcm,	gus_mixer_write_pcm	},
  { " Line ",	GUS_MIXER_ID_LINE,	GUS_MIXER_MASK_LINE,
		gus_mixer_read_line,	gus_mixer_write_line	},
  { " MIC  ",	GUS_MIXER_ID_MIC,	GUS_MIXER_MASK_MIC,
		gus_mixer_read_mic,	gus_mixer_write_mic	},
  { "  CD  ",	GUS_MIXER_ID_CD,	GUS_MIXER_MASK_CD,
		gus_mixer_read_cd,	gus_mixer_write_cd	},
  { " Soft ",	GUS_MIXER_ID_SOFT,	GUS_MIXER_MASK_SOFT,
		gus_mixer_read_imix,	gus_mixer_write_imix	},
  { " Gain ",	GUS_MIXER_ID_GAIN,	GUS_MIXER_MASK_GAIN,
		gus_mixer_read_reclev,	gus_mixer_write_reclev	},
  { " Loop ",	GUS_MIXER_ID_LOOP,	GUS_MIXER_MASK_LOOP,
		gus_mixer_read_loopback, gus_mixer_write_loopback },
  { NULL, 	0,			0,
  		NULL,			NULL			}
};

struct channelInfo *channels[ SOUND_MIXER_NRDEVICES ];

/*
 *  Functions
 */

void setcolor( int color, int attrib )
{
  if ( colormode )
    attrset( COLOR_PAIR( color ) | A_BOLD );
   else
    attrset( attrib );
}

void xsetcolor( WINDOW *w, int color, int attrib )
{
  if ( colormode )
    wattrset( w, COLOR_PAIR( color ) | A_BOLD );
   else
    wattrset( w, attrib );
}

inline void wsetcolor( int color, int attrib )
{
  xsetcolor( w_panel, color, attrib );
}

void initChannels( void )
{
  struct channelInfo *chi;
  
  for ( chi = ichannels, bars = 0; chi -> name; chi++ )
    if ( chi -> mask & devmask )
      channels[ bars++ ] = chi;
}

void initColors(void)
{
  start_color();
  init_pair( TITLE1_COLOR,		COLOR_WHITE,	COLOR_BLUE );
  init_pair( TITLE2_COLOR,		COLOR_YELLOW,	COLOR_BLUE );
  init_pair( TITLE3_COLOR,		COLOR_RED,	COLOR_BLUE );
  init_pair( BORDER_COLOR,		COLOR_CYAN,	COLOR_BLACK );
  init_pair( RECSRC_COLOR, 		COLOR_RED,	COLOR_BLACK );
  init_pair( ACTIVE_RECSRC_COLOR,	COLOR_GREEN,	COLOR_BLACK );
  init_pair( BAR1_COLOR,		COLOR_WHITE,	COLOR_BLACK );
  init_pair( BAR2_COLOR,		COLOR_GREEN,	COLOR_BLACK );
  init_pair( BAR3_COLOR,		COLOR_RED,	COLOR_BLACK );
  init_pair( BAR_LABEL_COLOR,		COLOR_WHITE,	COLOR_BLUE );
  init_pair( BAR_FRAME_COLOR,		COLOR_CYAN,	COLOR_BLACK );
  init_pair( SELECT_COLOR,		COLOR_YELLOW, 	COLOR_BLACK );
  init_pair( HELP_BORDER_COLOR,		COLOR_BLACK, 	COLOR_CYAN );
  init_pair( HELP_TITLE_COLOR,		COLOR_WHITE,	COLOR_CYAN );
  init_pair( HELP_HOTKEY_COLOR,		COLOR_YELLOW,	COLOR_CYAN );
  init_pair( HELP_TEXT_COLOR,		COLOR_RED,	COLOR_CYAN );
}

void getBarPos( int index, int *x, int *y )
{
  int i;

  i = ( COLS - 6 ) / bars;
  *x = ( i * index ) + 3 + ( 4 + ( COLS - 6 ) - ( i * bars ) ) / 2;
  *y = ( LINES - 24 ) / 2;
}

void drawStaticThing( int index )
{
  int posX, posY, i;
  
  getBarPos( index, &posX, &posY );
  wsetcolor( BAR_FRAME_COLOR, A_NORMAL );
  mvwaddch( w_panel, posY +  4, posX + 0, ACS_ULCORNER );
  mvwaddch( w_panel, posY +  4, posX + 3, ACS_URCORNER );
  mvwaddch( w_panel, posY + 18, posX + 0, ACS_LLCORNER );
  mvwaddch( w_panel, posY + 18, posX + 3, ACS_LRCORNER );
  for ( i = 0; i < 13; i++ )
    {
      mvwaddch( w_panel, posY + 5 + i, posX + 0, ACS_VLINE );
      mvwaddch( w_panel, posY + 5 + i, posX + 3, ACS_VLINE );
    }
}

void drawStaticThings()
{
  int i;

  for ( i = 0; i < bars; i++ )
    drawStaticThing( i );
}

void drawBar( int index )
{
  int posX, posY, i, left, right, rcol, lcol;
  struct channelInfo *chi;
  char str[ 5 ], rstr[5], lstr[5];
  
  chi = channels[ index ];
  if ( ( left = chi -> read() ) < 0 ) return;
  getBarPos( index, &posX, &posY );
  posX++;
  if ( recmask & chi -> mask )
    {
      if ( gus_mixer_read_recsrc() & chi -> mask )
        {
          wsetcolor( ACTIVE_RECSRC_COLOR, A_REVERSE );
          mvwaddstr( w_panel, posY + 2, posX, "ON" );
        }
       else
        {
          wsetcolor( RECSRC_COLOR, A_NORMAL );
          mvwaddstr( w_panel, posY + 2, posX, "--" );
        }
    }
#if 0
  printf( "\n\n\n\ngus_mixer_l = 0x%x, mask = 0x%x\n", gus_mixer_read_devs_lmute(), chi -> mask );
#endif
  if ( gus_mixer_read_devs_lmute() & chi -> mask )
    {
      wsetcolor( ACTIVE_RECSRC_COLOR, A_REVERSE );
      mvwaddch( w_panel, posY + 3, posX, 'M' );
    }
   else
    {
      wsetcolor( RECSRC_COLOR, A_REVERSE );
      mvwaddch( w_panel, posY + 3, posX, ' ' );
    }
  if ( gus_mixer_read_devs_rmute() & chi -> mask )
    {
      wsetcolor( ACTIVE_RECSRC_COLOR, A_REVERSE );
      mvwaddch( w_panel, posY + 3, posX + 1, 'M' );
    }
   else
    {
      wsetcolor( RECSRC_COLOR, A_REVERSE );
      mvwaddch( w_panel, posY + 3, posX + 1, ' ' );
    }
  right = ( left >> 8 ) & 0xff;
  left &= 0xff;
  sprintf(lstr, "%3d<", left);
  sprintf(rstr, ">%-3d", right);
  left = ( ( left * 20 ) / ( 6 * 21 ) ) - 1;
  right = ( ( right * 20 ) / ( 6 * 21 ) ) - 1;
  str[ 4 ] = 0;
  rcol=3;
  lcol=3;
  wsetcolor( BAR3_COLOR, A_NORMAL );
  for ( i = 0; i < 7; i++ )
    {
      strcpy( str, "  " );
      if ( left >= 6 - i ) { str[ 0 ] = '#'; lcol=3; };
      if ( right >= 6 - i ) { str[ 1 ] = '#'; rcol=3; };
      mvwaddstr( w_panel, posY + i + 12, posX, str );
    }
  wsetcolor( BAR2_COLOR, A_NORMAL );
  for ( i = 0; i < 5; i++ )
    {
      strcpy( str, "  " );
      if ( left >= 11 - i ) { str[ 0 ] = '#'; lcol=2; };
      if ( right >= 11 - i ) { str[ 1 ] = '#'; rcol=2; };
      mvwaddstr( w_panel, posY + i + 7, posX, str );
    }
  wsetcolor( BAR1_COLOR, A_BOLD );
  for ( i = 0; i < 3; i++ )
    {
      strcpy( str, "  " );
      if ( left >= 14 - i ) { str[ 0 ] = '#'; lcol=1; };
      if ( right >= 14 - i ) { str[ 1 ] = '#'; rcol=1; };
      mvwaddstr( w_panel, posY + i + 4, posX, str );
    }
  switch (lcol) {
  	case 2:  wsetcolor( BAR2_COLOR, A_NORMAL ); break;
  	case 3:  wsetcolor( BAR3_COLOR, A_NORMAL ); break;
  	default: wsetcolor( BAR1_COLOR, A_BOLD ); break;
  }
  mvwaddstr( w_panel, posY + 19, posX - 3, lstr );
  switch (rcol) {
  	case 2:  wsetcolor( BAR2_COLOR, A_NORMAL ); break;
  	case 3:  wsetcolor( BAR3_COLOR, A_NORMAL ); break;
  	default: wsetcolor( BAR1_COLOR, A_BOLD ); break;
  }
  mvwaddstr( w_panel, posY + 19, posX + 1, rstr );
  wsetcolor( BAR_LABEL_COLOR, A_REVERSE );
  mvwaddstr( w_panel, posY + 20, posX - 2, chi -> name ); 
  wrefresh( w_panel );
}

void drawBars( void )
{
  int i;

  for ( i = 0; i < bars; i++ )
    drawBar( i );
}

void drawInterWaveSerial( void )
{
  static char *strings[] = {
"  serial transfer: none  ",
"  serial transfer: synth DSP -> RECORD  ",
"  serial transfer: synth DSP -> PLAYBACK  ",
"  serial transfer: RECORD -> PLAYBACK  ",
"  serial transfer: DSP -> external OUT  ",
"  serial transfer: RECORD -> external OUT  ",
"  serial transfer: external IN - PLAYBACK (1)  ",
"  serial transfer: external IN - PLAYBACK (2)  ",
};
  int i;
  char *s;

  if ( gus_mixer_version() == GUS_MIXER_ENH_CODEC )
    {
      i = gus_mixer_read_interwave_serial();
      if ( i < 0 )
        s = "  serial transfer: error  ";
       else
        s = strings[ i & 7 ];
      wsetcolor( BORDER_COLOR, A_BOLD );
      wborder(w_panel, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
  		   ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
      wsetcolor( BORDER_COLOR, A_BOLD );
      mvwaddstr( w_panel, LINES-3, ( COLS - strlen( s ) ) / 2, s );
    }
}

void selectBar( int index, int select )
{
  int posX, posY;

  getBarPos( index, &posX, &posY );
  wsetcolor( SELECT_COLOR, A_BOLD );
  mvwaddch( w_panel, posY + 20, posX - 2, select ? '>' : ' ' );
  mvwaddch( w_panel, posY + 20, posX + 5, select ? '<' : ' ' );
}

void changeVolume( int index, int direction, int what )
{
  struct channelInfo *chi;
  int left, right;
  
  chi = channels[ index ];
  if ( ( left = chi -> read() ) < 0 ) return;
  right = ( left >> 8 ) & 0xff;
  left &= 0xff;
  if ( what & 4 )
    left = right = ( left + right ) / 2;
  if ( what & 1 )
    {
      left += direction;
      if ( left < 0 ) left = 0;
      if ( left > 100 ) left = 100;
    }
  if ( what & 2 )
    {
      right += direction;
      if ( right < 0 ) right = 0;
      if ( right > 100 ) right = 100;
    }
  chi -> write( left | ( right << 8 ) );
  drawBar( index );
}

void changeRecSrc( int index )
{
  struct channelInfo *chi;
  int val;
  
  chi = channels[ index ];
  if ( !(chi -> mask & recmask) ) return;
  if ( ( val = gus_mixer_read_recsrc() ) < 0 ) return;
  val ^= chi -> mask;
  gus_mixer_write_recsrc( val ); 
  drawBar( index );
}

void changeMute( int index, int what )
{
  struct channelInfo *chi;
  int val;
  
  chi = channels[ index ];
  if ( what & 1 )
    {
      if ( ( val = gus_mixer_read_devs_lmute() ) < 0 ) return;
      val = ( chi -> mask & val ) == 0;
      gus_mixer_write_devs_lmute( chi -> id, val );
   }
  if ( what & 2 )
    {
      if ( ( val = gus_mixer_read_devs_rmute() ) < 0 ) return;
      val = ( chi -> mask & val ) == 0;
      gus_mixer_write_devs_rmute( chi -> id, val );
   }
  drawBar( index );
}

void changeInterWaveSerial( void )
{
  int val;
  
  if ( gus_mixer_version() == GUS_MIXER_ENH_CODEC )
    {
      val = gus_mixer_read_interwave_serial();
      gus_mixer_write_interwave_serial( ( val + 1 ) % 8 );
      drawInterWaveSerial();
    }
}

void updateAll( int index )
{
  drawStaticThings();
  drawBars();
  drawInterWaveSerial();
  selectBar( index, 1 );
}

void help( int index )
{
#define	NUM_HLINES	16
  struct {
    char *hot;
    char *text;
  } *hl, helpLines[NUM_HLINES] = {
    { "F1              ", "Help                               " },
    { "ESC             ", "Exit                               " },
    { "PgUp,LEFT       ", "Previous volume bar                " },
    { "PgDn,RIGHT,TAB  ", "Next volume bar                    " },
    { "UP,+,w,W        ", "Highter volume                     " },
    { "DOWN,-,x,X      ", "Lower volume                       " },
    { "HOME            ", "Highest volume                     " },
    { "END             ", "Lowest volume                      " },
    { "q,Q             ", "Highter volume for left channel    " },
    { "z,Z,y,Y         ", "Lower volume for left channel      " },
    { "e,E             ", "Highter volume for right channel   " },
    { "c,C             ", "Lower volume for right channel     " },
    { "b,B             ", "Balance left and right channel     " },
    { "m,M,<,>         ", "MUTE / UNMUTE                      " },
    { "SPACE           ", "Toggle record source               " },
    { "s,S             ", "Toggle serial transfer mode        " },
  };

  int key, i;
  WINDOW *w;
  char str[ 60 ];

  w = newwin( NUM_HLINES + 8, 55, ( ( LINES - NUM_HLINES - 8 ) / 2 ) + ( LINES < 25 ? 0 : 1 ), ( COLS - 55 ) / 2 );
  xsetcolor( w, HELP_BORDER_COLOR, A_REVERSE );
  wclear( w );
  wborder(w, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
  	     ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
  wmove( w, 1, 1 );
  waddstr( w, "                                                     " );
  wmove( w, 2, 1 );
  xsetcolor( w, HELP_TITLE_COLOR, A_REVERSE );
  sprintf( str, "                 GusMix v%s Help                  ", VERSION );
  waddstr( w, str );
  wmove( w, 3, 1 );
  waddstr( w, "                                                     " );
  for ( i = 0, hl = helpLines; i < NUM_HLINES; i++, hl++ )
    {
      if ( gus_mixer_version() != GUS_MIXER_ENH_CODEC && i == NUM_HLINES - 1 ) hl = NULL;
      wmove( w, i + 4, 1 );
      xsetcolor( w, HELP_BORDER_COLOR, A_REVERSE );
      waddstr( w, "  " );
      wmove( w, i + 4, 3 );
      xsetcolor( w, HELP_HOTKEY_COLOR, A_REVERSE );
      waddstr( w, hl ? hl -> hot : "                " );
      wmove( w, i + 4, 19 );
      xsetcolor( w, HELP_TEXT_COLOR, A_REVERSE );
      waddstr( w, hl ? hl -> text : "                                   " );
    }
  wmove( w, NUM_HLINES + 3 + 1, 1 );
  xsetcolor( w, HELP_BORDER_COLOR, A_REVERSE );
  waddstr( w, "                                                     " );
  wmove( w, NUM_HLINES + 3 + 2, 1 );
  xsetcolor( w, HELP_HOTKEY_COLOR, A_REVERSE );
  waddstr( w, "             Press ENTER to continue...              " );
  wmove( w, NUM_HLINES + 3 + 3, 1 );
  xsetcolor( w, HELP_BORDER_COLOR, A_REVERSE );
  waddstr( w, "                                                     " );
  wrefresh( w );
  while ( 1 ) {
    if ( ( key = getch() ) == 10 ) break;	/* Enter */
#if 0
    fprintf( stderr, "key = %i\n", key );
#endif
  }
  delwin( w );
  redrawwin( stdscr );
  redrawwin( w_panel );
  refresh();
}

void process( void )
{
  int key, selected = 0, direction, channel;

  updateAll( selected );
  while ( 1 ) {
    wrefresh( w_panel );
    key = getch();
    direction = channel = 0;
    switch ( key ) {
      case 27:		/* Escape */
        return;
      case 9:		/* Tab */
      case KEY_RIGHT:
      case KEY_NPAGE:
        direction++;
      case KEY_LEFT:
      case KEY_PPAGE:
        if ( !direction ) direction--;
        selectBar( selected, 0 );
        selected += direction;
        if ( selected < 0 ) selected = bars - 1;
        selected %= bars;
        selectBar( selected, 1 );
        break;
      case '+':
      case 'w':
      case 'W':
      case KEY_UP:
        direction++;
      case '-':
      case 'x':
      case 'X':
      case KEY_DOWN:
        if ( !direction ) direction--;
        direction *= 2;
        changeVolume( selected, direction, 3 );
        break;
      case 'q':
      case 'Q':
        channel = 1;
        direction++;
      case 'y':
      case 'Y':
      case 'z':
      case 'Z':
        if ( !direction )
          {
            channel = 1;
            direction--;
          }
      case 'e':
      case 'E':
        if ( !direction )
          {
            channel = 2;
            direction++;
          }
      case 'c':
      case 'C':
        if ( !direction )
          {
            channel = 2;
            direction--;
          }
        changeVolume( selected, direction, channel );
        break;
      case 'b':		/* balance left and right channel */
      case 'B':
        changeVolume( selected, 0, 7 );
        break;
      case 'm':		/* mute/unmute */
      case 'M':
        changeMute( selected, 3 );
        break;
      case '<':		/* mute/unmute left channel */
      case ',':
        changeMute( selected, 1 );
        break;
      case '>':
      case '.':		/* mute/unmute right channel */
        changeMute( selected, 2 );
        break;
      case ' ':
        changeRecSrc( selected );
        break;
      case KEY_BEG:
      case KEY_HOME:
        changeVolume( selected, +100, 3 );
        break;
      case KEY_LL:
      case KEY_END:
        changeVolume( selected, -100, 3 );
        break;
      case KEY_F(1):
        help( selected );
        break;
      case 'S':
      case 's':
        changeInterWaveSerial();
        break;
    }
  }
}

static void signalHandler( int sgnl )
{
  clear();
  refresh();
  endwin();
  exit( 1 );
}

int main( int argc, char *argv[] )
{
  int i, j, gcard, card = 0;
  char str[ 81 ];

  while ( 1 )
    {
      int c;
      
      if ( ( c = getopt( argc, argv, "hc:" ) ) < 0 ) break;
      switch ( c ) {
        case '?':
        case 'h':
          printf( "Usage: gusmix [-c <card # or id>]\n" );
          return 1;
        case 'c':
          card = gus_mixer_look_for_card( optarg );
          if ( card < 0 )
            {
              fprintf( stderr, "Card '%s' not found.\n", optarg );
              exit( 1 );
            }
          break;
      }
    }

  if ( ( gcard = gus_mixer_open( card ) ) < 0 )
    {
      fprintf(stderr, "Error opening mixer.\n");
      exit( 1 );
    }

  mversion = gus_mixer_version();

  devmask = gus_mixer_read_devmask();
  recmask = gus_mixer_read_recmask();
  stereodevs = gus_mixer_read_stereodevs();

  if ( !devmask ) 
    {
      fprintf(stderr, "No device found.\n");
      exit( 1 );
    }
  if ( mversion != GUS_MIXER_NONE &&
       mversion != GUS_MIXER_ICS &&
       mversion != GUS_MIXER_CODEC &&
       mversion != GUS_MIXER_ENH_CODEC )
    {
      fprintf(stderr, "Unknown mixer type.\n");
      exit( 1 );
    }
  mmaskversion = 1 << mversion;

  initChannels();

  initscr();

  if ( LINES < 24 || COLS < 80 )
    {
      endwin();
      fprintf(stderr, "LINES < 24 (%i) or COLUMNS < 80 (%i)\n", LINES, COLS );
      exit( 1 );
    }

  leaveok( stdscr, TRUE );
  keypad( stdscr, TRUE );
  cbreak();
  noecho();

  signal( SIGINT, signalHandler );
  signal( SIGTERM, signalHandler );
  
  colormode = has_colors();
#if 0
  if ( !colormode )
    fprintf(stderr, "WARNING: no color capability detected.");
#endif

  initColors();
  move( 0, 0 );
  setcolor( TITLE1_COLOR, A_BOLD | A_REVERSE );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  sprintf( str, "Gravis UltraSound %s mixer chip",
  	mversion == GUS_MIXER_NONE ?	"without" :
  	mversion == GUS_MIXER_ICS ?	"with ICS2101" :
  	mversion == GUS_MIXER_CODEC ?	"with CS4231" :
  					"with InterWave" );
  i = j = ( COLS - 8 - strlen( str ) ) / 2;
  while ( i-- > 0 ) addch( ' ' );
  printw( str );
  j += strlen( str ) & 1;
  while ( j-- > 0 ) addch( ' ' );
  setcolor( TITLE1_COLOR, A_BOLD | A_REVERSE );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  addch( ACS_DIAMOND );
  
  move(LINES-1, 0);
  i=COLS;
  while ( i-- > 0 ) addch( ' ' );
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  addstr("   F1" );
  setcolor( TITLE3_COLOR, A_REVERSE );
  addstr(" Help  " );
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  addstr("LEFT/RIGHT" );
  setcolor( TITLE3_COLOR, A_REVERSE );
  addstr(" Prev/Next  " );
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  addstr("UP/DOWN" );
  setcolor( TITLE3_COLOR, A_REVERSE );
  addstr(" Higher/Lower  ");
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  addstr("SPACE" );
  setcolor( TITLE3_COLOR, A_REVERSE );
  addstr(" RecSrc  ");
  setcolor( TITLE2_COLOR, A_BOLD | A_REVERSE );
  addstr("ESC" );
  setcolor( TITLE3_COLOR, A_REVERSE );
  addstr(" Exit  " );
  refresh();

  w_panel = newwin(LINES - 2, COLS, 1, 0);
  wsetcolor( BORDER_COLOR, A_DIM );
  wclear(w_panel);
  wsetcolor( BORDER_COLOR, A_BOLD );
  wborder(w_panel, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
  		   ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);

  process();  

  gus_mixer_close( gcard );

  wsetcolor( BORDER_COLOR, A_BOLD );
  clear();
  refresh();
  endwin();

  signal( SIGINT, SIG_DFL );
  signal( SIGTERM, SIG_DFL );

  return 0;
}
