/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for PCM (VoxWare 3.00 compatible)
 */

#include "driver.h"
#include "pcm.h"

#if 0
#define PCM_DEBUG_BUFFERS
#endif

#define VOX_FORMATS() ( !card -> use_codec ? \
AFMT_MU_LAW | AFMT_U8 | AFMT_S8 | AFMT_S16_LE | AFMT_U16_LE : \
AFMT_MU_LAW | AFMT_A_LAW | AFMT_IMA_ADPCM | AFMT_U8 | AFMT_S16_LE | AFMT_S16_BE )

static void pcm_fill_with_neutral( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn );
static void pcm_compute_blocks( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn );

/*
 *
 */

void gus_pcm_init( gus_card_t *card )
{
  card -> pcm.flags = PCM_LFLG_NONE;
}
 
static unsigned short pcm_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return PCM_LFLG_PLAY;
    case O_RDONLY:	return PCM_LFLG_RECORD;
    default:		return PCM_LFLG_BOTH;
  }
}

/*
 *  user to dma
 */

static int pcm_user_to_dma( gus_card_t *card, char *buf, int count )
{
  unsigned long flags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  int result, tmp;

  if ( count <= 0 || !buf ) return 0;
  if ( VERIFY_AREA( VERIFY_READ, buf, count ) ) return -EACCES;

  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
#if 0
  PRINTK( "pcm_user_to_dma: buf=0x%x, count=0x%x (%s)\n", (int)buf, count, pchn -> flags & PCM_FLG_NONBLK ? "non blocked" : "blocked" );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~PCM_FLG_ABORT;
  if ( pchn -> flags & PCM_FLG_MMAP ) return -EIO;	/* go to hell... */
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( card, pchn );
  if ( pchn -> flags & PCM_FLG_NEUTRAL )
    pcm_fill_with_neutral( card, pchn );

  result = 0;

  /*
   * OK. This is patch for .au files which begin with .snd header...
   * This is a little bit hard - it's application work to do conversions...
   */
  if ( (pchn -> mode & PCM_MODE_ULAW) && !pchn -> processed_bytes && !pchn -> used && count > 31 )
    {
      unsigned char buffer[ 8 ];
      
      MEMCPY_FROMFS( buffer, buf, 8 );
      if ( !memcmp( buffer, ".snd\0\0\0\x20", 8 ) )
        {
          buf += 32;
          count -= 32;
          result += 32;
        }
    }

  while ( count > 0 )
    {
      while ( pchn -> used >= pchn -> blocks )
        {
          if ( pchn -> flags & PCM_FLG_NONBLK ) return result;
          if ( !(pchn -> flags & PCM_FLG_ENABLE) ) return result;
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( card, playback, 10 * HZ );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( TABORT( playback ) )
            {
              pchn -> flags |= PCM_FLG_ABORT;
              return result;
            }
          if ( pchn -> used >= pchn -> blocks && TIMEOUT( card, playback ) )
            {
              PRINTK( "pcm_user_to_dma: timeout, new block discarded\n" );
              return -EIO;
            }
        }
        
      /* ok.. interresant part.. we must find first free & not locked block.. */
        
      tmp = 0;
      CLI( &flags );
      while ( pchn -> locks & ( 1 << pchn -> head ) )
        {
          pchn -> discarded++;
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          if ( pchn -> used >= pchn -> blocks )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */

      /* ok.. now we have right block - fill it */
      
      tmp = pchn -> block_size - pchn -> sizes[ pchn -> head ];
      if ( tmp > count ) tmp = count;	/* correction */

#if 0
      printk( "to dma: size = %i\n", tmp );
#endif
      if ( VERIFY_AREA( VERIFY_READ, buf, tmp ) )
        return -EACCES;
      pchn -> hw_dma( card, pchn, buf, tmp );

      count -= tmp;
      result += tmp;
      buf += tmp;

      pchn -> sizes[ pchn -> head ] += tmp;
      if ( pchn -> sizes[ pchn -> head ] == pchn -> block_size )
        {
#if 0
          printk( "block filled = %i\n", pchn -> head );
#endif
          pchn -> used++;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          if ( (pchn -> flags & PCM_FLG_ENABLE) &&
               (pchn -> used > (card -> pcm.flags & PCM_LFLG_REALTIME ? 0 : 1)) )
            pchn -> hw_init( card );
        }
#if 0
      PRINTK( "pcm_user_to_dma: end count=0x%x\n", count );
#endif
    }
#if 0
  PRINTK( "pcm_user_to_dma: end\n" );
#endif
  return result;
}

/*
 *  dma to user
 */

static int pcm_dma_to_user( gus_card_t *card, char *buf, int count )
{
  unsigned long flags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  int result, tmp;

  if ( count <= 0 ) return 0;
  if ( VERIFY_AREA( VERIFY_WRITE, buf, count ) ) return -EACCES;
  
  pchn = &card -> pcm.chn[ PCM_RECORD ];
#if 0
  PRINTK( "pcm_user_from_dma: buf=0x%x, count=0x%x (%s)\n", (int)buf, count, pchn -> flags & PCM_FLG_NONBLK ? "non blocked" : "blocked" );
#endif
  count &= ~( (pchn -> voices * (pchn -> mode & PCM_MODE_16 ? 2 : 1)) - 1 );
  pchn -> flags &= ~PCM_FLG_ABORT;
  if ( pchn -> flags & PCM_FLG_MMAP ) return -EIO;
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( card, pchn );
  if ( pchn -> flags & PCM_FLG_NEUTRAL )
    pcm_fill_with_neutral( card, pchn );

  if ( pchn -> flags & PCM_FLG_ENABLE )
    pchn -> hw_init( card );
  result = 0;
  
  while ( count > 0 )
    {
      while ( !pchn -> used )
        {
          if ( pchn -> flags & PCM_FLG_NONBLK ) return result;
          if ( !(pchn -> flags & PCM_FLG_ENABLE) ) return result;
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEP( card, record, 10 * HZ );
          pchn -> flags &= ~PCM_FLG_SLEEP;
          if ( TABORT( record ) )
            {
              pchn -> flags |= PCM_FLG_ABORT;
              return -EINTR;
            }
          if ( !pchn -> used && TIMEOUT( card, record ) )
            {
              PRINTK( "pcm_dma_to_user: data timeout\n" );
              return -EIO;
            }
        }
      
      tmp = 0;
      CLI( &flags );
      while ( pchn -> locks & ( 1 << pchn -> tail ) )
        {
          pchn -> discarded++;
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
          if ( !pchn -> used )
            {
              tmp++;
              break;
            }
        }
      STI( &flags );
      if ( tmp ) continue;		/* go to sleep loop */
        
      tmp = count <= pchn -> sizes[ pchn -> tail ] ?
		      	       count : pchn -> sizes[ pchn -> tail ];
      if ( VERIFY_AREA( VERIFY_WRITE, buf, tmp ) )
        return -EACCES;
      pchn -> hw_dma( card, pchn, buf, tmp );

      buf += tmp;
      count -= tmp;
      result += tmp;

      pchn -> sizes[ pchn -> tail ] -= tmp; 
      if ( !pchn -> sizes[ pchn -> tail ] )
        {
          pchn -> used--;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
        }
    }

  return result;
}

/*
 *  synchronize playback
 */

static void pcm_sync_playback( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  if ( pchn -> flags & PCM_FLG_ABORT ) return;
  pchn -> flags |= PCM_FLG_SYNC;
  if ( pchn -> used < pchn -> blocks && pchn -> sizes[ pchn -> head ] > 0 )
    {
      pchn -> hw_dma_neutral( card, pchn );
      pchn -> used++;
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
    }
  if ( !pchn -> used ) goto __end;
  pchn -> hw_sync( card );
  pcm_fill_with_neutral( card, pchn );
  pchn -> head = pchn -> tail = pchn -> used = 0;
  __end:
  pchn -> flags &= ~PCM_FLG_SYNC;
}

static void pcm_sync_record( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_RECORD ];
  pchn -> hw_sync( card );
  pcm_fill_with_neutral( card, pchn );
  pchn -> head = pchn -> tail = pchn -> used = 0;
}

/*
 *  other things
 */

void gus_pcm_set_dma_struct( gus_card_t *card, int dma_number, struct GUS_STRU_PCM_CHANNEL *pchn )
{
  int i;
  struct GUS_STRU_DMA *pdma;

  pdma = card -> dmas[ dma_number ];
  pchn -> _dma = pdma;
  pchn -> dma = pdma -> dma;
  pchn -> size = pdma -> usize;
  pchn -> buffer = pdma -> buf;
  for ( i = 0; i < PCM_MAX_BLOCKS; i++ )
    pchn -> sizes[ i ] = 0;
  pchn -> flags |= PCM_FLG_DMAOK;
}

static void pcm_fill_with_neutral( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn )
{
  MEMSET( pchn -> buffer, pchn -> neutral_byte, pchn -> used_size );
  pchn -> flags &= ~PCM_FLG_NEUTRAL;
}

static void pcm_compute_blocks( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn )
{
  int record;
  unsigned int block, blocks, bps;
  unsigned long flags;
#if 0
  unsigned int max;
#endif
  
  record = pchn != &card -> pcm.chn[ PCM_PLAYBACK ];
  if ( !record )
    pcm_sync_playback( card );
   else
    pcm_sync_record( card );
  CLI( &flags );
  if ( pchn -> requested_block_size )
    {
      block = pchn -> requested_block_size;
      if ( block > pchn -> size / pchn -> requested_blocks )
        block = pchn -> size / pchn -> requested_blocks;
      block &= ~3;	/* align to 4 */
      if ( !card -> use_codec )
        block &= ~31;
      blocks = pchn -> size / block;
      if ( blocks > pchn -> requested_blocks )
        blocks = pchn -> requested_blocks;
    }
   else
    {
      bps = pchn -> rate * pchn -> voices;
      if ( pchn -> mode & PCM_MODE_16 ) bps <<= 1;
      if ( pchn -> mode & PCM_MODE_ADPCM ) bps >>= 2;
      block = pchn -> size;
      while ( block > bps ) block >>= 1;
      if ( block == pchn -> size ) block >>= 1;
      if ( record )
        block /= 4;			/* small fragment when recording */
      block &= ~3;			/* align to 4 */
      blocks = pchn -> size / block;
    }
#if 0
   max = pchn -> hw_max_dma_block( card );
   if ( max < block ) block = max;
#endif
   if ( blocks > PCM_MAX_BLOCKS ) blocks = PCM_MAX_BLOCKS;
   pchn -> used_size = blocks * block;
   pchn -> blocks = blocks;
   pchn -> block_size = block;
   pchn -> flags |= PCM_FLG_NEUTRAL;
   pchn -> flags &= ~PCM_FLG_BUFFERS;
   STI( &flags );
#ifdef PCM_DEBUG_BUFFERS
   PRINTK( "used_size = %i, blocks = %i, blocks_size = %i, mmap = %s\n", pchn -> used_size, pchn -> blocks, pchn -> block_size, pchn -> flags & PCM_FLG_MMAP ? "yes" : "no" );
#endif
}

static int pcm_set_subdivision( gus_card_t *card, unsigned short flags, unsigned int subdivision )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        if ( pchn -> requested_subdivision != subdivision )
          {
            if ( idx == 1 )
              pcm_sync_playback( card );
             else
              pcm_sync_record( card );
            pchn -> requested_subdivision = subdivision;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }  
  return 0;
}

static int pcm_set_fragment( gus_card_t *card, unsigned short flags, unsigned int fragment )
{
  int idx, size, bytes, count;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  bytes = fragment & 0xffff;
  if ( bytes < 4 ) bytes = 4;
  if ( !card -> use_codec && bytes < 9 ) bytes = 9;
  if ( bytes > 17 ) bytes = 17;
  size = 1 << bytes;

  count = fragment >> 16;
  if ( count < 2 ) count = 2; 
  if ( count > PCM_MAX_BLOCKS ) count = PCM_MAX_BLOCKS;
  if ( !card -> use_codec )
    {
      if ( bytes < 11 && count < 3 ) count++;
      if ( bytes < 11 && count < 4 ) count++;
    }
   else
    if ( bytes < 10 && count < 3 ) count++;

#ifdef PCM_DEBUG_BUFFERS
  printk( "set fragment: size = 0x%x, count = 0x%x\n", size, count );
#endif

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        if ( pchn -> requested_block_size != size ||
             pchn -> requested_blocks != count )
          {
            if ( idx == 1 )
              pcm_sync_playback( card );
             else
              pcm_sync_record( card );
            pchn -> requested_block_size = size;
            pchn -> requested_blocks = count;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }  
  return ( count << 16 ) | bytes;
}


static unsigned int pcm_get_block_size( gus_card_t *card, unsigned short flags )
{
  if ( ( flags & PCM_LFLG_PLAY ) && ( card -> pcm.chn[ PCM_PLAYBACK ].flags & PCM_FLG_BUFFERS ) )
    pcm_compute_blocks( card, &card -> pcm.chn[ PCM_PLAYBACK ] );
  if ( ( flags & PCM_LFLG_RECORD ) && ( card -> pcm.chn[ PCM_RECORD ].flags & PCM_FLG_BUFFERS ) )
    pcm_compute_blocks( card, &card -> pcm.chn[ PCM_RECORD ] );
  return card -> pcm.chn[ ( flags - 1 ) & 1 ].block_size;
}

static void pcm_set_mode( gus_card_t *card, unsigned short flags, unsigned int mode )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        if ( pchn -> mode != mode )
          {
            if ( idx == 1 )
              pcm_sync_playback( card );
             else 
              pcm_sync_record( card );
            pchn -> neutral_byte = ( mode & PCM_MODE_16 ) ? 0x00 : 0x80;
            pchn -> mode = mode;
            pchn -> flags |= PCM_FLG_NEUTRAL | PCM_FLG_BUFFERS;
          }
      }
}

static unsigned int pcm_get_mode( gus_card_t *card, unsigned short flags )
{
  return card -> pcm.chn[ ( flags - 1 ) & 1 ].mode;
}

static void pcm_set_format( gus_card_t *card, unsigned short flags, unsigned int format )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        pchn -> format = format;
      }
}

static unsigned int pcm_get_format( gus_card_t *card, unsigned short flags )
{
  return card -> pcm.chn[ ( flags - 1 ) & 1 ].format;
}

static void pcm_set_rate( gus_card_t *card, unsigned short flags, unsigned int rate )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        if ( pchn -> rate != rate )
          {
            if ( idx == 1 )
              pcm_sync_playback( card );
             else
              pcm_sync_record( card );
            pchn -> rate = rate;
            pchn -> flags |= PCM_FLG_BUFFERS; 
          }
    }
}

static unsigned int pcm_get_rate( gus_card_t *card, unsigned short flags )
{
  return card -> pcm.chn[ ( flags - 1 ) & 1 ].rate;
}

static int set_format( gus_card_t *card, unsigned short flags, int format )
{
  unsigned int new_mode;
  unsigned int new_format;

  if ( format != AFMT_QUERY )
    {
#if 0
      printk( "format = 0x%x\n", format );
#endif
      if ( !format )
        {
          new_format = pcm_get_format( card, flags );
          new_mode = pcm_get_mode( card, flags );
        }
       else
        {
          new_format = format & VOX_FORMATS();
          if ( !new_format )		/* not supported format */
            new_format = AFMT_U8;	/* always supported */
          new_mode = pcm_get_mode( card, flags ) & ~PCM_MODE_TYPE;
          new_mode |= PCM_MODE_VALID;
          if ( new_format & ( AFMT_MU_LAW | AFMT_A_LAW | AFMT_U8 | 
                              AFMT_U16_LE | AFMT_U16_BE ) )
            new_mode |= PCM_MODE_U;
          if ( new_format & ( AFMT_S16_LE | AFMT_S16_BE | AFMT_U16_LE | 
                              AFMT_U16_BE | AFMT_IMA_ADPCM ) )
            new_mode |= PCM_MODE_16;
          if ( new_format & AFMT_MU_LAW )
            new_mode |= PCM_MODE_ULAW;
          if ( new_format & AFMT_A_LAW )
            new_mode |= PCM_MODE_ALAW;
          if ( new_format & AFMT_IMA_ADPCM )
            new_mode |= PCM_MODE_ADPCM;
        }
      if ( new_mode != pcm_get_mode( card, flags ) )
        {
          pcm_set_format( card, flags, new_format );
          pcm_set_mode( card, flags, new_mode );
        }
    }

  return pcm_get_format( card, flags );
}

static int set_rate( gus_card_t *card, unsigned short flags, unsigned int rate )
{
  unsigned int max_rate;

  if ( rate > 0 ) 
    {
      max_rate = !card -> use_codec ? 44100 : 48000;
      if ( rate > max_rate ) rate = max_rate;
      pcm_set_rate( card, flags, rate );
    }
   else
    rate = pcm_get_rate( card, flags );

  return rate;
}

static int pcm_set_channels( gus_card_t *card, unsigned short flags, int channels )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

#if 0
  printk( "set channels = %i\n", channels );
#endif
  if ( channels >= 0 )
    {
      if ( channels < 1 || channels > 32 ) return -EINVAL;
      if ( (card -> use_codec || (flags & PCM_LFLG_RECORD)) && channels > 2 ) return -EINVAL;
      for ( idx = 1; idx <= 2; idx++ )
        if ( flags & idx )
          {
            pchn = &card -> pcm.chn[ idx - 1 ];
            if ( idx == 1 )
              pcm_sync_playback( card );
             else
              pcm_sync_record( card );
            pchn -> voices = channels;
            pchn -> flags |= PCM_FLG_BUFFERS;
          }
    }
   else
    {
      for ( idx = 1; idx <= 2; idx++ )
        if ( flags & idx )
          {
            pchn = &card -> pcm.chn[ idx - 1 ];
            channels = pchn -> voices;
          }
    }
  return channels;
}

static void pcm_reset( gus_card_t *card, unsigned short flags )
{
  int idx;
  struct GUS_STRU_PCM_CHANNEL *pchn;

#ifdef PCM_DEBUG_BUFFERS
  printk( "pcm_reset!!!\n" );
#endif
  for ( idx = 1; idx <= 2; idx++ )
    if ( flags & idx )
      {
        pchn = &card -> pcm.chn[ idx - 1 ];
        if ( idx == 1 )
          pcm_sync_playback( card );
         else
          pcm_sync_record( card );
#if 0
        pchn -> requested_block_size =
        pchn -> requested_blocks =
        pchn -> requested_subdivision = 0;
        pchn -> processed_bytes = 0;
        pchn -> interrupts = 0;
        pchn -> flags |= PCM_FLG_BUFFERS;
#endif
        pchn -> flags &= ~PCM_FLG_ABORT;
      }
}

static int pcm_get_trigger( gus_card_t *card, unsigned short flags )
{
  unsigned long iflags;
  int result;

  result = 0;
  CLI( &iflags );
  if ( ( card -> pcm.chn[ PCM_PLAYBACK ].flags & PCM_FLG_TRIGGER ) && ( flags & PCM_LFLG_PLAY ) ) result |= PCM_ENABLE_OUTPUT;
  if ( ( card -> pcm.chn[ PCM_RECORD ].flags & PCM_FLG_TRIGGER ) && ( flags & PCM_LFLG_RECORD ) ) result |= PCM_ENABLE_INPUT;
  STI( &iflags );
#if 0
  printk( "get trigger = 0x%x\n", result );
#endif
  return result;
}

static int pcm_set_trigger( gus_card_t *card, unsigned short flags, int trigger )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

#if 0
  printk( "set trigger = 0x%x\n", trigger );
#endif
  if ( flags & PCM_LFLG_PLAY )
    {
      pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
      if ( trigger & PCM_ENABLE_OUTPUT )
        {
          pchn -> flags |= PCM_FLG_ENABLE;
          if ( !(pchn -> flags & PCM_FLG_TRIGGER) )
            {
              if ( (pchn -> flags & PCM_FLG_MMAP) || pchn -> used > 0 )
                {
                  if ( pchn -> flags & PCM_FLG_BUFFERS )
                    pcm_compute_blocks( card, pchn );
                  pchn -> hw_init( card );
                }
            }
        }
       else
        {
          pchn -> flags &= ~PCM_FLG_ENABLE;
          if ( pchn -> flags & PCM_FLG_TRIGGER )
            pchn -> hw_done( card );
        }
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      pchn = &card -> pcm.chn[ PCM_RECORD ];
      if ( trigger & PCM_ENABLE_INPUT )
        {
          pchn -> flags |= PCM_FLG_ENABLE;
          if ( !(pchn -> flags & PCM_FLG_TRIGGER) )
            {
              if ( pchn -> flags & PCM_FLG_BUFFERS )
                pcm_compute_blocks( card, pchn );
              pchn -> hw_init( card );
            }
        }
       else
        {
          pchn -> flags &= ~PCM_FLG_ENABLE;
          if ( pchn -> flags & PCM_FLG_TRIGGER )
            pchn -> hw_done( card );
        }
    }
  return pcm_get_trigger( card, flags );
}

static int pcm_get_space( gus_card_t *card, unsigned short flags, int record, audio_buf_info *arg )
{
  unsigned long iflags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  audio_buf_info info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &card -> pcm.chn[ record ];
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( card, pchn );
  CLI( &iflags );
  info.fragments = record ? pchn -> used : pchn -> blocks - pchn -> used;
  info.fragstotal = pchn -> blocks;
  info.fragsize = pchn -> block_size;
  info.bytes = info.fragments * info.fragsize;
  if ( !record && pchn -> sizes[ pchn -> head ] )
    {
      info.fragments--;
      info.bytes -= pchn -> sizes[ pchn -> head ];
    }
  STI( &iflags );
#ifdef PCM_DEBUG_BUFFERS
  printk( "ospace: frags = %i, total = %i, size = %i, bytes = %i\n", info.fragments, info.fragstotal, info.fragsize, info.bytes );
#endif
  pchn -> flags |= PCM_FLG_ENABLE;
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

static int pcm_get_ptr( gus_card_t *card, unsigned short flags, int record, count_info *arg )
{
  unsigned long iflags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  count_info info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &card -> pcm.chn[ record ];
  if ( pchn -> flags & PCM_FLG_BUFFERS )
    pcm_compute_blocks( card, pchn );
  CLI( &iflags );
  info.bytes = pchn -> processed_bytes;
  info.ptr = pchn -> hw_pointer( card );
  info.blocks = pchn -> interrupts; pchn -> interrupts = 0;
  if ( pchn -> flags & PCM_FLG_MMAP ) 
    info.bytes += info.ptr % pchn -> block_size;
#if 0
  printk( "ptr: bytes = %i, ptr = %i, blocks = %i\n", info.bytes, info.ptr, info.blocks );
#endif
  STI( &iflags );
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

static int pcm_get_mapbuf( gus_card_t *card, unsigned short flags, int record, buffmem_desc *arg )
{
  unsigned long iflags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  buffmem_desc info;

  if ( !( flags & ( 1 << record ) ) ) return -EIO;	/* bad file mode */
  if ( VERIFY_AREA( VERIFY_WRITE, arg, sizeof( info ) ) ) return -EIO;
  pchn = &card -> pcm.chn[ record ];
  CLI( &iflags );
  info.buffer = (unsigned *)pchn -> buffer;
  info.size = pchn -> size;
  STI( &iflags );
  MEMCPY_TOFS( arg, &info, sizeof( info ) );
  return 0;
}

static int pcm_gus_info_flags( gus_card_t *card, struct GUS_PCM_INFO *info )
{
  struct GUS_PCM_INFO ginfo;
  struct GUS_STRU_PCM_CHANNEL *pchn_p, *pchn_r;

  if ( VERIFY_AREA( VERIFY_WRITE, info, sizeof( ginfo ) ) ) return -EIO;
  MEMSET( &ginfo, 0, sizeof( ginfo ) );
  pchn_p = &card -> pcm.chn[ PCM_PLAYBACK ];
  pchn_r = &card -> pcm.chn[ PCM_RECORD ];
  if ( card -> use_codec )
    {
      ginfo.flags |= GUS_PCM_INFO_CODEC;
      sprintf( ginfo.name, "CODEC PCM - %s\n",
      			card -> pnp_flag ? "InterWave" : "CS4231" );
      ginfo.max_rate_play = ginfo.max_rate_record = 48000;
      ginfo.formats_play = ginfo.formats_record = AFMT_MU_LAW | AFMT_A_LAW | AFMT_IMA_ADPCM | AFMT_U8 | AFMT_S16_LE | AFMT_S16_BE;
    }
   else
    {
      strcpy( ginfo.name, "GF1 PCM" );
      ginfo.max_rate_play = ginfo.max_rate_record = 41000;
      ginfo.formats_play = AFMT_MU_LAW | AFMT_U8 | AFMT_S8 | AFMT_S16_LE | AFMT_U16_LE;
      ginfo.formats_record = AFMT_MU_LAW | AFMT_U8 | AFMT_S8;
    }
  ginfo.max_channels_play = ginfo.max_channels_record = 2;
  ginfo.dma_size_play = pchn_p -> size;
  ginfo.dma_size_record = pchn_r -> size;
  MEMCPY_TOFS( info, &ginfo, sizeof( ginfo ) );
  return 0;
}

static int gus_open_pcm_card( gus_card_t *card, short minor, struct file *file )
{
  unsigned short flags;
  int res;

  if ( ( flags = pcm_file_flags( file -> f_flags ) ) & card -> pcm.flags )
    return -EBUSY;			/* channel(s) already used */
#if 0
  PRINTK( "gus_open_pcm - minor = %i, flags = %i\n", minor, flags );
#endif
  if ( flags == PCM_LFLG_BOTH &&
       ( card -> equal_dma || card -> daughter_flag ) )
    return -EIO;
  pcm_set_format( card, flags, 0 );
  pcm_set_mode( card, flags, 0 );
  switch ( minor ) {
    case GUS_MINOR_AUDIO:
      set_format( card, flags, AFMT_MU_LAW );
      break;
    case GUS_MINOR_PCM_8:
      set_format( card, flags, AFMT_U8 );
      break; 
    case GUS_MINOR_PCM_16:
      set_format( card, flags, AFMT_U16_LE );
      break;
    default:
      PRINTK( "pcm: bad minor value\n" );
      return -EINVAL;
  }
  pcm_set_rate( card, flags, PCM_DEFAULT_RATE );
  pcm_set_channels( card, flags, 1 );
  if ( flags & PCM_LFLG_PLAY )
    {
      if ( ( res = card -> pcm.chn[ PCM_PLAYBACK ].hw_open( card ) ) < 0 )
        return res;
      card -> pcm.chn[ PCM_PLAYBACK ].flags |= PCM_FLG_ENABLE;
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      if ( ( res = card -> pcm.chn[ PCM_RECORD ].hw_open( card ) ) < 0 )
        {
          if ( flags & PCM_LFLG_PLAY )
            {
              card -> pcm.chn[ PCM_PLAYBACK ].hw_close( card );
              card -> pcm.chn[ PCM_PLAYBACK ].flags &= ~PCM_FLG_ENABLE;
            }
          return res;
        }
      card -> pcm.chn[ PCM_RECORD ].flags |= PCM_FLG_ENABLE;
    }
  card -> pcm.flags |= flags;

#if 0
  printk( "pcm open - done...\n" );
#endif

  return 0;
}

static void gus_close_pcm_card( gus_card_t *card, struct file *file )
{
  unsigned short flags;
  
  if ( !card ) return;
#if 0
  PRINTK( "gus_release_pcm\n" );
#endif
  flags = pcm_file_flags( file -> f_flags ) & card -> pcm.flags;
  if ( flags & PCM_LFLG_PLAY )
    {
      pcm_sync_playback( card );	/* synchronize playback */
      card -> pcm.chn[ PCM_PLAYBACK ].hw_close( card );
      card -> pcm.flags &= ~PCM_LFLG_PLAY;
      MEMSET( &card -> pcm.chn[ PCM_PLAYBACK ], 0, sizeof( struct GUS_STRU_PCM_CHANNEL ) - ( PCM_HW_FUNCTIONS * sizeof( long ) ) - PCM_ZERO_RESERVED );
    }
  if ( flags & PCM_LFLG_RECORD )
    {
      pcm_sync_record( card );		/* and record */
      card -> pcm.chn[ PCM_RECORD ].hw_close( card );
      card -> pcm.flags &= ~PCM_LFLG_RECORD;
      MEMSET( &card -> pcm.chn[ PCM_RECORD ], 0, sizeof( struct GUS_STRU_PCM_CHANNEL ) - ( PCM_HW_FUNCTIONS * sizeof( long ) ) - PCM_ZERO_RESERVED );
    }
#if 0
  printk( "release pcm: done\n" );
#endif
}

int gus_open_pcm( unsigned short minor, struct file *file )
{
  short dev;
  int res;

  if ( minor < GUS_MINOR_GBEGIN )
    {
      dev = minor >> 4;
      if ( dev >= gus_cards_count ) dev = 0;
      if ( ( res = gus_open_pcm_card( file -> private_data = gus_cards[ dev ], minor & GUS_MINOR_USS_MASK, file ) ) < 0 )
        {
          file -> private_data = NULL;
          return res;
        }
    }
   else
    {
      minor &= GUS_MINOR_GDEVMASK;
      if ( minor >= gus_cards_count ) return -ENODEV;
      if ( ( res = gus_open_pcm_card( file -> private_data = gus_cards[ minor ], GUS_MINOR_PCM_16, file ) ) < 0 )
        {
          file -> private_data = NULL;
          return res;
        }
    }
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_pcm( struct file *file )
{
  gus_close_pcm_card( (gus_card_t *)file -> private_data, file );
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_pcm( struct file *file, unsigned int cmd, unsigned long arg )
{
  gus_card_t *card;
  unsigned short flags;

  card = (gus_card_t *)file -> private_data;
  flags = pcm_file_flags( file -> f_flags );
  if ( ( ( cmd >> 8 ) & 0xff ) == 'M' )	/* mixer ioctl - for USS (grrr) compatibility */
    return gus_ioctl_mixer( card, file, cmd, arg );
  if ( ( ( cmd >> 8 ) & 0xff ) != 'P' ) return -EIO;
#if 0
  printk( "cmd = 0x%x, arg = 0x%x, flags = 0x%x\n", cmd, IOCTL_IN( arg ), flags );
#endif
  switch ( cmd ) {
    case SOUND_PCM_CARDS:
      return IOCTL_OUT( arg, gus_cards_count );
    case SOUND_PCM_WRITE_RATE:
      return IOCTL_OUT( arg, set_rate( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_READ_RATE:
      return IOCTL_OUT( arg, pcm_get_rate( card, flags ) );
    case SNDCTL_DSP_STEREO:
      return IOCTL_OUT( arg, pcm_set_channels( card, flags, IOCTL_IN( arg ) > 0 ? 2 : 1 ) - 1 );
    case SOUND_PCM_WRITE_CHANNELS:
      return IOCTL_OUT( arg, pcm_set_channels( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_READ_CHANNELS:
      return IOCTL_OUT( arg, pcm_set_channels( card, flags, -1 ) );
    case SOUND_PCM_GETFMTS:
      return IOCTL_OUT( arg, VOX_FORMATS() );
    case SOUND_PCM_SETFMT:
      return IOCTL_OUT( arg, set_format( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_GETCAPS:
      return IOCTL_OUT( arg,
      		( ( !card -> equal_dma && !card -> daughter_flag ) ? DSP_CAP_DUPLEX : 0 ) |
      		DSP_CAP_REALTIME |
      		( !card -> use_codec ? DSP_CAP_BATCH : 0 ) |
      		DSP_CAP_TRIGGER |
      		DSP_CAP_MMAP );
    case SOUND_PCM_READ_BITS:
      return IOCTL_OUT( arg, pcm_get_format( card, flags ) );
    case SNDCTL_DSP_GETBLKSIZE:
      return IOCTL_OUT( arg, pcm_get_block_size( card, flags ) );
    case SOUND_PCM_SUBDIVIDE:
      return IOCTL_OUT( arg, pcm_set_subdivision( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_SETFRAGMENT:
      return IOCTL_OUT( arg, pcm_set_fragment( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_SYNC:
    case SOUND_PCM_POST:	/* wrong implementation */
      if ( flags & PCM_LFLG_PLAY ) pcm_sync_playback( card );
      if ( flags & PCM_LFLG_RECORD ) pcm_sync_record( card );
      return 0;
    case SOUND_PCM_RESET:
      pcm_reset( card, flags );
      return 0;
    case SOUND_PCM_WRITE_FILTER:
    case SOUND_PCM_READ_FILTER:
      return -EINVAL;
    case SOUND_PCM_NONBLOCK:
      if ( flags & PCM_LFLG_PLAY ) card -> pcm.chn[ PCM_PLAYBACK ].flags |= PCM_FLG_NONBLK;
      if ( flags & PCM_LFLG_RECORD ) card -> pcm.chn[ PCM_RECORD ].flags |= PCM_FLG_NONBLK;
      return 0;
    case SOUND_PCM_GETTRIGGER:
      return IOCTL_OUT( arg, pcm_get_trigger( card, flags ) );
    case SOUND_PCM_SETTRIGGER:
      return IOCTL_OUT( arg, pcm_set_trigger( card, flags, IOCTL_IN( arg ) ) );
    case SOUND_PCM_GETOSPACE:
    case SOUND_PCM_GETISPACE:
      return pcm_get_space( card, flags, cmd == SOUND_PCM_GETISPACE, (audio_buf_info *)arg );
    case SOUND_PCM_GETIPTR:
    case SOUND_PCM_GETOPTR:
      return pcm_get_ptr( card, flags, cmd == SOUND_PCM_GETIPTR, (count_info *)arg );
    case SOUND_PCM_MAPINBUF:
    case SOUND_PCM_MAPOUTBUF:
      return pcm_get_mapbuf( card, flags, cmd == SOUND_PCM_MAPINBUF, (buffmem_desc *)arg );
    case SNDCTL_DSP_SETDUPLEX:
      if ( card -> equal_dma || card -> daughter_flag )
        return -EIO;
      return 0;
    case SOUND_PCM_SETSYNCRO:
      /* stop DMA now.. */
      return 0;
    case SOUND_PCM_GUSINFO:
      return pcm_gus_info_flags( card, (struct GUS_PCM_INFO *)arg );
    default:
      PRINTK( "pcm: unknown command = 0x%x\n", cmd ); 
  }
  return -EIO;
}

int gus_read_pcm( struct file *file, char *buf, int count )
{
  gus_card_t *card;

  card = (gus_card_t *)file -> private_data;
  if ( !card ) return -EIO;
  if ( !( pcm_file_flags( file -> f_flags ) & PCM_LFLG_RECORD ) ||
       !( card -> pcm.flags & PCM_LFLG_RECORD ) ) return -EIO;
  return pcm_dma_to_user( card, buf, count );
}

int gus_write_pcm( struct file *file, char *buf, int count )
{
  gus_card_t *card;

  card = (gus_card_t *)file -> private_data;
  if ( !card ) return -EIO;
  if ( !( pcm_file_flags( file -> f_flags ) & PCM_LFLG_PLAY ) ||
       !( card -> pcm.flags & PCM_LFLG_PLAY ) ) return -EIO;
  return pcm_user_to_dma( card, buf, count );
}

#ifdef GUS_POLL
unsigned int gus_poll_pcm( struct file *file, poll_table *wait )
{
  gus_card_t *card;
  unsigned long flags;
  unsigned int mask;
  struct GUS_STRU_PCM_CHANNEL *record, *playback;

  card = (gus_card_t *)file -> private_data;
  if ( !card ) return 0;

  record = &card -> pcm.chn[ PCM_RECORD ];
  playback = &card -> pcm.chn[ PCM_PLAYBACK ];
  CLI( &flags );
  record -> flags |= PCM_FLG_SLEEP;
  SLEEP_POLL( card, record, wait );
  playback -> flags |= PCM_FLG_SLEEP;
  SLEEP_POLL( card, playback, wait );
  STI( &flags );

  mask = 0;
  if ( record -> flags & PCM_FLG_MMAP )
    {
      if ( record -> interrupts ) mask |= POLLIN | POLLRDNORM;
    }
   else
    {
      if ( record -> used || (record -> flags & PCM_FLG_BUFFERS) ) mask |= POLLIN | POLLRDNORM;
    }
  if ( playback -> flags & PCM_FLG_MMAP )
    {
      if ( playback -> interrupts ) mask |= POLLOUT | POLLWRNORM;
    }
   else
    {
      if ( playback -> used < playback -> blocks || (playback -> flags & PCM_FLG_BUFFERS) ) mask |= POLLOUT | POLLWRNORM;
    }

  return mask;
}
#else
int gus_select_pcm( struct file *file, int sel_type, select_table *wait )
{
  int ok;
  gus_card_t *card;
  unsigned long flags;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  card = (gus_card_t *)file -> private_data;
  if ( !card ) return -EIO;
  switch ( sel_type ) {
    case SEL_IN:
      pchn = &card -> pcm.chn[ PCM_RECORD ];
      CLI( &flags );
      if ( pchn -> flags & PCM_FLG_MMAP )
        ok = !pchn -> interrupts; 
       else
        ok = !pchn -> used && !(pchn -> flags & PCM_FLG_BUFFERS);
      if ( ok )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEPS( card, record, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
      CLI( &flags );
      if ( pchn -> flags & PCM_FLG_MMAP )
        ok = !pchn -> interrupts; 
       else
        ok = pchn -> used >= pchn -> blocks && !(pchn -> flags & PCM_FLG_BUFFERS);
      if ( ok )
        {
          pchn -> flags |= PCM_FLG_SLEEP;
          SLEEPS( card, playback, wait );
          STI( &flags );
          return 0;
        }
      pchn -> flags &= ~PCM_FLG_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

int gus_mmap_pcm( struct inode *inode, struct file *file, struct vm_area_struct *vma )
{
  gus_card_t *card;
  unsigned short flags;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  unsigned long size;

  card = (gus_card_t *)file -> private_data;
  flags = pcm_file_flags( file -> f_flags );
  if ( ( vma -> vm_flags & ( VM_READ | VM_WRITE ) ) == ( VM_READ | VM_WRITE ) )
    return -EINVAL;
  if ( vma -> vm_flags & VM_READ )
    {
      if ( !(flags & PCM_LFLG_RECORD) ) return -EINVAL;
      pchn = &card -> pcm.chn[ PCM_RECORD ];
    }
   else
  if ( vma -> vm_flags & VM_WRITE )
    {
      if ( !(flags & PCM_LFLG_PLAY) ) return -EINVAL;
      pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
    }
   else
    return -EINVAL;
  if ( vma -> vm_offset != 0 ) return -EINVAL;
  size = vma -> vm_end - vma -> vm_start;
  if ( size != pchn -> size )
    printk( "pcm: wrong mmap() size 0x%x, should be 0x%x\n", (unsigned int)size, pchn -> size );
#if 0
  printk( "remap: to=0x%x from=0x%x size=0x%x\n", (int)vma -> vm_start, (int)virt_to_bus( pchn -> buffer ), (int)size );
#endif
  if ( remap_page_range( vma -> vm_start,
                         virt_to_bus( pchn -> buffer ), 
                         size,
                         vma -> vm_page_prot ) )
    return -EAGAIN;
#if LinuxVersionCode( 2, 1, 45 ) > LINUX_VERSION_CODE
  vma -> vm_inode = inode;
#if LinuxVersionCode( 2, 1, 43 ) > LINUX_VERSION_CODE
  inode -> i_count++;
#else
  atomic_inc( &inode -> i_count );
#endif
#else
  vma -> vm_dentry = dget( file -> f_dentry );
#endif
  pchn -> _dma -> mmaped = 1;
  pchn -> flags |= PCM_FLG_MMAP;
  pcm_fill_with_neutral( card, pchn );
  return 0;
}

void gus_pcm_info( gus_card_t *card, gus_info_buffer_t *buffer )
{
  gus_iprintf( buffer, "PCM:\n" );
  gus_iprintf( buffer, "  realtime buffering : %s\n", card -> pcm.flags & PCM_LFLG_REALTIME ? "on" : "off" );
  gus_iprintf( buffer, "  playback underflow : %i\n", card -> pcm.chn[ PCM_PLAYBACK ].discarded );
  gus_iprintf( buffer, "  record overflow    : %i\n", card -> pcm.chn[ PCM_RECORD ].discarded );
}
