/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for the MIDI interface
 */

#include "driver.h"
#include "midi.h"

#ifdef GUSCFG_MIDI

#define MIDI_MODE_NONE		0x0000
#define MIDI_MODE_IN		0x0001
#define MIDI_MODE_OUT		0x0002
#define MIDI_MODE_BOTH		(MIDI_MODE_IN|MIDI_MODE_OUT)

#define MIDI_TIMER_NOT_LOADED	0x0000
#define MIDI_TIMER_STOPED	0x0001
#define MIDI_TIMER_RUNNING	0x0002

static unsigned short midi_mode;

static unsigned char *midi_rx_buf;
static unsigned int midi_rx_size;
static unsigned int midi_rx_used;
static unsigned int midi_rx_head;
static unsigned int midi_rx_tail;

static unsigned char *midi_tx_buf;
static unsigned int midi_tx_size;
static unsigned int midi_tx_used;
static unsigned int midi_tx_head;
static unsigned int midi_tx_tail;
static unsigned int midi_tx_threshold;

static unsigned int midi_rx_ticks;
static unsigned int midi_tx_ticks;

static unsigned short midi_timer_state;

static unsigned char *midi_buf_cmd;
static unsigned char *midi_buf_realtime;

static int midi_flush_flag;

static gus_card_t *midi_master_card;

static void process_output_events( int now );

static unsigned short midi_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return MIDI_MODE_OUT;
    case O_RDONLY:	return MIDI_MODE_IN;
    default:		return MIDI_MODE_BOTH;
  }
}

static int timer_stop( void )
{
  if ( midi_master_card )
    return gus_timer_ioctl( midi_master_card, GUS_IOCTL_TIMER_STOP, 0 );
  return 0;
}

static int midi_timer_base( int base );
static int midi_timer_tempo( int tempo );

static int midi_open( unsigned int what, unsigned short mode )
{
  unsigned int card_number;
  unsigned int dev;
  int tmp;
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;
  
#if 0
  printk( "midi open - mode = 0x%x, what = 0x%x\n", mode, what );
#endif
  card_number = ( what >> 4 ) & 0x000f;
  dev = what & 0x000f;
  what >>= 8;
  if ( !( mode & MIDI_MODE_OUT ) && ( what & GUS_MIDI_OPEN_MODE_WRITE ) ) return -EINVAL;
  if ( !( mode & MIDI_MODE_IN ) && ( what & GUS_MIDI_OPEN_MODE_READ ) ) return -EINVAL;
  if ( card_number >= gus_cards_count || dev > GUS_MIDID_LAST ) return -EINVAL;
  if ( dev == GUS_MIDID_SYNTH && ( what & GUS_MIDI_OPEN_MODE_READ ) ) return -ENODEV;
  card = gus_cards[ card_number ];
  if ( !midi_master_card && ( card -> gf1.mode & GF1_MODE_ENGINE ) ) return -EBUSY;
  midi = &card -> midi[ dev ];
  midi -> dev = ( card_number << 4 ) | dev;
  if ( what & GUS_MIDI_OPEN_MODE_WRITE ) 
    {
      if ( midi -> flags & GUS_MIDIF_USED_OUT ) return -EBUSY;
      midi -> tx_size = GUS_MIDI_BUF_SIZE;
      midi -> tx_used = midi -> tx_head = midi -> tx_tail = 0;
      if ( ( tmp = midi -> init_write( card ) ) < 0 ) return tmp;
      midi -> flags |= GUS_MIDIF_USED_OUT;
    }
  if ( what & GUS_MIDI_OPEN_MODE_READ )
    {
      if ( midi -> flags & GUS_MIDIF_USED_IN )
        {
          if ( what & GUS_MIDI_OPEN_MODE_WRITE )
            {
              midi -> done_write( card );
              midi -> flags &= ~GUS_MIDIF_USED_OUT;
            }
          return -EBUSY;
        }
      midi -> rx.size = GUS_MIDI_BUF_SIZE;
      midi -> rx.dev = midi -> dev;
      midi -> rx.ack = gus_midi_rx_command;
      gus_midi_cmd_init( &midi -> rx, 1 );
      if ( ( tmp = midi -> init_read( card ) ) < 0 )
        {
          if ( what & GUS_MIDI_OPEN_MODE_WRITE )
            {
              midi -> done_write( card );
              midi -> flags &= ~GUS_MIDIF_USED_OUT;
            }
          return tmp;
        }
      midi -> flags |= GUS_MIDIF_USED_IN;
      midi -> thru_mask = 0;
    }
  if ( !midi_master_card )
    {
      if ( gus_timer_open( card, "midi" ) < 0 )
        PRINTK( "gus: MIDI - can't open timer\n" );
      if ( gus_timer_set_master( midi_master_card = card ) < 0 )
        PRINTK( "gus: MIDI - timer set master - unexpected error\n" );
      card -> gf1.timer_midi = 1;
      midi_timer_base( 320 );			/* default timer base */
      midi_timer_tempo( 60 );			/* 60Hz - default tempo */
    }
   else
    {
      if ( !card -> gf1.timer_slave && card != midi_master_card )
        if ( gus_timer_set_slave( midi_master_card, card ) < 0 )
          PRINTK( "gus: MIDI - timer set slave - unexpected error\n" );
    }
  return 0;  
}

static void midi_done_write( gus_card_t *card, int device )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ device ];
  midi -> done_write( card );
  midi -> flags &= ~GUS_MIDIF_USED_OUT;
}

static void midi_done_read( gus_card_t *card, int device )
{
  struct GUS_STRU_MIDI *midi;

  midi = &card -> midi[ device ];
  midi -> done_read( card );
  midi -> flags &= ~GUS_MIDIF_USED_IN;
}

static int midi_close_all( unsigned short mode )
{
  static void purge_rx( void );
  static void purge_tx( void );
  int i, j;
  gus_card_t *card;

  if ( !midi_master_card ) return 0;
  timer_stop();
  purge_rx();
  purge_tx();
  gus_timer_close( midi_master_card );
  midi_master_card -> gf1.timer_midi = 0;
  midi_master_card = NULL;
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      for ( j = 0; j <= GUS_MIDID_LAST; j++ )
        {
          int flags;
        
          flags = card -> midi[ j ].flags;
          if ( ( mode & MIDI_MODE_OUT ) && ( flags & GUS_MIDIF_USED_OUT ) )
            midi_done_write( card, j );
          if ( ( mode & MIDI_MODE_IN ) && ( flags & GUS_MIDIF_USED_IN ) )
            midi_done_read( card, j );
        }
    }
  purge_rx();		/* for sure - maybe some data is still waiting */
  purge_tx();		/* for sure */
  return 0;
}

static int midi_open_all( struct GUS_STRU_MIDI_OPEN *mopen, int mode )
{
  int i, res;

  struct GUS_STRU_MIDI_OPEN smopen;
  
  if ( VERIFY_AREA( VERIFY_READ, mopen, sizeof( *mopen ) ) ) return -EIO;
  MEMCPY_FROMFS( &smopen, mopen, sizeof( smopen ) );
  mopen = &smopen;
  
  if ( !mopen -> devices_used ||
       mopen -> devices_used > GUS_MIDID_LAST * GUS_CARDS ) return -EINVAL;
  for ( i = 0; i < mopen -> devices_used; i++ )
    if ( ( res = midi_open( ( mopen -> devices[ i ].mode << 8 ) | mopen -> devices[ i ].device, mode ) ) < 0 )
      {
        midi_close_all( mode );
        return res;
      }
  return 0;
}

/*
 *  -------------
 */

static void purge_rx( void )
{
  midi_rx_used = midi_rx_head = midi_rx_tail = 0;
}

static void put_rx_byte( unsigned char b )
{
  midi_rx_buf[ midi_rx_head++ ] = b;
  midi_rx_head %= midi_rx_size;
}

static unsigned char get_rx_byte( void )
{
  unsigned char result;
  
  result = midi_rx_buf[ midi_rx_tail++ ];
  midi_rx_tail %= midi_rx_size;
  return result;
}

static void back_rx_byte( void )
{
  if ( !midi_rx_tail ) midi_rx_tail = midi_rx_size - 1; else midi_rx_tail--;
}

void gus_midi_rx_command( unsigned int device, unsigned char *buffer, unsigned int count )
{
  unsigned long flags;
  unsigned int count0;
  unsigned char *buffer0;

  CLI( &flags );
  if ( midi_rx_ticks )
    {
      if ( midi_rx_used + 8 > GUS_MIDI_QUEUE_SIZE )
        {
          STI( &flags );
          PRINTK( "gus: MIDI Rx buffer overflow - command discarded\n" );
          return; 
        }
      put_rx_byte( 0 );
      put_rx_byte( 5 );
      put_rx_byte( 0xff );
      put_rx_byte( GUS_MCMD_WAIT );
      put_rx_byte( midi_rx_ticks >> 24 );
      put_rx_byte( midi_rx_ticks >> 16 );
      put_rx_byte( midi_rx_ticks >> 8 );
      put_rx_byte( midi_rx_ticks );
      midi_rx_ticks = 0;
      midi_rx_used += 8;
    }

  if ( midi_rx_used + count + 3 > GUS_MIDI_QUEUE_SIZE )
    {
      STI( &flags );
      PRINTK( "gus: MIDI Rx buffer overflow - command discarded\n" );
      return; 
    }
  count0 = count;
  buffer0 = buffer;
  put_rx_byte( count >> 8 );
  put_rx_byte( count );
  put_rx_byte( (unsigned char)device );
  while ( count0-- )
    put_rx_byte( *buffer0++ );
  midi_rx_used += count + 3;
  STI( &flags );
  if ( GETLOCK( midi_master_card, midi_in ) & WK_SLEEP )
    {
      GETLOCK( midi_master_card, midi_in ) &= ~WK_SLEEP;
      WAKEUP( midi_master_card, midi_in );
    }

  /*
   *  MIDI THRU part
   */

  {
    short i;
    unsigned char channel;
    struct GUS_STRU_MIDI *sender, *receiver;
    
    sender = &gus_cards[ ( device >> 4 ) & 0x0f ] -> midi[ device & 0x0f ];
    channel = *buffer & 0x0f;
#if 0
    printk( "thru_mask = 0x%x\n", sender -> thru_mask );
#endif
    if ( ( ( *buffer & 0xf0 ) != 0xf0 ) &&
         ( sender -> thru_mask & ( 1 << channel ) ) )
      {
        for ( i = 0; i <= GUS_MIDID_LAST; i++ )
          {
            unsigned char d_device;
            unsigned char d_channel;
            unsigned char d_velocity;
            gus_card_t *d_card;
            
            d_device = sender -> thru[ channel ][ i ].device;
            if ( d_device == 0xff ) continue;		/* nothing */
            d_channel = sender -> thru[ channel ][ i ].channel;
            d_velocity = sender -> thru[ channel ][ i ].velocity;
#if 0
            printk( "thru - d_device = 0x%x, d_channel = 0x%x, cmd = 0x%x\n", d_device, d_channel, buffer[ 0 ] );
#endif
            d_card = gus_cards[ ( d_device >> 4 ) & 0x0f ];
            receiver = &d_card -> midi[ d_device & 0x0f ];
            *buffer &= 0xf0;
            if ( d_velocity < 128 )
              switch ( *buffer ) {
                case GUS_MCMD_NOTE_OFF:
                case GUS_MCMD_NOTE_ON:
                  if ( buffer[ 2 ] != 0 )
                    buffer[ 2 ] = d_velocity;
                  break;
              }
            *buffer |= d_channel & 0x0f;
            receiver -> putcmd( d_card, buffer, count );
          }
      }
  }
}

/*
 *
 */

static void purge_tx( void )
{
  midi_tx_used = midi_tx_head = midi_tx_tail = 0;
}

static void put_tx_byte( unsigned char b )
{
  midi_tx_buf[ midi_tx_head++ ] = b;
  midi_tx_head %= midi_tx_size;
}

static unsigned char get_tx_byte( void )
{
  unsigned char result;

  result = midi_tx_buf[ midi_tx_tail++ ];
  midi_tx_tail %= midi_tx_size;
  return result;
}

static int gus_midi_tx_command( unsigned int device, unsigned char *buffer, unsigned int count, int space )
{
  unsigned int count1;
  
  if ( device != GUS_MIDID_COMMON )
    {
      if ( device >> 4 >= gus_cards_count ) return -ENODEV;
      if ( ( device & 0x0f ) > GUS_MIDID_LAST ) return -ENODEV;
      if ( !(gus_cards[ device >> 4 ] -> midi[ device & 0x0f ].flags & GUS_MIDIF_USED_OUT) ) return -ENODEV;
    }
  if ( count + 3 > GUS_MIDI_QUEUE_SIZE ) return -ENOMEM;
  while ( midi_tx_used + count + 3 > GUS_MIDI_QUEUE_SIZE )
    {
      GETLOCK( midi_master_card, midi_out ) |= WK_SLEEP;
      SLEEP( midi_master_card, midi_out, HZ * 60 );
      GETLOCK( midi_master_card, midi_out ) &= ~WK_SLEEP;
      if ( TABORT( midi_out ) ) return 0;
      if ( TIMEOUT( midi_master_card, midi_out ) )
        {
          PRINTK( "gus: midi timeout\n" );
          return -EIO;
        }
    }
  put_tx_byte( count >> 8 );
  put_tx_byte( count );
  put_tx_byte( device );
  count1 = count;
  if ( space == SP_USER )
    {
      while ( count1-- )
        put_tx_byte( get_fs_byte( buffer++ ) );
    }
   else
    {
      while ( count1-- )
        put_tx_byte( *buffer++ );
    }
  midi_tx_used += count + 3;
  if ( midi_timer_state != MIDI_TIMER_RUNNING )
    process_output_events( 1 );
  return 0;
}

/*
 *  -------------
 */

static void process_output_events( int now )
{
  unsigned long flags;
  int device, count, count1, end;
  gus_card_t *card;
  unsigned char *buf;
  struct GUS_STRU_MIDI *midi;

  if ( midi_mode & MIDI_MODE_OUT )
    {
      if ( midi_tx_ticks > 0 )
        {
          if ( !now )
            midi_tx_ticks--;
          return;
        }
      end = 0;
#if 0
      if ( midi_tx_used )
        printk( "midi out - used = %i\n", midi_tx_used );
#endif
      while ( midi_tx_used > 2 && !end )
        {
          count = get_tx_byte() << 8;
          count |= get_tx_byte();
          device = get_tx_byte();
          if ( device != GUS_MIDID_COMMON )
            {
              if ( count <= GUS_MIDI_CMD_SIZE )
                {
                  card = gus_cards[ device >> 4 ];
                  midi = &card -> midi[ device & 0x0f ];
                  buf = midi_buf_cmd;
                  count1 = count;
                  while ( count1-- > 0 )
                    *buf++ = get_tx_byte();
                  if ( !midi )
                    PRINTK( "gus: unknown midi device 0x%x\n", device );
                   else
                    midi -> putcmd( card, midi_buf_cmd, count );
                }
            }
           else
            {
              switch ( get_tx_byte() ) {
                case GUS_MCMD_WAIT:
                  midi_tx_ticks = get_tx_byte() << 24;
                  midi_tx_ticks |= get_tx_byte() << 16;
                  midi_tx_ticks |= get_tx_byte() << 8;
                  midi_tx_ticks |= get_tx_byte();
#if 0
                  printk( "midi_tx_ticks = %i\n", midi_tx_ticks );
#endif
#if 0
		  midi_tx_ticks *= 4;
#endif
                  if ( midi_tx_ticks ) 
                    {
                      midi_tx_ticks--;
                      end = 1;
                    }
                  break;
                case GUS_MCMD_TEMPO:
                  count1 = get_tx_byte() << 8;
                  count1 |= get_tx_byte();
                  gus_timer_tempo( midi_master_card, count1 );
                  break;
                case GUS_MCMD_TBASE:
                  count1 = get_tx_byte() << 8;
                  count1 |= get_tx_byte();
                  gus_timer_base( midi_master_card, count1 );
                  break;
                case GUS_MCMD_ECHO:
                  count1 = count - 1;
#if 0
		  printk( "echo - size = %i\n", count1 );
#endif
		  if ( (midi_mode & MIDI_MODE_IN) && 
		       midi_rx_used + count + 3 <= GUS_MIDI_QUEUE_SIZE )
		    {
		      CLI( &flags );
		      put_rx_byte( count >> 8 );
		      put_rx_byte( count );
		      put_rx_byte( GUS_MIDID_COMMON );
		      put_rx_byte( GUS_MCMD_ECHO );
                      while ( count1-- )
                        put_rx_byte( get_tx_byte() );
                      midi_rx_used += count + 3;
                      STI( &flags );
                      if ( GETLOCK( midi_master_card, midi_in ) & WK_SLEEP )
                        {
                          GETLOCK( midi_master_card, midi_in ) &= ~WK_SLEEP;
                          WAKEUP( midi_master_card, midi_in );
                        }
                    }
                   else
                    while ( count1-- ) get_tx_byte();
                  break;
              }    
            }
          midi_tx_used -= count + 3;
        }
      if ( GETLOCK( midi_master_card, midi_out ) & WK_SLEEP )
        {
          if ( ( midi_flush_flag && !midi_tx_ticks && !midi_tx_used ) ||
               ( midi_tx_used < midi_tx_threshold || midi_rx_used > 0 ) )
            {
              GETLOCK( midi_master_card, midi_out ) &= ~WK_SLEEP;
              WAKEUP( midi_master_card, midi_out );
            }
        }
    }
}

void gus_midi_interrupt( void )
{
#if 0
  PRINTK( "gus_midi_interrupt - start - mode = 0x%x\n", midi_mode );
#endif
  if ( midi_mode & MIDI_MODE_IN )
    midi_rx_ticks++;
  if ( midi_mode & MIDI_MODE_OUT )
    process_output_events( 0 );
#if 0
  PRINTK( "gus_midi_interrupt - end\n" );
#endif
}

/*
 *
 */
 
static int midi_flush( void )
{
  int res = 0;

  if ( !midi_master_card ) return -ENODEV;
  if ( midi_timer_state != MIDI_TIMER_RUNNING )
    {
      purge_tx();
      midi_flush_flag = 0;
      return 0;
    }
  midi_flush_flag = 1;
  while ( midi_tx_ticks || midi_tx_used )
    {
      GETLOCK( midi_master_card, midi_out ) |= WK_SLEEP;
      SLEEP( midi_master_card, midi_out, 60 * HZ );
      GETLOCK( midi_master_card, midi_out ) &= ~WK_SLEEP;
      if ( TABORT( midi_out ) ) break;
      if ( TIMEOUT( midi_master_card, midi_out ) )
        {
          printk( "gus: midi flush timeout?\n" );
          res = -EIO;
          break;
        }
    }
  midi_flush_flag = 0;
  return res;
}

static int midi_realtime( struct GUS_STRU_MIDI_REALTIME *realtime )
{
  struct GUS_STRU_MIDI_REALTIME srealtime;
  unsigned int count, count1;
  int cmd_size, dev; 
  unsigned char *buf, *buf1;
  gus_card_t *card;
  struct GUS_STRU_MIDI *midi;

  if ( !midi_master_card ) return -ENODEV;
  if ( VERIFY_AREA( VERIFY_READ, realtime, sizeof( *realtime ) ) ) return -EIO;
  MEMCPY_FROMFS( &srealtime, realtime, sizeof( srealtime ) );
  realtime = &srealtime;

  buf = realtime -> buffer;
  count = realtime -> count;

  while ( count > 2 )
    {
      cmd_size = get_fs_byte( buf++ ) << 8;
      cmd_size |= get_fs_byte( buf++ );
      if ( count < cmd_size )
        {
          PRINTK( "gus_midi_realtime: unexpected count\n" );
          return -EINVAL;
        }
      dev = get_fs_byte( buf++ );
      if ( dev != GUS_MIDID_COMMON && dev < ( GUS_CARDS << 4 ) )
        {
          card = gus_cards[ dev >> 4 ];
          if ( card && cmd_size <= GUS_MIDI_CMD_SIZE )
            {
              midi = &card -> midi[ dev & 0x0f ];
              buf1 = midi_buf_realtime;
              count1 = cmd_size;
              while ( count1-- > 0 )
                *buf1++ = get_fs_byte( buf++ );
              midi -> putcmd( card, midi_buf_realtime, cmd_size );
            }
           else
            buf += cmd_size;
        }
       else
        buf += cmd_size;
      count -= cmd_size + 3;
    }
  if ( count )
    PRINTK( "gus_midi_realtime: unexpected count\n" );
  
  return 0;
}

static int midi_abort( void )
{
  purge_tx();
  purge_rx();
  return 0;
}

/*
 *  -------------
 */
 
static gus_card_t *midi_verify_card;
 
static struct GUS_STRU_MIDI *midi_verify_device( unsigned char device )
{
  if ( device >> 4 >= gus_cards_count ) return NULL;
  if ( ( device & 0x0f ) > GUS_MIDID_LAST ) return NULL;
  return &(midi_verify_card = gus_cards[ device >> 4 ]) -> midi[ device ];
}
 
static int midi_set_thru( struct GUS_STRU_MIDI_THRU *thru )
{
  struct GUS_STRU_MIDI_THRU sthru;
  struct GUS_STRU_MIDI *sender, *receiver;
  int i, j;

  if ( VERIFY_AREA( VERIFY_READ, thru, sizeof( *thru ) ) ) return -EIO;
  MEMCPY_FROMFS( &sthru, thru, sizeof( sthru ) );
  thru = &sthru;

  sender = midi_verify_device( thru -> device );
  if ( !sender ) return -ENODEV;
  if ( !( sender -> flags & GUS_MIDIF_USED_IN ) ) return -EINVAL;
  for ( i = 0; i < 16; i++ )
    for ( j = 0; j <= GUS_MIDID_LAST; j++ )
      {
        unsigned char d_device;
        unsigned char d_channel;
        unsigned char d_velocity;
        
        d_device = thru -> routing[ i ][ j ].device;
        d_channel = thru -> routing[ i ][ j ].channel;
        d_velocity = thru -> routing[ i ][ j ].velocity;
        if ( d_device == 0xff ) continue;		/* nothing */
        if ( d_device >> 4 >= gus_cards_count ) return -ENODEV;
        if ( ( d_device & 0x0f ) > GUS_MIDID_LAST ) return -ENODEV;
        if ( d_channel & 0xf0 ) return -EINVAL;
        if ( d_velocity > 127 && d_velocity < 255 ) return -EINVAL;
        receiver = &gus_cards[ d_device >> 4 ] -> midi[ d_device & 0x0f ];
        if ( !( receiver -> flags & GUS_MIDIF_USED_OUT ) ) return -EINVAL;
      }
  sender -> thru_mask = 0;
  MEMSET( &sender -> thru, 0xff, sizeof( receiver -> thru ) );
  for ( i = 0; i < 16; i++ )
    for ( j = 0; j <= GUS_MIDID_LAST; j++ )
      {
        unsigned char d_device;
        
        d_device = thru -> routing[ i ][ j ].device;
        if ( d_device == 0xff ) continue;		/* nothing */
#if 0
        printk( "setup: d_device = 0x%x, channel = 0x%x, velocity = 0x%x\n", d_device, thru -> routing[ i ][ j ].channel, thru -> routing[ i ][ j ].velocity );
#endif
        sender -> thru_mask |= 1 << i;
        sender -> thru[ i ][ j ].device = d_device;
        sender -> thru[ i ][ j ].channel = thru -> routing[ i ][ j ].channel;
        sender -> thru[ i ][ j ].velocity = thru -> routing[ i ][ j ].velocity;
      }
  return 0;
}

static int midi_get_thru( struct GUS_STRU_MIDI_THRU *thru )
{
  struct GUS_STRU_MIDI_THRU sthru;
  struct GUS_STRU_MIDI *sender;
  int i, j;

  if ( VERIFY_AREA( VERIFY_READ, thru, sizeof( *thru ) ) ) return -EIO;
  MEMCPY_FROMFS( &sthru, thru, sizeof( sthru ) );
  
  sender = midi_verify_device( sthru.device );
  if ( !sender ) return -ENODEV;
  MEMSET( &sthru.routing, 0xff, sizeof( sthru.routing ) );  
  for ( i = 0; i < 16; i++ )
    if ( sender -> thru_mask & ( 1 << i ) )
      for ( j = 0; j <= GUS_MIDID_LAST; j++ )
        {
          sthru.routing[ i ][ j ].device = sender -> thru[ i ][ j ].device;
          sthru.routing[ i ][ j ].channel = sender -> thru[ i ][ j ].channel;
          sthru.routing[ i ][ j ].velocity = sender -> thru[ i ][ j ].velocity;
        }
        
  if ( VERIFY_AREA( VERIFY_WRITE, thru, sizeof( *thru ) ) ) return -EIO;
  MEMCPY_TOFS( thru, &sthru, sizeof( sthru ) );
  return 0;
}

static int midi_preload_bank( struct GUS_STRU_MIDI_PRELOAD_BANK *bank )
{
  struct GUS_STRU_MIDI_PRELOAD_BANK sbank;
  
  if ( VERIFY_AREA( VERIFY_READ, bank, sizeof( *bank ) ) ) return -EIO;
  MEMCPY_FROMFS( &sbank, bank, sizeof( sbank ) );
  bank = &sbank;

  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( bank -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( bank -> device ) ) return -ENODEV;
  
  gus_gf1_daemon_preload( midi_verify_card, bank -> name );
  return 0;
}

static int midi_preload_instruments( struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS *instr )
{
  int i, j, in;
  struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS sinstr;
  
  if ( VERIFY_AREA( VERIFY_READ, instr, sizeof( *instr ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinstr, instr, sizeof( sinstr ) );
  instr = &sinstr;

  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( instr -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( instr -> device ) ) return -ENODEV;
  
  if ( !instr -> instruments_used || instr -> instruments_used > 64 ) return -EINVAL;
  for ( i = 0; i < instr -> instruments_used; i++ )
    {
      struct GUS_STRU_INSTRUMENT *pinstr;
    
      in = instr -> instruments[ i ];
      pinstr = gus_instrument_look( &midi_verify_card -> gf1.mem_alloc, in );
      if ( pinstr ) continue;
      for ( j = 0; j < i; j++ )
        if ( instr -> instruments[ j ] == in ) break;
      if ( j < i ) continue;
      gus_gf1_daemon_request_sample( midi_verify_card, in );
    }
  gus_gf1_daemon_preload( midi_verify_card, (char *)NULL );
  return 0;
}

static int midi_timer_base( int base )
{ 
  if ( !midi_master_card ) return -EIO;
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  gus_timer_base( midi_master_card, base );
#if 0
  printk( "ok.. new tempo = %i\n", midi_master_card -> gf1.timer_midi_tempo );
#endif
  return 0;
}

static int midi_timer_tempo( int tempo )
{ 
  if ( !midi_master_card ) return -EIO;
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  gus_timer_tempo( midi_master_card, tempo );
#if 0
  printk( "ok.. new tempo = %i\n", midi_master_card -> gf1.timer_midi_tempo );
#endif
  return 0;
}

static int midi_emulation_get( struct GUS_STRU_MIDI_EMULATION *emul )
{
  struct GUS_STRU_MIDI_EMULATION semul;
  
  if ( VERIFY_AREA( VERIFY_READ, emul, sizeof( *emul ) ) ) return -EIO;
  MEMCPY_FROMFS( &semul, emul, sizeof( semul ) );
  
  if ( ( semul.device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( semul.device ) ) return -ENODEV;
  semul.emulation = midi_verify_card -> gf1.midi_emul;
  
  if ( VERIFY_AREA( VERIFY_WRITE, emul, sizeof( *emul ) ) ) return -EIO;
  MEMCPY_TOFS( emul, &semul, sizeof( semul ) );
  return 0;
}

static int midi_emulation_set( struct GUS_STRU_MIDI_EMULATION *emul )
{
  struct GUS_STRU_MIDI_EMULATION semul;
  
  if ( VERIFY_AREA( VERIFY_READ, emul, sizeof( *emul ) ) ) return -EIO;
  MEMCPY_FROMFS( &semul, emul, sizeof( semul ) );
  
  if ( ( semul.device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( semul.device ) ) return -ENODEV;
  if ( semul.emulation == GUS_MIDI_EMUL_AUTO )
    semul.emulation = midi_verify_card -> gf1.default_midi_emul;
  if ( semul.emulation > GUS_MIDI_EMUL_MT32 ) return -EINVAL;
  if ( semul.emulation != midi_verify_card -> gf1.midi_emul )
    {
      midi_verify_card -> gf1.midi_emul = semul.emulation;
      gus_gf1_daemon_midi_emul_change( midi_verify_card );
    }
  return 0;
}

static int midi_memory_reset( struct GUS_STRU_MIDI_MEMORY_RESET *reset )
{
  struct GUS_STRU_MIDI_MEMORY_RESET sreset;
  
  if ( VERIFY_AREA( VERIFY_READ, reset, sizeof( *reset ) ) ) return -EIO;
  MEMCPY_FROMFS( &sreset, reset, sizeof( sreset ) );
  reset = &sreset;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( reset -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( reset -> device ) ) return -ENODEV;
  return gus_gf1_midi_engine_reset( midi_verify_card );
}

static int midi_memory_test( struct GUS_STRU_MIDI_INSTRUMENT *minstr )
{
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  if ( VERIFY_AREA( VERIFY_READ, minstr, sizeof( *minstr ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinstr, minstr, sizeof( sinstr ) );
  minstr = &sinstr;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( minstr -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( minstr -> device ) ) return -ENODEV;
  return gus_memory_alloc( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, minstr -> instrument, MTST_ONE, SP_USER );
}

static int midi_memory_alloc( struct GUS_STRU_MIDI_INSTRUMENT *minstr )
{
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  if ( VERIFY_AREA( VERIFY_READ, minstr, sizeof( *minstr ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinstr, minstr, sizeof( sinstr ) );
  minstr = &sinstr;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( minstr -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( minstr -> device ) ) return -ENODEV;
  return gus_memory_alloc( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, minstr -> instrument, MTST_NONE, SP_USER );
}

static int midi_memory_free( struct GUS_STRU_MIDI_INSTRUMENT *minstr )
{
  struct GUS_STRU_MIDI_INSTRUMENT sinstr;
  
  if ( VERIFY_AREA( VERIFY_READ, minstr, sizeof( *minstr ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinstr, minstr, sizeof( sinstr ) );
  minstr = &sinstr;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( minstr -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( minstr -> device ) ) return -ENODEV;
  return gus_memory_free( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, minstr -> instrument, SP_USER );
}

static int midi_memory_pack( struct GUS_STRU_MIDI_MEMORY_PACK *pack )
{
  struct GUS_STRU_MIDI_MEMORY_PACK spack;
  
  if ( VERIFY_AREA( VERIFY_READ, pack, sizeof( *pack ) ) ) return -EIO;
  MEMCPY_FROMFS( &spack, pack, sizeof( spack ) );
  pack = &spack;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( pack -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( pack -> device ) ) return -ENODEV;
  return gus_memory_pack( midi_verify_card, &midi_verify_card -> gf1.mem_alloc );
}

static int midi_memory_block_alloc( struct GUS_STRU_MIDI_MEMORY_BLOCK *block )
{
  struct GUS_STRU_MIDI_MEMORY_BLOCK sblock;
  
  if ( VERIFY_AREA( VERIFY_READ, block, sizeof( *block ) ) ) return -EIO;
  MEMCPY_FROMFS( &sblock, block, sizeof( sblock ) );
  block = &sblock;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( block -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( block -> device ) ) return -ENODEV;
  return gus_memory_block_alloc( midi_verify_card, block -> block, GUS_MEMORY_LOCK_USER, SP_USER );
}

static int midi_memory_block_free( struct GUS_STRU_MIDI_MEMORY_BLOCK *block )
{
  struct GUS_STRU_MIDI_MEMORY_BLOCK sblock;
  
  if ( VERIFY_AREA( VERIFY_READ, block, sizeof( *block ) ) ) return -EIO;
  MEMCPY_FROMFS( &sblock, block, sizeof( sblock ) );
  block = &sblock;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( block -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( block -> device ) ) return -ENODEV;
  return gus_memory_block_free( midi_verify_card, block -> block, GUS_MEMORY_LOCK_USER, SP_USER );
}

static int midi_memory_get_name( struct GUS_STRU_MIDI_INSTRUMENT_NAME *instr )
{
  struct GUS_STRU_MIDI_INSTRUMENT_NAME sinstr;
  
  if ( VERIFY_AREA( VERIFY_READ, instr, sizeof( *instr ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinstr, instr, sizeof( sinstr ) );
  instr = &sinstr;
  
  if ( ( instr -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( instr -> device ) ) return -ENODEV;
  return gus_memory_get_name( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, instr -> name, SP_USER );
}

static int midi_memory_dump( struct GUS_STRU_MIDI_MEMORY_DUMP *dump )
{
  struct GUS_STRU_MIDI_MEMORY_DUMP sdump;
  
  if ( VERIFY_AREA( VERIFY_READ, dump, sizeof( *dump ) ) ) return -EIO;
  MEMCPY_FROMFS( &sdump, dump, sizeof( sdump ) );
  dump = &sdump;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( dump -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( dump -> device ) ) return -ENODEV;
  return gus_memory_dump( midi_verify_card, dump -> dump, SP_USER );
}

static int midi_memory_list( struct GUS_STRU_MIDI_MEMORY_LIST *list )
{
  struct GUS_STRU_MIDI_MEMORY_LIST slist;
  
  if ( VERIFY_AREA( VERIFY_READ, list, sizeof( *list ) ) ) return -EIO;
  MEMCPY_FROMFS( &slist, list, sizeof( slist ) );
  list = &slist;
  
  if ( midi_timer_state == MIDI_TIMER_RUNNING ) return -EBUSY;
  if ( ( list -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( list -> device ) ) return -ENODEV;
  return gus_memory_list( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, list -> list, SP_USER );
}

static int midi_card_info( struct GUS_STRU_MIDI_CARD_INFO *info )
{
  struct GUS_STRU_MIDI_CARD_INFO sinfo;
  
  if ( VERIFY_AREA( VERIFY_READ, info, sizeof( *info ) ) ) return -EIO;
  MEMCPY_FROMFS( &sinfo, info, sizeof( sinfo ) );
  info = &sinfo;
  
  if ( ( info -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( info -> device ) ) return -ENODEV;
  return gus_info( midi_verify_card, &midi_verify_card -> gf1.mem_alloc, info -> info );
}

static int midi_effect_reset( struct GUS_STRU_MIDI_EFFECT_RESET *reset )
{
  struct GUS_STRU_MIDI_EFFECT_RESET sreset;
  
  if ( VERIFY_AREA( VERIFY_READ, reset, sizeof( *reset ) ) ) return -EIO;
  MEMCPY_FROMFS( &sreset, reset, sizeof( sreset ) );
  reset = &sreset;
  
  if ( ( reset -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( reset -> device ) ) return -ENODEV;
  return gus_effects_reset( midi_verify_card );
}

static int midi_effect_setup( struct GUS_STRU_MIDI_EFFECT *effect )
{
  struct GUS_STRU_MIDI_EFFECT seffect;
  
  if ( VERIFY_AREA( VERIFY_READ, effect, sizeof( *effect ) ) ) return -EIO;
  MEMCPY_FROMFS( &seffect, effect, sizeof( seffect ) );
  effect = &seffect;
  
  if ( ( effect -> device & 0x0f ) != GUS_MIDID_SYNTH || !midi_verify_device( effect -> device ) ) return -ENODEV;
  return gus_effects_setup( midi_verify_card, effect -> effect, SP_USER );
}

static int midi_devices( struct GUS_STRU_MIDI_DEVICES *udevices )
{
  int i;
  struct GUS_STRU_MIDI_DEVICES devices;

  if ( VERIFY_AREA( VERIFY_WRITE, udevices, sizeof( *udevices ) ) ) return -EIO;
  MEMSET( &devices, 0, sizeof( devices ) );
  devices.count = gus_cards_count * 2;
  for ( i = 0; i < devices.count; i++ )
    devices.devices[ i ] = ( ( i & ~1 ) << 3 ) | ( i & 1 );
  MEMCPY_TOFS( udevices, &devices, sizeof( devices ) );
  return 0;  
}

static int midi_device_info( struct GUS_STRU_MIDI_DEVICE_INFO *udevice )
{
  int cnumber, dnumber;
  gus_card_t *card;
  struct GUS_STRU_MIDI_DEVICE_INFO device;
  
  if ( VERIFY_AREA( VERIFY_READ, udevice, sizeof( *udevice ) ) ) return -EIO;
  if ( VERIFY_AREA( VERIFY_WRITE, udevice, sizeof( *udevice ) ) ) return -EIO;
  MEMCPY_FROMFS( &device, udevice, sizeof( device ) );
  if ( device.device >= 0x80 ) return -EINVAL;
  cnumber = device.device >> 4;
  if ( cnumber >= gus_cards_count ) return -EINVAL;
  card = gus_cards[ cnumber ];
  dnumber = device.device & 0x0f;
  switch ( dnumber ) {
    case GUS_MIDID_UART:
      strcpy( device.name, "MIDI UART 6850" );
      device.cap = GUS_MIDI_CAP_INPUT | GUS_MIDI_CAP_OUTPUT;
      break;
    case GUS_MIDID_SYNTH:
      strcpy( device.name, card -> gf1.enh_mode ? "InterWave MIDI emulation" : "GF1 MIDI emulation" );
      device.cap = GUS_MIDI_CAP_OUTPUT | GUS_MIDI_CAP_SYNTH | GUS_MIDI_CAP_MEMORY;
      break;
    default:
      return -EINVAL;
  }
  MEMCPY_TOFS( udevice, &device, sizeof( device ) );
  return 0;
}


/*
 *  -------------
 */

int gus_open_midi( struct file *file )
{
  unsigned short mode;
  short i;

  if ( midi_mode != MIDI_MODE_NONE ) return -EBUSY;
  mode = midi_file_flags( file -> f_flags );
  if ( midi_mode == MIDI_MODE_NONE )
    {
      for ( i = 0; i < gus_cards_count; i++ )
        gus_cards[ i ] -> gf1.mode |= GF1_MODE_MIDI;
    }
  if ( mode & MIDI_MODE_IN )
    {
      midi_rx_buf = (char *)gus_malloc( midi_rx_size = GUS_MIDI_QUEUE_SIZE );
      if ( midi_rx_buf == NULL ) return -ENOMEM;
      purge_rx();
    }
  if ( mode & MIDI_MODE_OUT )
    {
      midi_tx_buf = (char *)gus_malloc( midi_tx_size = GUS_MIDI_QUEUE_SIZE );
      midi_buf_cmd = (char *)gus_malloc( GUS_MIDI_CMD_SIZE );
      midi_buf_realtime = (char *)gus_malloc( GUS_MIDI_CMD_SIZE );
      if ( midi_tx_buf == NULL || midi_buf_cmd == NULL || midi_buf_realtime == NULL )
        {
          gus_free( midi_tx_buf, GUS_MIDI_QUEUE_SIZE );
          gus_free( midi_buf_cmd, GUS_MIDI_CMD_SIZE );
          gus_free( midi_buf_realtime, GUS_MIDI_CMD_SIZE );
          return -ENOMEM;
        }
      purge_tx();
      midi_flush_flag = 0;
      midi_tx_threshold = ( midi_tx_size * 80 ) / 100;
    }
  midi_tx_ticks = midi_rx_ticks = 0;
  midi_mode |= mode;
  midi_timer_state = MIDI_TIMER_NOT_LOADED;
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_midi( struct file *file )
{
  unsigned short mode;
  short i;
  
  mode = midi_file_flags( file -> f_flags );
  midi_close_all( mode );
  if ( mode & MIDI_MODE_IN )
    gus_free( midi_rx_buf, GUS_MIDI_QUEUE_SIZE );
  if ( mode & MIDI_MODE_OUT )
    {
      gus_free( midi_tx_buf, GUS_MIDI_QUEUE_SIZE );
      gus_free( midi_buf_cmd, GUS_MIDI_CMD_SIZE );
      gus_free( midi_buf_realtime, GUS_MIDI_CMD_SIZE );
    }
  midi_mode &= ~mode;
  if ( midi_mode == MIDI_MODE_NONE )
    {
      for ( i = 0; i < gus_cards_count; i++ )
        gus_cards[ i ] -> gf1.mode &= ~GF1_MODE_MIDI;
    }
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_midi( struct file *file, unsigned int cmd, unsigned long arg )
{
  unsigned short mode;
  
#if 0
  printk( "ioctl = 0x%x\n", cmd );
#endif
  mode = midi_file_flags( file -> f_flags );
  switch ( cmd ) {
    case GUS_MIDI_CARDS:
      return IOCTL_OUT( arg, gus_cards_count );
    case GUS_MIDI_DEVICE_OPEN:
      return midi_open_all( (struct GUS_STRU_MIDI_OPEN *)arg, mode );
    case GUS_MIDI_DEVICE_CLOSE:
      return midi_close_all( mode );
    case GUS_MIDI_TIMER_BASE:
      return midi_timer_base( IOCTL_IN( arg ) );
    case GUS_MIDI_TIMER_TEMPO:
      return midi_timer_tempo( IOCTL_IN( arg ) );
    case GUS_MIDI_TIMER_START:
      midi_tx_ticks = 0;
      if ( !midi_master_card ) return -EINVAL;
      midi_timer_state = MIDI_TIMER_RUNNING;
      return gus_timer_ioctl( midi_master_card, GUS_IOCTL_TIMER_START, 0 );
    case GUS_MIDI_TIMER_STOP:
      midi_timer_state = MIDI_TIMER_STOPED;
      return gus_timer_ioctl( midi_master_card, GUS_IOCTL_TIMER_STOP, 0 );
    case GUS_MIDI_TIMER_CONTINUE:
      if ( !midi_master_card ) return -EINVAL;
      midi_timer_state = MIDI_TIMER_RUNNING;
      return gus_timer_ioctl( midi_master_card, GUS_IOCTL_TIMER_CONTINUE, 0 );
    case GUS_MIDI_FLUSH:
      return midi_flush();
    case GUS_MIDI_REALTIME:
      return midi_realtime( (struct GUS_STRU_MIDI_REALTIME *)arg );
    case GUS_MIDI_ABORT:
      return midi_abort();
    case GUS_MIDI_SET_THRU:
      return midi_set_thru( (struct GUS_STRU_MIDI_THRU *)arg );
    case GUS_MIDI_GET_THRU:
      return midi_get_thru( (struct GUS_STRU_MIDI_THRU *)arg );
    case GUS_MIDI_PRELOAD_BANK:
      return midi_preload_bank( (struct GUS_STRU_MIDI_PRELOAD_BANK *)arg );
    case GUS_MIDI_PRELOAD_INSTRUMENTS:
      return midi_preload_instruments( (struct GUS_STRU_MIDI_PRELOAD_INSTRUMENTS *)arg );
    case GUS_MIDI_THRESHOLD:
      {
        int tmp = IOCTL_IN( arg );
        
        if ( tmp < 1 ) tmp = 1;
        if ( tmp > 100 ) tmp = 100;
        midi_tx_threshold = ( midi_tx_size * tmp ) / 100;
        return 0;
      }
    case GUS_MIDI_GET_EMULATION:
      return midi_emulation_get( (struct GUS_STRU_MIDI_EMULATION *)arg );
    case GUS_MIDI_SET_EMULATION:
      return midi_emulation_set( (struct GUS_STRU_MIDI_EMULATION *)arg );

    case GUS_MIDI_MEMORY_RESET:
      return midi_memory_reset( (struct GUS_STRU_MIDI_MEMORY_RESET *)arg );
    case GUS_MIDI_MEMORY_TEST:
      return midi_memory_test( (struct GUS_STRU_MIDI_INSTRUMENT *)arg );
    case GUS_MIDI_MEMORY_ALLOC:
      return midi_memory_alloc( (struct GUS_STRU_MIDI_INSTRUMENT *)arg );
    case GUS_MIDI_MEMORY_FREE:
      return midi_memory_free( (struct GUS_STRU_MIDI_INSTRUMENT *)arg );
    case GUS_MIDI_MEMORY_PACK:
      return midi_memory_pack( (struct GUS_STRU_MIDI_MEMORY_PACK *)arg );
    case GUS_MIDI_MEMORY_BALLOC:
      return midi_memory_block_alloc( (struct GUS_STRU_MIDI_MEMORY_BLOCK *)arg );
    case GUS_MIDI_MEMORY_BFREE:
      return midi_memory_block_free( (struct GUS_STRU_MIDI_MEMORY_BLOCK *)arg );
    case GUS_MIDI_MEMORY_GET_NAME:
      return midi_memory_get_name( (struct GUS_STRU_MIDI_INSTRUMENT_NAME *)arg );
    case GUS_MIDI_MEMORY_DUMP:
      return midi_memory_dump( (struct GUS_STRU_MIDI_MEMORY_DUMP *)arg );
    case GUS_MIDI_MEMORY_LIST:
      return midi_memory_list( (struct GUS_STRU_MIDI_MEMORY_LIST *)arg );

    case GUS_MIDI_CARD_INFO:
      return midi_card_info( (struct GUS_STRU_MIDI_CARD_INFO *)arg );
  
    case GUS_MIDI_EFFECT_RESET:
      return midi_effect_reset( (struct GUS_STRU_MIDI_EFFECT_RESET *)arg );
    case GUS_MIDI_EFFECT_SETUP:
      return midi_effect_setup( (struct GUS_STRU_MIDI_EFFECT *)arg );
      
    case GUS_MIDI_DEVICES:
      return midi_devices( (struct GUS_STRU_MIDI_DEVICES *)arg );
    case GUS_MIDI_DEVICE_INFO:
      return midi_device_info( (struct GUS_STRU_MIDI_DEVICE_INFO *)arg );
  }
  return -EINVAL;
}

int gus_read_midi( struct file *file, char *buf, int count )
{
  unsigned long flags;
  int size, count0, count1;
  
  if ( !midi_master_card ) return -ENODEV;
  size = 0;
  while ( count > 0 && midi_rx_used > 0 )
    {
      CLI( &flags );
      count0 = count1 = ( get_rx_byte() << 8 ) | get_rx_byte();
      if ( count0 + 3 > count )
        {
          back_rx_byte();
          back_rx_byte();
          STI( &flags );
          break;
        }
      STI( &flags );
      put_fs_byte( count0 >> 8, buf++ );
      put_fs_byte( count0, buf++ );
      put_fs_byte( get_rx_byte(), buf++ );	/* device */
      while ( count1-- > 0 )
        put_fs_byte( get_rx_byte(), buf++ );
      count0 += 3;
      midi_rx_used -= count0;
      size += count0;
      count -= count0;
    }
  return size;
}

int gus_write_midi( struct file *file, char *buf, int count )
{
  int cmd_size, i, dev, old_count;
  
  if ( !midi_master_card ) return -ENODEV;
  old_count = count;
  while ( count > 2 )
    {
      cmd_size = get_fs_byte( buf++ ) << 8;
      cmd_size |= get_fs_byte( buf++ );
      if ( count < cmd_size )
        {
          PRINTK( "gus_write_midi: unexpected count\n" );
          return -EINVAL;
        }
      dev = get_fs_byte( buf++ );
      if ( ( i = gus_midi_tx_command( dev, buf, cmd_size, SP_USER ) ) < 0 )
        return i;
      count -= cmd_size + 3;
      buf += cmd_size;
    }
  return old_count;
}

#ifdef GUS_POLL
unsigned int gus_poll_midi( struct file *file, poll_table *wait )
{
  unsigned long flags;
  unsigned int mask;
  
  CLI( &flags );
  GETLOCK( midi_master_card, midi_in ) |= WK_SLEEP;
  SLEEP_POLL( midi_master_card, midi_in, wait );
  GETLOCK( midi_master_card, midi_out ) |= WK_SLEEP;
  SLEEP_POLL( midi_master_card, midi_out, wait );
  STI( &flags );
  
  mask = 0;
  if ( midi_rx_used )
    mask |= POLLIN | POLLRDNORM;
  if ( midi_rx_used <= midi_tx_threshold )
    mask |= POLLOUT | POLLWRNORM;

  return mask;
}
#else
int gus_select_midi( struct file *file, int sel_type, select_table *wait )
{
  unsigned long flags;

  switch ( sel_type ) {
    case SEL_IN:
      CLI( &flags );
      if ( !midi_rx_used )
        {
          GETLOCK( midi_master_card, midi_in ) |= WK_SLEEP;
          SLEEPS( midi_master_card, midi_in, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( midi_master_card, midi_in ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      CLI( &flags );
      if ( midi_tx_used > midi_tx_threshold )
        {
          GETLOCK( midi_master_card, midi_out ) |= WK_SLEEP;
          SLEEPS( midi_master_card, midi_out, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( midi_master_card, midi_out ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

void gus_init_midi( gus_card_t *card )
{
  midi_mode = MIDI_MODE_NONE;
  midi_master_card = NULL;
}

#endif /* GUSCFG_MIDI */
