/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Information about gus driver + user control interface
 */

#include "driver.h"
#include "pcm.h"
#include <stdarg.h>

#ifdef GUSCFG_DEBUG_MEMORY
#define INFO_BUFFER_SIZE	( 700 * 1024 )
#else
#define INFO_BUFFER_SIZE	( 32 * 1024 )
#endif
#define INFO_WR_BUFFER_SIZE	( 4 * 1024 )

static char *info_buf = NULL;
static size_t info_buf_size;
static char *info_wr_buf = NULL;
static char *info_wr_buf_ptr = NULL;
static size_t info_wr_buf_size;
static int info_open_flag = 0;
static gus_info_buffer_t info_buffer;

int gus_iprintf( gus_info_buffer_t *buffer, char *fmt, ... )
{
  va_list args;
  int res;
  char sbuffer[ 512 ];
  
  if ( buffer -> stop ) return 0;
  va_start( args, fmt );
  res = vsprintf( sbuffer, fmt, args );
  va_end( args );
  if ( buffer -> size + res >= INFO_BUFFER_SIZE )
    {
      buffer -> stop = 1;
      return 0;
    }
  strcpy( buffer -> curr, sbuffer );
  buffer -> curr += res;
  buffer -> size += res;
  return res;
}

static void gus_ignore_info_line( char *message, char *buffer )
{
  PRINTK( "gus: wrong %s in line '%s'\n", message, buffer );
}

static void gus_info_set_dma( gus_card_t *card, int dma, char *buffer )
{
  int value;
  struct GUS_STRU_DMA *pdma;
  unsigned long flags;
  
  if ( !strncmp( buffer, "auto", 4 ) )
    value = -1;
   else
    {
      value = simple_strtoul( buffer, &buffer, 0 );
      if ( value > 128 ) value = 128;
      if ( value < 4 ) value = 4;
      value *= 1024;
    }
  dma = dma == 1 ? GUS_DMA_GPLAY : GUS_DMA_GRECORD;
  pdma = card -> dmas[ dma ];
  if ( pdma )
    {
      CLI( &flags );
      pdma -> ursize = value > 0 ? value : pdma -> rsize;
      if ( !pdma -> used && pdma -> static_alloc )
        {
          pdma -> usize = pdma -> ursize;
          if ( pdma -> usize > pdma -> size )
            pdma -> usize = pdma -> size;
        }
      STI( &flags );
    }
}

static void gus_info_set_mixer( gus_card_t *card, int device, char *buffer )
{
  unsigned short mute;
  int level_left, level_right;
  int mute_left, mute_right;

  level_left = card -> mixer.curr[ device ].left;
  level_right = card -> mixer.curr[ device ].right;
  mute = card -> mixer.curr[ device ].mute;
  mute_left = mute & MIX_MUTE_LEFT ? 1 : 0;
  mute_right = mute & MIX_MUTE_RIGHT ? 1 : 0;
  while ( *buffer )
    {
      while ( *buffer && *buffer <= ' ' ) buffer++;
      if ( !strncmp( buffer, "left=", 5 ) ) { buffer += 5; level_left = simple_strtoul( buffer, &buffer, 0 ); } else
      if ( !strncmp( buffer, "right=", 6 ) ) { buffer += 6; level_right = simple_strtoul( buffer, &buffer, 0 ); } else
      if ( !strncmp( buffer, "both=", 5 ) ) { buffer += 5; level_left = level_right = simple_strtoul( buffer, &buffer, 0 ); } else
      if ( !strncmp( buffer, "mute=on", 7 ) ) { buffer += 7; mute_left = mute_right = 0; } else
      if ( !strncmp( buffer, "mute=off", 8 ) ) { buffer += 8; mute_left = mute_right = 1; } else
      if ( !strncmp( buffer, "mleft=on", 8 ) ) { buffer += 8; mute_left = 0; } else
      if ( !strncmp( buffer, "mleft=off", 9 ) ) { buffer += 9; mute_left = 1; } else
      if ( !strncmp( buffer, "mright=on", 9 ) ) { buffer += 9; mute_right = 0; } else
      if ( !strncmp( buffer, "mright=off", 10 ) ) { buffer += 10; mute_right = 1; }
    }
  if ( level_left < 0 ) level_left = 0;
  if ( level_left > 100 ) level_left = 100;
  if ( level_right < 0 ) level_right = 0;
  if ( level_right > 100 ) level_right = 100;
  gus_mixer_update_level( card, device, level_left, level_right, mute_left, mute_right );
}

static void gus_info_set_enhanced_mode( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_INTERWAVE
  int disable = 0;

  if ( !strncmp( buffer, "disable", 7 ) ) disable++;
  gus_init_set_enhanced_mode( card, !disable );
#endif
}

static void gus_info_set_pcm_memory( gus_card_t *card, char *buffer )
{
  int disable = 0;

  if ( !strncmp( buffer, "disable", 7 ) ) disable++;
  gus_init_set_pcm_memory( card, !disable );
}

static void gus_info_set_smooth_pan( gus_card_t *card, char *buffer )
{
  int disable = 0;

  if ( !strncmp( buffer, "disable", 7 ) ) disable++;
  card -> gf1.default_smooth_pan = !disable;
}

static void gus_info_set_full_range_pan( gus_card_t *card, char *buffer )
{
  int disable = 0;

  if ( !strncmp( buffer, "disable", 7 ) ) disable++;
  card -> gf1.default_full_range_pan = !disable;
}

static void gus_info_set_volume_ramp( gus_card_t *card, char *buffer )
{
  int value;
  
  if ( !strncmp( buffer, "disable", 7 ) )
    value = 0;
   else
    {
      value = simple_strtoul( buffer, &buffer, 0 );
      if ( value > 255 ) value = 255;
    }
  card -> gf1.default_volume_ramp = value;
}

static void gus_info_set_pcm_realtime( gus_card_t *card, char *buffer )
{
  if ( !strncmp( buffer, "on", 2 ) )
    card -> pcm.flags |= PCM_LFLG_REALTIME;
   else
    card -> pcm.flags &= ~PCM_LFLG_REALTIME;
}

static void gus_info_set_codec( gus_card_t *card, char *buffer )
{
}

static void gus_info_set_pcm_fifo( gus_card_t *card, int record, char *buffer )
{
#ifdef GUSCFG_INTERWAVE
  int value;
  
  value = simple_strtoul( buffer, &buffer, 0 );
  if ( *buffer == 'k' ) value *= 1024;
  gus_init_set_pcm_fifo( card, record, value );
#endif
}

static void gus_info_set_midi_voices( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_MIDI_DEVICES
  int voices;

  voices = simple_strtoul( buffer, &buffer, 0 );
  if ( voices < 14 ) voices = 14;
  if ( voices > 32 ) voices = 32;
  if ( card -> gf1.enh_mode ) voices = 32;	/* forced */
  card -> gf1.midi_mix_voices = voices;
#endif
}

static void gus_info_set_midi_emul( gus_card_t *card, char *buffer )
{
  if ( !strncmp( buffer, "GM", 2 ) ) card -> gf1.default_midi_emul = GUS_MIDI_EMUL_GM; else
  if ( !strncmp( buffer, "GS", 2 ) ) card -> gf1.default_midi_emul = GUS_MIDI_EMUL_GS; else
  if ( !strncmp( buffer, "MT32", 4 ) ) card -> gf1.default_midi_emul = GUS_MIDI_EMUL_MT32; else
    PRINTK( "gus: unknown midi emulation '%s'\n", buffer );
  gus_gf1_daemon_midi_emul_change( card );
}

static void gus_info_set_midi_uart( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_MIDI_DEVICES
  int disable;

  if ( !strncmp( buffer, "disable", 7 ) ) disable = 1; else disable = 0;
  card -> gf1.uart_enable = !disable;
#endif
}

static void gus_info_set_midi_raw_out( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_MIDI_DEVICES
  int synth;

  if ( !strncmp( buffer, "synth", 7 ) ) synth = 1; else synth = 0;
  card -> gf1.midi_raw_out = synth;
#endif
}

static void gus_info_set_midi_oss_effect( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_USS
  unsigned int i;

  if ( !strncmp( buffer, "disable", 7 ) )
    card -> seq.iw_effect[ 0 ] = card -> seq.iw_effect[ 1 ] = -1;
   else
    { 
      i = simple_strtoul( buffer, &buffer, 0 );
      if ( i > 7 ) i = -1;
      card -> seq.iw_effect[ 0 ] = i;
      i = simple_strtoul( buffer+1, &buffer, 0 );
      if ( i > 7 ) i = -1;
      card -> seq.iw_effect[ 1 ] = i;
    }
#endif
}

static void gus_info_set_midi_oss_swap_d0_to_d2( gus_card_t *card, char *buffer )
{
#ifdef GUSCFG_USS
  if ( !strncmp( buffer, "disable", 7 ) )
    card -> seq.flags &= ~GUS_SEQ_F_D0SWAP;
   else
    card -> seq.flags |= GUS_SEQ_F_D0SWAP;
#endif
}

#ifdef GUSCFG_DEBUG
static void gus_info_show_voice_registers( gus_card_t *card )
{
  unsigned long flags;
  int old_voice;
  int i, j;
  
  CLI( &flags );
  old_voice = card -> gf1.active_voice;
  for ( i = 0; i < GF1_VOICE_RANGES; i++ )
    for ( j = card -> gf1.voice_ranges[ i ].min; j <= card -> gf1.voice_ranges[ i ].max; j++ )
      {
        gf1_select_voice( card, j );
        gus_print_voice_registers( card );
      }
  STI( &flags );
}
#endif

static int gus_process_info_line( char *buffer )
{
  char id[ 9 ];
  char keyword[ 41 ];
  char *ptr, *orig;
  int count;
  gus_card_t *card;

  while ( *buffer && *buffer <= ' ' ) buffer++;
  if ( !(*buffer) ) return 0;
  orig = buffer;
  for ( ptr = id, count = 0; *buffer > ' ' && count < 8; buffer++, ptr++, count++ ) *ptr = *buffer;
  if ( *buffer > ' ' )
    {
      gus_ignore_info_line( "CARD ID", orig );
      return 0;
    }
  *ptr = 0;
  for ( count = 0; count < gus_cards_count; count++ )
    {
      card = gus_cards[ count ];
      if ( !strncmp( card -> id, id, 8 ) ) break;
    }
  if ( count >= gus_cards_count )
    {
      gus_ignore_info_line( "CARD ID", orig );
      return 0;
    }
  while ( *buffer && *buffer <= ' ' ) buffer++;
  for ( ptr = keyword, count = 0; *buffer > ' ' && count < 40; buffer++, ptr++, count++ ) *ptr = *buffer;
  if ( *buffer > ' ' )
    {
      gus_ignore_info_line( "KEYWORD", orig );
      return 0;
    }
  *ptr = 0;
  while ( *buffer && *buffer <= ' ' ) buffer++;
  if ( !strcmp( keyword, "dma1_size" ) ) gus_info_set_dma( card, 1, buffer ); else
  if ( !strcmp( keyword, "dma2_size" ) ) gus_info_set_dma( card, 2, buffer ); else
  if ( !strcmp( keyword, "mix_mic" ) ) gus_info_set_mixer( card, MIX_MIC, buffer ); else
  if ( !strcmp( keyword, "mix_line" ) ) gus_info_set_mixer( card, MIX_LINE, buffer ); else
  if ( !strcmp( keyword, "mix_cd" ) ) gus_info_set_mixer( card, MIX_CD, buffer ); else
  if ( !strcmp( keyword, "mix_synth" ) || !strcmp( keyword, "mix_gf1" ) ) gus_info_set_mixer( card, MIX_GF1, buffer ); else
  if ( !strcmp( keyword, "mix_pcm" ) ) gus_info_set_mixer( card, MIX_PCM, buffer ); else
  if ( !strcmp( keyword, "mix_gain" ) ) gus_info_set_mixer( card, MIX_GAIN, buffer ); else
  if ( !strcmp( keyword, "mix_master" ) ) gus_info_set_mixer( card, MIX_MASTER, buffer ); else
  if ( !strcmp( keyword, "mix_soft" ) ) gus_info_set_mixer( card, MIX_GF1_MASTER, buffer ); else
  if ( !strcmp( keyword, "mix_loop" ) ) gus_info_set_mixer( card, MIX_LOOPBACK, buffer ); else
  if ( !strcmp( keyword, "mix_effects" ) ) gus_info_set_mixer( card, MIX_EFFECTS, buffer ); else
  if ( !strcmp( keyword, "enhanced_mode" ) ) gus_info_set_enhanced_mode( card, buffer ); else
  if ( !strcmp( keyword, "pcm_memory" ) ) gus_info_set_pcm_memory( card, buffer ); else
  if ( !strcmp( keyword, "smooth_pan" ) ) gus_info_set_smooth_pan( card, buffer ); else
  if ( !strcmp( keyword, "full_range_pan" ) ) gus_info_set_full_range_pan( card, buffer ); else
  if ( !strcmp( keyword, "volume_ramp" ) ) gus_info_set_volume_ramp( card, buffer ); else
  if ( !strcmp( keyword, "pcm_realtime" ) ) gus_info_set_pcm_realtime( card, buffer ); else
  if ( !strcmp( keyword, "codec" ) ) gus_info_set_codec( card, buffer ); else
  if ( !strcmp( keyword, "interwave_fifo_playback" ) ) gus_info_set_pcm_fifo( card, 0, buffer ); else
  if ( !strcmp( keyword, "interwave_fifo_record" ) ) gus_info_set_pcm_fifo( card, 1, buffer ); else
  if ( !strcmp( keyword, "midi_voices" ) ) gus_info_set_midi_voices( card, buffer ); else
  if ( !strcmp( keyword, "midi_emul" ) ) gus_info_set_midi_emul( card, buffer ); else
  if ( !strcmp( keyword, "midi_uart" ) ) gus_info_set_midi_uart( card, buffer ); else
  if ( !strcmp( keyword, "midi_raw_out" ) ) gus_info_set_midi_raw_out( card, buffer ); else
  if ( !strcmp( keyword, "midi_oss_effect" ) ) gus_info_set_midi_oss_effect( card, buffer ); else
  if ( !strcmp( keyword, "midi_oss_swap_d0_to_d2" ) ) gus_info_set_midi_oss_swap_d0_to_d2( card, buffer ); else
#ifdef GUSCFG_DEBUG
  if ( !strcmp( keyword, "show_setup_registers" ) ) gus_print_setup_registers( card ); else
  if ( !strcmp( keyword, "show_global_registers" ) ) gus_print_global_registers( card ); else
  if ( !strcmp( keyword, "show_voice_registers" ) ) gus_info_show_voice_registers( card ); else
#ifdef GUSCFG_CODEC
  if ( !strcmp( keyword, "show_codec_registers" ) ) codec_debug( card ); else
#endif
#endif
    {
      gus_ignore_info_line( "KEYWORD", orig );
      return 0;
    }
  return 0;
}

int gus_read_info( char *buf, off_t *offset, int count )
{
  int size, size1;

  if ( !info_buf ) return -ENOMEM;
  if ( !info_buf_size )
    {
      info_buffer.stop = 0;
      info_buffer.buffer = info_buffer.curr = info_buf;
      info_buffer.size = 0;
      info_buf_size = get_info_init( &info_buffer );
    }
  if ( *offset >= info_buf_size ) return 0;
  size = info_buf_size < count ? info_buf_size : count;
  size1 = info_buf_size - *offset;
  if ( size1 < size ) size = size1;
  MEMCPY_TOFS( buf, info_buf + *offset, size );
  *offset += size;
  return size;
}

int gus_write_info( char *buf, int count )
{
  char c;
  int old_count, res;

  if ( !info_wr_buf ) return -ENOMEM;
  old_count = count;
  while ( count-- > 0 )
    {
      c = get_fs_byte( buf++ );
      if ( c == 0x0a )			/* LF */
        {
          *info_wr_buf_ptr = 0;
          if ( ( res = gus_process_info_line( info_wr_buf ) ) < 0 )
            return res;
          info_wr_buf_ptr = info_wr_buf;
          info_wr_buf_size = 0;
        }
       else
        {
          if ( info_wr_buf_size < INFO_WR_BUFFER_SIZE - 1 )
            {
              *(info_wr_buf_ptr++) = c;
              info_wr_buf_size++;
            }
        }
    }
  return count;
}

int gus_open_info( void )
{
  if ( !info_open_flag )
    {
      info_buf = (char *)VMALLOC( INFO_BUFFER_SIZE );
      info_buf_size = 0;
      info_wr_buf = info_wr_buf_ptr = (char *)VMALLOC( INFO_WR_BUFFER_SIZE );
      info_wr_buf_size = 0;
      info_open_flag++;
      MOD_INC_USE_COUNT;
      return 0;
    }
  return -EBUSY;
}

void gus_release_info( void )
{
  if ( info_wr_buf_size > 0 )
    gus_process_info_line( info_wr_buf );
  if ( info_wr_buf ) VFREE( info_wr_buf );
  if ( info_buf ) VFREE( info_buf );
  info_open_flag = 0;
  MOD_DEC_USE_COUNT;
}

#ifdef GUSCFG_USS

int gus_read_sndstat( char *buf, off_t *offset, int count )
{
  char *s = SNDSTAT_STRING;
  int size;

  size = strlen( s ) + 1;
  if ( *offset >= size || count < size - *offset )
    return 0;
  size -= *offset;
  MEMCPY_TOFS( buf, s + *offset, size );
  *offset += size;
  return size;
}

int gus_open_sndstat( void )
{
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_sndstat( void )
{
  MOD_DEC_USE_COUNT;
}

#endif
