/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  GUS's memory allocation routines
 */

#include "driver.h"

static int gus_memory_manager_find( gus_card_t *card, gus_mem_t *alloc, gus_mem_block_t *block, int size, int w_16, int align );
static gus_mem_block_t *gus_memory_manager_look( gus_card_t *card, gus_mem_t *alloc, unsigned int address );

static void gus_instrument_insert( gus_mem_t *alloc, gus_instrument_t *insert );
static void gus_instrument_delete( gus_mem_t *alloc, gus_instrument_t *delete );

/*
 *  Hardware layer routines
 */

void gus_download_over_poke( gus_card_t *card,
	  	             unsigned int mem_ptr, 
			     unsigned char *ptr, 
			     unsigned int length,
			     unsigned short invert,
			     unsigned short w_16,
			     int space )
{
  unsigned char buffer[ 512 ];
  unsigned long flags;
  unsigned char *source;
  unsigned int source_length;
  unsigned int count;
 
  if ( w_16 )
    {
      if ( (mem_ptr & 1) || (length & 1) )
        {
          PRINTK( "gus_download_over_poke: warning! unaligned poke size\n" );
          length &= ~1;
        }
    }
    
  while ( length > 0 )
    {
      if ( space == SP_USER || space == SP_KERNEL_WAVE_USER )
        {
          source_length = length > sizeof( buffer ) ? sizeof( buffer ) : length;
          if ( VERIFY_AREA( VERIFY_READ, ptr, source_length ) ) return;
          MEMCPY_FROMFS( buffer, ptr, source_length );
          source = buffer;
        }
       else
        {
          source_length = length > sizeof( buffer ) ? sizeof( buffer ) : length;
          source = ptr;
        }

      ptr += source_length;
      length -= source_length;
      count = source_length;
      if ( card -> gf1.enh_mode )
        {
          if ( w_16 )
            {
              CLI( &flags );
              gus_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 | ( invert ? 0x08 : 0x00 ) );
              gus_dram_addr( card, mem_ptr );
              OUTB( 0x51, GUSP( card, GF1REGSEL ) );
              OUTSW( GUSP( card, GF1DATALOW ), source, count >> 1 );
              gus_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 );              
              STI( &flags );
            }
           else
            {
              CLI( &flags );
              gus_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 | ( invert ? 0x08 : 0x00 ) );
              gus_dram_addr( card, mem_ptr );
              OUTB( 0x51, GUSP( card, GF1REGSEL ) );
              OUTSB( GUSP( card, DRAM ), source, count );
              gus_write8( card, GF1_GB_MEMORY_CONTROL, 0x01 );              
              STI( &flags );
            }
          mem_ptr += source_length;
        }
       else
        {
          invert = invert ? 0x80 : 0x00;
          if ( w_16 )
            {
              count >>= 1;
              while ( count-- )
                {
                  gus_poke( card, mem_ptr++, *source++ );
                  gus_poke( card, mem_ptr++, *source++ ^ invert );
                }
            }
           else
            {
              while ( count-- )
                gus_poke( card, mem_ptr++, *source++ ^ invert );
            }
        }
    }
}

void gus_download_over_dma( gus_card_t *card,
			    unsigned int mem_ptr,
			    unsigned char *ptr, 
			    unsigned int length,
			    unsigned short invert,
			    unsigned short w_16,
			    int space )
{
  int dma_flag;
  unsigned int transfer_size;
  unsigned long flags;
  
  if ( space == SP_USER || space == SP_KERNEL_WAVE_USER )
    if ( VERIFY_AREA( VERIFY_READ, ptr, length ) ) return;

  transfer_size = card -> gf1.enh_mode ? 1 : 31;
  transfer_size = ( mem_ptr + transfer_size ) & ~transfer_size;
  if ( transfer_size > mem_ptr )
    {
      transfer_size -= mem_ptr;
      if ( length < transfer_size ) transfer_size = length;
      gus_download_over_poke( card, mem_ptr, ptr, transfer_size, invert, w_16, space );
      mem_ptr += transfer_size;
      ptr += transfer_size;
      length -= transfer_size;
    }

  dma_flag = 0;
  if ( !card -> use_codec ) dma_flag = 1;
#ifdef GUSCFG_CODEC
  else if ( card -> daughter_flag || card -> codec.mode == CODEC_MODE_NONE ) dma_flag = 1;
#endif  
  if ( (card -> gf1.mode & GF1_MODE_ENGINE) && dma_flag )
    {
      CLI( &flags );
      while ( !card -> dmas[ GUS_DMA_GPLAY ] -> lock && length >= 512 )
        {
          card -> dmas[ GUS_DMA_GPLAY ] -> lock = WK_LOCK;
          STI( &flags );
          if ( length > card -> dmas[ GUS_DMA_GPLAY ] -> usize ) 
            transfer_size = card -> dmas[ GUS_DMA_GPLAY ] -> usize; 
           else
            transfer_size = length;

          if ( !card -> gf1.enh_mode &&
               mem_ptr >> GUS_MEM_BANK_SHIFT != ( mem_ptr + transfer_size - 1 ) >> 18 )
            transfer_size = ( ( mem_ptr + ( 1 << GUS_MEM_BANK_SHIFT ) ) &
            		    ~( GUS_MEM_BANK_SIZE - 1 ) ) - mem_ptr;
          if ( space == SP_USER || space == SP_KERNEL_WAVE_USER )
            MEMCPY_FROMFS( card -> dmas[ GUS_DMA_GPLAY ] -> buf, ptr, transfer_size );
           else
            MEMCPY( card -> dmas[ GUS_DMA_GPLAY ] -> buf, ptr, transfer_size );
          gus_gf1_init_dma_transfer( card, mem_ptr, card -> dmas[ GUS_DMA_GPLAY ] -> buf, transfer_size, invert, w_16 );
	  GETLOCK( card, dma1 ) |= WK_SLEEP;
	  while ( 1 )
	    {
              SLEEP( card, dma1, DMA_DOWNLOAD_SLEEP_TIME );
              if ( !(GETLOCK( card, dma1 ) & WK_SLEEP) ) break;
              if ( TABORT( card ) )
                {
                  gus_gf1_done_dma_transfer( card );
                  GETLOCK( card, dma1 ) &= ~WK_SLEEP;
                  return;
                }
              if ( TIMEOUT( card, dma1 ) )
                {
         	  gus_gf1_done_dma_transfer( card );
         	  GETLOCK( card, dma1 ) &= ~WK_SLEEP;
                  PRINTK( "gus: GF1 DMA transfer time out\n" );
                  break;
                }
            }
          ptr += transfer_size;
          length -= transfer_size;
          mem_ptr += transfer_size;
          CLI( &flags );
#ifdef GUSCFG_GF1PCM
          if ( GETLOCK( card, dma1_lock ) & WK_SLEEP )
            {
              GETLOCK( card, dma1_lock ) &= ~WK_SLEEP;
              WAKEUP( card, dma1_lock );
            }
           else
#endif
            card -> dmas[ GUS_DMA_GPLAY ] -> lock = 0;
        }
      STI( &flags );
    }

  if ( length > 0 )
    gus_download_over_poke( card, mem_ptr, ptr, length, invert, w_16, SP_USER );
}

/*
 *  GUS memory manager
 */

static int gus_memory_manager_alloc( gus_card_t *card, gus_mem_t *alloc, gus_mem_block_t *block )
{
  gus_mem_block_t *pblock, *nblock;
  
  nblock = (gus_mem_block_t *)gus_malloc( sizeof( gus_mem_block_t ) );
  if ( !nblock ) return -ENOMEM;
  MEMCPY( nblock, block, sizeof( gus_mem_block_t ) );
  pblock = alloc -> first;
  while ( pblock )
    {
      if ( pblock -> ptr > nblock -> ptr ) 
        {
          nblock -> prev = pblock -> prev;
          nblock -> next = pblock;
          pblock -> prev = nblock;
          if ( pblock == alloc -> first )
            alloc -> first = nblock;
           else
            nblock -> prev -> next = nblock;
          return 0;
        }
      pblock = pblock -> next;
    }
  nblock -> next = NULL;
  if ( !alloc -> last )
    {
      nblock -> prev = NULL;
      alloc -> first = alloc -> last = nblock;
    }
   else
    {
      nblock -> prev = alloc -> last;
      alloc -> last -> next = nblock;
      alloc -> last = nblock;
    }
  return 0;
}

static int gus_memory_manager_free( gus_card_t *card, gus_mem_t *alloc, gus_mem_block_t *block )
{
  if ( block -> share )			/* ok.. shared block */
    {
      block -> share--;
      return 0;
    }
  if ( alloc -> first == block )
    { 
      alloc -> first = block -> next;
      if ( block -> next ) block -> next -> prev = NULL;
    }
   else
    {
      block -> prev -> next = block -> next;
      if ( block -> next ) block -> next -> prev = block -> prev;
    }
  if ( alloc -> last == block )
    {
      alloc -> last = block -> prev;
      if ( block -> prev ) block -> prev -> next = NULL;
    }
   else
    {
      block -> next -> prev = block -> prev;
      if ( block -> prev ) block -> prev -> next = block -> next;
    } 
  if ( block -> owner == GUS_MEMORY_OWNER_USER )
    gus_free_str( block -> owner_data.name );
  gus_free( block, sizeof( gus_mem_block_t ) );
  return 0;
}

static int gus_memory_manager_init( gus_card_t *card, gus_mem_t *alloc )
{
  int i;
  gus_mem_block_t block;

  alloc -> first = alloc -> last = NULL;

  if ( !card -> gf1.memory ) return 0;
  
  block.flags = GUS_MEMORY_BLOCK_LOCKED;
  block.owner = GUS_MEMORY_OWNER_DRIVER;
  block.lock_group = GUS_MEMORY_LOCK_DRIVER;
  block.share = block.share_id1 = block.share_id2 = 0;
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.enh_mode )
    {
      block.ptr = 0;
      block.size = 1024;
      block.owner_data.name = "InterWave LFOs";
      if ( ( i = gus_memory_manager_alloc( card, alloc, &block ) ) < 0 ) return i;
    }
  if ( card -> codec.playback_fifo_size > 0 )
    {
      if ( gus_memory_manager_find( card, &card -> gf1.mem_alloc, &block, card -> codec.playback_fifo_size, 0, 256 ) < 0 )
        {
          if ( alloc == &card -> gf1.mem_alloc )
            {
              card -> codec.playback_fifo_size = 0;
              card -> codec.interwave_fifo_reg &= ~0x001f;
            }
        }
       else
        {
          block.owner_data.name = "playback PCM FIFO";
          if ( ( i = gus_memory_manager_alloc( card, alloc, &block ) ) < 0 ) return i;
          if ( alloc == &card -> gf1.mem_alloc )
            card -> codec.playback_fifo_block = gus_memory_manager_look( card, alloc, block.ptr );
        }
    }
  if ( card -> codec.record_fifo_size > 0 )
    {
      if ( gus_memory_manager_find( card, &card -> gf1.mem_alloc, &block, card -> codec.record_fifo_size, 0, 256 ) < 0 )
        {
          if ( alloc == &card -> gf1.mem_alloc )
            {
              card -> codec.record_fifo_size = 0;
              card -> codec.interwave_fifo_reg &= ~0x1f00;
            }
        }
       else
        {
          block.owner_data.name = "record PCM FIFO";
          if ( ( i = gus_memory_manager_alloc( card, alloc, &block ) ) < 0 ) return i;
          if ( alloc == &card -> gf1.mem_alloc )
            card -> codec.record_fifo_block = gus_memory_manager_look( card, alloc, block.ptr );
        }
    }
#endif
  block.ptr = card -> gf1.default_voice_address;
  block.size = 4;
  block.owner_data.name = "Voice default (NULL's)";
  i = gus_memory_manager_alloc( card, alloc, &block );  
  if ( i < 0 ) return i;
#ifdef GUSCFG_GF1PCM
  if ( card -> gf1.pcm_memory && !card -> use_codec )
    {
      block.ptr = ( card -> gf1.default_voice_address + block.size + 31 ) & ~31;
      card -> gf1.pcm_mem = block.ptr;
      block.size = card -> dmas[ GUS_DMA_GPLAY ] -> rsize;
      block.owner_data.name = "GF1 PCM DATA";
      i = gus_memory_manager_alloc( card, alloc, &block );
      if ( i < 0 ) return i;
    }
   else
    card -> gf1.pcm_mem = 0;
#endif
  return 0;
}

static int gus_memory_manager_done( gus_card_t *card, gus_mem_t *alloc )
{
  gus_mem_block_t *block;

  for ( block = alloc -> first; block; block = block -> next )
    gus_memory_manager_free( card, alloc, block );
  return 0;
}

static gus_mem_block_t *gus_memory_manager_look( gus_card_t *card, gus_mem_t *alloc, unsigned int address )
{
  gus_mem_block_t *block;
  
  for ( block = alloc -> first; block; block = block -> next )
    if ( block -> ptr == address ) return block;
  return NULL;
}

static gus_mem_block_t *gus_memory_manager_share( gus_card_t *card, gus_mem_t *alloc, unsigned int share_id1, unsigned int share_id2 )
{
  gus_mem_block_t *block;
  
  if ( !share_id1 && !share_id2 ) return NULL;
  for ( block = alloc -> first; block; block = block -> next )
    if ( block -> share_id1 == share_id1 && block -> share_id2 == share_id2 ) return block;
  return NULL;
}

static int gus_memory_manager_find( gus_card_t *card, gus_mem_t *alloc, gus_mem_block_t *block, int size, int w_16, int align )
{
  struct GUS_STRU_BANK_INFO *info = w_16 ? card -> gf1.banks_16 : card -> gf1.banks_8;
  unsigned int idx, boundary;
  int size1;
  gus_mem_block_t *pblock;
  unsigned int ptr1, ptr2;
  
  align--;
  block -> flags = w_16 ? GUS_MEMORY_BLOCK_16BIT : 0;
  block -> owner = GUS_MEMORY_OWNER_DRIVER;
  block -> lock_group = GUS_MEMORY_LOCK_DRIVER;
  block -> share = block -> share_id1 = block -> share_id2 = 0;
  block -> owner_data.name = NULL;
  block -> prev = block -> next = NULL;
  for ( pblock = alloc -> first, idx = 0; pblock; pblock = pblock -> next )
    {
      while ( pblock -> ptr >= ( boundary = info[ idx ].address + info[ idx ].size ) ) idx++;
      while ( pblock -> ptr + pblock -> size >= ( boundary = info[ idx ].address + info[ idx ].size ) ) idx++;
      ptr2 = boundary;
      if ( pblock -> next )
        {
          if ( pblock -> ptr + pblock -> size == pblock -> next -> ptr ) continue;
          if ( pblock -> next -> ptr < boundary )
            ptr2 = pblock -> next -> ptr;
        }
      ptr1 = ( pblock -> ptr + pblock -> size + align ) & ~align;
      if ( ptr1 >= ptr2 ) continue;
      size1 = ptr2 - ptr1;
      if ( size <= size1 )
        {
          block -> ptr = ptr1;
          block -> size = size;
          return 0;
        }
    }
  while ( ++idx < GUS_MEMORY_BANKS )
    if ( size <= info[ idx ].size )
      {
        block -> ptr = info[ idx ].address;	/* I assume that bank address is already aligned.. */
        block -> size = size;
        return 0;
      }
  return -ENOMEM;
}

unsigned int gus_memory_manager_malloc( gus_card_t *card, gus_mem_t *alloc, char *name, int size, int w_16, int align )
{
  gus_mem_block_t block;
  
  if ( gus_memory_manager_find( card, alloc, &block, size, w_16, align ) < 0 ) return -1;
  block.owner_data.name = name;
  return gus_memory_manager_alloc( card, alloc, &block ) < 0 ? -1 : block.ptr;
}

int gus_memory_manager_mfree( gus_card_t *card, gus_mem_t *alloc, unsigned int address )
{
  gus_mem_block_t *block;

  if ( ( block = gus_memory_manager_look( card, alloc, address ) ) != NULL )
    return gus_memory_manager_free( card, alloc, block );
  return -1;
}

#ifdef GUSCFG_DEBUG_MEMORY
void gus_memory_manager_debug( gus_card_t *card, gus_info_buffer_t *buffer, gus_mem_t *alloc )
{
  gus_instrument_t *instrument;
  gus_mem_block_t *block;
  int i, total = 0, not_found = 0, no_mem = 0, alias = 0;

  for ( instrument = alloc -> instruments; instrument; instrument = instrument -> next )
    {
      total++;
      if ( instrument -> flags & GUS_INSTR_F_NOT_FOUND ) not_found++;
      if ( instrument -> flags & GUS_INSTR_F_ALIAS ) alias++;
    }
  gus_iprintf( buffer, "Memory usage:\n" );
  gus_iprintf( buffer, "  Total instruments    : %i\n", total );
  gus_iprintf( buffer, "    Not found          : %i\n", not_found );
  gus_iprintf( buffer, "    Not enough memory  : %i\n", no_mem );
  gus_iprintf( buffer, "    Alias(es)          : %i\n", alias );
  gus_iprintf( buffer, "  8-bit banks          : \n    " );
  for ( i = 0; i < GUS_MEMORY_BANKS; i++ )
    gus_iprintf( buffer, "0x%06x (%04ik)%s", card -> gf1.banks_8[ i ].address, card -> gf1.banks_8[ i ].size >> 10, i + 1 < GUS_MEMORY_BANKS ? "," : "" );
  gus_iprintf( buffer, "\n"
                       "  16-bit banks         : \n    " );
  for ( i = no_mem = 0; i < GUS_MEMORY_BANKS; i++ )
    {
      gus_iprintf( buffer, "0x%06x (%04ik)%s", card -> gf1.banks_16[ i ].address, card -> gf1.banks_16[ i ].size >> 10, i + 1 < GUS_MEMORY_BANKS ? "," : "" );
      no_mem += card -> gf1.banks_16[ i ].size;
    }
  gus_iprintf( buffer, "\n" );  
  total = 0;
  for ( block = alloc -> first, i = 0; block; block = block -> next, i++ )
    {
      total += block -> size;
      gus_iprintf( buffer, "  Block %i at 0x%lx GUS 0x%x size %i (0x%x):\n", i, (long)block, block -> ptr, block -> size, block -> size );
      gus_iprintf( buffer, "    Share          : %i [id1 0x%x] [id2 0x%x]\n", block -> share, block -> share_id1, block -> share_id2 );
      gus_iprintf( buffer, "    Flags          :%s%s\n",
      			block -> flags & GUS_MEMORY_BLOCK_LOCKED ? " locked" : "",
      			block -> flags & GUS_MEMORY_BLOCK_16BIT ? " 16-bit" : "" );
      gus_iprintf( buffer, "    Owner          : " );
      switch ( block -> owner ) {
        case GUS_MEMORY_OWNER_DRIVER:
          gus_iprintf( buffer, "driver - %s\n", block -> owner_data.name );
          break;
        case GUS_MEMORY_OWNER_WAVE:
          gus_iprintf( buffer, "wave at 0x%lx\n", (long)block -> owner_data.wave );
          break;
        case GUS_MEMORY_OWNER_USER:
          gus_iprintf( buffer, "user - '%s'\n", block -> owner_data.name );
          break;
        default:
          gus_iprintf( buffer, "uknown\n" );
      }
      gus_iprintf( buffer, "    Lock group     : " );
      switch ( block -> lock_group ) {
        case GUS_MEMORY_LOCK_DRIVER: gus_iprintf( buffer, "driver\n" ); break;
        case GUS_MEMORY_LOCK_DAEMON: gus_iprintf( buffer, "daemon\n" ); break;
        case GUS_MEMORY_LOCK_USER: gus_iprintf( buffer, "user\n" ); break;
        default:
          gus_iprintf( buffer, "unknown\n" );
      }
    }
  gus_iprintf( buffer, "  Total: used = %i, free = %i, memory = %i\n",
  			total, no_mem - total, no_mem );
  gus_iprintf( buffer, "  Verify: free = %i, max 8-bit block = %i, max 16-bit block = %i\n",
  			gus_memory_free_size( card, &card -> gf1.mem_alloc ),
  			gus_memory_free_block( card, &card -> gf1.mem_alloc, 0 ),
  			gus_memory_free_block( card, &card -> gf1.mem_alloc, 1 ) );
}
#endif

/*
 *  User layer routines
 */

int gus_memory_init( gus_card_t *card, gus_mem_t *alloc )
{
  alloc -> mode = GUS_ALLOC_MODE_NORMAL;
  alloc -> instruments = NULL;
  return gus_memory_manager_init( card, alloc );
}

int gus_memory_done( gus_card_t *card, gus_mem_t *alloc )
{
  gus_memory_reset( card, alloc, GUS_ALLOC_MODE_NORMAL, 1 );
  gus_memory_manager_done( card, alloc );
  return 0;
}

int gus_memory_reset( gus_card_t *card, gus_mem_t *alloc, int mode, int close_flag )
{
  int res;
  gus_mem_block_t *block;

  res = gus_memory_free( card, alloc, alloc -> instruments, SP_KERNEL );
  if ( res < 0 ) return res;
  if ( close_flag )
    {
      gus_memory_reset_daemon( card, alloc, 0 );
      for ( block = alloc -> first; block; block = block -> next )
        if ( !(block -> flags & GUS_MEMORY_BLOCK_LOCKED) )
          gus_memory_manager_free( card, alloc, block );
    }
  alloc -> mode = mode;
  return 0;
}

void gus_memory_reset_daemon( gus_card_t *card, gus_mem_t *alloc, int close_flag )
{
  int notice = 0;
  gus_mem_block_t *block;
  
  for ( block = alloc -> first; block; block = block -> next )
    {
      if ( !close_flag && (block -> flags & GUS_MEMORY_BLOCK_LOCKED) ) continue;
      if ( block -> lock_group == GUS_MEMORY_LOCK_DAEMON ) notice++;
    }
  if ( !close_flag && notice )		/* notice to daemon: block is invalid */
    gus_gf1_daemon_free_block( card );
  /*
   * ok.. free all daemon block...
   * this work is already done by gusd daemon... but for sure...
   */
  for ( block = alloc -> first; block; block = block -> next )
    {
      if ( !close_flag && (block -> flags & GUS_MEMORY_BLOCK_LOCKED) ) continue;
      if ( block -> lock_group == GUS_MEMORY_LOCK_DAEMON )
        gus_memory_manager_free( card, alloc, block );
    }
}

unsigned int gus_memory_free_size( gus_card_t *card, gus_mem_t *alloc )
{
  int used;
  gus_mem_block_t *block;
  
  for ( block = alloc -> first, used = 0; block; block = block -> next )
    used += block -> size;
  return card -> gf1.memory - used;
}

unsigned int gus_memory_free_block( gus_card_t *card, gus_mem_t *alloc, short w_16 )
{
  struct GUS_STRU_BANK_INFO *info = w_16 ? card -> gf1.banks_16 : card -> gf1.banks_8;
  gus_mem_block_t *block;
  unsigned int idx;
  int size, result;
  
  result = 0;
  for ( block = alloc -> first, idx = 0; block; block = block -> next )
    {
      while ( block -> ptr >= info[ idx ].address + info[ idx ].size ) idx++;
      while ( block -> ptr + block -> size >= info[ idx ].address + info[ idx ].size ) idx++;
      if ( block -> next )
        {
          if ( block -> ptr + block -> size == block -> next -> ptr ) continue;
          if ( block -> next -> ptr >= info[ idx ].address + info[ idx ].size )
            size = info[ idx ].size - ( block -> ptr - info[ idx ].address ) - block -> size;
           else       
            size = block -> next -> ptr - block -> ptr - block -> size;
        }
       else
        size = info[ idx ].size - ( block -> ptr - info[ idx ].address ) - block -> size;
      if ( result < size ) result = size;
    }
  while ( ++idx < GUS_MEMORY_BANKS )
    if ( result < info[ idx ].size ) result = info[ idx ].size;
  return result;
}

int gus_memory_block_alloc( gus_card_t *card, struct GUS_STRU_MEMORY_BLOCK *block, int lock, int space )
{
  struct GUS_STRU_MEMORY_BLOCK sblock, *oblock;
  gus_mem_block_t mblock, *pblock;
  char id[ sizeof( sblock.id ) + 1 ];
  
  oblock = block;
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, block, sizeof( *block ) ) ) return -EIO;
      MEMCPY_FROMFS( &sblock, block, sizeof( sblock ) );
      block = &sblock;
    }

  for ( pblock = card -> gf1.mem_alloc.first; pblock; pblock = pblock -> next )
    if ( pblock -> owner == GUS_MEMORY_OWNER_USER &&
         !strncmp( pblock -> owner_data.name, block -> id, sizeof( block -> id ) ) )
      break;

  if ( pblock )
    {
      block -> flags = ( pblock -> flags & ( GUS_MEMORY_BLOCK_LOCKED | GUS_MEMORY_BLOCK_16BIT ) ) | GUS_MEMORY_BLOCK_ALREADY;
      block -> address = pblock -> ptr;
      block -> size = pblock -> size;
      pblock -> share++;
      
      if ( space == SP_USER )
        MEMCPY_TOFS( oblock, block, sizeof( *block ) );
      return 0;
    }
  
  if ( gus_memory_manager_find( card, &card -> gf1.mem_alloc, &mblock, block -> size, block -> flags & GUS_MEMORY_BLOCK_16BIT, block -> flags & GUS_MEMORY_BLOCK_16BIT ? 2 : 1 ) < 0 )
    return -ENOMEM;

  mblock.owner = GUS_MEMORY_OWNER_USER;
  mblock.lock_group = lock;
  if ( block -> flags & GUS_MEMORY_BLOCK_LOCKED )
    mblock.flags |= GUS_MEMORY_BLOCK_LOCKED;
  
  strncpy( id, block -> id, sizeof( sblock.id ) );
  id[ sizeof( sblock.id ) ] = 0;
  
  mblock.owner_data.name = gus_malloc_strdup( id, SP_KERNEL );

  gus_memory_manager_alloc( card, &card -> gf1.mem_alloc, &mblock );

  block -> address = mblock.ptr;
  if ( space == SP_USER )
    MEMCPY_TOFS( oblock, block, sizeof( *block ) );

  return 0;
}

int gus_memory_block_free( gus_card_t *card, struct GUS_STRU_MEMORY_BLOCK *block, int lock, int space )
{
  struct GUS_STRU_MEMORY_BLOCK sblock;
  gus_mem_block_t *mblock;
  
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, block, sizeof( *block ) ) ) return -EIO;
      MEMCPY_FROMFS( &sblock, block, sizeof( sblock ) );
      block = &sblock;
    }

  if ( block -> address == ~0 || block -> size == ~0 )	/* find id */
    {
      for ( mblock = card -> gf1.mem_alloc.first; mblock; mblock = mblock -> next )
        if ( mblock -> owner == GUS_MEMORY_OWNER_USER &&
             !strncmp( mblock -> owner_data.name, block -> id, sizeof( block -> id ) ) )
          break;
    }
   else
    mblock = gus_memory_manager_look( card, &card -> gf1.mem_alloc, block -> address );

  if ( !mblock ) return -ENOENT;
  if ( mblock -> size != ( ( block -> size + 1 ) & ~1 ) ) return -EINVAL;
  if ( mblock -> lock_group == GUS_MEMORY_LOCK_DRIVER ) return -EPERM;
  if ( mblock -> share )
    {
      mblock -> share--;
      return 0;
    }
  
  gus_memory_manager_free( card, &card -> gf1.mem_alloc, mblock );
  
  return 0;
}

static void gus_memory_free_iw_envelope( struct GUS_STRU_IW_ENV_RECORD *record )
{
  struct GUS_STRU_IW_ENV_RECORD *orecord;

  if ( !record ) return;
  while ( record )
    {
      orecord = record;
      record = record -> next;
      gus_free( orecord -> points, sizeof( short ) * 2 * ( orecord -> nattack + orecord -> nrelease ) );
      gus_free( orecord, sizeof( struct GUS_STRU_IW_ENV_RECORD ) );
    }
}

static struct GUS_STRU_IW_ENV_RECORD *gus_memory_alloc_iw_envelope( struct GUS_STRU_IW_ENV_RECORD *record, int space )
{
  unsigned short points, *points_ptr;
  struct GUS_STRU_IW_ENV_RECORD *first_record, *nrecord, *precord, srecord;

  first_record = precord = NULL;
  while ( record ) {
    if ( space == SP_USER )
      {
        if ( VERIFY_AREA( VERIFY_READ, record, sizeof( *record ) ) ) goto __free_records;
        MEMCPY_FROMFS( &srecord, record, sizeof( srecord ) );
        record = &srecord;
      }
    points = record -> nattack + record -> nrelease;
    points_ptr = NULL;
    if ( points )
      {
        if ( !record -> points ) goto __free_records;
        points_ptr = gus_malloc( sizeof( short ) * 2 * points );
        if ( !points_ptr ) goto __free_records;
        if ( space == SP_USER )
          {
            if ( VERIFY_AREA( VERIFY_READ, record -> points, sizeof( short ) * 2 * points ) )
              {
                gus_free( points_ptr, sizeof( short ) * 2 * points );
                goto __free_records;
              }
	    MEMCPY_FROMFS( points_ptr, record -> points, sizeof( short ) * 2 * points );
	  }
	 else
	  MEMCPY( points_ptr, record -> points, sizeof( short ) * 2 * points );
      }
    nrecord = gus_malloc( sizeof( struct GUS_STRU_IW_ENV_RECORD ) );
    if ( !nrecord )
      {
        gus_free( points_ptr, sizeof( short ) * 2 * points );
        goto __free_records;
      }
    MEMCPY( nrecord, record, sizeof( struct GUS_STRU_IW_ENV_RECORD ) );
    nrecord -> next = NULL;
    (unsigned short *)nrecord -> points = points_ptr;
    if ( !first_record )
      first_record = nrecord;
     else
      precord -> next = nrecord;
    precord = nrecord;
    record = record -> next;
  }
  return first_record;
  __free_records:
  gus_memory_free_iw_envelope( first_record );
  return (void *)-1;
}

static void gus_memory_free_wave( gus_card_t *card, gus_mem_t *alloc, struct GUS_STRU_WAVE *wave )
{
  struct GUS_STRU_WAVE *owave;
  gus_mem_block_t *block;

  while ( wave )
    {
      owave = wave;
      wave = wave -> next;
      if ( !(owave -> format & (GUS_WAVE_RAM | GUS_WAVE_ROM)) )
        {
          block = gus_memory_manager_look( card, alloc, owave -> begin.memory >> 4 );
          if ( !block )
            {
#ifdef GUSCFG_DEBUG_INSTRUMENTS
              PRINTK( "gus: memory block isn't allocated (gus_memory_free_wave)?\n" );
#endif
              continue;
            }
          gus_memory_manager_free( card, alloc, block );
        }
      gus_free( owave, sizeof( struct GUS_STRU_WAVE ) );
    }
}

static struct GUS_STRU_WAVE *gus_memory_alloc_wave( gus_card_t *card, gus_mem_t *alloc, unsigned char mode, struct GUS_STRU_WAVE *wave, int test_mode, int space )
{
  struct GUS_STRU_WAVE swave, *res_wave, *pwave, *nwave;
  gus_mem_block_t block, *pblock;

  res_wave = pwave = NULL;
  while ( wave ) {
    if ( space == SP_USER )
      {
        if ( VERIFY_AREA( VERIFY_READ, wave, sizeof( *wave ) ) ) goto __free_waves;
        MEMCPY_FROMFS( &swave, wave, sizeof( swave ) );
        wave = &swave;
      }

    pblock = NULL;
    if ( !(wave -> format & ( GUS_WAVE_RAM | GUS_WAVE_ROM )) )
      {
        if ( wave -> share_id1 || wave -> share_id2 )		/* try find shared wave */
          pblock = gus_memory_manager_share( card, alloc, wave -> share_id1, wave -> share_id2 );
        if ( !pblock && gus_memory_manager_find( card, alloc, &block, wave -> size, wave -> format & GUS_WAVE_16BIT, wave -> format & GUS_WAVE_16BIT ? 2 : 1 ) < 0 )
          goto __free_waves;		/* no enough memory */
      }

    nwave = gus_malloc( sizeof( struct GUS_STRU_WAVE ) );
    if ( !nwave ) goto __free_waves;
    MEMCPY( nwave, wave, sizeof( struct GUS_STRU_WAVE ) );
    nwave -> next = NULL;

#ifdef GUSCFG_DEBUG_INSTRUMENTS
    if ( !nwave -> loop_end )
      PRINTK( "wave alloc: loop end is zero?\n" );
#endif

    if ( !(wave -> format & ( GUS_WAVE_RAM | GUS_WAVE_ROM )) )
      {
        if ( !pblock )
          {
            block.owner = GUS_MEMORY_OWNER_WAVE;
            block.owner_data.wave = nwave;
            block.share_id1 = wave -> share_id1;
            block.share_id2 = wave -> share_id2;
            if ( gus_memory_manager_alloc( card, alloc, &block ) < 0 )
              {
                gus_free( nwave, sizeof( struct GUS_STRU_WAVE ) );
                goto __free_waves;
              }
            if ( !test_mode && alloc -> mode == GUS_ALLOC_MODE_NORMAL )
              gus_download_over_dma( card, block.ptr, wave -> begin.ptr,
                                     wave -> size,
                                     wave -> format & GUS_WAVE_INVERT,
                                     wave -> format & GUS_WAVE_16BIT, space );
            nwave -> begin.memory = block.ptr << 4;
          }
         else
          {
            pblock -> share++;
            nwave -> begin.memory = pblock -> ptr << 4;
          }
      }

    if ( !res_wave )
      res_wave = nwave;
     else
      pwave -> next = nwave;
    
    pwave = nwave;
    wave = wave -> next;
  }
  return res_wave;
__free_waves:
  gus_memory_free_wave( card, alloc, res_wave );
  return NULL;
}

static void gus_memory_free_layer( gus_card_t *card, gus_mem_t *alloc, struct GUS_STRU_LAYER *layer )
{
  struct GUS_STRU_LAYER *olayer;

  while ( layer )
    {
      olayer = layer;
      layer = layer -> next;
      gus_memory_free_wave( card, alloc, olayer -> wave );
      if ( olayer -> mode == GUS_INSTR_INTERWAVE )
        {
          gus_memory_free_iw_envelope( olayer -> data.iw.penv.record );
          gus_memory_free_iw_envelope( olayer -> data.iw.venv.record );
        }
      gus_free( olayer, sizeof( struct GUS_STRU_LAYER ) );
    }
}

static struct GUS_STRU_LAYER *gus_memory_alloc_layer( gus_card_t *card, gus_mem_t *alloc, unsigned char mode, struct GUS_STRU_LAYER *layer, int test_mode, int space )
{
  struct GUS_STRU_LAYER slayer, *res_layer, *player, *nlayer;
  struct GUS_STRU_WAVE *wave;

  res_layer = player = NULL;
  while ( layer ) {
    if ( space == SP_USER )
      {
        if ( VERIFY_AREA( VERIFY_READ, layer, sizeof( *layer ) ) ) goto __free_layers;
        MEMCPY_FROMFS( &slayer, layer, sizeof( slayer ) );
        layer = &slayer;
      }
    if ( !layer -> wave ) goto __free_layers;
    wave = gus_memory_alloc_wave( card, alloc, mode, layer -> wave, test_mode, space );
    if ( !wave ) goto __free_layers;
    if ( test_mode != MTST_ONE )
      {
        nlayer = gus_malloc( sizeof( struct GUS_STRU_LAYER ) );
        if ( !nlayer ) goto __free_layers;
        MEMCPY( nlayer, layer, sizeof( struct GUS_STRU_LAYER ) );
        nlayer -> next = NULL;
        nlayer -> wave = wave;
        if ( nlayer -> mode == GUS_INSTR_INTERWAVE )	/* allocate envelopes, too */
          {
            nlayer -> data.iw.penv.record = gus_memory_alloc_iw_envelope( layer -> data.iw.penv.record, space );
            nlayer -> data.iw.venv.record = gus_memory_alloc_iw_envelope( layer -> data.iw.venv.record, space );
            if ( nlayer -> data.iw.penv.record == (void *)-1 ||
                 nlayer -> data.iw.venv.record == (void *)-1 ) goto __free_layers;
          }
        if ( !res_layer )
          res_layer = nlayer;
         else
          player -> next = nlayer;
        player = nlayer;
      }
     else
      if ( !res_layer ) res_layer = layer;
    layer = layer -> next;    
  }
  return res_layer;
__free_layers:
  gus_memory_free_layer( card, alloc, res_layer );
  return NULL;
}

int gus_memory_get_name( gus_card_t *card, gus_mem_t *alloc, gus_instrument_name_t *instrument, int space )
{
  gus_instrument_name_t instr, *oinstr = NULL;
  gus_instrument_t *pinstr;
  unsigned int ninstr;

  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, instrument, sizeof( *instrument ) ) ) return -EIO;
      MEMCPY_FROMFS( &instr, instrument, sizeof( instr ) );
      oinstr = instrument;
      instrument = &instr;
    }
  ninstr = instrument -> number.instrument;
  MEMSET( instrument, 0, sizeof( instr ) );
  if ( ( pinstr = gus_instrument_look( &card -> gf1.mem_alloc, ninstr ) ) == NULL )
    instrument -> flags = GUS_INSTR_F_NOT_LOADED;
   else
    {
      if ( pinstr -> name )
        {
          strncpy( instrument -> name, pinstr -> name, sizeof( instrument -> name ) );
          instrument -> name[ sizeof( instrument -> name ) - 1 ] = 0;
        }
      instrument -> mode = pinstr -> mode;
      instrument -> flags = pinstr -> flags;
    }
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_WRITE, oinstr, sizeof( *oinstr ) ) ) return -EIO;
      MEMCPY_TOFS( oinstr, instrument, sizeof( *oinstr ) );
    }
  return 0;
}

int gus_memory_list( gus_card_t *card, gus_mem_t *alloc, struct GUS_STRU_MEMORY_LIST *list, int space )
{
  unsigned int idx, size, *instruments, *instruments1;
  struct GUS_STRU_MEMORY_LIST slist;
  gus_instrument_t *instr;

  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, list, sizeof( *list ) ) ) return -EIO;
      MEMCPY_FROMFS( &slist, list, sizeof( slist ) );
      list = &slist;
    }
  if ( !list -> count ) return -EINVAL;
  size = list -> count * sizeof( unsigned int );
  if ( VERIFY_AREA( VERIFY_WRITE, list -> info.instruments, size ) ) return -EIO;
  if ( ( instruments = gus_malloc( size ) ) == NULL ) return -ENOMEM;
  MEMSET( instruments, 0xff, size );
  instruments1 = instruments;
  for ( instr = alloc -> instruments, idx = list -> count; instr && idx; instr = instr -> next, idx-- )
    *instruments1++ = instr -> number.instrument;
  MEMCPY_TOFS( list -> info.instruments, instruments, size );
  gus_free( instruments, size );
  return 0;
}

int gus_memory_alloc( gus_card_t *card, gus_mem_t *alloc, gus_instrument_t *instrument, int test_mode, int space )
{
  int res;
  gus_instrument_t instr, *first, *pinstr, *ninstr;
  struct GUS_STRU_LAYER *layer;
  char *name;

  res = 0;
  first = NULL;
  while ( instrument ) {
    if ( space == SP_USER )
      {
        if ( VERIFY_AREA( VERIFY_READ, instrument, sizeof( *instrument ) ) )
          {
            res = -EIO;
            goto __end;
          }
        MEMCPY_FROMFS( &instr, instrument, sizeof( instr ) );
        instrument = &instr;
      }
    if ( gus_instrument_look( &card -> gf1.mem_alloc, instrument -> number.instrument ) != NULL )
      {
        /* ok.. this maybe wrong if not all instruments will be loaded.. */
        /* this instrument will be deleted in this situation.. */
        instrument = instrument -> next;
        continue;
      }
    if ( !gus_engine_instrument_register_ask( instrument -> mode ) )
      {
        res = -EINVAL;
        goto __end;
      }
    layer = NULL;
    if ( instrument -> name )
      {
        name = gus_malloc_strdup( instrument -> name, space == SP_USER ? SP_USER : SP_KERNEL );
        if ( !name )
          {
            res = -ENOMEM;
            goto __end;
          }
      }
     else
      name = NULL;
    switch ( instrument -> flags ) {
      case GUS_INSTR_F_NOT_FOUND:
      case GUS_INSTR_F_ALIAS:
        break;
      default:		/* ok.. layer mode.. */
        layer = gus_memory_alloc_layer( card, alloc, instrument -> mode, instrument -> info.layer, test_mode, space );
        if ( !layer )
          {
            res = -ENOMEM;
            gus_free_str( name );
            goto __end;
          }
    }
    pinstr = alloc -> instruments;        
    ninstr = gus_malloc( sizeof( gus_instrument_t ) );
    if ( !ninstr )
      {
        res = -ENOMEM;
        gus_memory_free_layer( card, alloc, layer );
        gus_free_str( name );
        goto __end;
      }
    if ( !first ) first = ninstr;
    MEMCPY( ninstr, instrument, sizeof( gus_instrument_t ) );
    ninstr -> next = NULL;		/* for sure */
    ninstr -> name = name;
    if ( layer )
      ninstr -> info.layer = layer;
    gus_instrument_insert( alloc, ninstr );
    instrument = instrument -> next;
  }
__end:
  if ( res < 0 || test_mode == MTST_ONE )
    {
      gus_memory_free( card, alloc, first, SP_KERNEL );
      return res;
    }
  return 0;
}

int gus_memory_free( gus_card_t *card, gus_mem_t *alloc, gus_instrument_t *instrument, int space )
{
  int res = 0;
  gus_instrument_t instr, *pinstr, *ninstr;

  while ( instrument ) {
    if ( space == SP_USER )
      {
        if ( VERIFY_AREA( VERIFY_READ, instrument, sizeof( *instrument ) ) ) return -EIO;
        MEMCPY_FROMFS( &instr, instrument, sizeof( instr ) );
        instrument = &instr;
      }
      pinstr = gus_instrument_look( alloc, instrument -> number.instrument );
      if ( !pinstr )
        {
          res = -EINVAL;
          instrument = instrument -> next;
          continue;
        }

      gus_instrument_delete( alloc, pinstr );
      if ( alloc == &card -> gf1.mem_alloc )
        gus_engine_invalidate_instrument( card, pinstr );

      if ( pinstr -> flags == GUS_INSTR_F_NORMAL )
        gus_memory_free_layer( card, alloc, pinstr -> info.layer );
      ninstr = instrument -> next;
      gus_free_str( pinstr -> name );
      gus_free( pinstr, sizeof( gus_instrument_t ) );

      instrument = ninstr;
    }
  return res;
}

int gus_memory_pack( gus_card_t *card, gus_mem_t *alloc )
{
  return 0;
}

#ifdef GUSCFG_DEBUG_INSTRUMENTS

static char *gus_memory_debug_instruments_get_mode( int mode )
{
  switch ( mode ) {
    case GUS_INSTR_SIMPLE: return "simple"; break;
    case GUS_INSTR_PATCH: return "patch"; break;
    case GUS_INSTR_INTERWAVE: return "InterWave"; break;
  }
  return "unknown";
}

static void gus_memory_debug_instruments_iw_modulation( gus_info_buffer_t *buffer, char *title, struct GUS_STRU_IW_LFO *modulation )
{
  gus_iprintf( buffer, "        %s (freq %i,depth %i,sweep %i,shape %i,delay %i)\n",
  				title,
              			modulation -> freq,
              			modulation -> depth,
              			modulation -> sweep,
              			modulation -> shape,
              			modulation -> delay );
}

static void gus_memory_debug_instruments_iw_envelope( gus_info_buffer_t *buffer, char *title, struct GUS_STRU_IW_ENV *envelope )
{
  int idx, idxp;
  struct GUS_STRU_IW_ENV_RECORD *record;

  gus_iprintf( buffer, "        %s envelope (flags 0x%x, mode 0x%x, index 0x%x)\n",
  				title,
  				envelope -> flags,
  				envelope -> mode,
  				envelope -> index );
  for ( record = envelope -> record, idx = 0; record; record = record -> next, idx++ )
    {
      gus_iprintf( buffer, "          %03i nattack=%i nrelease=%i soffset=%i srate=%i rrate=%i hirange=%i\n",
      				idx,
      				record -> nattack,
      				record -> nrelease,
      				record -> sustain_offset,
      				record -> sustain_rate,
      				record -> release_rate,
      				record -> hirange );
      for ( idxp = 0; idxp < record -> nattack + record -> nrelease; idxp++ )
        gus_iprintf( buffer, "            %03i offset=%i rate=%i\n",
        		idxp,
        		record -> points[ idxp ].offset,
        		record -> points[ idxp ].rate );
    }
}

void gus_memory_debug_instruments( gus_card_t *card, gus_info_buffer_t *buffer, gus_mem_t *alloc )
{
  int idx, idxl, idxw;
  gus_instrument_t *instrument;
  gus_layer_t *layer;
  gus_wave_t *wave;
  char *s1, *s2, *s3;
  
  instrument = alloc -> instruments;
  gus_iprintf( buffer, "Instruments:\n" );
  if ( !instrument )
    {
      gus_iprintf( buffer, "  -none-\n" );
      return;
    }  
  for ( idx = 0; instrument; instrument = instrument -> next, idx++ )
    {
#if 0
      printk( "instr = 0x%lx, number = %i\n", (long)instrument, instrument -> number.instrument );
#endif
      s2 = s3 = "unknown";
      s1 = gus_memory_debug_instruments_get_mode( instrument -> mode );
      switch ( instrument -> exclusion ) {
        case GUS_INSTR_E_NONE: s2 = "none"; break;
        case GUS_INSTR_E_SINGLE: s2 = "single"; break;
        case GUS_INSTR_E_MULTIPLE: s2 = "multiple"; break;
      }
      switch ( instrument -> layer ) {
        case GUS_INSTR_L_NONE: s3 = "none"; break;
        case GUS_INSTR_L_ON: s3 = "on"; break;
        case GUS_INSTR_L_VELOCITY: s3 = "velocity"; break;
        case GUS_INSTR_L_FREQUENCY: s3 = "frequency"; break;
      }
      gus_iprintf( buffer, "  %03i [%i-%i] <%s> flags(%s%s) exclusion(%s) layer(%s) egrp(%i)\n",
                        idx, instrument -> number.midi.bank, instrument -> number.midi.prog, s1,
      			instrument -> flags & GUS_INSTR_F_NOT_FOUND ? "-not found-" : "",
      			instrument -> flags & GUS_INSTR_F_ALIAS ? "-alias-" : "",
      			s2, s3,
      			instrument -> exclusion_group );
      gus_iprintf( buffer, "    name '%s'\n", instrument -> name ? instrument -> name : "<none>" );
      if ( instrument -> mode == GUS_INSTR_INTERWAVE )
        gus_iprintf( buffer, "    effect1 = %i, depth = %i, effect2 = %i, depth = %i\n",
        		instrument -> data.iw.effect1,
        		instrument -> data.iw.effect1_depth,
        		instrument -> data.iw.effect2,
        		instrument -> data.iw.effect2_depth );
      if ( instrument -> flags == GUS_INSTR_F_NOT_FOUND ) continue;
      if ( ( instrument -> flags & GUS_INSTR_F_ALIAS ) )
        {
          gus_iprintf( buffer, "    alias = %i (0x%x)\n", instrument -> info.alias, instrument -> info.alias );
          continue;
        }
      gus_iprintf( buffer, "    layers:\n" );
      if ( !instrument -> info.layer )
        gus_iprintf( buffer, "      -none-\n" );
      for ( layer = instrument -> info.layer, idxl = 0; layer; layer = layer -> next, idxl++ )
        {
          gus_iprintf( buffer, "      %03i <%s>\n", idxl, gus_memory_debug_instruments_get_mode( layer -> mode ) );
          if ( layer -> mode == GUS_INSTR_INTERWAVE )
            {
              gus_iprintf( buffer, "        flags = 0x%x, velocity_mode = 0x%x, layer_event = 0x%x\n",
           		   	layer -> data.iw.flags,
           		   	layer -> data.iw.velocity_mode,
           		   	layer -> data.iw.layer_event );
              gus_iprintf( buffer, "        low_range = %i, high_range = %i, pan = %i, pan_freq_scale = %i\n",
              			layer -> data.iw.low_range,
              			layer -> data.iw.high_range,
              			layer -> data.iw.pan,
              			layer -> data.iw.pan_freq_scale );
              gus_memory_debug_instruments_iw_modulation( buffer, "tremolo", &layer -> data.iw.tremolo );
              gus_memory_debug_instruments_iw_modulation( buffer, "vibrato", &layer -> data.iw.vibrato );
              gus_iprintf( buffer, "        attenuation = %i, freq_scale = %i, freq_center = %i\n",
              			layer -> data.iw.attenuation,
              			layer -> data.iw.freq_scale,
              			layer -> data.iw.freq_center );
              gus_memory_debug_instruments_iw_envelope( buffer, "pitch", &layer -> data.iw.penv );
              gus_memory_debug_instruments_iw_envelope( buffer, "volume", &layer -> data.iw.venv );
            }
          gus_iprintf( buffer, "      waves:\n" );
          if ( !layer -> wave )
            gus_iprintf( buffer, "        -none-\n" );
          for ( wave = layer -> wave, idxw = 0; wave; wave = wave -> next, idxw++ )
            {
              gus_iprintf( buffer, "        %03i at 0x%lx <%s> format=(%s%s%s%s%s%s%s%s)\n",
              		idxw,
              		(long)wave,
			gus_memory_debug_instruments_get_mode( wave -> mode ),
              		wave -> format & GUS_WAVE_16BIT ? "-16bit-" : "",
              		wave -> format & GUS_WAVE_INVERT ? "-invert-" : "",
              		wave -> format & GUS_WAVE_BACKWARD ? "-back-" : "",
              		wave -> format & GUS_WAVE_LOOP ? "-loop-" : "",
              		wave -> format & GUS_WAVE_BIDIR ? "-bidir-" : "",
              		wave -> format & GUS_WAVE_ULAW ? "-uLaw-" : "",
              		wave -> format & GUS_WAVE_RAM ? "-RAM-" : "",
              		wave -> format & GUS_WAVE_ROM ? "-ROM-" : "" );
              gus_iprintf( buffer, "          share_id=0x%x/0x%x,mem=0x%x,size=0x%x,start=0x%x,ls=0x%x,le=0x%x,lr=%i\n",
              		wave -> share_id1,
              		wave -> share_id2,
              		wave -> begin.memory,
              		wave -> size,
              		wave -> start,
              		wave -> loop_start,
              		wave -> loop_end,
              		wave -> loop_repeat );
              if ( wave -> mode == GUS_INSTR_PATCH )
                {
                  gus_iprintf( buffer, "          flags=0x%x,rate=%i,low_freq=%i,high_freq=%i,root_freq=%i\n",
                  	wave -> data.patch.flags,
                  	wave -> data.patch.sample_rate,
                  	wave -> data.patch.low_frequency,
                  	wave -> data.patch.high_frequency,
                  	wave -> data.patch.root_frequency );
                  gus_iprintf( buffer, "          tune=%i,balance=%i\n",
                  	wave -> data.patch.tune,
                  	wave -> data.patch.balance );
                  gus_iprintf( buffer, "          envelope rates (0x%x,0x%x,0x%x,0x%x,0x%x,0x%x)\n",
                  	wave -> data.patch.envelope_rate[ 0 ],
                  	wave -> data.patch.envelope_rate[ 1 ],
                  	wave -> data.patch.envelope_rate[ 2 ],
                  	wave -> data.patch.envelope_rate[ 3 ],
                  	wave -> data.patch.envelope_rate[ 4 ],
                  	wave -> data.patch.envelope_rate[ 5 ] );
                  gus_iprintf( buffer, "          envelope offsets (0x%x,0x%x,0x%x,0x%x,0x%x,0x%x)\n",
                  	wave -> data.patch.envelope_offset[ 0 ],
                  	wave -> data.patch.envelope_offset[ 1 ],
                  	wave -> data.patch.envelope_offset[ 2 ],
                  	wave -> data.patch.envelope_offset[ 3 ],
                  	wave -> data.patch.envelope_offset[ 4 ],
                  	wave -> data.patch.envelope_offset[ 5 ] );
                  gus_iprintf( buffer, "          tremolo (sweep=%i, rate=%i, depth=%i)\n",
                  	wave -> data.patch.tremolo_sweep,
                  	wave -> data.patch.tremolo_rate,
                  	wave -> data.patch.tremolo_depth );
                  gus_iprintf( buffer, "          vibrato (sweep=%i, rate=%i, depth=%i)\n",
                  	wave -> data.patch.vibrato_sweep,
                  	wave -> data.patch.vibrato_rate,
                  	wave -> data.patch.vibrato_depth );
                  gus_iprintf( buffer, "          scale_freq=%i, scale_factor=%i\n",
                  	wave -> data.patch.scale_frequency,
                  	wave -> data.patch.scale_factor );
                }
              if ( wave -> mode == GUS_INSTR_INTERWAVE )
                {
                  gus_iprintf( buffer, "          sample_ratio=%i,atten=%i,low_note=%i,high_note=%i\n",
                  	wave -> data.iw.sample_ratio,
                  	wave -> data.iw.attenuation,
                  	wave -> data.iw.low_note,
                  	wave -> data.iw.high_note );
                }
            }
        }
    }
}

#endif /* GUSCFG_DEBUG_INSTRUMENTS */

int gus_memory_dump( gus_card_t *card, struct GUS_STRU_MEMORY_DUMP *dump, int space )
{
  unsigned int dump_size, size, size1, addr;
  unsigned char *ptr, *ptrb;
  unsigned char buffer[ 512 ];
  struct GUS_STRU_MEMORY_DUMP sdump;
  
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, dump, sizeof( *dump ) ) ) return -EIO;
      MEMCPY_FROMFS( &sdump, dump, sizeof( sdump ) );
      dump = &sdump;
    }

  if ( dump -> flags & GUS_MEMORY_DUMP_ROM )
    {
      if ( dump -> flags & GUS_MEMORY_DUMP_WRITE ) return -EINVAL;
      if ( !card -> pnp_flag ) return -EINVAL;
      if ( dump -> address + dump -> size > 16L * 1024L * 1024L ) return -EINVAL;
    }
   else
    {
      if ( dump -> flags & GUS_MEMORY_DUMP_WRITE )
        {
          gus_download_over_dma( card,
                                 dump -> address,
                                 dump -> dump,
          			 dump -> size,
          			 dump -> flags & GUS_MEMORY_DUMP_INVERT,
          			 dump -> flags & GUS_MEMORY_DUMP_16BIT,
          			 SP_USER );
          return 0;
        }
      if ( dump -> address + dump -> size > 16L * 1024L * 1024L ) return -EINVAL;
    }

  ptr = dump -> dump;
  dump_size = dump -> size;
  addr = dump -> address;
  while ( dump_size )
    {
      if ( dump_size > sizeof( buffer ) ) size = sizeof( buffer ); else size = dump_size;
      dump_size -= size;
      ptrb = buffer;
      size1 = size;
      
#ifdef GUSCFG_PNP
      if ( card -> pnp_flag )
        {
          unsigned long flags;
          
          CLI( &flags );
          gus_write8( card, 0x53, dump -> flags & GUS_MEMORY_DUMP_ROM ? 0x03 : 0x01 );
          gus_dram_addr( card, addr );
          INSB( GUSP( card, DRAM ), buffer, size );
	  gus_write8( card, 0x53, 0x01 );		/* always RAM */
          STI( &flags );
          addr += size;
        }
       else
#endif
        while ( size1-- )
          *ptrb++ = gus_peek( card, addr++ );

      MEMCPY_TOFS( ptr, buffer, size );
      ptr += size;
    }
  
  return 0;  
}

/*
 *  Instrument interface...
 * 
 *  I'm using AVL tree for instrument number search...
 *  Code is simple, but tree is build after each insert/remove action...
 */

static gus_instrument_t *gus_instrument_index( gus_mem_t *alloc, int index )
{
  gus_instrument_t *result;
  
  for ( result = alloc -> instruments; index > 0; index-- )
    result = result -> next;
  return result;
}

static gus_instrument_t *gus_instrument_rebuild_tree( gus_mem_t *alloc, int min, int max )
{
  int middle;
  gus_instrument_t *result;

  if ( min > max ) return NULL;
  if ( min == max )
    return gus_instrument_index( alloc, min );
  middle = ( min + max ) >> 1;
  result = gus_instrument_index( alloc, middle );
  if ( result )
    {
      result -> left = gus_instrument_rebuild_tree( alloc, min, middle - 1 );
      result -> right = gus_instrument_rebuild_tree( alloc, middle + 1, max );
#if 0 
      if ( result -> left &&
           result -> left -> number.instrument >= result -> number.instrument )
        printk( "left failed!!!!\n" );
      if ( result -> right &&
           result -> right -> number.instrument <= result -> number.instrument )
        printk( "right failed!!!!\n" );
#if 0
      if ( alloc -> instruments_count == 15 )    
        printk( " - %i (%i) - left = (%i), right = (%i)\n",
          		middle, result -> number.instrument,
          		result -> left ? result -> left -> number.instrument : -1,
          		result -> right ? result -> right -> number.instrument : -1 );
#endif
#endif
    }
  return result;
}

static void gus_instrument_rebuild_avl( gus_mem_t *alloc )
{
  gus_instrument_t *pi;

  alloc -> instruments_locked++;
  pi = alloc -> instruments;
  while ( pi )
    {
      pi -> left = pi -> right = NULL;
      pi = pi -> next;
    }
  alloc -> instruments_root = gus_instrument_rebuild_tree( alloc, 0, alloc -> instruments_count - 1 );
  alloc -> instruments_locked--;
}

static void gus_instrument_insert( gus_mem_t *alloc, gus_instrument_t *insert )
{
  unsigned int number;
  gus_instrument_t *pi, *old_pi;

  insert -> left = insert -> right = insert -> next = NULL;
  number = insert -> number.instrument;
  pi = alloc -> instruments;
  if ( !pi )
    {
      alloc -> instruments_count = 1;
      alloc -> instruments = alloc -> instruments_root = insert;
      return;
    }
  old_pi = NULL;
  while ( pi && number > pi -> number.instrument )
    {
      old_pi = pi;
      pi = pi -> next;
    }
  if ( old_pi )
    {
      insert -> next = old_pi -> next;
      old_pi -> next = insert;
    }
   else
    {
      insert -> next = alloc -> instruments;
      alloc -> instruments = insert;
    }
  alloc -> instruments_count++;
  gus_instrument_rebuild_avl( alloc );
}

static void gus_instrument_delete( gus_mem_t *alloc, gus_instrument_t *delete )
{
  gus_instrument_t *pi, *old_pi;
  
  old_pi = NULL;
  pi = alloc -> instruments;
  while ( pi && delete -> number.instrument != pi -> number.instrument )
    {
      old_pi = pi;
      pi = pi -> next;
    }
  if ( old_pi )
    old_pi -> next = pi -> next;
   else
    alloc -> instruments = pi -> next;
  alloc -> instruments_count--;
  gus_instrument_rebuild_avl( alloc );
}
 
gus_instrument_t *gus_instrument_look( gus_mem_t *alloc, unsigned int instrument )
{
  register unsigned int number;
  register gus_instrument_t *pi;
  
  if ( alloc -> instruments_locked )	/* AVL is building */
    {
      pi = alloc -> instruments;
      while ( pi )
        {
          if ( pi -> number.instrument == instrument ) return pi;
          pi = pi -> next;
        }
      return NULL;
    }
#if 1
  pi = alloc -> instruments_root;
  while ( pi )
    {
      number = pi -> number.instrument;
      if ( instrument == number ) return pi;
      pi = instrument < number ? pi -> left : pi -> right;
    }
#else
  {
    int xx = 0;
    
    pi = alloc -> instruments_root;
    while ( pi )
      {
        number = pi -> number.instrument;
        if ( instrument == number ) return pi;
        pi = instrument < number ? pi -> left : pi -> right;
        if ( ++xx > 10 )
          {
            printk( "depth overflow - %i\n", instrument );
            break;
          }
      }
    pi = alloc -> instruments;
    while ( pi )
      {
        if ( pi -> number.instrument == instrument )
          {
            printk( "not found, instrument = %i\n", instrument );
            return pi;
          }
        pi = pi -> next;
      } 
  }
#endif
  return NULL;
}
