/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for the GF1 daemon interface
 */

#include "driver.h"
#include "midi.h"

/*
 *  -------------
 */

int gus_gf1_daemon_request_sample( gus_card_t *card, unsigned int instrument )
{
#if 0
  printk( "daemon request instrument - %i\n", instrument );
#endif
  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return -1;
  if ( card -> gf1.daemon_instruments_count >= 64 )
    {
      PRINTD( "gus: daemon request sample overflow\n" );
      return -1;
    }
  card -> gf1.daemon_instruments[ card -> gf1.daemon_instruments_count++ ] = instrument;
  return 0;
}

void gus_gf1_daemon_preload( gus_card_t *card, char *bank_name )
{
  unsigned long flags;

  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return;
  CLI( &flags );
  card -> gf1.daemon_request++;
  if ( bank_name )
    {
      card -> gf1.daemon_preload = 1;
      card -> gf1.daemon_preload_bank_name = bank_name;
    }
   else
    if ( !card -> gf1.daemon_instruments_count )
      card -> gf1.daemon_request--;
  STI( &flags );
  if ( card -> gf1.daemon_request )
    gus_gf1_daemon_wait( card );
  CLI( &flags );
  if ( bank_name )
    {
      card -> gf1.daemon_request -= card -> gf1.daemon_preload;
      card -> gf1.daemon_preload = 0;
      card -> gf1.daemon_preload_bank_name = NULL;
    }
   else
    {
      card -> gf1.daemon_request -= card -> gf1.daemon_instruments_count > 0 ? 1 : 0;
    }
  STI( &flags );
}

void gus_gf1_daemon_midi_emul_change( gus_card_t *card )
{
  unsigned long flags;

  if ( !(card -> gf1.mode & GF1_MODE_ENGINE) )
    card -> gf1.midi_emul = card -> gf1.default_midi_emul;
  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return;
  if ( card -> gf1.midi_emul != card -> gf1.daemon_midi_emul )
    {
      CLI( &flags );
      card -> gf1.daemon_request++;
      card -> gf1.daemon_emul = 1;
      STI( &flags );
      gus_gf1_daemon_wait( card );
      CLI( &flags );
      card -> gf1.daemon_request -= card -> gf1.daemon_emul;
      card -> gf1.daemon_emul = 0;
      STI( &flags );
    }
}

void gus_gf1_daemon_free_block( gus_card_t *card )
{
  unsigned long flags;

  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return;
  CLI( &flags );
  card -> gf1.daemon_request++;
  card -> gf1.daemon_free = 1;
  STI( &flags );
  gus_gf1_daemon_wait( card );
  CLI( &flags );
  card -> gf1.daemon_request -= card -> gf1.daemon_free;
  card -> gf1.daemon_free = 0;
  STI( &flags );
}

#ifdef GUSCFG_USS
void gus_gf1_daemon_iw_effect( gus_card_t *card )
{
  unsigned long flags;

  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return;
  CLI( &flags );
  card -> gf1.daemon_request++;
  card -> gf1.daemon_iw_effect = 1;
  STI( &flags );
  gus_gf1_daemon_wait( card );
  CLI( &flags );
  card -> gf1.daemon_request -= card -> gf1.daemon_iw_effect;
  card -> gf1.daemon_iw_effect = 0;
  STI( &flags );
}
#endif

void gus_gf1_daemon_wait( gus_card_t *card )
{
  if ( !(card -> gf1.mode & GF1_MODE_DAEMON) ) return;
  if ( card -> gf1.daemon_request )
    card -> gf1.daemon_active = 1;
  while ( card -> gf1.daemon_active || card -> gf1.daemon_lock )
    {
      if ( !card -> gf1.daemon_lock && (GETLOCK( card, daemon ) & WK_SLEEP) )
        {
          GETLOCK( card, daemon ) &= ~WK_SLEEP;
          WAKEUP( card, daemon );
        }
      GETLOCK( card, daemon_wait ) |= WK_SLEEP;
      SLEEP( card, daemon_wait, 600 * HZ );
      GETLOCK( card, daemon_wait ) &= ~WK_SLEEP;
#if 0	/* don't change this.. gusd is another process and we must wait for command complete */
      if ( TABORT( daemon_wait ) ) break;
#endif
    }
}

/*
 *  -------------
 */
 
static void gus_gf1_daemon_clear( gus_card_t *card )
{
  card -> gf1.daemon_lock = 0;
  card -> gf1.daemon_request = 0;
  card -> gf1.daemon_preload = 0;
  card -> gf1.daemon_emul = 0;
  card -> gf1.daemon_free = 0;
  card -> gf1.daemon_instruments_count = 0;
  card -> gf1.daemon_midi_emul = card -> gf1.default_midi_emul;
}
 
static int gus_open_gf1_daemon_card( gus_card_t *card )
{
  if ( card -> gf1.mode & GF1_MODE_DAEMON ) return -EBUSY;
  gus_gf1_daemon_clear( card );
  card -> gf1.mode |= GF1_MODE_DAEMON; 
  return 0;
}

static void gus_close_gf1_daemon_card( gus_card_t *card )
{
  if ( card )
    {
      card -> gf1.mode &= ~GF1_MODE_DAEMON;
      gus_gf1_daemon_clear( card );
      gus_memory_reset_daemon( card, &card -> gf1.mem_alloc, 1 );
    }
}

/*
 *  -------------
 */

int gus_open_gf1_daemon( unsigned short minor, struct file *file )
{
  int res;

  if ( ( file -> f_flags & O_ACCMODE ) == O_WRONLY ||
       ( file -> f_flags & O_ACCMODE ) == O_RDONLY ) return -EACCES;
  minor &= GUS_MINOR_GDEVMASK;
  if ( minor >= gus_cards_count ) return -ENODEV;
  if ( ( res = gus_open_gf1_daemon_card( gus_cards[ minor ] ) ) < 0 )
    return res;
  file -> private_data = gus_cards[ minor ];
  MOD_INC_USE_COUNT;
  return 0;
}

void gus_release_gf1_daemon( struct file *file )
{
  gus_close_gf1_daemon_card( (gus_card_t *)file -> private_data );
  file -> private_data = NULL;
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_gf1_daemon( struct file *file, unsigned int cmd, unsigned long arg )
{
  gus_card_t *card;
  
  if ( ( cmd & 0xff00 ) != ( 'g' << 8 ) ) return -EINVAL;
  card = file -> private_data;
  switch ( cmd ) {
    case GUS_IOCTL_DMN_INFO:
      return gus_info( card, NULL, (struct GUS_STRU_INFO *)arg );
    case GUS_IOCTL_DMN_INSTRUMENT:
      {
        struct GUS_STRU_INSTRUMENT *instr; 
        
        instr = gus_instrument_look( &card -> gf1.mem_alloc, IOCTL_IN( arg ) );
        return IOCTL_OUT( arg, !instr ? GUS_INSTR_F_NOT_LOADED : instr -> flags );
      }
      break;
    case GUS_IOCTL_DMN_FREE:
      return IOCTL_OUT( arg, gus_memory_free_size( card, &card -> gf1.mem_alloc ) );
    case GUS_IOCTL_DMN_MEMORY_DUMP:
      return gus_memory_dump( card, (struct GUS_STRU_MEMORY_DUMP *)arg, SP_USER );
    case GUS_IOCTL_DMN_MEMORY_BALLOC:
      return gus_memory_block_alloc( card, (struct GUS_STRU_MEMORY_BLOCK *)arg, GUS_MEMORY_LOCK_DAEMON, SP_USER );
    case GUS_IOCTL_DMN_MEMORY_BFREE:
      return gus_memory_block_free( card, (struct GUS_STRU_MEMORY_BLOCK *)arg, GUS_MEMORY_LOCK_DAEMON, SP_USER );
    case GUS_IOCTL_DMN_LOCK:
      if ( card -> gf1.daemon_lock ) return -EAGAIN;
      card -> gf1.daemon_lock = 1;
      break;
    case GUS_IOCTL_DMN_UNLOCK:
      if ( !card -> gf1.daemon_lock ) return -EIO;
      if ( GETLOCK( card, daemon_wait ) & WK_SLEEP )
        {
          GETLOCK( card, daemon_wait ) &= ~WK_SLEEP;
          WAKEUP( card, daemon_wait );
        }
      card -> gf1.daemon_lock = 0;
      break;
    case GUS_IOCTL_DMN_GET_EMULATION:
      return IOCTL_OUT( arg, card -> gf1.midi_emul );
  }
  return -EIO;
}

int gus_read_gf1_daemon( struct file *file, char *buf, int count )
{
  gus_card_t *card;
  struct GUS_STRU_DAEMON_MESSAGE message;
  int i;

  card = file -> private_data;
  if ( !card -> gf1.daemon_request ) return 0;
  if ( count < sizeof( message ) ) return -EIO;
  MEMSET( &message, 0, sizeof( message ) );
  message.direction = GUS_DAEMON_MSG_REQUEST;
#ifdef GUSCFG_USS
  if ( card -> gf1.daemon_iw_effect )
    {
      message.command = GUS_DAEMON_MSG_IWEFF;
      message.info.iw_effect[ 0 ] = card -> seq.iw_effect[ 0 ];
      message.info.iw_effect[ 1 ] = card -> seq.iw_effect[ 1 ];
      card -> gf1.daemon_iw_effect = 0;
    }
   else
#endif
  if ( card -> gf1.daemon_free )
    {
      message.command = GUS_DAEMON_MSG_BFREE;
      card -> gf1.daemon_free = 0;
    }
   else
  if ( card -> gf1.daemon_emul )
    {
      message.command = GUS_DAEMON_MSG_EMUL;
      message.info.emul = card -> gf1.daemon_midi_emul = card -> gf1.midi_emul;
      card -> gf1.daemon_emul = 0;
    }
   else
  if ( card -> gf1.daemon_preload )
    {
      message.command = GUS_DAEMON_MSG_PRELOAD;
      strcpy( message.info.bank_name, card -> gf1.daemon_preload_bank_name );
      card -> gf1.daemon_preload = 0;
    }
   else
  if ( card -> gf1.daemon_instruments_count > 0 )
    {
      message.command = GUS_DAEMON_MSG_DOWNLOAD;
      MEMSET( &message.info.instruments, 0xff, sizeof( message.info.instruments ) );
      for ( i = 0; i < card -> gf1.daemon_instruments_count; i++ )
        message.info.instruments[ i ] = card -> gf1.daemon_instruments[ i ];
      card -> gf1.daemon_instruments_count = 0;
    }
#if 0
  printk( "message.command = 0x%x, requests = %i\n", message.command, card -> gf1.daemon_request );
#endif
  if ( message.command == 0 )
    {
#ifdef GUSCFG_DEBUG
      printk( "gus: unknown daemon command - aborting..\n" );
#endif
      gus_gf1_daemon_clear( card );
      return -EIO;
    }
  card -> gf1.daemon_request--;
  MEMCPY_TOFS( buf, &message, sizeof( message ) );
  return sizeof( message );
}

static void gus_daemon_download( gus_card_t *card, struct GUS_STRU_DAEMON_MESSAGE *message )
{
  if ( !( card -> gf1.mode & GF1_MODE_ENGINE ) ) return;
  gus_memory_alloc( card, &card -> gf1.mem_alloc, message -> info.instrument, MTST_NONE, SP_USER );
}

int gus_write_gf1_daemon( struct file *file, char *buf, int count )
{
  gus_card_t *card;
  int res;
  struct GUS_STRU_DAEMON_MESSAGE message;

  card = file -> private_data;
  if ( count != sizeof( message ) ) return -EIO;
  MEMCPY_FROMFS( &message, buf, sizeof( message ) );
  if ( message.direction != GUS_DAEMON_MSG_REPLY ) return -EINVAL;
  if ( !( card -> gf1.mode & GF1_MODE_DAEMON ) ) return -EINTR;
#if 0
  PRINTK( "ok.. we have reply - %i\n", message.command );
#endif
  res = 0;
  switch ( message.command ) {
    case GUS_DAEMON_MSG_END_OF_REQUEST:
      if ( card -> gf1.daemon_active )
        {
          card -> gf1.daemon_active = 0;
          if ( GETLOCK( card, daemon_wait ) & WK_SLEEP )
            {
              GETLOCK( card, daemon_wait ) &= ~WK_SLEEP;
              WAKEUP( card, daemon_wait );
            }
        }
      break;
    case GUS_DAEMON_MSG_DOWNLOAD:
      gus_daemon_download( card, &message );
      break;
#ifdef GUSCFG_USS
    case GUS_DAEMON_MSG_IWEFF:
      gus_effects_setup( card, message.info.effect, SP_USER );
      break;
#endif
    default:
      PRINTK( "gus: wrong daemon write command %i\n", message.command );
  }
  return sizeof( message );
}

#ifdef GUS_POLL
unsigned int gus_poll_gf1_daemon( struct file *file, poll_table *wait )
{
  gus_card_t *card;
  unsigned long flags;
  unsigned int mask;

  card = (gus_card_t *)file -> private_data;
  if ( !card ) return 0;
  
  CLI( &flags );
  GETLOCK( card, daemon ) |= WK_SLEEP;
  SLEEP_POLL( card, daemon, wait );
  STI( &flags );
  
  mask = 0;
  if ( card -> gf1.daemon_request )
    mask |= POLLIN | POLLRDNORM;

  return mask;
}
#else
int gus_select_gf1_daemon( struct file *file, int sel_type, select_table *wait )
{
  unsigned long flags;
  gus_card_t *card;

  card = (gus_card_t *)file -> private_data;
  switch ( sel_type ) {
    case SEL_IN:
      CLI( &flags );
      if ( !card -> gf1.daemon_request )
        {
          GETLOCK( card, daemon ) |= WK_SLEEP;
          SLEEPS( card, daemon, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( card, daemon ) &= ~WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      return 1;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

void gus_gf1_daemon_init( gus_card_t *card )
{
}
