/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#define GUS_MAIN_OBJECT_FILE
#include "driver.h"
#ifdef __alpha__
#include "gus_cfg.h"
#endif

char *gus_config_string;
int gus_config_dma_static = 0;
#ifdef MODULE_PARM              /* hey - we have new 2.1.18+ kernel... */
MODULE_AUTHOR( "Jaroslav Kysela <perex@jcu.cz>" );
MODULE_DESCRIPTION( "Low-level driver for the Gravis UltraSound cards." );
MODULE_SUPPORTED_DEVICE( "sound" );
MODULE_PARM( gus_config_string, "s" );
MODULE_PARM_DESC( gus_config_string, "Not for user hands... Used by insgus!!!" );
MODULE_PARM( gus_config_dma_static, "i" );
MODULE_PARM_DESC( gus_config_dma_static, "Not for user hands... Used by insgus!!!" );
#endif

#ifdef LINUX_2_1
static long long gus_lseek( struct inode *inode, struct file *file, long long offset, int orig )
#else
static int gus_lseek( struct inode *inode, struct file *file, off_t offset, int orig )
#endif
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:
#endif
    case GUS_MINOR_GINFO:	return -EIO;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:
    case GUS_MINOR_GSYNTH_CTL:
    case GUS_MINOR_GINSMAN:
    case GUS_MINOR_GMIXER:
    case GUS_MINOR_GPCM:	return -EIO;
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return -EIO;
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_lseek_sequencer( file, offset, orig );
    case GUS_MINOR_MIDI:
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;
}

#ifdef LINUX_2_1
static long gus_read( struct inode *inode, struct file *file, char *buf, unsigned long count )
#else
static int gus_read( struct inode *inode, struct file *file, char *buf, int count )
#endif
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_read_midi( file, buf, count );
#endif
    case GUS_MINOR_GINFO:
      return gus_read_info( buf, (off_t *)&file -> f_pos, count );
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_read_gf1( file, buf, count );
    case GUS_MINOR_GSYNTH_CTL:	return gus_read_gf1_ctl( file, buf, count );
    case GUS_MINOR_GINSMAN:	return gus_read_gf1_daemon( file, buf, count );
    case GUS_MINOR_GMIXER:	return -EIO;
    case GUS_MINOR_GPCM:	return gus_read_pcm( file, buf, count );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return -EIO;
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_read_pcm( file, buf, count );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_read_sequencer( file, buf, count );
    case GUS_MINOR_MIDI:	return -EIO;
    case GUS_MINOR_SNDSTAT:
      return gus_read_sndstat( buf, (off_t *)&file -> f_pos, count );
#endif
  }
  return -ENODEV;
}

#ifdef LINUX_2_1
static long gus_write( struct inode *inode, struct file *file, const char *buf, unsigned long count )
#else
static int gus_write( struct inode *inode, struct file *file, const char *buf, int count )
#endif
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_write_midi( file, (char *)buf, count );
#endif
    case GUS_MINOR_GINFO:	return gus_write_info( (char *)buf, count );
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_write_gf1( file, (char *)buf, count );
    case GUS_MINOR_GSYNTH_CTL:  return gus_write_gf1_ctl( file, (char *)buf, count );
    case GUS_MINOR_GINSMAN:	return gus_write_gf1_daemon( file, (char *)buf, count );
    case GUS_MINOR_GMIXER:	return -EIO;
    case GUS_MINOR_GPCM:	return gus_write_pcm( file, (char *)buf, count );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return -EIO;
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_write_pcm( file, (char *)buf, count );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_write_sequencer( file, (char *)buf, count );
    case GUS_MINOR_MIDI:	return -EIO;
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;
}

static int gus_open( struct inode *inode, struct file *file )
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_open_midi( file );
#endif
    case GUS_MINOR_GINFO:	return gus_open_info();
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_open_gf1( minor, file );
    case GUS_MINOR_GSYNTH_CTL:	return gus_open_gf1_ctl( minor, file );
    case GUS_MINOR_GINSMAN:	return gus_open_gf1_daemon( minor, file );
    case GUS_MINOR_GMIXER:	return gus_open_mixer( minor, file );
    case GUS_MINOR_GPCM:	return gus_open_pcm( minor, file );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return gus_open_mixer( minor, file );
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_open_pcm( minor, file );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_open_sequencer( minor, file );
    case GUS_MINOR_MIDI:	return gus_open_oss_midi( minor, file );
    case GUS_MINOR_SNDSTAT:	return gus_open_sndstat();
#endif
  }
  printk( "gus: uknown minor number - %i\n", minor );
  return -ENODEV;
}

static int gus_release( struct inode * inode, struct file *file )
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	gus_release_midi( file );	return 0;
#endif
    case GUS_MINOR_GINFO:	gus_release_info();		return 0;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	gus_release_gf1( file );	return 0;
    case GUS_MINOR_GSYNTH_CTL:	gus_release_gf1_ctl( file );	return 0;
    case GUS_MINOR_GINSMAN:	gus_release_gf1_daemon( file ); return 0;
    case GUS_MINOR_GMIXER:	gus_release_mixer( file );	return 0;
    case GUS_MINOR_GPCM:	gus_release_pcm( file );	return 0;
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	gus_release_mixer( file );	return 0;
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	gus_release_pcm( file );	return 0;
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	gus_release_sequencer( file );	return 0;
    case GUS_MINOR_MIDI:	gus_release_oss_midi( file );	return 0;
    case GUS_MINOR_SNDSTAT:	gus_release_sndstat();		return 0;
#endif
  }
  return -ENODEV;
}

#ifdef GUS_POLL
static unsigned int gus_poll( struct file *file, poll_table *wait )
{
#if LinuxVersionCode( 2, 1, 45 ) > LINUX_VERSION_CODE
  unsigned short minor = MINOR( file -> f_inode -> i_rdev );
#else
  unsigned short minor = MINOR( file -> f_dentry -> d_inode -> i_rdev );
#endif

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_poll_midi( file, wait );
#endif
    case GUS_MINOR_GINFO:	return -EIO;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_poll_gf1( file, wait );
    case GUS_MINOR_GSYNTH_CTL:	return -EIO;
    case GUS_MINOR_GINSMAN:	return gus_poll_gf1_daemon( file, wait );
    case GUS_MINOR_GMIXER:	return -EIO;
    case GUS_MINOR_GPCM:	return gus_poll_pcm( file, wait );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return -EIO;
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_poll_pcm( file, wait );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_poll_sequencer( file, wait );
    case GUS_MINOR_MIDI:
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;
}                       
#else
static int gus_select( struct inode *inode, struct file *file, 
                       int sel_type, select_table *wait )
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_select_midi( file, sel_type, wait );
#endif
    case GUS_MINOR_GINFO:	return -EIO;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_select_gf1( file, sel_type, wait );
    case GUS_MINOR_GSYNTH_CTL:	return -EIO;
    case GUS_MINOR_GINSMAN:	return gus_select_gf1_daemon( file, sel_type, wait );
    case GUS_MINOR_GMIXER:	return -EIO;
    case GUS_MINOR_GPCM:	return gus_select_pcm( file, sel_type, wait );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return -EIO;
    case GUS_MINOR_PCM_8: 
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_select_pcm( file, sel_type, wait );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_select_sequencer( file, sel_type, wait );
    case GUS_MINOR_MIDI:
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;
}                       
#endif

static int gus_ioctl( struct inode *inode, struct file *file,
		      unsigned int cmd, unsigned long arg )
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:	return gus_ioctl_midi( file, cmd, arg );
#endif
    case GUS_MINOR_GINFO:	return -EIO;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:	return gus_ioctl_gf1( file, cmd, arg );
    case GUS_MINOR_GSYNTH_CTL:	return gus_ioctl_gf1_ctl( file, cmd, arg );
    case GUS_MINOR_GINSMAN:	return gus_ioctl_gf1_daemon( file, cmd, arg );
    case GUS_MINOR_GMIXER:	return gus_ioctl_mixer( NULL, file, cmd, arg );
    case GUS_MINOR_GPCM:	return gus_ioctl_pcm( file, cmd, arg );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return gus_ioctl_mixer( NULL, file, cmd, arg );
    case GUS_MINOR_PCM_8:
    case GUS_MINOR_PCM_16: 
    case GUS_MINOR_AUDIO:	return gus_ioctl_pcm( file, cmd, arg );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:	return gus_ioctl_sequencer( file, cmd, arg );
    case GUS_MINOR_MIDI:
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;
}

static int gus_mmap( struct inode *inode, struct file *file, struct vm_area_struct *vma )
{
  unsigned short minor = MINOR( inode -> i_rdev );

  switch ( minor ) {
#ifdef GUSCFG_MIDI
    case GUS_MINOR_GMIDI:
#endif
    case GUS_MINOR_GINFO:	return -EIO;
  }
  switch ( minor >> 3 ) {
    case GUS_MINOR_GSYNTH:
    case GUS_MINOR_GSYNTH_CTL:
    case GUS_MINOR_GINSMAN:
    case GUS_MINOR_GMIXER:
    case GUS_MINOR_GPCM:	return gus_mmap_pcm( inode, file, vma );
  }
  switch ( minor & GUS_MINOR_USS_MASK ) {
    case GUS_MINOR_MIXER:	return -EIO;
    case GUS_MINOR_PCM_8:
    case GUS_MINOR_PCM_16:
    case GUS_MINOR_AUDIO:	return gus_mmap_pcm( inode, file, vma );
#ifdef GUSCFG_USS
    case GUS_MINOR_SEQUENCER:
    case GUS_MINOR_MUSIC:
    case GUS_MINOR_MIDI:
    case GUS_MINOR_SNDSTAT:	return -EIO;
#endif
  }
  return -ENODEV;  
}

static struct file_operations gus_fops = {
  gus_lseek,	/* (l)lseek */
  gus_read,	/* read */
  gus_write,	/* write */
  NULL,		/* readdir */
#ifdef GUS_POLL
  gus_poll,	/* poll */
#else
  gus_select,	/* select */
#endif
  gus_ioctl,	/* ioctl */
  gus_mmap,	/* mmap */
  gus_open,	/* open */
#ifdef LINUX_2_1
  gus_release,	/* release */
#else
  (void (*)( struct inode * inode, struct file *file ))gus_release,
#endif
};

#ifndef LINUX_2_1

int gus_ioctl_in( long *addr )
{
  if ( verify_area( VERIFY_READ, addr, sizeof( int ) ) != 0 )
    {
      printk( "gus_ioctl_in: bad address 0x%lx\n", (long)addr );
      return 0;
    }
  return get_fs_int( addr );
}

int gus_ioctl_out( long *addr, int value )
{
  int err;
  
  if ( ( err = verify_area( VERIFY_WRITE, addr, sizeof( int ) ) ) != 0 )
    return err;
  put_fs_int( value, addr );
  return 0;
}

#else

int gus_ioctl_in( long *addr )
{
  int value;
  
  get_user( value, addr );
  return value;
}

int gus_ioctl_out( long *addr, int value )
{
  int err;

  if ( ( err = verify_area( VERIFY_WRITE, addr, sizeof( int ) ) ) != 0 )
    return err;
  put_user( value, addr );
  return 0;
}

#endif /* !LINUX_2_1 */

static void gus_print_info( void )
{
  int i, j, k, count;
  gus_card_t *card, *cards[ GUS_CARDS ];

  MEMSET( cards, 0, sizeof( cards ) );
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      cards[ card -> number % GUS_CARDS ] = card;
    }
  for ( i = j = count = 0; i < GUS_CARDS; i++ )
    {
      card = cards[ i ];
      if ( !card )
        {
          for ( k = 0; k < GUS_CARDS; k++ )
            {
              card = cards[ k ];
              if ( !card ) continue;
              if ( card -> number >= i ) card -> number--;
            }
          continue;
        }
      gus_cards[ j++ ] = card;
      count++;
    }
  gus_cards_count = count;
}

static struct GUS_STRU_CFG *gus_get_cfg( int card )
{
  static struct GUS_STRU_CFG cfg;
  char *dest, *src;
  int cnt;
  
#if 0
  PRINTK( "card = %i\n", card );
  PRINTK( "config = %s\n", &gus_config_string[ 8 + ( card * sizeof( struct GUS_STRU_CFG ) * 2 ) ] );
#endif
  for ( src = &gus_config_string[ 8 + ( card * sizeof( struct GUS_STRU_CFG ) * 2 ) ],
        dest = (char *)&cfg, cnt = sizeof( cfg ); cnt > 0; cnt-- )
    {
      *dest = ( *src >= 'A' ? ( *src - 'A' ) + 10 : *src - '0' ) << 4;
      src++;
      *dest |= *src >= 'A' ? ( *src - 'A' ) + 10 : *src - '0';
      src++;
      dest++;
    }
#if 0
  PRINTK( "gus: mix_gf1 = %i, dma1_size = %i\n", cfg.mix_gf1, cfg.dma1_size );
#endif
  return &cfg;
}

static void gus_free_resources( gus_card_t *card )
{
  if ( card -> dma1_ok ) { disable_dma( card -> dma1 ); free_dma( card -> dma1 ); }
  if ( card -> dma2_ok ) { disable_dma( card -> dma2 ); free_dma( card -> dma2 ); }
  if ( card -> gf1_irq_ok ) { disable_irq( card -> gf1.irq ); gus_free_irq( card -> gf1.irq, card ); }
#ifdef GUSCFG_GUSDB16
  if ( card -> dmad_ok ) { disable_dma( card -> dmad ); free_dma( card -> dmad ); }
  if ( card -> codec_irq_ok ) { disable_irq( card -> codec.irq ); gus_free_irq( card -> codec.irq, card ); }
#endif
}

static int gus_look_for_next_card( gus_card_t *card )
{
  if ( gus_probe( card ) ) return -ENODEV;
#if 0
  PRINTK( "gus_probe ok (0x%x)\n", card -> gf1.port );
#endif
  if ( gus_init_pre( card ) ) return -ENODEV;
  if ( gus_request_irq( card -> gf1.irq, gus_interrupt, SA_GUS_FLAGS, "gus", card ) )
    {
      gus_free_resources( card );
      PRINTK( "gus: unable to grab IRQ %d\n", (int)card -> gf1.irq );
      return -EIO;
    }
  card -> gf1_irq_ok = 1;
  if ( request_dma( card -> dma1, card -> equal_dma ? "gus" : "gus (dma1)" ) )
    {
      gus_free_resources( card );
      PRINTK( "gus: unable to grab DMA1 %d\n", (int)card -> dma1 );
      return -EIO;
    }
  card -> dma1_ok = 1;
  if ( !card -> equal_dma )
    {
      if ( request_dma( card -> dma2, "gus (dma2)" ) )
        {
          gus_free_resources( card );
          PRINTK( "gus: unable to grab DMA2 %d\n", (int)card -> dma2 );
          return -EIO;
        }
      card -> dma2_ok = 1;
    }
#ifdef GUSCFG_GUSDB16
  if ( card -> daughter_flag )
    {
      if ( gus_request_irq( card -> codec.irq, gus_interrupt_db16, SA_GUS_FLAGS, "gus DB16", card ) )
        {
          gus_free_resources( card );
          PRINTK( "gus: unable to grab DB16 IRQ %d\n", (int)card -> codec.irq );
          return -EIO;
        }
      card -> codec_irq_ok = 1;
      if ( request_dma( card -> dmad, "gus DB16" ) )
        {
          gus_free_resources( card );
          PRINTK( "gus: unable to grab DB16 DMA %d\n", (int)card -> dmad );
          return -EIO;
        }
      card -> dmad_ok = 1;
      gus_cards_irq_index[ card -> codec.irq ] = card;
      enable_irq( card -> codec.irq );
    }
#endif
  gus_cards_irq_index[ card -> gf1.irq ] = card;
  enable_irq( card -> gf1.irq );
  if ( gus_config_dma_static )
    {
      if ( gus_dma_malloc( card, GUS_DMA_GPLAY, "static dma1", -1 ) < 0 )
        {
          gus_free_resources( card );
          PRINTK( "gus: unable to allocate DMA1 buffer\n" );
          return -ENOMEM;
        }
      if ( !card -> equal_dma )
        if ( gus_dma_malloc( card, GUS_DMA_GRECORD, "static dma2", -1 ) < 0 )
          {
            gus_free_resources( card );
            gus_dma_free( card, GUS_DMA_GPLAY, -1 );
            PRINTK( "gus: unable to allocate DMA2 buffer\n" );
            return -ENOMEM;
          }
#ifdef GUSCFG_GUSDB16
      if ( card -> daughter_flag )
        {
          if ( gus_dma_malloc( card, GUS_DMA_CPLAY, "static db16 dma", -1 ) < 0 )
            {
              gus_free_resources( card );
              gus_dma_free( card, GUS_DMA_GPLAY, -1 );
              if ( !card -> equal_dma )
                gus_dma_free( card, GUS_DMA_GRECORD, -1 );
              PRINTK( "gus: unable to allocate DB16 DMA buffer\n" );
              return -ENOMEM;
            }
        }
#endif
    }
  if ( gus_init( card ) )
    {
      gus_free_resources( card );
      return -ENODEV;
    }
  request_region( card -> gf1.port, 16,			    "gus (region 1)" );
  request_region( card -> gf1.port + 0x100, 16,		    "gus (region 2)" );
  request_region( card -> gf1.port + ( 0x726 - 0x220 ), 1,  "gus (region 3)" );
#ifdef GUSCFG_GUSDB16
  if ( card -> daughter_flag )
    request_region( card -> codec.port, 4, "gus (DB16)" );
#endif
  return 0;
}

static void gus_cleanup_card( gus_card_t *card )
{
  if ( !card ) return;
  if ( card -> gf1_irq_ok ) disable_irq( card -> gf1.irq );
  gus_done( card );
  gus_free_resources( card );
#if 0
  if ( card -> dmas[ GUS_DMA_GPLAY ] -> mmaped )
    gus_free_pages( card -> dmas[ GUS_DMA_GPLAY ] -> buf, card -> dmas[ GUS_DMA_GPLAY ] -> size );
  if ( card -> dmas[ GUS_DMA_GRECORD ] -> mmaped )
    gus_free_pages( card -> dmas[ GUS_DMA_GRECORD ] -> buf, card -> dmas[ GUS_DMA_GRECORD ] -> size );
#endif
#if 0
  if ( card -> dmas[ GUS_DMA_GPLAY ] -> used )
    PRINTK( "gus: Warning! DMA %i is still allocated!!!\n", card -> dmas[ GUS_DMA_GPLAY ] -> dma );
  if ( !card -> equal_dma && card -> dmas[ GUS_DMA_GRECORD ] -> used )
    PRINTK( "gus: Warning! DMA %i is still allocated!!!\n", card -> dmas[ GUS_DMA_GRECORD ] -> dma );
  if ( card -> daughter_flag && card -> dmas[ GUS_DMA_GPLAY ] -> used )
    PRINTK( "gus: Warning! DMA %i is still allocated!!!\n", card -> dmas[ GUS_DMA_CPLAY ] -> dma );
#endif
  if ( gus_config_dma_static )
    {
      gus_dma_free( card, GUS_DMA_GPLAY, -1 );
      if ( !card -> equal_dma )
        gus_dma_free( card, GUS_DMA_GRECORD, -1 );
#ifdef GUSCFG_GUSDB16
      if ( card -> daughter_flag )
        gus_dma_free( card, GUS_DMA_CPLAY, -1 );
#endif
    }
  release_region( card -> gf1.port, 16 );
  release_region( card -> gf1.port + 0x100, 16 );
  release_region( card -> gf1.port + ( 0x726 - 0x220 ), 1 );
#ifdef GUSCFG_GUSDB16
  if ( card -> daughter_flag ) release_region( card -> codec.port, 4 );
#endif
  gus_free( card, sizeof( gus_card_t ) );
}
  
int init_module( void )
{
  gus_card_t *card;
  int gus_cards_to_probe;
  struct GUS_STRU_CFG *cfg;
  
  if ( gus_config_string == NULL || strncmp( gus_config_string, "GUS-CFG*", 8 ) )
    {
      PRINTK( "gus: please, use 'insgus' program for insert gus module to kernel\n" );
      return -EIO;
    }
  if ( ( ( strlen( gus_config_string ) - 8 ) % ( sizeof( struct GUS_STRU_CFG ) * 2 ) ) != 0 )
    {
      PRINTK( "gus: bad size of configuration array (wrong version of insgus?)\n" );
      return -EIO;
    }
  gus_cards_to_probe = ( strlen( gus_config_string ) - 8 ) / ( sizeof( struct GUS_STRU_CFG ) * 2 );
  if ( gus_cards_to_probe > GUS_CARDS ) gus_cards_to_probe = GUS_CARDS;
  gus_cards_count = 0;
  if ( register_chrdev( GUSCFG_MAJOR, "gus", &gus_fops ) )
    {
      PRINTK("gus: unable to get major device number %d\n", GUSCFG_MAJOR);
      return -EIO;
    }
  gus_malloc_init();
  gus_driver_init();
#if 0
  PRINTK( "gus: sizeof( gus_card_t ) = %i\n", sizeof( gus_card_t ) );
#endif
  do {
    card = NULL;
    if ( gus_cards_to_probe-- <= 0 ) break;
    cfg = gus_get_cfg( gus_cards_count );
    if ( cfg -> version != GUS_STRU_CFG_VERSION )
      {
        PRINTK( "gus: wrong configuration for card #%i (wrong version of insgus?)\n", gus_cards_count + 1 );
        break;
      }
    card = gus_malloc( sizeof( gus_card_t ) );
    gus_card_init( card, cfg, gus_cards_count );
    gus_cards[ gus_cards_count ] = card;
    gus_cards_count++;
  } while ( !gus_look_for_next_card( card ) );
  if ( card )
    {
      gus_free( card, sizeof( gus_card_t ) );
      gus_cards[ --gus_cards_count ] = NULL;
    }
  if ( !gus_cards_count )
    {
      unregister_chrdev( GUSCFG_MAJOR, "gus" );
      printk( "gus: card not found...\n" );
      return -ENODEV;
    }
  gus_print_info();
  return 0;
}

void cleanup_module( void )
{
  short i;

  for ( i = 0; i < gus_cards_count; i++ )
    gus_cleanup_card( gus_cards[ i ] );
  gus_malloc_done();
  if ( unregister_chrdev( GUSCFG_MAJOR, "gus" ) != 0 )
    PRINTK( "gus_cleanup: failed" );
}
 