/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of CS4231 (CODEC) chip
 */
  
#include  "driver.h"
#include  "pcm.h"

#ifdef GUSCFG_CODEC

/* playback flags */

#define PFLG_NONE	0x0000
#define PFLG_USED	0x0001
#define PFLG_FLUSH	0x0002

/* record flags */

#define RFLG_NONE	0x0000
#define RFLG_USED	0x0001
#define RFLG_FLUSH	0x0002

/*
 *  Some variables
 */

static struct GUS_STRU_CODEC_FREQ codec_freq[14] = {
  {  5,  5510, 0x00 | CODEC_XTAL2 },
  {  6,  6620, 0x0E | CODEC_XTAL2 },
  {  8,  8000, 0x00 | CODEC_XTAL1 },
  {  9,  9600, 0x0E | CODEC_XTAL1 },
  { 11, 11025, 0x02 | CODEC_XTAL2 },
  { 16, 16000, 0x02 | CODEC_XTAL1 },
  { 18, 18900, 0x04 | CODEC_XTAL2 },
  { 22, 22050, 0x06 | CODEC_XTAL2 },
  { 27, 27042, 0x04 | CODEC_XTAL1 },
  { 32, 32000, 0x06 | CODEC_XTAL1 },
  { 33, 33075, 0x0C | CODEC_XTAL2 },
  { 37, 37800, 0x08 | CODEC_XTAL2 },
  { 44, 44100, 0x0A | CODEC_XTAL2 },
  { 48, 48000, 0x0C | CODEC_XTAL1 }
};

static struct GUS_STRU_CODEC_IMAGE codec_original_image = {
  0x00,						/* 00 - lic */
  0x00,						/* 01 - ric */
  0x80,						/* 02 - la1ic */
  0x80,						/* 03 - ra1ic */
  0x80,						/* 04 - la2ic */
  0x80,						/* 05 - ra2ic */
  0x80,						/* 06 - loc */
  0x80,						/* 07 - roc */
  0x20,						/* 08 - pdfr */
  CODEC_AUTOCALIB,				/* 09 - ic */
  0x00,						/* 0a - pc */
  0x00,						/* 0b - ti */
  CODEC_MODE2,					/* 0c - mi */
  0x00,						/* 0d - lbc */
  0x00,						/* 0e - pbru */
  0x00,						/* 0f - pbrl */
  0x80,						/* 10 - afei */
  0x01,						/* 11 - afeii */
  0x00,						/* 12 - llic */
  0x00,						/* 13 - rlic */
  0x00,						/* 14 - tlb */
  0x00,						/* 15 - thb */
  0x80,						/* 16 - la3ic */
  0x80,						/* 17 - ra3ic */
  0x00,						/* 18 - afs */
  0x80,						/* 19 - lamic */
  0x00,						/* 1a - mioc */
  0x80,						/* 1b - ramic */
  0x20,						/* 1c - cdfr */
  0x00,						/* 1d - res1 */ 
  0x00,						/* 1e - cbru */
  0x00,						/* 1f - cbrl */
};

/*
 *  CODEC detection
 */
  
static void codec_mce_up( gus_card_t *card )
{
  unsigned long flags;
  int timeout = 1000;
  
  while ( timeout-- > 0 && ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT ) );
  CLI( &flags );
  card -> codec.mce_bit = CODEC_MCE;
  if ( INB( CODECP( card, REGSEL ) ) & CODEC_MCE )
    {
      STI( &flags );
      return;
    }
  OUTB( CODEC_MCE, CODECP( card, REGSEL ) );
  STI( &flags );
}
  
static void codec_mce_down( gus_card_t *card )
{
  unsigned long flags;
  int timeout = 1000;
  
  while ( timeout-- > 0 && ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT ) );
  CLI( &flags );
  card -> codec.mce_bit = 0x00;
  timeout = INB( CODECP( card, REGSEL ) );
  OUTB( 0x00, CODECP( card, REGSEL ) );
  if ( ( timeout & CODEC_MCE ) == 0 )
    {
      STI( &flags );
      return;
    }
  OUTB( 0x00, CODECP( card, REGSEL ) );
  STI( &flags );
  /* calibration process */
  timeout = 100000;
  while ( timeout-- > 0 && ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT ) );
  if ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT )
    PRINTK( "gus: codec_mce_down - auto calibration time out (1)\n" );
  timeout = 100;
  while ( timeout-- > 0 && ( codec_in( card, CODEC_TEST_INIT ) & CODEC_CALIB_IN_PROGRESS ) == 0 );
  if ( ( codec_in( card, CODEC_TEST_INIT ) & CODEC_CALIB_IN_PROGRESS ) == 0 ) return;
  timeout = 80000;
  while ( timeout-- > 0 && ( codec_in( card, CODEC_TEST_INIT ) & CODEC_CALIB_IN_PROGRESS ) );
  if ( codec_in( card, CODEC_TEST_INIT ) & CODEC_CALIB_IN_PROGRESS )
    PRINTK( "gus: codec_mce_down - auto calibration time out (2)\n" );
}

static unsigned int codec_get_count( unsigned char format, unsigned int size )
{
  switch ( format & 0xe0 ) {
    case CODEC_LINEAR_16:
    case CODEC_LINEAR_16_BIG:
      size >>= 1;
      break;
    case CODEC_ADPCM_16:
      return size >> 2;
  }
  if ( format & CODEC_STEREO ) size >>= 1;
  return size;
}

#if 0
static void codec_zero( gus_card_t *card )
{
  unsigned char format, silence, status;
  unsigned int size, i, timeout;
  
  card -> codec.image.pc &= ~CODEC_IRQ_ENABLE;
  codec_out( card, CODEC_PIN_CTRL, card -> codec.image.pc );

  format = codec_in( card, CODEC_PLAYBK_FORMAT );
  silence = ( format & 0xe0 ) == CODEC_LINEAR_8 ? 0x80 : 0x00;
  format = ( format & 0xf0 ) | 0x0b;
  codec_mce_up( card );
  codec_out( card, CODEC_PLAYBK_FORMAT, format );
  codec_mce_down( card );
  size = codec_get_count( format, 32 ) - 1;
  codec_out( card, CODEC_PLY_LWR_CNT, (unsigned char)size );
  codec_out( card, CODEC_PLY_UPR_CNT, (unsigned char)(size >> 8) );
  card -> codec.image.ic |= CODEC_PLAYBACK_ENABLE | CODEC_PLAYBACK_PIO;
  codec_mce_up( card );
  codec_out( card, CODEC_IFACE_CTRL, card -> codec.image.ic );
  codec_mce_down( card );
  for ( i = 0, timeout = 10000; i < 32 && timeout > 0; timeout-- )
    {
      OUTB( silence, CODECP( card, PIO ) );
      status = codec_in( card, CODEC_IRQ_STATUS );
      if ( status == 0xff ) break;
      if ( !(status & 0x02) ) i++; else INB( CODECP( card, STATUS ) );
    }
  if ( !timeout )
    PRINTK( "codec_zero: timeout (1)\n" );
  timeout = 10000;
  do {
    status = codec_in( card, CODEC_IRQ_STATUS );
    if ( status == 0xff ) break;
    timeout--;
  } while ( !(status & 0x01) && timeout > 0 );
  if ( !timeout )
    PRINTK( "codec_zero: timeout (2)\n" );
  card -> codec.image.ic &= ~(CODEC_PLAYBACK_ENABLE | CODEC_PLAYBACK_PIO);
  codec_mce_up( card );
  codec_out( card, CODEC_IFACE_CTRL, card -> codec.image.ic );
  codec_mce_down( card );

  card -> codec.image.pc |= CODEC_IRQ_ENABLE;
  codec_out( card, CODEC_PIN_CTRL, card -> codec.image.pc );
}
#endif

static short codec_version( gus_card_t *card )
{
  short res;

  codec_out( card, CODEC_MISC_INFO, card -> codec.image.mi );
  res = INB( CODECP( card, REG ) ) & 0x0f;
  return res;
}

int codec_init( gus_card_t *card )
{
  unsigned long flags;
  short i, j;
  short version;
  char *ptr;

  CLI( &flags );
  INB( CODECP( card, STATUS ) );	/* clear any pendings IRQ */
  OUTB( 0, CODECP( card, STATUS ) );
  MB();
  STI( &flags );
  
#if 1
  card -> codec.image.mi = card -> pnp_flag ? CODEC_MODE3 : CODEC_MODE2;
#else
  card -> codec.image.mi = CODEC_MODE2;
#endif

  for ( i = 0; i < 1000; i++ )
    {
      MB();
      if ( INB( CODECP( card, REGSEL ) ) & CODEC_INIT )
        for ( j = 0; j < 50; j++ ) gus_delay( card );
       else
        {
          version = codec_version( card );
          if ( version >= 1 && version < 15 ) break;
        }
    }
  
  version = codec_version( card );
  if ( version >= 1 && version < 15 )
    {
      card -> codec.image.ic = 
        ( card -> codec.image.ic & ~CODEC_SINGLE_DMA ) | 
        ( ( card -> daughter_flag || card -> equal_dma ) ? CODEC_SINGLE_DMA : 0 );
      card -> codec.image.afei = 0x80;
      card -> codec.image.afeii = card -> codec.image.mi == CODEC_MODE3 ? 0xc2 : 0x01;
      ptr = (char *)&card -> codec.image;
      codec_mce_down( card );
      for ( i = 0; i < 32; i++ )		/* ok.. fill all CODEC registers */
        codec_out( card, i, *ptr++ );
      codec_mce_up( card );
      codec_mce_down( card );
      return version;
    }
  return 0;
}

/*
 *  CODEC I/O
 */

static unsigned char codec_get_freq( gus_card_t *card, unsigned int freq )	/* freq in Hz */
{
  int i;
  
  freq /= 1000;
#if 0
  PRINTK( "pcm_rate: %d\n", freq );
#endif
  if ( freq > 48 ) freq = 48; 
  for ( i = 0; i < 14; i++ )
    if ( freq <= codec_freq[ i ].hertz )
      return codec_freq[ i ].bits;
  return codec_freq[ 13 ].bits;
}

static unsigned char codec_get_format( gus_card_t *card, unsigned int mode, int voices )
{
  unsigned char format;

  format = 0;
  if ( mode & PCM_MODE_16 )
    {
      if ( mode & PCM_MODE_ADPCM )
        format |= CODEC_ADPCM_16;
       else
      if ( mode & PCM_MODE_BIG )
        format |= CODEC_LINEAR_16_BIG;
       else
        format |= CODEC_LINEAR_16;
    }
   else
    {
      if ( mode & PCM_MODE_ALAW )
        format |= CODEC_ALAW_8;
       else
      if ( mode & PCM_MODE_ULAW )
        format |= CODEC_ULAW_8;
#if 0
       else
        format |= CODEC_LINEAR_8;	/* I know, OR with 0 */
#endif
    }
  if ( voices == 2 )
    format |= CODEC_STEREO;
#if 0
  PRINTK( "codec_get_format: 0x%x (mode=0x%x)\n", format, mode );
#endif
  return format;
}

static void codec_set_play_format( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  card -> codec.image.pdfr = 
    codec_get_format( card, pchn -> mode, pchn -> voices ) |
    codec_get_freq( card, pchn -> rate );
#if 0
  PRINTK( ">>> pmode = 0x%x, pdfr = 0x%x\n", card -> pcm.chn[ PCM_PLAYBACK ].mode, card -> codec.image.pdfr );
#endif
}
 
static void codec_set_rec_format( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_RECORD ];
  card -> codec.image.cdfr = 
    codec_get_format( card, pchn -> mode, pchn -> voices ) |
    codec_get_freq( card, pchn -> rate );
#if 0
  PRINTK( ">>> rmode = 0x%x, cdfr = 0x%x\n", card -> pcm.chn[ PCM_RECORD ].mode, card -> codec.image.cdfr );
#endif
}

#if 0

/*
 *  prepared for future use
 */

static void codec_timer_start( gus_card_t *card, unsigned int value )
{
  if ( value > 0xffff ) value = 0xffff;
  card -> codec.image.afei |= CODEC_TIMER_ENABLE;
  codec_out( card, CODEC_TIMER_LOW, (unsigned char)value );
  codec_out( card, CODEC_TIMER_HIGH, (unsigned char)( value >> 8 ) );
  codec_out( card, CODEC_ALT_FEATURE_1, card -> codec.image.afei );
  card -> codec.timer_value = value;
}

static void codec_timer_stop( gus_card_t *card )
{
  card -> codec.image.afei &= ~CODEC_TIMER_ENABLE;
  codec_out( card, CODEC_ALT_FEATURE_1, card -> codec.image.afei );
}

#endif

void codec_open( gus_card_t *card )
{
  if ( card -> codec.mode & CODEC_MODE_OPEN ) return;
  codec_mce_down( card );
  codec_out( card, CODEC_ALT_FEATURE_1, card -> codec.image.afei );
  codec_out( card, CODEC_TRD | CODEC_ALT_FEATURE_2, card -> codec.image.afeii );
  codec_set_rec_format( card );
  codec_set_play_format( card );
  card -> codec.image.ic &= ~(CODEC_PLAYBACK_ENABLE|CODEC_PLAYBACK_PIO|
                              CODEC_RECORD_ENABLE|CODEC_RECORD_PIO);
  codec_mce_up( card );
  codec_out( card, CODEC_IFACE_CTRL, card -> codec.image.ic );
  codec_out( card, CODEC_PLAYBK_FORMAT, card -> codec.image.pdfr );
  if ( card -> codec.image.mi != CODEC_MODE3 )
    codec_out( card, CODEC_REC_FORMAT, card -> codec.image.cdfr );
   else
    codec_out( card, CODEC_REC_FORMAT, card -> codec.image.pdfr );
  codec_mce_down( card );
  /* ok. now enable CODEC IRQ */
  codec_out( card, CODEC_IRQ_STATUS, CODEC_PLAYBACK_IRQ | 
                                     CODEC_RECORD_IRQ | 
                                     CODEC_TIMER_IRQ );
  codec_out( card, CODEC_IRQ_STATUS, 0 );
  OUTB( 0, CODECP( card, STATUS ) );		/* clear IRQ */
  card -> codec.image.pc |= CODEC_IRQ_ENABLE;
  codec_out( card, CODEC_PIN_CTRL, card -> codec.image.pc );
  codec_out( card, CODEC_IRQ_STATUS, CODEC_PLAYBACK_IRQ | 
                                     CODEC_RECORD_IRQ | 
                                     CODEC_TIMER_IRQ );
  codec_out( card, CODEC_IRQ_STATUS, 0 );
  card -> mixer.codec_mute_output = 1;
  codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
}

void codec_close( gus_card_t *card )
{
  if ( card -> codec.mode & CODEC_MODE_OPEN ) return;
  card -> mixer.codec_mute_output = 1;
  codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
  /* now disable record & playback */
  card -> codec.image.ic &= ~(CODEC_PLAYBACK_ENABLE|CODEC_PLAYBACK_PIO|
                              CODEC_RECORD_ENABLE|CODEC_RECORD_PIO);
  codec_mce_up( card );
  codec_out( card, CODEC_IFACE_CTRL, card -> codec.image.ic );
  codec_mce_down( card );
  /* now disable CODEC IRQ */
  codec_out( card, CODEC_IRQ_STATUS, 0 );
  OUTB( 0, CODECP( card, STATUS ) );		/* clear IRQ */
  card -> codec.image.pc &= ~CODEC_IRQ_ENABLE;
  codec_out( card, CODEC_PIN_CTRL, card -> codec.image.pc );
  /* disable both dma channels */
  if ( card -> pcm.flags & PCM_LFLG_RECORD )
    disable_dma( card -> dmas[ GUS_DMA_CRECORD ] -> dma );
  if ( card -> pcm.flags & PCM_LFLG_PLAY )
    disable_dma( card -> dmas[ GUS_DMA_CPLAY ] -> dma );
}

/*
 *
 */

static void codec_trigger( gus_card_t *card, unsigned char what, int enable )
{
#if 0
  printk( "codec trigger!!! - what = %i, enable = %i\n", what, enable );
#endif
  if ( enable )
    {
      if ( card -> codec.image.ic & what ) return;
      card -> codec.image.ic |= what;
    }
   else
    {
      if ( !(card -> codec.image.ic & what) ) return;
      card -> codec.image.ic &= ~what;
    }
  /* codec_mce_up( card ); */
  codec_out( card, CODEC_IFACE_CTRL, card -> codec.image.ic );	/* go! */
  /* codec_mce_down( card ); */
}
 
static void codec_start_playback( gus_card_t *card )
{
  unsigned long flags;
  int size;
  short gus_dma2;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  
  codec_set_play_format( card );
  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  switch ( card -> codec.interwave_serial_port ) {
    case ( 2 << 5 ) | 0x1f:
    case ( 3 << 5 ) | 0x1f:
      gus_i_write8( card, 0x59, card -> codec.interwave_serial_port );
      break;
  }
#if 0
  PRINTK( "codec_start_playback\n" );
#endif
  gus_dma2 = card -> dmas[ GUS_DMA_CPLAY ] -> dma;
  card -> codec.image.ic &= ~(CODEC_PLAYBACK_ENABLE|CODEC_PLAYBACK_PIO);
  if ( card -> codec.image.mi == CODEC_MODE3 ||
       ( codec_in( card, CODEC_IFACE_CTRL ) & CODEC_RECORD_ENABLE ) == 0 )
    {
      codec_mce_up( card );
      codec_out( card, CODEC_PLAYBK_FORMAT, card -> codec.image.pdfr );
      if ( card -> codec.image.mi != CODEC_MODE3 )
        codec_out( card, CODEC_REC_FORMAT, card -> codec.image.pdfr );
      codec_mce_down( card );
    }
  CLI( &flags );
  disable_dma( gus_dma2 );
  if ( card -> max_flag && gus_dma2 > 3 )
    {
      OUTB( card -> max_cntrl_val & ~0x20, GUSP( card, MAXCNTRLPORT ) );
      OUTB( card -> max_cntrl_val, GUSP( card, MAXCNTRLPORT ) );
    }
  STI( &flags );
  clear_dma_ff( gus_dma2 );
  set_dma_mode( gus_dma2, DMA_MODE_WRITE | 0x10 );
  set_dma_addr( gus_dma2, (long)virt_to_bus( pchn -> buffer ) );
  clear_dma_ff( gus_dma2 );
  set_dma_count( gus_dma2, size = pchn -> used_size );
  enable_dma( gus_dma2 );
  size = codec_get_count( card -> codec.image.pdfr, size );
  size /= pchn -> blocks;
  size--;
  codec_out( card, CODEC_PLY_LWR_CNT, (unsigned char)size );
  codec_out( card, CODEC_PLY_UPR_CNT, (unsigned char)(size >> 8) );
  pchn -> locks = 1 << pchn -> tail;
  codec_trigger( card, CODEC_PLAYBACK_ENABLE, 1 );
  pchn -> flags |= PCM_FLG_TRIGGER;
  card -> codec.pflags |= PFLG_USED;
  card -> mixer.codec_mute_output = 0;	/* enable output */
  codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
#if 0
  codec_debug( card );
#endif
#if 0
  PRINTK( "residue0: 0x%x\n", get_dma_residue( card -> dma2 ) );
#endif
}

static void codec_stop_playback( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;
  short gus_dma2;

#if 0
  printk( "stop_playback\n" );
#endif
  card -> mixer.codec_mute_output = 1;
  codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
  if ( card -> codec.pflags & PFLG_FLUSH ) { WAKEUP( card, playback ); }
  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  pchn -> locks = 0;
  card -> codec.pflags &= ~( PFLG_USED | PFLG_FLUSH );	/* done */
  codec_trigger( card, CODEC_PLAYBACK_ENABLE, 0 );
  gus_dma2 = card -> dmas[ GUS_DMA_CPLAY ] -> dma;
  disable_dma( gus_dma2 );
  clear_dma_ff( gus_dma2 );
  pchn -> flags &= ~PCM_FLG_TRIGGER;
  switch ( gus_i_look8( card, 0x59 ) ) {
    case ( 2 << 5 ) | 0x1f:
    case ( 3 << 5 ) | 0x1f:
      gus_i_write8( card, 0x59, 0x1f );
      break;
  }
}

static void codec_start_record( gus_card_t *card )
{
  unsigned long flags;
  int size;
  short gus_dma1;
  struct GUS_STRU_PCM_CHANNEL *pchn;
  
  codec_set_rec_format( card );
  switch ( card -> codec.interwave_serial_port ) {
    case ( 1 << 5 ) | 0x1f:
    case ( 5 << 5 ) | 0x1f:
      gus_i_write8( card, 0x59, card -> codec.interwave_serial_port );
      break;
  }
  pchn = &card -> pcm.chn[ PCM_RECORD ];
#if 0
  PRINTK( "codec_start_record: start (size=0x%x)\n", size );
#endif
  gus_dma1 = card -> dmas[ GUS_DMA_CRECORD ] -> dma;
  card -> codec.image.ic &= ~(CODEC_RECORD_ENABLE|CODEC_RECORD_PIO);
  if ( card -> codec.image.mi == CODEC_MODE3 ||
       ( codec_in( card, CODEC_IFACE_CTRL ) & CODEC_PLAYBACK_ENABLE ) == 0 )
    {
      codec_mce_up( card );
      codec_out( card, CODEC_REC_FORMAT, card -> codec.image.cdfr );
      if ( card -> codec.image.mi != CODEC_MODE3 )
        codec_out( card, CODEC_PLAYBK_FORMAT, card -> codec.image.cdfr );
      codec_mce_down( card );
    }
  CLI( &flags );
  disable_dma( gus_dma1 );
  if ( card -> max_flag && gus_dma1 > 3 )
    {
      OUTB( card -> max_cntrl_val & ~0x10, GUSP( card, MAXCNTRLPORT ) );
      OUTB( card -> max_cntrl_val, GUSP( card, MAXCNTRLPORT ) );
    }
  STI( &flags );
  clear_dma_ff( gus_dma1 );
  set_dma_mode( gus_dma1, DMA_MODE_READ | 0x10 );
  set_dma_addr( gus_dma1, (long)virt_to_bus( pchn -> buffer ) );
  set_dma_count( gus_dma1, size = pchn -> used_size );
  enable_dma( gus_dma1 );
  size = codec_get_count( card -> codec.image.cdfr, size );
  size /= pchn -> blocks;
  size--;
  codec_out( card, CODEC_REC_LWR_CNT, (unsigned char)size );
  codec_out( card, CODEC_REC_UPR_CNT, (unsigned char)(size >> 8) );
  pchn -> locks = 1 << pchn -> head;
  codec_trigger( card, CODEC_RECORD_ENABLE, 1 );
  pchn -> flags |= PCM_FLG_TRIGGER;
  card -> codec.rflags |= RFLG_USED;
}

static void codec_stop_record( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;
  short gus_dma1;

  if ( card -> codec.rflags & PFLG_FLUSH ) { WAKEUP( card, record ); }
  pchn = &card -> pcm.chn[ PCM_RECORD ];
  pchn -> locks = 0;
  card -> codec.rflags &= ~( RFLG_USED | RFLG_FLUSH );	/* done */
  codec_trigger( card, CODEC_RECORD_ENABLE, 0 );
  gus_dma1 = card -> dmas[ GUS_DMA_CRECORD ] -> dma;
  disable_dma( gus_dma1 );
  clear_dma_ff( gus_dma1 );
  pchn -> flags &= ~PCM_FLG_TRIGGER;
  switch ( gus_i_look8( card, 0x59 ) ) {
    case ( 1 << 5 ) | 0x1f:
    case ( 5 << 5 ) | 0x1f:
      gus_i_write8( card, 0x59, 0x1f );
      break;
  }
}

void codec_interrupt( gus_card_t *card, unsigned char status )
{
#ifdef IN_INTERRUPT_DEBUG
  static in_interrupt = 0;
#endif
  struct GUS_STRU_PCM_CHANNEL *pchn;

#ifdef IN_INTERRUPT_DEBUG
  if ( in_interrupt )
    PRINTK( "gus: Eiaaa, interrupt routine reentered - %d times (codec)!!!\n", in_interrupt );
  in_interrupt++;
#endif

#if 0
  PRINTK( "codec_interrupt: status=0x%x\n", status );
#endif

  if ( ( card -> codec.pflags & PFLG_USED ) && ( status & CODEC_PLAYBACK_IRQ ) )
    {
#if 0
      PRINTK( "residue: 0x%x\n", get_dma_residue( card -> dmas[ GUS_DMA_CPLAY ] -> dma ) );
#endif
      pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
      if ( !( pchn -> flags & PCM_FLG_MMAP ) )
        {
          MEMSET( pchn -> buffer + pchn -> block_size * pchn -> tail, pchn -> neutral_byte, 16 );
          if ( pchn -> used == 1 )
            {
              card -> mixer.codec_mute_output = 1;
              codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
              codec_trigger( card, CODEC_PLAYBACK_ENABLE, 0 );
              pchn -> flags &= ~PCM_FLG_TRIGGER;
              if ( !(pchn -> flags & PCM_FLG_SYNC) )
                pchn -> discarded++;
            }
          pchn -> processed_bytes += pchn -> sizes[ pchn -> tail ];
          pchn -> sizes[ pchn -> tail++ ] = 0;
          pchn -> tail %= pchn -> blocks;
          pchn -> used--;
          pchn -> locks = pchn -> used > 0 ? 1 << pchn -> tail : 0;
        }
       else 
        {
          pchn -> interrupts++;
          pchn -> processed_bytes += pchn -> block_size;
          pchn -> tail++;
          pchn -> tail %= pchn -> blocks;
        }
      if ( !pchn -> used && ( card -> codec.pflags & PFLG_FLUSH ) )
        {
          codec_stop_playback( card );
          card -> codec.pflags &= ~PFLG_FLUSH;
          WAKEUP( card, playback );
        }
      if ( pchn -> flags & PCM_FLG_SLEEP )
        {
          pchn -> flags &= ~PCM_FLG_SLEEP;
          WAKEUP( card, playback );
        }
    }
    
  if ( ( card -> codec.rflags & RFLG_USED ) && ( status & CODEC_RECORD_IRQ ) )
    {
      pchn = &card -> pcm.chn[ PCM_RECORD ];
      if ( !( pchn -> flags & PCM_FLG_MMAP ) )
        {
          if ( pchn -> used < pchn -> blocks )
            pchn -> used++;
           else
            card -> codec.record_overflow++;
          pchn -> sizes[ pchn -> head ] = pchn -> block_size;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
          pchn -> processed_bytes += pchn -> block_size;
          pchn -> locks = 1 << pchn -> head;
        }
       else
        {
          pchn -> interrupts++;
          pchn -> processed_bytes += pchn -> block_size;
          pchn -> head++;
          pchn -> head %= pchn -> blocks;
        }
      if ( card -> codec.rflags & RFLG_FLUSH )	/* stop this as soon is possible */
        codec_stop_record( card );
      if ( pchn -> flags & PCM_FLG_SLEEP )
        {
          pchn -> flags &= ~PCM_FLG_SLEEP;
          WAKEUP( card, record );
        }
    }

#ifdef IN_INTERRUPT_DEBUG
  in_interrupt--;
#endif
#if 0
  PRINTK( "codec_interrupt: end\n" );
#endif
}

/*
 *
 */

static int codec_open_playback( gus_card_t *card )
{
  if ( gus_dma_malloc( card, GUS_DMA_CPLAY, "CODEC PCM - playback", 1 ) < 0 )
    return -ENOMEM;
  gus_pcm_set_dma_struct( card, GUS_DMA_CPLAY, &card -> pcm.chn[ PCM_PLAYBACK ] );
  codec_open( card );
  card -> codec.mode |= CODEC_MODE_PLAY;
  card -> codec.pflags = RFLG_NONE;
  return 0;
}

static int codec_open_record( gus_card_t *card )
{
  if ( gus_dma_malloc( card, GUS_DMA_CRECORD, "CODEC PCM - record", 1 ) < 0 )
    return -ENOMEM;
  gus_pcm_set_dma_struct( card, GUS_DMA_CRECORD, &card -> pcm.chn[ PCM_RECORD ] );
  codec_open( card );
  card -> codec.mode |= CODEC_MODE_RECORD;
  card -> codec.rflags = RFLG_NONE;
  return 0;
}

static void codec_close_playback( gus_card_t *card )
{
  if ( card -> codec.pflags & PFLG_USED )
    codec_stop_playback( card );
  card -> codec.pflags = RFLG_NONE;
  card -> codec.mode &= ~CODEC_MODE_PLAY;  
  codec_close( card );
  gus_dma_free( card, GUS_DMA_CPLAY, 1 );
}

static void codec_close_record( gus_card_t *card )
{
  if ( card -> codec.rflags & RFLG_USED )
    codec_stop_record( card );
  card -> codec.rflags = RFLG_NONE;
  card -> codec.mode &= ~CODEC_MODE_RECORD;
  codec_close( card );
  gus_dma_free( card, GUS_DMA_CRECORD, 1 );
}

static void codec_init_playback( gus_card_t *card )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  if ( !(pchn -> flags & PCM_FLG_MMAP) && !pchn -> used ) return;
  if ( !( card -> codec.pflags & PFLG_USED ) )
    codec_start_playback( card );
   else
    {
      if ( card -> mixer.codec_mute_output )
        {
          card -> mixer.codec_mute_output = 0;
          codec_set_mute( card, CODEC_OUTPUT, -1, -1 );
        }
      codec_trigger( card, CODEC_PLAYBACK_ENABLE, 1 );
      card -> pcm.chn[ PCM_PLAYBACK ].flags |= PCM_FLG_TRIGGER;
    }
}

static void codec_done_playback( gus_card_t *card )
{
#if 0
  printk( "codec_done_playback!!!\n" );
#endif
  if ( card -> codec.pflags & PFLG_USED )
    {
      card -> codec.pflags |= PFLG_FLUSH;
      while ( card -> codec.pflags & PFLG_USED )
        {
          SLEEP( card, playback, HZ * 10 );
          if ( card -> codec.pflags & PFLG_USED )
            {
              if ( TABORT( playback ) )
                card -> pcm.chn[ PCM_PLAYBACK ].flags |= PCM_FLG_ABORT;
               else
              if ( TIMEOUT( card, playback ) )
                PRINTK( "pcm: flush failed (playback)\n" );
               else
                continue;
              codec_stop_playback( card );
              card -> codec.pflags = PFLG_NONE;
            }
        }
      card -> codec.pflags = PFLG_NONE;
    }
}

static void codec_init_record( gus_card_t *card )
{
  if ( !( card -> codec.rflags & RFLG_USED ) )
    codec_start_record( card );
}

static void codec_done_record( gus_card_t *card )
{
  if ( card -> codec.rflags & RFLG_USED )
    {
#if 0          
      codec_stop_record( card );
#else
      card -> codec.rflags |= RFLG_FLUSH;
      while ( card -> codec.rflags & RFLG_USED )
        {
          SLEEP( card, record, HZ * 10 );
          if ( card -> codec.rflags & RFLG_USED )
            {
              if ( TABORT( record ) )
                card -> pcm.chn[ PCM_RECORD ].flags |= PCM_FLG_ABORT;
               else
              if ( TIMEOUT( card, record ) )
                PRINTK( "pcm: flush failed (record)\n" );
               else
                continue;
              codec_stop_record( card );
              card -> codec.rflags = PFLG_NONE;
            }
        }
#endif
    }
}

static void codec_dma_playback( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn, char *buffer, unsigned int count )
{
  unsigned char *dest;

  dest = pchn -> buffer + ( pchn -> head * pchn -> block_size ) + pchn -> sizes[ pchn -> head ];
  MEMCPY_FROMFS( dest, buffer, count );
}

static void codec_dma_playback_neutral( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn )
{
  unsigned int size;
  unsigned int count;
  unsigned char *dest;
  unsigned char neutral;
  
  if ( card -> codec.pflags & PFLG_USED )
    {
      size = pchn -> sizes[ pchn -> head ];
      count = pchn -> block_size - size;
      neutral = pchn -> neutral_byte;
      if ( count > 0 )
        {
          dest = pchn -> buffer + ( pchn -> head * pchn -> block_size ) + size;
          if ( size < pchn -> block_size ) neutral = *( dest - ( pchn -> mode & PCM_MODE_BIG ? 2 : 1 ) );
          MEMSET( dest, neutral, count );
        }
    }
}

static void codec_dma_record( gus_card_t *card, struct GUS_STRU_PCM_CHANNEL *pchn, char *buffer, unsigned int count )
{
  MEMCPY_TOFS( buffer,
               pchn -> buffer + ( pchn -> tail * pchn -> block_size ) + ( pchn -> block_size - pchn -> sizes[ pchn -> tail ] ),
               count );
}

static void codec_sync_playback( gus_card_t *card )
{
  codec_init_playback( card );
#if 0
  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  while ( pchn -> used > 0 )
    {
      pchn -> flags |= PCM_FLG_SLEEP;
      SLEEP( card, playback, HZ * 10 );
      if ( pchn -> used > 0 )
        {
          if ( TABORT( playback ) )
            {
              pchn -> flags |= PCM_FLG_ABORT;
              return;
            }
        }
      pchn -> flags &= ~PCM_FLG_SLEEP;
    }
#endif
  codec_done_playback( card );
}

static void codec_sync_record( gus_card_t *card )
{
  codec_done_record( card );
}

static unsigned int codec_pointer_playback( gus_card_t *card )
{
  unsigned int res = 0;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  if ( ( pchn -> flags & ( PCM_FLG_TRIGGER | PCM_FLG_MMAP ) ) == ( PCM_FLG_TRIGGER | PCM_FLG_MMAP ) )
    res = pchn -> used_size - get_dma_residue( card -> dmas[ GUS_DMA_CPLAY ] -> dma );
  return res;
}

static unsigned int codec_pointer_record( gus_card_t *card )
{
  unsigned int res = 0;
  struct GUS_STRU_PCM_CHANNEL *pchn;

  pchn = &card -> pcm.chn[ PCM_RECORD ];
  if ( ( pchn -> flags & ( PCM_FLG_TRIGGER | PCM_FLG_MMAP ) ) == ( PCM_FLG_TRIGGER | PCM_FLG_MMAP ) )
    res = pchn -> used_size - get_dma_residue( card -> dmas[ GUS_DMA_CRECORD ] -> dma );
  return res;
}

/*
 *
 */
 
void gus_codec_info( gus_card_t *card, gus_info_buffer_t *buffer )
{
  if ( !card -> use_codec ) return;
  gus_iprintf( buffer, "CODEC:\n" );
  gus_iprintf( buffer, "  mode               : %s\n", card -> codec.image.mi != CODEC_MODE3 ? "2" : "3" );
  gus_iprintf( buffer, "  record overflow    : %i\n", card -> codec.record_overflow );
  if ( card -> pnp_flag )
    {
      gus_iprintf( buffer, "  playback fifo      : %i\n", card -> codec.playback_fifo_size );
      gus_iprintf( buffer, "  record fifo        : %i\n", card -> codec.record_fifo_size );
    }
}

/*
 *
 */

void gus_init_codec_pcm( gus_card_t *card, int image )
{
  struct GUS_STRU_PCM_CHANNEL *pchn;

  if ( image )
    {
      MEMCPY( &card -> codec.image, &codec_original_image, sizeof( codec_original_image ) );
      card -> codec.mce_bit = 0x00;
    }
  card -> codec.pflags = PFLG_NONE;
  card -> codec.rflags = RFLG_NONE;
  
  pchn = &card -> pcm.chn[ PCM_PLAYBACK ];
  pchn -> hw_open = codec_open_playback;
  pchn -> hw_close = codec_close_playback;
  pchn -> hw_init = codec_init_playback;
  pchn -> hw_done = codec_done_playback;
  pchn -> hw_dma = codec_dma_playback;
  pchn -> hw_dma_neutral = codec_dma_playback_neutral;
  pchn -> hw_sync = codec_sync_playback;
  pchn -> hw_pointer = codec_pointer_playback;

  pchn++;
  pchn -> hw_open = codec_open_record;
  pchn -> hw_close = codec_close_record;
  pchn -> hw_init = codec_init_record;
  pchn -> hw_done = codec_done_record;
  pchn -> hw_dma = codec_dma_record;
  pchn -> hw_dma_neutral = NULL;
  pchn -> hw_sync = codec_sync_record;
  pchn -> hw_pointer = codec_pointer_record;
}

#endif /* GUSCFG_CODEC */
