/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Native synthesizer support for GF(A)1 chip
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */

#if 0
#define DEBUG_QUEUE
#endif

#define FILE_GUS		"/dev/gus%i"
#define FILE_GUS_CTL		"/dev/gusctl%i"
 
/*
 *  structures
 */

typedef struct {
  int card;
  int handle;
  int memory_mode;
  int non_block;
  unsigned char *queue_buffer;
  size_t queue_buffer_size;
  unsigned char *queue_ptr;
  size_t queue_len;
  int info_flag;
  struct GUS_STRU_INFO info_data;
  int active_voices;
  int channel_voices;
  unsigned int freq_control;
  unsigned short volume_control;
  unsigned short pan_control;
} CARD;

/*
 *  variables
 */

static CARD *gus_card_ptrs[ GUS_CARDS ] =
			{ NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
static CARD *acard;

/*
 *  local functions
 */

static void gus_reset_variables( void )
{
  acard -> info_flag = 0;
  acard -> queue_ptr = acard -> queue_buffer;
  acard -> queue_len = acard -> queue_buffer_size;
}

static CARD *gus_alloc_card( int card, size_t queue_buffer_size )
{
  acard = gus_card_ptrs[ card ];
  if ( !acard )
    {
      acard = gus_card_ptrs[ card ] = (CARD *)malloc( sizeof( CARD ) );
      if ( !acard ) return NULL;
    }
  acard -> card = card;
  acard -> handle = -1;
  acard -> queue_buffer = NULL;
 
  /* init variables */

  if ( queue_buffer_size > 0 &&
       ( acard -> queue_buffer = (unsigned char *)malloc( queue_buffer_size ) ) == NULL )
    {
      gus_dprintf( "gus_open: allocation error" );
      free( acard -> queue_buffer );
      return NULL;
    }

  acard -> queue_buffer_size = queue_buffer_size;
  acard -> queue_ptr = acard -> queue_buffer;
  acard -> queue_len = acard -> queue_buffer_size;
  gus_reset_variables();
  return acard;
}

/*
 *  EXPORTED FUNCTIONS
 */

int gus_cards( void )
{
  int handle, cards;
  char filename[ 16 ];
  
  sprintf( filename, FILE_GUS_CTL, 0 );
  if ( ( handle = open( filename, O_RDWR ) ) < 0 )
    {
      gus_dprintf( "gus_cards: open - %s", strerror( errno ) );
      return 0;
    }
  if ( ioctl( handle, GUS_IOCTL_CTL_CARDS, &cards ) < 0 )
    {
      gus_dprintf( "gus_cards: ioctl" );
      cards = 0;
    }
  close( handle );
  return cards;
}

int gus_look_for_card( char *id )
{
  char filename[ 16 ];
  int card, fd;
  gus_info_t info;
  
  card = atoi( id );
  if ( card >= 1 && card <= GUS_CARDS )
    {
      card--;
      sprintf( filename, FILE_GUS_CTL, card );
      if ( ( fd = open( filename, O_RDONLY ) ) < 0 ) return -1;
      close( fd );
      return card;
    }
  for ( card = gus_cards() - 1; card >= 0; card-- )
    {
      sprintf( filename, FILE_GUS_CTL, card );
      if ( ( fd = open( filename, O_RDONLY ) ) < 0 ) continue;
      if ( ioctl( fd, GUS_IOCTL_CTL_INFO, &info ) < 0 ) continue;
      close( fd );
      if ( !strncmp( info.id, id, sizeof( info.id ) ) )
        return card;
    }
  return -1;
}

/*
 *  ----------------- open / close
 */

int gus_open( int card, size_t queue_buffer_size, int non_block )
{
  int handle;
  int i;
  char filename[ 16 ];

  errno = 0;
  
  if ( card < 0 || card >= GUS_CARDS )
    {
      gus_dprintf( "gus_open: requested card %i out of range", card );
      return -1;
    }
  if ( gus_card_ptrs[ card ] && gus_card_ptrs[ card ] -> handle >= 0 )
    {
      gus_dprintf( "gus_open: device already open" );
      return -1;
    }
  if ( queue_buffer_size < 512 || queue_buffer_size > 1024 * 1024 )
    {
      gus_dprintf( "gus_open: queue buffer size isn't valid" );
      return -1;
    }
    
  sprintf( filename, FILE_GUS, card );
  if ( ( handle = open( filename, O_RDWR | ( non_block ? O_NONBLOCK : 0 ) ) ) < 0 )
    {
      gus_dprintf( "gus_open: %s", strerror( errno ) );
      return -1;
    }
  if ( ioctl( handle, GUS_IOCTL_VERSION, &i ) < 0 )
    {
      gus_dprintf( "gusOpen: gus ioctl - VERSION" );
      close( handle );
      return -1;
    }
  
  if ( ( i >> 8 ) != ( GUS_SYNTH_VERSION >> 8 ) ||
       ( GUS_SYNTH_VERSION & 0xff ) > ( i & 0xff ) )
    {
      gus_dprintf( "gus_open: uncompatible version of gus driver" );
      gus_dprintf( "gus_open: required version of synth protocol %i.%02i (driver reports %i.%02i)",
		 	      GUS_SYNTH_VERSION >> 8, GUS_SYNTH_VERSION & 0xff,
			  i >> 8, i & 0xff );
      close( handle );
      return -1;
    }

  if ( !gus_alloc_card( card, queue_buffer_size ) )
    {
      gus_dprintf( "gus_open: error by allocation of card\n" );
      close( handle );
      return -1;
    }
  acard -> handle = handle;
  acard -> non_block = non_block != 0;
  acard -> freq_control = GUS_FREQ_HZ;
  acard -> volume_control = GUS_VOLUME_LINEAR;
  acard -> pan_control = GUS_PAN_LINEAR;
 
  /* default setup */
  
  return card;
}

int gus_close( int card )
{
  int res;
  
  if ( gus_select( card ) ) return -1;
  res = close( acard -> handle ) < 0 ? -1 : 0;
  acard -> handle = -1;
  free( acard -> queue_buffer );
  acard -> queue_buffer = NULL;
  gus_card_ptrs[ acard -> card ] = NULL;
  free( acard );
  acard = NULL;
  return res;
}

int gus_select( int card )
{
  if ( acard && acard -> card == card ) return 0;	/* ok. card is selected */
  if ( card < 0 || card >= GUS_CARDS || gus_card_ptrs[ card ] == NULL ) return -1;
  acard = gus_card_ptrs[ card ];
  return 0;
}

int gus_get_handle()
{
  return !acard ? -1 : acard -> handle;
}

/*
 *  ----------------- info
 */

int gus_info( struct GUS_STRU_INFO *info, int reread )
{
  int null;

  null = info == NULL;
  if ( !acard -> info_flag || reread )
    {
      if ( null ) info = &acard -> info_data;
      if ( ioctl( acard -> handle, GUS_IOCTL_INFO, info ) < 0 )
        {
          gus_dprintf( "gus_read_info: gus ioctl - INFO" );
          return -1;
        }
      if ( !null )
        memcpy( &acard -> info_data, info, sizeof( struct GUS_STRU_INFO ) );
      acard -> info_flag = 1;
    }
   else
    if ( !null )
      memcpy( info, &acard -> info_data, sizeof( struct GUS_STRU_INFO ) );
  return 0;
}

/*
 *  ----------------- setup
 */

int gus_ultraclick_get( gus_ultraclick_t *ultraclick )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_GET_ULTRACLICK, ultraclick ) < 0 )
    {
      gus_dprintf( "gus_ultraclick_get: gus ioctl - GET_ULTRACLICK\n" );
      return -1;
    }
  return 0;  
}

int gus_ultraclick_set( gus_ultraclick_t *ultraclick )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_SET_ULTRACLICK, ultraclick ) < 0 )
    {
      gus_dprintf( "gus_ultraclick_set: gus ioctl - SET_ULTRACLICK\n" );
      return -1;
    }
  return 0;  
}

int gus_emulation_get( void )
{
  int i;

  if ( ioctl( acard -> handle, GUS_IOCTL_GET_MIDI_EMUL, &i ) < 0 )
    {
      gus_dprintf( "gus_emulation_get: gus ioctl - GET_MIDI_EMUL\n" );
      return -1;
    }
  return i;  
}

int gus_emulation_set( int emulation )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_SET_MIDI_EMUL, &emulation ) < 0 )
    {
      gus_dprintf( "gus_emulation_set: gus ioctl - SET_MIDI_EMUL\n" );
      return -1;
    }
  return 0;  
}

int gus_effect_reset( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_EFFECT_RESET ) < 0 )
    {
      gus_dprintf( "gus_effect_reset: gus ioctl - EFFECT_RESET\n" );
      return -1;
    }
  return 0;
}

int gus_effect_setup( struct GUS_STRU_EFFECT *effect )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_EFFECT_SETUP, effect ) < 0 )
    {
      gus_dprintf( "gus_effect_setup: gus ioctl - EFFECT_SETUP\n" );
      return -1;
    }
  return 0;  
}

/*
 *  ----------------- memory
 */

int gus_memory_size( void )
{
  if ( gus_info( NULL, 0 ) ) return -1;
  return acard -> info_data.memory_size;
}

int gus_memory_free_size( void )
{
  if ( gus_info( NULL, 1 ) ) return -1;
  return acard -> info_data.memory_free;
}

int gus_memory_free_block( int w_16bit )
{
  if ( gus_info( NULL, 1 ) ) return -1;
  return w_16bit ? acard -> info_data.memory_block_16 : acard -> info_data.memory_block_8;
}

int gus_memory_rom_size( void )
{
  if ( gus_info( NULL, 0 ) ) return -1;
  return acard -> info_data.memory_rom_size;  
}

int gus_memory_rom_banks( void )
{
  if ( gus_info( NULL, 0 ) ) return -1;
  return acard -> info_data.memory_rom_banks;  
}

/*
 *  ----------------- reset
 */

int gus_reset( int voices, unsigned int channel_voices )
{
  struct GUS_STRU_RESET reset;

  gus_reset_variables();
  reset.voices = acard -> active_voices = voices;
  reset.channel_voices = acard -> channel_voices = channel_voices;
  if ( ioctl( acard -> handle, GUS_IOCTL_RESET, &reset ) < 0 )
    {
      gus_dprintf( "gus_reset: gus ioctl - RESET\n" );
      return -1;
    }
  return reset.voices;
}

int gus_reset_engine_only( void )
{
  struct GUS_STRU_RESET reset;

  acard -> queue_ptr = acard -> queue_buffer;
  acard -> queue_len = acard -> queue_buffer_size;
  reset.voices = 0;
  reset.channel_voices = 0;
  if ( ioctl( acard -> handle, GUS_IOCTL_RESET0, &reset ) < 0 )
    {
      gus_dprintf( "gus_reset_engine_only: gus ioctl - RESET0\n" );
      return -1;
    }
  return reset.voices;
}

/*
 *  ----------------- timer
 */

int gus_timer_start( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_START ) < 0 )
    {
      gus_dprintf( "gus_timer_start: gus ioctl - TIMER_START\n" );
      return -1;
    }
  return 0;
}

int gus_timer_stop( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_STOP ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_STOP\n" );
      return -1;
    }
  return 0;
}

int gus_timer_continue( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_CONTINUE ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_CONTINUE\n" );
      return -1;
    }
  return 0;
}

int gus_timer_base( int base )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_BASE, &base ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_BASE\n" );
      return -1;
    }
  return 0;
}

int gus_timer_tempo( int tempo )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_TEMPO, &tempo ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_TEMPO\n" );
      return -1;
    }
  return 0;
}

int gus_timer_master( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_MASTER ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_MASTER\n" );
      return -1;
    }
  return 0;
  
}

int gus_timer_slave( int slave_card )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_TIMER_SLAVE, &slave_card ) < 0 )
    {
      gus_dprintf( "gus_timer_stop: gus ioctl - TIMER_SLAVE\n" );
      return -1;
    }
  return 0;
}

/*
 *  ----------------- sample
 */

int gus_memory_reset( int mode )
{
  acard -> info_flag = 0;
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_RESET, &mode ) < 0 )
    {
      gus_dprintf( "gus_sample_download_reset: gus ioctl - DOWNLOAD_RESET\n" );
      return -1;
    }
  acard -> memory_mode = mode;
  return 0;
}

int gus_memory_test( gus_instrument_t *instrument )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_TEST, instrument ) < 0 )
    return -1;
  return 0;
}

int gus_memory_get_name( gus_instrument_name_t *instrument )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_GET_NAME, instrument ) < 0 )
    return -1;
  return 0;
}

int gus_memory_alloc( gus_instrument_t *instrument )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_ALLOC, instrument ) < 0 )
    {
#ifdef DEBUG
      if ( acard -> memory_mode != GUS_DOWNLOAD_MODE_TEST )
        gus_dprintf( "gus_sample_alloc: gus ioctl - MEMORY_ALLOC\n" );
#endif 
      return -1;
    }
  return 0;
}

int gus_memory_free( gus_instrument_t *instrument )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_FREE, instrument ) < 0 )
    {
#ifdef DEBUG
      if ( acard -> memory_mode != GUS_DOWNLOAD_MODE_TEST )
        gus_dprintf( "gus_sample_alloc: gus ioctl - MEMORY_FREE\n" );
#endif 
      return -1;
    }
  return 0;
}

int gus_memory_pack( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_PACK ) < 0 )
    {
#ifdef DEBUG
      if ( acard -> memory_mode != GUS_DOWNLOAD_MODE_TEST )
        gus_dprintf( "gus_sample_alloc: gus ioctl - MEMORY_PACK\n" );
#endif 
      return -1;
    }
  return 0;
}

int gus_memory_list( struct GUS_STRU_MEMORY_LIST *list )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_LIST, list ) < 0 )
    return -1;
  return 0;
}

int gus_memory_block_alloc( struct GUS_STRU_MEMORY_BLOCK *block )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_BALLOC, block ) < 0 )
    {
#ifdef DEBUG
      if ( acard -> memory_mode != GUS_DOWNLOAD_MODE_TEST )
        gus_dprintf( "gus_sample_alloc: gus ioctl - MEMORY_BALLOC\n" );
#endif 
      return -1;
    }
  return 0;
}

int gus_memory_block_free( struct GUS_STRU_MEMORY_BLOCK *block )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_BFREE, block ) < 0 )
    {
#ifdef DEBUG
      if ( acard -> memory_mode != GUS_DOWNLOAD_MODE_TEST )
        gus_dprintf( "gus_sample_alloc: gus ioctl - MEMORY_BFREE\n" );
#endif 
      return -1;
    }
  return 0;
}

int gus_memory_dump( struct GUS_STRU_MEMORY_DUMP *dump )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_MEMORY_DUMP, dump ) < 0 )
    {
      gus_dprintf( "gus_memory_dump: gus ioctl - MEMORY_DUMP\n" );
      return -1;
    }
  return 0;
}

/*
 *  ----------------- queue
 */

int gus_queue_write_set_size( int items )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_WQUEUE_SET_SIZE, &items ) < 0 )
    {
      gus_dprintf( "gus_queue_write_set_size: gus ioctl - WQUEUE_SET_SIZE (%s)\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_queue_write_get_size( void )
{
  int items;

  if ( ioctl( acard -> handle, GUS_IOCTL_WQUEUE_GET_SIZE, &items ) < 0 )
    {
      gus_dprintf( "gus_queue_write_get_size: gus ioctl - WQUEUE_GET_SIZE\n" );
      return -1;
    }
  return 0;
}

int gus_queue_write_free( void )
{
  int res;

  if ( ioctl( acard -> handle, GUS_IOCTL_WQUEUE_FREE, &res ) < 0 )
    {
      gus_dprintf( "gus_queue_free: gus ioctl - WQUEUE_FREE\n" );
      return -1;
    }
  return res;
}

int gus_queue_write_threshold( int threshold )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_WQUEUE_THRESHOLD, &threshold ) < 0 )
    {
      gus_dprintf( "gus_queue_threshold: gus ioctl - WQUEUE_THRESHOLD\n" );
      return -1;
    }
  return 0;
}

int gus_queue_read_set_size( int items )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_RQUEUE_SET_SIZE, &items ) < 0 )
    {
      gus_dprintf( "gus_queue_read_set_size: gus ioctl - RQUEUE_SET_SIZE (%s)\n", strerror( errno ) );
      return -1;
    }
  return 0;
}

int gus_queue_read_get_size( void )
{
  int items;

  if ( ioctl( acard -> handle, GUS_IOCTL_RQUEUE_GET_SIZE, &items ) < 0 )
    {
      gus_dprintf( "gus_queue_read_get_size: gus ioctl - RQUEUE_TIME\n" );
      return -1;
    }
  return 0;
}

int gus_queue_read_used( void )
{
  int res;

  if ( ioctl( acard -> handle, GUS_IOCTL_RQUEUE_USED, &res ) < 0 )
    {
      gus_dprintf( "gus_queue_free: gus ioctl - RQUEUE_USED\n" );
      return -1;
    }
  return res;
}

int gus_queue_flush( void )
{
  int res;

  do {
    res = gus_do_flush();
    if ( res < 0 ) return res;
  } while ( acard -> non_block && res > 0 );
  if ( ioctl( acard -> handle, GUS_IOCTL_FLUSH ) < 0 )
    {
      gus_dprintf( "gus_queue_flush: gus ioctl - FLUSH\n" );
      return -1;
    }
  return 0;
}

int gus_queue_abort( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_ABORT ) < 0 )
    {
      gus_dprintf( "gus_queue_abort: gus ioctl - ABORT\n" );
      return -1;
    }
  return 0;
}

int gus_queue_abort_to_stop( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_QABORT ) < 0 )
    {
      gus_dprintf( "gus_queue_abort_to_stop: gus ioctl - QABORT\n" );
      return -1;
    }
  return 0;
}

int gus_queue_stop_playback( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_STOP ) < 0 )
    {
      gus_dprintf( "gus_queue_stop_playback: gus ioctl - STOP\n" );
      return -1;
    }
  return 0;
}

int gus_queue_continue_playback( void )
{
  if ( ioctl( acard -> handle, GUS_IOCTL_CONTINUE ) < 0 )
    {
      gus_dprintf( "gus_queue_continue_playback: gus ioctl - CONTINUE\n" );
      return -1;
    }
  return 0;
}

/*
 *  ----------------- queue commands
 */

int gus_do_flush( void )
{
  int error;
  size_t res, size;

  error = 0;
  size = acard -> queue_ptr - acard -> queue_buffer;
  if ( size > 0 )
    {
      if ( ( res = write( acard -> handle, acard -> queue_buffer, size ) ) != size )
        {
          if ( acard -> non_block && res >= 0 )
            {
              if ( res > 0 )
                {
                  size -= res;
                  memcpy( acard -> queue_buffer, acard -> queue_buffer + res, size );
                  acard -> queue_ptr = acard -> queue_buffer + size;
                  acard -> queue_len = acard -> queue_buffer_size - size;
                }
              return res;
            }
          error = -errno;
          gus_dprintf( "gus_do_flush: write error (%i)", error );
        }
      acard -> queue_ptr = acard -> queue_buffer;
      acard -> queue_len = acard -> queue_buffer_size;
    }
  return error;
}

static void _gus_cmd1_( unsigned char cmd, unsigned char voice, unsigned char *args )
{
  if ( acard -> queue_len < 8 ) gus_do_flush();
  *acard -> queue_ptr = cmd;
  *(acard -> queue_ptr + 1) = voice & 0x1f;
  if ( args )
    memcpy( acard -> queue_ptr + 2, args, 6 );
  acard -> queue_ptr += 8;
  acard -> queue_len -= 8;
}

static void _gus_cmd2_( unsigned char cmd, unsigned char *args, unsigned char extended )
{
  if ( acard -> queue_len < 8 ) gus_do_flush();
  *acard -> queue_ptr = cmd;
  *(acard -> queue_ptr + 1) = extended;
  if ( args )
    memcpy( acard -> queue_ptr + 2, args, 6 );
  acard -> queue_ptr += 8;
  acard -> queue_len -= 8;
}

static inline unsigned int _gus_expand_( unsigned int pos )
{
  return pos;
}

static inline unsigned int _gus_freq_expand_( unsigned int freq )
{
  if ( freq > GUS_FREQ_DATA ) printf( "libgus: frequency overflow!!!! 0x%x\n", freq );
  return ( freq > GUS_FREQ_DATA ? GUS_FREQ_DATA : freq ) | acard -> freq_control;
}

static inline unsigned short _gus_volume_expand_( unsigned short volume )
{
  return ( volume > GUS_VOLUME_DATA ? GUS_VOLUME_DATA : volume ) | acard -> volume_control;
}

static inline unsigned short _gus_pan_expand_( unsigned short pan )
{
  return ( pan > GUS_PAN_DATA ? GUS_PAN_DATA : pan ) | acard -> pan_control;
}

void gus_do_program( unsigned char voice, unsigned int program )
{
  unsigned char args[ 6 ];

  gus_put_dword( args, 0, program );
  _gus_cmd1_( GUS_CMD_VOICE_PROGRAM, voice, args );
}

void gus_do_voice_start( unsigned char voice,
			 unsigned int program,
			 unsigned int freq,
			 unsigned short volume,
			 unsigned short pan )
{
  unsigned char args[ 6 ];
  
  if ( program != (unsigned short)~0 )
    {
      gus_put_dword( args, 0, program );
      _gus_cmd1_( GUS_CMD_VOICE_PROGRAM, voice, args );
    }
  gus_put_byte( args, 0, pan != (unsigned short)~0 ? GUS_CMD_VOICE_PAN : 
                         volume != (unsigned short)~0 ? GUS_CMD_VOICE_VOLUME :
                         GUS_CMD_VOICE_FREQ );
  _gus_cmd1_( GUS_CMD_VOICE_START, voice, args );
  gus_put_dword( args, 0, _gus_freq_expand_( freq ) );
  _gus_cmd1_( GUS_CMD_VOICE_FREQ, voice, args );
  if ( volume != (unsigned short)~0 )
    {
      gus_put_word( args, 0, _gus_volume_expand_( volume ) );
      _gus_cmd1_( GUS_CMD_VOICE_VOLUME, voice, args );
    }
  if ( pan != (unsigned short)~0 )
    {
      gus_put_word( args, 0, _gus_pan_expand_( pan ) );
      _gus_cmd1_( GUS_CMD_VOICE_PAN, voice, args );
    }
}

void gus_do_voice_start_position(
			 unsigned char voice,
			 unsigned int program,
			 unsigned int freq,
			 unsigned short volume,
			 unsigned short pan,
			 unsigned int pos )
{
  unsigned char args[ 6 ];
  
  if ( program != (unsigned short)~0 )
    {
      gus_put_dword( args, 0, program );
      _gus_cmd1_( GUS_CMD_VOICE_PROGRAM, voice, args );
    }
  gus_put_byte( args, 0, GUS_CMD_VOICE_POS );
  _gus_cmd1_( GUS_CMD_VOICE_START, voice, args );
  freq = _gus_freq_expand_( freq );
  if ( freq > GUS_FREQ_DATA ) freq = GUS_FREQ_DATA;
  gus_put_dword( args, 0, freq );
  _gus_cmd1_( GUS_CMD_VOICE_FREQ, voice, args );
  if ( volume != (unsigned short)~0 )
    {
      gus_put_word( args, 0, _gus_volume_expand_( volume ) );
      _gus_cmd1_( GUS_CMD_VOICE_VOLUME, voice, args );
    }
  if ( pan != (unsigned short)~0 )
    {
      gus_put_word( args, 0, _gus_pan_expand_( pan ) );
      _gus_cmd1_( GUS_CMD_VOICE_PAN, voice, args );
    }
  gus_put_dword( args, 0, _gus_expand_( pos ) );
  _gus_cmd1_( GUS_CMD_VOICE_POS, voice, args );
}

void gus_do_voice_stop( unsigned char voice, unsigned char mode )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, mode );
  _gus_cmd1_( GUS_CMD_VOICE_STOP, voice, args );
}

void gus_do_voice_control( unsigned char voice, unsigned char cntrl )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, cntrl );
  _gus_cmd1_( GUS_CMD_VOICE_CONTROL, voice, args );
}

void gus_do_voice_frequency( unsigned char voice, unsigned int freq )
{
  unsigned char args[ 6 ];

  gus_put_dword( args, 0, _gus_freq_expand_( freq ) );
  _gus_cmd1_( GUS_CMD_VOICE_FREQ, voice, args );
}

void gus_do_voice_volume( unsigned char voice, unsigned short vol )
{
  unsigned char args[ 6 ];

  gus_put_word( args, 0, _gus_volume_expand_( vol ) );
  _gus_cmd1_( GUS_CMD_VOICE_VOLUME, voice, args );
}

void gus_do_voice_loop_range( unsigned char voice, unsigned int start, unsigned int end )
{
  unsigned char args[ 6 ];

  start = _gus_expand_( start );
  end = _gus_expand_( end );
  gus_put_dword( args, 0, start );
  _gus_cmd1_( GUS_CMD_VOICE_SLOOP, voice, args );
  gus_put_dword( args, 0, end );
  _gus_cmd1_( GUS_CMD_VOICE_ELOOP, voice, args );
}

void gus_do_voice_ramp( unsigned char voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control )
{
  unsigned char args[ 6 ];

  gus_put_byte( args, 0, start );
  gus_put_byte( args, 1, end );
  gus_put_byte( args, 2, rate );
  gus_put_byte( args, 3, control );
  _gus_cmd1_( GUS_CMD_VOICE_RAMP, voice, args );
}

void gus_do_voice_position( unsigned char voice, unsigned int pos )
{
  unsigned char args[ 6 ];

  gus_put_dword( args, 0, _gus_expand_( pos ) );
  _gus_cmd1_( GUS_CMD_VOICE_POS, voice, args );
}

void gus_do_voice_pan( unsigned char voice, unsigned short pan )
{
  unsigned char args[ 6 ];

  gus_put_word( args, 0, _gus_pan_expand_( pan ) );
  _gus_cmd1_( GUS_CMD_VOICE_PAN, voice, args );
}

void gus_do_voice_lfo_setup( unsigned char voice, int type, int freq, int final_depth, int sweep, int shape )
{
  unsigned char args[ 6 ];
  unsigned short temp;
  
  args[ 0 ] = GUS_LFO_SETUP;
  if ( type ) args[ 0 ] |= 0x80;
  gus_put_byte( args, 1, sweep > 255 ? 255 : sweep );
  temp = freq & 0x7ff;
  if ( shape ) temp |= 0x2000;
  gus_put_word( args, 2, temp );
  gus_put_word( args, 4, final_depth );
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_lfo_freq( unsigned char voice, int type, int freq )
{
  unsigned char args[ 6 ];
  
  args[ 0 ] = GUS_LFO_FREQ;
  if ( type ) args[ 0 ] |= 0x80;
  gus_put_word( args, 2, freq & 0x7ff );
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_lfo_depth( unsigned char voice, int type, int depth )
{
  unsigned char args[ 6 ];
  
  args[ 0 ] = GUS_LFO_FREQ;
  if ( type ) args[ 0 ] |= 0x80;
  gus_put_word( args, 2, depth & 0x1fff );
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_lfo_enable( unsigned char voice, int type )
{
  unsigned char args[ 6 ];
  
  args[ 0 ] = GUS_LFO_ENABLE;
  if ( type ) args[ 0 ] |= 0x80;
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_lfo_disable( unsigned char voice, int type )
{
  unsigned char args[ 6 ];
  
  args[ 0 ] = GUS_LFO_DISABLE;
  if ( type ) args[ 0 ] |= 0x80;
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_lfo_shutdown( unsigned char voice, int type )
{
  unsigned char args[ 6 ];
  
  args[ 0 ] = GUS_LFO_SHUTDOWN;
  if ( type ) args[ 0 ] |= 0x80;
  _gus_cmd1_( GUS_CMD_VOICE_LFO, voice, args );
}

void gus_do_voice_interwave_effect( unsigned char voice, unsigned short eff_vol, int eff1, int eff2, int eff3, int eff4 )
{
  unsigned char args[ 6 ];

  if ( eff1 < 0 || eff1 > 7 ) eff1 = 15;
  if ( eff2 < 0 || eff2 > 7 ) eff2 = 15;
  if ( eff3 < 0 || eff3 > 7 ) eff3 = 15;
  if ( eff4 < 0 || eff4 > 7 ) eff4 = 15;
  gus_put_byte( args, 0, GUS_PRIV1_IW_EFFECT );
  gus_put_word( args, 2, _gus_volume_expand_( eff_vol ) );
  gus_put_byte( args, 4, eff1 | ( eff2 << 4 ) );
  gus_put_byte( args, 5, eff3 | ( eff4 << 4 ) );
  _gus_cmd1_( GUS_CMD_VOICE_PRIVATE1, voice, args );
}

void gus_do_voice_set_frequency_mode( unsigned int mode )
{
  mode &= GUS_FREQ_MASK;
  acard -> freq_control = mode;
}

void gus_do_voice_set_volume_mode( unsigned short mode )
{
  mode &= GUS_VOLUME_MASK;
  acard -> volume_control = mode;
}

void gus_do_voice_set_pan_mode( unsigned short mode )
{
  mode &= GUS_PAN_MASK;
  acard -> pan_control = mode;
}

void gus_do_chn_program( unsigned char channel, unsigned short program )
{
  unsigned char args[ 6 ];
  
  gus_put_word( args, 0, program );
  _gus_cmd1_( GUS_CMD_CHN_PROGRAM, channel, args );
}

void gus_do_chn_note_on( unsigned char channel, unsigned char note, unsigned char velocity, unsigned char priority )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, note );
  gus_put_byte( args, 1, velocity );
  gus_put_byte( args, 2, priority );
  _gus_cmd1_( GUS_CMD_CHN_NOTE_ON, channel, args );
}

void gus_do_chn_note_off( unsigned char channel, unsigned char note, unsigned char velocity )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, note );
  gus_put_byte( args, 1, velocity );
  _gus_cmd1_( GUS_CMD_CHN_NOTE_ON, channel, args );
}

void gus_do_chn_pitchbend( unsigned char channel, unsigned short pitchbend )
{
  unsigned char args[ 6 ];
  
  gus_put_word( args, 0, pitchbend );
  _gus_cmd1_( GUS_CMD_CHN_PITCHBEND, channel, args );
}

void gus_do_chn_control( unsigned char channel, unsigned char p1, unsigned char p2 )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, p1 );
  gus_put_byte( args, 1, p2 );
  _gus_cmd1_( GUS_CMD_CHN_CONTROL, channel, args );
}

void gus_do_chn_note_pressure( unsigned char channel, unsigned char p1, unsigned char p2 )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, p1 );
  gus_put_byte( args, 1, p2 );
  _gus_cmd1_( GUS_CMD_CHN_NOTE_PRESSURE, channel, args );
}

void gus_do_chn_pressure( unsigned char channel, unsigned char p1 )
{
  unsigned char args[ 6 ];
  
  gus_put_byte( args, 0, p1 );
  _gus_cmd1_( GUS_CMD_CHN_PRESSURE, channel, args );
}

void gus_do_tempo( unsigned int tempo )
{
  unsigned char args[ 6 ];

  gus_put_dword( args, 0, tempo );
  _gus_cmd2_( GUS_CMD_TEMPO, args, 0 );
}

void gus_do_wait( unsigned int us )
{
  unsigned char args[ 6 ];

  gus_put_dword( args, 0, us );
  _gus_cmd2_( GUS_CMD_WAIT, args, 0 );
}

void gus_do_stop( void )
{
  _gus_cmd2_( GUS_CMD_STOP, NULL, 0 );
}

void gus_do_echo( unsigned char *src, int size )
{
  char args[ 7 ];

  if ( size > 7 ) size = 7;
  memset( args, 0, 7 );
  memcpy( args, src, size );
  _gus_cmd2_( GUS_CMD_ECHO, args + 1, args[ 0 ] );
}

/*
 * Input multiplexer...
 */

int gus_do_input( gus_callbacks_t *callbacks )
{
  int errors = 0, size;
  unsigned char buf[ 64 ], *ptr;

  if ( callbacks && callbacks -> version < 2 ) return -1;
  while ( 1 )
    {
      if ( ( size = read( acard -> handle, buf, sizeof( buf ) ) ) == 0 ) break;
      ptr = buf;
      while ( size >= 8 )
        {
          switch ( ptr[ 0 ] ) {
            case GUS_CMD_ECHO:
              if ( callbacks && callbacks -> call_echo )
                callbacks -> call_echo( callbacks -> private, ptr + 1 );
              break;
            case GUS_CMD_VOICES_CHANGE:
              if ( callbacks && callbacks -> call_voices_change )
                callbacks -> call_voices_change( callbacks -> private, *(ptr + 1) );
              break;
            default:
              errors++;
          }
          size -= 8;
          ptr += 8;
        }
      if ( size != 0 ) errors++;
    }
  return errors ? -1 : 0;
}
