/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  PCM (Pulse Code Modulation) support
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */

#define FILE_PCM		"/dev/guspcm%i"
 
/*
 *  structures
 */

typedef struct {
  int card;
  int handle;
  int direction;
  struct GUS_PCM_INFO info;
} CARD;

/*
 *  variables
 */

static CARD *gus_card_ptrs[ GUS_CARDS ] =
		{ NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
static CARD *acard;

/*
 *  local functions
 */

/*
 *  EXPORTED FUNCTIONS
 */

int gus_pcm_cards( void )
{
  return gus_cards();
}

int gus_pcm_look_for_card( char *id )
{
  return gus_look_for_card( id );
}

/*
 *  ----------------- open / close
 */

int gus_pcm_open( int card, int direction, int flags )
{
  int handle;
  char filename[ 16 ];

  if ( card < 0 || card >= GUS_CARDS ) return -1;
  if ( gus_card_ptrs[ card ] ) return -1;
  if ( direction != GUS_PCM_DIRECTION_PLAY &&
       direction != GUS_PCM_DIRECTION_RECORD &&
       direction != GUS_PCM_DIRECTION_DUPLEX ) return -1;

  sprintf( filename, FILE_PCM, card );
  if ( ( handle = open( filename, direction | ( flags & GUS_PCM_OF_NONBLOCK ? O_NONBLOCK : 0 ) ) ) < 0 )
    {
      gus_dprintf( "gus_pcm_open: %s", strerror( errno ) );
      return -1;
    }
  
  acard = gus_card_ptrs[ card ] = (CARD *)malloc( sizeof( CARD ) );  
  if ( !acard )
    {
      gus_dprintf( "gus_pcm_open: malloc problem" );
      close( handle );
      return -1;
    }
  acard -> handle = handle;
  acard -> card = card;
  acard -> direction = direction;
  if ( ioctl( handle, SOUND_PCM_GUSINFO, &acard -> info ) < 0 )
    {
      gus_dprintf( "gus_pcm_open: info ioctl error" );
      free( gus_card_ptrs[ card ] );
      gus_card_ptrs[ card ] = NULL;
      acard = NULL;
      close( handle );
      return -1;
    }
 
  return card;
}

int gus_pcm_close( int card )
{
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  close( gus_card_ptrs[ card ] -> handle );
  free( gus_card_ptrs[ card ] );
  acard = gus_card_ptrs[ card ] = NULL;
  return 0;
}

int gus_pcm_get_handle( int card )
{
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  return gus_card_ptrs[ acard -> card ] -> handle;
}

int gus_pcm_select_card( int card )
{
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  if ( gus_card_ptrs[ card ] == NULL ) return -EINVAL;
  acard = gus_card_ptrs[ card ];
  return 0;
}

int gus_pcm_mode_set( int direction, unsigned int mode )
{
  if ( ioctl( acard -> handle, SOUND_PCM_SETFMT, &mode ) < 0 )
    return -1;
  return 0;
}

unsigned int gus_pcm_mode_get( int direction )
{
  unsigned int res;

  if ( ioctl( acard -> handle, SOUND_PCM_READ_BITS, &res ) < 0 )
    return -1;
  return res;
}

unsigned int gus_pcm_mode_supported( int direction )
{
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAY:
      return acard -> info.formats_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.formats_record;
    default:
      return -1;
  } 
}

int gus_pcm_channels_set( int direction, int channels )
{
  if ( ioctl( acard -> handle, SOUND_PCM_WRITE_CHANNELS, &channels ) < 0 )
    return -1;
  return 0;
}

int gus_pcm_channels_get( int direction )
{
  unsigned int res;

  if ( ioctl( acard -> handle, SOUND_PCM_READ_CHANNELS, &res ) < 0 )
    return -1;
  return res;
}

int gus_pcm_channels_supported( int direction )
{
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAY:
      return acard -> info.max_channels_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.max_channels_record;
    default:
      return -1;
  } 
}

int gus_pcm_rate_set( int direction, int rate )
{
  if ( ioctl( acard -> handle, SOUND_PCM_WRITE_RATE, &rate ) < 0 )
    return -1;
  return 0;
}

int gus_pcm_rate_get( int direction )
{
  int res;
  
  if ( ioctl( acard -> handle, SOUND_PCM_READ_RATE, &res ) < 0 )
    return -1;
  return res;
}

int gus_pcm_rate_supported( int direction )
{
  switch ( direction ) {
    case GUS_PCM_DIRECTION_PLAY:
      return acard -> info.max_rate_play;
    case GUS_PCM_DIRECTION_RECORD:
      return acard -> info.max_rate_record;
    default:
      return -1;
  }
}

int gus_pcm_dma_size_set( int direction, int fragments, int fragment_size )
{
  fragments <<= 16;
  fragments |= fragment_size > 0xffff ? 0xffff : fragment_size;
  if ( ioctl( acard -> handle, SOUND_PCM_SETFRAGMENT, &fragments ) < 0 )
    return -1;
  return fragments;
}

int gus_pcm_dma_size_get( int direction, int current )
{
  if ( !current )
    {
      switch ( direction ) {
        case GUS_PCM_DIRECTION_PLAY:
          return acard -> info.dma_size_play;
        case GUS_PCM_DIRECTION_RECORD:
          return acard -> info.dma_size_record;
        default:
          return -1;
      }
    }
   else
    {
      int res;
    
      if ( ioctl( acard -> handle, SNDCTL_DSP_GETBLKSIZE, &res ) < 0 )
        return -1;
      return res;
    }
}

int gus_pcm_sync( void )
{
  if ( ioctl( acard -> handle, SOUND_PCM_SYNC ) < 0 )
    return -errno;
  return 0;
}

int gus_pcm_read( unsigned char *buffer, int count )
{
  int res;

  res = read( acard -> handle, buffer, count );
  return res < 0 ? -1 : res;
}

int gus_pcm_write( unsigned char *buffer, int count )
{
  int res;

  res = write( acard -> handle, buffer, count );
  return res < 0 ? -1 : res;
}
