/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Mixer support routines
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"

#define FILE_MIXER	"/dev/gusmixer%i"

/*
 *  structures
 */

typedef struct {
  int card;
  int handle;
  int devmask;
  int recmask;
  int stereodevs;
  int version;
} CARD;

/*
 *  variables
 */

static CARD *gus_card_ptrs[ GUS_CARDS ] =
		{ NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
static CARD *acard = NULL;

/*
 *  EXPORTED FUNCTIONS
 */ 

int gus_mixer_cards( void )
{
  return gus_cards();
}

int gus_mixer_look_for_card( char *id )
{
  return gus_look_for_card( id );
}

int gus_mixer_open( int card )
{
  int handle, i, mversion; 
  int devmask, recmask, stereodevs;
  char filename[ 16 ];

  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  if ( gus_card_ptrs[ card ] ) return -EBUSY;
  sprintf( filename, FILE_MIXER, card );
  if ( ( handle = open( filename, O_RDWR ) ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: file open error %i", errno );
      return -1;
    }
  mversion = GUS_MIXER_NONE;
  if ( ioctl( handle, SOUND_MIXER_GUS_VERSION, &i ) < 0 )
    {
      gus_dprintf( "gus_mixer_open: read gus version error" );
      close( handle );
      return -1;
    }
  if ( i & 0x10000 )		/* daughter board */
    mversion = GUS_MIXER_CODEC;
   else
    {
      i &= 0xffff;
      if ( i >= 0x35 && i <= 0x37 ) mversion = GUS_MIXER_ICS; else
      if ( i == 0xa0 ) mversion = GUS_MIXER_CODEC;
      if ( i == 0x100 ) mversion = GUS_MIXER_ENH_CODEC;
    }
  i = 0;
  if ( ioctl( handle, SOUND_MIXER_READ_DEVMASK, &devmask ) < 0 ) i = -1;
  if ( ioctl( handle, SOUND_MIXER_READ_RECMASK, &recmask ) < 0 ) i = -1;
  if ( ioctl( handle, SOUND_MIXER_READ_STEREODEVS, &stereodevs ) < 0 ) i = -1;
  if ( i < 0 )
    {
      gus_dprintf( "gus_mixer_open: device mask or record mask or stereo devices ioctl error" );
      close( handle );
      return -1;
    }
    
  acard = gus_card_ptrs[ card ] = (CARD *)malloc( sizeof( CARD ) );
  if ( !acard )
    {
      gus_dprintf( "gus_mixer_open: malloc problem" );
      close( handle );
      return -1;
    }
 
  acard -> card = card;
  acard -> handle = handle;
  acard -> devmask = devmask;
  acard -> recmask = recmask;
  acard -> stereodevs = stereodevs;
  acard -> version = mversion;
    
  return card;
}

int gus_mixer_close( int card )
{
  int res;
  
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  res = close( gus_card_ptrs[ card ] -> handle ) < 0 ? -1 : 0;
  free( gus_card_ptrs[ card ] );
  acard = gus_card_ptrs[ card ] = NULL;
  return 0;
}

int gus_mixer_select( int card )
{
  if ( card < 0 || card >= GUS_CARDS ) return -ENODEV;
  if ( gus_card_ptrs[ card ] == NULL ) return -EINVAL;
  acard = gus_card_ptrs[ card ];
  return 0;
}

int gus_mixer_version( void )
{
  return acard -> version;
}

int gus_mixer_read_devmask( void )
{
  return acard -> devmask;
}

int gus_mixer_read_recmask( void )
{
  return acard -> recmask;
}

int gus_mixer_read_stereodevs( void )
{
  return acard -> stereodevs; 
}

#ifdef DEBUG

#define gus_mixer_read( what, mesg ) \
	_gus_mixer_read_( SOUND_MASK_##what, SOUND_MIXER_READ_##what, mesg )

static int _gus_mixer_read_( int mask, int what, char *mesg )
{
  int res = -1;

  if ( acard -> devmask & mask )
    if ( ioctl( acard -> handle, what, &res ) < 0 )
      gus_dprintf( "gus_mixer_read: %s", mesg );
  return res;
}

#else

#define gus_mixer_read( what, mesg ) \
	_gus_mixer_read_( SOUND_MASK_##what, SOUND_MIXER_READ_##what )

static inline int _gus_mixer_read_( int mask, int what )
{
  int res = -1;

  if ( acard -> devmask & mask )
    return ioctl( acard -> handle, what, &res ) < 0 ? -1 : res;
   else
    return -1;
}

#endif

int gus_mixer_read_recsrc( void )
{
  int res = -1;

  if ( ioctl( acard -> handle, SOUND_MIXER_READ_RECSRC, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_recsrc: error" );
  return res;
}

int gus_mixer_read_devs_lmute( void )
{
  int res = -1;

  if ( ioctl( acard -> handle, SOUND_MIXER_READ_DEVS_LMUTE, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_devs_lmute: error" );
  return res;
}

int gus_mixer_read_devs_rmute( void )
{
  int res = -1;

  if ( ioctl( acard -> handle, SOUND_MIXER_READ_DEVS_RMUTE, &res ) < 0 )
    gus_dprintf( "gus_mixer_read_devs_rmute: error" );
  return res;
}

int gus_mixer_read_mic( void )
{
  return gus_mixer_read( MIC, "MIC" );
}

int gus_mixer_read_cd( void )
{
  return gus_mixer_read( CD, "CD" );
}

int gus_mixer_read_line( void )
{
  return gus_mixer_read( LINE, "LINE" );
}

int gus_mixer_read_synth( void )
{
  return gus_mixer_read( SYNTH, "SYNTH" );
}

int gus_mixer_read_pcm( void )
{
  return gus_mixer_read( PCM, "PCM" );
}

int gus_mixer_read_reclev( void )
{
  return gus_mixer_read( RECLEV, "RECLEV" );
}

int gus_mixer_read_volume( void )
{
  return gus_mixer_read( VOLUME, "VOLUME" );
}

int gus_mixer_read_imix( void )
{
  return gus_mixer_read( IMIX, "IMIX" );
}

int gus_mixer_read_loopback( void )
{
  return gus_mixer_read( LINE1, "LOOPBACK" );
}

int gus_mixer_read_effect( void )
{
  return gus_mixer_read( LINE2, "EFFECT" );
}

#ifdef DEBUG

#define gus_mixer_write( what, data, mesg ) \
	_gus_mixer_write_( SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data, mesg )

static int _gus_mixer_write_( int mask, int what, int data, char *mesg )
{
  int res = -1;

  if ( acard -> devmask & mask )
    if ( ( res = ioctl( acard -> handle, what, &data ) ) < 0 )
      gus_dprintf( "gus_mixer_write: 0x%x, %s", data, mesg );
  return res;
}

#else

#define gus_mixer_write( what, data, mesg ) \
	_gus_mixer_write_( SOUND_MASK_##what, SOUND_MIXER_WRITE_##what, data )

static inline int _gus_mixer_write_( int mask, int what, int data )
{
  if ( acard -> devmask & mask )
    return ioctl( acard -> handle, what, &data ) < 0 ? -1 : 0;
   else
    return -1;
}

#endif

int gus_mixer_write_recsrc( int value )
{
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_RECSRC, &value ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_recsrc: error" );
      return -1;
    }
  return 0;
}

int gus_mixer_write_devs_lmute( int dev, int mute )
{
  if ( mute ) dev |= SOUND_MIXER_MUTE_FLAG;
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_DEVS_LMUTE, &dev ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_devs_lmute: error" );
      return -1;
    }
  return 0;  
}

int gus_mixer_write_devs_rmute( int dev, int mute )
{
  if ( mute ) dev |= SOUND_MIXER_MUTE_FLAG;
  if ( ioctl( acard -> handle, SOUND_MIXER_WRITE_DEVS_RMUTE, &dev ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_devs_rmute: error" );
      return -1;
    }
  return 0;  
}

int gus_mixer_write_mic( int value )
{
  return gus_mixer_write( MIC, value, "MIC" );
}

int gus_mixer_write_cd( int value )
{
  return gus_mixer_write( CD, value, "CD" );
}

int gus_mixer_write_line( int value )
{
  return gus_mixer_write( LINE, value, "LINE" );
}

int gus_mixer_write_synth( int value )
{
  return gus_mixer_write( SYNTH, value, "SYNTH" );
}

int gus_mixer_write_pcm( int value )
{
  return gus_mixer_write( PCM, value, "PCM" );
}

int gus_mixer_write_reclev( int value )
{
  return gus_mixer_write( RECLEV, value, "RECLEV" );
}

int gus_mixer_write_volume( int value )
{
  return gus_mixer_write( VOLUME, value, "VOLUME" );
}

int gus_mixer_write_imix( int value )
{
  return gus_mixer_write( IMIX, value, "IMIX" );
}

int gus_mixer_write_loopback( int value )
{
  return gus_mixer_write( LINE1, value, "LOOPBACK" );
}

int gus_mixer_write_effect( int value )
{
  return gus_mixer_write( LINE2, value, "EFFECT" );
}

/*
 * special things
 */

int gus_mixer_read_interwave_serial( void )
{
  struct GUS_MIXER_SPECIAL special;
  
  special.what = GUS_MIXER_S_IW;
  if ( ioctl( acard -> handle, SOUND_MIXER_SPECIAL_READ, &special ) < 0 )
    {
      gus_dprintf( "gus_mixer_read_interwave_serial: error\n" );
      return -1;
    }
  return special.data.interwave.serial;
}

int gus_mixer_write_interwave_serial( int serial )
{
  struct GUS_MIXER_SPECIAL special;
  
  special.what = GUS_MIXER_S_IW;
  special.data.interwave.serial = serial;
  if ( ioctl( acard -> handle, SOUND_MIXER_SPECIAL_WRITE, &special ) < 0 )
    {
      gus_dprintf( "gus_mixer_write_interwave_serial: error\n" );
      return -1;
    }
  return special.data.interwave.serial;
}
