/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for playing MIDI file
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include "libgus.h"
#include "libgus_local.h"

#if 0
#define MIDIPLAY_DEBUG
#endif

/* Non-standard MIDI file formats */
#define RIFF                    0x52494646
#define CTMF                    0x43544d46
/* Standard MIDI file format definitions */
#define MThd                    0x4d546864
#define MTrk                    0x4d54726b

struct midiplay_channel {
  int emulation;		/* current emulation/mode for this channel */
  int bank;			/* current instrument bank # */
  int program;			/* loaded program */
  gus_midi_device_t *mdevice;	/* output device for this channel -> more info */
  int device;			/* output device for this channel */
};

struct midiplay_track {
  unsigned char *data;		/* complete track */
  unsigned int length;		/* complete length of track */
  unsigned char *ptr;		/* current pointer */
  unsigned int size;		/* current size to end of track */
  unsigned int tick;		/* current tick */
  unsigned char old_cmd;	/* old command */
  int clock;			/* next sequence is clock */
};

struct midiplay_info {
  unsigned short format;
  char *format_name;
  unsigned short ntrks;
  unsigned short division;
  unsigned int tracklen;
  unsigned int default_tempo;
  unsigned int tempo;
  unsigned int bpm;
  unsigned int ticks_per_quarter;
  struct midiplay_channel *channels;
  struct midiplay_track *tracks;
};

static int global_stop;
static int global_end_flag;


/*
 *  code for mread() was taken from playmidi-2.2
 */

static int mread( struct midiplay_info *info, 
		  unsigned char *song_data, unsigned int song_size, int song_number )
{
  unsigned char *ptr;
  unsigned int i, tracklen;

#ifdef MIDIPLAY_DEBUG
  printf( "mread\n" );
#endif  
  memset( info, 0, sizeof( *info ) );
  i = 0;
  ptr = song_data;
  while ( ptr < song_data + song_size - 32 )
    if ( !strncmp( ptr, "MThd", 4 ) )
      {
        if ( i == song_number ) break;
        i++;
        ptr += 4;
      }
     else
      ptr++;
  if ( i != song_number ) return -1;		/* song not found */
#ifdef MIDIPLAY_DEBUG
  printf( "song found ok..\n" );
#endif
  i = gus_get_Ldword( ptr, 0 );
  ptr += 4;
  if ( i == RIFF )
    {
      i = gus_get_Ldword( ptr, 16 );
      ptr += 20;
    }
  if ( i == MThd ) 
    {
      info -> tracklen = gus_get_Ldword( ptr, 0 );
      info -> format = gus_get_Lword( ptr, 4 );
      info -> ntrks = gus_get_Lword( ptr, 6 );
      info -> division = gus_get_Lword( ptr, 8 );
      if ( info -> format == 0 )
        info -> format_name = strdup( "single track MIDI" );
       else
        info -> format_name = strdup( "multi track MIDI" );
      ptr += 10;
#ifdef MIDIPLAY_DEBUG
      printf( "mread - len = %i, format = %i, tracks = %i, division = %i\n", info -> tracklen, info -> format, info -> ntrks, info -> division );
#endif
    }
   else
  if ( i == CTMF )
    {
      info -> tracklen = gus_get_Lbyte( ptr, 4 ) | ( gus_get_Lbyte( ptr, 5 ) << 8 );
      info -> default_tempo = gus_get_Lbyte( ptr, 8 ) | ( gus_get_Lbyte( ptr, 9 ) << 8 ) * 3000;
      info -> format = 0;
      info -> ntrks = 1;
      info -> division = 40;
      info -> tracks = malloc( sizeof( struct midiplay_track ) );
      info -> tracks[ 0 ].data = song_data + info -> tracklen;
      info -> tracks[ 0 ].length = song_size - info -> tracklen;
      info -> format_name = strdup( "single track CTMF" );
      return 0;
    }
   else
    return -1;
  info -> tracks = malloc( sizeof( struct midiplay_track ) * info -> ntrks );
  if ( !info -> tracks )
    {
      free( info -> format_name );
      info -> format_name = NULL;
      return -1;
    }
  for ( i = 0; i < info -> ntrks; i++ )
    {
      if ( gus_get_Ldword( ptr, 0 ) != MTrk )
        {
          free( info -> tracks ); info -> tracks = NULL;
          free( info -> format_name ); info -> format_name = NULL;
          return -1;
        }
      tracklen = gus_get_Ldword( ptr, 4 );
      ptr += 8;
      if ( ptr + tracklen > song_data + song_size )
        tracklen = song_data + song_size - ptr;
      info -> tracks[ i ].data = ptr;
      info -> tracks[ i ].length = tracklen;
      ptr += tracklen;
    }
  info -> default_tempo = 60000000L / 120L;
  info -> bpm = 120;
  info -> ticks_per_quarter = info -> division;
  return 0;
}

int gus_midiplay_songs( unsigned char *song_data, unsigned int song_size )
{
  unsigned char *ptr;
  unsigned int result;

  if ( song_size < 32 ) return -1;
  if ( gus_get_Ldword( song_data, 0 ) == RIFF )
    {
      song_data += 16;
      song_size -= 16;
    }
  if ( gus_get_Ldword( song_data, 0 ) == CTMF )
    return 1;
  result = 0;
  ptr = song_data;
  while ( ptr < song_data + song_size - 32 )
    if ( !strncmp( ptr, "MThd", 4 ) )
      {
        result++;
        ptr += 4;
      }
     else
      ptr++;
  return result;
}

static void msysex( int device, unsigned char *buffer, int size, int tick_time )
{
  static unsigned char gm_on[ 4 ] = { 0x7e, 0x7f, 0x09, 0x01 };
  static unsigned char gm_off[ 4 ] = { 0x7e, 0x7f, 0x09, 0x02 };
  static unsigned char gs_on[ 9 ] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x00, 0x41 };
  static unsigned char gs_off[ 9 ] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x7f, 0x41 };
  int wait_flag = 0;
                  
#if 0
  printf( "sysex - %i\n", size );
#endif
  if ( size == 4 )
    {
      if ( !memcmp( gm_on, buffer, 4 ) ) wait_flag++;
      if ( !memcmp( gm_off, buffer, 4 ) ) wait_flag++;
    }
  if ( size == 9 )
    {
      if ( !memcmp( gs_on, buffer, 9 ) ) wait_flag++;
      if ( !memcmp( gs_off, buffer, 9 ) ) wait_flag++;
    }
  gus_midi_sysex( device, buffer, size );
  if ( wait_flag )
    gus_midi_wait( 50000 / tick_time );
}

static void mecho( void *private, unsigned char *data, int len )
{
  unsigned int arg[ 2 ];
  gus_midiplay_callbacks_t *callbacks = (gus_midiplay_callbacks_t *)private;

  if ( len == sizeof( arg ) )
    {
      memcpy( &arg, data, sizeof( arg ) );
      switch ( arg[ 0 ] ) {
        case 0:
          if ( callbacks -> call_time )
            callbacks -> call_time( callbacks -> private, arg[ 1 ] );
          break;
        case 1:
          global_end_flag++;
          break;
      }
    }
}

static void mwait( struct midiplay_info *info, gus_midiplay_callbacks_t *callbacks, int read_only )
{
  static gus_midi_callbacks_t input = {
    17,
    NULL,		/* private */
    NULL,		/* note on */
    NULL,		/* note off */
    NULL,		/* note pressure */
    NULL,		/* channel pressure */
    NULL,		/* bender */
    NULL,		/* program change */
    NULL,		/* control */
    NULL,		/* sysex */
    mecho,		/* echo */
    NULL,		/* wait */
    NULL,		/* mtc quarter */
    NULL,		/* song select */
    NULL,		/* song position */
    NULL,		/* tune request */
    NULL,		/* start */
    NULL,		/* continue */
    NULL,		/* stop */
  };
  fd_set read_fds, write_fds;
  int midi_handle, old_global_stop;

  midi_handle = gus_midi_get_handle();
  if ( global_stop != 2 )
    gus_midi_write();
  do { 
    int res;
     
    memcpy( &read_fds, &callbacks -> read_fds, sizeof( fd_set ) );
    memcpy( &write_fds, &callbacks -> write_fds, sizeof( fd_set ) );
    FD_SET( midi_handle, &read_fds );
    if ( !global_stop && !read_only )
      FD_SET( midi_handle, &write_fds );
    res = select( callbacks -> last_fd + 1, &read_fds, &write_fds, NULL, NULL );
    if ( callbacks -> call_select )
      {
        old_global_stop = global_stop;
        callbacks -> call_select( callbacks -> private, &read_fds, &write_fds );
        if ( global_stop != old_global_stop ) break;
      }
    if ( FD_ISSET( midi_handle, &read_fds ) )
      {
        input.private = callbacks;
        gus_midi_input( &input );
      }
  } while ( ( !read_only && !global_stop && !FD_ISSET( midi_handle, &write_fds ) ) ||
            ( read_only && !FD_ISSET( midi_handle, &read_fds ) ) );
}

static unsigned int mgetvar( struct midiplay_track *ptrack )
{
  unsigned int result;
  unsigned char c;

  result = ~0;
  if ( ptrack -> size )
    {
      result = *ptrack -> ptr++; ptrack -> size--;
      if ( result & 0x80 )
        {
          result &= 0x7f;
          do {
            if ( !ptrack -> size ) return ~0;
            result <<= 7;
            result |= ( c = *ptrack -> ptr++ ) & 0x7f;
            ptrack -> size--;
          } while ( c & 0x80 );
        }
    }
  return result;
}

static unsigned int compute_tick_time( struct midiplay_info *info )
{
  unsigned int result;

  result = info -> bpm * ( ( info -> ticks_per_quarter * 5 ) / 3 );
  result = ( 100000000L + ( result >> 1 ) ) / result;
  return result;
}

static int mplay( int device, struct midiplay_info *info, gus_midiplay_callbacks_t *callbacks, int play )
{
  int track, ctrack, size, chn, emul;
  unsigned int low_time, clock, current_tick, delta, tick_time, sec_time, sec_time_modulo;
  struct midiplay_track *ptrack;
  struct midiplay_channel *pchn;
  unsigned char cmd;
  char text[ 512 ];
  gus_midi_device_t *mdevice;
  unsigned char sysex[ 270 ];
  int sysex_ptr, raw_ptr_end;

#if 0
  if ( play ) return 0;
#endif
#ifdef MIDIPLAY_DEBUG
  printf( "mplay - %i\n", play );
#endif
  if ( !play )
    if ( callbacks -> call_info )
      callbacks -> call_info( callbacks -> private, info -> format, info -> format_name, info -> ntrks );
  emul = GUS_MIDI_EMUL_GM;
  if ( device != GUS_MIDID_COMMON )
    {
      emul = gus_midi_emulation_get( device );
      if ( emul < 0 ) emul = GUS_MIDI_EMUL_GM;
    }
  for ( chn = 0; chn < 16; chn++ )
    {
      info -> channels[ chn ].bank = 0;
      info -> channels[ chn ].program = -1;
      info -> channels[ chn ].emulation = emul;
      gus_midi_program_change( device, chn, 0 );
      if ( device != GUS_MIDID_COMMON )
        {
          info -> channels[ chn ].mdevice = gus_midi_output_device( device );
          info -> channels[ chn ].device = device;
        }
    }
  if ( device == GUS_MIDID_COMMON )
    for ( mdevice = gus_midi_output_devices(); mdevice; mdevice = mdevice -> next )
      {
        emul = gus_midi_emulation_get( mdevice -> device );
        if ( emul >= 0 )
          for ( chn = 0; chn < 16; chn++ )
            if ( mdevice -> channels & ( 1 << chn ) )
              {
                info -> channels[ chn ].emulation = emul;
                info -> channels[ chn ].mdevice = mdevice;
                info -> channels[ chn ].device = mdevice -> device;
              }
      }
  for ( track = 0; track < info -> ntrks; track++ )
    {
      ptrack = &info -> tracks[ track ];
      ptrack -> ptr = ptrack -> data;
      ptrack -> size = ptrack -> length;
      ptrack -> tick = 0;
      ptrack -> old_cmd = 0;
      ptrack -> clock = 1;
    }
  info -> tempo = info -> default_tempo;
  if ( play )
    {
      unsigned int arg[ 2 ];
    
      gus_midi_timer_base( ( info -> ticks_per_quarter * 5 ) / 3 );
      info -> bpm = ( 60000000L + ( info -> tempo >> 1 ) ) / info -> tempo;
      gus_midi_timer_tempo( info -> bpm );
      gus_midi_timer_start();
      arg[ 0 ] = 0;		/* time stamp */
      arg[ 1 ] = 0;		/* zero time */
      gus_midi_echo( (char *)&arg, sizeof( arg ) );
    }
  current_tick = 0;
  tick_time = compute_tick_time( info );
  sec_time = 0; sec_time_modulo = 0;
  while ( !global_stop || global_stop == 2 )
    {
      if ( play )
        {
          if ( global_stop == 2 )		/* pause */
            {
              gus_midi_timer_stop();
              while ( global_stop == 2 )
                mwait( info, callbacks, 0 );
              if ( global_stop ) break;
              gus_midi_timer_continue();
            }
           else
            {
              mwait( info, callbacks, 0 );
              if ( global_stop == 1 ) break;
            }
        }
      low_time = ~0; ctrack = -1;
      for ( track = 0; track < info -> ntrks; track++ )
        {
          ptrack = &info -> tracks[ track ];
          if ( ptrack -> tick < low_time )
            {
              ctrack = track;
              low_time = ptrack -> tick;
            }
        }
      if ( ctrack < 0 )
        if ( play )
          {
            unsigned int arg[ 2 ];
        
            if ( global_stop != 1 )
              {
                global_end_flag = 0;
                arg[ 0 ] = 1;			/* stop */
                arg[ 1 ] = 0;
                gus_midi_echo( (char *)&arg, sizeof( arg ) );
                while ( global_stop != 1 && !global_end_flag )
                  {
                    if ( global_stop == 2 )		/* pause */
                      {
                        gus_midi_timer_stop();
                        while ( global_stop == 2 && !global_end_flag )
                          mwait( info, callbacks, 1 );
                        if ( global_stop == 1 ) break;
                        gus_midi_timer_continue();
                      }
                     else
                      mwait( info, callbacks, 1 );
                  }
              }
            break;
          }
         else
          break;
      ptrack = &info -> tracks[ ctrack ];
      if ( !ptrack -> size )
        {
          ptrack -> tick = ~0;
          continue;
        }
      if ( ptrack -> tick > current_tick )
        {
          delta = ptrack -> tick - current_tick;
          /* delay longer than 40sec - cutoff to small delay */
          if ( 40000000 < delta * tick_time )
            {
              delta = info -> bpm * 5;
              for ( track = 0; track < info -> ntrks; track++ )
                {
                  info -> tracks[ track ].size = 0;
                  info -> tracks[ track ].tick = current_tick + delta;
                }
            }
          if ( play )
            {
              unsigned int tmp, tmp1;
              unsigned int arg[ 2 ];
        
              /* ok.. each 0.1sec put time marker to echo queue */
              tmp = sec_time_modulo + ( delta * tick_time );
#if 0
              printf( "(0) modulo = %i, tick_time = %i, tmp = %i\n", sec_time_modulo, tick_time, tmp );
#endif
              while ( 100000 <= tmp )
                {
                  tmp1 = ( 100000 - sec_time_modulo ) + ( tick_time - 1 );
                  tmp1 /= tick_time;
#if 0
                  printf( "tmp1 = %i\n", tmp1 );
#endif
                  gus_midi_wait( tmp1 );
                  delta -= tmp1;
                  sec_time_modulo += tmp1 * tick_time;
                  sec_time_modulo %= 100000;
                  tmp = sec_time_modulo + ( delta * tick_time );
                  arg[ 0 ] = 0;
                  arg[ 1 ] = ++sec_time;
#if 0
                  printf( " --- echo = %i\n", sec_time );
#endif
                  gus_midi_echo( (char *)&arg, sizeof( arg ) );
                }
              sec_time_modulo = tmp;
#if 0
              printf( "(1) modulo = %i, tick_time = %i, tmp = %i\n", sec_time_modulo, tick_time, tmp );
#endif
              if ( delta )
                {
#if 0
                  printf( "delta = %i\n", delta );
#endif
                  gus_midi_wait( delta );
                }
            }
           else
            {
              unsigned int tmp;
              
              tmp = sec_time_modulo + ( delta * tick_time );
              sec_time += tmp / 100000;
              sec_time_modulo = tmp % 100000;
            }
          current_tick = ptrack -> tick;
        }      
      sysex_ptr = -1;      
      raw_ptr_end = -1;
      while ( ptrack -> size ) {
#if 0
        printf( "raw_ptr_end = %i, ptrack -> size = %i\n", raw_ptr_end, ptrack -> size );
#endif
        if ( raw_ptr_end >= 0 )
          if ( raw_ptr_end >= ptrack -> size ) raw_ptr_end = -1;
        if ( raw_ptr_end < 0 && ptrack -> clock )
          {
            clock = mgetvar( ptrack );
#if 0
            printf( "clock = %i\n", clock );
#endif
#if 0
            printf( "0: tick = %i, ctrack = %i, ptrack -> size = %i, data = 0x%x, 0x%x, 0x%x\n", current_tick, ctrack, ptrack -> size, *(ptrack -> ptr + 0), *(ptrack -> ptr + 1), *(ptrack -> ptr + 2) );
#endif
            if ( clock == ~0 )
              {
                ptrack -> size = 0;
                continue;
              }
            if ( clock > 0 )
              {
                ptrack -> tick += clock;
                ptrack -> clock = 0;
                break;
              }
          }
        ptrack -> clock = 1;
#if 0
        printf( "1: tick = %i, ctrack = %i, ptrack -> size = %i, data = 0x%x, 0x%x, 0x%x\n", current_tick, ctrack, ptrack -> size, *(ptrack -> ptr + 0), *(ptrack -> ptr + 1), *(ptrack -> ptr + 2) );
#endif
        if ( *ptrack -> ptr == 0xff )		/* meta event */
          {
            ptrack -> ptr++; ptrack -> size--;
            if ( ptrack -> size < 2 )
              {
                ptrack -> size = 0;
                break;
              }
            cmd = *ptrack -> ptr++; ptrack -> size--;
            size = mgetvar( ptrack );
#if 0
            printf( "meta event - cmd = 0x%x, size = %i\n", cmd, size );
#endif
            if ( ptrack -> size < size )
              {
                ptrack -> size = 0;
                break;
              }
            switch ( cmd ) {
              case 0x01:		/* text */
              case 0x02:
              case 0x03:
              case 0x04:
              case 0x05:
              case 0x06:
              case 0x07:
                if ( size > ptrack -> size ) size = ptrack -> size;
                if ( size >= sizeof( text ) )
                  {
                    strncpy( text, ptrack -> ptr, sizeof( text ) );
                    text[ sizeof( text ) - 1 ] = 0;
                  }
                 else
                  {
                    strncpy( text, ptrack -> ptr, size );
                    text[ size ] = 0;
                  }
                {
                  char *ptr = text;
                  
                  while ( *ptr )
                    {
                      if ( *ptr < ' ' ) *ptr = ' ';
                      ptr++;
                    }
                }
#if 0
                printf( "cmd = 0x%x, text = '%s'\n", cmd, text );
#endif
                ptrack -> ptr += size;
                ptrack -> size -= size;
                switch ( cmd ) {
                  case 0x01:		/* text */
                    if ( !play && callbacks -> call_text )
                      callbacks -> call_text( callbacks -> private, ctrack, text );
                    break;
                  case 0x02:		/* copyright */
                    if ( !play && callbacks -> call_copyright )
                      callbacks -> call_copyright( callbacks -> private, ctrack, text );
                    break;
                  case 0x03:		/* sequence */
                    if ( !play && callbacks -> call_sequence )
                      callbacks -> call_sequence( callbacks -> private, ctrack, text );
                    break;
                  case 0x04:		/* instrument */
                    if ( !play && callbacks -> call_instrument )
                      callbacks -> call_instrument( callbacks -> private, ctrack, text );
                    break;
                  case 0x05:		/* lyric */
                    if ( play && callbacks -> call_lyric )
                      callbacks -> call_lyric( callbacks -> private, ctrack, text );
                    break;
                  case 0x06:		/* marker */
                    if ( play && callbacks -> call_marker )
                      callbacks -> call_marker( callbacks -> private, ctrack, text );
                    break;
                  case 0x07:		/* cue-point */
                    if ( play && callbacks -> call_cuepoint )
                      callbacks -> call_cuepoint( callbacks -> private, ctrack, text );
                    break;
                }
                break;
              case 0x2f:		/* end of track */
                ptrack -> size = 0;
                break;
              case 0x51:		/* tempo */
                if ( ptrack -> size < 3 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                info -> tempo = *ptrack -> ptr++ << 16;
                info -> tempo |= *ptrack -> ptr++ << 8;
                info -> tempo |= *ptrack -> ptr++;
                ptrack -> size -= 3;
                info -> bpm = ( 60000000L + ( info -> tempo >> 1 ) ) / info -> tempo;
                tick_time = compute_tick_time( info );
                if ( play )
                  gus_midi_timer_tempo( info -> bpm );
                break;
              default:
                ptrack -> ptr += size;
                ptrack -> size -= size;
            }
          }
         else
          {
            if ( *ptrack -> ptr == GUS_MCMD_COMMON_SYSEX )
              {
                int sysex_size;
                unsigned char *start;
                 
                size = 0;
                start = ++ptrack -> ptr;
                ptrack -> size--;
                sysex_size = mgetvar( ptrack );
                if ( ptrack -> size < sysex_size )
                  {
                    ptrack -> size = 0;
                    continue;
                  }
                if ( sysex_size == 1 && *(ptrack -> ptr + 1) == 0 )
                  {
                    sysex[ sysex_ptr = 0 ] = *ptrack -> ptr;
                    ptrack -> ptr++;
                    ptrack -> size--;
                    continue;
                  }
                if ( *ptrack -> ptr >= 0x80 && *ptrack -> ptr != 0xf0 )	/* this doesn't seems as SysEx */
                  {
                    raw_ptr_end = ptrack -> size - sysex_size;
                    continue;
                  }
                if ( *ptrack -> ptr == 0xf0 )	/* already */
                  {
                    ptrack -> ptr++;
                    ptrack -> size--;
                    sysex_size--;
                  }
                /* ok.. standard SysEx processing */
                if ( sysex_size )
                  {
                    if ( play )
                      msysex( device, ptrack -> ptr,
                              sysex_size - ( *(ptrack -> ptr + sysex_size - 1) == 0xf7 ? 1 : 0 ),
                              tick_time );
                    ptrack -> ptr += sysex_size;
                    ptrack -> size -= sysex_size;
                  }
                continue;
              }
            if ( *ptrack -> ptr == GUS_MCMD_COMMON_SYSEX_END )
              {
                if ( sysex_ptr < 0 )
                  {
                    ptrack -> size--;
                    ptrack -> ptr++;
                    continue;
                  }
                if ( ptrack -> size < 3 ||
                     *(ptrack -> ptr + 1) != 0x01 )	/* wrong data for this track */
                  {
                    sysex_ptr = -1;
                    ptrack -> size = 0;
                    ptrack -> tick = ~0;
                    continue;
                  }
                if ( *(ptrack -> ptr + 2) == GUS_MCMD_COMMON_SYSEX_END )
                  {
                    if ( play )
                      msysex( device, sysex, sysex_ptr + 1, tick_time );
                    sysex_ptr = -1;
                  }
                 else
                  {
                    if ( sysex_ptr < 269 )
                      sysex[ ++sysex_ptr ] = *(ptrack -> ptr + 2);
                  }
                ptrack -> ptr += 3;
                ptrack -> size -= 3;
                continue;
              }
            cmd = ptrack -> old_cmd;
            if ( *ptrack -> ptr & 0x80 )	/* new event */
              {
                ptrack -> old_cmd = cmd = *ptrack -> ptr++;
                ptrack -> size--;
              }
            if ( (unsigned char)cmd < 0x80 )
              {
#ifdef MIDIPLAY_DEBUG
                printf( "ctrack = %i, cmd = 0x%x, sync failed!!!\n", ctrack, cmd );
#endif
                ptrack -> size = 0;
                ptrack -> tick = ~0;
                continue;
              }
            chn = cmd & 0x0f;              
            pchn = &info -> channels[ chn ];
            switch ( cmd & 0xf0 ) {
              case GUS_MCMD_NOTE_OFF:
                if ( ptrack -> size < 2 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play && pchn -> device >= 0 )
                  gus_midi_note_off( pchn -> device, chn, ptrack -> ptr[ 0 ], ptrack -> ptr[ 1 ] );
                ptrack -> ptr += 2;
                ptrack -> size -= 2;
                break;
              case GUS_MCMD_NOTE_ON:
                if ( ptrack -> size < 2 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play )
                  {
                    if ( pchn -> device >= 0 )
                      gus_midi_note_on( pchn -> device, chn, ptrack -> ptr[ 0 ], ptrack -> ptr[ 1 ] );
                  }
                 else
                  {
                    if ( chn == 9 && pchn -> device >= 0 )
                      {
                        int program;
                        
                        program = ptrack -> ptr[ 0 ] | 0x80;
                        program |= pchn -> bank << 16;
                        if ( callbacks -> call_download_program )
                          callbacks -> call_download_program( callbacks -> private, pchn -> mdevice, program );
                      }
                     else
                      {
                        if ( pchn -> program < 0 )
                          {
                            pchn -> program = 0;
                            if ( callbacks -> call_download_program )
                              callbacks -> call_download_program( callbacks -> private, pchn -> mdevice, pchn -> program );
                          }
                      }
                  }
                ptrack -> ptr += 2;
                ptrack -> size -= 2;
                break;
              case GUS_MCMD_NOTE_PRESSURE:
                if ( ptrack -> size < 2 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play && pchn -> device >= 0 )
                  gus_midi_note_pressure( pchn -> device, chn, ptrack -> ptr[ 0 ], ptrack -> ptr[ 1 ] );
                ptrack -> ptr += 2;
                ptrack -> size -= 2;
                break;
              case GUS_MCMD_CONTROL:
                if ( ptrack -> size < 2 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play && pchn -> device >= 0 )
                  gus_midi_control( pchn -> device, chn, ptrack -> ptr[ 0 ], ptrack -> ptr[ 1 ] );
                switch ( ptrack -> ptr[ 0 ] ) {
                  case GUS_MCTL_MSB_BANK:
                    if ( pchn -> emulation == GUS_MIDI_EMUL_GS )
                      pchn -> bank = ptrack -> ptr[ 1 ] & 0x7f;
                     else
                      {
                        pchn -> bank &= 0x7f;
                        pchn -> bank |= ( ptrack -> ptr[ 1 ] & 0x7f ) << 7;
                      }
                    break;
                  case GUS_MCTL_LSB_BANK:
                    if ( pchn -> emulation != GUS_MIDI_EMUL_GS )
                      {
                        pchn -> bank &= ~0x7f;
                        pchn -> bank |= ptrack -> ptr[ 1 ] & 0x7f;
                      }
                    break;
                }
                ptrack -> ptr += 2;
                ptrack -> size -= 2;
                break;
              case GUS_MCMD_PGM_CHANGE:
                if ( ptrack -> size < 1 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play )
                  {
                    if ( pchn -> device >= 0 )
                      gus_midi_program_change( device, chn, ptrack -> ptr[ 0 ] );
                  }
                 else
                  {
                    int program;
                  
                    program = ptrack -> ptr[ 0 ];
                    if ( chn == 9 )	/* drum channel */
                      {
                        if ( pchn -> emulation == GUS_MIDI_EMUL_GS )
                          pchn -> bank = program;
                      }
                     else
                      {
                        pchn -> program = program;
                        program |= pchn -> bank << 16;
                        if ( callbacks -> call_download_program && pchn -> device >= 0 )
                          callbacks -> call_download_program( callbacks -> private, pchn -> mdevice, ptrack -> ptr[ 0 ] );
                      }
                  }
                ptrack -> ptr++;
                ptrack -> size--;
                break;
              case GUS_MCMD_CHANNEL_PRESSURE:
                if ( ptrack -> size < 1 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play && pchn -> device >= 0 )
                  gus_midi_channel_pressure( pchn -> device, chn, ptrack -> ptr[ 0 ] );
                ptrack -> ptr++;
                ptrack -> size--;
                break;
              case GUS_MCMD_BENDER:
                if ( ptrack -> size < 2 )
                  {
                    ptrack -> size = 0;
                    break;
                  }
                if ( play && pchn -> device >= 0 )
                  gus_midi_bender( pchn -> device, chn, ( ( ptrack -> ptr[ 1 ] & 0x7f ) << 7 ) | ( ptrack -> ptr[ 0 ] & 0x7f ) );
                ptrack -> ptr += 2;
                ptrack -> size -= 2;
                break;
            }
          }
      }
    }
  if ( play )
    {
      if ( global_stop == 1 )
        {
          gus_midi_timer_stop();
          gus_midi_abort();
        }
       else
        {
          gus_midi_flush();
          gus_midi_timer_stop();
        }
    }
   else
    {
      if ( sec_time_modulo ) sec_time++;
      if ( callbacks -> call_totaltime )
        callbacks -> call_totaltime( callbacks -> private, sec_time );
    }
  return 0;
}

int gus_midiplay_song( int device, unsigned char *song_data, unsigned int song_size, int song_number, gus_midiplay_callbacks_t *callbacks )
{
  struct midiplay_info info;

#ifdef MIDIPLAY_DEBUG
  printf( "gus_midiplay_song: data = 0x%lx, size = %i, song = %i\n", (long)song_data, song_size, song_number );
#endif
  if ( song_data == NULL || song_size <= 32 || song_number < 0 ||
       callbacks == NULL || callbacks -> version < 15 )
    {
      gus_dprintf( "mplay: wrong arguments (0x%x, %li, %i)\n", song_data, song_size, song_number );
      return -1;
    }
  if ( mread( &info, song_data, song_size, song_number ) < 0 )
    {
      gus_dprintf( "mplay: read error\n" );
      strcpy( gus_midi_error, "unknown format" );
      return -1;
    }
  global_stop = 0;
  gus_midi_reset();
  info.channels = malloc( sizeof( struct midiplay_channel ) * 16 );
  if ( !mplay( device, &info, callbacks, 0 ) )	/* first look */
    mplay( device, &info, callbacks, 1 );	/* now play data */
  gus_midi_detach();
  free( info.channels );
  free( info.tracks );
  free( info.format_name );
  return 0;
}

int gus_midiplay_stop( int pause )
{
  global_stop = pause;
  return 0;
}
