/*
 * Application interface library for the GUS driver
 * Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#ifndef __LIBGUS_H
#define __LIBGUS_H

#include <sys/types.h>
#include <fcntl.h>
#include <stdarg.h>
#if defined( __GUS_PATH_APP__ )
#include "./sys/gus.h"
#elif defined( __GUS_PATH_LIB__ )
#include "../../../include/sys/gus.h"
#else
#include <sys/gus.h>
#endif

#define LIBGUS_VERSION_MAJOR	0x0003
#define LIBGUS_VERSION_MINOR	0x0300
#define LIBGUS_VERSION		( ( LIBGUS_VERSION_MAJOR << 16 ) | LIBGUS_VERSION_MINOR )

  /*
   * all functions return value -1 if error occured
   */

/***************************************************************************
 *                                                                         *
 *              Native synthesizer support functions                       *
 *                                                                         *
 ***************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#define GUS_WAVE_DELTA		0x0100

int gus_cards( void );
  /* 
   * return value:	number of GUS cards installed in system or
   *			zero if driver isn't installed
   */
int gus_look_for_card( char *id );
  /*
   * input value:	card number (from 1 to GUS_CARDS) or id of requested card
   * return value:	card number (from 0 to GUS_CARDS-1) if card was found
   *			-1 if card wasn't found
   */
int gus_open( int card, size_t queue_buffer_size, int non_block );
  /*
   * input values:	card number,
   *			size of command queue buffer (512-1MB)
   *			buffer is allocated dynamically,
   *			non block mode
   * return value:	zero if success
   * note 1:		this function must be called as first
   * 			open file /dev/gus
   * note 2:		you can open more cards with one process
   */
int gus_close( int card );
  /*
   * close file (gus synthesizer) previously opened with gusOpen function
   * return value:	zero if success
   */
int gus_select( int card );
  /*
   * select specified card
   * return value:	zero if success
   */
int gus_get_handle( void );
  /*
   * return value:	file handle (descriptor) for /dev/gus
   */

int gus_info( gus_info_t *info, int reread );
  /*
   * return value:	filled info variable with actual values
   * 			(look at gus.h header file for more informations)
   * version field:	0x0024	- GUS revision 2.4
   *			0x0035	- GUS revision 3.7 with flipped mixer channels
   *			0x0037	- GUS revision 3.7
   *			0x0090	- GUS ACE
   *			0x00a0	- GUS MAX revision 10
   *			0x00a1	- GUS MAX revision 11
   *                    0x0100  - InterWave (full version)
   * flags field:	see to GUS_STRU_INFO_F_???? constants (gus.h header file)
   * port field:	port number (for example 0x220)
   * irq field:		irq number (for example 11)
   * dma1 field:	dma1 number (for example 5)
   * dma2 field:	dma2 number (for example 6)
   * note:		dma1 and dma2 could be same in case of only one dma channel used
   */

#define gus_get_ultraclick( x ) gus_ultraclick_get( x )
#define gus_set_ultraclick( x ) gus_ultraclick_set( x )
int gus_ultraclick_get( gus_ultraclick_t *ultraclick );
  /*
   * return value:	filled gus_ultraclick_t structure
   */
int gus_ultraclick_set( gus_ultraclick_t *ultraclick );
  /*
   * input_value:	filled gus_ultraclick_t structure
   */
int gus_emulation_get( void );		/* midi emulation - gus_do_chn_xxxx */
  /*
   * return value:	see to GUS_MIDI_EMUL_XXXX constants
   *			-1 if error
   */
int gus_emulation_set( int emulation );	/* midi emulation - gus_do_chn_xxxx */
  /*
   * input value:	see to GUS_MIDI_EMUL_XXXX constants
   * return value:	zero if success
   */
int gus_effect_reset( void );
  /*
   * remove effect(s) voices and disable effects
   */
int gus_effect_setup( gus_effect_t *effect ); 
  /*
   * setup effect
   *	for an InterWave board can be used gus_effect_interwave() function
   *	which fills gus_effect_t structure with two of selected effects
   * note:
   *	this setup assign 8 voices to effect processing, so only
   *	24 voices can be used with gus_do_voice_xxxx() functions!!!
   */
 
int gus_memory_size( void );
  /*
   * return value:	gus memory size in bytes
   */
int gus_memory_free_size( void );
  /*
   * return value:	unused gus memory in bytes
   * warning:		reset function must be called before
   */
int gus_memory_free_block( int w_16bit );
  /*
   * return value:	current largest free block for 8-bit or 16-bit wave
   */
int gus_memory_rom_size( void );
  /*
   * return value:	gus memory size (ROM) in bytes
   */
int gus_memory_rom_banks( void );
  /*
   * return value:	gus memory ROM banks
   */

int gus_reset( int voices, unsigned int channel_voices );
  /*
   * input values:	active voices and channel voices (for dynamic allocation)
   * return value:	number of active voices if reset was successfull (GF1 chip active)
   */
int gus_reset_engine_only( void );
  /*
   * return value:	same as gus_reset function
   * note:		this command doesn't change number of active
   *			voices and doesn't do hardware reset
   */
   
int gus_timer_start( void );
  /*
   * return value:	zero if successfull
   */
int gus_timer_stop( void );
  /*
   * return value:	zero if timer was stoped
   */
int gus_timer_continue( void );
  /*
   * return value:	zero if timer will be continue
   */
int gus_timer_base( int base );
  /*
   * return value:	zero if setup was success (default timebase = 100)
   */
int gus_timer_tempo( int ticks );
  /*
   * return value:	zero if setup was success
   */
int gus_timer_master( void );
int gus_timer_slave( int slave_card );

int gus_memory_reset( int mode );
  /*
   * input value:	see to GUS_DOWNLOAD_MODE_XXXX constants (gus.h) 
   * return value:	zero if samples & instruments was successfully removed
   *			from GF1 memory manager
   */
int gus_memory_test( gus_instrument_t *instrument );
  /*
   * input value:	look at gus.h for more details about gus_instrument_t structure
   * return value:	zero if sample !can be! downloaded
   * note:		sample isn't allocated in GF1 memory manager
   */
int gus_memory_get_name( gus_instrument_name_t *instrument );
  /*
   * 			get name of instrument and instrument flags
   * input value:	look at gus.h for more details about gus_instrument_t structure
   * return value:	zero if success
   */
int gus_memory_alloc( gus_instrument_t *instrument );
  /*
   * input value:	look at gus.h for more details about gus_instrument_t structure
   * return value:	zero if instrument was successfully allocated
   */
int gus_memory_free( gus_instrument_t *instrument );
  /*
   * input value:	look at gus.h for more details about gus_instrument_t structure
   * return value:	zero if instrument was successfully removed
   */
int gus_memory_pack( void );
  /*
   * return value:	zero if success
   */
int gus_memory_list( gus_memory_list_t *list );
  /*
   * input value:	list -> count = maximum instruments to read
   *			list -> info.instruments = pointer to unsigned int array
   *			  which is filled by driver, -1 = no more instruments
   * return value:	zero if success
   */
int gus_memory_block_alloc( gus_memory_block_t *block );
  /*
   * undocumented at this time - see to gus.h for more info
   */
int gus_memory_block_free( gus_memory_block_t *block );
  /*
   * undocumented at this time - see to gus.h for more info
   */
int gus_memory_dump( gus_memory_dump_t *dump );
  /*
   * undocumented at this time - see to gus.h for more info
   */

int gus_queue_write_set_size( int items );
  /*
   * input value:	write queue size in items (each item have 8 bytes)
   */
int gus_queue_write_get_size( void );
  /*
   * output value:	write queue size in items (each item have 8 bytes)
   */
int gus_queue_write_free( void );
  /*
   * return value:	free items (ticks) in queue
   */
int gus_queue_write_threshold( int threshold );
  /*
   * input value:	threshold in queue items (ticks)
   *			range 1 - gus_queue_write_get_size()
   *			good value = items - ( items / 3 )
   * return value:	zero if success
   */

int gus_queue_time_set_size( int items );
  /*
   * input value:	time queue size in items (each item have 4 bytes)
   */
int gus_queue_time_get_size( void );
  /*
   * output value:	time queue size in items (each item have 4 bytes)
   */
int gus_queue_time_free( void );
  /*
   * return value:	free items (ticks) in queue
   */
int gus_queue_time_threshold( int threshold );
  /*
   * input value:	threshold in queue items (ticks)
   *			range 1 - gus_queue_time_get_size()
   *			good value = items - ( items / 3 )
   * return value:	zero if success
   */

int gus_queue_read_set_size( int items );
  /*
   * input value:	echo buffer size in items (if 0 - erase echo buffer)
   */
int gus_queue_read_get_size( void );
  /*
   * output value:	echo buffer size in items
   */
int gus_queue_read_used( void );
  /*
   * output value:	used items in read queue
   */

int gus_queue_flush( void );
  /*
   * return value:	zero if command queue was successfully flushed
   */
int gus_queue_abort( void );
  /*
   * return value:	zero if command queue was successfully aborted and
   *			GF1 chip was stoped
   * note:		reset must be called after this command for
   *			more sound!
   */
int gus_queue_abort_to_stop( void );
  /*
   * return value:	zero if command queue was successfully aborted
   *			to queue command STOP; all voices was stoped
   */
int gus_queue_stop_playback( void );
  /*
   *			zero if command was successfull
   */
int gus_queue_continue_playback( void );
  /*
   *			zero if command was successfull
   */
   
/*
 *  Queue commands
 *
 *  This commands are interpreted by block. Blocks are separated by command
 *  gus_do_wait or gus_do_stop (for abort_to_stop command). Queue command
 *  isn't sent to GUS driver in same time user calls it. Queue commands
 *  are stored into internal buffer dynamically allocated by gus library
 *  (look at gus_open function). Internal buffer is flushed (forwarded to
 *  GUS driver) when is full or in case user forces buffer flush with
 *  gus_do_flush function.
 */

int gus_do_flush( void );
  /*
   * return value:	zero if command queue was successfully flushed
   *			in non block mode - number of written bytes
   */


void gus_do_program( unsigned char voice, unsigned int program );
void gus_do_voice_start( unsigned char voice, unsigned int program, unsigned int freq, unsigned short volume, unsigned short pan );
  /*
   * start voice
   *		voice	 : voice #
   *		program	 : program # or ~0 = current
   *		freq	 : frequency in Hz
   *		volume	 : volume level (0-16384) or ~0 = current
   *		pan	 : pan level (0-16384) or ~0 = current
   */
void gus_do_voice_start_position( unsigned char voice, unsigned int program, unsigned int freq, unsigned short volume, unsigned short pan, unsigned int position );
  /*
   * start voice
   *		voice	 : voice #
   *		program	 : program # or ~0 = current
   *		freq	 : frequency in Hz
   *		volume	 : volume level (0-16384) or ~0 = current
   *		pan	 : pan level (0-16384) or ~0 = current
   *		position : offset to wave in bytes * 16 (lowest 4 bits - fraction)
   */
void gus_do_voice_stop( unsigned char voice, unsigned char mode );
  /*
   * stop voice
   *		mode = 0 : stop voice now
   *		mode = 1 : disable wave loop and finish it
   */

void gus_do_voice_control( unsigned char voice, unsigned char cntrl );
  /*
   * set voice control register:
   *	bit 0. - 0 = voice is running, 1 = voice is stopped
   *	bit 1. - 0 = nothing, 1 = stop voice
   *	bit 2. - 0 = 8 bit wave, 1 = 16 bit wave
   *	bit 3. - 0 = nothing, 1 = enable looping
   *	bit 4. - 0 = nothing, 1 = bidirectional looping
   *	bit 5. - 0 = nothing, 1 = enable wavetable IRQ
   *	bit 6. - 0 = increasing addresses, 1 = decreasing addresses
   *	bit 7. - 1 = copy bits 0. & 1. from actual voice control register
   * note: don't set bits 5 & 7 - this thing must be controled by driver
   */
void gus_do_voice_frequency( unsigned char voice, unsigned int freq );
  /*
   * set voice frequency in Hz
   */
void gus_do_voice_volume( unsigned char voice, unsigned short vol );
  /*
   * set voice volume level 0-16384 (linear)
   */
void gus_do_voice_loop_range( unsigned char voice, unsigned int start, unsigned int end );
  /*
   * set voice loop start & end positions in sample
   * values are relative from the begining of sample
   */
void gus_do_voice_ramp( unsigned char voice, unsigned char start, unsigned char end, unsigned char rate, unsigned char control );
  /*
   * set voice ramp start value
   *    bits 7-4: exponent
   *    bits 3-0: mantisa
   * set voice ramp end value
   *    bits 7-4: exponent
   *    bits 3-0: mantisa
   * set voice volume ramping rate
   * (look at UltraSound Lowlevel Toolkit documentation for more details)
   *    bits 7-6: rate
   *	bits 5-0: volume increment
   * set ramp control register
   *   bit 0. - 0 = volume ramping in progress, 1 = volume ramping is inactive
   *   bit 1. - 0 = nothing, 1 = stop ramp
   *   bit 2. - 0 = nothing, 1 = roll over condition
   *   bit 3. - 0 = nothing, 1 = ramp loop enable
   *   bit 4. - 0 = nothing, 1 = bidirectional ramp loop enable
   *   bit 5. - 0 = nothing, 1 = volume ramp IRQ enable
   *   bit 6. - 0 = increasing direction, 1 = decreasing direction
   *   bit 7. - 0 = nothing, 1 = volume ramp IRQ pending
   * note: don't set bits 2,5,7 - these things must be controled by driver
   */
void gus_do_voice_position( unsigned char voice, unsigned int pos );
  /*
   * set current voice position
   * value is relative from the begining of sample
   */
void gus_do_voice_pan( unsigned char voice, unsigned short pan );
  /*
   * set voice pan (0-16384) (full left - full right)
   */

void gus_do_voice_lfo_setup( unsigned char voice, int type, int freq, int final_depth, int sweep, int shape );
void gus_do_voice_lfo_freq( unsigned char voice, int type, int freq );
void gus_do_voice_lfo_depth( unsigned char voice, int type, int depth );
void gus_do_voice_lfo_enable( unsigned char voice, int type );
void gus_do_voice_lfo_disable( unsigned char voice, int type );
void gus_do_voice_lfo_shutdown( unsigned char voice, int type );

void gus_do_voice_interwave_effect( unsigned char voice, unsigned short eff_volume, int eff1, int eff2, int eff3, int eff4 );
  /*
   * select effects for this voice and setup effect volume
   * effect_volume level 0-16384
   * if effX == -1 -> no effect
   * if this command is used with gus_effect_interwave() function,
   *   reverb effect = 0 and chorus effect = 2
   */
void gus_do_voice_frequency_mode( unsigned int mode );
  /*
   * see to GUS_FREQ_XXXX contants in gus.h
   */
void gus_do_voice_volume_mode( unsigned short mode );
  /*
   * see to GUS_VOLUME_XXXX contants in gus.h
   */
void gus_do_voice_pan_mode( unsigned short mode );
  /*
   * see to GUS_PAN_XXXX contants in gus.h
   */

void gus_do_chn_program( unsigned char channel, unsigned short program );
void gus_do_chn_note_on( unsigned char channel, unsigned char note, unsigned char velocity, unsigned char priority );
void gus_do_chn_note_off( unsigned char channel, unsigned char note, unsigned char velocity );
void gus_do_chn_pitchbend( unsigned char channel, unsigned short pitchbend );
void gus_do_chn_control( unsigned char channel, unsigned char p1, unsigned char p2 );
void gus_do_chn_note_pressure( unsigned char channel, unsigned char p1, unsigned char p2 );
void gus_do_chn_pressure( unsigned char channel, unsigned char p1 );

void gus_do_tempo( unsigned int tempo );
  /*
   * set new tempo
   */
void gus_do_wait( unsigned int ticks );
  /*
   * wait x ticks - this command is block separator
   * all commands between blocks are interpreted in the begining of one tick
   */
void gus_do_stop( void );
  /*
   * this command is remark for gus_queue_abort_to_stop
   */
void gus_do_echo( unsigned char *src, int size );
  /*
   * echo data at *src (max 7 bytes) in specified time to file (stream) input
   */

#ifdef __cplusplus
}
#endif

struct GUS_STRU_CALLBACKS {
  unsigned int version;			/* version of this structure - 2 */
  void *private;			/* user data */
  void (*call_echo)( void *private, unsigned char *data );
  void (*call_voices_change)( void *private, int voices );
};

typedef struct GUS_STRU_CALLBACKS gus_callbacks_t;

#ifdef __cplusplus
extern "C" {
#endif
   
int gus_do_input( gus_callbacks_t *callbacks );
  /*
   * return value:	zero if success, -1 if error
   */

#ifdef __cplusplus
}
#endif

/***************************************************************************
 *                                                                         *
 *             Support for various instrument formats                      *
 *                                                                         *
 ***************************************************************************/

typedef struct __gus_rom_interwave_header {
  char iwave[ 8 ];
  unsigned char rom_hdr_revision;
  unsigned char series_number;
  char series_name[ 16 ];
  char date[ 10 ];
  unsigned short vendor_revision_major;
  unsigned short vendor_revision_minor;
  unsigned int rom_size;
  char copyright[ 128 ];
  char vendor_name[ 64 ];
  char rom_description[ 128 ];
} gus_rom_interwave_header_t;

#ifdef __cplusplus
extern "C" {
#endif

gus_instrument_t *gus_instr_alloc( void );
int gus_instr_free( gus_instrument_t *instrument );
int gus_instr_patch_load( gus_instrument_t *instr, char *filename, int s8bit );
int gus_instr_ffff_access_synth( void );
int gus_instr_ffff_access_daemon( void );
int gus_instr_ffff_access_midi( int midi_device );
int gus_instr_ffff_get_rom_header( int fd, int access, int bank, gus_rom_interwave_header_t *header );
int gus_instr_ffff_open( char *fff_filename, char *dat_filename );
int gus_instr_ffff_open_rom( int fd, int access, int bank, int file );
int gus_instr_ffff_close( int handle );
int gus_instr_ffff_download( int handle, int fd, int access, int temporary );
int gus_instr_ffff_download_free( int handle, int fd, int access, int temporary );
int gus_instr_ffff_load( gus_instrument_t *instr, int handle, int bank, int prog );

#ifdef __cplusplus
}
#endif

/***************************************************************************
 *                                                                         *
 *                            Effects support                              *
 *                                                                         *
 ***************************************************************************/

/* effect1 - known as reverb */
#define GUS_EFFECT_INTERWAVE_1_ROOM1		0
#define GUS_EFFECT_INTERWAVE_1_ROOM2		1
#define GUS_EFFECT_INTERWAVE_1_ROOM3		2
#define GUS_EFFECT_INTERWAVE_1_HALL1		3
#define GUS_EFFECT_INTERWAVE_1_HALL2		4	/* default in DOS */
#define GUS_EFFECT_INTERWAVE_1_PLATE		5
#define GUS_EFFECT_INTERWAVE_1_DELAY		6
#define GUS_EFFECT_INTERWAVE_1_PANNING_DELAY	7
/* effect3 - known as chorus */
#define GUS_EFFECT_INTERWAVE_3_CHORUS1		0
#define GUS_EFFECT_INTERWAVE_3_CHORUS2		1
#define GUS_EFFECT_INTERWAVE_3_CHORUS3		2	/* default in DOS */
#define GUS_EFFECT_INTERWAVE_3_CHORUS4		3
#define GUS_EFFECT_INTERWAVE_3_FEEDBACK_CHORUS	4
#define GUS_EFFECT_INTERWAVE_3_FLANGER		5
#define GUS_EFFECT_INTERWAVE_3_SHORT_DELAY	6
#define GUS_EFFECT_INTERWAVE_3_SHORT_DELAY_FB	7	/* feed back? */

int gus_effect_interwave( gus_effect_t *effect, int effect1, int effect3 );

/***************************************************************************
 *                                                                         *
 *                    GUS mixer support functions                          *
 *                                                                         *
 ***************************************************************************/

#define GUS_MIXER_NONE		0	/* GUS rev. <3.7 */
#define GUS_MIXER_ICS		1	/* GUS rev. 3.7 & above */
#define GUS_MIXER_CODEC		2	/* GUS MAX or 16-bit Daughter Board */
#define GUS_MIXER_ENH_CODEC	3	/* enhanced codec - GUS PnP */

#define GUS_MIXER_ID_MASTER	SOUND_MIXER_VOLUME
#define GUS_MIXER_ID_SYNTH	SOUND_MIXER_SYNTH
#define GUS_MIXER_ID_PCM	SOUND_MIXER_PCM
#define GUS_MIXER_ID_LINE	SOUND_MIXER_LINE
#define GUS_MIXER_ID_MIC	SOUND_MIXER_MIC
#define GUS_MIXER_ID_CD		SOUND_MIXER_CD
#define GUS_MIXER_ID_SOFT	SOUND_MIXER_IMIX
#define GUS_MIXER_ID_GAIN	SOUND_MIXER_RECLEV
#define GUS_MIXER_ID_LOOP	SOUND_MIXER_LINE1
#define GUS_MIXER_ID_EFFECT	SOUND_MIXER_LINE2

#define GUS_MIXER_MASK_MASTER	SOUND_MASK_VOLUME
#define GUS_MIXER_MASK_SYNTH	SOUND_MASK_SYNTH
#define GUS_MIXER_MASK_PCM	SOUND_MASK_PCM
#define GUS_MIXER_MASK_LINE	SOUND_MASK_LINE
#define GUS_MIXER_MASK_MIC	SOUND_MASK_MIC
#define GUS_MIXER_MASK_CD	SOUND_MASK_CD
#define GUS_MIXER_MASK_SOFT	SOUND_MASK_IMIX
#define GUS_MIXER_MASK_GAIN	SOUND_MASK_RECLEV
#define GUS_MIXER_MASK_LOOP	SOUND_MASK_LINE1
#define GUS_MIXER_MASK_EFFECT	SOUND_MASK_LINE2

#ifdef __cplusplus
extern "C" {
#endif

int gus_mixer_cards( void );
  /*
   * return value:	cards installed in the system
   */
int gus_mixer_look_for_card( char *id );
  /*
   * see to gus_look_for_card()
   */
int gus_mixer_open( int card );
  /*
   * input value:	selected card
   * return value:	selected card if successfull
   */
int gus_mixer_close( int card );
  /*
   * input value:	card to close
   * return value:	zero if successfull
   */
int gus_mixer_select( int card );
  /*
   * input value:	card to select
   * return value:	zero if successfull
   */

int gus_mixer_version( void );
  /*
   * return value:	see to GUS_MIXER_XXXX constants
   */
int gus_mixer_read_devmask( void );
  /*
   * return value:	available devices mask
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_read_stereodevs( void );
  /*
   * return value:	available stereo devices mask
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_read_recmask( void );
  /*
   * return value:	available record devices mask
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_read_recsrc( void );
  /*
   * return value:	current record source
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_write_recsrc( int value );
  /*
   * input value:	new record source
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_read_devs_lmute( void );
  /*
   * return value:	bitmask with muted devices (left channel)
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_read_devs_rmute( void );
  /*
   * return value:	bitmask with muted devices (right channel)
   *			(look at GUS_MIXER_MASK_XXXX)
   */
int gus_mixer_write_devs_lmute( int dev, int mute );
  /*
   * input value:	device and mute flag ( 0 - unmute, != 0 - mute)
   *			left channel
   *			(look at GUS_MIXER_MASK_XXXX - soundcard.h)
   * return value:	zero if successfull
   */
int gus_mixer_write_devs_rmute( int dev, int mute );
  /*
   * input value:	device and mute flag ( 0 - unmute, != 0 - mute)
   *			right channel
   *			(look at GUS_MIXER_MASK_XXXX - soundcard.h)
   * return value:	zero if successfull
   */

  /*
   * functions bellow use following form of the input/output values:
   *	bits 0-7 (low byte): left channel volume level (0-100)
   *	bits 8-15 (high byte): right channel volume level (0-100)
   */

int gus_mixer_read_mic( void );		/* all (NONE & CODEC - mono) */
int gus_mixer_read_cd( void );		/* ICS, CODEC, ENH_CODEC */
int gus_mixer_read_line( void );	/* all (NONE - mono) */
int gus_mixer_read_synth( void );	/* all (NONE - mono) */
int gus_mixer_read_pcm( void );		/* all */
int gus_mixer_read_reclev( void );	/* CODEC, ENH_CODEC - mono */
int gus_mixer_read_volume( void );	/* ICS, ENH_CODEC */
int gus_mixer_read_imix( void );	/* ICS, CODEC, ENH_CODEC - all mono */
int gus_mixer_read_loopback( void );	/* CODEC, ENH_CODEC - mono */
int gus_mixer_read_effect( void );	/* InterWave only - mono */

int gus_mixer_write_mic( int value );
int gus_mixer_write_cd( int value );
int gus_mixer_write_line( int value );
int gus_mixer_write_synth( int value );
int gus_mixer_write_pcm( int value );
int gus_mixer_write_reclev( int value );
int gus_mixer_write_volume( int value );
int gus_mixer_write_imix( int value );
int gus_mixer_write_loopback( int value );
int gus_mixer_write_effect( int value );

int gus_mixer_read_interwave_serial( void );
int gus_mixer_write_interwave_serial( int serial );

#ifdef __cplusplus
}
#endif

/***************************************************************************
 *                                                                         *
 *                      Conversion routines                                *
 *                                                                         *
 ***************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

void gus_convert_delta( unsigned int type, unsigned char *dest,
			unsigned char *src, size_t size );
  /*
   * note: dest and src pointers can be equal
   */
void gus_convert_8bit_to_16bit( unsigned short *dest, unsigned char *src,
				size_t size );
  /*
   * note: dest and src pointers can be equal
   */
void gus_convert_spline_oversampling_8bit(
			signed char *dest, signed char *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */
void gus_convert_spline_oversampling_16bit(
			signed short *dest, signed short *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */
void gus_convert_spline_oversampling_8bit_to_16bit(
			signed short *dest, signed char *src,
			size_t size, float factor, int unsigned_ );
  /*
   * note: output array is always signed
   */

#ifdef __cplusplus
}
#endif

/***************************************************************************
 *                                                                         *
 *                             PCM routines                                *
 *                                                                         *
 ***************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

int gus_pcm_cards( void );
  /*
   * return value:	cards installed in the system
   */
int gus_pcm_look_for_card( char *id );
  /*
   * see to gus_look_for_card()
   */

#define GUS_PCM_DIRECTION_PLAY		O_WRONLY
#define GUS_PCM_DIRECTION_RECORD	O_RDONLY
#define GUS_PCM_DIRECTION_DUPLEX	O_RDWR

#define GUS_PCM_OF_NONE			0x00000000
#define GUS_PCM_OF_NONBLOCK		0x00000001

int gus_pcm_open( int card, int direction, int flags );
int gus_pcm_close( int card );
int gus_pcm_get_handle( int card );

int gus_pcm_select_card( int card );

#define GUS_PCM_MODE_NONE		AFMT_QUERY
#define GUS_PCM_MODE_S8			AFMT_S8
#define GUS_PCM_MODE_U8			AFMT_U8
#define GUS_PCM_MODE_S16_LE		AFMT_S16_LE	/* little endian */
#define GUS_PCM_MODE_U16_LE		AFMT_U16_LE	/* little endian */
#define GUS_PCM_MODE_S16_BE		AFMT_S16_BE	/* big endian */
#define GUS_PCM_MODE_U16_BE		AFMT_U16_BE	/* big endian */
#define GUS_PCM_MODE_MU_LAW		AFMT_MU_LAW
#define GUS_PCM_MODE_A_LAW		AFMT_A_LAW
#define GUS_PCM_MODE_IMA_ADPCM		AFMT_IMA_ADPCM

int gus_pcm_mode_set( int direction, unsigned int mode );
unsigned int gus_pcm_mode_get( int direction );
unsigned int gus_pcm_mode_supported( int direction );

int gus_pcm_channels_set( int direction, int channels );
int gus_pcm_channels_get( int direction );
int gus_pcm_channels_supported( int direction );

int gus_pcm_rate_set( int direction, int rate );
int gus_pcm_rate_get( int direction );
int gus_pcm_rate_supported( int direction );

int gus_pcm_dma_size_set( int direction, int fragments, int fragment_size );
int gus_pcm_dma_size_get( int direction, int current );

int gus_pcm_sync( void );

int gus_pcm_read( unsigned char *buffer, int count );
int gus_pcm_write( unsigned char *buffer, int count );

#ifdef __cplusplus
}
#endif

/***************************************************************************
 *                                                                         *
 *                             MIDI routines                               *
 *                                                                         *
 ***************************************************************************/

#define GUS_MIDI_CHANNELS		16
#define GUS_MIDI_PROGRAM( bank, prog )	(((unsigned int)bank<<16)|prog)
#define GUS_MIDI_PROGRAM_BANK( prog )	(prog >> 16)
#define GUS_MIDI_PROGRAM_PROG( prog )	(prog & 0xffff)

typedef struct __gus_midi_device {
  unsigned char mode;		/* mode.. GUS_MIDI_OPEN_MODE_XXXX */
  unsigned char device;		/* input/output device number */
  unsigned short channels;	/* midi channels processed by this device */
  unsigned char midi_device;	/* midi device # */
  unsigned char emulation;	/* emulation - GUS_MIDI_EMUL_XXXX */
  unsigned int cap;		/* device capabilities - GUS_MIDI_CAP_XXXX */
  unsigned short midi_id_number;
  unsigned short midi_family_code;
  unsigned short midi_family_number_code;
  unsigned int midi_software_revision;
  unsigned int reserved[ 4 ];	/* reserved for future use */
  struct __gus_midi_device *next; /* next device */
} gus_midi_device_t;

extern char gus_midi_error[];

#ifdef __cplusplus
extern "C" {
#endif

int gus_midi_cards( void );
int gus_midi_devices( gus_midi_devices_t *devices );
int gus_midi_device_info( gus_midi_device_info_t *info );
char *gus_midi_device_info_manufacture( gus_midi_device_t *device );
char *gus_midi_device_info_family( gus_midi_device_t *device );
char *gus_midi_device_info_product( gus_midi_device_t *device );
int gus_midi_autodetection( int device );
gus_midi_device_t *gus_midi_output_device( int device );
gus_midi_device_t *gus_midi_input_device( int device );
gus_midi_device_t *gus_midi_output_devices( void );
gus_midi_device_t *gus_midi_input_devices( void );

#define GUS_MIDI_IN		O_RDONLY
#define GUS_MIDI_OUT		O_WRONLY
#define GUS_MIDI_BOTH		O_RDWR

#define GUS_MIDI_OF_NONE	0x00000000
#define GUS_MIDI_OF_NONBLOCK	0x00000001
#define GUS_MIDI_OF_ECHO_ENABLE	0x00000002

int gus_midi_fill_device_structure( gus_midi_device_t *device, int mode, int device_number, unsigned short channels );
int gus_midi_open( int file_flags, gus_midi_device_t *cards, size_t buffer_size, int flags );
int gus_midi_open_intelligent( int file_flags, char *cfg_file, size_t buffer_size, int flags );
int gus_midi_close( void );
int gus_midi_get_handle( void );

int gus_midi_realtime( int enable );

int gus_midi_gs_sysex_checksum( unsigned char *data, int size );

int gus_midi_reset( void );
int gus_midi_detach( void );
int gus_midi_emulation_set( int device, int emulation );	/* see to GUS_MIDI_EMUL_XXXX */
int gus_midi_emulation_get( int device );			/* see to GUS_MIDI_EMUL_XXXX */

int gus_midi_note_on( int device, int chn, int note, int vel );
int gus_midi_note_off( int device, int chn, int note, int vel );
int gus_midi_note_pressure( int device, int chn, int note, int vel );
int gus_midi_channel_pressure( int device, int chn, int vel );
int gus_midi_bender( int device, int chn, int pitch );	/* range: 0-16383 */
int gus_midi_program_change( int device, int chn, int program );
int gus_midi_control( int device, int chn, int p1, int p2 );
int gus_midi_sysex( int device, unsigned char *data, int len );
int gus_midi_master_volume( int device, int volume );

int gus_midi_raw_data( int device, unsigned char *data, int len );
int gus_midi_echo( unsigned char *data, int len );

int gus_midi_timer_base( int base );
int gus_midi_timer_tempo( int tempo );
int gus_midi_timer_start( void );
int gus_midi_timer_stop( void );
int gus_midi_timer_continue( void );
int gus_midi_wait( int ticks );
int gus_midi_write( void );
int gus_midi_flush( void );
int gus_midi_abort( void );
int gus_midi_threshold( int threshold );

#ifdef __cplusplus
}
#endif

typedef struct __gus_midi_callbacks {
  unsigned int version;			/* version of this structure - 17 */
  void *private;			/* user data */
  void (*call_note_on)( void *private, int device, int chn, int note, int vel );
  void (*call_note_off)( void *private, int device, int chn, int note, int vel );
  void (*call_note_pressure)( void *private, int device, int chn, int note, int vel );
  void (*call_channel_pressure)( void *private, int device, int chn, int vel );
  void (*call_bender)( void *private, int device, int chn, int pitch );
  void (*call_program_change)( void *private, int device, int chn, int program );
  void (*call_control)( void *private, int device, int chn, int p1, int p2 );
  void (*call_sysex)( void *private, int device, unsigned char *data, int len );
  void (*call_echo)( void *private, unsigned char *data, int len );
  void (*call_wait)( void *private, unsigned int tick );
  void (*call_mtc_quarter)( void *private, int device, int quarter );
  void (*call_song_select)( void *private, int device, int song );
  void (*call_song_position)( void *private, int device, int position );
  void (*call_tune_request)( void *private, int device );
  void (*call_start)( void *private, int device );
  void (*call_continue)( void *private, int device );
  void (*call_stop)( void *private, int device );
} gus_midi_callbacks_t;

#ifdef __cplusplus
extern "C" {
#endif

int gus_midi_input( gus_midi_callbacks_t *callbacks );

int gus_midi_thru_set( gus_midi_thru_t *thru );
int gus_midi_thru_get( gus_midi_thru_t *thru );
int gus_mdii_thru_clear( int i_dev );
int gus_midi_thru_channel_set( int i_dev, int i_chn, int o_dev, int o_chn, int o_vel );
int gus_midi_thru_channel_clear( int i_dev, int i_chn, int o_dev );

int gus_midi_preload_bank( int device, char *bank_name );
int gus_midi_preload_program( int device, int *instruments, int count );

int gus_midi_icfg_open( int device, char *filename );
int gus_midi_icfg_close( int device );
int gus_midi_icfg_reopen( int device );
void gus_midi_icfg_error_handler( int device,
	void (*error)( int device, char *filename, int line, char *format, va_list va ),
	void (*warning)( int device, char *filename, int line, char *format, va_list va ) );
int gus_midi_icfg_download_bank( int device, char *bank_name );
int gus_midi_icfg_download_program( int device, int *instruments, int count );

int gus_midi_memory_reset_all( void );
int gus_midi_memory_reset( int device );
int gus_midi_memory_test( int device, gus_instrument_t *instr );
int gus_midi_memory_alloc( int device, gus_instrument_t *instr );
int gus_midi_memory_free( int device, gus_instrument_t *instr );
int gus_midi_memory_pack( int device );
int gus_midi_memory_list( int device, gus_memory_list_t *list );
int gus_midi_memory_block_alloc( int device, gus_memory_block_t *block );
int gus_midi_memory_block_free( int device, gus_memory_block_t *block );
int gus_midi_memory_get_name( int device, gus_instrument_name_t *name );
int gus_midi_memory_dump( int device, gus_memory_dump_t *dump );

int gus_midi_synth_info( int device, gus_info_t *info );

int gus_midi_effect_reset( int device );
int gus_midi_effect_setup( int device, gus_effect_t *effect );

#ifdef __cplusplus
}
#endif

typedef struct __gus_midiplay_callbacks {
  unsigned int version;			/* version of this structure - 15 */
  void *private;			/* user data */
  int last_fd;				/* last file descriptor - for select() */
  fd_set read_fds;
  fd_set write_fds;
  void (*call_select)( void *private, fd_set *read_fds, fd_set *write_fds );
  void (*call_info)( void *private, int format, char *format_name, int tracks );
  void (*call_download_program)( void *private, gus_midi_device_t *device, unsigned int program );
  void (*call_totaltime)( void *private, unsigned int time );	/* in 1/10 of sec */
  void (*call_time)( void *private, unsigned int time );	/* in 1/10 of sec */
  void (*call_text)( void *private, int track, char *text );
  void (*call_copyright)( void *private, int track, char *text );
  void (*call_sequence)( void *private, int track, char *text );
  void (*call_instrument)( void *private, int track, char *text );
  void (*call_lyric)( void *private, int track, char *text );
  void (*call_marker)( void *private, int track, char *text );
  void (*call_cuepoint)( void *private, int track, char *text );
} gus_midiplay_callbacks_t;

#ifdef __cplusplus
extern "C" {
#endif

int gus_midiplay_songs( unsigned char *song_data, unsigned int song_size );
int gus_midiplay_song( int device, unsigned char *song_data, unsigned int song_size, int song_number, gus_midiplay_callbacks_t *callbacks );
int gus_midiplay_stop( int pause );
  /*
   * pause == 1 -> abort
   * pause == 2 -> pause
   * pause == 0 -> continue
   */

#ifdef __cplusplus
}
#endif

/*
 *
 */
 
#endif /* __LIBGUS_H */
