/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for InterWave ROM control
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include "libgus.h"
#include "libgus_local.h"

/*
 *  defines
 */

#undef IW_ROM_DEBUG

#define IW_ROM_PROGRAM_VERSION_MAJOR	1
#define IW_ROM_PROGRAM_VERSION_MINOR	0

#define IW_FORMAT_8BIT		0x01
#define IW_FORMAT_SIGNED	0x02
#define IW_FORMAT_FORWARD	0x04
#define IW_FORMAT_LOOP		0x08
#define IW_FORMAT_BIDIR		0x10
#define IW_FORMAT_ULAW		0x20
#define IW_FORMAT_ROM		0x80

/*
 *  structures
 */

union ID {
  unsigned char c[ 4 ];
  unsigned int v;
  struct {
    unsigned short major;
    unsigned short minor;
  } version;
};

struct header {
  union ID id;
  unsigned int length;
};

struct program {
  /* 00 */ union ID id;
  /* 04 */ union ID version;
};

#define PATCH_SIZE 20

struct patch {
  /* 00 */ union ID id;
  /* 04 */ short nlayers;
  /* 06 */ char layer_mode;
  /* 07 */ char exclusion_mode;
  /* 08 */ short exclusion_group;
  /* 10 */ char effect1;
  /* 11 */ char effect1_depth;
  /* 12 */ char effect2;
  /* 13 */ char effect2_depth;
  /* 14 */ unsigned char bank;
  /* 15 */ unsigned char program;
  /* 16 */ char *layer;
};

#define LFO_SIZE 8

struct LFO {
  /* 00 */ unsigned short freq;
  /* 02 */ short depth;
  /* 04 */ short sweep;
  /* 06 */ unsigned char shape;
  /* 07 */ unsigned char delay;
};

#define LAYER_SIZE 42

struct layer {
  /* 00 */ union ID id;
  /* 04 */ char nwaves;
  /* 05 */ char flags;
  /* 06 */ char high_range;
  /* 07 */ char low_range;
  /* 08 */ char pan;
  /* 09 */ char pan_freq_scale;
  /* 10 */ struct LFO tremolo;
  /* 18 */ struct LFO vibrato;
  /* 26 */ char velocity_mode;
  /* 27 */ char attenuation;
  /* 28 */ short freq_scale;
  /* 30 */ char freq_center;
  /* 31 */ char layer_event;
  /* 32 */ union ID penv;
  /* 34 */ union ID venv;
  /* 38 */ char *wave;
};

#define WAVE_SIZE 37

struct wave {
  /* 00 */ union ID id;
  /* 04 */ unsigned int size;
  /* 08 */ unsigned int start;
  /* 12 */ unsigned int loopstart;
  /* 16 */ unsigned int loopend;
  /* 20 */ unsigned int m_start;
  /* 24 */ unsigned int sample_ratio;
  /* 28 */ char attenuation;
  /* 29 */ char low_note;
  /* 30 */ char high_note;
  /* 31 */ unsigned char format;
  /* 32 */ unsigned char m_format;
  /* 33 */ union ID data_id;
};

#define ENVELOPE_SIZE 8

struct envelope {
  /* 00 */ union ID id;
  /* 04 */ char num_envelopes;
  /* 05 */ char flags;
  /* 06 */ char mode;
  /* 07 */ char index_type;
};

#define ENVELOPE_RECORD_SIZE 12

struct envelope_record {
  /* 00 */ short nattack;
  /* 02 */ short nrelease;
  /* 04 */ unsigned short sustain_offset;
  /* 06 */ unsigned short sustain_rate;
  /* 08 */ unsigned short release_rate;
  /* 10 */ unsigned char hirange;
  /* 11 */ unsigned char pad;
};

#define ROM_HDR_SIZE 512

struct rom_hdr {
  /* 000 */ char iwave[ 8 ];
  /* 008 */ unsigned char rom_hdr_revision;
  /* 009 */ unsigned char series_number;
  /* 010 */ char series_name[ 16 ];
  /* 026 */ char date[ 10 ];
  /* 036 */ unsigned short vendor_revision_major;
  /* 038 */ unsigned short vendor_revision_minor;
  /* 040 */ unsigned int rom_size;
  /* 044 */ char copyright[ 128 ];
  /* 172 */ char vendor_name[ 64 ];
  /* 236 */ char rom_description[ 128 ];
  /* 364 */ char pad[ 147 ];
  /* 511 */ unsigned char csum;
};

/*
 *  variables
 */

static union ID file_header = 		{ { 'F', 'F', 'F', 'F' } };
static union ID patch_header = 		{ { 'P', 'T', 'C', 'H' } };
static union ID program_header = 	{ { 'P', 'R', 'O', 'G' } };
static union ID layer_header = 		{ { 'L', 'A', 'Y', 'R' } };
static union ID wave_header = 		{ { 'W', 'A', 'V', 'E' } };
static union ID envp_header = 		{ { 'E', 'N', 'V', 'P' } };
#if 0
static union ID text_header = 		{ { 'T', 'E', 'X', 'T' } };
static union ID data_header = 		{ { 'D', 'A', 'T', 'A' } };
static union ID copyright_header = 	{ { 'C', 'P', 'R', 'T' } };
#endif

static struct iw_file {
  unsigned int handle;
  unsigned char *data;
  unsigned int size;
  unsigned int start_addr;
  unsigned int start_addr_temp;
  char *fff_filename;
  char *dat_filename;
  struct iw_file *next;
} *iw_files = NULL;

static unsigned int share_id2;

/*
 *  local functions
 */

int gus_instr_ffff_access_synth( void )
{
  return 0;
}

int gus_instr_ffff_access_daemon( void )
{
  return 1;
}

int gus_instr_ffff_access_midi( int midi_device )
{
  return 0x100 | midi_device;
}

static unsigned int rom_pos;

static int iw_rom_seek( int fd, int access, unsigned int address )
{
  return rom_pos = address;
}

static unsigned int iw_rom_tell( int fd, int access )
{
  return rom_pos;
}

static int iw_rom_read( int fd, int access, char *buffer, int count )
{
  struct GUS_STRU_MEMORY_DUMP dump;

  dump.flags = GUS_MEMORY_DUMP_ROM;
  dump.address = rom_pos;
  dump.size = count;
  dump.dump = buffer;
  if ( access & 0x100 )		/* midi */
    {
      struct GUS_STRU_MIDI_MEMORY_DUMP mdump;
      
      mdump.device = (unsigned char)access;
      mdump.dump = &dump;
      if ( ioctl( fd, GUS_MIDI_MEMORY_DUMP, &mdump ) < 0 ) return -1;
    }
   else
    if ( ioctl( fd, access ? GUS_IOCTL_DMN_MEMORY_DUMP : GUS_IOCTL_MEMORY_DUMP, &dump ) < 0 ) return -1;
  rom_pos += count;
  return count;
}

int gus_instr_ffff_get_rom_header( int fd, int access, int bank, gus_rom_interwave_header_t *header )
{
  gus_info_t info;
  struct rom_hdr hdr;

  if ( access & 0x100 )
    {
      struct GUS_STRU_MIDI_CARD_INFO minfo;
      
      minfo.device = (unsigned char)access;
      minfo.info = &info;
      if ( ioctl( fd, GUS_MIDI_CARD_INFO, &minfo ) < 0 ) return -1;
    }
   else
    if ( ioctl( fd, access ? GUS_IOCTL_DMN_INFO : GUS_IOCTL_INFO, &info ) < 0 ) return -1;
  if ( bank < 0 || bank >= info.memory_rom_banks ) return -1;
  iw_rom_seek( fd, access, bank * 4L * 1024L * 1024L );
  if ( iw_rom_read( fd, access, (char *)&hdr, sizeof( hdr ) ) == sizeof( hdr ) )
    {
      strncpy( header -> iwave, hdr.iwave, sizeof( hdr.iwave ) );
      header -> rom_hdr_revision = gus_get_byte( &hdr.rom_hdr_revision, 0 );
      header -> series_number = gus_get_byte( &hdr.series_number, 0 );
      strncpy( header -> series_name, hdr.series_name, sizeof( hdr.series_name ) );
      strncpy( header -> date, hdr.date, sizeof( hdr.date ) );
      header -> vendor_revision_major = gus_get_word( (char *)&hdr.vendor_revision_major, 0 );
      header -> vendor_revision_minor = gus_get_word( (char *)&hdr.vendor_revision_minor, 0 );
      strncpy( header -> copyright, hdr.copyright, sizeof( hdr.copyright ) );
      strncpy( header -> vendor_name, hdr.vendor_name, sizeof( hdr.vendor_name ) );
      strncpy( header -> rom_description, hdr.rom_description, sizeof( hdr.rom_description ) );
      return 0;
    }
  return -1;
}

int gus_instr_ffff_open( char *fff_filename, char *dat_filename )
{
  static int handle_number = 0;
  struct iw_file *file, *pfile;
  struct stat info;
  struct header ffff;
  int fd;

  if ( stat( fff_filename, &info ) < 0 ) return -1;
  if ( stat( dat_filename, &info ) < 0 ) return -1;
  for ( pfile = iw_files; pfile && pfile -> next; pfile = pfile -> next );
  /* ok.. at first - look for FFFF ROM */
  if ( ( fd = open( fff_filename, O_RDONLY ) ) < 0 )
    {
      gus_dprintf( "InterWave FILE: file '%s' not found\n", fff_filename );
      return -1;
    }
  if ( read( fd, &ffff, sizeof( ffff ) ) != sizeof( ffff ) )
    {
      gus_dprintf( "InterWave FILE: file '%s' - read error\n", fff_filename );
      close( fd );
      return -1;
    }
  file = malloc( sizeof( struct iw_file ) );
  if ( !file )
    {
      close( fd );
      return -1;
    }
  memset( file, 0, sizeof( *file ) );
  ++handle_number; handle_number &= 0xffff;
  file -> handle = handle_number;
  file -> size = gus_get_dword( (char *)&ffff.length, 0 );
  file -> fff_filename = strdup( fff_filename );
  file -> dat_filename = strdup( dat_filename );
  file -> data = malloc( file -> size );
  if ( !file -> data )
    {
      free( file );
      close( fd );
      return -1;
    }
  if ( read( fd, file -> data, file -> size ) != file -> size )
    {
      gus_dprintf( "InterWave FILE: file '%s' - read error\n", fff_filename );
      free( file -> data );
      free( file );
      close( fd );
      return -1;
    }
  file -> start_addr = ~0;
  file -> start_addr_temp = ~0;
  file -> next = NULL;
  if ( !iw_files )
    iw_files = file;
   else
    pfile -> next = file;
  return file -> handle;
}

int gus_instr_ffff_open_rom( int fd, int access, int bank, int file )
{
  gus_info_t info;
  unsigned int next_ffff;
  struct header ffff;
  struct iw_file *xfile, *pfile;
  int index;

  index = 0;
  if ( bank > 255 || file > 255 ) return -1; 
  for ( pfile = iw_files; pfile && pfile -> next; pfile = pfile -> next );
  if ( access & 0x100 )
    {
      struct GUS_STRU_MIDI_CARD_INFO minfo;
      
      minfo.device = (unsigned char)access;
      minfo.info = &info;
      if ( ioctl( fd, GUS_MIDI_CARD_INFO, &minfo ) < 0 ) return -1;
    }
   else
    if ( ioctl( fd, access ? GUS_IOCTL_DMN_INFO : GUS_IOCTL_INFO, &info ) < 0 ) return -1;
  if ( info.version != 0x100 ) return -1;		/* not InterWave board */
  if ( !info.memory_rom_size || !info.memory_rom_banks )
    {
      gus_dprintf( "InterWave ROM: init - InterWave board without ROM?\n" );
      return -1;
    }
  if ( bank < 0 || bank >= info.memory_rom_banks )
    {
      gus_dprintf( "InterWave ROM: bank out of range\n" );
      return -1;
    }
  /* ok.. maybe some checking will be here, but this is already done by driver */
  /* here is real start of ROM */
  iw_rom_seek( fd, access, bank * ( 4L * 1024L * 1024L ) + ROM_HDR_SIZE );
  while ( iw_rom_read( fd, access, (char *)&ffff, sizeof( ffff ) ) == sizeof( ffff ) )
    {
      if ( ffff.id.v != file_header.v ) break;
      ffff.length = gus_get_dword( (char *)&ffff.length, 0 );
      next_ffff = iw_rom_tell( fd, access ) + ffff.length;
      if ( file == index )
        {
#ifdef IW_ROM_DEBUG
          fprintf( stderr, "file header at 0x%x size 0x%x\n", rom_pos - sizeof( ffff ), ffff.length );
#endif
          xfile = malloc( sizeof( struct iw_file ) );
          if ( xfile )
            {
              memset( xfile, 0, sizeof( *xfile ) );
              xfile -> data = malloc( ffff.length );
              if ( !xfile -> data )
                {
                  free( xfile );
                  return -1;
                }
              iw_rom_read( fd, access, xfile -> data, ffff.length );
              xfile -> handle = 0x10000 + ( bank << 8 ) + file;
              xfile -> size = ffff.length;
              xfile -> start_addr = bank * 4L * 1024L * 1024L;
              xfile -> start_addr_temp = ~0;
              xfile -> next = NULL;
              if ( !iw_files )
                iw_files = xfile;
               else
                pfile -> next = xfile;
              return xfile -> handle;
            }
          return -1;
        }
      index++;
      iw_rom_seek( fd, access, next_ffff );
    }
  return -1;
}

int gus_instr_ffff_close( int handle )
{
  struct iw_file *file, *ofile;
  
  for ( file = iw_files; file; file = file -> next )
    if ( file -> handle == handle )
      {
        if ( file == iw_files )
          iw_files = iw_files -> next;
        ofile = file;
        file = file -> next;
        free( ofile -> dat_filename );
        free( ofile -> fff_filename );
        free( ofile -> data );
        free( ofile );
        return 0;
      }
  return -1;
}

int gus_instr_ffff_download( int handle, int fd, int access, int temporary )
{
  struct GUS_STRU_MEMORY_BLOCK block;
  off_t size;
  int fd_read;
  struct iw_file *iw_file;
  int error = 0;
  
  for ( iw_file = iw_files; iw_file; iw_file = iw_file -> next )
    if ( iw_file -> handle == handle ) break;
  if ( !iw_file ) return -1;
  if ( iw_file -> handle >= 0x10000 ) return -1;
  if ( temporary )
    {
      if ( iw_file -> start_addr != ~0 )			/* fixed */
        gus_instr_ffff_download_free( handle, fd, access, 0 );
       else
        {
          if ( iw_file -> start_addr_temp != ~0 ) return 0;	/* already */
        }
    }
   else
    {
      if ( iw_file -> start_addr_temp != ~0 )
        gus_instr_ffff_download_free( handle, fd, access, 1 );
       else
        {
          if ( iw_file -> start_addr != ~0 ) return 0;		/* already */
        }
    }
  if ( ( fd_read = open( iw_file -> dat_filename, O_RDONLY ) ) < 0 )
    {
      gus_dprintf( "InterWave FFFF (download): open error - '%s'\n", iw_file -> dat_filename );
      return -1;
    }
  size = lseek( fd_read, 0, SEEK_END );
  lseek( fd_read, 0, SEEK_SET );
  block.flags = temporary ? 0 : GUS_MEMORY_BLOCK_LOCKED;
  block.address = 0;
  block.size = size;
  strncpy( block.id, iw_file -> dat_filename, sizeof( block.id ) );
  if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_LOCK );
  if ( access & 0x100 )		/* midi */
    {
      struct GUS_STRU_MIDI_MEMORY_BLOCK mblock;
      
      mblock.device = (unsigned char)access;
      mblock.block = &block;
      if ( ioctl( fd, GUS_MIDI_MEMORY_BALLOC, &mblock ) < 0 ) error++;
    }
   else
    if ( ioctl( fd, access ? GUS_IOCTL_DMN_MEMORY_BALLOC : GUS_IOCTL_MEMORY_BALLOC, &block ) < 0 ) error++;
  if ( error )
    {
      gus_dprintf( "InterWave FFFF (download): GUS memory block alloc error - '%s'\n", strerror( errno ) );
      if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
      close( fd_read );
      return -1;
    }
  if ( !(block.flags & GUS_MEMORY_BLOCK_ALREADY) )
    {
      unsigned char *buffer = NULL;
      unsigned int bsize = ( 64 * 1024 ) * 2;
      struct GUS_STRU_MEMORY_DUMP dump;
                  
      do {
        bsize >>= 1;
        buffer = malloc( bsize );
      } while ( bsize > 0 && !buffer );
      if ( !buffer )
        {
          gus_dprintf( "InterWave FFFF (download): malloc problem\n" );
          if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
          close( fd_read );
          return -1;
        }
      dump.flags = GUS_MEMORY_DUMP_RAM | GUS_MEMORY_DUMP_WRITE;
      dump.dump = buffer;
      while ( size > 0 )
        {
          if ( bsize > size ) bsize = size;
          dump.address = block.address + lseek( fd_read, 0, SEEK_CUR );
          dump.size = bsize;
          if ( read( fd_read, buffer, bsize ) != bsize )
            {
              gus_dprintf( "InterWave FFFF (download): read error - '%s'\n", strerror( errno ) );
              goto __free1;
            }
          if ( access & 0x100 )
            {
              struct GUS_STRU_MIDI_MEMORY_DUMP mdump;
            
              mdump.device = (unsigned char)access;
              mdump.dump = &dump;
              if ( ioctl( fd, GUS_MIDI_MEMORY_DUMP, &mdump ) < 0 ) error++;
            }
           else
            if ( ioctl( fd, access ? GUS_IOCTL_DMN_MEMORY_DUMP : GUS_IOCTL_MEMORY_DUMP, &dump ) < 0 ) error++;
          if ( error )
            {
              gus_dprintf( "InterWave FFFF (download): dump error - '%s'\n", strerror( errno ) );
              __free1:
              free( buffer );
              if ( access & 0x100 )
                {
                  struct GUS_STRU_MIDI_MEMORY_BLOCK mblock;
                  
                  mblock.device = (unsigned char)access;
                  mblock.block = &block;
                  ioctl( fd, GUS_MIDI_MEMORY_BFREE, &block );
                }
               else
                ioctl( fd, access ? GUS_IOCTL_DMN_MEMORY_BFREE : GUS_IOCTL_MEMORY_BFREE, &block );
              if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
              close( fd_read );
              return -1;
            }
          size -= bsize;
        }
      free( buffer );
    }
  if ( temporary )
    {
      iw_file -> start_addr_temp = block.address;
      iw_file -> start_addr = ~0;
    }
   else
    {
      iw_file -> start_addr = block.address;
      iw_file -> start_addr_temp = ~0;
    }
  if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
  close( fd_read );
  return 0;
}

int gus_instr_ffff_download_free( int handle, int fd, int access, int temporary )
{
  struct GUS_STRU_MEMORY_BLOCK block;
  off_t size;
  int fd_read;
  struct iw_file *iw_file;
  int error = 0;
  
  for ( iw_file = iw_files; iw_file; iw_file = iw_file -> next )
    if ( iw_file -> handle == handle ) break;
  if ( !iw_file ) return -1;
  if ( iw_file -> handle >= 0x10000 ) return -1;
  if ( temporary && iw_file -> start_addr_temp == ~0 ) return -1;
  if ( !temporary && iw_file -> start_addr == ~0 ) return -1;
  if ( ( fd_read = open( iw_file -> dat_filename, O_RDONLY ) ) < 0 )
    {
      gus_dprintf( "InterWave FFFF (download_free): open error - '%s'\n", iw_file -> dat_filename );
      return -1;
    }
  size = lseek( fd_read, 0, SEEK_END );
  lseek( fd_read, 0, SEEK_SET );
  block.flags = temporary ? 0 : GUS_MEMORY_BLOCK_LOCKED;
  block.address = temporary ? iw_file -> start_addr_temp : iw_file -> start_addr;
  block.size = size;
  strncpy( block.id, iw_file -> dat_filename, sizeof( block.id ) );
  if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_LOCK );
#if 0
  printf( "ffff: block.address = 0x%x, block.size = 0x%x, access = 0x%x\n", block.address, block.size, access );
#endif
  if ( access & 0x100 )		/* midi */
    {
      struct GUS_STRU_MIDI_MEMORY_BLOCK mblock;
      
      mblock.device = (unsigned char)access;
      mblock.block = &block;
      if ( ioctl( fd, GUS_MIDI_MEMORY_BFREE, &mblock ) < 0 ) error++;
    }
   else
    if ( ioctl( fd, access ? GUS_IOCTL_DMN_MEMORY_BFREE : GUS_IOCTL_MEMORY_BFREE, &block ) < 0 ) error++;
  if ( error )
    {
      if ( errno != ENOENT )
        gus_dprintf( "InterWave FFFF (download_free): GUS memory block free error - '%s'\n", strerror( errno ) );
      if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
      close( fd_read );
      return -1;
    }
  if ( access == 1 ) ioctl( fd, GUS_IOCTL_DMN_UNLOCK );
  close( fd_read );
  iw_file -> start_addr_temp =
  iw_file -> start_addr = ~0;
  return 0;
}

static char *look_for_id( struct iw_file *file, char *start, char *end, union ID *id )
{
  if ( !start ) return NULL;
  while ( (long)start < (long)end )
    {
      if ( ((struct header *)start) -> id.v == id -> v ) return start;
      start += sizeof( struct header ) + gus_get_dword( (char *)&((struct header *)start) -> length, 0 );
    }
  return NULL;
}

static void copy_modulation( struct GUS_STRU_IW_LFO *glfo, unsigned char *buffer )
{
  glfo -> freq = gus_get_word( buffer, 0 );
  glfo -> depth = gus_get_word( buffer, 2 );
  glfo -> sweep = gus_get_word( buffer, 4 );
  glfo -> shape = gus_get_byte( buffer, 6 );
  glfo -> delay = gus_get_byte( buffer, 7 );
}

static void copy_envelope( struct iw_file *file, struct GUS_STRU_IW_ENV *genv, union ID *eid )
{
  int idx;
  unsigned char *ptr, *end;
  unsigned char *envelope, *record;
  struct GUS_STRU_IW_ENV_RECORD *grecord, *precord;
  
  if ( !eid -> v ) return;
  ptr = file -> data;
  end = file -> data + file -> size;
  while ( 1 )
    {
      ptr = look_for_id( file, ptr, end, &envp_header );
      if ( !ptr ) break;
      envelope = ptr + sizeof( struct header );
      if ( ((union ID *)envelope) -> v == eid -> v )
        {
          genv -> flags = gus_get_byte( envelope, 5 );
          genv -> mode = gus_get_byte( envelope, 6 );
          genv -> index = gus_get_byte( envelope, 7 );
          genv -> record = precord = NULL;
          record = envelope + ENVELOPE_SIZE;
          for ( idx = 0; idx < gus_get_byte( envelope, 4 ); idx++ )
            {
              grecord = calloc( 1, sizeof( *grecord ) );
              if ( !grecord ) return;
              grecord -> nattack = gus_get_word( record, 0 );
              grecord -> nrelease = gus_get_word( record, 2 );
              grecord -> sustain_offset = gus_get_word( record, 4 );
              grecord -> sustain_rate = gus_get_word( record, 6 );
              grecord -> release_rate = gus_get_word( record, 8 );
              grecord -> hirange = gus_get_byte( record, 10 );
              grecord -> points = calloc( 1, sizeof( short ) * 2 * ( grecord -> nattack + grecord -> nrelease ) );
              if ( grecord -> points )
                {
                  int idx;
                  short *points;
                  
                  points = (short *)(record + ENVELOPE_RECORD_SIZE);
                  for ( idx = 0; idx < grecord -> nattack + grecord -> nrelease; idx++ )
                    {
                      grecord -> points[ idx ].offset = *points++;
                      grecord -> points[ idx ].rate = *points++;
                    }
                }
              grecord -> next = NULL;
              if ( !genv -> record )
                genv -> record = grecord;
               else
                precord -> next = grecord;
              precord = grecord;
              record += ENVELOPE_RECORD_SIZE + 2 * sizeof( short ) * ( grecord -> nattack + grecord -> nrelease );
            }
          return;
        }
      ptr += sizeof( struct header ) + gus_get_dword( (char *)&((struct header *)ptr) -> length, 0 );
    }
  gus_dprintf( "InterWave FFFF: envelope 0x%x wasn't found?\n", eid -> v );
}

static char *load_iw_wave( struct iw_file *file, unsigned int start, unsigned int size )
{
  int fd;
  char *result;
  
  if ( ( fd = open( file -> dat_filename, O_RDONLY ) ) < 0 )
    {
      gus_dprintf( "InterWave FFFF: file '%s' not found\n", file -> dat_filename );
      return NULL;
    }
  if ( lseek( fd, start, SEEK_SET ) < 0 )
    {
      gus_dprintf( "InterWave FFFF: seek error (%s)\n", file -> dat_filename );
      close( fd );
      return NULL;
    }
  result = malloc( size );
  if ( !result )
    {
      gus_dprintf( "InterWave FFFF: malloc problem\n" );
      close( fd );
      return NULL;
    }
  if ( read( fd, result, size ) != size )
    {
      gus_dprintf( "InterWave FFFF: read error (%s)\n", file -> dat_filename );
      free( result );
      close( fd );
      return NULL;
    }
  close( fd );
  return result;
}

static int load_iw_patch( gus_instrument_t *instr, struct iw_file *file, unsigned char *patch, int temp )
{
  unsigned char *layer, *wave;
  gus_layer_t *glayer, *player;
  gus_wave_t *gwave, *pwave;
  int idx_layer, idx_wave;
  char *current;

#ifdef IW_ROM_DEBUG
  gus_dprintf( "load_iw_patch - nlayers = %i\n", gus_get_word( patch, 4 ) );
#endif
  instr -> mode = GUS_INSTR_INTERWAVE;
  instr -> layer = gus_get_byte( patch, 6 );
  instr -> exclusion = gus_get_byte( patch, 7 );
  instr -> exclusion_group = gus_get_word( patch, 8 );
  instr -> data.iw.effect1 = gus_get_byte( patch, 10 );
  instr -> data.iw.effect1_depth = gus_get_byte( patch, 11 );
  instr -> data.iw.effect2 = gus_get_byte( patch, 12 );
  instr -> data.iw.effect2_depth = gus_get_byte( patch, 13 );
  current = (char *)patch + sizeof( struct patch );
  instr -> info.layer = player = NULL;
  for ( idx_layer = 0; idx_layer < gus_get_word( patch, 4 ); idx_layer++ )
    {
      if ( ((struct header *)current) -> id.v != layer_header.v )
        {
          gus_dprintf( "InterWave FFFF: wrong layer signature\n" );
          break;
        }
      layer = current + sizeof( struct header );
      glayer = calloc( 1, sizeof( gus_layer_t ) );
      glayer -> mode = GUS_INSTR_INTERWAVE;
      glayer -> data.iw.flags = gus_get_byte( layer, 5 );
      glayer -> data.iw.high_range = gus_get_byte( layer, 6 );
      glayer -> data.iw.low_range = gus_get_byte( layer, 7 );
      glayer -> data.iw.pan = gus_get_byte( layer, 8 );
      glayer -> data.iw.pan_freq_scale = gus_get_byte( layer, 9 );
      copy_modulation( &glayer -> data.iw.tremolo, layer + 10 );
      copy_modulation( &glayer -> data.iw.vibrato, layer + 18 );
      glayer -> data.iw.velocity_mode = gus_get_byte( layer, 26 );
      glayer -> data.iw.attenuation = gus_get_byte( layer, 27 );
      glayer -> data.iw.freq_scale = gus_get_word( layer, 28 );
      glayer -> data.iw.freq_center = gus_get_byte( layer, 30 );
      glayer -> data.iw.layer_event = gus_get_byte( layer, 31 );
      copy_envelope( file, &glayer -> data.iw.penv, (union ID *)(layer + 32) );
      copy_envelope( file, &glayer -> data.iw.venv, (union ID *)(layer + 36) );
      current += sizeof( struct header ) + gus_get_dword( (char *)&((struct header *)current) -> length, 0 );
      glayer -> wave = pwave = NULL;
      for ( idx_wave = 0; idx_wave < gus_get_byte( layer, 4 ); idx_wave++ )
        {
          unsigned char format;
          int w_16;          
        
          if ( ((struct header *)current) -> id.v != wave_header.v )
            {
              gus_dprintf( "InterWave FFFF: wrong wave signature\n" );
              break;
            }
          wave = current + sizeof( struct header );
          format = gus_get_byte( wave, 31 );
          w_16 = format & IW_FORMAT_8BIT ? 1 : 2; 
          gwave = calloc( 1, sizeof( gus_wave_t ) );
          gwave -> mode = GUS_INSTR_INTERWAVE;
          gwave -> size = gus_get_dword( wave, 4 ) * w_16;
          gwave -> start = 0;
          if ( file -> handle >= 0x10000 )
            gwave -> begin.memory = ( gus_get_dword( wave, 8 ) + file -> start_addr ) << 4;
          gwave -> loop_start = gus_get_dword( wave, 12 ) * w_16;
          gwave -> loop_end = gus_get_dword( wave, 16 ) * w_16;
          gwave -> data.iw.sample_ratio = gus_get_dword( wave, 24 );
          gwave -> data.iw.attenuation = gus_get_byte( wave, 28 );
          gwave -> data.iw.low_note = gus_get_byte( wave, 29 );
          gwave -> data.iw.high_note = gus_get_byte( wave, 30 );
          if ( !(format & IW_FORMAT_8BIT) ) gwave -> format |= GUS_WAVE_16BIT;
          if ( !(format & IW_FORMAT_SIGNED) ) gwave -> format |= GUS_WAVE_UNSIGNED;
          if ( !(format & IW_FORMAT_FORWARD) ) gwave -> format |= GUS_WAVE_BACKWARD;
          if ( format & IW_FORMAT_LOOP ) gwave -> format |= GUS_WAVE_LOOP;
          if ( format & IW_FORMAT_BIDIR ) gwave -> format |= GUS_WAVE_BIDIR;
          if ( format & IW_FORMAT_ULAW ) gwave -> format |= GUS_WAVE_ULAW;
          if ( file -> handle >= 0x10000 )
            {
              if ( !(format & IW_FORMAT_ROM) )
                gus_dprintf( "InterWave FFFF: wrong wave ROM format?\n" );
              gwave -> format |= GUS_WAVE_ROM;
            }
           else
            {
              if ( file -> start_addr == ~0 && ( !temp || file -> start_addr_temp == ~0 ) )
                {
                  gwave -> share_id1 = share_id2;
                  gwave -> share_id2 = gus_get_dword( wave, 8 ) | 0x80000000;
                  gwave -> begin.ptr = load_iw_wave( file, gus_get_dword( wave, 8 ), gwave -> size );
                  if ( !gwave -> begin.ptr )
                    {
                      free( gwave );
                      goto __skip_wave;
                    }
                }
               else
                {
                  gwave -> format |= GUS_WAVE_RAM;
                  if ( file -> start_addr == ~0 )
                    gwave -> begin.memory = file -> start_addr_temp;
                   else
                    gwave -> begin.memory = file -> start_addr;
                  gwave -> begin.memory += gus_get_dword( wave, 8 );
                  gwave -> begin.memory <<= 4;
                }
            }
          if ( !glayer -> wave )
            glayer -> wave = gwave;
           else
            pwave -> next = gwave;
          pwave = gwave;
          __skip_wave:
          current += sizeof( struct header ) + gus_get_dword( (char *)&((struct header *)current) -> length, 0 );
        }
      if ( !instr -> info.layer )
        instr -> info.layer = glayer;
       else
        player -> next = glayer;
      player = glayer;
    }
  return 0;
}

int gus_instr_ffff_load( gus_instrument_t *instr, int handle, int bank, int prog )
{
  struct iw_file *pfile;
  char *ptr, *end;
  unsigned char *program, *patch;
  struct header *header;

#if 0
  printf( "handle = %i, bank = %i, prog = %i\n", handle, bank, prog );
#endif
  for ( pfile = iw_files; pfile; pfile = pfile -> next )
    if ( pfile -> handle == handle ) break;
  if ( !pfile )
    {
      gus_dprintf( "InterWave FFFF: handle %i not found\n", handle );
      return -1; 
    }
  ptr = pfile -> data;
  end = pfile -> data + pfile -> size;
  share_id2 = handle;
  while ( 1 )
    {
      ptr = look_for_id( pfile, ptr, end, &program_header );
      if ( !ptr ) break;
      program = ptr + sizeof( struct header );
      if ( gus_get_word( program, 4 ) != IW_ROM_PROGRAM_VERSION_MAJOR ||
           gus_get_word( program, 6 ) != IW_ROM_PROGRAM_VERSION_MINOR )
        {
          gus_dprintf( "InterWave FFFF: unknown version of program\n" );
          return -1;
        }
      header = (struct header *)(ptr + sizeof( struct header ) + sizeof( struct program ));
      if ( header -> id.v == patch_header.v )
        {
          patch = ptr + sizeof( struct header ) + sizeof( struct program ) + sizeof( struct header );
          if ( gus_get_byte( patch, 14 ) == ( bank & 0xff ) && gus_get_byte( patch, 15 ) == ( prog & 0xff ) )
            return load_iw_patch( instr, pfile, patch, prog & 0x100 ? 1 : 0 );
        }
      ptr += sizeof( struct header ) + gus_get_dword( (char *)&((struct header *)ptr) -> length, 0 );
    }
  return -1;
}
