%{

/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/stat.h>

#include "icfg.h"

#define MAX_CONDITION			32

#define DEF_FLAGS_IW_ROM_PROG		0x00000001
#define DEF_FLAGS_IW_ROM_BANK		0x00000002
#define DEF_FLAGS_IW_FILE_PROG		0x00000004
#define DEF_FLAGS_IW_FILE_BANK		0x00000008

#define yyerror				gus_icfg_error

#if 0
#define DEBUG_PARSER
#endif
#ifdef DEBUG_PARSER
#define dprintf(args...)	printf(## args )
#else
#define dprintf(args...)	/* nothing */
#endif

	/* insgus_lexer.c */

int yylex( void );

extern int line_count;
extern FILE *yyin;
	
	/* local variables */

static int condition_true, condition_false, condition_stack_ptr;
static int condition_stack[ MAX_CONDITION ];
static struct iwfile *iw_ffff = NULL;
static struct bank *p_bank = NULL;
static struct instrument *p_prog = NULL;
static unsigned int bank_prog_flags = 0;
static int bank_prog_defaults_flag = 0;
static struct instrument *bank_prog_defaults = NULL;
static unsigned int bank_prog_defaults_flags = 0;
static unsigned char number_bitmap[ 256 / 8 ];
static struct preload *preload_bank = NULL;
static struct preload_format *p_format = NULL;
static int bank_prog_format_alias_type = 0;
static int preload_download_bank_number = -1;
static int preload_aliases_bank_number = -1;
static int preload_aliases_dest_bank_number = -1;
static int preload_aliases_dest_prog_number = -1;

	/* local functions */

static void condition_go( int true );
static void condition_end( void );

static int file_exist( char *file );
static int patch_exist( char *patch );
static int rom_exist( int bank, char *name );

static void add_gf1_patch_path( char *path );

static void iw_ffff_file_start( void );
static void iw_ffff_file_stop( void );
static void iw_ffff_file_number( int value );
static void iw_ffff_file_name( char *name );
static void iw_ffff_file_data( char *data );
static void iw_ffff_file_download( void );
static void iw_ffff_file_rom_name( char *name );
static void iw_ffff_file_rom_bank( int value );
static void iw_ffff_file_rom_file( int value );

static void bank_start( void );
static void bank_stop( void );
static void bank_number( int value );
static void bank_name( char *name );
static void bank_copy_from_bank( int value );
static void bank_defaults_start( void );
static void bank_defaults_stop( void );
static void bank_defaults_preffered_start( void );
static void bank_defaults_preffered_stop( void );
static void bank_defaults_preffered_instrument( int type );
static void bank_prog_start( void );
static void bank_prog_stop( void );
static void bank_prog_number( int value );
static void bank_prog_number_range( int value1, int value2 );
static void bank_prog_text( char *text );
static void bank_prog_patch_start( void );
static void bank_prog_patch_stop( void );
static void bank_prog_patch_file( char *name );
static void bank_prog_patch_group( int value );
static void bank_prog_patch_exclusion( int value );
static void bank_prog_patch_8bit( void );
static void bank_prog_patch_disable( void );
static void bank_prog_iwrom_start( void );
static void bank_prog_iwrom_stop( void );
static void bank_prog_iwrom_file( int value );
static void bank_prog_iwrom_bank( int value );
static void bank_prog_iwrom_prog( int value );
static void bank_prog_iwrom_disable( void );
static void bank_prog_iwfile_start( void );
static void bank_prog_iwfile_stop( void );
static void bank_prog_iwfile_file( int value );
static void bank_prog_iwfile_bank( int value );
static void bank_prog_iwfile_prog( int value );
static void bank_prog_iwfile_disable( void );
static void bank_prog_alias_bank( int bank );
static void bank_prog_alias_prog( int prog );
static void bank_prog_format_alias_start( int format );
static void bank_prog_format_alias_bank( int bank );
static void bank_prog_format_alias_prog( int prog );

static void preload_start( void );
static void preload_stop( void );
static void preload_name( char *name );
static void preload_format_start( int type );
static void preload_format_stop( void );
static void preload_patch_8bit( void );
static void preload_ffff_whole( void );
static void preload_ffff_file( int value );
static void preload_download_start( void );
static void preload_download_stop( void );
static void preload_download_bank( int value );
static void preload_download_prog( int value );
static void preload_download_prog_range( int value1, int value2 );
static void preload_aliases_start( void );
static void preload_aliases_stop( void );
static void preload_aliases_dest_bank( int value );
static void preload_aliases_dest_prog( int value );
static void preload_aliases_bank( int value );
static void preload_aliases_prog( int value );
static void preload_aliases_prog_range( int value1, int value2 );

%}

%start lines

%union {
    int i_value;
    char *s_value;
  };

%token <i_value> L_INTEGER L_TRUE L_FALSE
%token <s_value> L_STRING L_ETC_DIR L_LIB_DIR

%token L_LE L_LEQ L_GT L_GTE L_EQ L_NEQ L_OR L_AND

%token L_IF L_IFE L_ELSE
%token L_DEFINED L_DEFINE L_UNDEF

%token L_RANGE L_FILE_EXIST L_PATCH_EXIST L_ROM_EXIST
%token L_ETC_DIR L_LIB_DIR
%token L_GF1_PATCH_PATH L_IW_FFFF_FILE L_BANK L_PRELOAD L_DOWNLOAD L_SYSLOG L_ABORT
%token L_ROM_NAME L_ROM_BANK L_ROM_FILE L_FFFF_NAME L_DATA_NAME
%token L_NUMBER L_NAME L_DEFAULTS L_PROG L_COPY_FROM_BANK
%token L_PREFFERED
%token L_NUMBERS L_TEXT L_PATCH L_IWROM L_IWFILE
%token L_FILE L_GROUP L_EXCLUSION L_DISABLE
%token L_WIDE_8BIT L_WHOLE L_DEST_BANK L_DEST_PROG L_ALIAS L_PROGS L_VALUE

%type <i_value> bool integer
%type <s_value> string

%left L_LE L_NEQ L_GT L_GTQ
%left L_OR L_AND
%left '-' '+'
%left '*' '/'
%left '&' '|'
%left NEG
%left L_RSHIFT L_LSHIFT

%%

lines	: line
	| lines line
	;

	/*
         *  main
	 */

line	: L_GF1_PATCH_PATH string { add_gf1_patch_path( $2 ); }
	| L_IW_FFFF_FILE
		{ iw_ffff_file_start(); }
		'{' iw_ffff_flags '}'
		{ iw_ffff_file_stop(); }
	| L_BANK
		{ bank_start(); }
		'{' bank_flags '}'
		{ bank_stop(); }
	| L_PRELOAD
		{ preload_start(); }
		'{' preload_flags '}'
		{ preload_stop(); }
	| L_DEFINE string	{ if ( condition_true ) gus_icfg_lexer_defines_add( $2 ); free( $2 ); }
	| L_UNDEF string	{ if ( condition_true ) gus_icfg_lexer_defines_remove( $2 ); free( $2 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' lines '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' lines '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' lines '}' { condition_end(); }
	| L_SYSLOG string	{ if ( condition_true ) gus_icfg_warning( $2 ); free( $2 ); }
	| L_ABORT		{ if ( condition_true ) { gus_icfg_error( "Usert abort..." ); exit( 0 ); } }
	| error			{ yyerror( "unknown keyword in main section" ); }
	;

	/*
         *  iw_ffff_file
	 */

iw_ffff_flags
	: iw_ffff_flag
	| iw_ffff_flags iw_ffff_flag
	;

iw_ffff_flag
	: L_NUMBER integer 	{ iw_ffff_file_number( $2 ); }
	| L_FFFF_NAME string 	{ iw_ffff_file_name( $2 ); }
	| L_DATA_NAME string	{ iw_ffff_file_data( $2 ); }
	| L_DOWNLOAD 		{ iw_ffff_file_download(); }
	| L_ROM_NAME string	{ iw_ffff_file_rom_name( $2 ); }
	| L_ROM_BANK integer	{ iw_ffff_file_rom_bank( $2 ); }
	| L_ROM_FILE integer	{ iw_ffff_file_rom_file( $2 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' iw_ffff_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' iw_ffff_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' iw_ffff_flags '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in iw_ffff_file{} section" ); }
	;

	/*
         *  bank
	 */

bank_flags
	: bank_flag
	| bank_flags bank_flag
	;

bank_flag
	: L_NUMBER integer	{ bank_number( $2 ); }
	| L_NAME string 	{ bank_name( $2 ); }
	| L_DEFAULTS
		{ bank_defaults_start(); }
		'{' bank_defaults_flags '}'
		{ bank_defaults_stop(); }
	| L_PROG
		{ bank_prog_start(); }
		'{' bank_prog_flags '}'
		{ bank_prog_stop(); }
	| L_COPY_FROM_BANK integer { bank_copy_from_bank( $2 ); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' bank_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' bank_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' bank_flags '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in bank{} section" ); }
	;


bank_defaults_flags
	: bank_defaults_flag
	| bank_defaults_flags bank_defaults_flag
	;

bank_defaults_flag
	: L_PREFFERED
		{ bank_defaults_preffered_start(); }
		'{' bank_defaults_preffered_flags '}'
		{ bank_defaults_preffered_stop(); }
	| L_PATCH
		{ bank_prog_patch_start(); }
		'{' bank_prog_patch_flags '}'
		{ bank_prog_patch_stop(); }
	| L_IWROM
		{ bank_prog_iwrom_start(); }
		'{' bank_prog_iwrom_flags '}'
		{ bank_prog_iwrom_stop(); }
	| L_IWFILE
		{ bank_prog_iwfile_start(); }
		'{' bank_prog_iwfile_flags '}'
		{ bank_prog_iwfile_stop(); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' bank_defaults_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' bank_defaults_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' bank_defaults_flags '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in defaults{} section" ); }
	;


bank_defaults_preffered_flags
	: bank_defaults_preffered_flag
	| bank_defaults_preffered_flags bank_defaults_preffered_flag
	;

bank_defaults_preffered_flag
	: L_PATCH	{ bank_defaults_preffered_instrument( IT_GF1_PATCH ); }
	| L_IWROM	{ bank_defaults_preffered_instrument( IT_IW_ROM ); }
	| L_IWFILE	{ bank_defaults_preffered_instrument( IT_IW_FILE ); }
	| error		{ yyerror( "uknown instrument type" ); }
	;


bank_prog_flags
	: bank_prog_flag
	| bank_prog_flags bank_prog_flag
	;

bank_prog_flag
	: L_NUMBER integer { bank_prog_number( $2 ); }
	| L_NUMBERS '{' bank_prog_numbers_flags '}'
	| L_TEXT string { bank_prog_text( $2 ); }
	| L_PATCH
		{ bank_prog_patch_start(); }
		'{' bank_prog_patch_flags '}'
		{ bank_prog_patch_stop(); }
	| L_IWROM
		{ bank_prog_iwrom_start(); }
		'{' bank_prog_iwrom_flags '}'
		{ bank_prog_iwrom_stop(); }
	| L_IWFILE
		{ bank_prog_iwfile_start(); }
		'{' bank_prog_iwfile_flags '}'
		{ bank_prog_iwfile_stop(); }
	| L_ALIAS		'{' bank_prog_alias_flags '}'
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' bank_prog_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' bank_prog_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' bank_prog_flags '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in prog{} section" ); }
	;


bank_prog_numbers_flags
	: bank_prog_numbers_flag
	| bank_prog_numbers_flags bank_prog_numbers_flag
	;

bank_prog_numbers_flag
	: L_VALUE integer	{ bank_prog_number( $2 ); }
	| L_RANGE integer '@' integer { bank_prog_number_range( $2, $4 ); }
	| error 		{ yyerror( "bad program number" ); }
	;


bank_prog_patch_flags
	: bank_prog_patch_flag
	| bank_prog_patch_flags bank_prog_patch_flag
	;

bank_prog_patch_flag
	: L_FILE string 	{ bank_prog_patch_file( $2 ); }
	| L_GROUP integer 	{ bank_prog_patch_group( $2 ); }
	| L_EXCLUSION integer 	{ bank_prog_patch_exclusion( $2 ); }
	| L_WIDE_8BIT		{ bank_prog_patch_8bit(); }
	| L_DISABLE 		{ bank_prog_patch_disable(); }
	| L_ALIAS		{ bank_prog_format_alias_start( IT_GF1_PATCH ); }
				'{' bank_prog_format_alias_flags '}'
	| error 		{ yyerror( "unknown keyword in patch{} section" ); }
	;


bank_prog_iwrom_flags
	: bank_prog_iwrom_flag
	| bank_prog_iwrom_flags bank_prog_iwrom_flag
	;

bank_prog_iwrom_flag
	: L_FILE integer 	{ bank_prog_iwrom_file( $2 ); }
	| L_BANK integer 	{ bank_prog_iwrom_bank( $2 ); }
	| L_PROG integer 	{ bank_prog_iwrom_prog( $2 ); }
	| L_DISABLE 		{ bank_prog_iwrom_disable(); }
	| L_ALIAS		{ bank_prog_format_alias_start( IT_IW_ROM ); }
				'{' bank_prog_format_alias_flags '}'
	| error			{ yyerror( "unknown keyword in iwrom{} section" ); }
	;


bank_prog_iwfile_flags
	: bank_prog_iwfile_flag
	| bank_prog_iwfile_flags bank_prog_iwfile_flag
	;

bank_prog_iwfile_flag
	: L_FILE integer	{ bank_prog_iwfile_file( $2 ); }
	| L_BANK integer	{ bank_prog_iwfile_bank( $2 ); }
	| L_PROG integer	{ bank_prog_iwfile_prog( $2 ); }
	| L_DISABLE		{ bank_prog_iwfile_disable(); }
	| L_ALIAS		{ bank_prog_format_alias_start( IT_IW_FILE ); }
				'{' bank_prog_format_alias_flags '}'
	| error			{ yyerror( "unknown keyword in iwfile{} section" ); }
	;

bank_prog_alias_flags
	: bank_prog_alias_flag
	| bank_prog_alias_flags bank_prog_alias_flag
	;

bank_prog_alias_flag
	: L_BANK integer	{ bank_prog_alias_bank( $2 ); }
	| L_PROG integer	{ bank_prog_alias_prog( $2 ); }
	| error			{ yyerror( "unknown keyword in alias{} section" ); }
	;

bank_prog_format_alias_flags
	: bank_prog_format_alias_flag
	| bank_prog_format_alias_flags bank_prog_format_alias_flag
	;

bank_prog_format_alias_flag
	: L_BANK integer	{ bank_prog_format_alias_bank( $2 ); }
	| L_PROG integer	{ bank_prog_format_alias_prog( $2 ); }
	| error			{ yyerror( "unknown keyword in alias{} section" ); }
	;

	/*
	 *  preload
	 */

preload_flags
	: preload_flag
	| preload_flags preload_flag
	;

preload_flag
	: L_NAME string 	{ preload_name( $2 ); }
	| L_PATCH
		{ preload_format_start( IT_GF1_PATCH ); }
		'{' preload_patch_flags '}'
		{ preload_format_stop(); }
	| L_IWROM
		{ preload_format_start( IT_IW_ROM ); }
		'{' preload_ffff_flags '}'
		{ preload_format_stop(); }
	| L_IWFILE
		{ preload_format_start( IT_IW_FILE ); }
		'{' preload_ffff_flags '}'
		{ preload_format_stop(); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' preload_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' preload_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' preload_flags '}' { condition_end(); }
	| error			{ yyerror( "unknown keyword in preload{} section" ); }
	;

preload_patch_flags
	: preload_patch_flag
	| preload_patch_flags preload_patch_flag
	;

preload_patch_flag
	: L_WIDE_8BIT 	 	{ preload_patch_8bit(); }
	| L_DOWNLOAD
		{ preload_download_start(); }
		'{' preload_download_flags '}'
		{ preload_download_stop(); }
	| L_ALIAS
		{ preload_aliases_start(); }
		'{' preload_aliases_flags '}'
		{ preload_aliases_stop(); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' preload_patch_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' preload_patch_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' preload_patch_flags '}' { condition_end(); }
	| error 		{ yyerror( "unknown keyword in patch{} section" ); }	
	;

preload_ffff_flags
	: preload_ffff_flag
	| preload_ffff_flags preload_ffff_flag
	;

preload_ffff_flag
	: L_WHOLE 	 	{ preload_ffff_whole(); }
	| L_FILE integer	{ preload_ffff_file( $2 ); }
	| L_DOWNLOAD
		{ preload_download_start(); }
		'{' preload_download_flags '}'
		{ preload_download_stop(); }
	| L_ALIAS
		{ preload_aliases_start(); }
		'{' preload_aliases_flags '}'
		{ preload_aliases_stop(); }
	| L_IF '(' bool ')' 	{ condition_go( $3 ); } '{' preload_ffff_flags '}' { condition_end(); }
	| L_IFE '(' bool ')' 	{ condition_go( $3 ); } '{' preload_ffff_flags '}' { condition_end(); }
				L_ELSE
			     	{ condition_go( !($3) ); } '{' preload_ffff_flags '}' { condition_end(); }
	| error 		{ yyerror( "unknown keyword in iwrom{} section" ); }	
	;


preload_download_flags
	: preload_download_flag
	| preload_download_flags preload_download_flag
	;

preload_download_flag
	: L_BANK integer	{ preload_download_bank( $2 ); }
	| L_PROG integer	{ preload_download_prog( $2 ); }
	| L_PROGS		'{' preload_download_progs_flags '}'
	| error 		{ yyerror( "unknown keyword in download{} section" ); }	
	;

preload_download_progs_flags
	: preload_download_progs_flag
	| preload_download_progs_flags preload_download_progs_flag
	;

preload_download_progs_flag
	: L_VALUE integer	{ preload_download_prog( $2 ); }
	| L_RANGE integer '@' integer { preload_download_prog_range( $2, $4 ); }
	| error			{ yyerror( "unknown keyword in progs{} section" ); }
	;

preload_aliases_flags
	: preload_aliases_flag
	| preload_aliases_flags preload_aliases_flag
	;

preload_aliases_flag
	: L_DEST_BANK integer	{ preload_aliases_dest_bank( $2 ); }
	| L_DEST_PROG integer	{ preload_aliases_dest_prog( $2 ); }
	| L_BANK integer	{ preload_aliases_bank( $2 ); }
	| L_PROG integer	{ preload_aliases_prog( $2 ); }
	| L_PROGS		'{' preload_aliases_progs_flags '}'
	| error 		{ yyerror( "unknown keyword in aliases{} section" ); }	
	;

preload_aliases_progs_flags
	: preload_aliases_progs_flag
	| preload_aliases_progs_flags preload_aliases_progs_flag
	;

preload_aliases_progs_flag
	: L_VALUE integer	{ preload_aliases_prog( $2 ); }
	| L_RANGE integer '@' integer { preload_aliases_prog_range( $2, $4 ); }
	| error			{ yyerror( "unknown keyword in progs{} section" ); }
	;

	/*
         *  types
	 */

bool	: L_TRUE		{ $$ = 1; }
	| L_FALSE		{ $$ = 0; }
	| L_DEFINED string ')'	{ $$ = gus_icfg_lexer_defines_test( $2 ) != 0 ? 1 : 0; free( $2 ); }
	| L_FILE_EXIST string ')' { $$ = file_exist( $2 ) != 0 ? 1 : 0; }
	| L_PATCH_EXIST string ')' { $$ = patch_exist( $2 ) != 0 ? 1 : 0; }
	| L_ROM_EXIST integer ',' string ')' { $$ = rom_exist( $2, $4 ) != 0 ? 1 : 0; }
	| integer L_GT integer	{ $$ = $1 > $3; }
	| integer L_GTQ integer { $$ = $1 >= $3; }
	| integer L_LE integer	{ $$ = $1 < $3; }
	| integer L_LEQ integer	{ $$ = $1 <= $3; }
	| integer L_EQ integer	{ $$ = $1 == $3; }
	| integer L_NEQ integer	{ $$ = $1 != $3; }
	| string L_EQ string	{ $$ = strcmp( $1, $3 ) == 0; free( $1 ); free( $3 ); }
	| string L_NEQ string	{ $$ = strcmp( $1, $3 ) != 0; free( $1 ); free( $3 ); }
	| bool L_OR bool	{ $$ = $1 || $3; }
	| bool L_AND bool	{ $$ = $1 && $3; }
	| '!' bool %prec NEG	{ $$ = $2 != 0 ? 0 : 1; }
	| '(' bool ')'		{ $$ = $2; }
	| error			{ yyerror( "unknown boolean value" ); }
	;

integer : L_INTEGER		{ $$ = $1; }
	| integer '+' integer	{ $$ = $1 + $3; }
	| integer '-' integer	{ $$ = $1 - $3; }
	| integer '*' integer	{ $$ = $1 * $3; }
	| integer '/' integer	{ $$ = $1 / $3; }
	| integer L_RSHIFT integer { $$ = $1 >> $3; }
	| integer L_LSHIFT integer { $$ = $1 << $3; }
	| integer '&' integer	{ $$ = $1 & $3; }
	| integer '|' integer	{ $$ = $1 | $3; }
	| '-' integer %prec NEG { $$ = -$2; }
	| '(' integer ')'	{ $$ = $2; }
	/*
	| error			{ yyerror( "unknown integer value" ); }
	*/
	;

string	: L_ETC_DIR L_STRING ')' { char *x = (char *)malloc( sizeof( GUS_PATH_ETC ) + strlen( $2 ) );
                                  strcpy( x, GUS_PATH_ETC );
                                  strcat( x, $2 );
                                  free( $2 );
                                  $$ = x;
                                }
	| L_LIB_DIR L_STRING ')' { char *x = (char *)malloc( sizeof( GUS_PATH_LIB ) + strlen( $2 ) );
                                  strcpy( x, GUS_PATH_LIB );
                                  strcat( x, $2 );
                                  free( $2 );
                                  $$ = x;
                                }
	| L_STRING		{ $$ = $1; }
	/*
	| error			{ yyerror( "uknown string" ); }
	*/
	;


%%

#if 0

void yyerror( char *string, ... )
{
  char msg[ 512 ];

  va_list vars;
  va_start( vars, string );
  vsprintf( msg, string, vars );
  va_end( vars );
  fprintf( stderr, "gusd: error in configuration file '%s': (line %i) %s\n", config_filename, line_count + 1, msg );
  syslog( LOG_ERR, "error in configuration file '%s': (line %i) %s", config_filename, line_count + 1, msg );
  exit( 1 );
}

#endif

void gus_icfg_parser_init( void )
{
  condition_true = 1;
  condition_false = 0;
  condition_stack_ptr = 0;
  gus_icfg_lexer_init();
}

void gus_icfg_parser_done( void )
{
  gus_icfg_lexer_done();
}

static inline void number_bitmap_clear( void )
{
  memset( number_bitmap, 0, sizeof( number_bitmap ) );
}

static inline void number_bitmap_set( int value )
{
  number_bitmap[ ( value >> 3 ) & 0x1f ] |= 0x80 >> ( value & 0x07 );
}

static inline int number_bitmap_get( int value )
{
  return ( number_bitmap[ ( value >> 3 ) & 0x1f ] & ( 0x80 >> ( value & 0x07 ) ) ) != 0;
}

static void condition_go( int true )
{
  condition_stack[ condition_stack_ptr++ ] = condition_true;
  if ( condition_true )
    {
      condition_true = true ? 1 : 0;
      condition_false = true ? 0 : 1;
    }
}

static void condition_end( void )
{
  if ( !condition_stack_ptr )
    yyerror( "condition stack underflow" );
  condition_true = condition_stack[ --condition_stack_ptr ];
  condition_false = !condition_true;
}

static int file_exist( char *file )
{
  struct stat st;

  if ( condition_true && !stat( file, &st ) )
    {
      free( file );
      return 1;
    }
  free( file );
  return 0;
}

static int patch_exist( char *patch )
{
  if ( condition_true && gus_icfg_look_for_patch_file( patch, ".pat" ) != NULL )
    {
      free( patch );
      return 1;
    }
  free( patch );
  return 0;
}

static int rom_exist( int bank, char *name )
{
  gus_rom_interwave_header_t header;

  if ( condition_true )
    if ( gus_instr_ffff_get_rom_header( gus_icfg_config -> fd, gus_icfg_config -> ffff_access, bank, &header ) >= 0 )
      if ( !strncmp( header.series_name, name, 16 ) )
        {
          free( name );
          return 1;
        }
  free( name );
  return 0;
}

static void add_gf1_patch_path( char *path )
{
  struct gf1path *xpath, *npath;

  if ( condition_false )
    {
      free( path );
      return;
    }
  npath = malloc( sizeof( struct gf1path ) );
  if ( !npath ) yyerror( "alloc error" );
  npath -> path = path;
  npath -> next = NULL;
  for ( xpath = gus_icfg_config -> gf1_paths; xpath && xpath -> next; xpath = xpath -> next );
  if ( !xpath )
    gus_icfg_config -> gf1_paths = npath;
   else
    xpath -> next = npath;
}

static void iw_ffff_file_start( void )
{
  if ( condition_false ) return;
  iw_ffff = (struct iwfile *)malloc( sizeof( struct iwfile ) );
  memset( iw_ffff, 0, sizeof( *iw_ffff ) );
  if ( !iw_ffff ) yyerror( "alloc error" );
}

static void iw_ffff_file_stop( void )
{
  struct iwfile *iw;

  if ( condition_false ) return;
  for ( iw = gus_icfg_config -> iwfiles; iw && iw -> next; iw = iw -> next );
  iw_ffff -> next = NULL;
  if ( !iw )
    gus_icfg_config -> iwfiles = iw_ffff;
   else
    iw -> next = iw_ffff;
  iw_ffff = NULL;
}

static void iw_ffff_file_number( int value )
{
  if ( condition_false ) return;
  iw_ffff -> number = value;
}

static void iw_ffff_file_name( char *name )
{
  if ( condition_false )
    {
      free( name );
      return;
    }
  free( iw_ffff -> spec.file.name );
  iw_ffff -> spec.file.name = name;
}

static void iw_ffff_file_data( char *data )
{
  if ( condition_false )
    {
      free( data );
      return;
    }
  free( iw_ffff -> spec.file.data ); 
  iw_ffff -> spec.file.data = data;
}

static void iw_ffff_file_download( void )
{
  if ( condition_false ) return;
  iw_ffff -> flags |= IWF_DOWNLOAD;
}

static void iw_ffff_file_rom_name( char *name )
{
  if ( condition_false )
    {
      free( name );
      return;
    }
  iw_ffff -> flags |= IWF_ROM;
  free( iw_ffff -> spec.rom.name );
  iw_ffff -> spec.rom.name = name;
}

static void iw_ffff_file_rom_bank( int value )
{
  if ( condition_false ) return;
  iw_ffff -> spec.rom.bank = value;
}

static void iw_ffff_file_rom_file( int value )
{
  if ( condition_false ) return;
  iw_ffff -> spec.rom.file = value;
}

static void bank_start( void )
{
  if ( condition_false ) return;

  dprintf( "bank start\n" );

  p_bank = (struct bank *)malloc( sizeof( struct bank ) );
  if ( !p_bank ) yyerror( "alloc error" );
  memset( p_bank, 0, sizeof( *p_bank ) );

  p_prog = NULL;
  bank_prog_defaults_flag = 0;
  bank_prog_defaults = NULL;
}

static void bank_stop( void )
{
  struct bank *bank;

  if ( condition_false ) return;
  dprintf( "bank stop\n" );
  for ( bank = gus_icfg_config -> banks; bank && bank -> next; bank = bank -> next )
    if ( bank -> number == p_bank -> number ) yyerror( "bank %i already defined", p_bank -> number );
  if ( !bank )
    gus_icfg_config -> banks = p_bank;
   else
    {
      if ( bank -> number == p_bank -> number ) yyerror( "bank %i already defined", p_bank -> number );
      bank -> next = p_bank;
    }
  p_bank = NULL;
  if ( bank_prog_defaults )
    {
      free( bank_prog_defaults -> name );
      free( bank_prog_defaults );
      bank_prog_defaults = NULL;
    }
}

static void bank_number( int value )
{
  if ( condition_false ) return;
  dprintf( "bank number = %i\n", value );
  p_bank -> number = value;
}

static void bank_name( char *name )
{
  if ( condition_false )
    {
      free( name );
      return;
    }
  dprintf( "bank name = '%s'\n", name );
  free( p_bank -> name );
  p_bank -> name = name;
}

static void bank_copy_from_bank( int value )
{
  struct bank *bank;

  if ( condition_false ) return;
  dprintf( "bank_copy_from_bank = %i\n", value );
  for ( bank = gus_icfg_config -> banks; bank; bank = bank -> next )
    if ( bank -> number == value ) break;
  if ( !bank ) yyerror( "parent bank isn't defined" );
  p_bank -> parent = value;
}

static void bank_defaults_start( void )
{
  if ( condition_false ) return;
  dprintf( "bank_defaults_start\n" );
  bank_prog_defaults_flag = 1;
  if ( !bank_prog_defaults )
    {
      bank_prog_defaults = (struct instrument *)calloc( 1, sizeof( struct instrument ) );
      if ( !bank_prog_defaults ) yyerror( "alloc error" );
      memset( bank_prog_defaults, 0, sizeof( bank_prog_defaults ) );
      bank_prog_defaults_flags = 0;
    }
  p_prog = bank_prog_defaults;
}

static void bank_defaults_stop( void )
{
  if ( condition_false ) return;
  dprintf( "bank_defaults_stop\n" );
  bank_prog_defaults_flag = 0;
  p_prog = NULL;
}

static void bank_defaults_preffered_start( void )
{
  int i;  

  if ( condition_false ) return;
  dprintf( "bank_defaults_preffered_start\n" );
  for ( i = 0; i < IT_COUNT; i++ )
    p_bank -> preffered[ i ] = 0;
}

static void bank_defaults_preffered_stop( void )
{
  if ( condition_false ) return;
  dprintf( "bank_defaults_preffered_stop\n" );
}

static void bank_defaults_preffered_instrument( int type )
{
  int i;

  if ( condition_false ) return;
  dprintf( "bank_defaults_preffered_instrument = %i\n", type );
  for ( i = 0; i < IT_COUNT; i++ )
    if ( p_bank -> preffered[ i ] == IT_NONE )
      {
        p_bank -> preffered[ i ] = type;
        return;
      }
  yyerror( "instrument type out of range" );
}

static void bank_prog_start( void )
{
  if ( condition_false ) return;
  if ( !bank_prog_defaults_flag )
    {
      p_prog = (struct instrument *)malloc( sizeof( struct instrument ) );
      if ( !p_prog ) yyerror( "alloc error" ); 
      memcpy( p_prog, bank_prog_defaults, sizeof( struct instrument ) );      
      bank_prog_flags = 0;
    }
  number_bitmap_clear();
}

static void bank_prog_stop( void )
{
  int i, first;

  if ( condition_false ) return;
  dprintf( "bank_prog_stop\n" );
  if ( !bank_prog_defaults_flag )
    {
      if ( !(p_prog -> flags & IF_ALIAS) )
        {
          if ( !(bank_prog_flags & DEF_FLAGS_IW_ROM_BANK) &&
               (bank_prog_defaults_flags & DEF_FLAGS_IW_ROM_BANK) )
            p_prog -> data.spec.iwrom.spec.bank = p_bank -> number;
          if ( !(bank_prog_flags & DEF_FLAGS_IW_FILE_BANK) &&
               (bank_prog_defaults_flags & DEF_FLAGS_IW_FILE_BANK) )
            p_prog -> data.spec.iwfile.spec.bank = p_bank -> number;
          for ( i = 0, first = -1; i < 256; i++ )
            if ( number_bitmap_get( i ) ) { first = i; break; }
          if ( first < 0 )
            yyerror( "program number wasn't defined!!!" );
          if ( !(p_prog -> flags & IF_IWROM_DISABLE) &&
               !(bank_prog_flags & DEF_FLAGS_IW_ROM_PROG) )
            {
              if ( bank_prog_defaults_flags & DEF_FLAGS_IW_ROM_PROG )
                p_prog -> data.spec.iwrom.spec.prog = first;
              else
                yyerror( "please - specify program in 'iwrom {}' section" );
            }
          if ( !(p_prog -> flags & IF_IWFILE_DISABLE) &&
               !(bank_prog_flags & DEF_FLAGS_IW_FILE_PROG) )
            {
              if ( bank_prog_defaults_flags & DEF_FLAGS_IW_FILE_PROG )
                p_prog -> data.spec.iwfile.spec.prog = first;
               else
                yyerror( "please - specify program in 'iwfile {}' section" );
            }   
        }
      for ( i = 0, first = -1; i < 256; i++ )
        if ( number_bitmap_get( i ) )
          {
            struct instrument *instr, *alias;

            for ( instr = p_bank -> first; instr && instr -> next; instr = instr -> next )
              if ( instr -> number == i ) yyerror( "program %i already defined\n", i );
            if ( instr )
              if ( instr -> number == i ) yyerror( "program %i already defined\n", i );
            if ( first < 0 )
              {
                p_prog -> number = first = i;
                if ( !instr )
                  p_bank -> first = p_prog;
                 else
                  instr -> next = p_prog;
              }
             else
              {
                alias = malloc( sizeof( *alias ) );
                if ( !alias ) yyerror( "alloc error" );
                memset( alias, 0, sizeof( alias ) );
  	        alias -> name = strdup( p_prog -> name );
                alias -> flags |= IF_ALIAS;
                alias -> flags |= p_prog -> flags & IF_DISABLE_ALL;
                instr -> next = alias;
              }
          }
#if 0
      printf( "rom: file = %i, bank = %i, prog = %i\n",
		p_prog -> data.spec.iwrom.spec.file,
		p_prog -> data.spec.iwrom.spec.bank,
		p_prog -> data.spec.iwrom.spec.prog );
#endif
    }
  p_prog = NULL;
}

static void bank_prog_number( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_number = %i\n", value );
  if ( value < 0 || value > 255 ) yyerror( "program value out of range" );
  number_bitmap_set( value );
  p_prog -> number = value;
}

static void bank_prog_number_range( int value1, int value2 )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_number_range = %i-%i\n", value1, value2 );
  if ( value1 < 0 || value1 > 255 ) yyerror( "program range out of range" );
  if ( value2 < 0 || value2 > 255 ) yyerror( "program range out of range" );
  if ( value1 > value2 ) yyerror( "bad range" );
  while ( value1 <= value2 ) bank_prog_number( value1++ );
}

static void bank_prog_text( char *text )
{
  if ( condition_false )
    {
      free( text );
      return;
    }
  dprintf( "bank_prog_text = '%s'\n", text );
  free( p_prog -> name );
  p_prog -> name = text;
}

static void bank_prog_patch_start( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_start\n" );
  p_prog -> data.spec.patch.spec.exclusion = GUS_INSTR_E_NONE;
  if ( bank_prog_defaults_flag )
    p_prog -> flags &= ~IF_PATCH_DISABLE;
}

static void bank_prog_patch_stop( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_stop\n" );
}

static void bank_prog_patch_file( char *name )
{
  if ( condition_false )
    {
      free( name );
      return;
    }
  dprintf( "bank_prog_patch_file = '%s'\n", name );
  strncpy( p_prog -> data.spec.patch.spec.filename, name, sizeof( p_prog -> data.spec.patch.spec.filename ) );
  free( name );
}

static void bank_prog_patch_group( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_group = %i\n", value );
  p_prog -> data.spec.patch.spec.exclusion_group = value;
}

static void bank_prog_patch_exclusion( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_exclusion = %i\n", value );
  switch ( value ) {
    case 0x20001: 
      p_prog -> data.spec.patch.spec.exclusion = GUS_INSTR_E_SINGLE;
      break;
    case 0x20002:
      p_prog -> data.spec.patch.spec.exclusion = GUS_INSTR_E_MULTIPLE;
      break;
    default:
      p_prog -> data.spec.patch.spec.exclusion = GUS_INSTR_E_NONE;
  }
}

static void bank_prog_patch_8bit( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_8bit\n" );
  p_prog -> flags |= IF_8BIT;
}

static void bank_prog_patch_disable( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_patch_disable\n" );
  p_prog -> flags |= IF_PATCH_DISABLE;
}

static void bank_prog_iwrom_start( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_start\n" );
  if ( bank_prog_defaults_flag )
    p_prog -> flags &= ~IF_IWROM_DISABLE;
}

static void bank_prog_iwrom_stop( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_stop\n" );
}

static void bank_prog_iwrom_file( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_file = %i\n", value );
  p_prog -> data.spec.iwrom.spec.file = value;
}

static void bank_prog_iwrom_bank( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_bank = %i\n", value );
  if ( bank_prog_defaults_flag && value == 0x10000 )
    {
      bank_prog_defaults_flags |= DEF_FLAGS_IW_ROM_BANK;
      return;
    }
  if ( value < 0 || value > 0x3fff ) yyerror( "bank out of range" );
  p_prog -> data.spec.iwrom.spec.bank = value;
  bank_prog_flags |= DEF_FLAGS_IW_ROM_BANK;
}

static void bank_prog_iwrom_prog( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_prog = %i\n", value );
  if ( bank_prog_defaults_flag && value == 0x10000 )
    {
      bank_prog_defaults_flags |= DEF_FLAGS_IW_ROM_PROG;
      return;
    }
  if ( value < 0 || value > 255 ) yyerror( "program out of range" );
  p_prog -> data.spec.iwrom.spec.prog = value;
  bank_prog_flags |= DEF_FLAGS_IW_ROM_PROG;
}

static void bank_prog_iwrom_disable( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwrom_disable\n" );
  p_prog -> flags |= IF_IWROM_DISABLE;
}

static void bank_prog_iwfile_start( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_start\n" );
  if ( bank_prog_defaults_flag )
    p_prog -> flags &= ~IF_IWFILE_DISABLE;
}

static void bank_prog_iwfile_stop( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_stop\n" );
}

static void bank_prog_iwfile_file( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_file = %i\n", value );
  p_prog -> data.spec.iwfile.spec.file = value;
}

static void bank_prog_iwfile_bank( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_bank = %i\n", value );
  if ( bank_prog_defaults_flag && value == 0x10000 )
    {
      bank_prog_defaults_flags |= DEF_FLAGS_IW_FILE_PROG;
      return;
    }
  if ( value < 0 || value > 0x3fff ) yyerror( "bank out of range" );
  p_prog -> data.spec.iwfile.spec.bank = value;
  bank_prog_flags |= DEF_FLAGS_IW_FILE_BANK;
}

static void bank_prog_iwfile_prog( int value )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_prog = %i\n", value );
  if ( bank_prog_defaults_flag && value == 0x10000 )
    {
      bank_prog_defaults_flags |= DEF_FLAGS_IW_FILE_PROG;
      return;
    }
  if ( value < 0 || value > 255 ) yyerror( "program out of range" );
  p_prog -> data.spec.iwfile.spec.prog = value;
  bank_prog_flags |= DEF_FLAGS_IW_FILE_PROG;
}

static void bank_prog_iwfile_disable( void )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_iwfile_disable\n" );
  p_prog -> flags |= IF_IWFILE_DISABLE;
}

static void bank_prog_alias_bank( int bank )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_alias_bank: bank = %i\n", bank );
  p_prog -> flags |= IF_ALIAS;
  p_prog -> data.alias.bank = bank;
}

static void bank_prog_alias_prog( int prog )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_alias_prog: prog = %i\n", prog );
  p_prog -> flags |= IF_ALIAS;
  p_prog -> data.alias.prog = prog;
}

static void bank_prog_format_alias_start( int format )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_format_alias_start: format = %i\n", format );
  bank_prog_format_alias_type = format;  
}

static void bank_prog_format_alias_bank( int bank )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_format_alias_bank: bank = %i\n", bank );
  switch ( bank_prog_format_alias_type ) {
    case IT_GF1_PATCH:
      p_prog -> data.spec.patch.alias.bank = bank;
      break;
    case IT_IW_ROM:
      p_prog -> data.spec.iwrom.alias.bank = bank;
      break;
    case IT_IW_FILE:
      p_prog -> data.spec.iwfile.alias.bank = bank;
      break;
  }
}

static void bank_prog_format_alias_prog( int prog )
{
  if ( condition_false ) return;
  dprintf( "bank_prog_format_alias_bank: prog = %i\n", prog );
  switch ( bank_prog_format_alias_type ) {
    case IT_GF1_PATCH:
      p_prog -> data.spec.patch.alias.prog = prog;
      break;
    case IT_IW_ROM:
      p_prog -> data.spec.iwrom.alias.prog = prog;
      break;
    case IT_IW_FILE:
      p_prog -> data.spec.iwfile.alias.prog = prog;
      break;
  }
}

static void preload_start( void )
{
  if ( condition_false ) return;
  dprintf( "preload_start\n" );
  preload_bank = malloc( sizeof( struct preload ) );
  memset( preload_bank, 0, sizeof( struct preload ) );
}

static void preload_stop( void )
{
  if ( condition_false ) return;
  dprintf( "preload_stop\n" );
  if ( !gus_icfg_config -> preload_banks )
    gus_icfg_config -> preload_banks = preload_bank;
   else
    {
      struct preload *p = gus_icfg_config -> preload_banks;
      while ( p -> next ) p = p -> next;
      p -> next = preload_bank;
    }
  preload_bank = NULL;
}

static void preload_name( char *name )
{
  if ( condition_false )
    {
      free( name );
      return;
    }
  dprintf( "preload name = '%s'\n", name );
  free( preload_bank -> name );
  preload_bank -> name = name;
}

static void preload_format_start( int type )
{
  if ( condition_false ) return;
  p_format = malloc( sizeof( *p_format ) );
  if ( !p_format ) yyerror( "alloc error" );
  memset( p_format, 0, sizeof( *p_format ) );
  p_format -> type = type;
}

static void preload_format_stop( void )
{
  struct preload_format *formats;

  if ( condition_false ) return;
  for ( formats = preload_bank -> formats; formats && formats -> next; formats = formats -> next );
  if ( !formats )
    preload_bank -> formats = p_format;
   else
    formats -> next = p_format;
  p_format = NULL;
}

static void preload_patch_8bit( void )
{
  if ( condition_false ) return;
  dprintf( "preload_patch_8bit\n" );
  p_format -> flags |= PF_8BIT;
}

static void preload_ffff_whole( void )
{
  if ( condition_false ) return;
  dprintf( "preload_ffff_whole\n" );
  p_format -> flags |= PF_FFFF_WHOLE;
}

static void preload_ffff_file( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_ffff_file: value = %i\n", value );
  if ( p_format -> type == IT_IW_ROM )
    p_format -> spec.iwrom.file = value;
   else
    p_format -> spec.iwfile.file = value;
}

static void preload_download_start( void )
{
  if ( condition_false ) return;
  dprintf( "preload_download_start\n" );
  preload_download_bank_number = -1;
}

static void preload_download_stop( void )
{
  if ( condition_false ) return;
  dprintf( "preload_download_stop\n" );
}

static void preload_download_bank( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_download_bank: value = %i\n", value );
  preload_download_bank_number = value;
}

static void preload_download_prog( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_download_prog: value = %i\n", value );
  preload_download_prog_range( value, value );
}

static void preload_download_prog_range( int value1, int value2 )
{
  struct preload_download *download, *pd;

  if ( condition_false ) return;
  dprintf( "preload_download_prog_range: value1 = %i, value2 = %i\n", value1, value2 );
  if ( preload_download_bank_number < 0 ) yyerror( "define bank number first\n" );
  download = malloc( sizeof( *download ) );
  if ( !download ) yyerror( "alloc error" );
  memset( download, 0, sizeof( *download ) );
  download -> bank = preload_download_bank_number;
  download -> min = value1;
  download -> max = value2;
  download -> next = NULL;
  for ( pd = p_format -> download; pd && pd -> next; pd = pd -> next );
  if ( !pd )
    p_format -> download = download;
   else
    pd -> next = download;
}

static void preload_aliases_start( void )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_start\n" );
  preload_aliases_bank_number = -1;
  preload_aliases_dest_bank_number = -1;
  preload_aliases_dest_prog_number = -1;
}

static void preload_aliases_stop( void )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_stop\n" );
}

static void preload_aliases_dest_bank( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_dest_bank: value = %i\n", value );
  preload_aliases_dest_bank_number = value;
}

static void preload_aliases_dest_prog( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_dest_prog: value = %i\n", value );
  preload_aliases_dest_prog_number = value;
}

static void preload_aliases_bank( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_bank: value = %i\n", value );
  preload_aliases_bank_number = value;
}

static void preload_aliases_prog( int value )
{
  if ( condition_false ) return;
  dprintf( "preload_aliases_prog: value = %i\n", value );
  preload_aliases_prog_range( value, value );
}

static void preload_aliases_prog_range( int value1, int value2 )
{
  struct preload_aliases *alias, *pa;

  if ( condition_false ) return;
  dprintf( "preload_aliases_prog_range: value1 = %i, value2 = %i\n", value1, value2 );
  if ( preload_aliases_bank_number < 0 ) yyerror( "define bank number first\n" );
  if ( preload_aliases_dest_bank_number < 0 ) yyerror( "define destonation bank number first\n" );
  if ( preload_aliases_dest_prog_number < 0 ) yyerror( "define destonation program number first\n" );
  alias = malloc( sizeof( *alias ) );
  if ( !alias ) yyerror( "alloc error" );
  memset( alias, 0, sizeof( *alias ) );
  alias -> bank = preload_aliases_bank_number;
  alias -> dest_bank = preload_aliases_dest_bank_number;
  alias -> dest_prog = preload_aliases_dest_prog_number;
  alias -> min = value1;
  alias -> max = value2;
  alias -> next = NULL;
  for ( pa = p_format -> aliases; pa && pa -> next; pa = pa -> next );
  if ( !pa )
    p_format -> aliases = alias;
   else
    pa -> next = alias;
}
