/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

%{

#include <string.h>
#include "icfg.h"
#include "icfg_parser.h"

#define MAX_INCLUDE		16
#define MAX_DEFINES		128

#undef YY_BUF_SIZE
#define YY_BUF_SIZE		2048
#define YY_NEVER_INTERACTIVE	1
#define YY_STACK_USED		1
#define YY_NO_TOP_STATE		1
#define YY_NO_UNPUT
#undef YY_CDECL
#define YY_CDECL int YY_PROTO(yylex( void ));
#define yyerror gus_icfg_error

static int defines_count;
static char *defines_heap[ MAX_DEFINES ];
static int include_stack_ptr;
static YY_BUFFER_STATE include_stack[ MAX_INCLUDE ];
static int include_line_count[ MAX_INCLUDE ];
static char *include_file[ MAX_INCLUDE ];

%}

DIGIT		[0-9]
HEXA		[0-9a-f]
STRING		[a-z0-9/\~@-_\+:,\.]

%x include
%x comment

%%

^include		yy_push_state( include );
<include>[ \t]+		/* eat the whitespace */
<include>[^ \t\n]+	{
			  if ( ++include_stack_ptr >= MAX_INCLUDE )
			    yyerror( "includes nested too deeply" );
			  include_stack[ include_stack_ptr ] = YY_CURRENT_BUFFER;
			  include_line_count[ include_stack_ptr ] = gus_icfg_config -> line_count;
			  include_file[ include_stack_ptr ] = gus_icfg_config -> filename;
			  gus_icfg_config -> line_count = 0;
			  if ( !strncmp( yytext, "etc_dir(", 8 ) ||
                               !strncmp( yytext, "lib_dir(", 8 ) )
                            {
                              char *x = (char *)malloc( 512 );
                              strcpy( x, yytext[ 0 ] == 'e' ? GUS_PATH_ETC : GUS_PATH_LIB );
                              strcat( x, yytext + 8 );
                              x[ strlen(x) - 1 ] = 0;
   			      gus_icfg_config -> filename = strdup( x );
                              free( x );
                            }
                           else
  			    gus_icfg_config -> filename = strdup( yytext );
			  yyin = fopen( gus_icfg_config -> filename, "r" );
			  if ( !yyin )
			    yyerror( "can't open include file '%s'", gus_icfg_config -> filename );
			  yy_switch_to_buffer( yy_create_buffer( yyin, YY_BUF_SIZE ) );
			  yy_pop_state();
			}
<<EOF>>			{
			  if ( include_stack_ptr < 0 )
			    yyterminate();
			   else
			    {
			      yy_delete_buffer( YY_CURRENT_BUFFER );
                              fclose( yyin );
			      gus_icfg_config -> line_count = include_line_count[ include_stack_ptr ];
			      if ( include_stack_ptr > 0 )
                                free( gus_icfg_config -> filename );
			      gus_icfg_config -> filename = include_file[ include_stack_ptr ];
			      yy_switch_to_buffer( include_stack[ include_stack_ptr-- ] );
			    }
			}

	/* special characters */

"{"|"}"			return yytext[0];
"("|")"			return yytext[0];
"*"|"+"|"-"|"/"		return yytext[0];
">>"			return L_RSHIFT;
"<<"			return L_LSHIFT;
"||"			return L_OR;
"&&"			return L_AND;
"&"|"|"			return yytext[0];
"<"			return L_LE;
"<="			return L_LEQ;
">"			return L_GT;
">="			return L_GTQ;
"=="			return L_EQ;
"!="			return L_NEQ;
"!"			return yytext[0];
"@"			return yytext[0];
","			return yytext[0];

	/* keywords - conditions */

if			return L_IF;
ife			return L_IFE;
else			return L_ELSE;

	/* keywords - level 1 */

gf1_patch_path		return L_GF1_PATCH_PATH;
iw_ffff_file		return L_IW_FFFF_FILE;
bank			return L_BANK;
preload			return L_PRELOAD;
syslog			return L_SYSLOG;
abort			return L_ABORT;
"file_exist("		return L_FILE_EXIST;
"patch_exist("		return L_PATCH_EXIST;
"rom_exist("		return L_ROM_EXIST;
"etc_dir("		return L_ETC_DIR;
"lib_dir("		return L_LIB_DIR;

	/* keywords - iw_ffff_file definition */

ffff_name		return L_FFFF_NAME;
data_name		return L_DATA_NAME;
download		return L_DOWNLOAD;
rom_name		return L_ROM_NAME;
rom_file		return L_ROM_FILE;
rom_bank		return L_ROM_BANK;

	/* keywords - bank definition */

number			return L_NUMBER;
name			return L_NAME;
defaults		return L_DEFAULTS;
prog			return L_PROG;
copy_from_bank		return L_COPY_FROM_BANK;

	/* keywords - defaults definition */

preffered		return L_PREFFERED;

	/* keywords - prog definition */

numbers			return L_NUMBERS;
text			return L_TEXT;
patch			return L_PATCH;
iwrom			return L_IWROM;
iwfile			return L_IWFILE;
alias			return L_ALIAS;

	/* keywords - patch definition */

file			return L_FILE;
group			return L_GROUP;
exclusion		return L_EXCLUSION;
disable			return L_DISABLE;
wide_8bit		return L_WIDE_8BIT;

	/* keywords - preload definition */

whole			return L_WHOLE;
dest_bank		return L_DEST_BANK;
dest_prog		return L_DEST_PROG;
progs			return L_PROGS;
value			return L_VALUE;

	/* boolean */

true|yes|on		{ gus_icfg_lval.i_value = 1; return L_TRUE; }
false|no|off		{ gus_icfg_lval.i_value = 0; return L_FALSE; }
"defined("		return L_DEFINED;
define			return L_DEFINE;
undef			return L_UNDEF;

	/* integers */

[0-9]+			{ gus_icfg_lval.i_value = atoi( yytext ); return L_INTEGER; }
0x[0-9a-f]+		{ char *end;
                          gus_icfg_lval.i_value = strtol( yytext, &end, 0 );
                          return L_INTEGER; }
current			{ gus_icfg_lval.i_value = 0x10000; return L_INTEGER; }
range			return L_RANGE;
none			{ gus_icfg_lval.i_value = 0x20000; return L_INTEGER; }
single			{ gus_icfg_lval.i_value = 0x20001; return L_INTEGER; }
multiple		{ gus_icfg_lval.i_value = 0x20002; return L_INTEGER; }

$CARD_NUMBER		{ gus_icfg_lval.i_value = gus_icfg_config -> card + 1; return L_INTEGER; }
$RAM_SIZE		{ gus_icfg_lval.i_value = gus_icfg_config -> info_data.memory_size / 1024; return L_INTEGER; }
$ROM_SIZE		{ gus_icfg_lval.i_value = gus_icfg_config -> info_data.memory_rom_size / 1024; return L_INTEGER; }
$ROM_BANKS		{ gus_icfg_lval.i_value = gus_icfg_config -> info_data.memory_rom_banks / 1024; return L_INTEGER; }

	/* strings */

\"[^\"]*\"		{ yytext[ strlen( yytext ) - 1 ] = 0;
			  gus_icfg_lval.s_value = strdup( &yytext[ 1 ] );
			  return L_STRING; }
\'[^\']*\'		{ yytext[ strlen( yytext ) - 1 ] = 0;
			  gus_icfg_lval.s_value = strdup( &yytext[ 1 ] );
			  return L_STRING; }
{STRING}+		{ gus_icfg_lval.s_value = strdup( yytext );
			  return L_STRING; }
${STRING}+		{ char *str = getenv( &yytext[ 1 ] );
			  if ( !str ) str = "";
			  gus_icfg_lval.s_value = strdup( str );
			  return L_STRING;
			}

	/* comments & whitespaces */

"/*"			yy_push_state( comment );
<comment>[^*\n]*	/* eat anything that's not a '*' */
<comment>"*"+[^*/\n]*	/* eat up '*'s not followed by '/'s */
<comment>\n		++gus_icfg_config -> line_count;
<comment>"*"+"/"	yy_pop_state();

#\ [0-9]+[^\n]*		gus_icfg_config -> line_count = atoi( yytext + 2 ) - 2;

[\;][^\n]*\n		gus_icfg_config -> line_count++;
[ \t]+			;
\n			gus_icfg_config -> line_count++;
.			gus_icfg_warning( "discarding char '%c'", yytext[ 0 ] );

%%

#ifndef gus_icfg_wrap
int gus_icfg_wrap(void)	/* do this avoid to do -lfl */
{
  return 1;
}
#endif

void gus_icfg_lexer_init( void )
{
  int i;
  char id[ 9 ];
  char str[ 32 ];

  gus_icfg_config -> line_count = 0;
  defines_count = 0; 
  include_stack_ptr = -1;
  strncpy( id, gus_icfg_config -> info_data.id, 8 );
  id[ 8 ] = 0;
  for ( i = 0; i < sizeof( id ); i++ )
    {
      unsigned char c = id[ i ];
      id[ i ] = c >= 'a' && c <= 'z' ? c - ( 'a' - 'A' ) : c;
    }
  sprintf( str, "CARD_ID_%s", id );
  gus_icfg_lexer_defines_add( str );
  if ( gus_icfg_config -> pnp_flag )
    gus_icfg_lexer_defines_add( "CHIP_INTERWAVE" );
   else
    gus_icfg_lexer_defines_add( "CHIP_GF1" );
  switch ( gus_icfg_config -> midi_emul ) {
    case GUS_MIDI_EMUL_GM:
      gus_icfg_lexer_defines_add( "EMUL_GM" );
      break;
    case GUS_MIDI_EMUL_GS:
      gus_icfg_lexer_defines_add( "EMUL_GS" );
      break;
    case GUS_MIDI_EMUL_MT32:
      gus_icfg_lexer_defines_add( "EMUL_MT32" );
      break;
    default:
      yyerror( "unknown midi emulation" );
  }
}

void gus_icfg_lexer_done( void )
{
  int i;

  for ( i = 0; i < defines_count; i++ )
    free( defines_heap[ i ] );
  defines_count = 0;
}

void gus_icfg_lexer_defines_add( char *s )
{
#if 0
  printf( "add = '%s'\n", s );
#endif
  gus_icfg_lexer_defines_remove( s );
  if ( defines_count < MAX_DEFINES )
    defines_heap[ defines_count++ ] = strdup( s );
   else
    yyerror( "define head overflow" );
}

void gus_icfg_lexer_defines_remove( char *s )
{
  int i, j;

  for ( i = 0; i < defines_count; i++ )
    if ( !strcmp( defines_heap[ i ], s ) )
      {
        for ( j = i + 1; j < defines_count; j++ )
          defines_heap[ j - 1 ] = defines_heap[ j ];
        defines_count--;
      }
}

int gus_icfg_lexer_defines_test( char *s )
{
  int i;

  for ( i = 0; i < defines_count; i++ )
    if ( !strcmp( defines_heap[ i ], s ) )
      return 1;
  return 0;
}
