#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <malloc.h>
#include <sys/stat.h>
#include "icfg.h"

#define yyerror gus_icfg_error

extern FILE *gus_icfg_in;
int gus_icfg_parse( void );

struct gus_icfg_config *gus_icfg_config = NULL;
static struct gus_icfg_config *gus_icfg_config_first;

static int global_write_status;
static int global_whole_flag;

/* ---- */

#ifdef DEBUG

void __gus_icfg_dprintf( char *fmt, ... )
{
  va_list va;
  
  fprintf( stderr, "icfg error >>> " );
  va_start( va, fmt );
  vfprintf( stderr, fmt, va );
  va_end( va );
  fprintf( stderr, "\n" );
  fflush( stderr );
}

#endif

void gus_icfg_error( char *format, ... )
{
  va_list va;
  
  va_start( va, format );
  gus_icfg_config -> error( gus_icfg_config -> key, gus_icfg_config -> filename, gus_icfg_in != NULL ? gus_icfg_config -> line_count + 1 : -1, format, va );
  va_end( va );
  exit( 1 );
}
              
void gus_icfg_warning( char *format, ... )
{
  va_list va;
  
  va_start( va, format );
  gus_icfg_config -> warning( gus_icfg_config -> key, gus_icfg_config -> filename, gus_icfg_in != NULL ? gus_icfg_config -> line_count + 1 : -1, format, va );
  va_end( va );
}
              
/* ---- */

char *gus_icfg_look_for_patch_file( char *filename, char *extension )
{
  static char result[ 256 + 1 ];
  struct stat st;
  struct gf1path *path;

  for ( path = gus_icfg_config -> gf1_paths; path; path = path -> next )
    {
      strcpy( result, path -> path );
      strcat( result, filename );
      strcat( result, extension );
#if 0
      printf( "look_for_file '%s'..\n", result );
#endif
      if ( !stat( result, &st ) )
        return result;
    }
  return NULL;
}

static inline struct bank *look_for_bank( unsigned short bank )
{
  struct bank *pbank;
  
  for ( pbank = gus_icfg_config -> banks; pbank; pbank = pbank -> next )
    if ( pbank -> number == bank ) break;
  return pbank;
}

static inline struct instrument *look_for_instrument( struct bank *bank, unsigned short instrument )
{
  struct instrument *instr;

  if ( !bank ) return NULL;
  for ( instr = bank -> first; instr; instr = instr -> next )
    if ( instr -> number == instrument ) break;
  return instr;
}

static struct iwfile *get_ffff_file( int file )
{
  struct iwfile *ifile;

  for ( ifile = gus_icfg_config -> iwfiles; ifile; ifile = ifile -> next )
    if ( ifile -> number == file ) return ifile;
  return NULL;
}

int download_ffff_blocks( void )
{
  struct iwfile *iwf;

  for ( iwf = gus_icfg_config -> iwfiles; iwf; iwf = iwf -> next )
    if ( iwf -> flags & IWF_ROM )
      {
        gus_rom_interwave_header_t header;
      
        if ( gus_instr_ffff_get_rom_header( gus_icfg_config -> fd, gus_icfg_config -> ffff_access, iwf -> spec.rom.bank, &header ) < 0 )
          {
            gus_dprintf( "can't get header of internal ROM '%s'.. aborting..", iwf -> spec.rom.name );
            return -1;
          }
        if ( strncmp( header.series_name, iwf -> spec.rom.name, 16 ) )
          {
            gus_dprintf( "wrong header name = '%s' - current = '%s'.. aborting..", iwf -> spec.rom.name, header.series_name );
            return -1;
          }
        if ( ( iwf -> handle = gus_instr_ffff_open_rom( gus_icfg_config -> fd, gus_icfg_config -> ffff_access, iwf -> spec.rom.bank, iwf -> spec.rom.file ) ) < 0 )
          {
            gus_dprintf( "can't open internal ROM '%s'.. aborting..", iwf -> spec.rom.name );
            return -1;
          }
      }
     else
      {
        if ( ( iwf -> handle = gus_instr_ffff_open( iwf -> spec.file.name, iwf -> spec.file.data ) ) < 0 )
          {
            gus_dprintf( "can't open/read file '%s' - '%s'.. aborting..", iwf -> spec.file.name, iwf -> spec.file.data );
            return -1;
          }
        if ( (iwf -> flags & IWF_DOWNLOAD) && gus_instr_ffff_download( iwf -> handle, gus_icfg_config -> fd, gus_icfg_config -> ffff_access, 0 ) < 0 )
          {
            gus_dprintf( "can't download whole file '%s' to onboard RAM.. aborting..", iwf -> spec.file.data );
            return -1;
          }
      }
  return 0;
}

int download_ffff_blocks_free( void )
{
  struct iwfile *iwf;

  for ( iwf = gus_icfg_config -> iwfiles; iwf; iwf = iwf -> next )
    {
      if ( !(iwf -> flags & IWF_ROM) )
        {
          if ( iwf -> flags & IWF_DOWNLOAD )
            gus_instr_ffff_download_free( iwf -> handle, gus_icfg_config -> fd, gus_icfg_config -> ffff_access, 0 );
           else
          if ( iwf -> flags & IWF_TEMP )
            gus_instr_ffff_download_free( iwf -> handle, gus_icfg_config -> fd, gus_icfg_config -> ffff_access, 1 );
        }
      gus_instr_ffff_close( iwf -> handle );
    }
  return 0;
}

/* --- */

int gus_icfg_open( int key )
{
  struct gus_icfg_config *config, *config1;
  
  gus_icfg_in = NULL;
  if ( ( config = malloc( sizeof( struct gus_icfg_config ) ) ) == NULL )
    return -1;
  memset( config, 0, sizeof( struct gus_icfg_config ) );
  config -> key = key;
  for ( config1 = gus_icfg_config_first; config1 && config1 -> next && config1 -> key != key; config1 = config1 -> next );
  if ( !config1 )
    gus_icfg_config_first = config;
   else
    {
      if ( config1 -> key == key )
        {
          free( config );
          return -1;
        }
      config1 -> next = config;
    }
  return 0;
}

int gus_icfg_close( int key )
{
  struct gus_icfg_config *config, *config1;
  
  config = NULL;
  for ( config1 = gus_icfg_config_first; config1 && config1 -> key != key; config1 = config1 -> next ) config = config1;
  if ( !config1 ) return -1;
  if ( config )
    config -> next = config1 -> next;
   else
    gus_icfg_config_first = config1 -> next;
  config = gus_icfg_config;
  if ( config == config1 ) config = NULL;
  gus_icfg_config = config1;
  gus_icfg_unload();
  free( config1 -> filename );
  free( config1 );
  gus_icfg_config = config;
  return 0;
}

int gus_icfg_select( int key )
{
  struct gus_icfg_config *config1;

  for ( config1 = gus_icfg_config_first; config1 && config1 -> key != key; config1 = config1 -> next );
  if ( !config1 ) return -1;
  gus_icfg_config = config1;
  return 0;
}

int gus_icfg_load( void )
{
  int res;

  gus_icfg_config -> banks = NULL;
  gus_icfg_config -> gf1_paths = NULL;
  gus_icfg_config -> iwfiles = NULL;
  gus_icfg_config -> preload_banks = NULL;
  gus_icfg_in = NULL;
  if ( ( gus_icfg_in = fopen( gus_icfg_config -> filename, "r" ) ) == NULL )
    return -1;
  gus_icfg_parser_init();
  res = gus_icfg_parse();
  gus_icfg_parser_done();
  fclose( gus_icfg_in );
  gus_icfg_in = NULL;
  if ( !res )
    {
      if ( ( res = download_ffff_blocks() ) < 0 )
        download_ffff_blocks_free();
    }
  return res ? -1 : 0;
}

void gus_icfg_unload( void )
{
  struct bank *pbank, *obank;
  struct instrument *pinstr, *oinstr;
  struct preload *ppre, *opre;
  struct preload_format *pfor, *ofor;
  struct preload_download *pdow, *odow;
  struct preload_aliases *pali, *oali;
  struct gf1path *pgf1, *ogf1;
  struct iwfile *piw, *oiw;
  
  download_ffff_blocks_free();
  pbank = gus_icfg_config -> banks;
  while ( pbank )
    {
      obank = pbank;
      pbank = pbank -> next;
      pinstr = obank -> first;
      while ( pinstr )
        {
          oinstr = pinstr;
          pinstr = pinstr -> next;
          free( oinstr -> name );
          free( oinstr );
        }
      free( obank -> name );
      free( obank );
    }
  ppre = gus_icfg_config -> preload_banks;
  while ( ppre )
    {
      opre = ppre;
      ppre = ppre -> next;
      pfor = opre -> formats;
      while ( pfor ) 
        {
          ofor = pfor;
          pfor = pfor -> next;
          pdow = ofor -> download;
          while ( pdow )
            {
              odow = pdow;
              pdow = pdow -> next;
              free( odow );
            }
          pali = ofor -> aliases;
          while ( pali )
            {
              oali = pali;
              pali = pali -> next;
              free( oali );
            }
          free( ofor );
        }
      free( opre -> name );
      free( opre );
    }  
  pgf1 = gus_icfg_config -> gf1_paths;
  while ( pgf1 )
    {
      ogf1 = pgf1;
      pgf1 = pgf1 -> next;
      free( ogf1 -> path );
      free( ogf1 );
    }
  piw = gus_icfg_config -> iwfiles;
  while ( piw )
    {
      oiw = piw;
      piw = piw -> next;
      if ( oiw -> flags & IWF_ROM )
        {
          free( oiw -> spec.rom.name );
        }
       else
        {
          free( oiw -> spec.file.name );
          free( oiw -> spec.file.data );
        }
      free( oiw );
    }
  gus_icfg_config -> banks = NULL;
  gus_icfg_config -> gf1_paths = NULL;
  gus_icfg_config -> iwfiles = NULL;
  gus_icfg_config -> preload_banks = NULL;  
}

/* --- */

static int write_not_found( unsigned int instrument )
{
  gus_instrument_t *ginstr;
  struct instrument *instr;

  ginstr = gus_instr_alloc();
  if ( !ginstr ) return -1;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
  ginstr -> number.instrument = instrument;
  instr = look_for_instrument( look_for_bank( instrument >> 16 ), instrument );
  if ( instr )
    ginstr -> name = strdup( instr -> name );
  gus_icfg_config -> memory_alloc( gus_icfg_config, ginstr );
  gus_instr_free( ginstr );
  return 0;
}

static int write_instrument( unsigned int instrument, int type, int s8bit )
{
  gus_instrument_t *ginstr;
  struct instrument *instr;
  struct iwfile *iwf;
  int alias;

#if 0
  printf( "instrument = %i, type = %i, 8bit = %i\n", instrument, type, s8bit );
#endif
  if ( !gus_icfg_config -> pnp_flag && type == IT_IW_ROM )
    {
      global_write_status = 0;
      return 0;
    }
  __again:
  global_write_status = 0;		/* not found */
  if ( gus_icfg_config -> memory_loaded( gus_icfg_config, instrument ) )
    {
      global_write_status++;
      return 0;
    }
  alias = 0;
  ginstr = gus_instr_alloc();
  if ( !ginstr ) return 0;
  instr = look_for_instrument( look_for_bank( instrument >> 16 ), instrument );
  if ( !instr || ( instr && (instr -> flags & IF_DISABLED( type )) ) )
    {
      gus_instr_free( ginstr );
      return 0;
    } 
  ginstr -> number.instrument = instrument;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
  ginstr -> name = strdup( instr -> name );
  if ( instr -> flags & IF_ALIAS )
    {
      ginstr -> flags = GUS_INSTR_F_ALIAS;
      ginstr -> info.alias = alias = ( instr -> data.alias.bank << 16 ) | instr -> data.alias.prog;
      alias |= 0x80000000;
      global_write_status++;
    }
   else
  if ( instr -> data.spec.aliases & IFA_TEST( type ) )
    {
      switch ( type ) {
        case IT_GF1_PATCH:
          alias = ( instr -> data.spec.patch.alias.bank << 16 ) | instr -> data.spec.patch.alias.prog;
          break;
        case IT_IW_ROM:
          alias = ( instr -> data.spec.iwrom.alias.bank << 16 ) | instr -> data.spec.iwrom.alias.prog;
          break;
        case IT_IW_FILE:
          alias = ( instr -> data.spec.iwfile.alias.bank << 16 ) | instr -> data.spec.iwfile.alias.prog;
          break;
        default:
          alias = 0xffffffff;
          break;
      }
      if ( alias != 0xffffffff )
        {
          ginstr -> flags |= GUS_INSTR_F_ALIAS;
          ginstr -> info.alias = alias;
          alias |= 0x80000000;
        }
       else
        ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
    }
   else
  switch ( type ) {
    case IT_GF1_PATCH:
      {
        char patch_filename[ 9 ];
        char *filename;
        
        strncpy( patch_filename, instr -> data.spec.patch.spec.filename, 8 );
        patch_filename[ 8 ] = 0;
        if ( ( filename = gus_icfg_look_for_patch_file( patch_filename, ".pat" ) ) == NULL )
          gus_dprintf( "patch file '%s' not found", instr -> data.spec.patch.spec.filename );
         else
        if ( gus_instr_patch_load( ginstr, filename, s8bit ) >= 0 )
          {
            ginstr -> flags = GUS_INSTR_F_NORMAL;
            ginstr -> exclusion = instr -> data.spec.patch.spec.exclusion;
            ginstr -> exclusion_group = instr -> data.spec.patch.spec.exclusion_group;
            global_write_status++;
          }
      }
      break;
    case IT_IW_ROM:
      iwf = get_ffff_file( instr -> data.spec.iwrom.spec.file );
      if ( iwf && gus_instr_ffff_load( ginstr, iwf -> handle, instr -> data.spec.iwrom.spec.bank, instr -> data.spec.iwrom.spec.prog ) >= 0 )
        {
          ginstr -> flags = GUS_INSTR_F_NORMAL;
          global_write_status++;
        }
      break;
    case IT_IW_FILE:
      iwf = get_ffff_file( instr -> data.spec.iwfile.spec.file );
      if ( iwf && gus_instr_ffff_load( ginstr, iwf -> handle, instr -> data.spec.iwfile.spec.bank, global_whole_flag | instr -> data.spec.iwfile.spec.prog ) >= 0 )
        {
          ginstr -> flags = GUS_INSTR_F_NORMAL;
          global_write_status++;
        }
      break;
    default:
      ginstr -> flags = GUS_INSTR_F_NOT_FOUND;
      break;
  }
  gus_icfg_config -> memory_alloc( gus_icfg_config, ginstr );
  gus_instr_free( ginstr );
  if ( alias )
    {
      instrument = (unsigned short)alias;
      goto __again;
    }
  return 0;
}

static int write_alias( unsigned int instrument, unsigned int to_instrument, int type, int s8bit )
{
  gus_instrument_t *ginstr;
  struct instrument *instr;
  
  ginstr = gus_instr_alloc();
  ginstr -> flags |= GUS_INSTR_F_ALIAS;
  ginstr -> mode = GUS_INSTR_SIMPLE;
  ginstr -> number.instrument = instrument;
  ginstr -> info.alias = to_instrument;
  instr = look_for_instrument( look_for_bank( instrument >> 16 ), (unsigned short)instrument );
  if ( instr )
    ginstr -> name = strdup( instr -> name );
  gus_icfg_config -> memory_alloc( gus_icfg_config, ginstr );
  gus_instr_free( ginstr );
  return 0;
}

static int write_bank( struct preload_format *formats )
{
  struct preload_download *download;
  struct preload_aliases *aliases;
  struct iwfile *iwf;
  int i, handle = -1;

  while ( formats )
    {
      if ( !gus_icfg_config -> pnp_flag && formats -> type == IT_IW_ROM )
        {
          formats = formats -> next;
          continue;
        }
      global_whole_flag = 0;
      if ( formats -> type == IT_IW_ROM )
        {
          iwf = get_ffff_file( formats -> spec.iwrom.file );
          if ( iwf ) handle = iwf -> handle;
        }
       else
      if ( formats -> type == IT_IW_FILE )
        {
          iwf = get_ffff_file( formats -> spec.iwfile.file );
          if ( iwf ) handle = iwf -> handle;
          if ( !(iwf -> flags & IWF_DOWNLOAD) && (formats -> flags & PF_FFFF_WHOLE) )
            {
              if ( gus_instr_ffff_download( handle, gus_icfg_config -> fd, 1, 1 ) < 0 )
                gus_dprintf( "download error of file '%s'\n", (iwf ? (char *)iwf -> spec.file.data : "???") );
               else 
                {
                  if ( iwf ) iwf -> flags |= IWF_TEMP;
                  global_whole_flag = 0x100;
                }
            }
        }
      for ( download = formats -> download; download; download = download -> next )
        for ( i = download -> min; i <= download -> max; i++ )
          write_instrument( ( download -> bank << 16 ) | i, formats -> type, formats -> flags & PF_8BIT );
      for ( aliases = formats -> aliases; aliases; aliases = aliases -> next )
        for ( i = aliases -> min; i <= aliases -> max; i++ )
          write_alias( ( aliases -> bank << 16 ) | i, ( aliases -> dest_bank << 16 ) | aliases -> dest_prog, formats -> type, formats -> flags & PF_8BIT );
      formats = formats -> next;
      global_whole_flag = 0;
    }
  return 0;
}

/* --- */

int gus_icfg_preload( char *bank_name )
{
  struct preload *pre;
  
  for ( pre = gus_icfg_config -> preload_banks; pre; pre = pre -> next )
    {
      if ( strcmp( bank_name, pre -> name ) ) continue;
      return write_bank( pre -> formats );
    }
  gus_dprintf( "preload - bank '%s' not found\n", bank_name );
  return -1;
}

int gus_icfg_download( unsigned int instrument )
{
  struct bank *bank;
  int j;
  
  __again:
  bank = look_for_bank( instrument >> 16 );
  global_whole_flag = 0;
  global_write_status = 0;
  if ( bank )
    {
      for ( j = 0; j < IT_COUNT; j++ )
      if ( bank -> preffered[ j ] != IT_NONE )
        {
          if ( write_instrument( instrument, bank -> preffered[ j ], 0 ) ) return -1;
          if ( global_write_status > 0 ) break;
        }
    }
  if ( !global_write_status )
    {
      if ( ( instrument >> 16 ) > 0 )
        {
          instrument &= 0xffff;
          goto __again;
        }
      return write_not_found( instrument );
    }
  return 0;
}
