/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include "version.h"
#include "insgus.h"

extern int line_count;
extern FILE *yyin;
int yyparse( void );

#if 0
#define INSGUS_DEBUG
#endif

int card_number = -1;
struct GUS_STRU_CFG card;
int static_dma = 0;

static int kerneld_flag = 0;

static char ins_dir[ 256 ] = GUS_PATH_MODUTL;
char cfg_file[ 256 ] = GUS_FILE_INSGUS_CONF;
static char drv_file[ 256 ] = GUS_PATH_ETC "gus.o";
static char dmn_file[ 256 ] = GUS_PATH_DMN "gusd";
static char icfg_file[ 256 ] = GUS_FILE_SYNTH_CONF;

static struct GUS_STRU_CFG *first;

void error( char *fmt, ... )
{
  va_list va;
  
  va_start( va, fmt );
  fprintf( stderr, "insgus: " );
  vfprintf( stderr, fmt, va );
  fprintf( stderr, "\n" );
  fflush( stderr );
  va_end( va );
  exit( 1 );
}

void card_init( void )
{
  memset( &card, 0, sizeof( card ) );
}

void card_next( void )
{
  struct GUS_STRU_CFG *c, *c1;
  
  c = malloc( sizeof( struct GUS_STRU_CFG ) );
  memmove( c, &card, sizeof( struct GUS_STRU_CFG ) );
  c -> next = NULL;
  if ( !first )
    first = c;
   else
    {
      c1 = first;
      while ( c1 -> next ) c1 = c1 -> next;
      c1 -> next = c;
    }
}

void card_erase( void )
{
  struct GUS_STRU_CFG *c, *c1;
  struct driver_command *cmd, *cmd1;
  
  if ( ( c = first ) == NULL ) return;
  do {
    c1 = c -> next;
    cmd = (struct driver_command *)c -> user_data;
    while ( cmd )
      {
        cmd1 = cmd;
        cmd = cmd -> next;
        free( cmd1 -> command );
        free( cmd1 );
      }
    free( c );
  } while ( ( c = c1 ) != NULL );
  first = NULL;
}

#ifdef INSGUS_DEBUG

void card_print( void )
{
  struct GUS_STRU_CFG *c;
  int i;

  for ( c = first, i = 0; c; c = c -> next, i++ )
    {
      printf( "card %i\n\n", i + 1 );
      printf( "  port = %s%x\n", c -> port > 0 ? "0x" : "", c -> port );
      printf( "  irq = %i\n", c -> irq );
      printf( "  dma1 = %i\n" , c -> dma1 );
      printf( "  dma1_size = %i\n", c -> dma1_size );
      printf( "  dma2 = %i\n", c -> dma2 );
      printf( "  dma2_size = %i\n", c -> dma2_size );
      printf( "\n" );
    }
}

#endif /* INSGUS_DEBUG */

long get_memory_size( void )
{
  FILE *p;
  char str[ 256 ];
  long total, used, free, shared, buffers;
  
  if ( ( p = fopen( "/proc/meminfo", "r" ) ) == NULL )
    error( "proc filesystem required for DMA size autoconfiguration" );
  fgets( str, sizeof( str ) - 1, p );
  if ( !strncmp( str, "MemTotal:", 9 ) )	/* new 2.1.40+ kernels */
    {
      sscanf( str + 9, "%li", &total );
      total <<= 10;
    }
   else
    fscanf( p, "%s %li %li %li %li %li", str, &total, &used, &free, &shared, &buffers );
  fclose( p );
  if ( total <= 0 )
    error( "proc filesystem error" );
#ifdef INSGUS_DEBUG
  printf( "total memory size = %li\n", total );
#endif
  return total;
}

void card_process( void )
{
  struct GUS_STRU_CFG *c;
  short t;
  long l;
  int i, card_count;
  int logical[ GUS_CARDS ];

  for ( i = 0; i < GUS_CARDS; i++ ) logical[ i ] = 0;
  for ( c = first, i = 0, card_count = 0; c; c = c -> next, i++, card_count++ )
    {
      c -> version = GUS_STRU_CFG_VERSION;
      if ( logical[ i ] )
        error( "wrong logical number for card %i (this number is already used)", i + 1 );
      logical[ i ] = c -> logical;
      if ( c -> dma1_size == 0x100 || c -> dma2_size == 0x100 )
        {
          l = get_memory_size();
          if ( l <=  4 * 1024 * 1024 ) t = 8; else
          if ( l <=  8 * 1024 * 1024 ) t = 16; else
          if ( l <  16 * 1024 * 1024 ) t = 32; else
          if ( l <  24 * 1024 * 1024 ) t = 64; else
                                       t = 128;
          if ( c -> dma1_size == 0x100 ) c -> dma1_size = c -> dma1 < 0x100 && c -> dma1 < 4 && t > 64 ? 64 : t;
          if ( c -> dma2_size == 0x100 ) c -> dma2_size = c -> dma2 < 0x100 && c -> dma2 < 4 && t > 64 ? 64 : t;
          if ( c -> daughter_dma_size == 0x100 ) c -> daughter_dma_size = t > 64 ? 64 : t;
        }
      if ( c -> dma1 == c -> dma2 && 
           c -> dma1 < 0x100 && c -> dma2 < 0x100 && 
           c -> dma1_size != c -> dma2_size )
        error( "sizes of DMA buffers for card %i must be same", i + 1 );
    }
  for ( i = 0; i < card_count; i++ )
    if ( !logical[ i ] )
      error( "all cards must have unique logical number" );
}

char *get_cfg_string( int header )
{
  struct GUS_STRU_CFG *c;
  unsigned char *s, *dest, *src;
  int cards;
  int i, cnt;
  int kv1, kv2, kv3, new_insmod;
  FILE *in;
    
  new_insmod = 0;
  if ( ( in = fopen( "/proc/sys/kernel/osrelease", "r" ) ) == NULL )
    error( "/proc filesystem isn't mounted" );
  fscanf( in, "%i.%i.%i", &kv1, &kv2, &kv3 );
  fclose( in );
  if ( kv1 > 2 || ( kv1 == 2 && kv2 > 1 ) || ( kv1 == 2 && kv2 == 1 && kv3 > 17 ) )
    new_insmod++;
  for ( c = first, cards = 0; c; c = c -> next, cards++ ) ;
  s = malloc( ( sizeof( struct GUS_STRU_CFG ) * cards * 2 ) + 32 );
  s[ 0 ] = 0;
  if ( header )
    strcpy( s, "gus_config_string=" );
  if ( new_insmod )
    strcat( s, "\"" );
  strcat( s, "GUS-CFG*" );
  dest = s + strlen( s );
  for ( c = first; c; c = c -> next )
    for ( src = (char *)c, cnt = sizeof( struct GUS_STRU_CFG ); cnt > 0; src++, cnt-- )
      {
        i = *src >> 4;
        *dest++ = i < 10 ? i + '0' : i + 'A' - 10;
        i = *src & 0x0f;
        *dest++ = i < 10 ? i + '0' : i + 'A' - 10;
      }
  *dest = 0;
  if ( new_insmod )
    strcat( dest, "\\0\"" );
#ifdef INSGUS_DEBUG
  printf( "-------> %s\n", s );
#endif
  return s;
}

void help( void )
{
  printf(
"Usage: insgus [options]\n"
"\n"
"	-i <modutils>	path to insmod/rmmod program\n"
"	-f <cfg file>	full path to configuration file\n"
"	-o <driver>	full path to driver file (gus.o)\n"
"	-d <daemon>	full path to instrument daemon executable (gusd)\n"
"\n"
"	-n		don't start instrument daemon(s)\n" 
"	-c <icfg file>	full path to instrument daemon configuration file\n"
"\n"
"	-m		generate map file (insmod -m)\n"
"	-a		always insert (insmod -f)\n"
"\n"
"	-r		restart instrument daemon(s)\n"
"	-k		kill driver and instrument daemon(s)\n"
"\n"
"	-V		print version of insgus\n"
"\n"
"Current path to modutils   : %s\n"
"Current configuration file : %s\n"
"Current driver file	   : %s\n"
"Current daemon file	   : %s\n"
"Current daemon cfg file    : %s\n"
, ins_dir, cfg_file, drv_file, dmn_file, icfg_file );
}

int driver_present( void )
{
  int fd, cards;
  
  /* ok. I'm using mixer interface for get number of installed GUS cards */
  if ( ( fd = open( "/dev/gusmixer0", O_RDWR ) ) < 0 )
    return -1;
  if ( ioctl( fd, SOUND_MIXER_CARDS, &cards ) < 0 )
    {
      close( fd );
      return -1;
    }
  close( fd );

  return cards;
}

void driver_remove( void )
{
  int pid;
  char str[ 128 ];

  if ( driver_present() > 0 )
    {
      sprintf( str, "%srmmod", ins_dir );
      if ( ( pid = fork() ) == 0 )
        {
          if ( execle( str, "rmmod", "gus", NULL, NULL ) < 0 )
            fprintf( stderr, "rmmod exec error - %s\n", strerror( errno ) );
        }
       else
      if ( pid > 0 )
        wait( NULL );
    }
}

void driver_commands( void )
{
  struct GUS_STRU_CFG *c;
  struct driver_command *cmd;
  FILE *file;
  
  if ( driver_present() <= 0 ) return;
  file = fopen( "/dev/gusinfo", "w" );
  if ( !file )
    {
      perror( "open error of /dev/gusinfo" );
      return;
    }
  for ( c = first; c; c = c -> next )
    for ( cmd = (struct driver_command *)c -> user_data; cmd; cmd = cmd -> next )
      fprintf( file, "%s %s\n", c -> id, cmd -> command );
  fclose( file );
}

void daemons_start( void )
{
  int pid;
  int i, cards;
  char card_number[ 5 ];

  if ( ( cards = driver_present() ) < 0 )
    {
      fprintf( stderr, "daemons start - driver not present?\n" );
      return;
    }
  for ( i = 0; i < cards; i++ )
    {
      if ( ( pid = fork() ) == 0 )
        {
          sprintf( card_number, "%i", i + 1 );
          if ( execle( dmn_file, dmn_file, "-sf", icfg_file, "-c", card_number, NULL, NULL ) < 0 )
            fprintf( stderr, "can't start %i. daemon '%s'", i + 1, dmn_file );
        }
       else
      if ( pid > 0 )
        wait( NULL );
    }
}

void daemons_remove( void )
{
  int i, pid, present;
  char path[ 128 ], path1[ 32 ];
  FILE *in;
  struct stat st;

  present = driver_present();
  for ( i = 0; i < GUS_CARDS; i++ )
    {
      sprintf( path, "%sgusd%i.pid", GUS_PATH_PID, i );
      if ( ( in = fopen( path, "r" ) ) == NULL ) continue;
      fscanf( in, "%i", &pid );
      fclose( in );
      sprintf( path1, "/proc/%i", pid );
      if ( stat( path1, &st ) < 0 || present <= 0 )
        {
          remove( path );
          continue;
        }
      if ( pid > 0 )
        {
          kill( pid, SIGTERM );
          /* is this loop needed? */
          while ( 1 )
            {
              if ( stat( path, &st ) < 0 ) break;
              usleep( 100 );
            }
        }
    }
}

int main( int argc, char *argv[] )
{
  char *s;
  int res;
  int force = 0, map_file = 0;
  int write_cfg_str = 0;
  int kill_daemons = 0, daemons_not = 0, restart_daemons = 0;
  char str[ 256 ];

  if ( getgid() != 0 )
    {
      if ( setregid( 0, 0 ) < 0 )
        {
          fprintf( stderr, "insgus: permission denied (run this program as root)\n" );
          return 1;
        }
    }

  while ( 1 )
    {
      int c;
      
      if ( ( c = getopt( argc, argv, "ahi:f:o:d:c:wknmrVK" ) ) < 0 ) break;
      switch ( c ) {
        case '?':
        case 'h':
          help();
          return 1;
        case 'a':
          force = 1;
          break;
        case 'i':
          strcpy( ins_dir, optarg );
          if ( ins_dir[ 0 ] > 0 && ins_dir[ strlen( ins_dir ) - 1 ] != '/' )
            strcat( ins_dir, "/" );
          break;
        case 'f':
          strcpy( cfg_file, optarg );
          break;
        case 'o':
          strcpy( drv_file, optarg );
          break;
        case 'd':
          strcpy( dmn_file, optarg );
          break;
        case 'c':
          strcpy( icfg_file, optarg );
          break;
        case 'w':
          write_cfg_str = 1;
          break;
        case 'k':
          kill_daemons = 1;
          break;
        case 'n':
          daemons_not = 1;
          break;
        case 'm':
          map_file = 1;
          break;
        case 'r':
          restart_daemons = 1;
          break;
        case 'V':
          printf( "insgus: Version " GUS_VERSION "\n" );
          return 0;
        case 'K':
          kerneld_flag = 1;
          break;
      }
    }
  if ( restart_daemons )
    {
      daemons_remove();
      daemons_start();
      return 0;
    }
  if ( kill_daemons )
    {
      daemons_remove();
      driver_remove();
      return 0;
    }
  if ( ( yyin = fopen( cfg_file, "r" ) ) == NULL )
    error( "can't open configuration file '%s'", cfg_file );
  line_count = 0;
  res = yyparse();
  fclose( yyin );
  if ( res )
    error( "error in configuration file '%s'", cfg_file );
  card_process();
#ifdef INSGUS_DEBUG
  card_print();
#endif
  if ( !first )
    fprintf( stderr, "insgus: configuration not found...\n" );
   else
  if ( write_cfg_str )
    {
      FILE *out;
      
      if ( write_cfg_str )
        {
          out = fopen( "gus_cfg.h", "w+" );
          fprintf( out, "#define GUS_CFG_STRING \"%s\"\n", get_cfg_string( 0 ) );
          fclose( out );
        }
    }
   else
    {
      char static_dma_str[ 24 ];
    
      if ( !kerneld_flag )
        {
          daemons_remove();		/* for sure */
          driver_remove();
        }
      s = get_cfg_string( 1 );
      sprintf( static_dma_str, "gus_config_dma_static=%i", static_dma > 0 ? 1 : 0 );
      {
        static int res = 0;
        int pid, i;
        char *p[ 8 ];

        sprintf( str, "%sinsmod", ins_dir );
        if ( ( pid = fork() ) == 0 )
          {
            i = 0;
            p[ i++ ] = "insmod";
            if ( force ) p[ i++ ] = "-f";
            if ( map_file ) p[ i++ ] = "-m";
            if ( kerneld_flag ) p[ i++ ] = "-k";
            p[ i++ ] = drv_file;
            p[ i++ ] = s;
            p[ i++ ] = static_dma_str;
            p[ i++ ] = NULL;
            res = execv( str, p );
            if ( res < 0 )
              {
                fprintf( stderr, "insgus: exec error - %s\n", strerror( errno ) );
                fprintf( stderr, "insgus: probably insmod program can't be found..\n" );
              }
          }
         else
        if ( pid > 0 )
          wait( NULL );
      }
      driver_commands();
      if ( !daemons_not ) daemons_start();
      free( s );
    }
  card_erase();
  return 0;
}
