#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <stdarg.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <syslog.h>
#include <errno.h>
#include "version.h"
#include "gusd.h"

extern char **environ;

#if 0
#define CONFIG_DEBUG_CFG
#endif

static char ident[ 8 ];
static char *pid_path;
static int standalone;
static char *config_filename;
static int card;
static int dmn_fd;
static struct GUS_STRU_DAEMON_MESSAGE dmn_msg;

/* --- */

static void pid_write( void )
{
  char filename[ 256 ];
  FILE *out;
 
  strcpy( filename, PID_PATH );
  strcat( filename, ident );
  strcat( filename, ".pid" );
  pid_path = strdup( filename );
  if ( ( out = fopen( filename, "w" ) ) == NULL )
    {
      fprintf( stderr, "cann't open pid file '%s'", filename );
      return;
    }  
  fprintf( out, "%i", getpid() );
  fclose( out );
}

static void pid_remove( void )
{
  remove( pid_path );
  free( pid_path );
}

/* --- */

void gusd_error( int key, char *filename, int line, char *format, va_list va )
{
  char message[ 512 ];

  sprintf( message, "error - file %s, line %i - ", filename, line );
  vsprintf( message + strlen( message ) - 1, format, va );
  syslog( LOG_ERR, message );
}

void gusd_warning( int key, char *filename, int line, char *format, va_list va )
{
  char message[ 512 ];

  sprintf( message, "warning - file %s, line %i - ", filename, line );
  vsprintf( message + strlen( message ) - 1, format, va );
  syslog( LOG_ERR, message );
}

/* --- */

static void free_message( struct GUS_STRU_DAEMON_MESSAGE *msg )
{
  switch ( msg -> command ) {
    case GUS_DAEMON_MSG_DOWNLOAD:
      memset( &msg -> info.instrument, 0, sizeof( msg -> info.instrument ) );
      break;
  }
}

/* --- */

static int write_command( struct GUS_STRU_DAEMON_MESSAGE *msg )
{
  if ( write( dmn_fd, msg, sizeof( *msg ) ) != sizeof( *msg ) )
    {
      if ( errno != EINTR )
        syslog( LOG_ERR, "driver write error (%i)", msg -> command );
       else
        return 1;		/* break */
    }
  return 0;
}

/* --- */

static int gusd_memory_alloc( struct gus_icfg_config *cfg, gus_instrument_t *instrument )
{
  struct GUS_STRU_DAEMON_MESSAGE msg;
  int res;
  
  memset( &msg, 0, sizeof( msg ) );
  msg.direction = GUS_DAEMON_MSG_REPLY;
  msg.command = GUS_DAEMON_MSG_DOWNLOAD;
  msg.info.instrument = instrument;
  res = write_command( &msg );
  free_message( &msg );
  return res;
}

static int gusd_memory_loaded( struct gus_icfg_config *cfg, unsigned int instrument )
{
  if ( ioctl( dmn_fd, GUS_IOCTL_DMN_INSTRUMENT, &instrument ) < 0 )
    {
      syslog( LOG_ERR, "instrument ioctl error - %s", strerror( errno ) );
      return 0;
    } 
  if ( instrument != GUS_INSTR_F_NOT_LOADED ) return 1;
  return 0;
}

/* --- */

static int reply_end( void )
{
  struct GUS_STRU_DAEMON_MESSAGE msg;

  memset( &msg, 0, sizeof( msg ) );
  msg.direction = GUS_DAEMON_MSG_REPLY;
  msg.command = GUS_DAEMON_MSG_END_OF_REQUEST;
  return write_command( &msg );
}

static void interwave_effect( void )
{
  struct GUS_STRU_EFFECT effect;
  struct GUS_STRU_DAEMON_MESSAGE msg;

  if ( gus_effect_interwave( &effect, dmn_msg.info.iw_effect[ 0 ], dmn_msg.info.iw_effect[ 1 ] ) >= 0 )
    {
      memset( &msg, 0, sizeof( msg ) );
      msg.direction = GUS_DAEMON_MSG_REPLY;
      msg.command = GUS_DAEMON_MSG_IWEFF;
      msg.info.effect = &effect;
      write_command( &msg );
      free_message( &msg );
    }
}

static void decode_message( void )
{
  int i;
  int new_midi_emul;
  struct iwfile *iwf;

#if 0
  printf( "decode message - command = %i\n", dmn_msg.command );
#endif
  if ( dmn_msg.direction != GUS_DAEMON_MSG_REQUEST )
    {
      syslog( LOG_ERR, "decode_message - wrong message direction" );
      return;
    }
  switch ( dmn_msg.command ) {
    case GUS_DAEMON_MSG_PRELOAD:
      gus_icfg_preload( dmn_msg.info.bank_name );
      break;
    case GUS_DAEMON_MSG_DOWNLOAD:
      for ( i = 0; i < 64; i++ )
        {
          if ( dmn_msg.info.instruments[ i ] != 0xffffffff )
            gus_icfg_download( dmn_msg.info.instruments[ i ] );
        }
      break;
    case GUS_DAEMON_MSG_EMUL:
      new_midi_emul = 0xffffffff;
      ioctl( dmn_fd, GUS_IOCTL_DMN_GET_EMULATION, &new_midi_emul );
      if ( new_midi_emul != 0xffffffff && new_midi_emul != gus_icfg_config -> midi_emul )
        {
          gus_icfg_unload();
          gus_icfg_config -> midi_emul = new_midi_emul;
          gus_icfg_load();
        }
      break;
    case GUS_DAEMON_MSG_BFREE:
      for ( iwf = gus_icfg_config -> iwfiles; iwf; iwf = iwf -> next )
        if ( !(iwf -> flags & IWF_ROM) && (iwf -> flags & IWF_TEMP) )
          {
#if 0
            syslog( LOG_ERR, "Free block - %i\n", iwf -> handle );
#endif
            iwf -> flags &= ~IWF_TEMP;
            gus_instr_ffff_download_free( iwf -> handle, dmn_fd, gus_icfg_config -> ffff_access, 1 );
          }
      break;
    case GUS_DAEMON_MSG_IWEFF:
      interwave_effect();
      break;
    default:
      syslog( LOG_ERR, "decode_message - unknown command" );
  }
}

static void main_loop( void )
{
  char filename[ 15 ];
  fd_set read_fds;

  sprintf( filename, "/dev/insman%i", card );
  if ( ( dmn_fd = open( filename, O_RDWR ) ) < 0 )
    {
      syslog( LOG_ERR, "can't open file '%s'", filename );
      return;
    }
  if ( gus_icfg_open( dmn_fd ) < 0 )
    {
      syslog( LOG_ERR, "main_loop: error (1)" );
      return;
    }
  if ( gus_icfg_select( dmn_fd ) < 0 )
    {
      syslog( LOG_ERR, "main_loop: error (2)" );
      return;
    }
  gus_icfg_config -> fd = dmn_fd;
  gus_icfg_config -> filename = strdup( config_filename );
  gus_icfg_config -> ffff_access = gus_instr_ffff_access_daemon();
  gus_icfg_config -> card = card;
  gus_icfg_config -> memory_alloc = gusd_memory_alloc;
  gus_icfg_config -> memory_loaded = gusd_memory_loaded;
  gus_icfg_config -> error = gusd_error;
  gus_icfg_config -> warning = gusd_warning;
  if ( ioctl( dmn_fd, GUS_IOCTL_DMN_INFO, &gus_icfg_config -> info_data ) < 0 )
    {
      syslog( LOG_ERR, "daemon info ioctl error - aborting" );
      close( dmn_fd );
      return;
    }
  if ( ioctl( dmn_fd, GUS_IOCTL_DMN_GET_EMULATION, &gus_icfg_config -> midi_emul ) < 0 )
    {
      syslog( LOG_ERR, "daemon emulation ioctl error - aborting" );
      close( dmn_fd );
      return;
    }
  gus_icfg_config -> pnp_flag = gus_icfg_config -> info_data.memory_rom_size ? 1 : 0;
  if ( gus_icfg_load() < 0 )
    {
      gus_icfg_close( dmn_fd ); 
      close( dmn_fd );
      return;
    }
  syslog( LOG_NOTICE, "Ready to serve requests." );
  while ( 1 )
    {
      FD_ZERO( &read_fds );
      FD_SET( dmn_fd, &read_fds );
      select( dmn_fd + 1, &read_fds, NULL, NULL, NULL );
      if ( FD_ISSET( dmn_fd, &read_fds ) )
        {
          int i;
        
          while ( ( i = read( dmn_fd, &dmn_msg, sizeof( dmn_msg ) ) ) == sizeof( dmn_msg ) )
            {
              decode_message();
              reply_end();
            }
          if ( i > 0 )
            syslog( LOG_WARNING, "driver read error (size = %i)?", i );
        }
    }
  gus_icfg_close( dmn_fd );
  close( dmn_fd );
}

/*
 *  Help
 */

static void help( void )
{
  printf(
"Usage: gusd [options]\n"
"\n"
"	-c <card>	select card (1-%i)\n"
"	-f <file>	configuration file\n"
"	-h		this help screen\n"
"	-s		standalone mode\n"
"	-V		print version\n"
, GUS_CARDS );
}

/*
 *  Signals
 */

void terminate( int signal )
{  
  syslog( LOG_NOTICE, "Terminate." );
  
  gus_icfg_close( dmn_fd );
  close( dmn_fd );
  closelog();
  if ( standalone )
    pid_remove();
  exit( 0 );
}

/*
 *  process id string
 */

void inst_str(char *dst[], int argc, char *src)
{
  if ( strlen( src ) <= strlen( dst[ 0 ] ) )
    {
      char *ptr;

      for ( ptr = dst[ 0 ]; *ptr; *(ptr++) = '\0' );
      strcpy( dst[ 0 ], src );
    } 
   else
    {
      /* stolen from the source to perl 4.036 (assigning to $0) */
      char *ptr, *ptr2;
      int count;
      ptr = dst[ 0 ] + strlen( dst[ 0 ] );
      for ( count = 1; count < argc; count++ ) 
        {
          if ( dst[ count ] == ptr + 1 )
            ptr += strlen( ++ptr );
        }
      if ( environ[ 0 ] == ptr + 1 ) 
        for ( count = 0; environ[ count ]; count++ )
          if ( environ[ count ] == ptr + 1 )
            ptr += strlen( ++ptr );
      count = 0;
      for ( ptr2 = dst[ 0 ]; ptr2 <= ptr; ptr2++ )
        {
          *ptr2 = '\0';
           count++;
        }
      strncpy( dst[ 0 ], src, count );
    }
}

/*
 *
 */

int main( int argc, char *argv[] )
{
  int pid;
  char str[ 16 ];

  card = 0;
  standalone = 0;
  config_filename = strdup( GUS_FILE_SYNTH_CONF );
  while ( 1 )
    {
      int c;
      
      if ( ( c = getopt( argc, argv, "c:f:hsV" ) ) < 0 ) break;
      switch ( c ) {
        case '?':
        case 'h':
          help();
          exit( 0 );
        case 'c':
          card = atoi( optarg ) - 1;
          if ( card < 0 || card >= GUS_CARDS )
            {
              help();
              exit( 0 );
            }
          break;
        case 'f':
          free( config_filename );
          config_filename = strdup( optarg );
          break;
        case 's':
          standalone = 1;
          break;
        case 'V':
          printf( "gusd: Version " GUS_VERSION "\n" );
          return 0;
      }
    }
  sprintf( ident, "gusd%i", card );
  if ( standalone )
    {
      if ( ( pid = fork() ) < 0 )
        { 
          fprintf( stderr, "fork error\n" );
          exit( 1 );
        }
      if ( pid != 0 )
        return 0;
      pid_write();
    }
  sprintf( str, "gusd <card #%i>", card + 1 );
  inst_str( argv, argc, str );
  openlog( ident, LOG_PID, LOG_DAEMON );
  signal( SIGTERM, terminate );
  signal( SIGINT, terminate );
  main_loop();
  signal( SIGTERM, SIG_DFL );
  signal( SIGINT, SIG_DFL );
  closelog();
  if ( standalone ) 
    pid_remove();
  return 0;
}
