
#ifndef __MIME_H__
#define __MIME_H__

typedef struct mime_t
{
    char *key;
    char *mimetype;
    char **apps;			
}
mime_t;

typedef struct _xfmime_functions xfmime_functions;

struct _xfmime_functions {
   void (*mime_add)(const gchar *sfx, const gchar *command);
   gboolean (*is_valid_command)(const gchar *cmd_fmt);
   const gchar *(*mime_typeinfo)(const gchar *type);
   const gchar *(*mime_key_type) (const gchar *file);
   const gchar *(*mime_get_type)(const gchar *file, gboolean try_magic);
   const gchar *(*mime_command)(const gchar *file);
   const gchar **(*mime_apps)(const gchar *file);
   const gchar *(*mk_command_line)(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold);
};
#define MIME_typeinfo (*(load_mime_module()->mime_typeinfo))
#define MIME_key_type (*(load_mime_module()->mime_key_type))
#define MIME_get_type (*(load_mime_module()->mime_get_type))
#define MIME_command (*(load_mime_module()->mime_command))
#define MIME_apps (*(load_mime_module()->mime_apps))
#define MIME_add (*(load_mime_module()->mime_add))
#define MIME_mk_command_line (*(load_mime_module()->mk_command_line))
#define MIME_is_valid_command (*(load_mime_module()->is_valid_command))


/****************************************************************/
/**
 * mime_typeinfo:
 * @type: the mimetype for which information is being requested
 *
 * Returns: A string with mimetype information. 
 * 
 **/
G_MODULE_EXPORT
const gchar *
mime_typeinfo(const gchar *type);

/****************************************************************/
/**
 *  mime_key_type:
 *  @file: Filename on which to do mime type determination.
 *  
 *  This function does not do the magic mimetype determination. Use
 *  mime_get_type() for that.
 *
 *  Returns: The mimetype associated with @file. This is done by
 *  extracting the file extension. Returns NULL is no mimetype is
 *  associated.
 **/
G_MODULE_EXPORT
const gchar *
mime_key_type (const gchar *file);


/****************************************************************/
/**
 * mime_command:
 * @file: Path to query for associated application
 *
 * As there  may be more than one application associated,
 * use mime_apps() if you want all of them. This function
 * returns the default application, which is the last associated
 * application entered into the association hash.
 * 
 * Returns: string with the default application for @file, or
 * NULL if there is no application associated. * 
 **/
G_MODULE_EXPORT
const gchar *
mime_command(const gchar *file);

/****************************************************************/
/**
 * mime_apps:
 * @file: Path of file to query for associated applications.
 *
 * Returns: array of strings with associated applications. The last
 * element of the array is a NULL pointer. If no applications are
 * associated, returns NULL.
 *
 **/
G_MODULE_EXPORT
const gchar **
mime_apps(const gchar *file);

/****************************************************************/
/**
 * mk_command_line:
 * @command_fmt: Command string to execute.
 * @path: Path to file to send with command.
 * @interm: TRUE if to execute within a terminal window.
 * @hold: TRUE if to hold the output of a terminal window on 
 * completion of command.
 * 
 * A command string may have %s to indicate the position where the
 * path is to be inserted. Otherwise the path is appended.
 *
 * Returns: A string with the command line constructed, suitable for
 * a g_spawn_async() or similar.
 * 
 **/
G_MODULE_EXPORT
const gchar *
mk_command_line(const gchar *command_fmt,const gchar *path,gboolean interm,gboolean hold);

/****************************************************************/
/**
 * mime_get_type:
 * @file: NULL terminated string with full path of file to query.
 * @try_magic: use magic rules if all else fails.
 *
 *
 * Determines mime type based on file name, file type and magic rules. 
 * If @try_magic is FALSE, then magic rules will not be applied.
 *
 * Returns: string with the mime type, or NULL if no mime type is
 * associated.
 *
 **/
G_MODULE_EXPORT
const gchar *
mime_get_type(const gchar *file, gboolean try_magic);

/****************************************************************/
/**
 * mime_add:
 * @file: File to associate command with.
 * @command: Command to associate.
 *
 * This associates a command with a file type, determined by the file
 * extension. When this command is used, the associated command becomes
 * the default command for the file type while the module remains 
 * loaded. 
 *
 * Returns: nothing.
 *
 **/
G_MODULE_EXPORT
void 
mime_add(const gchar *file, const gchar *command);

/****************************************************************/
/**
 * is_valid_command:
 * @cmd_fmt: command to check for.
 *
 * Checks whether @cmd_fmt is in $PATH and whether it can be executed.
 *
 *
 * Returns: TRUE if @cmd_fmt can be executed, FALSE otherwise.
 **/
G_MODULE_EXPORT
gboolean 
is_valid_command(const char *cmd_fmt);   


#endif
